/*
 * msynctool - A command line client for the opensync framework
 * Copyright (C) 2004-2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2006-2007  Daniel Friedrich <daniel.friedrich@opensync.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 * 
 */
 
#include <opensync/opensync.h>
#include <opensync/opensync-group.h>
#include <opensync/opensync-format.h>
#include <opensync/opensync-plugin.h>
#include <opensync/opensync-engine.h>
#include <opensync/opensync-data.h>
#include <opensync/opensync-merger.h>


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <glib.h>
#include <errno.h>

#include "config.h"

osync_bool manual = FALSE;
OSyncConflictResolution conflict = OSYNC_CONFLICT_RESOLUTION_UNKNOWN;
int winner = 0;

static void usage (char *name, int ecode)
{
  fprintf (stderr, "Usage: %s\n", name);
  fprintf (stderr, "Information about installation:\n");
  fprintf (stderr, "--listplugins    Lists all plugins\n");
  fprintf (stderr, "--listformats    Lists all formats\n");
  fprintf (stderr, "--version        Shows the version of opensync and msynctool\n");
  fprintf (stderr, "--help           Show this help\n");
  fprintf (stderr, "\n");
  fprintf (stderr, "Information about configured groups:\n");
  fprintf (stderr, "--listgroups\n");
  fprintf (stderr, "    Lists all groups\n");
  fprintf (stderr, "--showgroup <groupname>\n");
  fprintf (stderr, "    Lists all members of the group\n");
  fprintf (stderr, "--showobjtypes <groupname>\n");
  fprintf (stderr, "    Lists all objtypes that the group can synchronize\n");
  fprintf (stderr, "--showfilter <groupname>\n");
  fprintf (stderr, "    Lists the filters for a group\n");
  fprintf (stderr, "--showcapabilities <groupname>\n");
  fprintf (stderr, "    List all capabilities of the group members\n");
  fprintf (stderr, "\n");
  fprintf (stderr, "Group configuration:\n");
  fprintf (stderr, "--addgroup <groupname>\n");
  fprintf (stderr, "    Add a new group\n");
  fprintf (stderr, "--delgroup <groupname>\n");
  fprintf (stderr, "    Delete the given group\n");
  fprintf (stderr, "--configure-filter <groupname>\n");
  fprintf (stderr, "    Configures the filters of a group\n");
  fprintf (stderr, "--enable-objtype <groupname> <objtype>\n");
  fprintf (stderr, "    Enables the synchronization of a objtype for a group\n");
  fprintf (stderr, "--disable-objtype <groupname> <objtype>\n");
  fprintf (stderr, "    Disables the synchronization of a objtype for a group\n");
  fprintf (stderr, "--enable-merger <groupname>\n");
  fprintf (stderr, "    Enable the merger of a group (default: enabled)\n");
  fprintf (stderr, "--disable-merger <groupname>\n");
  fprintf (stderr, "    Disable the merger of a group\n");
  fprintf (stderr, "--enable-converter <groupname>\n");
  fprintf (stderr, "    Enable the converter of a group (default: enabled)\n");
  fprintf (stderr, "--disable-converter <groupname>\n");
  fprintf (stderr, "    Disable the converter of a group (Recommended for backups)\n");
  fprintf (stderr, "\n");
  fprintf (stderr, "Group member configuration:\n");
  fprintf (stderr, "--addmember <groupname> <plugintype> \n");
  fprintf (stderr, "    Add a member to the group\n");
  fprintf (stderr, "--delmember <groupname> <memberid>\n");
  fprintf (stderr, "    Delete a member from the group\n");
  fprintf (stderr, "--configure <groupname> <memberid>\n");
  fprintf (stderr, "    Configure a member. memberid as returned by --showgroup\n");
  fprintf (stderr, "--configure-capabilities <groupname> <memberid>\n");
  fprintf (stderr, "    Configures the capabilities of a member from the group\n");
  fprintf (stderr, "--discover <groupname> [<memberid>]\n");
  fprintf (stderr, "    Detect which objtypes are supported by one specified or all members\n");
  fprintf (stderr, "--disable-readonly <groupname> <memberid> [<objtype>]\n");
  fprintf (stderr, "    Enable writing (of objtype sink) for a member (default)\n");
  fprintf (stderr, "--enable-readonly <groupname> <memberid> [<objtype>]\n");
  fprintf (stderr, "    Disable writing (of objtype sink) for a member\n");
  fprintf (stderr, "\n");
  fprintf (stderr, "Synchronization:\n");
  fprintf (stderr, "--sync <groupname>\n");
  fprintf (stderr, "    Sync all members in a group\n");
  fprintf (stderr, "[--wait]\n");
  fprintf (stderr, "    Wait for a client to initialize the sync instead of starting immediately\n");
  fprintf (stderr, "[--multi]\n");
  fprintf (stderr, "    Repeat to wait for sync alerts\n");
  fprintf (stderr, "[--slow-sync <objtype>]\n");
  fprintf (stderr, "    Perform a slow-sync of all members in the group\n");
  fprintf (stderr, "[--manual]\n");
  fprintf (stderr, "    Make manual engine iterations. Only for debugging.\n");
  fprintf (stderr, "[--configdir]\n");
  fprintf (stderr, "    Set a different configuration directory than ~/.opensync\n");
  fprintf (stderr, "[--conflict 1-9/d/i/n] \n");
  fprintf (stderr, "    Resolve all conflicts as side [1-9] wins, [d]uplicate, [i]gnore, or\n");
  fprintf (stderr, "    keep [n]ewer\n");
  exit (ecode);
}

typedef enum  {
	NONE,
	MSYNCTOOL_INSTALL_LISTPLUGINS,
	MSYNCTOOL_INSTALL_LISTFORMATS,
	MSYNCTOOL_INSTALL_GETVERSION,
	MSYNCTOOL_SHOW_GROUPS,
	MSYNCTOOL_SHOW_GROUP,
	MSYNCTOOL_SHOW_OBJTYPES,
	MSYNCTOOL_SHOW_FILTER,
	MSYNCTOOL_SHOW_CAPABILITIES,
	MSYNCTOOL_CONFIGURE_ADDGROUP,
	MSYNCTOOL_CONFIGURE_DELGROUP,
	MSYNCTOOL_CONFIGURE_ADDMEMBER,
	MSYNCTOOL_CONFIGURE,
	MSYNCTOOL_CONFIGURE_DELMEMBER,
	MSYNCTOOL_CONFIGURE_ENABLE_OBJTYPE,
	MSYNCTOOL_CONFIGURE_DISABLE_OBJTYPE,
	MSYNCTOOL_CONFIGURE_ENABLE_MERGER,
	MSYNCTOOL_CONFIGURE_DISABLE_MERGER,
	MSYNCTOOL_CONFIGURE_ENABLE_CONVERTER,
	MSYNCTOOL_CONFIGURE_DISABLE_CONVERTER,
	MSYNCTOOL_CONFIGURE_FILTER,
	MSYNCTOOL_CONFIGURE_CAPABILITIES,
	MSYNCTOOL_CONFIGURE_DISCOVER,
	MSYNCTOOL_CONFIGURE_MEMBER_ENABLE_WRITE,
	MSYNCTOOL_CONFIGURE_MEMBER_DISABLE_WRITE,
	MSYNCTOOL_SYNC
} ToolAction;


static osync_bool msync_list_formats(OSyncFormatEnv *env, OSyncError **error)
{
	int i;
	for (i = 0; i < osync_format_env_num_objformats(env); i++) {
		OSyncObjFormat *format = osync_format_env_nth_objformat(env, i);
		printf("Format: %s\n", osync_objformat_get_name(format));
		printf("\tObject Type: %s\n", osync_objformat_get_objtype(format));
	}
	return TRUE;
}

static void msync_list_plugins(OSyncPluginEnv *env)
{
	int i;
	OSyncPlugin *plugin;
	
	osync_trace(TRACE_ENTRY, "%s(%p)", __func__, env);
	
	printf("Available plugins:\n");
	
	for (i = 0; i < osync_plugin_env_num_plugins(env); i++) {
		plugin = osync_plugin_env_nth_plugin(env, i);
		printf("%s\n", osync_plugin_get_name(plugin));
	}
	osync_trace(TRACE_EXIT, "%s", __func__);
}

static void msync_version(void)
{
	printf("This is msynctool version \"%s\"\n", MSYNCTOOL_VERSION);
	printf("using OpenSync version \"%s\"\n", osync_get_version());
}

static void msync_show_groups(OSyncGroupEnv *env)
{
	int i;
	OSyncGroup *group;
	
	printf("Available groups:\n");
	
	for (i = 0; i < osync_group_env_num_groups(env); i++) {
		group = osync_group_env_nth_group(env, i);
		printf("%s\n", osync_group_get_name(group));
	}
}

static const char *OSyncChangeType2String(OSyncChangeType type)
{
	switch (type) {
		case OSYNC_CHANGE_TYPE_ADDED: return "ADDED";
		case OSYNC_CHANGE_TYPE_UNMODIFIED: return "UNMODIFIED";
		case OSYNC_CHANGE_TYPE_DELETED: return "DELETED";
		case OSYNC_CHANGE_TYPE_MODIFIED: return "MODIFIED";
		case OSYNC_CHANGE_TYPE_UNKNOWN: return "?";
	}
	return NULL;
}

static void member_status(OSyncMemberUpdate *status, void *user_data)
{
	osync_trace(TRACE_ENTRY, "%s(%p (%i), %p)", __func__, status, status->type, user_data);
	
	char *sink = NULL;
	if (status->objtype == NULL) {
		sink = g_strdup("Main sink");
	} else {
		sink = g_strdup_printf("%s sink", status->objtype);
	}
	
	switch (status->type) {
		case OSYNC_CLIENT_EVENT_CONNECTED:
			printf("%s of member %lli of type %s just connected\n", sink, osync_member_get_id(status->member), osync_member_get_pluginname(status->member));
			break;
		case OSYNC_CLIENT_EVENT_DISCONNECTED:
			printf("%s of member %lli of type %s just disconnected\n", sink, osync_member_get_id(status->member), osync_member_get_pluginname(status->member));
			break;
		case OSYNC_CLIENT_EVENT_READ:
			printf("%s of member %lli of type %s just sent all changes\n", sink, osync_member_get_id(status->member), osync_member_get_pluginname(status->member));
			break;
		case OSYNC_CLIENT_EVENT_WRITTEN:
			printf("%s of member %lli of type %s committed all changes.\n", sink, osync_member_get_id(status->member), osync_member_get_pluginname(status->member));
			break;
		case OSYNC_CLIENT_EVENT_SYNC_DONE:
			printf("%s of member %lli of type %s reported sync done.\n", sink, osync_member_get_id(status->member), osync_member_get_pluginname(status->member));
			break;
		case OSYNC_CLIENT_EVENT_DISCOVERED:
			printf("%s of member %lli of type %s discovered its objtypes.\n", sink, osync_member_get_id(status->member), osync_member_get_pluginname(status->member));
			break;
		case OSYNC_CLIENT_EVENT_ERROR:
			printf("%s of member %lli of type %s had an error: %s\n", sink, osync_member_get_id(status->member), osync_member_get_pluginname(status->member), osync_error_print(&(status->error)));
			break;
	}
	
	g_free(sink);
	
	osync_trace(TRACE_EXIT, "%s", __func__);
}

static void entry_status(OSyncChangeUpdate *status, void *user_data)
{
	osync_trace(TRACE_ENTRY, "%s(%p (%i), %p)", __func__, status, status->type, user_data);
	
	switch (status->type) {
		case OSYNC_CHANGE_EVENT_READ:
			printf("Received an entry %s from member %lli (%s). Changetype %s\n",
					osync_change_get_uid(status->change),
					osync_member_get_id(status->member),
					osync_member_get_pluginname(status->member),
					OSyncChangeType2String(osync_change_get_changetype(status->change)));
			break;
		case OSYNC_CHANGE_EVENT_WRITTEN:
			printf("Sent an entry %s to member %lli (%s). Changetype %s\n",
					osync_change_get_uid(status->change),
					osync_member_get_id(status->member),
					osync_member_get_pluginname(status->member),
					OSyncChangeType2String(osync_change_get_changetype(status->change)));
			break;
		case OSYNC_CHANGE_EVENT_ERROR:
			printf("Error for entry %s and member %lli (%s): %s\n",
					osync_change_get_uid(status->change),
					osync_member_get_id(status->member),
					osync_member_get_pluginname(status->member),
					osync_error_print(&(status->error)));
			break;
	}
	
	osync_trace(TRACE_EXIT, "%s", __func__);
}

static void mapping_status(OSyncMappingUpdate *status, void *user_data)
{
	switch (status->type) {
		case OSYNC_MAPPING_EVENT_SOLVED:
			/*printf("Mapping solved\n");*/
			break;
		/*case OSYNC_MAPPING_EVENT_WRITTEN:
			printf("Mapping Synced\n");
			break;*/
		case OSYNC_MAPPING_EVENT_ERROR:
			printf("Mapping Error: %s\n", osync_error_print(&(status->error)));
			break;
	}
}

static void engine_status(OSyncEngineUpdate *status, void *user_data)
{
	switch (status->type) {
		case OSYNC_ENGINE_EVENT_CONNECTED:
			printf("All clients connected or error\n");
			break;
		case OSYNC_ENGINE_EVENT_READ:
			printf("All clients sent changes or error\n");
			break;
		case OSYNC_ENGINE_EVENT_WRITTEN:
			printf("All clients have written\n");
			break;
		case OSYNC_ENGINE_EVENT_DISCONNECTED:
			printf("All clients have disconnected\n");
			break;
		case OSYNC_ENGINE_EVENT_ERROR:
			printf("The sync failed: %s\n", osync_error_print(&(status->error)));
			break;
		case OSYNC_ENGINE_EVENT_SUCCESSFUL:
			printf("The sync was successful\n");
			break;
		case OSYNC_ENGINE_EVENT_PREV_UNCLEAN:
			printf("The previous synchronization was unclean. Slow-syncing\n");
			break;
		case OSYNC_ENGINE_EVENT_END_CONFLICTS:
			printf("All conflicts have been reported\n");
			break;
		case OSYNC_ENGINE_EVENT_SYNC_DONE:
			printf("All clients reported sync done\n");
			break;
	}
}

static void conflict_handler(OSyncEngine *engine, OSyncMappingEngine *mapping, void *user_data)
{
	OSyncError	*error = NULL;
	OSyncConflictResolution res = OSYNC_CONFLICT_RESOLUTION_UNKNOWN;
	OSyncChange *change = NULL;
	
	printf("Conflict for Mapping %p: ", mapping);
	fflush(stdout);
	
	if (conflict != OSYNC_CONFLICT_RESOLUTION_UNKNOWN)
		res = conflict;
	
	/* Check if the original value for the winning side was valid : memberid start from 1 thus winner + 1 */
	if (conflict == OSYNC_CONFLICT_RESOLUTION_SELECT) {
		if (!osync_mapping_engine_member_change(mapping, winner + 1)) {
			printf("Unable to find change #%i\n", winner + 1);
			res = OSYNC_CONFLICT_RESOLUTION_UNKNOWN;
		}
	}

	osync_bool supports_ignore = osync_mapping_engine_supports_ignore(mapping);
	osync_bool supports_use_latest = osync_mapping_engine_supports_use_latest(mapping);

	if (res == OSYNC_CONFLICT_RESOLUTION_UNKNOWN) {
		int i = 0;
		for (i = 0; i < osync_mapping_engine_num_changes(mapping); i++) {
			OSyncChange *change = osync_mapping_engine_nth_change(mapping, i);
			if (osync_change_get_changetype(change) != OSYNC_CHANGE_TYPE_UNKNOWN) {
				OSyncMember *member = osync_mapping_engine_change_find_member(mapping, change);
				OSyncData *data = osync_change_get_data(change);
				char *printable = osync_data_get_printable(data);
				printf("\nEntry %i:\nMember: %lli (%s)\nUID: %s\n%s\n", i+1, 
						osync_member_get_id(member), 
						osync_member_get_pluginname(member),
						osync_change_get_uid(change), 
						printable);
				g_free(printable);
			}
		}
		
		while (res == OSYNC_CONFLICT_RESOLUTION_UNKNOWN) {
			printf("\nWhich entry do you want to use? [1-9] To select a side, [D]uplicate");
			
			if (supports_ignore)
				printf(", [I]gnore");
			
			if (supports_use_latest)
				printf(", Keep [N]ewer");
			
			printf(": ");
			fflush(stdout);
			int inp = getchar();
			while (inp != '\n' && getchar() != '\n');
			
			if (inp == 'D' || inp == 'd')
				res = OSYNC_CONFLICT_RESOLUTION_DUPLICATE;
			else if (supports_ignore && (inp == 'i' || inp == 'I'))
				res = OSYNC_CONFLICT_RESOLUTION_IGNORE;
			else if (supports_use_latest && (inp == 'n' || inp == 'N'))
				res = OSYNC_CONFLICT_RESOLUTION_NEWER;
			else if (strchr("123456789", inp) != NULL) {
				char inpbuf[2];
				inpbuf[0] = inp;
				inpbuf[1] = 0;
				
				winner = atoi(inpbuf) - 1;
				if (winner >= 0) {
					if (!osync_mapping_engine_nth_change(mapping, winner))
						printf("Unable to find change #%i\n", winner + 1);
					else
						res = OSYNC_CONFLICT_RESOLUTION_SELECT;
				}
			}
		}
	}
		
	/* Did we get default conflict resolution ? */
	switch (res) {
		case OSYNC_CONFLICT_RESOLUTION_DUPLICATE:
			printf("Mapping duplicated\n");
			if (!osync_engine_mapping_duplicate(engine, mapping, &error))
				goto error;
			return;
		case OSYNC_CONFLICT_RESOLUTION_IGNORE:
			printf("Conflict ignored\n");
			if (!osync_engine_mapping_ignore_conflict(engine, mapping, &error))
				goto error;
			return;
		case OSYNC_CONFLICT_RESOLUTION_NEWER:
			printf("Newest entry used\n");
			if (!osync_engine_mapping_use_latest(engine, mapping, &error))
				goto error;
			return;
		case OSYNC_CONFLICT_RESOLUTION_SELECT:
			printf("Solving conflict\n");
			
			change = osync_mapping_engine_nth_change(mapping, winner);
				
			g_assert(change);
			if (!osync_engine_mapping_solve(engine, mapping, change, &error))
				goto error;
			return;
		case OSYNC_CONFLICT_RESOLUTION_UNKNOWN:
			g_assert_not_reached();
	}
	
	return;

error:
	printf("Conflict not resolved: %s\n", osync_error_print(&error));
	osync_error_unref(&error);
}

static osync_bool msync_synchronize(OSyncGroupEnv *env, char *groupname, osync_bool wait, osync_bool multi, GList *objtypes, OSyncError **error)
{
	osync_trace(TRACE_ENTRY, "%s(%p, %s, %i, %i, %p, %p)", __func__, env, groupname, wait, multi, objtypes, error);
	
	OSyncGroup *group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		return FALSE;
	}
	
	printf("Synchronizing group \"%s\" %s\n", osync_group_get_name(group), objtypes ? "[slow sync]" : "");

	OSyncEngine *engine = osync_engine_new(group, error);
	if (!engine)
		goto error;
	
	osync_engine_set_conflict_callback(engine, conflict_handler, NULL);
	osync_engine_set_changestatus_callback(engine, entry_status, NULL);
	osync_engine_set_mappingstatus_callback(engine, mapping_status, NULL);
	osync_engine_set_enginestatus_callback(engine, engine_status, NULL);
	osync_engine_set_memberstatus_callback(engine, member_status, NULL);
	
	/*osengine_set_memberstatus_callback(engine, member_status, NULL);
	osengine_set_changestatus_callback(engine, entry_status, NULL);
	osengine_set_enginestatus_callback(engine, engine_status, NULL);
	osengine_set_mappingstatus_callback(engine, mapping_status, NULL);
	osengine_set_conflict_callback(engine, conflict_handler, NULL);*/
	//osengine_set_message_callback(engine, plgmsg_function, "palm_uid_mismatch", NULL);
	//osengine_flag_only_info(engine);
	//if (manual)
	//	osengine_flag_manual(engine);
	
	if (!osync_engine_initialize(engine, error))
		goto error_free_engine;
	
	/* Set slowsync for requested object types */
	GList *o = NULL;
	for (o = objtypes; o; o = o->next) {
		char *objtype = (char *) o->data;
		/* osync_engine_find_objengine is only able to find
		   something if the engine is initialized! */
		OSyncObjEngine *objengine = osync_engine_find_objengine(engine, objtype);

		if (!objengine) {
			osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find Object Engine for object type  \"%s\"", objtype);
			goto error_finalize;
		}

		osync_obj_engine_set_slowsync(objengine, TRUE);
	}

	if (!osync_engine_synchronize_and_block(engine, error))
		goto error_finalize;
	
	/*if (!wait) {
		if (!manual) {
			if (!osengine_sync_and_block(engine, &error)) {
				printf("Error synchronizing: %s\n", osync_error_print(&error));
				osync_error_free(&error);
				return;
			}
		} else {
			if (!osengine_synchronize(engine, &error)) {
				printf("Error while starting synchronization: %s\n", osync_error_print(&error));
				osync_error_free(&error);
				return;
			}
			char buf[1024];
			while (1) {
				if (fgets(buf, sizeof(buf), stdin) == NULL)
					break;
				printf("+++++++++++++++++++++++++++\nNew Engine iteration:\n");
				osengine_one_iteration(engine);
				osengine_print_all(engine);
			}
		}
	} else {
		do {
			osengine_wait_sync_end(engine, NULL);
		} while (multi);
	}*/

	
	if (!osync_engine_finalize(engine, error))
		goto error_free_engine;
	
	osync_engine_unref(engine);
	
	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_finalize:
	osync_engine_finalize(engine, NULL);
error_free_engine:
	osync_engine_unref(engine);
error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;
}

static osync_bool msync_show_group(OSyncGroupEnv *env, const char *groupname, OSyncError **error)
{
	int i;
	OSyncMember *member = NULL;
	
	OSyncGroup *group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		return FALSE;
	}
	
	printf("Group: %s\n", osync_group_get_name(group));
	
	for (i = 0; i < osync_group_num_members(group); i++) {
		member = osync_group_nth_member(group, i);
		printf("Member %lli: %s\n", osync_member_get_id(member), osync_member_get_pluginname(member));

		/*
		 * Print the current configuration
		 */
		OSyncPluginConfig *config = osync_member_get_config(member, error);
		if (!config) {
			printf("\tNo Configuration found: %s\n", osync_error_print(error));
			osync_error_unref(error);
		} else {
			/* TODO: implement print funciton... print fields of OSyncPluginConfig.
			printf("\tConfiguration : %s\n", config);
			*/
		}
	}
	return TRUE;
}

static osync_bool msync_show_objtypes(OSyncGroupEnv *env, const char *groupname, OSyncError **error)
{
	int i;
	OSyncMember *member = NULL;
	
	OSyncGroup *group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		return FALSE;
	}
	
	for (i = 0; i < osync_group_num_members(group); i++) {
		member = osync_group_nth_member(group, i);
		int num = osync_member_num_objtypes(member);
		if (num == 0) {
			printf("Member %lli has no objtypes. Has it already been discovered?\n", osync_member_get_id(member));
		} else {
			printf("Member %lli Objtypes:\n", osync_member_get_id(member));
			
			int n = 0;
			for (n = 0; n < osync_member_num_objtypes(member); n++) {
				const char *objtype = osync_member_nth_objtype(member, n);
				printf("\tObjtype %s: %s\n", objtype, osync_member_objtype_enabled(member, objtype) ? "Enabled" : "Disabled");
				const OSyncList *formats = osync_member_get_objformats(member, objtype, error);
				if (!formats) {
					printf("\t\tNo formats found: %s\n", osync_error_print(error));
					osync_error_unref(error);
				}
				
				for (; formats; formats = formats->next) {
					OSyncObjFormatSink *formatsink = formats->data;
					const char * conversionconfig = osync_objformat_sink_get_config(formatsink);
					if (conversionconfig != NULL) {
						printf("\t\t\t\t conversion config: %s\n", conversionconfig);
					}
					else {
						printf("\t\t\t\t conversion config: not configured\n");
					}
				}
			}
		}
	}
	
	int max = osync_group_num_objtypes(group);
	if (max == 0) {
		printf("Group has no objtypes. Have the objtypes already been discovered?\n");
	} else {
		printf("Objtypes for the group:\n");
		for (i = 0; i < max; i++) {
			const char *objtype = osync_group_nth_objtype(group, i);
			printf("\t%s: %s\n", objtype, osync_group_objtype_enabled(group, objtype) ? "Enabled" : "Disabled");
		}
	}
	
	return TRUE;
}

static osync_bool msync_show_filter(OSyncGroupEnv *env, const char *groupname, OSyncError **error)
{
	return TRUE;
}

static osync_bool msync_show_capabilities(OSyncGroupEnv *env, const char *groupname, OSyncError **error)
{
	int i, size;
	char *data = NULL;
	OSyncMember *member = NULL;
	
	OSyncGroup *group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		return FALSE;
	}
	
	printf("Group: %s\n", osync_group_get_name(group));
	
	for (i = 0; i < osync_group_num_members(group); i++) {
		member = osync_group_nth_member(group, i);
		printf("Member %lli: %s\n", osync_member_get_id(member), osync_member_get_pluginname(member));

		OSyncCapabilities *capabilities = osync_member_get_capabilities(member);
		if(!capabilities) {
			printf("No capabilities found.\n");
			continue;
		}
		
		osync_capabilities_assemble(capabilities, &data, &size);
		printf("\n%s\n", data);
		g_free(data);
	}
	return TRUE;
}

static osync_bool msync_add_group(OSyncGroupEnv *env, char *groupname, OSyncError **error)
{
	OSyncGroup *group = osync_group_new(error);
	if (!group)
		goto error;
	
	osync_group_set_name(group, groupname);
	if (!osync_group_env_add_group(env, group, error))
		goto error_and_free;
	
	if (!osync_group_save(group, error))
		goto error_and_free;
	
	osync_group_unref(group);
	return TRUE;

error_and_free:
	osync_group_unref(group);
error:
	return FALSE;
}

static osync_bool msync_del_group(OSyncGroupEnv *env, char *groupname, OSyncError **error)
{
	OSyncGroup *group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		return FALSE;
	}
	
	if (!osync_group_delete(group, error))
		return FALSE;
	
	osync_group_env_remove_group(env, group);
	
	return TRUE;
}

static osync_bool msync_add_member(OSyncGroupEnv *env, OSyncPluginEnv *plugin_env, const char *groupname, const char *pluginname, OSyncError **error)
{
	OSyncMember *member = NULL;
	OSyncPlugin *plugin = NULL;
	OSyncGroup *group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		return FALSE;
	}
	
	plugin = osync_plugin_env_find_plugin(plugin_env, pluginname);
	if (!plugin) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find plugin with name %s", pluginname);
		return FALSE;
	}
	
	member = osync_member_new(error);
	if (!member) 
		return FALSE;
		
	osync_group_add_member(group, member);
	osync_member_set_pluginname(member, pluginname);
	
	if (!osync_member_save(member, error)) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to save member: %s", osync_error_print(error));
		return FALSE;
	}
	
	osync_member_unref(member);
	
	return TRUE;
}

static osync_bool msync_del_member(OSyncGroupEnv *env, const char *groupname, const char *memberid, OSyncError **error)
{
	long long id = 0;
	OSyncMember *member = NULL;
	
	OSyncGroup *group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		return FALSE;
	}
	
	id = atoi(memberid);
	member = osync_group_find_member(group, id);
	if (!member) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find member with id %s", memberid);
		return FALSE;
	}
	
	if (!osync_member_delete(member, error))
		return FALSE;
	
	osync_group_remove_member(group, member);

	return TRUE;
}

static osync_bool msync_enable_objtype(OSyncGroupEnv *env, const char *groupname, const char *objtype, osync_bool enable, OSyncError **error)
{
	OSyncGroup *group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		return FALSE;
	}
	
	osync_group_set_objtype_enabled(group, objtype, enable);
	
	if (!osync_group_save(group, error))
		return FALSE;
	
	return TRUE;
}

static osync_bool msync_configure_group_set_merger_enabled(OSyncGroupEnv *env, const char *groupname, osync_bool enable, OSyncError **error)
{
	OSyncGroup *group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		return FALSE;
	}
	
	osync_group_set_merger_enabled(group, enable);
	
	if (!osync_group_save(group, error))
		return FALSE;
	
	return TRUE;
}

static osync_bool msync_configure_group_set_converter_enabled(OSyncGroupEnv *env, const char *groupname, osync_bool enable, OSyncError **error)
{
	OSyncGroup *group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		return FALSE;
	}
	
	osync_group_set_converter_enabled(group, enable);
	
	if (!osync_group_save(group, error))
		return FALSE;
	
	return TRUE;
}

static osync_bool edit_data(const char* data, int size, char** edited_data, unsigned int* edited_size, OSyncError** error)
{
	int file = 0;
	char *tmpfile = NULL;
	char *editcmd = NULL;
	char *editor = NULL;
	
	tmpfile = g_strdup_printf("%s/msynctooltmp-XXXXXX", g_get_tmp_dir());
	file = g_mkstemp(tmpfile);
	if (!file) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to create temp file");
		goto error_free_tmp;
	}
	
	if (write(file, data, size) == -1) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to write to temp file: %i", errno);
		goto error_close_file;
	}
	
	if (close(file) == -1)  {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to close temp file: %i", errno);
		goto error_free_tmp;
	}
	
#ifdef _WIN32
	editcmd = g_strdup_printf("notepad %s", tmpfile);
#else
	if (!(editor = getenv("EDITOR")))
		editor = getenv("VISUAL");
	
	if (editor)
		editcmd = g_strdup_printf("%s %s", editor, tmpfile);
	else	
		editcmd = g_strdup_printf("vi %s", tmpfile);
#endif

	if (system(editcmd)) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to open editor. Aborting");
		g_free(editcmd);
		goto error_free_tmp;
	}

	g_free(editcmd);
	
	if (!osync_file_read(tmpfile, edited_data, edited_size, error))
		goto error_free_tmp;
	
	if (remove(tmpfile) != 0) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to remove file %s", tmpfile);
		g_free(edited_data);
		goto error_free_tmp;
	}
	
	g_free(tmpfile);
	
	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_close_file:
	close(file);
error_free_tmp:
	g_free(tmpfile);
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;	
}

static osync_bool edit_config(OSyncPluginConfig **config, OSyncError** error)
{
	int file = 0;
	char *tmpfile = NULL;
	char *editcmd = NULL;
	char *editor = NULL;
	
	tmpfile = g_strdup_printf("%s/msynctooltmp-XXXXXX", g_get_tmp_dir());
	/*	
	file = g_mkstemp(tmpfile);
	if (!file) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to create temp file");
		goto error_free_tmp;
	} */
	
	if (*config) {
		if(!osync_plugin_config_file_save(*config, tmpfile, error)) {
			osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to write to temp file: %i", errno);
			goto error_close_file;
		}
	}
	/*
	if (close(file) == -1)  {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to close temp file: %i", errno);
		goto error_free_tmp;
	} */
	
#ifdef _WIN32
	editcmd = g_strdup_printf("notepad %s", tmpfile);
#else
	if (!(editor = getenv("EDITOR")))
		editor = getenv("VISUAL");
	
	if (editor)
		editcmd = g_strdup_printf("%s %s", editor, tmpfile);
	else	
		editcmd = g_strdup_printf("vi %s", tmpfile);
#endif

	if (system(editcmd)) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to open editor. Aborting");
		g_free(editcmd);
		goto error_free_tmp;
	}

	g_free(editcmd);
	*config = osync_plugin_config_new(error); /* if a new config isn't created the loaded config is added to the old config */
	if (!osync_plugin_config_file_load(*config, tmpfile, NULL, error)) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to load config from tmpfile %s. Aborting", tmpfile);
		goto error_free_tmp;
	}
	
	if (remove(tmpfile) != 0) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to remove file %s", tmpfile);
		goto error_free_tmp;
	}
	
	g_free(tmpfile);
	
	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_close_file:
	close(file);
error_free_tmp:
	g_free(tmpfile);
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;	
}

static osync_bool msync_configure_member(OSyncGroupEnv *env, OSyncPluginEnv *pluginenv, const char *groupname, const char *memberid, OSyncError **error)
{
	long long id = 0;
	OSyncMember *member = NULL;
	OSyncGroup *group = NULL;
	OSyncPlugin *plugin = NULL;
	OSyncPluginConfig *config = NULL;
	OSyncConfigurationType type = OSYNC_PLUGIN_NO_CONFIGURATION;
	
	osync_trace(TRACE_ENTRY, "%s(%p, %p, %s, %s, %p)", __func__, env, pluginenv, groupname, memberid, error);
	
	group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		goto error;
	}
	
	id = atoi(memberid);
	member = osync_group_find_member(group, id);
	if (!member) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find member with id %s", memberid);
		goto error;
	}
	
	plugin = osync_plugin_env_find_plugin(pluginenv, osync_member_get_pluginname(member));
	if (!plugin) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find plugin with name %s", osync_member_get_pluginname(member));
		return FALSE;
	}
	
	type = osync_plugin_get_config_type(plugin);
	if (type == OSYNC_PLUGIN_NO_CONFIGURATION) {
		printf("This plugin has no options and does not need to be configured\n");
		osync_trace(TRACE_EXIT, "%s: No options", __func__);
		return TRUE;
	}
	
	config = osync_member_get_config_or_default(member, error);
	if (!config)
		goto error;

	if (!edit_config(&config, error))
		goto error;
	
	osync_member_set_config(member, config);
	
	if (!osync_member_save(member, error))
		goto error;

	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;
}

static osync_bool msync_configure_member_sink(OSyncGroupEnv *env, const char *groupname, const char *memberid, const char *objtype, osync_bool isEnable, OSyncError **error)
{
	long long id = 0;
	OSyncMember *member = NULL;
	OSyncGroup *group = NULL;
	
	osync_trace(TRACE_ENTRY, "%s(%p, %s, %s, %s, %i, %p)", __func__, env, groupname, memberid, objtype, isEnable, error);
	
	group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		goto error;
	}
	
	id = atoi(memberid);
	member = osync_group_find_member(group, id);
	if (!member) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find member with id %s", memberid);
		goto error;
	}

	/* If certain objtype is to be configured, just configure this one. */ 
	if (objtype) {
		OSyncObjTypeSink *sink = osync_member_find_objtype_sink(member, objtype);
		osync_objtype_sink_set_write(sink, isEnable);
	} else {
		/* Otherwise configure all sinks */
		int num = osync_member_num_objtypes(member);
		if (num == 0) {
			osync_error_set(error, OSYNC_ERROR_GENERIC, "Member %lli has no objtypes. Has it already been discovered?\n", osync_member_get_id(member));
			goto error;
		} else {
			int n = 0;
			for (n = 0; n < osync_member_num_objtypes(member); n++) {
				const char *objtype = osync_member_nth_objtype(member, n);
				OSyncObjTypeSink *sink = osync_member_find_objtype_sink(member, objtype);
				osync_objtype_sink_set_write(sink, isEnable);
			}
		}
	}
	
	if (!osync_member_save(member, error))
		goto error;

	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;
error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;
}

static osync_bool msync_configure_filter(OSyncGroupEnv *env, const char *groupname, OSyncError **error)
{
	return TRUE;
}

static osync_bool msync_configure_capabilities(OSyncGroupEnv *env, const char *groupname, const char *memberid, OSyncError **error)
{
	char *data = NULL;
	char *edited = NULL;
	int size = 0;
	long long id = 0;
	OSyncGroup *group = NULL;
	OSyncMember *member = NULL;

	osync_trace(TRACE_ENTRY, "%s(%p, %p, %s, %s, %p)", __func__, env, groupname, memberid, error);
	
	group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		goto error;
	}
	
	id = atoi(memberid);
	member = osync_group_find_member(group, id);
	if (!member) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find member with id %s", memberid);
		goto error;
	}
	
	OSyncCapabilities *capabilities = osync_member_get_capabilities(member);
	osync_bool rc;
	if(capabilities) {
		osync_capabilities_assemble(capabilities, &data, &size);
		rc = edit_data(data, strlen(data), &edited, (unsigned int *)&size, error);
		g_free(data);
	}else{
		rc = edit_data(NULL, 0, &edited, (unsigned int *)&size, error);
	}
	if(!rc)
		goto error;
	
	OSyncCapabilities *capabilitiesnew = osync_capabilities_parse(edited, size, error);
	g_free(edited);
	if(!capabilitiesnew)
		goto error;
	
	if(!osync_member_set_capabilities(member, capabilitiesnew, error))
		goto error;

	if (!osync_member_save(member, error))
		goto error;

	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;
}

static osync_bool msync_configure_discover_member(OSyncEngine *engine, OSyncMember *member, OSyncError **error)
{
	/* Discover the objtypes for the members */
	if (!osync_engine_discover_and_block(engine, member, error))
		return FALSE;

	int num = osync_member_num_objtypes(member);
	if (num == 0) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "discover failed: no objtypes returned");
		return FALSE;
	} else {
		printf("Discovered Objtypes:\n");
				
		int n = 0;
		for (n = 0; n < osync_member_num_objtypes(member); n++) {
			const char *objtype = osync_member_nth_objtype(member, n);
			printf("\t%s\n", objtype);
			const OSyncList *formats = osync_member_get_objformats(member, objtype, error);
			if (!formats) {
				printf("\t\tNo formats found: %s\n", osync_error_print(error));
				osync_error_unref(error);
			}
					
			for (; formats; formats = formats->next) {
				OSyncObjFormatSink *format_sink = formats->data;
				const char *objformat = osync_objformat_sink_get_objformat(format_sink);
				const char *config = osync_objformat_sink_get_config(format_sink);

				printf("\t\tFormat: %s\n", objformat);
				if (config)
					printf("\t\t\t\t conversion config: %s\n", config);
			}
		}
	}

	if (!osync_member_save(member, error))
		return FALSE;

	return TRUE;
}

static osync_bool msync_configure_discover(OSyncGroupEnv *env, OSyncPluginEnv *pluginenv, const char *groupname, const char *memberid, OSyncError **error)
{
	long long id = 0;
	OSyncMember *member = NULL;
	OSyncGroup *group = NULL;
	
	osync_trace(TRACE_ENTRY, "%s(%p, %p, %s, %s, %p)", __func__, env, pluginenv, groupname, memberid, error);
	
	group = osync_group_env_find_group(env, groupname);
	if (!group) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find group with name %s", groupname);
		goto error;
	}

	OSyncEngine *engine = osync_engine_new(group, error);
	if (!engine)
		goto error;
	
	if (memberid) {
		id = atoi(memberid);
		member = osync_group_find_member(group, id);
		if (!member) {
			osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find member with id %s", memberid);
			goto error;
		}

		if (!msync_configure_discover_member(engine, member, error))
			goto error_engine_finalize;
	} else {
		int i;
		for (i = 0; i < osync_group_num_members(group); i++) {
			member = osync_group_nth_member(group, i);
			if (!member) {
				osync_error_set(error, OSYNC_ERROR_GENERIC, "Unable to find member #%i", i);
				goto error;
			}

			if (!msync_configure_discover_member(engine, member, error))
				goto error_engine_finalize;
		}
	}

	//osync_engine_set_enginestatus_callback(engine, engine_status, NULL);
	//osync_engine_set_memberstatus_callback(engine, member_status, NULL);
	
	osync_engine_unref(engine);


	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_engine_finalize:
	osync_engine_finalize(engine, error);
	osync_engine_unref(engine);
error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;
}

int main (int argc, char *argv[])
{
	OSyncError *error = NULL;
	int i;
	char *groupname = NULL;
	char *membername = NULL;
	char *pluginname = NULL;
	ToolAction action = NONE;
	osync_bool wait = FALSE;
	osync_bool multi = FALSE;
	char *configdir = NULL;
	GList *slow_objects = NULL;
	char *objtype = NULL;
	OSyncGroupEnv *group_env = NULL;
	OSyncFormatEnv *format_env = NULL;
	OSyncPluginEnv *plugin_env = NULL;
	manual = FALSE;
	
	osync_trace(TRACE_ENTRY, "%s(%i, %p)", __func__, argc, argv);
	
	if (argc == 1)
		usage (argv[0], 1);
	
	for (i = 1; i < argc; i++) {
		char *arg = argv[i];
		if (!strcmp (arg, "--listplugins")) { //Install options
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_INSTALL_LISTPLUGINS;
		} else if (!strcmp (arg, "--listformats")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_INSTALL_LISTFORMATS;
		} else if (!strcmp (arg, "--version")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_INSTALL_GETVERSION;
		} else if (!strcmp (arg, "--listgroups")) { //Group info
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_SHOW_GROUPS;
		} else if (!strcmp (arg, "--showgroup")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_SHOW_GROUP;
			groupname = argv[i + 1];
			i++;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--showobjtypes")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_SHOW_OBJTYPES;
			groupname = argv[i + 1];
			i++;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--showfilter")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_SHOW_FILTER;
			groupname = argv[i + 1];
			i++;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--showcapabilities")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_SHOW_CAPABILITIES;
			groupname = argv[i + 1];
			i++;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--addgroup")) { //configure options
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_ADDGROUP;
			groupname = argv[i + 1];
			i += 1;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--delgroup")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_DELGROUP;
			groupname = argv[i + 1];
			i += 1;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--addmember")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_ADDMEMBER;
			groupname = argv[i + 1];
			pluginname = argv[i + 2];
			i += 2;
			if (!groupname || !pluginname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--configure")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE;
			groupname = argv[i + 1];
			membername = argv[i + 2];
			i += 2;
			if (!groupname || !membername)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--delmember")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_DELMEMBER;
			groupname = argv[i + 1];
			membername = argv[i + 2];
			i += 2;
			if (!groupname || !membername)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--enable-objtype")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_ENABLE_OBJTYPE;
			groupname = argv[i + 1];
			objtype = argv[i + 2];
			i += 2;
			if (!groupname || !objtype)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--disable-objtype")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_DISABLE_OBJTYPE;
			groupname = argv[i + 1];
			objtype = argv[i + 2];
			i += 2;
			if (!groupname || !objtype)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--enable-merger")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_ENABLE_MERGER;
			groupname = argv[i + 1];
			i += 1;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--disable-merger")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_DISABLE_MERGER;
			groupname = argv[i + 1];
			i += 1;
			if (!groupname)
				usage (argv[0], 1);		
		} else if (!strcmp (arg, "--enable-converter")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_ENABLE_CONVERTER;
			groupname = argv[i + 1];
			i += 1;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--disable-converter")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_DISABLE_CONVERTER;
			groupname = argv[i + 1];
			i += 1;
			if (!groupname)
				usage (argv[0], 1);	
		} else if (!strcmp (arg, "--configure-filter")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_FILTER;
			groupname = argv[i + 1];
			i += 1;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--configure-capabilities")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_CAPABILITIES;
			groupname = argv[i + 1];
			membername = argv[i + 2];
			i += 2;
			if (!groupname || !membername)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--disable-readonly")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_MEMBER_ENABLE_WRITE;
			groupname = argv[i + 1];
			membername = argv[i + 2];
			i += 2;
			if (!groupname || !membername)
				usage (argv[0], 1);

			if (argc > i) {
				objtype = argv[i + 1];
				i += 1;
			}
		} else if (!strcmp (arg, "--enable-readonly")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_MEMBER_DISABLE_WRITE;
			groupname = argv[i + 1];
			membername = argv[i + 2];
			i += 2;
			if (!groupname || !membername)
				usage (argv[0], 1);

			if (argc > i) {
				objtype = argv[i + 1];
				i += 1;
			}
		} else if (!strcmp (arg, "--discover")) {
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_CONFIGURE_DISCOVER;
			groupname = argv[i + 1];
			membername = argv[i + 2];
			i += 2;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--sync")) { //sync options
			if (action != NONE)
				usage (argv[0], 1);
			action = MSYNCTOOL_SYNC;			
			groupname = argv[i + 1];
			i += 1;
			if (!groupname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--slow-sync")) {
			objtype = argv[i + 1];
			slow_objects = g_list_append(slow_objects, objtype);
			i += 1;
			if (action != MSYNCTOOL_SYNC || !objtype)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--wait")) {
			wait = TRUE;
			if (action != MSYNCTOOL_SYNC)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--multi")) {
			multi = TRUE;
			if (action != MSYNCTOOL_SYNC)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--conflict")) {
			const char *conflictstr = argv[i + 1];
			i++;
			if (!conflictstr || action != MSYNCTOOL_SYNC)
				usage (argv[0], 1);
			
			if (conflictstr[0] == 'd' || conflictstr[0] == 'D')
				conflict = OSYNC_CONFLICT_RESOLUTION_DUPLICATE;
			else if (conflictstr[0] == 'i' || conflictstr[0] == 'I')
				conflict = OSYNC_CONFLICT_RESOLUTION_IGNORE;
			else if (conflictstr[0] == 'n' || conflictstr[0] == 'N')
				conflict = OSYNC_CONFLICT_RESOLUTION_NEWER;
			else if (strchr("123456789", conflictstr[0]) != NULL) {
				winner = atoi(conflictstr) - 1;
				if (winner < 0)
					usage (argv[0], 1);
				conflict = OSYNC_CONFLICT_RESOLUTION_SELECT;
			} else
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--configdir")) {
			configdir = argv[i + 1];
			i++;
			if (!configdir)
				usage (argv[0], 1);
		}  else if (!strcmp (arg, "--manual")) {
			manual = TRUE;
			if (action != MSYNCTOOL_SYNC)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--help")) {
			usage (argv[0], 0);
		} else if (!strcmp (arg, "--")) {
			break;
		} else if (arg[0] == '-') {
			usage (argv[0], 1);
		} else {
			usage (argv[0], 1);
		}
	}
	
	group_env = osync_group_env_new(&error);
	if (!group_env)
		goto error;
		
	format_env = osync_format_env_new(&error);
	if (!format_env)
		goto error_free_group_env;
		
	plugin_env = osync_plugin_env_new(&error);
	if (!plugin_env)
		goto error_free_format_env;
	
	switch (action) {
		case MSYNCTOOL_SHOW_GROUP:
		case MSYNCTOOL_SHOW_GROUPS:
		case MSYNCTOOL_CONFIGURE_DELMEMBER:
		case MSYNCTOOL_CONFIGURE_ADDGROUP:
		case MSYNCTOOL_CONFIGURE_DELGROUP:
		case MSYNCTOOL_SHOW_OBJTYPES:
		case MSYNCTOOL_SHOW_FILTER:
		case MSYNCTOOL_SHOW_CAPABILITIES:
		case MSYNCTOOL_CONFIGURE_ENABLE_OBJTYPE:
		case MSYNCTOOL_CONFIGURE_DISABLE_OBJTYPE:
		case MSYNCTOOL_CONFIGURE_ENABLE_MERGER:
		case MSYNCTOOL_CONFIGURE_DISABLE_MERGER:
		case MSYNCTOOL_CONFIGURE_ENABLE_CONVERTER:
		case MSYNCTOOL_CONFIGURE_DISABLE_CONVERTER:
		case MSYNCTOOL_CONFIGURE_MEMBER_ENABLE_WRITE:
		case MSYNCTOOL_CONFIGURE_MEMBER_DISABLE_WRITE:
		case MSYNCTOOL_CONFIGURE_FILTER:
			if (!osync_group_env_load_groups(group_env, configdir, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_INSTALL_LISTFORMATS:
			if (!osync_format_env_load_plugins(format_env, NULL, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_INSTALL_LISTPLUGINS:
			if (!osync_plugin_env_load(plugin_env, NULL, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_ADDMEMBER:
		case MSYNCTOOL_SYNC:
		case MSYNCTOOL_CONFIGURE:
		case MSYNCTOOL_CONFIGURE_CAPABILITIES:
		case MSYNCTOOL_CONFIGURE_DISCOVER:
			if (!osync_group_env_load_groups(group_env, configdir, &error))
				goto error_free_plugin_env;
				
			if (!osync_format_env_load_plugins(format_env, NULL, &error))
				goto error_free_plugin_env;
				
			if (!osync_plugin_env_load(plugin_env, NULL, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_INSTALL_GETVERSION:
			break;
		case NONE:
			osync_error_set(&error, OSYNC_ERROR_GENERIC, "No action given");
			goto error_free_plugin_env;
	}
	
	switch (action) {
		case MSYNCTOOL_INSTALL_LISTPLUGINS:
			msync_list_plugins(plugin_env);
			break;
		case MSYNCTOOL_INSTALL_LISTFORMATS:
			if (!msync_list_formats(format_env, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_INSTALL_GETVERSION:
			msync_version();
			break;
			
		case MSYNCTOOL_SHOW_GROUPS:
			msync_show_groups(group_env);
			break;
		case MSYNCTOOL_SHOW_GROUP:
			if (!msync_show_group(group_env, groupname, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_SHOW_OBJTYPES:
			if (!msync_show_objtypes(group_env, groupname, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_SHOW_FILTER:
			if (!msync_show_filter(group_env, groupname, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_SHOW_CAPABILITIES:
			if (!msync_show_capabilities(group_env, groupname, &error))
				goto error_free_plugin_env;	
			break;
		case MSYNCTOOL_CONFIGURE_ADDGROUP:
			if (!msync_add_group(group_env, groupname, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_DELGROUP:
			if (!msync_del_group(group_env, groupname, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_ADDMEMBER:
			if (!msync_add_member(group_env, plugin_env, groupname, pluginname, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE:
			if (!msync_configure_member(group_env, plugin_env, groupname, membername, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_DELMEMBER:
			if (!msync_del_member(group_env, groupname, membername, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_ENABLE_OBJTYPE:
			if (!msync_enable_objtype(group_env, groupname, objtype, TRUE, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_DISABLE_OBJTYPE:
			if (!msync_enable_objtype(group_env, groupname, objtype, FALSE, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_ENABLE_MERGER:
			if (!msync_configure_group_set_merger_enabled(group_env, groupname, TRUE, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_DISABLE_MERGER:
			if (!msync_configure_group_set_merger_enabled(group_env, groupname, FALSE, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_ENABLE_CONVERTER:
			if (!msync_configure_group_set_converter_enabled(group_env, groupname, TRUE, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_DISABLE_CONVERTER:
			if (!msync_configure_group_set_converter_enabled(group_env, groupname, FALSE, &error))
				goto error_free_plugin_env;
			break;	
		case MSYNCTOOL_CONFIGURE_FILTER:
			if (!msync_configure_filter(group_env, groupname, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_CAPABILITIES:
			if (!msync_configure_capabilities(group_env, groupname, membername, &error))
				goto error_free_plugin_env;
			break;	
		case MSYNCTOOL_CONFIGURE_DISCOVER:
			if (!msync_configure_discover(group_env, plugin_env, groupname, membername, &error))
				goto error_free_plugin_env;
			break;
		case MSYNCTOOL_CONFIGURE_MEMBER_ENABLE_WRITE:
			if  (!msync_configure_member_sink(group_env, groupname, membername, objtype, TRUE, &error))
				goto error_free_group_env;
			break;
		case MSYNCTOOL_CONFIGURE_MEMBER_DISABLE_WRITE:
			if  (!msync_configure_member_sink(group_env, groupname, membername, objtype, FALSE, &error))
				goto error_free_group_env;
			break;
		case MSYNCTOOL_SYNC:
			if (!msync_synchronize(group_env, groupname, wait, multi, slow_objects, &error)) {
				g_list_free(slow_objects);
				goto error_free_plugin_env;
			}
			break;
		case NONE:
			osync_error_set(&error, OSYNC_ERROR_GENERIC, "No action given");
			goto error_free_plugin_env;
	}

	g_list_free(slow_objects);
	osync_group_env_free(group_env);
	osync_format_env_free(format_env);
	osync_plugin_env_free(plugin_env);
	
	osync_trace(TRACE_EXIT, "%s", __func__);
	return 0;

error_free_plugin_env:
	osync_plugin_env_free(plugin_env);
error_free_format_env:
	osync_format_env_free(format_env);
error_free_group_env:
	osync_group_env_free(group_env);
error:
	fprintf(stderr, "ERROR: %s\n", osync_error_print(&error));
	osync_trace(TRACE_EXIT, "%s: %s", __func__, osync_error_print(&error));
	osync_error_unref(&error);
	return -1;
}
