# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Load/save config files.


# load libs
import ConfigParser
import os
import re
import sys
from copy import copy

# load modules
import defaults
from modules.mformula.main import mFormula


class mConfig:
    """Load and process INI files"""

    # ----
    def __init__ (self):

        # get current executing folder
        path = sys.path[0]
        if os.path.isfile(path):
            path = os.path.dirname(path)
        self.configPath = path + '/configs'

        # check config folder
        if not os.path.exists(self.configPath):
            try:
                os.mkdir(self.configPath)
            except:
                pass

        # load config files
        self.cfg = self.loadConfig()
        self.elem = self.loadElements()
        self.amino = self.loadAminoacids()
        self.mod = self.loadModifications()
        self.enz = self.loadEnzymes()
        self.ref = self.loadReferences()

        # count masses for formulas in cfg
        self.mFormula = mFormula(self.elem)
        self.amino = self.countMasses(self.amino)
        self.mod = self.countMasses(self.mod)

        # convert colours
        self.cfg['colours'] = self.convertColours(self.cfg['colours'])

        # remember main app defaults for preferences dialog
        self.cfg['ori'] = {}
        self.cfg['ori']['masstype'] = self.cfg['common']['masstype']
        self.cfg['ori']['errortype'] = self.cfg['common']['errortype']
        self.cfg['ori']['tolerance'] = self.cfg['common']['tolerance']
    # ----


    # ----
    def loadIni(self, filename):
        """ Load given ini file. """

        config = ConfigParser.ConfigParser()
        config.optionxform = unicode
        config.read(filename)
        data = {}
        for section in config.sections():
            data[section] = {}
            for option in config.options(section):
                value = config.get(section, option)
                data[section][option] = value.decode("utf-8")

        return data
    # ----


    # ----
    def loadConfig(self):
        """ Load main configuration file. """

        if os.path.exists(self.configPath+'/config.ini'):
            data = self.loadIni(self.configPath+'/config.ini')
            for section in data:
                for option in data[section]:
                    value = data[section][option]
                    if re.compile('^[0-9]+$').search(value):
                        data[section][option] = int(value)
                    elif re.compile('^[0-9]*[.]+[0-9]+$').search(value):
                        data[section][option] = float(value)
        else:
            data = defaults.config
            self.saveFile(data, 'cfg')

        return data
    # ----


    # ----
    def loadElements(self):
        """ Load elements' file. """

        if os.path.exists(self.configPath+'/elements.ini'):
            data = self.loadIni(self.configPath+'/elements.ini')
            for item in data:
                data[item]['mmass'] = float(data[item]['mmass'])
                data[item]['amass'] = float(data[item]['amass'])
        else:
            data = defaults.elements
            self.saveFile(data, 'elem')

        return data
    # ----


    # ----
    def loadAminoacids(self):
        """ Load amino acids' file. """

        if os.path.exists(self.configPath+'/aminoacids.ini'):
            data = self.loadIni(self.configPath+'/aminoacids.ini')
        else:
            data = defaults.aminoacids
            self.saveFile(data, 'amino')

        return data
    # ----


    # ----
    def loadModifications(self):
        """ Load modifications' file. """

        if os.path.exists(self.configPath+'/modifications.ini'):
            data = self.loadIni(self.configPath+'/modifications.ini')
            for item in data:
                data[item]['n-term'] = int(data[item]['n-term'])
                data[item]['c-term'] = int(data[item]['c-term'])
        else:
            data = defaults.modifications
            self.saveFile(data, 'mod')

        return data
    # ----


    # ----
    def loadEnzymes(self):
        """ Load enzymes' file. """

        if os.path.exists(self.configPath+'/enzymes.ini'):
            data = self.loadIni(self.configPath+'/enzymes.ini')
        else:
            data = defaults.enzymes
            self.saveFile(data, 'enz')

        return data
    # ----


    # ----
    def loadReferences(self):
        """ Load calibration references. """

        if os.path.exists(self.configPath+'/references.ini'):
            data = self.loadIni(self.configPath+'/references.ini')
        else:
            data = defaults.references
            self.saveFile(data, 'ref')

        return data
    # ----


    # ----
    def countMasses(self, data):
        """ Count masses for formulas in cfg files. """

        for item in data:
            mmass = self.mFormula.getMass(data[item]['formula'], 'mmass')
            amass = self.mFormula.getMass(data[item]['formula'], 'amass')
            data[item]['mmass'] = mmass
            data[item]['amass'] = amass

        return data
    # ----


    # ----
    def convertColours(self, data):
        """ Convert colours' string to tuple. """

        for item in data:
            tmp = data[item].split(',')
            data[item] = (int(tmp[0]), int(tmp[1]), int(tmp[2]))

        return data
    # ----


    # ----
    def saveCfgOnExit(self):
        """ Save main config file on application exit. """

        # revert some init data
        self.cfg['common']['errortype'] = self.cfg['ori']['errortype']
        self.cfg['common']['masstype'] = self.cfg['ori']['masstype']
        self.cfg['common']['tolerance'] = self.cfg['ori']['tolerance']

        # erase ori
        del self.cfg['ori']

        # revert colours to string
        for colour in self.cfg['colours']:
            self.cfg['colours'][colour] = '%d,%d,%d' % self.cfg['colours'][colour]

        # save data to file
        self.saveFile(self.cfg, 'cfg')
    # ----


    # ----
    def saveFile(self, data, dataType):
        """ Create configuration file with default parameters. """

        # make filename
        filename = ''
        if dataType == 'cfg':
            filename = '/config.ini'
            buff = '# application preferences'
        elif dataType == 'elem':
            filename = '/elements.ini'
            buff = '# list of elements'
        elif dataType == 'amino':
            filename = '/aminoacids.ini'
            buff = '# list of amino acids'
        elif dataType == 'mod':
            filename = '/modifications.ini'
            buff = '# list of modifications'
        elif dataType == 'enz':
            filename = '/enzymes.ini'
            buff = '# list of enzymes'
        elif dataType == 'ref':
            filename = '/references.ini'
            buff = '# list of calibration references'
        filename = self.configPath + filename

        # sort sections by name
        sections = []
        for section in data:
            sections.append(section)
        sections.sort()

        # paste data to buffer
        for section in sections:

            # sort option by name
            options = []
            for option in data[section]:
                options.append(option)
            options.sort()

            # add formated data
            buff += '\n\n[%s]' % section
            for option in options:
                buff += '\n%s=%s' % (option, data[section][option])

        # save file
        try:
            cfgFile = file(filename, 'w')
            cfgFile.write(buff.encode("utf-8"))
            cfgFile.close()
            return True
        except IOError:
            return False
    # ----
