# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Count masses from molecular formula.

# load libs
import re
import string


class mFormula:
    """Count masses from given molecular formula"""

    # ----
    def __init__(self, elementsCfg):

        # elements config
        self.elementsCfg = elementsCfg

        # define used patterns
        self.simplePattern = re.compile('^(([A-Z][a-z\*]?)([0-9]*))?$')
        self.complexPattern = re.compile('^(([A-Z][a-z\*]?[0-9]*)*)(-(([A-Z][a-z\*]?[0-9]*)*))?$')
        self.elementPattern = re.compile('^([A-Z][a-z\*]?)([0-9]*)$')
        self.splitSimplePattern = re.compile('([A-Z][a-z\*]?[0-9]*)')
    # ----


    # ----
    def getMass(self, formula, massType):
        """ Check molecular formula and count mass. """

        # switch formula type and get mass
        if not formula:
            return False
        elif self.simplePattern.match(formula):
            return self.countSimpleFormulaMass(formula, massType)
        elif self.complexPattern.match(formula):
            return self.countComplexFormulaMass(formula, massType)
        else:
            return False
    # ----


    # ----
    def countSimpleFormulaMass(self, formula, massType):
        """ Count mass for simple molecular formula. """

        # split formula to elements
        splited = self.splitSimplePattern.split(formula)

        # sum elements masses
        mass = 0
        for part in splited:
            count = 1
            if part != '':
                elements = self.elementPattern.match(part)
                element = elements.group(1)

                # get element count
                if elements.group(2) != '':
                    count = int(elements.group(2))

                # count mass for element
                try:
                    mass += self.elementsCfg[element][massType] * count
                except KeyError:
                    return False

        return mass
    # ----


    # ----
    def countComplexFormulaMass(self, formula, massType):
        """ Count molecular formula with plus and minus parts. """

        # split to 'plus' and 'minus' parts
        parts = self.complexPattern.match(formula)
        if parts == None:
            return False

        # count masses for 'plus' part
        plusMass = 0
        if parts.group(1):
            plusMass = self.countSimpleFormulaMass(parts.group(1), massType)
            if not plusMass:
                return False

        # count masses for 'minus' part
        minusMass = 0
        if parts.group(4):
            minusMass = self.countSimpleFormulaMass(parts.group(4), massType)
            if not minusMass:
                return False

        # sum masses
        return (plusMass - minusMass)
    # ----
