/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 1997-2000 by the German Cancer Research Center (Deutsches
 *   Krebsforschungszentrum, DKFZ Heidelberg) and Bastien Chevreux
 * Copyright (C) 2000 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */


#ifndef lint
static char vcid3[] = "$Id$";
#endif /* lint */

#include "contig.H"
#include "util/misc.H"


//#define CEBUGFLAG
#ifdef CEBUGFLAG
#define CEBUG(bla)   {cout << bla; cout.flush();}
#define CEBUGF(bla)  {cout << bla; cout.flush();}
#else
#define CEBUG(bla)
#define CEBUGF(bla)
#endif



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::stats(ostream &ostr)
{
  (void) vcid3;

  FUNCSTART("ostream & Contig::stats(ostream &ostr, Contig const  &con)");

  calcStats();

  // shouldn't be needed as setprecision() works only only once (Josuttis)
  //  but apparently gcc does not honour this
  streamsize oldprecision=cout.precision();

  if(CON_outtype==AS_HTML){
    ostr << "<h2>Statistics</h2><p>\n";
    ostr << "To be reworked!\n";
    //ostr << "Contig id: " << CON_id << "<br>" << '\n';
    //ostr << "Contig length: " << CON_counts.size() << "<br>" << '\n';
    //ostr << "Avg. contig coverage: " << CON_stats.avg_coverage << "<br>" << '\n';
    //
    //ostr << "Consensus contains: A: " << CON_stats.AinC << "\tC: " << CON_stats.CinC << "\tG: " << CON_stats.GinC << "\tT: " << CON_stats.TinC << "\tN: " << CON_stats.NinC << "\tIUPAC: " << CON_stats.IUPACinC << "\tFunny: " << CON_stats.FunnyInC << "\t*: " << CON_stats.starInC << "<p>" << "\n";
    //
    //
    //ostr << "\nNum reads: " << CON_stats.total_reads << "<br>" << '\n';
    //
    //ostr << "Avg. read length: " << CON_stats.avg_readlen << "<br>" << '\n';
    //
    //ostr << "Reads contain " << CON_stats.total_readlen-CON_stats.starInR-CON_stats.NinR << " bases, " << CON_stats.NinR << " Ns and " << CON_stats.starInR << " gaps." << "<p>" << "\n";
    //
    //ostr << "<p>\n";
  }else{
    ostr << "\n-------------- Contig statistics ----------------\n";
    ostr << "Contig id: " << CON_id << '\n';;
    
    ostr << "Contig length: " << CON_counts.size() << "\n\n";

    ostr << "\t\t";
    for(uint32 i=0;i<Read::SEQTYPE_END; i++){
#ifdef HIDEPACBIO
      if(i==Read::SEQTYPE_PACBIO) continue;
#endif
      ostr << setw(12) << Read::getNameOfSequencingType(i);
    }
    ostr << "\nNum. reads\t";
    for(uint32 i=0;i<Read::SEQTYPE_END; i++){
#ifdef HIDEPACBIO
      if(i==Read::SEQTYPE_PACBIO) continue;
#endif
      ostr << setw(12) << CON_stats.readsperst[i];
    }

    ostr << "\n100% merged reads\t"
	 << setw(4) << "-" << setw(12) << "-" << setw(12) << "-" << setw(12) << "-"
	 << setw(12) << CON_nummergedreads_perseqtype[0]
	 << setw(12) << CON_nummergedreads_perseqtype[1];

    ostr << "\nAvg. read len\t";
    for(uint32 i=0;i<Read::SEQTYPE_END; i++){
#ifdef HIDEPACBIO
      if(i==Read::SEQTYPE_PACBIO) continue;
#endif
      if(CON_stats.readsperst[i]>0){
	if(i==Read::SEQTYPE_SOLEXA){
	  ostr << setw(12) << CON_stats.readlenperst[i]/(CON_stats.readsperst[i]+CON_nummergedreads_perseqtype[0]);
	}else if(i==Read::SEQTYPE_ABISOLID){
	  ostr << setw(12) << CON_stats.readlenperst[i]/(CON_stats.readsperst[i]+CON_nummergedreads_perseqtype[1]);
	}else{
	  ostr << setw(12) << CON_stats.readlenperst[i]/CON_stats.readsperst[i];
	}
      }else{
	ostr << setw(12) << 0;
      }
    }

    ostr << "\nMax. coverage\t";
    for(uint32 i=0;i<Read::SEQTYPE_END; i++){
#ifdef HIDEPACBIO
      if(i==Read::SEQTYPE_PACBIO) continue;
#endif
      ostr << setw(12) << CON_stats.max_covperst[i];
    }

    ostr << "\nAvg. coverage\t";
    for(uint32 i=0;i<Read::SEQTYPE_END; i++){
#ifdef HIDEPACBIO
      if(i==Read::SEQTYPE_PACBIO) continue;
#endif
      ostr << setw(12) << setprecision(3) << CON_stats.avg_covperst[i];
    }

    ostr << "\n\nMax. contig coverage: " << CON_stats.max_coverage << '\n';
    ostr << "Avg. contig coverage: " << setprecision(3) << CON_stats.avg_coverage << '\n';
    
    ostr << "\nConsensus contains:\tA: " << CON_stats.AinC << "\tC: " << CON_stats.CinC << "\tG: " << CON_stats.GinC << "\tT: " << CON_stats.TinC << "\tN: " << CON_stats.NinC << "\n\t\t\tIUPAC: " << CON_stats.IUPACinC << "\tFunny: " << CON_stats.FunnyInC << "\t*: " << CON_stats.starInC << "\n";
    
    ostr << "GC content: ";
    if(CON_stats.CinC+CON_stats.GinC>0){
      ostr << 100.0/(CON_stats.AinC+CON_stats.CinC+CON_stats.GinC+CON_stats.TinC)*(CON_stats.CinC+CON_stats.GinC) << "%\n";
    }else{
      ostr << "0%\n";
    }
    
    //ostr << "\nReads contain " << CON_stats.total_readlen-CON_stats.starInR-CON_stats.NinR << " bases, " << CON_stats.NinR << " Ns and " << CON_stats.starInR << " gaps.\n";
    
    ostr << "-------------------------------------------------\n";
  }

  cout.precision(oldprecision);

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::setCoutType(uint8 type)
{
  switch(type){
  case AS_TEXT: 
  case AS_DEBUG: 
  case AS_FASTA: 
  case AS_FASTAQUAL: 
  case AS_FASTAPADDED: 
  case AS_FASTAPADDEDQUAL: 
  case AS_HTML: 
  case AS_CAF: 
  case AS_MAF: 
  case AS_SAM:
  case AS_ACE: 
  case AS_TCS: 
  case AS_GAP4DA: {
    CON_outtype=type;
    break;
  }
  default:{
    throw Notify(Notify::INTERNAL, "void Contig::setCoutType(uint8 type)", "Wrong type is not one of TEXT, HTML, CAF. ACE or GAP4DA.");
  }
  }

  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

ostream & operator<<(ostream &ostr, Contig const  &con)
{
  FUNCSTART("friend ostream & Contig::operator<<(ostream &ostr, Contig const  &con)");

  Contig & nonconstcon = const_cast<Contig &>(con);

  //nonconstcon.definalise();
  nonconstcon.finalise();

  if(con.CON_valid==0){
    throw Notify(Notify::INTERNAL, THISFUNC, "Object not valid.");
  }
  if(con.CON_valid!=CON_VALID){
    throw Notify(Notify::INTERNAL, THISFUNC, "Object probably trashed.");
  }

  if(con.CON_outtype==Contig::AS_DEBUG){
    nonconstcon.dumpAsDebug(ostr);
    FUNCEND();
    return ostr;
  }
  if(con.CON_outtype==Contig::AS_CAF){
    nonconstcon.dumpAsCAF(ostr);
    FUNCEND();
    return ostr;
  }
  if(con.CON_outtype==Contig::AS_MAF){
    nonconstcon.dumpAsMAF(ostr);
    FUNCEND();
    return ostr;
  }
  if(con.CON_outtype==Contig::AS_ACE){
    nonconstcon.dumpAsACE(ostr);
    FUNCEND();
    return ostr;
  }
  if(con.CON_outtype==Contig::AS_TCS){
    nonconstcon.dumpAsTCS(ostr);
    FUNCEND();
    return ostr;
  }
  if(con.CON_outtype==Contig::AS_FASTA){
    nonconstcon.dumpAsFASTA(ostr,false);
    FUNCEND();
    return ostr;
  }
  if(con.CON_outtype==Contig::AS_FASTAQUAL){
    nonconstcon.dumpAsFASTAQUAL(ostr,false);
    FUNCEND();
    return ostr;
  }
  if(con.CON_outtype==Contig::AS_FASTAPADDED){
    nonconstcon.dumpAsFASTA(ostr,true);
    FUNCEND();
    return ostr;
  }
  if(con.CON_outtype==Contig::AS_FASTAPADDEDQUAL){
    nonconstcon.dumpAsFASTAQUAL(ostr,true);
    FUNCEND();
    return ostr;
  }


  if(con.CON_outtype==Contig::AS_HTML){
    ostr << "<a NAME=\"" << con.getContigName() << "\"></a>\n";
    ostr << "<h1><center>" << con.getContigName() << "</center></h1>\n";
  }

  nonconstcon.stats(ostr);

  string consseq;
  vector<base_quality_t> consqual;
  nonconstcon.newConsensusGet(consseq, consqual);

  // TODO: ??? why ???
  //for(uint32 i=0; i<consseq.size(); i++){
  //  nonconstcon.CON_tmpcons[i]=consseq[i];
  //}


  if(con.CON_finalised==true){
    nonconstcon.dumpAsTextOrHTML(ostr, con.CON_outtype, consseq, consqual, 0, nonconstcon.getContigLength());
  }else{
    ostr << "Consensus not finalised, no more information to output.\n";
  }

  FUNCEND();

  return ostr;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpAsText(ostream &ostr, int32 frompos, int32 topos)
{
  string consseq;
  vector<base_quality_t> consqual;
  newConsensusGet(consseq, consqual);
  dumpAsTextOrHTML(ostr, Contig::AS_TEXT, consseq, consqual, frompos, topos);
}

void Contig::dumpAsHTML(ostream &ostr, int32 frompos, int32 topos)
{
  string consseq;
  vector<base_quality_t> consqual;
  newConsensusGet(consseq, consqual);
  dumpAsTextOrHTML(ostr, Contig::AS_HTML, consseq, consqual, frompos, topos);
}

void Contig::dumpAsTextOrHTML(ostream &ostr, const uint8 outtype, const string & consseq, const vector<base_quality_t> & consqual, int32 frompos, int32 topos)
{
  FUNCSTART("void Contig::dumpAsTextOrHTML(ostream &ostr, const uint8 outtype, const string & consseq, const vector<base_quality_t> & consqual, int32 frompos, int32 topos)");

  (void) consqual;

  if(frompos<0 || frompos > static_cast<int32>(CON_counts.size())) frompos=0;
  if(topos<0 || topos > static_cast<int32>(CON_counts.size())) topos=static_cast<int32>(CON_counts.size());

  if(outtype==Contig::AS_TEXT){
    ostr << "Sequence:\n";
  }else{
    ostr << "<h2>Sequence:</h2><p>\n";
  }

  const contig_parameters & con_params = (*(CON_miraparams))[0].getContigParams();

  int32 cpl=con_params.con_output_text_cpl;
  char egfillchar=con_params.con_output_text_gapfill;
  if(outtype==Contig::AS_HTML){
    cpl=con_params.con_output_html_cpl;
    egfillchar=con_params.con_output_html_gapfill;
  }
  if(cpl>topos-frompos) cpl=topos-frompos;
  if(egfillchar=='\t') egfillchar=' ';

  size_t longestnamesize=0;
  for(uint32 ri=0; ri<CON_reads.size(); ri++){
    longestnamesize=max(longestnamesize,CON_reads[ri].read.getName().size());
  }
  ++longestnamesize;

  BUGIFTHROW(CON_reads.size() != CON_outputorder.size(),"CON_reads size != CON_outputorder size?");

  string spanid; // for HTML output

  for(int32 conpos=frompos; conpos < topos; conpos+=cpl){
    if(outtype==Contig::AS_HTML) ostr << "<table CELLSPACING=0 CELLPADDING=0 NOSAVE >\n";
    if(outtype==Contig::AS_HTML){
      ostr << "<tr NOSAVE>\n<td NOSAVE><div align=right>";
    }
    if(outtype==Contig::AS_TEXT){
      ostr << right << setw(longestnamesize);
    }
    ostr << conpos << ' ';
    if(outtype==Contig::AS_HTML) ostr << "</div></td>\n";
    CEBUG("\t\t\t\t\t\t");
    if(outtype==Contig::AS_TEXT){
      //ostr << "\t\t\t";
      ostr << ' ';
      for(int32 i=0; i<((cpl-1)/10)+1; i++) {
	ostr << "|    .    ";
      }
      ostr << '\n';
    }else if(outtype==Contig::AS_HTML){
      ostr << "<td><tt>";
      for(int32 i=0; i<((cpl-1)/10)+1; i++) {
	ostr << "|&nbsp;&nbsp;&nbsp;&nbsp;.&nbsp;&nbsp;&nbsp;&nbsp;";
      }
      ostr << "</tt></td>\n</tr>\n";
    }

    for(uint32 i=0; i<CON_reads.size(); i++){
      uint32 origindex=CON_outputorder[i].original_index;
      if(!CON_outputRails && CON_reads[origindex].read.isRail()) continue;

      Read & actread=CON_reads[origindex].read;

      if((conpos >= CON_outputorder[i].offset_start
	  || conpos+cpl >= CON_outputorder[i].offset_start)
	 && conpos<static_cast<int32>(CON_outputorder[i].offset_start+actread.getLenClippedSeq())){
	       
#ifdef CEBUGFLAG
	ostr << "offset: " << CON_outputorder[i].offset_start;
	ostr << "\tsize: " << actread.getLenClippedSeq();
	ostr << "\torpid: " << CON_outputorder[i].orpid;
	ostr << "\t";
#endif 

	if(outtype==Contig::AS_HTML) {
	  ostr << "<tr><td>";
	}else if(outtype==Contig::AS_TEXT){
	  ostr << left << setw(longestnamesize);
	}
	ostr << actread.getName();

	vector<char>::const_iterator oI;
	if(CON_reads[origindex].direction>=0){
	  ostr << '+';
	  //Contig & nonconstcon = const_cast<Contig &>(con);
	  oI=actread.getClippedSeqIterator();
	}else{
	  ostr << '-';
	  //Contig & nonconstcon = const_cast<Contig &>(con);
	  oI=actread.getClippedComplementSeqIterator();
	}
	if(outtype==Contig::AS_TEXT){
	  //if(actread.getName().size() < 15) ostr << "\t";
	  ostr << ' ';
	}else if(outtype==Contig::AS_HTML){
	  ostr << "</td>\n<td><tt>";
	}

	int32 index = conpos-CON_outputorder[i].offset_start;
	//	  BOUNDCHECK(index, 0, CON_reads.size());  
	advance(oI, index);

	for(int32 j=0; j<cpl; j++, oI++){
	  if(index+j>=0 && 
	     index+j<static_cast<int32>(actread.getLenClippedSeq())){
//	      ostr << CON_reads[origindex].corrected[index+j];
	    char tmp=*oI;
	    bool setspan=false;
	    if(outtype==Contig::AS_HTML){
	      //Contig & nonconstcon = const_cast<Contig &>(con);
	      //int32 rp=nonconstcontigPosToUnclippedReadPos(conpos+j, CON_reads[origindex]);	
	      int32 rp=contigPosToUnclippedReadPos(conpos+j, CON_reads[origindex]);	

	      if(CON_reads[origindex].direction<0){
		rp=actread.getLenSeq()-rp-1;
	      }
	      //		int32 rp=index+j;

	      spanid.clear();
	      int32 spanprecedence=9999;
	      for(uint32 k=0;k<actread.getNumOfTags() && spanprecedence != 0;k++){
		multitag_t thetag=actread.getTag(k);
//		    if(conpos>=660 && conpos <= 6240){
//		      cout << actread.getName() << '\n';
//		      cout << "rp: " << rp << '\n';
//		      cout << "tt.f: " << thetag.from << '\n';
//		      cout << "tt.t: " << thetag.to << '\n';
//		    }
		if(rp >= static_cast<int32>(thetag.from)
		   && rp <= static_cast<int32>(thetag.to)){
		  if(thetag.identifier==Read::REA_tagentry_idED_D){
		    spanid="EDxD";
		    setspan=true;
		    spanprecedence=5;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idED_C){
		    spanid="EDxC";
		    setspan=true;
		    spanprecedence=5;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idED_I){
		    spanid="EDxI";
		    setspan=true;
		    spanprecedence=5;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idSRMr){
		    if(dptools::isValidStarBase(consseq[conpos+j])
		       && toupper(consseq[conpos+j])!=toupper(tmp)){
		      spanid="MISM";
		    }else{
		      spanid="SRMr";
		    }
		    setspan=true;
		    spanprecedence=1;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idWRMr){
		    spanid="WRMr";
		    setspan=true;
		    spanprecedence=1;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idSROr){
		    spanid="SROr";
		    setspan=true;
		    spanprecedence=0;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idSAOr){
		    spanid="SAOr";
		    setspan=true;
		    spanprecedence=0;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idSIOr){
		    spanid="SIOr";
		    setspan=true;
		    spanprecedence=0;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idFpAS){
		    spanid="POLY";
		    setspan=true;
		    spanprecedence=3;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idFCDS){
		    spanid="FCDS";
		    setspan=true;
		    spanprecedence=3;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idFtRN){
		    spanid="FtRNA";
		    setspan=true;
		    spanprecedence=3;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idFrRN){
		    spanid="FrRN";
		    setspan=true;
		    spanprecedence=3;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idFm_R){
		    spanid="FmxR";
		    setspan=true;
		    spanprecedence=3;
		  }
		  if(thetag.identifier==Read::REA_tagentry_idALUS){
		    spanid="ALUS";
		    setspan=true;
		    spanprecedence=4;
		  }
		}
	      }
	      if(setspan==false){
		if(toupper(consseq[conpos+j])!=toupper(tmp)){
		  spanid="MISM";
		  setspan=true;
		}
	      }
	      if(setspan){
		ostr << "<SPAN CLASS=\"" << spanid << "\">";
	      }
	    }

	    ostr << tmp;

	    if(setspan){
	      ostr << "</SPAN>";
	    }
	  }else{
	    if(outtype==Contig::AS_TEXT){
	      ostr << egfillchar;
	    }else if(outtype==Contig::AS_HTML){
	      switch(egfillchar) {
	      case ' ' : {
		ostr << "&nbsp;";
		break;
	      }
	      default : {
		ostr << egfillchar;
	      }
	      }
	    }
	    
	    // we're already over the limit, oI should point to .end()
	    // this has absolutely no relevance in normal code, but when compiling
	    //  in _GLIBCXX_DEBUG mode, we get an error complaining about the
	    //  fact that we try to increase the iterator past .end()
	    // therefore, reduce it before the next loop
	    if(index+j>=static_cast<int32>(actread.getLenClippedSeq())){
	      --oI;
	    }
	  }
	}

	if(outtype==Contig::AS_TEXT){
	  ostr << '\n';
	}else if(outtype==Contig::AS_HTML){
	  ostr << "</tt></td></tr>\n";
	}
      }
    }
    CEBUG("\t\t\t\t\t\t");
    if(outtype==Contig::AS_TEXT){
      ostr << setw(longestnamesize+2) << ' ';
      for(int32 j=0; j<cpl; j++){
	ostr << "-";
      }
      ostr << '\n' << left << setw(longestnamesize+2) << "Consensus:";
    }else if(outtype==Contig::AS_HTML){
      ostr << "<tr><td>Consensus:</td>\n<td><tt>";
    }

    CEBUG("\t\t\t\t\t\t");

    for(int32 j=0; j<cpl; j++){
      if(conpos+j<topos){
	bool setspan=false;
	if(outtype==Contig::AS_HTML){
	  for(size_t k=0;k<CON_consensus_tags.size();k++){
	    if(conpos+j>=static_cast<int32>(CON_consensus_tags[k].from) && conpos+j<=static_cast<int32>(CON_consensus_tags[k].to)){
	      if(CON_consensus_tags[k].identifier==CON_tagentry_idMCVc){
		ostr << "<SPAN CLASS=\"MCVc\">";
		setspan=true;
		break;
	      }
	      if(CON_consensus_tags[k].identifier==CON_tagentry_idED_D){
		ostr << "<SPAN CLASS=\"EDxD\">";
		setspan=true;
		break;
	      }
	      if(CON_consensus_tags[k].identifier==CON_tagentry_idED_C){
		ostr << "<SPAN CLASS=\"EDxC\">";
		setspan=true;
		break;
	      }
	      if(CON_consensus_tags[k].identifier==CON_tagentry_idED_I){
		ostr << "<SPAN CLASS=\"EDxI\">";
		setspan=true;
		break;
	      }
	      if(CON_consensus_tags[k].identifier==CON_tagentry_idSRMc){
		ostr << "<SPAN CLASS=\"SRMc\">";
		setspan=true;
		break;
	      }
	      if(CON_consensus_tags[k].identifier==CON_tagentry_idWRMc){
		ostr << "<SPAN CLASS=\"WRMc\">";
		setspan=true;
		break;
	      }
	      if(CON_consensus_tags[k].identifier==CON_tagentry_idSROc){
		ostr << "<SPAN CLASS=\"SROc\">";
		setspan=true;
		break;
	      }
	      if(CON_consensus_tags[k].identifier==CON_tagentry_idSAOc){
		ostr << "<SPAN CLASS=\"SAOc\">";
		setspan=true;
		break;
	      }
	      if(CON_consensus_tags[k].identifier==CON_tagentry_idSIOc){
		ostr << "<SPAN CLASS=\"SIOc\">";
		setspan=true;
		break;
	      }
	    }
	  }
	}
	if(outtype==Contig::AS_HTML 
	   && !setspan 
	   && !dptools::isValidBase(consseq[conpos+j])
	   && dptools::isValidIUPACBase(consseq[conpos+j])) {
	  setspan=true;
	  ostr << "<SPAN CLASS=\"IUPC\">";
	}
	ostr << consseq[conpos+j];
	if(setspan){
	  ostr << "</SPAN>";
	}
      }else{
	if(outtype==Contig::AS_TEXT){
	  ostr << egfillchar;
	}else if(outtype==Contig::AS_HTML){
	  switch(egfillchar) {
	  case ' ' : {
	    ostr << "&nbsp;";
	    break;
	  }
	  default : {
	    ostr << egfillchar;
	  }
	  }
	}
      }
    }
	   
    if(outtype==Contig::AS_TEXT){
      ostr << "\n\n";
    }else if(outtype==Contig::AS_HTML){
      ostr << "</tt>\n</td></tr>\n";
    }
    if(outtype==Contig::AS_HTML) ostr << "\n</table>\n<p>\n";
  }

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpConReads()
{
  cout << "%%% dumping contig reads" << '\n';
  for(uint32 i=0; i< CON_reads.size(); i++){
    cout << i << "\t" << CON_reads[i].orpid << "\t" << CON_reads[i].offset;
    cout << "\t" << CON_reads[i].direction << "\t" << CON_reads[i].read.getName();
    if(CON_reads[i].read.isBackbone()) cout << "\tbb";
    if(CON_reads[i].read.isRail()) cout << "\trail";
    cout << '\n';
  }
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpAsDebug(ostream & ostr)
{
  FUNCSTART("void Contig::dumpAsDebug(ostream & ostr)");

  // TODO: adapt to new struct

  uint32 count=0;
  cccontainer_t::const_iterator I=CON_counts.begin();
  while(I!=CON_counts.end()){
    ostr << count << ":\t" << I->A 
	 << '\t' << I->C 
	 << '\t' << I->G
	 << '\t' << I->T
	 << '\t' << I->N
	 << '\t' << I->X
	 << '\t' << I->star
	 << '\t' << I->total_cov
	 << '\t' << I->baselock
	 << '\t' << I->snplock
	 << '\t' << static_cast<char>(I->backbonechar)
	 << "\t[0] " << I->bbcounts[0]
	 << "\t[1] " << I->bbcounts[1]
	 << '\n';
    I++; count++;
  }
  
  ostr.flush();
  FUNCEND();

  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpAsTCS(ostream & ostr)
{
  FUNCSTART("void Contig::dumpAsTCS(ostream & ostr)");

  string consseq;
  vector<base_quality_t> consqual;

  //OLDgetConsensus1(consseq, consqual, false, 0, 0, -1, '@', &ostr);

  ostr << "TCS output currently not available, please contact the author.\n";
  
  ostr.flush();
  FUNCEND();

  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::buildFASTAHeader(ostringstream & ostr) const
{
  ostr << ">" << getContigName();
  // temporarily disabled: IGB 4.56 cannot correctly parse the
  //  FASTA name if there's more than the name in the line.
  //ostr << "    nseq=" << CON_reads.size();
  ostr << '\n';
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpAsFASTA(ostream & ostr, bool padded)
{
  FUNCSTART("void Contig::dumpAsFASTA(ostream & ostr, bool padded)");

  ostringstream tmp;
  buildFASTAHeader(tmp);
  ostr << tmp.str();;

  {
    string consseq;
    vector<base_quality_t> consqual;

    newConsensusGet(consseq, consqual);

    const char * cptr = consseq.c_str();

    uint32 cpl=0;
    for(;*cptr;cptr++){
      if(padded || *cptr!='*') {
	ostr << *cptr;
	if(cpl++==59){
	  cpl=0;
	  ostr << "\n";
	}
      }
    }
    if(cpl!=0) ostr << "\n";
  }

  ostr.flush();
  FUNCEND();

  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpAsFASTAQUAL(ostream & ostr, bool padded)
{
  FUNCSTART("void Contig::dumpAsFASTAQUAL(ostream & ostr, bool padded)");

  ostr << ">" << getContigName() << '\n';

  string consseq;
  vector<base_quality_t> consqual;

  // don't need to pass padded here, we just need info on stars *
  // BUT: it is probable that we can use the cached temporary cheat consensus!
  newConsensusGet(consseq, consqual);

  BUGIFTHROW(consseq.size() != consqual.size(), "sequence size != qual size ???");

  {
    const char * cptr = consseq.c_str();
    vector<base_quality_t>::const_iterator I=consqual.begin();
    uint32 cpl=0;
    for(;I!=consqual.end(); I++, cptr++){
      if(padded || *cptr!='*') {
	ostr << static_cast<uint16>(*I) << " ";
	if(cpl++==19){
	  cpl=0;
	  ostr << "\n";
	}
      }
    }
    if(cpl!=0) ostr << "\n";
  }

  ostr.flush();
  FUNCEND();

  return;
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpStrainAsFASTAQUAL(ostream & fout,ostream & qout, uint32 mincoverage, base_quality_t minqual, int32 strainidtotake, bool padded, bool fillholesinstrain)
{
  FUNCSTART("void Contig::dumpAsFASTA(ostream & fout, ostream & qout, base_quality_t minqual, int32 strainidtotake, bool padded, bool fillholesinstrain)");

  ostringstream tmp;
  buildFASTAHeader(tmp);
  fout << tmp.str();
  qout << tmp.str();

  {
    string strainseq;
    vector<base_quality_t> strainqual;
    calcConsensi(mincoverage,minqual);
    newConsensusGet(strainseq, strainqual, strainidtotake);
    const char * sptr = strainseq.c_str();
    vector<base_quality_t>::const_iterator S=strainqual.begin();

    string consseq;
    vector<base_quality_t> consqual;
    const char * cptr = sptr;
    vector<base_quality_t>::const_iterator C=S;
    if(fillholesinstrain){
      calcConsensi(mincoverage,0);
      newConsensusGet(consseq, consqual);
      BUGIFTHROW(consseq.size() != strainseq.size(), "cons size != strain size ???");
      cptr=consseq.c_str();
      C=consqual.begin();
    }

    uint32 cpl=0;
    uint32 qpl=0;
    for(;*sptr;sptr++,cptr++, S++, C++){
      if(padded || *sptr!='*') {
	if(fillholesinstrain && (*sptr=='N' || *sptr=='@')) {
	  fout << static_cast<char>(tolower(*cptr));
	  qout << static_cast<uint16>(*C) << ' ';
	} else {
	  fout << *sptr;
	  qout << static_cast<uint16>(*S) << ' ';
	}
	if(cpl++==59){
	  cpl=0;
	  fout << "\n";
	}
	if(qpl++==19){
	  qpl=0;
	  qout << "\n";
	}
      }
    }
    if(cpl!=0) fout << "\n";
    if(qpl!=0) qout << "\n";
  }

  fout.flush();
  qout.flush();

  FUNCEND();
  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpStrainAsFASTQ(ostream & fout, uint32 mincoverage, base_quality_t minqual, int32 strainidtotake, bool padded, bool fillholesinstrain)
{
  FUNCSTART("void Contig::dumpAsFASTq(ostream & fout, base_quality_t minqual, int32 strainidtotake, bool padded, bool fillholesinstrain)");


  fout << '@' << getContigName() << '\n';

  {
    string strainseq;
    vector<base_quality_t> strainqual;
    calcConsensi(mincoverage,minqual);
    newConsensusGet(strainseq, strainqual, strainidtotake);
    const char * sptr = strainseq.c_str();

    string consseq;
    vector<base_quality_t> consqual;
    const char * cptr=NULL;
    if(fillholesinstrain){
      calcConsensi(mincoverage,0);
      newConsensusGet(consseq, consqual);
      BUGIFTHROW(consseq.size() != strainseq.size(), "cons size != strain size ???");
      cptr = consseq.c_str();
    }

    for(;*sptr;sptr++,cptr++){
      if(padded || *sptr!='*') {
	if(fillholesinstrain && (*sptr=='N' || *sptr=='@')) {
	  fout << static_cast<char>(tolower(*cptr));
	} else {
	  fout << *sptr;
	}
      }
    }
    fout << "\n+\n";

    cptr = consseq.c_str();
    sptr = strainseq.c_str();
    vector<base_quality_t>::const_iterator C=consqual.begin();
    vector<base_quality_t>::const_iterator S=strainqual.begin();

    for(;*sptr;sptr++,cptr++, S++, C++){
      if(padded || *sptr!='*') {
	if(fillholesinstrain && (*sptr=='N' || *sptr=='@')) {
	  fout << (*C)+33;
	} else {
	  fout << (*S)+33;
	}
      }
    }
    fout <<'\n';
  }

  fout.flush();

  FUNCEND();
  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpAsCAF(ostream & ostr)
{
  FUNCSTART("void Contig::dumpAsCAF(ostream & ostr)");

  finalise();
  
  // make the consensus here as the routines also may set tags (STRM)
  //  to the contig
  // but first delete the old tags

  deleteTagsInReads(CON_tagentry_idSTMS);
  deleteTagsInReads(CON_tagentry_idSTMU);
  string consstring;
  vector<base_quality_t> consqual;

  newConsensusGet(consstring, consqual);

  Read::setCoutType(Read::AS_CAF);

  if(CON_reads.size() != CON_outputorder.size()){
    cerr << "Ouch! CON_reads.size() != CON_outputorder.size():  " << CON_reads.size() << " != " << CON_outputorder.size() << '\n';
    throw Notify(Notify::INTERNAL,THISFUNC,"Array size expected to be equal!");
  }

  // output rails first
  {
    for(uint32 ii=0; ii<CON_reads.size(); ii++){
      int32 origindex=CON_outputorder[ii].original_index;
      if(CON_outputRails && CON_reads[origindex].read.isRail()){
	ostr << CON_reads[origindex].read;
      }
    }
  }

  //ostr << "// dubidu" << '\n';

  // Reads
  {
    for(uint32 ii=0; ii<CON_reads.size(); ii++){
      int32 origindex=CON_outputorder[ii].original_index;
      if(CON_reads[origindex].read.isRail()) continue;
      ostr << CON_reads[origindex].read;
    }
  }

  ostr << "Sequence : " << getContigName() << '\n';
  ostr << "Is_contig\nPadded\n";
  
  // Assembled from
  {
    for(uint32 ii=0; ii<CON_outputorder.size(); ii++){
      int32 origindex=CON_outputorder[ii].original_index;
      if(!CON_outputRails && CON_reads[origindex].read.isRail()) continue;

      ostr << "Assembled_from " << CON_reads[origindex].read.getName() << " ";
      if(CON_reads[origindex].direction>0){
	ostr << CON_reads[origindex].offset+1 << " " << CON_reads[origindex].offset+CON_reads[origindex].read.getLenClippedSeq() << " ";
      }else{
	ostr << CON_reads[origindex].offset+CON_reads[origindex].read.getLenClippedSeq() << " " << CON_reads[origindex].offset+1 << " ";
      }
      ostr << CON_reads[origindex].read.getLeftClipoff()+1 << " " << CON_reads[origindex].read.getRightClipoff() << '\n';
    }
  }

  // Tags
  {
    vector<consensustag_t>::const_iterator I=CON_consensus_tags.begin();
    while(I!=CON_consensus_tags.end()){
      ostr << "Tag " << I->getIdentifierStr() << " ";
      if(I->strand=='-') {
	ostr << (I->to)+1 << " " << (I->from)+1 << " \"";
      }else{
	ostr << (I->from)+1 << " " << (I->to)+1 << " \"";
      }
      ostr << I->getCommentStr() << "\"\n";
      I++;
    }
  }

  ostr << "\n";

  {
    ostr << "DNA : " << getContigName() << '\n';

    const char * cptr = consstring.c_str();

    //makeTmpConsensus(0, CON_counts.size()+1);
    //char * cptr = CON_tmpcons;

    uint32 cpl=0;
    while(*cptr){
      if(*cptr=='*'){
	ostr << "-";
      }else{
	ostr << *cptr;
      }
      cptr++;
      if(cpl++==59){
	cpl=0;
	ostr << "\n";
      }
    }
    ostr << "\n\n";
  }

  {
    ostr << "BaseQuality : " << getContigName() << '\n';

    vector<base_quality_t>::const_iterator qI=consqual.begin();

    //makeTmpConsensus(0, CON_counts.size()+1);
    //char * cptr = CON_tmpcons;

    uint32 cpl=0;
    for(; qI!=consqual.end(); qI++){
      if(cpl++==24){
	ostr << static_cast<uint16>(*qI) << '\n';
	cpl=0;
      } else {
	ostr << static_cast<uint16>(*qI) << ' ';
      }
    }
    ostr << "\n\n";
  }


  ostr.flush();

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpAsMAF(ostream & ostr)
{
  FUNCSTART("void Contig::dumpAsMAF(ostream & ostr)");

  finalise();
  
  // make the consensus here as the routines also may set tags (STRM)
  //  to the contig
  // but first delete the old tags

  deleteTagsInReads(CON_tagentry_idSTMS);
  deleteTagsInReads(CON_tagentry_idSTMU);
  string consstring;
  vector<base_quality_t> consqual;

  newConsensusGet(consstring, consqual);

//  makeIntelligentConsensus(consstring,
//			   consqual,
//			   NULL,
//			   NULL,
//			   0, static_cast<int32>(CON_counts.size())
//			   ,false,false);

  Read::setCoutType(Read::AS_MAF);

  if(CON_reads.size() != CON_outputorder.size()){
    cerr << "Ouch! CON_reads.size() != CON_outputorder.size():  " << CON_reads.size() << " != " << CON_outputorder.size() << '\n';
    throw Notify(Notify::INTERNAL,THISFUNC,"Array size expected to be equal!");
  }

  // CO = contig name
  // NR = num reads (optional)
  // LC = contig length
  // CT = Consensus Tag (identifier from to comment).

  // scraped AF = assembled from (like in CAF: readname from to)
  // AT = assemble to (like in CAF AF, but no readname: from to)
  //      furthermore, they follow RD/ER block immediately so that
  //      the parser does not need to perform time consuming string lookups

  // CS = sequence (in one line, gaps as '*')
  // CQ = base quality (in one line, FASTQ-33 format)

  // \\ = start of read data
  // // = end of read data

  // EC = End Contig (marker for MAF parsing, mandatory)

  ostr << "CO\t" << getContigName()
       << "\nNR\t" << CON_reads.size()
       << "\nLC\t" << consstring.size() << '\n';

  // Tags
  {
    vector<consensustag_t>::const_iterator I=CON_consensus_tags.begin();
    while(I!=CON_consensus_tags.end()){
      ostr << "CT\t" << I->getIdentifierStr() << " ";
      if(I->strand=='-') {
	ostr << (I->to)+1 << " " << (I->from)+1 << " ";
      }else{
	ostr << (I->from)+1 << " " << (I->to)+1 << " ";
      }
      const string & com=I->getCommentStr();
      for(uint32 i=0; i<com.size(); i++){
	if(com[i]=='\n'){
	  ostr << "\\n";
	}else if(com[i]=='\\'){
	  ostr << "\\\\";
	}else{
	  ostr << com[i];
	}
      }
      ostr << '\n';
      I++;
    }
  }

  {
    ostr << "CS\t";
    ostr << consstring << '\n';
  }

  {
    ostr << "CQ\t";
    vector<base_quality_t>::const_iterator qI=consqual.begin();
    for(; qI!=consqual.end(); qI++) ostr << static_cast<char>((*qI) +33);
    ostr << '\n';
  }

  ostr << "\\\\\n";

  // output rails first
  {
    for(uint32 ii=0; ii<CON_reads.size(); ii++){
      int32 origindex=CON_outputorder[ii].original_index;
      if(CON_outputRails && CON_reads[origindex].read.isRail()){
	ostr << CON_reads[origindex].read;

	// and the AT line
	ostr << "AT\t";
	if(CON_reads[origindex].direction>0){
	  ostr << CON_reads[origindex].offset+1 << " " << CON_reads[origindex].offset+CON_reads[origindex].read.getLenClippedSeq() << " ";
	}else{
	  ostr << CON_reads[origindex].offset+CON_reads[origindex].read.getLenClippedSeq() << " " << CON_reads[origindex].offset+1 << " ";
	}
	ostr << CON_reads[origindex].read.getLeftClipoff()+1 << " " << CON_reads[origindex].read.getRightClipoff() << '\n';
      }
    }
  }

  //ostr << "// dubidu" << '\n';

  // Reads
  {
    for(uint32 ii=0; ii<CON_reads.size(); ii++){
      int32 origindex=CON_outputorder[ii].original_index;
      if(CON_reads[origindex].read.isRail()) continue;
      ostr << CON_reads[origindex].read;
      // and the AT line
      ostr << "AT\t";
      if(CON_reads[origindex].direction>0){
	ostr << CON_reads[origindex].offset+1 << " " << CON_reads[origindex].offset+CON_reads[origindex].read.getLenClippedSeq() << " ";
      }else{
	ostr << CON_reads[origindex].offset+CON_reads[origindex].read.getLenClippedSeq() << " " << CON_reads[origindex].offset+1 << " ";
      }
      ostr << CON_reads[origindex].read.getLeftClipoff()+1 << " " << CON_reads[origindex].read.getRightClipoff() << '\n';
    }
  }

  ostr << "//\n";

  ostr << "EC\n";

  ostr.flush();

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpAsSAM(ostream & ostr)
{
  FUNCSTART("void Contig::dumpAsSAM(ostream & ostr)");

  finalise();
  
  // make the consensus here as the routines also may set tags (STRM)
  //  to the contig
  // but first delete the old tags

  deleteTagsInReads(CON_tagentry_idSTMS);
  deleteTagsInReads(CON_tagentry_idSTMU);
  string consstring;
  vector<base_quality_t> consqual;

  newConsensusGet(consstring, consqual);

  //Read::setCoutType(Read::AS_SAM);

  if(CON_reads.size() != CON_outputorder.size()){
    cerr << "Ouch! CON_reads.size() != CON_outputorder.size():  " << CON_reads.size() << " != " << CON_outputorder.size() << '\n';
    throw Notify(Notify::INTERNAL,THISFUNC,"Array size expected to be equal!");
  }

  // make string for tags
  // they somehow should belong to the contig object, but gap5 imports them in reads *sigh*
  string constagstr;
  {
    ostringstream tmp;

    uint32 numtags=0;
    vector<consensustag_t>::const_iterator I=CON_consensus_tags.begin();
    for(; I!=CON_consensus_tags.end(); ++I, ++numtags){
      if(numtags) tmp << '\t';
      tmp << "Zc:Z:" << I->getIdentifierStr() 
	  << "|" << I->strand
	  << "|" << (I->from)+1
	  << "|" << (I->to)-(I->from);
      const string & com=I->getCommentStr();
      for(uint32 i=0; i<com.size(); i++){
	if(com[i]=='\n'){
	  tmp << "\\n";
	}else if(com[i]=='\\'){
	  tmp << "\\\\";
	}else{
	  tmp << com[i];
	}
      }
    }

    constagstr=tmp.str();
  }

  ostr << "@SQ\tSN:" << getContigName()
       << "\tLN:" << consstring.size() << '\n';

  {
    string cigar;
    for(uint32 ii=0; ii<CON_reads.size(); ii++){
      int32 origindex=CON_outputorder[ii].original_index;

      uint32 flags=0;
      if(CON_reads[origindex].direction<0) flags|=0x10;
      //ostr << CON_reads[origindex].read;

      CON_reads[origindex].read.getCIGARString(cigar);
      ostr << CON_reads[origindex].read.getName()
	   << '\t' << flags
	   << '\t' << getContigName()
	   << '\t' << CON_reads[origindex].offset+1
	   << '\t' << 50 //mapping quality
	   << '\t' << cigar
	   << "\t*" // TODO: RNEXT
	   << "\t0" // TODO: PNEXT
	   << "\t";

      vector<char>::const_iterator bI, eI;
      if(CON_reads[origindex].direction>0){
	bI=CON_reads[origindex].read.getSeqIteratorBegin();
	eI=CON_reads[origindex].read.getSeqIteratorEnd();
      }else{
	bI=CON_reads[origindex].read.getComplementSeqIteratorBegin();
	eI=CON_reads[origindex].read.getComplementSeqIteratorEnd();
      }
      for(; bI != eI; ++bI) ostr << *bI;

      ostr << '\t';
      for(uint32 posi=0; posi < CON_reads[origindex].read.getLenSeq(); ++posi){
	if(CON_reads[origindex].direction>0){
	  ostr << static_cast<char>(33+CON_reads[origindex].read.getQualityInSequence(posi));
	}else{
	  ostr << static_cast<char>(33+CON_reads[origindex].read.getQualityInComplementSequence(posi));
	}
      }
      ostr << '\n';
    }
  }

  ostr.flush();

  FUNCEND();
}



/*************************************************************************
 *
 * I need to count BS lines before output, so this function
 *  is called twice: once for counting only, once for output
 *
 *************************************************************************/

//#define CEBUG(bla)   {cout << bla; cout.flush();}
int32 Contig::helper_dumpAsACE_BSLines(string & consseq, bool alsodump, ostream & ostr)
{
  FUNCSTART("int32 Contig::helper_dumpAsACE_BSLines(string & consseq, bool alsodump, ostream & ostr)");

  int32 numBSlines=0;

  CEBUG("consseq: " << consseq << endl);

  const char * cptr = consseq.c_str();
  int32 conpos = 0;
  int32 bqindex= -1;
  int32 currentindex= -1;
  int32 stretch_begin=1;
  vector<int32> readidx;


  finalise();
  vector<out_order_t>::const_iterator outorderI = CON_outputorder.begin();

  // initialise the iterator for getting through the contig
  rcci_t rcci;
  {
    vector<int32> allowedstrainids; // empty would be all ids
    vector<uint8> allowedreadtypes;
    readColContigIteratorInit(rcci, 
			      allowedstrainids, 
			      allowedreadtypes, 
			      false,            // don't take rails
			      true,           // backbones
			      true);   // nor reads without readpool-reads
  }


  for(;*cptr; cptr++, conpos++, readColContigIteratorAdvance(rcci)){    
    char consbase = static_cast<char>(toupper(*cptr));
    base_quality_t bestqual=0;
    bqindex=-1;

    CEBUG("caa: conpos: " << conpos << '\t' << consbase << '\n');

    // look for best qual equal to cons base
    bool foundcurrindexincol=false;
    bool foundaread=false;
    for(uint32 i= 0; i<rcci.read_ids_in_col.size(); i++) {
      CEBUG("caa: read: " << CON_readpool->getRead(rcci.read_ids_in_col[i]).getName());
      if(i==0) {
	bqindex=rcci.read_ids_in_col[0];
	// do not set bestqual
	if(conpos==0) {
	  currentindex=rcci.read_ids_in_col[0];
	}
      }
      if(rcci.read_ids_in_col[i]==currentindex && toupper(getBaseInRead(conpos,rcci.read_ids_in_col[i])) == consbase){
	foundcurrindexincol=true;
      }
      if(toupper(getBaseInRead(conpos,rcci.read_ids_in_col[i])) == consbase
	 || (toupper(getBaseInRead(conpos,rcci.read_ids_in_col[i])) == 'X' && consbase=='N')) {
	base_quality_t q=getQualityInRead(conpos,rcci.read_ids_in_col[i]);
	if(q > bestqual) {
	  bestqual=q;
	  bqindex=rcci.read_ids_in_col[i];
	  CEBUG("\tbq");
	}
      }
      CEBUG('\n');
    }
    
    //// if no read chosen, look for best qual IN consbase
    //for(uint32 i= 0; i<rcci.read_ids_in_col.size(); i++) {
    //  CEBUG("caa: read: " << CON_readpool->getRead(rcci.read_ids_in_col[i]).getName() << '\n');
    //  if(toupper(getBaseInRead(conpos,rcci.read_ids_in_col[i])) == consbase
    //	 || (toupper(getBaseInRead(conpos,rcci.read_ids_in_col[i])) == 'X' && consbase=='N')) {
    //	base_quality_t q=getQualityInRead(conpos,rcci.read_ids_in_col[i]);
    //	if(q > bestqual) {
    //	  bestqual=q;
    //	  bq=rcci.read_ids_in_col[i];
    //	}
    //  }
    //}

    if(!foundaread){
      bqindex=rcci.read_ids_in_col[0];
      bestqual=getQualityInRead(conpos,bqindex);
    }

    CEBUG("bq: " << static_cast<uint16>(bestqual) << "\nbqindex: " << bqindex << '\n');

    // look if current read index (if set) is not within 5 of best qual 
    //  then output bs line
    if((!foundcurrindexincol || bestqual - getQualityInRead(conpos,currentindex) > 80)
      && bqindex != currentindex) {
      CEBUG("Jump new read: " << CON_readpool->getRead(bqindex).getName() << '\n');
      numBSlines++;
      if(alsodump){
	ostr << "BS " << stretch_begin << " " << conpos << " ";
	BUGIFTHROW(currentindex<0 || currentindex >= static_cast<int32>(CON_reads.size()),"1 / currentindex<0 || currentindex >= CON_reads.size() ???");
	ostr << CON_reads[currentindex].read.getName() << '\n';
      }
      stretch_begin=conpos+1;
      currentindex=bqindex;      
    }
  }

  numBSlines++;
  if(alsodump) {
    ostr << "BS " << stretch_begin << " " << conpos << " ";
    BUGIFTHROW(currentindex<0 || currentindex >= static_cast<int32>(CON_reads.size()),"2 / currentindex<0 || currentindex >= CON_reads.size() ???");
    ostr << CON_reads[currentindex].read.getName() << "\n\n";
  }

  FUNCEND();
  return numBSlines;
}
//#define CEBUG(bla)


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpAsACE(ostream & ostr)
{
  FUNCSTART("void Contig::dumpAsACE(ostream & ostr)");

  // 0 basesegments, don't know if i use the golden path
  //if(CON_reads.size() > 1){
  //  ostr << "CO C" << CON_id << " " << CON_counts.size() << " ";
  //}else{
  //  ostr << "CO S" << CON_id << " " << CON_counts.size() << " ";
  //}
  //ostr << CON_reads.size() << " 0 U" << '\n';



  string consseq;
  vector<base_quality_t> consqual;

  newConsensusGet(consseq, consqual);

  ostr << "CO " << getContigName() << " " << CON_counts.size() << " ";
  if(!CON_outputRails){
    uint32 numreads=0;
    for(uint32 i=0; i<CON_reads.size(); i++){
      if(!CON_reads[i].read.isRail()) numreads++;
    }
    ostr << numreads << " ";
  }else{ 
    ostr << CON_reads.size() << " ";
  }
  ostr << helper_dumpAsACE_BSLines(consseq,false,ostr) << " U\n";

  {
    const char * cptr = consseq.c_str();
    uint32 cpl=0;
    while(*cptr){
      ostr << *cptr;
      cptr++;
      if(cpl++==59){
	cpl=0;
	ostr << "\n";
      }
    }
    ostr << "\n\n";
  }

  {
    // qual for pads MUST NOT be output
    ostr << "BQ\n";
    vector<base_quality_t>::const_iterator I=consqual.begin();
    const char * cptr = consseq.c_str();
    uint32 cpl=0;
    while(I!=consqual.end()){
      if(*cptr !='*') {
	ostr << static_cast<uint16>(*I) << " ";
	if(cpl++==19){
	  cpl=0;
	  ostr << "\n";
	}
      }
      I++;
      cptr++;
    }
    ostr << "\n\n";
  }

  // AF lines
  for(uint32 i=0; i<CON_reads.size(); i++){
    if(!CON_outputRails && CON_reads[i].read.isRail()) continue;
    ostr << "AF " << CON_reads[i].read.getName();
    if(CON_reads[i].direction >0) {
      ostr << " U ";
      ostr << (static_cast<int32>(CON_reads[i].offset)+1 - static_cast<int32>(CON_reads[i].read.getLeftClipoff()) )<< '\n';
    } else {
      ostr << " C ";
      //ostr << (int32) CON_reads[i].offset << " ";
      //ostr << (int32) CON_reads[i].read.getLenSeq() << " ";
      //ostr << (int32) CON_reads[i].read.getLeftClipoff() << " ";
      //ostr << (int32) CON_reads[i].read.getRightClipoff() << " ";
      ostr << (static_cast<int32>(CON_reads[i].offset) + static_cast<int32>(CON_reads[i].read.getRightClipoff()) - static_cast<int32>(CON_reads[i].read.getLenSeq()) + 1 )<< '\n';
    }
  }
  ostr << "\n";

  // BS lines
  helper_dumpAsACE_BSLines(consseq,true,ostr);

  // Tags
  {
    vector<consensustag_t>::const_iterator I=CON_consensus_tags.begin();
    while(I!=CON_consensus_tags.end()){
      ostr << "CT{\n";
      //ostr << "C" << CON_id;
      ostr << getContigName() << ' ' << I->getIdentifierStr();
      if(I->strand=='-') {
	ostr << " MIRA " << (I->to)+1 << ' ' << (I->from)+1;
      }else{
	ostr << " MIRA " << (I->from)+1 << ' ' << (I->to)+1;
      }
      ostr << " 020202:121212 NoTrans\n";
      if(I->comment != CON_tagentry_coEmpty){
	ostr << "COMMENT{\n" << I->getCommentStr() << "\nC}\n";
      }
      ostr << "}\n\n";
      I++;
    }
   }

  // Reads
  {
    Read::setCoutType(Read::AS_ACE);
    for(uint32 i=0; i<CON_reads.size(); i++){
      if(!CON_outputRails && CON_reads[i].read.isRail()) continue;
      if (CON_reads[i].direction >0 ) {
	Read::setCoutType(Read::AS_ACE);
	ostr << CON_reads[i].read;
      } else {
	Read::setCoutType(Read::AS_ACE_COMPLEMENT);
	ostr << CON_reads[i].read;
      }
    }
  }

  ostr << '\n';


  ostr.flush();

  FUNCEND();
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::saveAsGAP4DA(const string & dirname, ostream & fofnstr)
{
  FUNCSTART("void Contig::saveAsGAP4DA()");

  finalise();
  
  // Reads
  {
    string anchor="";
    string APline="";
    for(uint32 cooi=0; cooi<CON_outputorder.size(); cooi++){
      int32 origindex=CON_outputorder[cooi].original_index;
      if(!CON_outputRails && CON_reads[origindex].read.isRail()) continue;
      if(APline.empty()){
	if(CON_reads[origindex].direction>0){
	  APline="*new* +";
	} else {
	  APline="*new* -";
	}
	anchor=CON_reads[origindex].read.getName();
      } else {
	ostringstream tmp;
	if(CON_reads[origindex].direction>0){
	  tmp << anchor << " + " << CON_reads[origindex].offset << " -1";
	}else{
	  tmp << anchor << " - " << CON_reads[origindex].offset << " -1";
	}
	APline=tmp.str();
      }

      // we really don't want EXP filenames with a | as part of the filename
      string expfilename=CON_reads[origindex].read.getName()+".exp";
      for(uint32 ii=0; ii<expfilename.size(); ii++){
	if(expfilename[ii]=='|') expfilename[ii]='_';
      }

      ofstream expout((dirname+"/"+expfilename).c_str(), ios::out | ios::trunc);
      CON_reads[origindex].read.dumpAsGAP4DA(expout, APline);


      // We're putting consensus tags all in the first exp file
      // james bonfield suggested the last file, but this lead to unexplainable
      //  off by 1,2,3,... shifts.
      if(cooi==0) {
	vector<consensustag_t>::const_iterator T=CON_consensus_tags.begin();
	for(;T!=CON_consensus_tags.end();T++) {
	  expout << "TC   " << T->getIdentifierStr();

	  if(CON_reads[origindex].direction>0) {
	    // if the first read of that contig is saved in + direction,
	    // we must add the left clippoff of the first read that
	    //  was written to the file
	    int32 lq=CON_reads[origindex].read.getLeftClipoff();
	    expout << " = " << (T->from)+1+lq << ".." << (T->to)+1+lq;
	  } else {
	    // if the first read of that contig is saved complemented,
	    //  save the tag positions also complemented
	    //  (and take right extend)
	    int32 lq=CON_reads[origindex].read.getRightClipoff();

	    expout << " = " << lq-static_cast<int32>(T->to) << ".." << lq-static_cast<int32>(T->from);
	    //if(lq-((T->to)) > CON_counts.size()) {
	    //  cout << "PANIC:\t";
	    //  Read::setCoutType(Read::AS_TEXTCLIPS);
	    //  cout << CON_reads[origindex].read << '\n' << '\n';
	    //  cout << "lq: " << lq << '\n';
	    //  T->dump();
	    //  cout << "Computed to: " << lq-((T->to)) << '\n';
	    //}
	  }
	  const char * cs=T->getCommentStr().c_str();
	  if (*cs!=0) expout << "\nTC        ";
	  while(*cs) {
	    if(*cs == '\n') expout << "\nTC        ";
	    expout << *cs;
	    cs++;
	  }
	  expout << '\n';
	}
      }

      expout.close();

      fofnstr << expfilename << '\n';
    }
  }

  FUNCEND();
  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpContigReadList_Head(ostream &ostr)
{
  ostr << "#\n# contig_name\tread_name\n#\n";
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpContigReadList_Body(ostream &ostr)
{
  vector<contigread_t>::const_iterator J=getContigReads().begin(); 
  while(J != getContigReads().end()) {
    ostr << getContigName() << "\t";
    ostr << J->read.getName() << "\n";
    J++;
  }
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpContigStatistics_Head(ostream &ostr)
{
  //ostr << "# name\t\tlength\tav.qual\t#-reads\tmx.cov.\tav.cov\tav.rlen\tAinCn\tCinCn\tGinCn\tTinCn\tIUPACinCn\tFunnyinCn\tNinCn\tXinCn\t*inCn\tNoCovinCn\n";
  //ostr << "# name\tlength\tav.qual\t#-reads\tmx.cov.\tav.cov\tGC%\tIUPACinCn\tFunnyinCn\tNinCn\tXinCn\t*inCn\tNoCovinCn\n";
  ostr << setw(20) << left << "# name";
  ostr << "\tlength\tav.qual\t#-reads\tmx.cov.\tav.cov\tGC%\tCnIUPAC\tCnFunny\tCnN\tCnX\tCnGap\tCnNoCov\n";
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpContigStatistics_Body(ostream &ostr)
{
  Contig::constats_t s=getStats();

  ostr << setw(20) << left << getContigName();
  ostr << '\t';
  ostr << s.conlength_nogap << '\t';
  ostr << s.avg_conqual << '\t';
  ostr << getNumReadsInContig() << '\t';
  ostr << s.max_coverage << '\t';
  ios_base::fmtflags f = ostr.flags();
  ostr.setf(ios::fixed, ios::floatfield);
  ostr.setf(ios::showpoint);
  ostr.precision(2);
  ostr << s.avg_coverage << '\t';
  ostr << s.gccontent << '\t';
  ostr.flags(f);

  //ostr << s.avg_readlen << '\t';
  //ostr << "--\t";
  
  //ostr << s.AinC << '\t';
  //ostr << s.CinC << '\t';
  //ostr << s.GinC << '\t';
  //ostr << s.TinC << '\t';
  ostr << s.IUPACinC << '\t'
       << s.FunnyInC << '\t'
       << s.NinC << '\t'
       << s.XinC << '\t'
       << s.starInC << '\t'
       << s.numnocoverage
       << '\n';

}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpReadTagList_Head(ostream &ostr)
{
  ostr << "#\n"
       << "# conName\tcFromPadded\tcToPadded\tcFromUnpadded\tcToUnpadded\ttype\trName\trFromPadded\trToPadded\trFromUnpadded\trToUnpadded\tcomment\n"
       << "#\n";
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpReadTagList_Body(ostream &ostr)
{
  string serialc;

  vector<contigread_t>::const_iterator R=getContigReads().begin(); 
  for(; R != getContigReads().end() ; R++) {
    const_cast<Read &>(R->read).sortTags();
    for(uint32 t=0; t < R->read.getNumOfTags(); t++){
      const multitag_t & thetag= R->read.getTag(t);
      int32 cpf=unclippedReadPosToContigPos(thetag.from, *R);
      int32 cpt=unclippedReadPosToContigPos(thetag.to, *R);
      ostr << getContigName() << "\t"
	   << cpf << "\t"
	   << cpt << "\t";
      if(cpf>=0) ostr << paddedPos2UnpaddedPos(cpf);
      ostr << "\t";
      if(cpt>=0) ostr << paddedPos2UnpaddedPos(cpt);
      ostr << "\t"
	   << thetag.getIdentifierStr() << "\t"
	   << R->read.getName() << "\t"
	   << thetag.from << "\t"
	   << thetag.to << "\t"
	   << R->read.getLowerNonGapAdjustmentPosOfReadPos(thetag.from) << "\t"
	   << R->read.getUpperNonGapAdjustmentPosOfReadPos(thetag.to) << "\t";
      thetag.serialiseComment(serialc);
      ostr << serialc << "\n";
    }
  }
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpConsensusTagList_Head(ostream &ostr)
{
  ostr << "#\n"
    "# conName\tfromPadded\ttoPadded\tfromUnPadded\ttoUnPadded\ttype\tSNPqual\tqualA\tqualC\tqualG\tqualT\tqual*\tcomment\n"
    "#" << endl;
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpConsensusTagList_Body(ostream &ostr)
{
  string serialc;

  sortConsensusTags();
  vector<Contig::consensustag_t>::const_iterator T=getConsensusTags().begin();
  for(; T != getConsensusTags().end() ; T++) {
    ostr << getContigName() << "\t";
    ostr << T->from << "\t";
    ostr << T->to << "\t";
    ostr << paddedPos2UnpaddedPos(T->from) << "\t";
    ostr << paddedPos2UnpaddedPos(T->to) << "\t";
    ostr << T->getIdentifierStr() << "\t";
    if(T->additionalinfo_initialised) {
      base_quality_t snpQual=255;
      //if((T->qualA)>0 && T->qualA<snpQual) snpQual=T->qualA;
      //if((T->qualC)>0 && T->qualC<snpQual) snpQual=T->qualC;
      //if((T->qualG)>0 && T->qualG<snpQual) snpQual=T->qualG;
      //if((T->qualT)>0 && T->qualT<snpQual) snpQual=T->qualT;
      //if((T->qualStar)>0 && T->qualStar<snpQual) snpQual=T->qualStar;
      
      for(uint8 i=0; i<5; i++) {
	if((T->qualACGTStar[i])>0 && T->qualACGTStar[i]<snpQual) snpQual=T->qualACGTStar[i];
      }
      
      if(snpQual==255) snpQual=0;
      
      ostr << static_cast<uint16>(snpQual) << "\t";
      for(uint8 i=0; i<5; i++) {
	ostr << static_cast<uint16>(T->qualACGTStar[i]) << "\t";
      }
      //ostr << static_cast<uint16>(T->qualACGTStar[0]) << "\t";
      //ostr << static_cast<uint16>(T->qualACGTStar[1]) << "\t";
      //ostr << static_cast<uint16>(T->qualACGTStar[2]) << "\t";
      //ostr << static_cast<uint16>(T->qualACGTStar[3]) << "\t";
      //ostr << static_cast<uint16>(T->qualACGTStar[4]) << "\t";
    }else{
      ostr << "\t\t\t\t\t\t";
    }
    T->serialiseComment(serialc);
    ostr << serialc << "\n";
  }
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpTCS_Head(ostream &ostr)
{
  ostr << "#TCS V1.0\n";
  ostr << "#\n";
  ostr << setw(20) << left << "# contig name"
       << setw(9) << right << "padPos" << setw(9) << "upadPos"
       << " | B" << "  Q |" 
       << setw(5) << "tcov" 
       << setw(5) << "covA" 
       << setw(5) << "covC" 
       << setw(5) << "covG" 
       << setw(5) << "covT" 
       << setw(5) << "cov*"
       << " |"
       << setw(3) << "qA" 
       << setw(3) << "qC" 
       << setw(3) << "qG" 
       << setw(3) << "qT" 
       << setw(3) << "q*" 
       << " |"
       << setw(3) << "S" 
       << " | Tags\n"
       << "#" << endl;
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpTCS_Body(ostream &ostr)
{
  dumpAsTCS(ostr);
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::dumpWiggle_Head(ostream &ostr)
{
  ostr << "#TCS V0.9\n"
    "#\n"
    "# conName paddedPos unpaddedPos Base bastyp bqual totcov qualA qualC qualG qualT qual*   covA covC covG covT cov*   Tags\n"
    "#" << endl;
}

/*************************************************************************
 *
 * dumps the coverage of a strain as wiggle file
 * must give padded consensus to routine (save recalculation)
 *
 *************************************************************************/

void Contig::dumpWiggle_Body(ostream &ostr, string & cons)
{
  FUNCSTART("void Contig::dumpWiggle_Body(ostream &ostr, string & cons)");
  BUGIFTHROW(cons.size() != CON_counts.size(), getContigName() << ": consensus.size() != CON_counts.size() ???");

  cccontainer_t::const_iterator ccI=CON_counts.begin();
  ccctype_t maxcov=0;
  for(; ccI!=CON_counts.end(); ccI++){
    maxcov=max(maxcov, ccI->total_cov);
  }

  uint32 maxstrains=CON_readpool->getNumOfStrainInReadpool();
  vector<bool> strainlookup(maxstrains,false);


  ostr << "track type=wiggle_0 name=\"" << getContigName() 
       << "cov\" description=\""
       << getContigName() << " ";

  vector<contigread_t>::const_iterator crI=CON_reads.begin();
  uint32 numbackbonestrains=0;
  for(;crI != CON_reads.end(); crI++){
    if(crI->orpid>=0 && crI->read.isBackbone()){
      if(!strainlookup[crI->read.getStrainID()]){
	strainlookup[crI->read.getStrainID()]=true;
	if(numbackbonestrains==0){
	  ostr << "BB strain:";
	}
	ostr << ' ' << crI->read.getStrain();
	++numbackbonestrains;
      }
    }
  }

  strainlookup.clear();
  strainlookup.resize(maxstrains,false);
  uint32 numstrains=0;
  crI=CON_reads.begin();
  for(;crI != CON_reads.end(); crI++){
    if(crI->orpid>=0 && !crI->read.isBackbone() && !crI->read.isRail()){
      if(!strainlookup[crI->read.getStrainID()]){
	strainlookup[crI->read.getStrainID()]=true;
	if(numstrains==0){
	  if(numbackbonestrains>0){
	    ostr << " Mapped strain(s):";
	  }else{
	    ostr << " Strain(s):";
	  }
	}
	if(numstrains<10){
	  ostr << ' ' << crI->read.getStrain();
	} else if(numstrains==10){
	  ostr << "...";
	}
	++numstrains;
      }
    }
  }

  uint32 span=4;

  BUGIFTHROW(span==0, "span for wiggle file cannot be 0");

  ostr << "\" visibility=full autoScale=off "
       << "viewLimits=0:" << maxcov << " color=0,200,100 "
       << "maxHeightPixels=100:50:20 graphType=bar priority=30\n"
       << "fixedStep chrom=" << getContigName() 
       << " start=1 step=" << span << " span=" << span << endl;

  ccI=CON_counts.begin();
  for(uint32 cpos=0; cpos<cons.size();){
    uint32 tcov=0;
    uint32 counter=0;
    uint32 maxto=min(cpos+span, static_cast<uint32>(cons.size()));
    bool haszerovalue=false;
    for(; cpos<maxto; cpos++, ccI++){
      if(cons[cpos] != '*'
	 && (cons[cpos] != '@' || (cons[cpos] == '@' && ccI->total_cov!=0))){
	if(ccI->total_cov){
	  tcov+=ccI->total_cov;
	}else{
	  haszerovalue=true; 
	  break;
	}
	counter++;
      }
    }
    if(haszerovalue || counter==0){
      ostr << "0\n";
    }else if(counter!=0) {
      ostr << tcov/counter << '\n';
    }
  }

  FUNCEND();
}

