/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 1997-2000 by the German Cancer Research Center (Deutsches
 *   Krebsforschungszentrum, DKFZ Heidelberg) and Bastien Chevreux
 * Copyright (C) 2000 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */

// 	$Id$	


#ifndef _mira_contig_h_
#define _mira_contig_h_

#include <assert.h>

#include <strings.h>

#include <iostream>
#include <iomanip>
#include <string>

#include "stdinc/stlincludes.H"

#include "stdinc/defines.H"
#include "errorhandling/errorhandling.H"

// for checking time restrictions
#include <time.h>
#include "mira/timerestrict.H"

//#include "mira/indexed_deque.H"
#include "mira/parameters.H"
#include "mira/readpool.H"
#include "mira/align.H"
#include "mira/ads.H"
#include "io/exp.H"
#include "io/gbf.H"

#include "util/dptools.H"





// structure used to give back information on GenBank tags in the contig
struct gbfsummary_t {
  uint32 cfrom;              // position of feature in contig
  uint32 cto;
  int32  readindex;          // index of read with this feature in CON_reads

  int32  strainid;
  string identifier;  // TODO: switch that to multitag_t::mte_id_t ??
  string locustag;
  string gene;
  string function;
  string ecnumber;
  string product;
  string note;
  int8   translationtable;
  int8   codonstart;
  int8   direction;
  bool   mustbetranslated;
  bool   isgene;             // Fgen, FCDS or one of the main RNAs

  friend ostream & operator<<(ostream &ostr, const gbfsummary_t & gbfs){
    ostr << "cfrom: " << gbfs.cfrom
	 << "\ncto: " << gbfs.cto
	 << "\nreadindex: " << gbfs.readindex
	 << "\nstrainid: " << gbfs.strainid
	 << "\nidentifier: " << gbfs.identifier
	 << "\nlocustag: " << gbfs.locustag
	 << "\ngene: " << gbfs.gene
	 << "\nfunction: " << gbfs.function
	 << "\necnumber: " << gbfs.ecnumber
	 << "\nproduct: " << gbfs.product
	 << "\nnote: " << gbfs.note
	 << "\ntranslationtable: " << static_cast<uint16>(gbfs.translationtable)
	 << "\ncodonstart: " << static_cast<uint16>(gbfs.codonstart)
	 << "\ndirection: " << static_cast<uint16>(gbfs.direction)
	 << "\nmustbetranslated: " << gbfs.mustbetranslated
	 << "\nisgene: " << gbfs.isgene << endl;
    return ostr;
  }
};


class Contig
{
public:
  enum{AS_TEXT=0, AS_HTML, AS_CAF, AS_MAF, AS_SAM, AS_ACE, AS_FASTA, AS_FASTAQUAL, 
	 AS_FASTAPADDED, AS_FASTAPADDEDQUAL, AS_GAP4DA, AS_TCS,
	 AS_DEBUG};

  enum addreadcode{ENOERROR=0, 
		   ENOALIGN,
		   EDROPINRELSCORE,
		   ETEMPLATEDIRECTION,
		   ETEMPLATESIZE,
		   ESRMBMISMATCH,
		   EDANGERZONE,
		   ESPECIALSRADDFAIL,
		   EREFIDNOTALLOWED,
		   EMAXCOVERAGEREACHED,
		   EFORCEDGROWTHNOTREACHED,
		   EGROWTHNOTALLOWED,
		   EUNSPECIFIED
  };




// Structure used for initialising a complete contig in initialiseContig(...)
  struct contig_init_read_t {
    int32 id;
    int32 offset_in_contig;
    int32 read_lclip;
    int32 read_rclip;
    int32 direction;
  };
  
  
  
  
  struct contigread_t{
    int32 orpid;              /* id of the original read in the readpool
				 can be < 0, then this means it is not
				 in the readpool.
				 At the moment this happens for coverage
				 mapping of short read */
    int32 offset;             // offset of good, clipped part in the read
    int32 direction;          // >0 forward, <0 complement
    Read  read;               /* read (might be modified and not be quite the
				 same as in the readpool) */

  friend ostream & operator<<(ostream &ostr, const contigread_t & cr){
    ostr << "CR: " << cr.orpid << "\t" << cr.offset << "\t" << cr.direction << "\t" << cr.read.getName() << endl;
    return ostr;
  }

//  ******************************************************************        
//  contigPosToUnclippedReadPos
//  We have a contig position and we want to know which position in the
//  UNCLIPPED FORWARD read corresponds to the contig position
//
//  ******************************************************************        

    inline int32 contigPos2UnclippedReadPos(const int32 contigpos) const
      {
	if (direction > 0) {
	  return contigpos - offset + read.getLeftClipoff();
	} else {
	  // FALSCH return ric.offset + ric.read.getRightClipoff() -1 - contigpos;
	  return read.getLenSeq() - read.getRightClipoff() + contigpos - offset;
	}
      };

    inline int32 contigPos2RawReadPos(const int32 contigpos) const
      {
	if (direction > 0) {
	  return contigPos2UnclippedReadPos(contigpos);
	} else {
	  return read.calcComplPos(contigPos2UnclippedReadPos(contigpos));
	}
      };


// *******************************************************************
//  unclippedReadPosToContigPos
//  We have a position in an UNCLIPPED read and we want to know the corresponding
//  position in the contig
//
//  ******************************************************************        
//weiter damit
    inline int32 unclippedReadPos2ContigPos(const int32 readpos) const
      {
	if (direction > 0) {
	  return readpos - read.getLeftClipoff() + offset ;
	} else {
	  return read.getRightClipoff()-readpos+offset -1 ;
	}
      }
    inline char getBase(const int32 contigpos) const
      {
	if(direction>0){
	  return read.getBaseInSequence(contigPos2UnclippedReadPos(contigpos));
	}else{
	  return read.getBaseInComplementSequence(contigPos2UnclippedReadPos(contigpos));
	}
      }
    inline base_quality_t getQuality(const int32 contigpos) const
      {
	if(direction>0){
	  return read.getQualityInSequence(contigPos2UnclippedReadPos(contigpos));
	}else{
	  return read.getQualityInComplementSequence(contigPos2UnclippedReadPos(contigpos));
	}
      }

  };
  

// we can currently merge 2 seqtypes: SOlexa, SOLiD
// SOLiD currently not used
#define NUMMERGESEQTYPES 2
  
  typedef uint32 ccctype_t;  // consensus_counts_count_type

  struct consensus_counts_t{
    ccctype_t A;             // ACGT are extended counters
    ccctype_t C;             // ccctype_t is enough for a coverage of 16k reads.
    ccctype_t G;
    ccctype_t T;
    
    ccctype_t N;             // N, X, star and coverage are normal counters.
    ccctype_t X;
    ccctype_t star;
    
    ccctype_t seqtype_cov[Read::SEQTYPE_END];  /* for each seqtype, count the
					     * coverage at this position. This
					     * number also includes the 100%
					     * mapped reads!
					     */
    ccctype_t total_cov;
    
    uint16 baselock;     // if > 0 then only one of A, C, T, G may be set, otherwise it's a misassembly
    
    uint16 snplock;

    // New for Short read mappings, but will also be used for normal
    //  backbone mappings
    // 

    uint8 forcemergearea;  /* if >0, short reads falling in this area will always
			      be merged, even if not mapping at 100% */
    

    // Currently, these MUST be at the end of this struct:
    //  MIRA depends on it by using CON_concounts_zero and 
    //  CON_concounts_zero_nobb to initialise quickly or rebuilding
    //  a consensus
    ccctype_t bbcounts[NUMMERGESEQTYPES];    /* coverage of reads supporting 100%
					     backbone 0 is Solexa, 1 is ABI
					     SOLiD */
    base_quality_t bbbestquals[NUMMERGESEQTYPES]; /* best quality of reads
						     supporting 100% backbone
						     0 is Solexa, 1 is ABI
						     SOLiD */
    uint8 bbstrains[NUMMERGESEQTYPES];   /* bitmap. which strains are mapped
					    at this pos max. 8 strains
					    possible. Strains are in 76543210
					    order */
    char   backbonechar;  /* original backbone consensus character (uppercase)
			     gaps are *, positions not initially covered by backbone
			     '@' */

    // that's more for debugging
    friend ostream & operator<<(ostream &ostr, const consensus_counts_t & cc){
      cout << "A: " << cc.A
	   << "\tC: " << cc.C
	   << "\tG: " << cc.G
	   << "\tT: " << cc.T
	   << "\tN: " << cc.N
	   << "\tX: " << cc.X
	   << "\t*: " << cc.star
	   << "\ttcov: " << cc.total_cov
	   << "\tblock: " << cc.baselock
	   << "\tslock: " << cc.snplock
	//<< "\n  arr"
	   << "\tbbc: " << cc.backbonechar
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
	   << "\tcSAN: " << cc.seqtype_cov[Read::SEQTYPE_SANGER]
	   << "\tc454: " << cc.seqtype_cov[Read::SEQTYPE_454GS20]
	   << "\tcION: " << cc.seqtype_cov[Read::SEQTYPE_IONTORRENT]
	   << "\tcPBS: " << cc.seqtype_cov[Read::SEQTYPE_PACBIO]
	   << "\tcSXA: " << cc.seqtype_cov[Read::SEQTYPE_SOLEXA]
	   << "\tcSID: " << cc.seqtype_cov[Read::SEQTYPE_ABISOLID]
	   << "\tbcSXA: " << cc.bbcounts[0]
	   << "\tbcSID: " << cc.bbcounts[1]
	   << "\tbqSXA: " << static_cast<ccctype_t>(cc.bbbestquals[0])
	   << "\tbqSID: " << static_cast<ccctype_t>(cc.bbbestquals[1])
	   << "\tsSXA: " << static_cast<ccctype_t>(cc.bbstrains[0])
	   << "\tsSID: " << static_cast<ccctype_t>(cc.bbstrains[1]);
      return ostr;
    }
  };

  // Contig exports the container with consensus_counts_t
  //  as type:  Contig::cccontainer_t

  typedef deque<consensus_counts_t> cccontainer_t;
  //typedef IndexedDeque<consensus_counts_t> cccontainer_t;


// Structure used for temporarily sorting read (needed for the << operator)
// also needed in other funcs for quick access to the contig 
struct out_order_t {
  int32 offset_start;
  int32 original_index;
  int32 orpid;
};




// we need an own tag type for the consensus as additional data must be 
//  stored
// lets use simple inheritance for that (first time for MIRA where
//  it is useful :-)
// Consensus tags have a quality identifier for each base (and *)
//  this is useful for example for SRMBs and SNPs
// This thing is a class of its own, especially
//  due to the copy operator for "upgrading" normal tags
// Lazy: no good constructors/destructors
// Contrary to the groups_t struct (farther below), only bases
//  which form a valid group have their quality set >0 here.

  class consensustag_t: public multitag_t {
  public:
    bool additionalinfo_initialised;

    base_quality_t qualACGTStar[5];  // qualities for A, C, G, T, *

    //base_quality_t qualA;
    //base_quality_t qualC;
    //base_quality_t qualG;
    //base_quality_t qualT;
    //base_quality_t qualStar;

    // TODO: find out where used, multitag_t???
    // upgrading copy operator: when copying a tag_t to contigtag_t
    Contig::consensustag_t const & operator=(tag_t const & other){
      from=other.from;
      to=other.to;
      strand=other.strand;
      identifier=multitag_t::newIdentifier(other.identifier);
      comment=multitag_t::newComment(other.comment);
      additionalinfo_initialised=false;

      qualACGTStar[0]=0;
      qualACGTStar[1]=0;
      qualACGTStar[2]=0;
      qualACGTStar[3]=0;
      qualACGTStar[4]=0;

      //qualA=0;
      //qualC=0;
      //qualG=0;
      //qualT=0;
      //qualStar=0;

      return *this;
    }

    // lazy constructor (need to define as copy-constructor is defined)
    consensustag_t() {}

    // upgrading copy constructor
    consensustag_t(tag_t const &other) {*this=other;}

  };

  struct constats_t {
    bool statsvalid;

    uint32 conlength;
    uint32 conlength_nogap;

    uint32 AinC;
    uint32 CinC;
    uint32 GinC;
    uint32 TinC;
    uint32 NinC;
    uint32 XinC;
    uint32 starInC;
    uint32 IUPACinC;
    uint32 FunnyInC;
    uint32 NinR;
    uint32 XinR;
    uint32 starInR;

    double gccontent;

    uint32 numSRMc;
    uint32 numWRMc;
    uint32 numSTMU;
    uint32 numSTMS;

    uint32 total_reads;
    uint32 readsperst[Read::SEQTYPE_END];    // EXCLUDING rail reads!!!
    uint32 readlenperst[Read::SEQTYPE_END];
    uint32 numreads_noqual;
    uint32 numreads_withqual;

    uint32 numnocoverage;

    uint32 max_coverage;
    uint32 max_covperst[Read::SEQTYPE_END];

    double avg_coverage;
    double avg_covperst[Read::SEQTYPE_END];

    uint32 avg_conqual;
  };


  struct errorstatus_t {
    int32         code;                   // the addread error code
    vector<int32> reads_affected;
    vector<int32> srmb_reads_affected;
    /* for mapping short reads: the contig proposes to cut back
       right quality clip to this position */
    int32         proposed_cutback;
  };


  // pacbio dark strobe edit
  struct pbdse_t {
    uint32 rid;               // rid in readpool
    int32  rposs;              // between read pos
    int32  rpose;              // and read pos
    int32  changeestim;        // >0: insert Ns; <0 delete Ns

    friend ostream & operator<<(ostream &ostr, const pbdse_t & pbdse){
      ostr << "rid: " << pbdse.rid << "\trposs: " << pbdse.rposs << "\trpose: " << pbdse.rpose << "\tce: " << pbdse.changeestim << endl;
      return ostr;
    }
  };

  // static variables
public:
  // Note: due to possible static initialization fiasco,
  //       the REA_tagentry_* variables are defined/initialised
  //       in "mira/multitag.C" and not in "mira/contig.C"
  static const multitag_t::mte_id_t CON_tagentry_idEmpty;
  static const multitag_t::mte_id_t CON_tagentry_idALUS;
  static const multitag_t::mte_id_t CON_tagentry_idREPT;
  static const multitag_t::mte_id_t CON_tagentry_idSRMc;
  static const multitag_t::mte_id_t CON_tagentry_idWRMc;
  static const multitag_t::mte_id_t CON_tagentry_idSAOc;
  static const multitag_t::mte_id_t CON_tagentry_idSROc;
  static const multitag_t::mte_id_t CON_tagentry_idSIOc;
  static const multitag_t::mte_id_t CON_tagentry_idFpAS;
  static const multitag_t::mte_id_t CON_tagentry_idPSHP;
  static const multitag_t::mte_id_t CON_tagentry_idED_D;
  static const multitag_t::mte_id_t CON_tagentry_idED_C;
  static const multitag_t::mte_id_t CON_tagentry_idED_I;
  static const multitag_t::mte_id_t CON_tagentry_idESDN;
  static const multitag_t::mte_id_t CON_tagentry_idSTMS;
  static const multitag_t::mte_id_t CON_tagentry_idSTMU;
  static const multitag_t::mte_id_t CON_tagentry_idUNSc;
  static const multitag_t::mte_id_t CON_tagentry_idIUPc;
  static const multitag_t::mte_id_t CON_tagentry_idMCVc;
  static const multitag_t::mte_id_t CON_tagentry_idDGPc;

  static const multitag_t::mte_co_t CON_tagentry_coEmpty;

  // special flag for debugging sessions. If set, the code should
  //  abort via a sigsev so that a debugger can catch it
  // should never ever happen in distributed packes, always commment out
  //  the code which performs the bombing
  static bool CON_abortflag;

private:
  static const consensus_counts_t CON_concounts_zero;
  static const consensus_counts_t CON_concounts_zero_nobb;

  static uint32 CON_id_counter;
  static uint32 CON_static_ok;

  static uint32 CON_railcounter;

  static uint32 CON_cer_idcounter;     // Coverage Equivalent Read id counter

  //static vector< string >   CON_danger_zones_ids;
  //static vector< string >   CON_baselock_ids;
  //static vector< string >   CON_snplock_ids;

  static vector<multitag_t::mte_id_t>   CON_danger_zones_ids;
  static vector<multitag_t::mte_id_t>   CON_baselock_ids;
  static vector<multitag_t::mte_id_t>   CON_snplock_ids;



  static uint8 CON_outtype;

  static bool  CON_outputRails;

  //Variables
private:
#define CON_VALID (((static_cast<uint32>('C')<<8)\
                     +static_cast<uint32>('O)'<<8)\
                     +static_cast<uint32>('N')<<8)\
                     +static_cast<uint32>('V'))
  uint32 CON_valid;

  bool CON_cebugflag; // only for special debugging in getConsensus() etc.

  vector<MIRAParameters> * CON_miraparams;


  uint32 CON_id;
  string CON_nameprefix;

  mutable string CON_name;  /* if this string is non-empty, then it's the name of
			       the contig. If it is empty, the name is made out of
			       the name prefix, some characters and the id.
			       see getContigName() for exact name scheme */

  bool CON_finalised;

  ReadPool * CON_readpool;
  vector<contigread_t>      CON_reads;
  cccontainer_t             CON_counts;
  multiset<int32>           CON_templates_present;
  vector<consensustag_t>    CON_consensus_tags;

  // if set and if an element >0, use this for checking free coverage
  //  when trying to add a read
  vector<ccctype_t>            CON_targetcoverageperst;

  /*
    This vector is here to save time in addRead() (other functions
    perhaps later)
    If active, it has the size of the readpool (CON_readpool) and 
    it maps the readpool ids of the reads to the ids of the reads
    in the CON_reads vector.
    If dirty, size is 0.
   */
  vector<int32> CON_maprpids_to_conreadsids;

  /*
    This vector is here to allow selective beef up the contig when mapping
    short reads.
    If active, it has the size of the readpool (CON_readpool) and each 
    element is either 1 (allowed) or 0 (not allowed), telling which
    reads are allowed or not to be a reference read when adding new
    reads to a contig.
    If dirty, size is 0.
   */
  vector<bool> CON_allowedrefids;


  //// fast cons function cached
  //uint32 CON_len_tmpcons;
  //char * CON_tmpcons;

  string CON_2tmpcons;

  /* Two flags that tell the contig how to add new reads:
   */

  bool CON_tmpcons_from_backbone;   /* if true, builds tmp consensus from
				       backbone chars if available */
  bool CON_mergenewsrreads;         /* if true, tries to use read merging
				       for short reads (Solexa, SOLiD */
  bool CON_hasforcemergeareas;      /* if true, some areas in the backbone
				       ask for forced merging */

  // Special conditions when aligning short reads (Solexa, SOLiD)
  bool CON_specialsraddconditions;   /* if true, applies special checks
					when adding short reads */
  int32 CON_ssrc_maxtotalerrors; // = to the sum of following two
  int32 CON_ssrc_maxgaps;        /* max number of gaps allowed for adding a
				    short read when using special SR
				    conditions */
  int32 CON_ssrc_maxmismatches;  /* max number of mismatches allowed for
				    adding a short read when using special SR
				    conditions */

  uint32 CON_nummergedreads_perseqtype[NUMMERGESEQTYPES];

  uint32 CON_index_leftestbbread;
  uint32 CON_index_rightestbbread;

  /*
    The "fixed" consensus is a consensus that the contig gets initialised with
    (e.g. loading from CAF). If set, then fetching consensus sequence and
    qualities will yield this instead of dynamically generated values from the
    CON_allcons* and CON_straincons* variables below. 

    Gets trashed when the contig changes)
   */
  string                 CON_fixedconsseq;
  vector<base_quality_t> CON_fixedconsqual;

  // new
  // when consenus is calculated, store here for every strain
  // it will be up to the caller to trash these values for recalc with other parameters
  // exception: if contig gets changed (added read, removed read etc.pp), these get trashed
  //  automatically by implicitly called definalise()
  // strains with no read in a given contig will still have the vectors filled (char '@', qual 0)

  int32          CON_conscalc_mincov;
  base_quality_t CON_conscalc_minqual;
  char           CON_conscalc_missingchar;

  string                  CON_allconsseq;
  vector<base_quality_t>  CON_allconsqual;
  vector<int32>           CON_alladjustments;

  vector<string>                  CON_strainconsseq;
  vector<vector<base_quality_t> > CON_strainconsqual;
  vector<vector<int32> >          CON_strainadjustments;

//  // TODO: old! to be phases out asap
//  // ---
//  string CON_cheat_intelcons;
//  vector<base_quality_t> CON_cheat_intelconsqual;
//  bool CON_cheat_intelcons_markspecials;
//  int32 CON_cheat_intelcons_mincov;
//  base_quality_t CON_cheat_intelcons_minqual;
//  int32 CON_cheat_intelcons_strainidtotake;
//  // ---


  // counter for num reads per strain. kept up to date by contig
  vector<uint32>                  CON_readsperstrain;


  //
  vector<out_order_t> CON_outputorder;

  vector<int32> CON_last_dangerous_overlaps;


  constats_t CON_stats;


  struct edit454command_t {
    uint32 conreadid;
    char base;
    uint32 readpos;
  };


  // set by pathfinder if the assembly mode is to keep long repeats 
  //  separated and this contig started with a multicopy read
  bool CON_contains_long_repeats_only;


public:


  //Functions
private:
  void foolCompiler();
  void init();
  void zeroVars();
  void definalise();

  //bool outputOrderSortCmp_(const out_order_t & a, const out_order_t & b);

  void buildFASTAHeader(ostringstream & ostr) const;

  void dumpAsTextOrHTML(ostream &ostr, 
			const uint8 outtype, 
			const string & consseq, 
			const vector<base_quality_t> & consqual, 
			const int32 frompos, 
			const int32 topos);
  void dumpAsDebug(ostream & ostr);
  void dumpAsCAF(ostream & ostr);
  void dumpAsMAF(ostream & ostr);
  void dumpAsSAM(ostream & ostr);
  int32 helper_dumpAsACE_BSLines(string & consseq, bool alsodump, ostream & ostr);
  void dumpAsACE(ostream & ostr);
  void dumpAsTCS(ostream & ostr);
  void dumpAsFASTA(ostream & ostr, bool padded);
  void dumpAsFASTAQUAL(ostream & ostr, bool padded);

  bool makeTmpConsensus(int32 from, int32 to);
  void updateCountVectors(const int32 from,
			  const int32 len,
			  vector<char>::const_iterator updateseq,
			  const uint32 seqtype,
			  bool addiftrue);
  void rebuildConCounts();
  void insertReadInContig(const AlignedDualSeq & ads, uint32 coffset,
			  int32 direction_frnid, int32 direction_global);
  bool insertMappedReadInContig(const AlignedDualSeq & ads, 
				const uint8 newreadseqtype,
				const uint32 coffset, 
				const int32 direction_frnid,
				const bool forcemerge);
  int32 analyseADSForCuttingBackCERMap(const AlignedDualSeq & ads, int32 direction_frnid);

  void finalise();
  void checkContig();
  void updateTagBaseInserted(uint32 contigpos);
  void updateTagBaseDeleted(uint32 contigpos);

  int deepThought(const contigread_t &aRead, int offset, char base);
		  //		  SCF_buffer &theBuffer);
  bool analyseRMBZones(const contigread_t & ric, vector<int32> & RMBdangeroverlaps);
  bool analyseDangerZones(const contigread_t & ric);
  bool analyseAllZones(const contigread_t & ric);

  void checkForLockedBase(const uint32 readpos,
			  const contigread_t ric,
			  int32 & baselock,
			  int32 & snplock);
  void checkForLockedBaseComplement(const uint32 readpos,
				    const contigread_t ric,
				    int32 & baselock,
				    int32 & snplock);

  void trimEnds();


  // TODO: see whether can be replace by the looking at sorted multitag_t vector?
  static bool consensustag_t_comparator(const consensustag_t & t1, 
					const consensustag_t & t2)
    {
      if(t1.from < t2.from) return true;
      if(t1.from > t2.from) return false;
      if(t1.to < t2.to) return true;
      if(t1.to > t2.to) return false;
      return (t1.getIdentifierStr() < t2.getIdentifierStr());
    };


  // rid high to low, within rid startpos high to low
  static bool pbdse_t_comparator(const pbdse_t & p1, 
				 const pbdse_t & p2)
    {
      if(p1.rid == p2.rid){
	return (p1.rposs < p2.rposs);
      }
      return p1.rid < p2.rid;
    };

  /* Sort ascending by offset start
     if equal, subsort by read_id */
  static bool out_order_t_comparator(const out_order_t & a, 
				     const out_order_t & b)
    {
      if(a.offset_start == b.offset_start){
	return (a.orpid < b.orpid);
      }
      return a.offset_start < b.offset_start;
    };


  // sort ascending, first by contigfrom, then to
  //  then by identifier: Fsrc, Fgen, FCDS, rest
  static bool gbfsummary_t_comparator(const gbfsummary_t & a, 
				      const gbfsummary_t & b)
    {
      if(a.cfrom!=b.cfrom) return (a.cfrom<b.cfrom);
      if(a.cto!=b.cto) return (a.cto<b.cto);
      if(a.identifier!=b.identifier) {
      	if(a.identifier.empty()) return true;
      	if(b.identifier.empty()) return false;
      	if(a.identifier=="Fsrc") return true;
      	if(b.identifier=="Fsrc") return false;
      	if(a.identifier=="Fgen") return true;
      	if(b.identifier=="Fgen") return false;
      	if(a.identifier=="FCDS") return true;
      }

      return false;
    };


  static bool edit454command_t_comparator(const edit454command_t & a, 
					  const edit454command_t & b)
    {
      if(a.conreadid == b.conreadid) {
	// sort within a conreadid by DESCENDING! readpos order
	return a.readpos > b.readpos;
      }else{
	return a.conreadid < b.conreadid;
      }
    };

  ///////////////////////////////////////////////////////

  class nngroups_t {
  public:
    char base;
    base_quality_t groupquality;

    bool valid;
    
    uint32 forwarddircounter;
    uint32 complementdircounter;

    // these vectors are 'linked': nth element of one corresponds to nth of another
    vector<int32> ids;
    vector<base_quality_t> quals;
    vector<int8>  directions;

    friend ostream & operator<<(ostream &ostr, const nngroups_t &nngroup){
      ostr << "    base " <<  nngroup.base << '\n';
      ostr << "    valid " <<  nngroup.valid << '\n';
      ostr << "    forwarddircounter " <<  nngroup.forwarddircounter << '\n';
      ostr << "    complementdircounter " <<  nngroup.complementdircounter << '\n';
      ostr << "    groupquality " <<  static_cast<uint32>(nngroup.groupquality) << '\n';
      ostr << "    ids.size " <<  nngroup.ids.size() << '\n';
      ostr << "    quals.size " <<  nngroup.quals.size() << '\n';
      ostr << "    directions.size " <<  nngroup.directions.size() << '\n';
      return ostr;
    }
  };

  ////////////////////////////////////////////////////////////////////////

  struct pos_rep_col_t {
    int32 contigpos;
    int32 numvalidgroups;
    vector<nngroups_t> groups;
    //vector<int32> allids;   not needed anyway (just for lazy debugging output)
    string type;

    bool is_dangerous;
    bool tagged;
  };
  void mpr__tagColumn(pos_rep_col_t & prc);
  ///////////////////////////////////////////////////////


  //void calcGroupQual(const groups_t & g);

  void makeIntelligentConsensus_helper3(
    char & thisbase, 
    base_quality_t & thisqual,
    const vector<nngroups_t> & groups, 
    const vector<char> & IUPACbasegroups);

  void makeIntelligentConsensus_helper2_calcSangerQual(
    char & thisbase,
    base_quality_t & thisqual,
    const uint32 actcontigpos,
    const vector<nngroups_t> & groups,
    vector<char> & IUPACbasegroups,
    const vector<char> & columnbases,
    const base_quality_t maxqual,
    const uint32 maxcount);

  void makeIntelligentConsensus_helper2_calcSOLEXA(
    char & thisbase, 
    base_quality_t & thisqual, 
    const uint32 actcontigpos,
    cccontainer_t::const_iterator ccI,
    const vector<nngroups_t> & groups, 
    vector<char> & IUPACbasegroups, 
    const base_quality_t maxqual,
    const uint32 maxcount,
    int32 strainidtotake);

  void makeIntelligentConsensus_helper2_calc454GS20(
    char & thisbase, 
    base_quality_t & thisqual, 
    const uint32 actcontigpos,
    const vector<nngroups_t> & groups, 
    vector<char> & IUPACbasegroups);

  void makeIntelligentConsensus_helper2_calcIonTorrent(
    char & thisbase, 
    base_quality_t & thisqual, 
    const uint32 actcontigpos,
    const vector<nngroups_t> & groups, 
    vector<char> & IUPACbasegroups);

  void makeIntelligentConsensus_helper2_calcPACBIO(
    char & thisbase, 
    base_quality_t & thisqual, 
    const uint32 actcontigpos,
    const vector<nngroups_t> & groups, 
    vector<char> & IUPACbasegroups);

  void makeIntelligentConsensus_helper1(
    char & thisbase, 
    base_quality_t & thisqual,
    const uint32 actcontigpos, 
    cccontainer_t::const_iterator ccI,
    const int32 mincoverage, 
    vector<nngroups_t> & groups,
    vector<nngroups_t> & maskedshadowgroups, 
    vector<char> & IUPACbasegroups, 
    vector<char> & columnbases, 
    const vector<int32> & read_ids_in_col, 
    vector<int8> & maskshadow,
    uint8 actreadtype,
    int32 strainidtotake,
    char missingcoveragechar
    );
  void makeIntelligentConsensus(string & target, 
				vector<base_quality_t> & qual, 
				vector<int32> * targetadjustments, 
				vector<int32> * targetadjustments_bla, 
				int32 from, 
				int32 to, 
				int32 mincoverage=0,
				base_quality_t minqual=0,
				int32 strainidtotake=-1,
				char missingcoveragechar='@');
//				ostream * ostr=NULL,
//				bool contagsintcs=true);
  
  int8 rateGoodnessLevelOfConsensus(cccontainer_t::const_iterator ccI, 
				    nngroups_t & groups, 
				    uint32 numpossiblebases, 
				    uint8 seqtype);


  void buildMaskShadow(vector <int8> & maskshaddow, 
		       vector<multitag_t::mte_id_t> & masktagtypes,
		       bool onlybackbone);
  void zeroStats();
  void calcStats();


  // ReadColContigIterator structure
  struct rcci_t {
    // a vector (read_ids_in_col) keeps the ids of the CON_reads that are
    //  covering a specific position of the contig
    // reserving 1000 position should be enough 99.9% of all cases,
    //  is automatically extended by STL if needed.
    vector<int32> read_ids_in_col; /* these are the CON_read[] indexes
				    */

    vector<out_order_t>::const_iterator outorderI;
    uint32 actcontigpos;

    vector<int32> allowedstrainids;
    vector<uint8> allowedreadtypes;
    bool takerails;
    bool takebackbones;
    bool takereadswithoutreadpool;
  };
  void readColContigIteratorInit(rcci_t & rcci, 
				 const vector<int32> & allowedstrainids,
				 const vector<uint8> & allowedsequencingtypes,
				 bool takerails,
				 bool takebackbones,
				 bool takereadswithoutreadpool);
  void readColContigIteratorUpdate(rcci_t & rcci);
  void readColContigIteratorAdvance(rcci_t & rcci);

  struct ercci_t {
    // per sequencing type and per strain ids,
    //  a vector keeps the ids of the reads that are
    //  covering a specific position of the contig
    // reserving 1000 position should be enough 99.9% of all cases,
    //  is automatically extended by STL if needed.
    
    // read ids per sequencing type per strain
    vector<vector<vector<uint32> > > rids_st_st;

    vector<out_order_t>::const_iterator outorderI;
    uint32 actcontigpos;

    bool takerails;
    bool takebackbones;

  };
  void extendedReadColContigIteratorInit(ercci_t & ercci, 
					 bool takerails,
					 bool takebackbones,
					 uint32 numstrains);
  void extendedReadColContigIteratorUpdate(ercci_t & ercci);
  void extendedReadColContigIteratorAdvance(ercci_t & ercci);


  ////////////////////////////////////////////////////////////////////////
  struct nnpos_rep_col_t {
    // the next two are linked, i.e., 1st value of each must
    //  be seen together, 2nd ... etc.
    string groupbases;
    vector<base_quality_t> groupquals;

    // which ids are in these groups
    vector<uint32> ids;

    multitag_t::mte_id_t type;       // fille with Read::REA_tagentry_id...
    int32 contigpos;

    bool is_dangerous;
    bool tagged;

    friend ostream & operator<<(ostream &ostr, const nnpos_rep_col_t & prc){
      ostr << "PRC:\n";
      ostr << "    is_dangerous: " <<  prc.is_dangerous << '\n';
      ostr << "    tagged: " <<  prc.tagged << '\n';
      ostr << "    groupbases: " <<  prc.groupbases << '\n';
      ostr << "    type: " <<  multitag_t::getIdentifierStr(prc.type) << '\n';
      ostr << "    contigpos: " <<  prc.contigpos << '\n';
      return ostr;
    }
  };

public:
  struct repeatmarker_stats_t {
    uint32 numSRMs;
    uint32 numWRMs;
    uint32 numSNPs;
  };
private:
  ////////////////////////////////////////////////////////////////////////

  void nmpr_firstfillin(const ercci_t & ercci, 
			const vector <int8> & maskshaddow,
			const vector<multitag_t::mte_id_t> & masktagtypes,
			vector<vector<vector <nngroups_t> > > & groups_st_st);
  void nmpr_rategroups(vector<vector<vector <nngroups_t> > > & groups_st_st,
		       cccontainer_t::const_iterator ccI);
  void calcGroupQual(const nngroups_t & g);
  void nmpr_secondfillin(const ercci_t & ercci, 
			 const vector <int8> & maskshaddow,
			 const vector<multitag_t::mte_id_t> & masktagtypes, 
			 vector<vector<vector <nngroups_t> > > & groups_st_st);
  void nmpr_cautiousMultiSeqTypeTagging(const ercci_t & ercci,
					const rcci_t & rcci,
					const vector<vector<vector <nngroups_t> > > & groups_st_st,
					const nnpos_rep_col_t & emptyprc,
					vector<bool> & readsmarkedsrm,
					repeatmarker_stats_t & repstats);
  void nmpr_evaluateOneSeqType(const uint32 actseqtype,
			       const ercci_t & ercci, 
  			       const vector<vector <nngroups_t> > & groups_st,
			       vector<nnpos_rep_col_t> & newprc,
			       const nnpos_rep_col_t & emptyprc);
  uint32 nmpr_appendPRCFieldsWithGroupsOfOneStrain(
    const vector <nngroups_t> & groups,
    nnpos_rep_col_t & newprc);

  void nmpr_tagColumn(nnpos_rep_col_t & prc, 
		      const rcci_t & rcci,
		      vector<bool> & readsmarkedsrm,
		      repeatmarker_stats_t & repstats);


  void coutAddReadClocks(const vector<suseconds_t> & suv);
  void getRailsAsReadsAffected(const int32 refid, 
			       vector<int32> & reads_affected,
			       const int32 xcut,
			       const int32 ycut);

  bool checkFreeCoverageForAddingRead(
    const uint8 newreadseqtype, 
    const int32 xcut, 
    const int32 ycut);

  inline uint8 getBBStrainMask(int32 strainid) {
    int32 strainmask=0;
    if(strainid>7 || strainid<0) {
      throw Notify(Notify::FATAL, "uint8 Contig::getBBStrainMask(int32 strainid)", "Mapping >=8 Solexa strains? Sorry, not possible yet.");
    }
    BITSET(strainid, strainmask);
    return static_cast<uint8>(strainmask);
  }
  void getMappedBBStrainIDsFromMask(vector<int32> & strains, uint8 mask);

  void myappend(string & a, const string & b, const string & concatstring) const;


public:
  Contig(vector<MIRAParameters> * params, ReadPool & readpool);
  Contig(Contig const &other);
  ~Contig();

  void reserveCapacity(uint32 expectedlen, uint32 expectedreads);
  
  void dumpStatus(ostream & ostr);
  void dumpAsText(ostream &ostr, 
//		  const string & consseq, 
//		  const vector<base_quality_t> & consqual, 
		  const int32 frompos = -1, 
		  const int32 topos = -1);
  void dumpAsHTML(ostream &ostr, 
//		  const string & consseq, 
//		  const vector<base_quality_t> & consqual, 
		  const int32 frompos = -1, 
		  const int32 topos = -1);

  void dumpStrainAsFASTAQUAL(ostream & fout,
			     ostream & qout, 
			     uint32 mincoverage,
			     base_quality_t minqual, 
			     int32 strainidtotake,
			     bool padded,
			     bool fillholesinstrain);
  void dumpStrainAsFASTQ(ostream & fout,
			 uint32 mincoverage,
			 base_quality_t minqual, 
			 int32 strainidtotake,
			 bool padded,
			 bool fillholesinstrain);

  void dumpConReads();

  static void dumpContigStatistics_Head(ostream &ostr);
  void dumpContigStatistics_Body(ostream &ostr);
  static void dumpContigReadList_Head(ostream &ostr);
  void dumpContigReadList_Body(ostream &ostr);
  static void dumpReadTagList_Head(ostream &ostr);
  void dumpReadTagList_Body(ostream &ostr);
  static void dumpConsensusTagList_Head(ostream &ostr);
  void dumpConsensusTagList_Body(ostream &ostr);
  static void dumpTCS_Head(ostream &ostr);
  void dumpTCS_Body(ostream &ostr);
  static void dumpWiggle_Head(ostream &ostr);
  void dumpWiggle_Body(ostream &ostr, string & cons);

  Contig const & operator=(Contig const & other);
  friend ostream & operator<<(ostream &ostr, Contig const &con);
  void saveAsGAP4DA(const string & dirname, ostream & fofnstr);

  void saveMem();

  void stats(ostream &ostr);
  const constats_t & getStats() { calcStats(); return CON_stats; };

  void setParams(vector<MIRAParameters> * params);

  static void setCoutType(uint8 type);

  uint32 getContigID() const {return CON_id;};
  void   setContigID(uint32 id) {CON_id=id;};
  void   setContigNamePrefix(const string & pname) {CON_nameprefix=pname;};
  const string & getContigName() const;
  void   setContigName(const string & name);

  void trashConsensusCache(bool qualonly) {
    nukeSTLContainer(CON_fixedconsqual);
    if(!qualonly) nukeSTLContainer(CON_fixedconsseq);
  };
//  void OLDgetConsensus1(string & target, 
//		    vector<base_quality_t> & qual, 
//		    bool markspecials,
//		    int32 mincoverage=0,
//		    base_quality_t minqual=0,
//		    int32 strainidtotake=-1,
//		    char missingcoveragechar='@',
//		    ostream * ostr=NULL,
//		    bool contagsintcs=true);

  void calcConsensi(int32 mincoverage=0,
		    base_quality_t minqual=0,
		    char missingcoveragechar='@');
  void newConsensusGet(string & target, 
		       vector<base_quality_t> & qual, 
		       int32 strainidtotake=-1);
  void ensureConsensus(int32 strainidtotake=-1);

  uint32 getNumReadsInContig() const {return static_cast<uint32>(CON_reads.size());};
  uint32 getContigLength() const {return static_cast<uint32>(CON_counts.size());};

  bool getLongReapeatStatus() const {return CON_contains_long_repeats_only;}
  void setLongReapeatStatus(bool b) {CON_contains_long_repeats_only=b;}

  void discard();

  void useBackbone4TmpConsensus(bool b) {CON_tmpcons_from_backbone=b;};
  void mergeNewSRReads(bool b) {CON_mergenewsrreads=b;};
  void setSpecialSRAddConditions(const int32 maxtotalerrors,
				 const int32 maxgaps,
				 const int32 maxmismatches);
  void allowedRefIDs_forbidAll();
  void allowedRefIDs_allowAll();
  void allowedRefIDs_allowSomeRailsOnShortReadCoverage(
    int32 belowcoverage, 
    int32 strainid,
    uint8 seqtype);
  const vector<bool> & getAllowedRefIDs() {return CON_allowedrefids;};

  void setContigCoverageTarget(vector<uint32>);
  void addFirstRead(int32 id, uint32 direction);
  void addRead(
    vector<Align> & aligncache,
    const AlignedDualSeqFacts * ads,
    int32 refid,
    int32 newid,
    int32 direction_frnid,
    bool  newid_ismulticopy,
    int32 forcegrow,
    errorstatus_t & errstat);
  void addRead_wrapped(
    vector<Align> & aligncache,
    const AlignedDualSeqFacts * ads,
    int32 refid,
    int32 newid,
    int32 direction_frnid,
    bool  newid_ismulticopy,
    int32 forcegrow,
    errorstatus_t & errstat);

  void addConsensusSequenceAsReadToContig(int32 strainid);

  void deleteRead(int32 id);
  const vector<int32> & getLastDangerzoneOverlapIDS() const {return CON_last_dangerous_overlaps;};

  void initialiseContig(uint32 contig_length, 
			const list<contig_init_read_t> & rlist,
			vector<tag_t> & tags,
			const string & contigname,
			string & fixedseq,
			vector<base_quality_t> & fixedqual);

  void markPossibleRepeats(uint32 & numSRMBs, 
			   uint32 & numWRMBs, 
			   uint32 & numSNPs);
  //void markMissedSNPs();
  void markFeaturesByConsensus(bool markSROs, bool markUNSs, bool markIUPACs);

  void newMarkPossibleRepeats(repeatmarker_stats_t & repstats, 
			      vector<bool> & readsmarkedsrm);

  uint32 getBaseRunLength(const Read & read, 
			  const uint32 readpos, 
			  const char base,
			  uint32 & from, 
			  uint32 & to, 
			  uint32 & zeroqualcounts,
			  bool useclipsasends=false);
  void largeGapRepeatMarker(uint32 mingaplen,
			    uint32 mingroupsize,
			    bool needbothstrands);
  void codonSingleBaseRepeatMarker(uint32 mingroupsize,
				   repeatmarker_stats_t & repstats, 
				   vector<bool> & readsmarkedsrm);
  void csbrm_fillin_groups_stst(
    const ercci_t & ercci,
    const vector<int8> & maskshadow,
    const vector<multitag_t::mte_id_t> & masktagtypes,
    vector<vector<vector <nngroups_t> > > & groups_st_st);
  bool csbrm_checkReadsForHomopolymers(uint32 actcontigpos,
				       uint32 maxrunallowed,
				       const vector<int32> & checkthesereads);
  bool csbrm_checkForCleanBlock(cccontainer_t::iterator ccI);

  char getBaseInRead(int32 contigposition,
		     int32 readindex) const;
  base_quality_t getQualityInRead(int32 contigposition,
		     int32 readindex) const;
  void changeBaseInRead(char base,
			int32 contigposition,
			int32 readindex, 			
			base_quality_t quality=BQ_UNDEFINED);
  void insertBaseInRead(char base,
			int32 contigposition,
			int32 readindex,
			base_quality_t quality=BQ_UNDEFINED);
  void deleteBaseInRead(int32 contigposition, int32 readindex);


  void adjustReadOffsets(int32 contigposition, int32 offset);
  void deleteStarOnlyColumns(int32 from, 
			     int32 to,
			     bool alsononly=false,
			     uint32 mincov=1,
			     bool updateconcounts=true);
  uint32 createPacBioDarkStrobeEdits(list<pbdse_t> & pbdsev);

  void addTagToRead(const uint32 contigpositionfrom,
		    const uint32 contigpositionto,
		    const int32 readindex,
		    const string & identifier,
		    const string & comment);
  void addTagToRead(const uint32 contigpositionfrom,
		    const uint32 contigpositionto,
		    const int32 readindex,
		    const char * identifier,
		    const char * comment);

  void addTagToConsensus(const uint32 contigpositionfrom,
			 const uint32 contigpositionto,
			 const char strand, 
			 const char * identifier,
			 const char * comment,
			 const bool doublecheck,
			 const bool additionalinfo=false,
			 const base_quality_t qualA=0,
			 const base_quality_t qualC=0,
			 const base_quality_t qualG=0,
			 const base_quality_t qualT=0,
			 const base_quality_t qualStar=0);

  // first function to use strings
  //bool hasConsensusTag(const string & identifier) const;
  bool hasConsensusTag(const multitag_t::mte_id_t identifier) const;

  void sortConsensusTags() {sort(CON_consensus_tags.begin(),CON_consensus_tags.end(),consensustag_t_comparator);};
  const vector<consensustag_t> & getConsensusTags() const {return CON_consensus_tags;};
  void clearConsensusTags() {CON_consensus_tags.clear();};
  //void deleteTagsInReads(const string & identifier);
  void deleteTagsInReads(const multitag_t::mte_id_t identifier);
  void reduceConsensusTags(const vector<multitag_t::mte_id_t> & mastertags,
			   const vector<multitag_t::mte_id_t> & deletecandidates);

  void updateStatsFromConsensusTags(bool countSRMcs,
				    bool countWRMcs, 
				    bool countIUPACs,
				    bool countSTMUs,
				    bool countSTMSs);

  const Read & getReadAtIndex(int32 index) const;
  const Read & getReadWithID(int32 readid) const;

  static void setIDCounter(uint32 num) { CON_id_counter=num;}
  const cccontainer_t & getConsensusCounts() const
    { return CON_counts;}
  const vector<contigread_t> & getContigReads() const
    { return CON_reads;}
  vector<int32> & getCRIndicesAtContigPosition(vector<int32> & vec,
                                               int32 pos1,
                                               int32 pos2=-1) const;
  vector<int32> & getReadIDsAtContigPosition(vector<int32> & vec, 
					     int32 pos1, 
					     int32 pos2=-1) const;

  //  int32 getContigPosAtReadPos(int32 readpos, int32 readindex) const;

  int32 unclippedReadPosToContigPos(const int32 readpos, const contigread_t &ric) const;
  int32 contigPosToUnclippedReadPos(const int32 contigpos, const contigread_t &ric) const;
  int32 getRealReadPos(const int32 contigpos, const uint32 readindex) const;
  int32 getRealReadPos(const int32 contigpos, const contigread_t &ric) const;

  int32 paddedPos2UnpaddedPos(uint32 padpos);

  void interpolateSRMValuesInCONcounts(cccontainer_t::iterator ccI);
  void initialiseBaseLocks();
  void updateReadTagsFromReadPool();
  void updateBaseLocks(const contigread_t & ric, bool addiftrue);
  void transposeReadSRMTagsToContig();

  size_t addRails(const uint32 raillength, 
		const uint32 railoverlap,
		const string & straintxt,
		const bool forcestrainset,
		const string & railfromstrain,
		const bool simulateonly);

  size_t addSubsequences(const uint32 raillength, 
			 const uint32 railoverlap,
			 const string & straintxt,
			 const bool forcestrainset,
			 const string & railfromstrain,
			 const bool asbackbone,
			 const string & bbseq, 
			 const vector<base_quality_t> & bbqualvec, 
			 const string & backbonename,
			 const bool initccbbvalues,
			 const bool simulateonly);

//  void addRails(const uint32 raillength, 
//		const string & straintxt, 
//		const int32 bbquals, 
//		const bool asbackbone = false,
//		const string & backbonename = "");

  void removeRails();
  static void resetCERNumbering() {CON_cer_idcounter=0;};
  void transformCERMappingsToCoverageReads();
  void recalcTemplateAndStrainIDsPresent();

  void getGBFSummary(list<gbfsummary_t> & allGBfeatures, 
		     const vector<multitag_t::mte_id_t> & allowedfeatures, 
		     const vector<multitag_t::mte_id_t> & forbiddenfeatures,
		     bool simulateintergenics) const; 
  void concatAllGBFInfoForLocus(const list<gbfsummary_t> & allGBfeatures,
				list<gbfsummary_t>::const_iterator gbfsI,
				const string & concatstring,
				string & gene,
				string & function,
				string & ecnumber,
				string & product,
				string & note) const;

  bool hasEditableOvercallData() const;
  bool hasPacBioData() const;

  uint32 editTrickyOvercalls(const bool onlysetPSHPtags,
				const bool noSRMreadsallowed,
				vector<bool> & readsmarkedsrm);
  uint32 edit454checkTrickies(const char basehypo, 
			      const uint32 actcontigpos, 
			      const vector<int32> & read_ids_in_col,
			      vector<edit454command_t> & editcommands,
			      const bool onlysetPSHPtags,
			      const bool noSRMreadsallowed);
  uint32 editSingleDiscrepancyNoHAFTag(vector<bool> & readsmarkedsrm);


  bool analyseReadCoverage(vector<uint32> & maxcovperread, 
			   vector<uint8> & multicopies,
			   vector<uint32> covperst);    // no reference here!

  void setupAsBackBoneContig();

  void blindContig();
};


#include "examine/scf_look.H"


#endif



