/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "dvb_cam.hh"
#include "exception_handler.hh"

#include <libdvbapi/dvbdemux.h>
#include <libdvbapi/dvbaudio.h>
#include <libucsi/mpeg/section.h>

class DvbCamHandlerLLCI : public DvbCamHandler
{
public:
	DvbCamHandlerLLCI();
	~DvbCamHandlerLLCI();

private:
	struct SlResource
	{
		en50221_app_public_resource_id resid;
		uint32_t binary_resource_id;
		en50221_sl_resource_callback callback;
		void *arg;
	};

	static int llci_rm_enq_callback(void *arg, uint8_t slot_id, uint16_t session_number);
	static int llci_rm_reply_callback(void *arg, uint8_t slot_id, uint16_t session_number, uint32_t resource_id_count, uint32_t *resource_ids);
	static int llci_rm_changed_callback(void *arg, uint8_t slot_id, uint16_t session_number);
	static int llci_lookup_callback(void *arg, uint8_t slot_id, uint32_t requested_resource_id, en50221_sl_resource_callback *callback_out, void **arg_out, uint32_t *connected_resource_id);
	static int llci_session_callback(void *arg, int reason, uint8_t slot_id, uint16_t session_number, uint32_t resource_id);

	void poll();
	bool registerCam(int ca_fd, uint8_t slot);
	bool sub_init();

	en50221_app_rm *RmResource;
	en50221_session_layer *SessionLayer;
	en50221_transport_layer *TransportLayer;

	SlResource Resources[3];
};

class DvbCamHandlerHLCI : public DvbCamHandler
{
public:
	DvbCamHandlerHLCI();
	~DvbCamHandlerHLCI();

private:
	static int hlci_send_data(void *arg, uint16_t session_number, uint8_t *data, uint16_t data_length);
	static int hlci_send_datav(void *arg, uint16_t session_number, iovec *vector, int iov_count);

	void poll();
	bool registerCam(int ca_fd, uint8_t slot);
	bool sub_init();
};

DvbCam::DvbCam(const String& path)
{
	stopped = true;
	pmt_size = 0;
	cam_thread = NULL;
	service_id = 0;

	if ( (fd = open( path.c_str(), O_RDWR )) < 0)
	{
		throw SystemException(_("Failed to open CA device"));
	}
	
	if (ioctl(fd, CA_GET_CAP, &cap) != 0)
	{
		throw SystemException(_("Failed get CA capabilities"));
	}
	
	if (cap.slot_num <= 0)
	{
		Log::write(_("No CI slot found"));
	}
	
	info.num = 0;
	if (ioctl(fd, CA_GET_SLOT_INFO, &info) != 0)
	{
		throw SystemException(_("Failed get CA info"));
	}

	if ( (info.type & CA_CI_LINK) !=0 )
	{
		Log::write(_("LLCI slot found"));
		if ( (info.flags & CA_CI_MODULE_PRESENT) == 0 )
		{
			throw Exception(_("No CA module present"));
		}

		Log::write(_("CA module present"));
	}
	else if ((info.type & CA_CI) != 0)
	{
		Log::write(_("HLCI slot found"));
		if ( (info.flags & CA_CI_MODULE_PRESENT) == 0 )
		{
			throw Exception(_("No CA module present"));
		}

		Log::write(_("CA module present"));
	}
	else
	{
		throw Exception(_("Unsupported CI interface"));
	}
}
	
DvbCam::~DvbCam()
{
	stop();
	if (fd != 0)
	{
		close(fd);
	}
}

void DvbCam::start(guint service_id)
{
	this->service_id = service_id;
	stopped = false;
	cam_thread = g_thread_create(cam_thread_function, this, TRUE, NULL);
	pmt_thread = g_thread_create(pmt_thread_function, this, TRUE, NULL);
}

void DvbCam::stop()
{
	stopped = true;
	g_thread_join(cam_thread);
	g_thread_join(pmt_thread);
}

gpointer DvbCam::cam_thread_function(gpointer data)
{
	TRY
	((DvbCam*)data)->cam_thread_handler();
	THREAD_CATCH
}

gpointer DvbCam::pmt_thread_function(gpointer data)
{
	TRY
	((DvbCam*)data)->pmt_thread_handler();
	THREAD_CATCH
}

void DvbCam::cam_thread_handler()
{
	Log::write(_("CamThread: started"));

	if ( info.type!=CA_CI )
	{
		if(dvbca_reset(fd, 0))
		{
			throw Exception(_("CamThread: [error] resetting cam slot failed"));
		}
	}

	while(!stopped)
	{
		bool cam_ready = false;
		switch(dvbca_get_cam_state(fd, 0))
		{
			case DVBCA_CAMSTATE_MISSING:
			{
				/*Log::write("CamThread: [error] no cam detected\n");
				close(fd);
				return; */ // FIXME: find a more reliable solution
				break;
			}
			case DVBCA_CAMSTATE_READY:
			{
				Log::write(_("CamThread: cam 0 is ready"));
				cam_ready = true;
				break;
			}
			case DVBCA_CAMSTATE_INITIALISING:
			{
				if ( info.type==CA_CI )
				{ // workaround needed for hlci
					Log::write(_("CamThread: cam 0 is ready [hlci workaround]"));
					cam_ready = true;
				}
				break;
			}
			default:
			{
				throw Exception(_("CamThread: [error] querying the cam state failed"));
			}
		}
		
		if(cam_ready)
		{
			break;
		}
		usleep(100000); // 100 ms
	}

	if(!stopped)
	{
		switch(dvbca_get_interface_type(fd, 0))
		{
			case DVBCA_INTERFACE_LINK:
			{
				Log::write(_("CamThread: LLCI cam slot detected"));
				cam_handler = new DvbCamHandlerLLCI();
				break;
			}
			case DVBCA_INTERFACE_HLCI:
			{
				Log::write(_("CamThread: HLCI cam slot detected"));
				cam_handler = new DvbCamHandlerHLCI();
				break;
			}
			default:
			{
				throw Exception(_("CamThread: [error] unknown cam slot type"));
			}
		}
	}

	if(!stopped)
	{
		if(!cam_handler->init())
		{
			delete cam_handler;
			cam_handler = NULL;
			throw Exception(_("CamThread: [error] cam slot initialization failed"));
		}
	}

	if(!stopped)
	{
		if(!cam_handler->registerCam(fd, 0))
		{
			delete cam_handler;
			cam_handler = NULL;
			throw Exception(_("CamThread: [error] registering cam 0 failed"));
		}
	}

	while(!stopped)
	{
		cam_handler->poll();
		if(pmt_size > 0) {
			if(cam_handler->sendPmt(pmt_buffer, pmt_size))
			{
				Log::write(_("CamThread: pmt sent to cam"));
				pmt_size = 0;
			}
		}
	}

	Log::write(_("CamThread: stopping requested"));

	delete cam_handler;
	cam_handler = NULL;

	Log::write(_("CamThread: stopped"));
}

void DvbCam::pmt_thread_handler()
{
	Log::write(_("PmtThread: started"));

	int demux_fd = createSectionFilter(TRANSPORT_PAT_PID, stag_mpeg_program_association);
	if(demux_fd < 0) {
		throw Exception("PmtThread: [error] opening demux device failed");
	}

	pollfd poll_desc;
	poll_desc.fd = demux_fd;
	poll_desc.events = POLLIN | POLLPRI;
	while(!stopped) {
		int ret = poll(&poll_desc, 1, 100); // 100 ms
		if(ret < 0) {
			Log::write("PmtThread: [error] polling demux device failed");
			close(demux_fd);
			return;
		}
		if((ret > 0) && (poll_desc.revents != 0) && ((poll_desc.revents & ~(POLLIN | POLLPRI)) == 0)) {
			int processed_pat = processPat(demux_fd);
			if(processed_pat >= 0) {
				close(demux_fd);
				demux_fd = createSectionFilter(processed_pat, stag_mpeg_program_map);
				if(demux_fd < 0) {
					throw Exception("PmtThread: [error] opening demux device failed");
				}
				poll_desc.fd = demux_fd;
				break;
			}
		}
	}

	while(!stopped) {
		int ret = poll(&poll_desc, 1, 100); // 100 ms
		if(ret < 0) {
			Log::write("PmtThread: [error] polling demux device failed");
			close(demux_fd);
			return;
		}
		if((ret > 0) && (poll_desc.revents != 0) && ((poll_desc.revents & ~(POLLIN | POLLPRI)) == 0)) {
			if(processPmt(demux_fd)) {
				Log::write("PmtThread: new pmt received");
				close(demux_fd);
				Log::write("PmtThread: stopped");
				return;
			}
		}
	}

	Log::write("PmtThread: stopping requested");

	close(demux_fd);

	Log::write("PmtThread: stopped");
}

bool DvbCam::processPmt(int demux_fd)
{
	// read section
	char si_buf[4096];
	int size = ::read(demux_fd, si_buf, sizeof(si_buf));
	if(size <= 0) {
		return false;
	}

	// parse section
	section *parsed_section = section_codec(reinterpret_cast<unsigned char *> (si_buf), size);
	if(parsed_section == NULL) {
		return false;
	}

	// parse section_ext
	section_ext *parsed_section_ext = section_ext_decode(parsed_section, 1); // crc check on
	if(parsed_section_ext == NULL) {
		return false;
	}

	// parse pmt
	mpeg_pmt_section *parsed_pmt = mpeg_pmt_section_codec(parsed_section_ext);
	if(parsed_pmt == NULL) {
		return false;
	}

	// translate it into a cam pmt
	pmt_size = en50221_ca_format_pmt(parsed_pmt, reinterpret_cast<unsigned char *> (pmt_buffer), sizeof(pmt_buffer), 0, CA_LIST_MANAGEMENT_ONLY, CA_PMT_CMD_ID_OK_DESCRAMBLING);
	if(pmt_size <= 0) {
		return false;
	}

	// the DvbCamCamThread will send it to the cam
	return true;
}

int DvbCam::createSectionFilter(uint16_t pid, uint8_t table_id)
{
	// open the demuxer
	int demux_fd = dvbdemux_open_demux(0, 0, 0);
	if(demux_fd < 0) {
		return -1;
	}

	// create a section filter
	uint8_t filter[18] = {table_id};
	uint8_t mask[18] = {0xff};
	if(dvbdemux_set_section_filter(demux_fd, pid, filter, mask, 1, 1)) { // crc check on
		close(demux_fd);
		return -1;
	}

	return demux_fd;
}

int DvbCam::processPat(int demux_fd)
{
	// read section
	char si_buf[4096];
	int size = ::read(demux_fd, si_buf, sizeof(si_buf));
	if(size < 0) {
		return -1;
	}

	// parse section
	section *parsed_section = section_codec(reinterpret_cast<unsigned char *> (si_buf), size);
	if(parsed_section == NULL) {
		return -1;
	}

	// parse section_ext
	section_ext *parsed_section_ext = section_ext_decode(parsed_section, 1); // crc check on
	if(parsed_section_ext == NULL) {
		return -1;
	}

	// parse pat
	mpeg_pat_section *parsed_pat = mpeg_pat_section_codec(parsed_section_ext);
	if(parsed_pat == NULL) {
		return -1;
	}

	// try and find the requested program
	mpeg_pat_program *cur_program;
	mpeg_pat_section_programs_for_each(parsed_pat, cur_program) {
		if(cur_program->program_number == service_id) {
			return cur_program->pid;
		}
	}

	Log::write("PmtThread: [warning] the requested service id couldn't be found");

	return -1;
}

DvbCamHandler::DvbCamHandler()
{
	AiResource = NULL;
	CaResource = NULL;

	SessionNumber = -1;
}

DvbCamHandler::~DvbCamHandler()
{
	if(CaResource != NULL) {
		en50221_app_ca_destroy((en50221_app_ca*)CaResource);
		CaResource = NULL;
	}
	if(AiResource != NULL) {
		en50221_app_ai_destroy((en50221_app_ai*)AiResource);
		AiResource = NULL;
	}
}

bool DvbCamHandler::init()
{
	AiResource = en50221_app_ai_create(&send_functions);
	CaResource = en50221_app_ca_create(&send_functions);

	if(CaResource != NULL) {
		en50221_app_ca_register_info_callback((en50221_app_ca*)CaResource, infoCallback, this);
	}

	return sub_init();
}

bool DvbCamHandler::sendPmt(char *pmt_buffer, int size)
{
	if((CaResource != NULL) && (SessionNumber >= 0)) {
		if(!en50221_app_ca_pmt((en50221_app_ca*)CaResource, SessionNumber, reinterpret_cast<unsigned char *> (pmt_buffer), size)) {
			return true;
		}
	}

	return false;
}

int DvbCamHandler::infoCallback(void *arg, uint8_t /*slot_id*/, uint16_t session_number, uint32_t /*ca_id_count*/, uint16_t */*ca_ids*/)
{
	(static_cast<DvbCamHandler *> (arg))->SessionNumber = session_number;
	return 0;
}

#define MAX_CARDS 1
#define MAX_TC 16
#define MAX_SESSIONS 16

DvbCamHandlerLLCI::DvbCamHandlerLLCI()
{
	RmResource     = NULL;
	SessionLayer   = NULL;
	TransportLayer = NULL;
}

DvbCamHandlerLLCI::~DvbCamHandlerLLCI()
{
	if(SessionLayer != NULL) {
		en50221_sl_destroy(SessionLayer);
		SessionLayer = NULL;
	}
	if(TransportLayer != NULL) {
		en50221_tl_destroy(TransportLayer);
		TransportLayer = NULL;
	}
	if(RmResource != NULL) {
		en50221_app_rm_destroy(RmResource);
		RmResource = NULL;
	}
}

int DvbCamHandlerLLCI::llci_rm_enq_callback(void *arg, uint8_t /*slot_id*/, uint16_t session_number)
{
	uint32_t resource_ids[] = {EN50221_APP_RM_RESOURCEID, EN50221_APP_AI_RESOURCEID, EN50221_APP_CA_RESOURCEID};
	en50221_app_rm_reply((en50221_app_rm*)arg, session_number, sizeof(resource_ids) / 4, resource_ids);
	return 0;
}

int DvbCamHandlerLLCI::llci_rm_reply_callback(void *arg, uint8_t /*slot_id*/, uint16_t session_number, uint32_t /*resource_id_count*/, uint32_t */*resource_ids*/)
{
	en50221_app_rm_changed((en50221_app_rm*)arg, session_number);
	return 0;
}

int DvbCamHandlerLLCI::llci_rm_changed_callback(void *arg, uint8_t /*slot_id*/, uint16_t session_number)
{
	en50221_app_rm_enq((en50221_app_rm*)arg, session_number);
	return 0;
}

int DvbCamHandlerLLCI::llci_lookup_callback(void *arg, uint8_t /*slot_id*/, uint32_t requested_resource_id, en50221_sl_resource_callback *callback_out, void **arg_out, uint32_t *connected_resource_id)
{
	// decode the resource id
	en50221_app_public_resource_id resid;
	if(!en50221_app_decode_public_resource_id(&resid, requested_resource_id)) {
		return -1;
	}

	// try and find an instance of the resource
	const SlResource *Resources = (static_cast<DvbCamHandlerLLCI *> (arg))->Resources;
	int i;
	for(i = 0; i < 3; ++i) {
		if((resid.resource_class == Resources[i].resid.resource_class) && (resid.resource_type == Resources[i].resid.resource_type)) {
			*callback_out = Resources[i].callback;
			*arg_out = Resources[i].arg;
			*connected_resource_id = Resources[i].binary_resource_id;
			return 0;
		}
	}
	return -1;
}

int DvbCamHandlerLLCI::llci_session_callback(void *arg, int reason, uint8_t /*slot_id*/, uint16_t session_number, uint32_t resource_id)
{
	if(reason == S_SCALLBACK_REASON_CAMCONNECTED) {
		if(resource_id == EN50221_APP_RM_RESOURCEID) {
			void *RmResource = (static_cast<DvbCamHandlerLLCI *> (arg))->Resources[0].arg;
			en50221_app_rm_enq((en50221_app_rm*)RmResource, session_number);
		}
		else if(resource_id == EN50221_APP_AI_RESOURCEID) {
			void *AiResource = (static_cast<DvbCamHandlerLLCI *> (arg))->Resources[1].arg;
			en50221_app_ai_enquiry((en50221_app_ai*)AiResource, session_number);
		}
		else if(resource_id == EN50221_APP_CA_RESOURCEID) {
			void *CaResource = (static_cast<DvbCamHandlerLLCI *> (arg))->Resources[2].arg;
			en50221_app_ca_info_enq((en50221_app_ca*)CaResource, session_number);
		}
	}
	return 0;
}

void DvbCamHandlerLLCI::poll()
{
	if(en50221_tl_poll(TransportLayer)) {
		Log::write("CamThread: [warning] polling the stack failed");
		usleep(10000); // wait 10 ms to not block
	}
}

bool DvbCamHandlerLLCI::registerCam(int ca_fd, uint8_t slot)
{
	// register the slot
	int slot_id = en50221_tl_register_slot(TransportLayer, ca_fd, slot, 1000, 100);
	if(slot_id < 0) {
		return false;
	}

	// create a new connection on the slot
	if(en50221_tl_new_tc(TransportLayer, slot_id) < 0) {
		return false;
	}

	return true;
}

bool DvbCamHandlerLLCI::sub_init()
{
	// create transport layer
	TransportLayer = en50221_tl_create(MAX_CARDS, MAX_TC);
	if(TransportLayer == NULL) {
		return false;
	}

	// create session layer
	SessionLayer = en50221_sl_create(TransportLayer, MAX_SESSIONS);
	if(SessionLayer == NULL) {
		en50221_tl_destroy(TransportLayer);
		TransportLayer = NULL;
		return false;
	}

	// create sendfuncs
	send_functions.arg        = SessionLayer;
	send_functions.send_data  = (int (*)(void *, uint16_t, uint8_t*, uint16_t))en50221_sl_send_data;
	send_functions.send_datav = (int (*)(void *, uint16_t, iovec*, int))en50221_sl_send_datav;

	// create the resource manager resource
	RmResource = en50221_app_rm_create(&send_functions);
	en50221_app_decode_public_resource_id(&Resources[0].resid, EN50221_APP_RM_RESOURCEID);
	Resources[0].binary_resource_id = EN50221_APP_RM_RESOURCEID;
	Resources[0].callback = (int (*)(void*, uint8_t, uint16_t, uint32_t, uint8_t*, uint32_t))en50221_app_rm_message;
	Resources[0].arg = RmResource;
	en50221_app_rm_register_enq_callback(RmResource, llci_rm_enq_callback, RmResource);
	en50221_app_rm_register_reply_callback(RmResource, llci_rm_reply_callback, RmResource);
	en50221_app_rm_register_changed_callback(RmResource, llci_rm_changed_callback, RmResource);

	// integrate the application information resource
	en50221_app_decode_public_resource_id(&Resources[1].resid, EN50221_APP_AI_RESOURCEID);
	Resources[1].binary_resource_id = EN50221_APP_AI_RESOURCEID;
	Resources[1].callback = (int (*)(void*, uint8_t, uint16_t, uint32_t, uint8_t*, uint32_t))en50221_app_ai_message;
	Resources[1].arg = AiResource;

	// integrate the ca resource
	en50221_app_decode_public_resource_id(&Resources[2].resid, EN50221_APP_CA_RESOURCEID);
	Resources[2].binary_resource_id = EN50221_APP_CA_RESOURCEID;
	Resources[2].callback = (int (*)(void*, uint8_t, uint16_t, uint32_t, uint8_t*, uint32_t))en50221_app_ca_message;
	Resources[2].arg = CaResource;

	// register session layer callbacks
	en50221_sl_register_lookup_callback(SessionLayer, llci_lookup_callback, this);
	en50221_sl_register_session_callback(SessionLayer, llci_session_callback, this);

	return true;
}

DvbCamHandlerHLCI::DvbCamHandlerHLCI()
{
}

DvbCamHandlerHLCI::~DvbCamHandlerHLCI()
{
}

int DvbCamHandlerHLCI::hlci_send_data(void *arg, uint16_t /*session_number*/, uint8_t *data, uint16_t data_length)
{
	return dvbca_hlci_write(static_cast<int> (reinterpret_cast<intptr_t> (arg)), data, data_length);
}

int DvbCamHandlerHLCI::hlci_send_datav(void *arg, uint16_t /*session_number*/, iovec *vector, int iov_count)
{
	// calculate the total length of the data to send
	uint32_t data_size = 0;
	for(int i = 0; i < iov_count; ++i) {
		data_size += vector[i].iov_len;
	}

	// allocate memory for it
	uint8_t *buf = new uint8_t[data_size];

	// merge the iovecs
	uint8_t *pos = buf;
	for(int i = 0; i < iov_count; ++i) {
		memcpy(pos, vector[i].iov_base, vector[i].iov_len);
		pos += vector[i].iov_len;
	}

	// send it
	int status = dvbca_hlci_write(static_cast<int> (reinterpret_cast<intptr_t> (arg)), buf, data_size);
	delete buf;
	return status;
}

void DvbCamHandlerHLCI::poll()
{
	// we do nothing here for the moment
	usleep(100000); // 100 ms
}

bool DvbCamHandlerHLCI::registerCam(int ca_fd, uint8_t /*slot*/)
{
	send_functions.arg = reinterpret_cast<void *> (ca_fd);

	// get application information
	if(en50221_app_ai_enquiry((en50221_app_ai*)AiResource, 0)) {
		Log::write("CamThread: [DEBUG #1]");
		return false;
	}

	uint8_t buf[256];
	int size = dvbca_hlci_read(ca_fd, TAG_APP_INFO, buf, sizeof(buf));
	if(size <= 0) {
		Log::write("CamThread: [DEBUG #2]");
		return false;
	}

	if(en50221_app_ai_message((en50221_app_ai*)AiResource, 0, 0, EN50221_APP_AI_RESOURCEID, buf, size)) {
		Log::write("CamThread: [DEBUG #3]");
		return false;
	}

	// FIXME: try to change this soon
	buf[0] = TAG_CA_INFO >> 16;
	buf[1] = (TAG_CA_INFO >> 8) & 0xFF;
	buf[2] = TAG_CA_INFO & 0xFF;
	buf[3] = 0;
	if(en50221_app_ca_message((en50221_app_ca*)CaResource, 0, 0, EN50221_APP_CA_RESOURCEID, buf, 4)) {
		Log::write("CamThread: [DEBUG #4]");
		return false;
	}

	Log::write("CamThread: [DEBUG #5]");

	return true;
}

bool DvbCamHandlerHLCI::sub_init()
{
	// create sendfuncs
	send_functions.arg        = NULL;
	send_functions.send_data  = hlci_send_data;
	send_functions.send_datav = hlci_send_datav;

	return true;
}


