/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file SymbolMode.cc
    \brief Implementation of the Template class SymbolMode.
    
    Magics Team - ECMWF 2004
    
    Started: Wed 21-Jan-2004
    
    Changes:
    
*/



#include "SymbolMode.h"
#include "LegendVisitor.h"
#include "CartesianTransformation.h"
#include "Histogram.h"
#include "HistoVisitor.h"
using namespace magics;


SymbolMode::SymbolMode() 
{
}


SymbolMode::~SymbolMode() 
{
}
void SymbolMode::visit(Data<GeoPoint>& data, HistoVisitor& visitor)
{

	IntervalMap<Colour> beans;
	Histogram<GeoPoint> helper;
	helper.visit(beans, data, data.points(), visitor);

}
void SymbolMode::visit(Data<UserPoint>& data, HistoVisitor& visitor)
{
	IntervalMap<Colour> beans;
	Histogram<UserPoint> helper;
	helper.visit(beans, data, data.points(), visitor);
		
}

/*!
 Class information are given to the output-stream.
*/		
void SymbolMode::print(ostream& out)  const
{
	out << "SymbolMode";
}

SymbolIndividualMode::SymbolIndividualMode() 
{
  
    update();
}



SymbolIndividualMode::~SymbolIndividualMode() 
{
}

void SymbolIndividualMode::update()
{
	marker_list_.clear();
	text_.clear();

	// we give a priority to symbol_name...
	if ( !symbol_.empty() ) 
		marker_ = -1;
    if ( marker_ != -1 ) {
    	marker_list_.push_back(marker_);
    	symbol_ = "";
    }
    else 
    	marker_ = 1;
    if (  marker_list_.empty() )  
    	marker_list_.push_back(marker_);
    if (  text_.empty() )  
        	text_.push_back("?");
    current_ = text_.begin();
    current_marker_ = marker_list_.begin();

    
}

void SymbolIndividualMode::properties() const
{

	properties_.colour_ = *colour_;
    properties_.height_ = height_;

    properties_.outline_ = outline_;
    properties_.outlineColour_ = *outline_colour_;
    properties_.outlineStyle_ = outline_style_;
    properties_.outlineThickness_ = outline_thickness_;

    properties_.label(*current_);
    ++current_;
    if (current_ == text_.end() ) 
    	current_ = text_.begin(); 
    MagFont font(text_font_name_);
    font.colour(text_font_colour_->automatic() ? *colour_ : *text_font_colour_);
    font.size(text_font_size_);
    font.style(text_font_style_);
    properties_.font_ = font;
    properties_.position_ = TextSymbol::M_ABOVE;
    properties_.setSymbol(symbol_, *current_marker_);
    ++current_marker_;
    if ( current_marker_ == marker_list_.end() ) 
    	current_marker_ = marker_list_.begin();
    
}

/*!
 Class information are given to the output-stream.
*/		
void SymbolIndividualMode::print(ostream& out)  const
{
    out << "SymbolIndividualMode[";
    SymbolIndividualModeAttributes::print(out);
    out << "]";
}






SymbolTableMode::SymbolTableMode() 
{
   
}


SymbolTableMode::~SymbolTableMode() 
{
}

/*!
 Class information are given to the output-stream.
*/		
void SymbolTableMode::print(ostream& out)  const
{
    out << "SymbolTableMode[";
    SymbolTableModeAttributes::print(out);
    out << "]";
}



SymbolProperties SymbolTableMode::operator()(double value) const 
{
	return map_.find(value, SymbolProperties());
    
    
}

void SymbolTableMode::prepare()
{
	if ( colour_.empty()) {
		MagLog::warning() << "SymbolTableMode: No colour table defined.\n";
		colour_.push_back("red");
	}
	if ( height_.empty()) {
		MagLog::warning() << "SymbolTableMode: No height table defined.\n";
		height_.push_back(0.2);
	}
	if ( !marker_.empty()) {
		symbol_.clear();
		for (vector<int>::const_iterator marker = marker_.begin(); marker != marker_.end(); ++marker) 
			symbol_.push_back(Symbol::convert(*marker));
	}
	
	if ( symbol_.empty()) {
		MagLog::warning() << "SymbolTableMode: No marker table defined.\n";
		symbol_.push_back("magics_1");
	}
    doublearray::iterator max = max_.begin();
    stringarray::iterator colour = colour_.begin();
    doublearray::iterator height = height_.begin();
    stringarray::iterator symbol = symbol_.begin();
    
    int index = 0;
    
  
    
    for (doublearray::const_iterator min = min_.begin(); min != min_.end(); ++min) {

        map_[Interval(*min, *max) ] = SymbolProperties(Colour(*colour),  *height,  *symbol);
        
        if ( ++colour == colour_.end()) --colour;  
        if ( ++height == height_.end())  --height;
        if ( ++symbol == symbol_.end()) --symbol;
//        label << *min << "-" << *max;
//        map[index].label_ = label.str();
     
        if ( ++max == max_.end()) --max;
        index++;
    }
} 

bool  SymbolTableMode::accept(double value) 
{
	try {
		map_.find(value);

		return true;
	}

	catch (...) {
		return false;
	}
}


void SymbolTableMode::visit(LegendVisitor& legend)
{
    
    IntervalMap<SymbolProperties>::const_iterator interval;
    
    for ( interval = map_.begin(); interval != map_.end(); ++interval) {
	        Symbol* symbol = new Symbol();
	        (*symbol).setColour(interval->second.colour_);
	        (*symbol).setSymbol(interval->second.marker_);
	        (*symbol).setHeight(interval->second.height_);
	        legend.add(new SimpleSymbolEntry( interval->first.min_,  interval->first.max_, symbol));
    } 	
}

void SymbolTableMode::visit(Data<GeoPoint>& data, HistoVisitor& visitor)
{
	IntervalMap<Colour> beans;
	if ( !visitor.basic() ) 
		buildBins(map_, beans);
	Histogram<GeoPoint> helper;
	helper.visit(beans, data, data.points(), visitor);

}
void SymbolTableMode::visit(Data<UserPoint>& data, HistoVisitor& visitor)
{
	IntervalMap<Colour> beans;
	if ( !visitor.basic() ) 
		buildBins(map_, beans);
	Histogram<UserPoint> helper;
	helper.visit(beans, data, data.points(), visitor);
		
}
void SymbolTableMode::buildBins(const IntervalMap<SymbolProperties>& in, IntervalMap<Colour>& out)
{
	for ( IntervalMap<SymbolProperties>::const_iterator interval = in.begin(); interval != in.end(); ++interval)
		out.insert(make_pair(Interval(interval->first.min_, interval->first.max_), interval->second.colour_));

}

void SymbolIndividualMode::visit(LegendVisitor& legend)
{
   
	  Symbol* symbol = properties_.symbol("marker");
      
	  legend.add(new SimpleSymbolEntry(" ", symbol));
 	
}

