/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file GribRegularInterpretor.cc
    \brief Implementation of the Template class GribRegularInterpretor.
    \author Graphics Section, ECMWF

    Started: Mon 18-Apr-2005

    Changes:
*/

#include <limits>

#include "CustomisedPoint.h"
#include "TeProjection.h"
#include "GribRegularInterpretor.h"
#include "GribDecoder.h"
#include "Matrix.h"
#include "LocalTable.h"
#include "RasterData.h"
#include "Timer.h"


using namespace magics;

GribRegularInterpretor::GribRegularInterpretor() 
{
}

GribRegularInterpretor::~GribRegularInterpretor() 
{
}

void GribRegularInterpretor::longitudesSanityCheck(double& west, double& east) const
{
	 // make sure that the west longitudes is always inferior to teh East longitudes and always 
	// betwwen -180 and 360...
	
	while ( east <= west) {		
		// We add 360 to the east ...
		east += 360;
	}
	return;
	// We reposition if needed 

	while ( east > 360. ) {
		west -= 360.;
		east -= 360.;
	}
}

void GribInterpretor::scaling(const GribDecoder& grib, double& scaling, double& offset) const
{
	scaling = 1;
	offset  = 0;

	// Fisrt check that they are not derived fields! 

	if (grib.getScaling())
	{
		long derived = grib.getLong("generatingProcessIdentifier");

		if (derived != 254) 
		{
			long edition = grib.getLong("edition");
            
			if (edition == 1)
			{
				// later...
				long id   = grib.getLong("paramId");
				
				
				long table   = 128;
				long centre  =98;
				try {
					const ParamDef& paramdef = LocalTable::localInfo(id, table, centre);
					scaling = paramdef.scaling();
					offset = paramdef.offset();
				}
				catch (...) {
					Log::warning() << " Can not find information for the parameter [" << id << "." << table << "]\n";
				}
			}
            
			// else, if not edition 1, then this table-based approach will not work,
			// and we'll just use the defaults set up at the top of this function
		}
	}
	else {
		scaling = grib.getScaling_factor();
		offset  = grib.getScaling_offset();
	}
}


void GribInterpretor::scaling(const GribDecoder& grib, Matrix** matrix) const
{
	double factor, offset;

	scaling(grib, factor, offset);

	(*matrix)->multiply(factor);
	(*matrix)->plus(offset);
}


void GribInterpretor::raw(const GribDecoder& grib, const Transformation& transformation, const string& key, CustomisedPointsList& points) const
{
	double factor, offset;
	scaling(grib, factor, offset);
	int err;

    grib_iterator* iter = grib_iterator_new(grib.handle(), 0,&err);                                     
    double missing = -std::numeric_limits<double>::max();
     grib.setDouble("missingValue", missing);    

    double lat, lon, value;
        /* Loop on all the lat/lon/values. */
    while(grib_iterator_next(iter,&lat,&lon,&value)) {   
  
      
      if (value != missing  ) {
    	  std::stack<GeoPoint>   duplicates;
    	  GeoPoint geo(lon, lat);
    	  value = (value*factor)+offset;
    	  
    	 if (  geo.in(transformation.getMinX(), transformation.getMaxX(), transformation.getMinY(), transformation.getMaxY(), duplicates) ) {
    	 	    			while (duplicates.empty() == false) {
    	 	    				CustomisedPoint *point = new CustomisedPoint(duplicates.top().x(), duplicates.top().y(), "");
    	 	    				point->insert(make_pair(key, value));
    	 	    				points.push_back(point);   	 	    				
    	 	    				duplicates.pop();
    	 	    			}
    	 	 }   	
    	
      }
       
    }

        /* At the end the iterator is deleted to free memory. */
    grib_iterator_delete(iter);               

	
}
/*!
 Class information are given to the output-stream.
*/		
void GribRegularInterpretor::print(ostream& out)  const
{
	out << "GribRegularInterpretor[";
	out << "]";
}

void GribRegularInterpretor::interpretAsMatrix(const GribDecoder& grib, Matrix** matrix, const Transformation& ) const
{
	interpretAsMatrix(grib, matrix);
}

void GribRegularInterpretor::interpretAsMatrix(const GribDecoder& grib, Matrix** matrix) const 
{
  Timer timer("gribapi", " read grib");
  Log::dev() << "GribRegularInterpretor::interpretAsMatrix" << "\n";
  long nblon = grib.getLong("numberOfPointsAlongAParallel");
  long nblat = grib.getLong("numberOfPointsAlongAMeridian");
  
  if ( *matrix == 0 ) *matrix = new Matrix(nblat, nblon);
  
  
  size_t nb;
  grib_get_size(grib.id(), "values", &nb);

  Log::dev() << "numberOfFieldValues[" << nb << "]" << "\n";
  double missing = int_MAX;
  grib.setDouble("missingValue", missing);    
  (*matrix)->missing(missing);
  (*matrix)->akimaEnabled();

  double north = grib.getDouble("latitudeOfFirstGridPointInDegrees");
  double west = grib.getDouble("longitudeOfFirstGridPointInDegrees");
  double south = grib.getDouble("latitudeOfLastGridPointInDegrees");; 
  double east = grib.getDouble("longitudeOfLastGridPointInDegrees");;
  longitudesSanityCheck(west, east);

  Log::dev() << "NewAPI---> area[" << west << ", " << north << ", " << east << ", " << south << "]" << "\n";
  double loni = longitudeIncrement(grib);
  
  double lon = (east-west)/(nblon-1);
  
  Log::dev() << "increment -->" << loni << " (from->" << west << " to-->" << west + (nblon-1) *loni << ")" <<  endl;
  Log::dev() << "calcul -->" << lon << " (from->" << west << " to-->" << west + (nblon-1) *lon << ")" <<  endl;

  latitudes(grib, (*matrix)->rowsAxis());
 
  double x = west;
  for (int i = 0; i < nblon; i++)
  {
	(*matrix)->columnsAxis().push_back(x);
	x  = west + (i+1)*lon;
  }
	
  (*matrix)->setMapsAxis();
  
  try
  { 
	(*matrix)->resize(nb);
  	size_t aux = size_t(nb);
  	grib_get_double_array(grib.id(),"values", &(*matrix)->front(),&aux); 
  	(*matrix)->missing(missing);
  } 
  catch (...) 
  {
	throw MagicsException("GribRegularInterpretor - Not enough memory");
  }
}


void  GribRegularInterpretor::latitudes(const GribDecoder& grib, vector<double>& latitudes) const
{
		double north = grib.getDouble("latitudeOfFirstGridPointInDegrees");	

		long nblat = grib.getLong("numberOfPointsAlongAMeridian");
		int scanning = grib.getLong("jScansPositively") ? 1 : -1;
		double lat =  scanning * grib.getDouble("jDirectionIncrementInDegrees");
	
		double y = north;
		for (int i = 0; i < nblat; i++)
		{
				latitudes.push_back(y);
				 y  += lat;
		}
}


void GribRegularGaussianInterpretor::latitudes(const GribDecoder& grib, vector<double>& latitudes) const 
{
	long res     = grib.getLong("numberOfParallelsBetweenAPoleAndTheEquator");
	double array[2 *res];
	grib_get_gaussian_latitudes(res, array);
	 double north = grib.getDouble("latitudeOfFirstGridPointInDegrees");
	 
	  double south = grib.getDouble("latitudeOfLastGridPointInDegrees");; 
	
	for ( int i = 0; i < 2*res; i++ )
	{
		//if ( array[i] <= north && array[i] >= south)
			latitudes.push_back(array[i]);
	}
	
}


double GribRegularInterpretor::longitudeIncrement(const GribDecoder& grib) const 
{
	int scanning = grib.getLong("iScansNegatively") ? -1 : 1;
	return scanning * grib.getDouble("iDirectionIncrementInDegrees");	
	
}




void GribRegularInterpretor::interpretAsRaster(const GribDecoder& grib, RasterData<GeoPoint>& raster,const Transformation& transformation) const
{
	Log::dev() << "GribRegularInterpretor::interpretAsRaster" << "\n";
	
	BoxMatrixHandler<GeoPoint> box(const_cast<GribDecoder* >(&grib)->matrix(), transformation); 
	
	int nblon = box.columns();
	int nblat = box.rows();
	double east = box.column(0,nblon-1);
	double west = box.column(0, 0);
	double south = box.row(0, 0);
	double north = box.row(nblat-1, 0);
	
	raster.setUpperRightCorner(east, north);
	raster.setLowerLeftCorner(west, south);

	double lon = (east-west)/(nblon-1);
	double lat = (north-south)/(nblat-1);
	
	raster.setXResolution(lon);
	raster.setYResolution(lat);
	
	raster.setColumns(nblon);
	raster.setRows(nblat);
//	
	raster.setProjection(new TeLatLong(TeDatum()));

	raster.reserve(nblon*nblat);
	
		for ( int j = nblat-1; j >= 0; j--)
			for ( int i = 0; i < nblon; i++)
				raster.push_back(box(j, i));
}


void GribReducedGaussianInterpretor::interpretAsRaster(const GribDecoder& grib, RasterData<GeoPoint>& raster,const Transformation& transformation) const
{
	Log::dev() << "GribRegularInterpretor::interpretAsRaster" << "\n";
	Timer timer("grib api", "read grib" );
	BoxMatrixHandler<GeoPoint> box(const_cast<GribDecoder* >(&grib)->matrix(), transformation); 
	
	int nblon = box.columns();
	int nblat = box.rows();
	double east = box.column(0,nblon-1);
	double west = box.column(0, 0);
	double south = box.row(0, 0);
	double north = box.row(nblat-1, 0);
	
	raster.setUpperRightCorner(east, north);
	raster.setLowerLeftCorner(west, south);

	double lon = (east-west)/(nblon-1);
	double lat = (north-south)/(nblat-1);
	
	raster.setXResolution(lon);
	raster.setYResolution(lat);
	
	raster.setColumns(nblon);
	raster.setRows(nblat);
//	
	raster.setProjection(new TeLatLong(TeDatum()));

	
	raster.reserve(nblon*nblat);
	
		for ( int j = nblat-1; j >= 0; j--)
			for ( int i = 0; i < nblon; i++)
				raster.push_back(box(j, i));
}


void GribReducedGaussianInterpretor::print(ostream& out)  const
{
	out << "GribRegularInterpretor[";
	out << "]";
}

#ifdef later
class GribMatrix : public Matrix
{
public:
	    GribMatrix(const GribDecoder& grib) : grib_(grib) {
	    	
	    }
	    
	    void prepare(double north = 90., double south = -90., double west = -180, double east = 180) 
	    {
	        Timer timer("gribapi", "subarea extraction");
	    	int error;
	    	box_ = grib_box_new(grib_.id(),&error);
	    	// deal with the error! 
	    	grib_points* points = grib_box_get_points(box_,north,west,south,east,&error);

	    	double value[points->n];
	    	grib_points_get_values(grib_.id(),points,value);
	    	
	    	map<double, vector<double> > positions;
	    	map<double, vector<double> > values;
	    	
	    	for (unsigned int i=0;i<points->n;i++) {
	    		map<double, vector<double> >::iterator pos = positions.find(points->latitudes[i]);
	    		map<double, vector<double> >::iterator val = values.find(points->latitudes[i]);
	    		if ( pos == positions.end() ) {
	    			positions.insert(make_pair(points->latitudes[i], vector<double>()) );
	    			values.insert(make_pair(points->latitudes[i], vector<double>()) );
	    			pos = positions.find(points->latitudes[i]);
	    			val = values.find(points->latitudes[i]);
	    		}
	    		pos->second.push_back(points->longitudes[i]);
	    		val->second.push_back(value[i]);	    		   
	    	}
	    	unsigned int max = 0;
	    	double lat = 0;
	    
	    	
	    	for (map<double, vector<double> >::iterator pos = positions.begin(); pos != positions.end(); ++pos) {
	    		if ( pos->second.size() > max ) {
	    			max = pos->second.size();
	    			lat = pos->first;
	    		}
	    		this->rowsAxis().push_back(pos->first);
	    	}
	    		vector<double>& longitudes = positions[lat];
	    		

	    		int nblon = longitudes.size();
	    		double width = longitudes.back() -  longitudes.front();
	    		double step = (width)/(nblon-1); 

	    		for (map<double, vector<double> >::iterator pos = positions.begin(); pos != positions.end(); ++pos) {
	    			
	    			map<double, vector<double> >::iterator val = values.find(pos->first);
	    			vector<double>& p =val->second;
	    			double lon = longitudes.front();
	    			unsigned int p1 = 0;
	    			unsigned int p2 = 1;
	    			double lon1 = lon;
	    			double lon2 = lon1 + (width/(p.size()-1));

	    			for ( unsigned int x = 0; x < longitudes.size(); x++ ) {
	    				if ( lon >= lon2 ) {
	    					p1++;
	    					p2++;
	    					lon1 = lon2;
	    					lon2 += (width)/(p.size()-1);
	    				}
	    				double d1 = (lon2 - lon)/(lon2-lon1);
	    				double d2 = 1-d1;
	    				double val;
	    				
	    				assert ( p1 < p.size() );
	    				if (p2 == p.size()) {
	    					this->push_back(p[p1]);
	    				}
	    				else {
	    					if (p[p1] == missing_ ||  p[p2] == missing_)
	    						val = missing_;
	    					else
	    						val = (p[p1] * d1) + (p[p2] * d2);
	    					this->push_back(val);
	    				}
	    				lon += step;
	    			}
	    		}
	    		
	    		for ( vector<double>::iterator lon = longitudes.begin(); lon != longitudes.end(); ++lon) 
	    				this->columnsAxis().push_back(*lon);
	    		
	    		setMapsAxis();
	    }
		  
protected:
	const GribDecoder& grib_;
	grib_box*                   box_;
};
#endif


void GribReducedGaussianInterpretor::interpretAsMatrix(const GribDecoder& grib, Matrix** matrix, const Transformation& transformation) const
{ 
	interpretAsMatrix(grib, matrix);
	//GribMatrix* helper = new GribMatrix(grib);
	//*matrix = helper;
	//double minlon, minlat, maxlon, maxlat;
    	      
	 //transformation.boundingBox(minlon, minlat, maxlon, maxlat);
	 
	// helper->prepare(maxlat, minlon, minlat, maxlon);
}

double GribReducedLatLonInterpretor::XResolution(const GribDecoder& grib) const
{
	long res     = grib.getLong("Nj");
	

	double west  = grib.getDouble("longitudeOfFirstGridPointInDegrees");
	double east  = grib.getDouble("longitudeOfLastGridPointInDegrees");;
	
	longitudesSanityCheck(west, east);
	
	return (east-west)/(2*res);
}
double GribReducedGaussianInterpretor::XResolution(const GribDecoder& grib) const
{
	long res     = grib.getLong("numberOfParallelsBetweenAPoleAndTheEquator");
	double west  = grib.getDouble("longitudeOfFirstGridPointInDegrees");
	
	double east  = grib.getDouble("longitudeOfLastGridPointInDegrees");;
		
	longitudesSanityCheck(west, east);
	return (east-west)/(4*res);
}

void GribReducedGaussianInterpretor::interpretAsMatrix(const GribDecoder& grib, Matrix** matrix) const
{
	Log::dev() << "GribRegularInterpretor::interpretAsMatrix" << "\n";
	Log::dev() << "GribRegularInterpretor::interpretAsMatrix" << "\n";
 
	Timer timer("gribapi", " read grib");
	*matrix = new Matrix();
	size_t nb;
	grib_get_size(grib.id(), "values", &nb);

	Log::dev() << "numberOfFieldValues[" << nb << "]" << "\n";
	double missing = std::numeric_limits<double>::max();
	grib.setDouble("missingValue", missing);
	(*matrix)->missing(missing);
	(*matrix)->akimaEnabled();

	double north = grib.getDouble("latitudeOfFirstGridPointInDegrees");
	double west  = grib.getDouble("longitudeOfFirstGridPointInDegrees");
	double south = grib.getDouble("latitudeOfLastGridPointInDegrees");; 
	double east  = grib.getDouble("longitudeOfLastGridPointInDegrees");;
	double plp   = grib.getDouble("PLPresent");
	long res     = grib.getLong("numberOfParallelsBetweenAPoleAndTheEquator");
	longitudesSanityCheck(west, east);
	Log::dev() << "NewAPI---> area[" << west << ", " << north << ", " << east << ", " << south << "]" << "\n";
	Log::dev() << "PLPresent---> " << plp << "\n";
	Log::dev() << "Res---> " << res << "\n";

	double pl[2*res];

	size_t aux = 2*res;
	grib_get_double_array(grib.id(),"pl",pl,&aux);
	// We have to determine if the field is global! 
	bool global = same(north-south, 180.);
	if (global) {
		Log::dev() << "YES THE FIELD IS GLOBAL" << endl;
		 // We have to determine if the field is global!
		bool global = same(north-south, 180.);
		if (global)
		{
        		Log::dev() << "YES THE FIELD IS GLOBAL" << endl;
        		east = west + 360;
		}
	}

	// compute the number of points we'll be adding to the matrix so that we can
	// allocate them in one go, rather than allowing the STL to re-allocate
	// when we reach the capacity
	(*matrix)->reserve(aux * 4 * res);

	double *data = new double[nb];

	size_t aux2 = size_t(nb);
	int nblon = 4*res;
	double width = east-west;
	double step = (width)/(nblon-1); 

	grib_get_double_array(grib.id(),"values",data,&aux2);
	int d = 0;
	for ( size_t i = 0; i < aux; i++)
	{
		vector<double> p;
		for ( int ii = 0; ii < pl[i]; ii++)
		{
			p.push_back(data[d]);
			d++;
		}
		
		double lon = west;
		unsigned int p1 = 0;
		unsigned int p2 = 1;
		double lon1 = west;
		double lon2 = lon1 + (width/(p.size()-1));

		for ( int x = 0; x < 4*res; x++ )
		{
			if ( lon >= lon2 )
			{
				p1++;
				p2++;
				lon1 = lon2;
				lon2 += (width)/(p.size()-1);
			}
			double d1 = (lon2 - lon)/(lon2-lon1);
			double d2 = 1-d1;
			double val;
			
			assert ( p1 < p.size() );
			if (p2 == p.size()) {
				(*matrix)->push_back(p[p1]);
			}
			else {
				if (p[p1] == missing ||  p[p2] == missing)
					val = missing;
				else
					val = (p[p1] * d1) + (p[p2] * d2);
				(*matrix)->push_back(val);
			}
			lon += step;
		}
	}

	delete [] data;


	
	double lon = (width) / (nblon-1);
	
		for (int x = 0; x < nblon; x++)
		{
			(*matrix)->columnsAxis().push_back(west+(x*lon));
		}
		
	double array[2 *res];
	long par = grib.getLong("numberOfParallelsBetweenAPoleAndTheEquator");
	grib_get_gaussian_latitudes(par, array);

	for ( int i = 0; i < 2*res; i++ )
	{
		(*matrix)->rowsAxis().push_back(array[i]);
	}
	(*matrix)->setMapsAxis();
}

void GribReducedLatLonInterpretor::print(ostream& out)  const
{
	out << "GribReducedLatLonInterpretor[";
	out << "]";
}

void GribReducedLatLonInterpretor::interpretAsMatrix(const GribDecoder& grib, Matrix** matrix) const
{
	
 
	Timer timer("gribapi", " read grib");
    *matrix = new Matrix();
	size_t nb;
	grib_get_size(grib.id(), "values", &nb);

	Log::dev() << "numberOfFieldValues[" << nb << "]" << "\n";
	double missing = std::numeric_limits<double>::max();
	grib.setDouble("missingValue", missing);
	(*matrix)->missing(missing);
	(*matrix)->akimaEnabled();

	double north = grib.getDouble("latitudeOfFirstGridPointInDegrees");
	double west  = grib.getDouble("longitudeOfFirstGridPointInDegrees");
	double south = grib.getDouble("latitudeOfLastGridPointInDegrees");; 
	double east  = grib.getDouble("longitudeOfLastGridPointInDegrees");;
	longitudesSanityCheck(west, east);
	size_t res     = grib.getLong("Nj");

	Log::dev() << "NewAPI---> area[" << west << ", " << north << ", " << east << ", " << south << "]" << "\n";
	Log::dev() << "Res---> " << res << "\n";

	double pl[res];
	
	long nblat = grib.getLong("numberOfPointsAlongAMeridian");
	int scanning = grib.getLong("jScansPositively") ? 1 : -1;
	double lat =  scanning * grib.getDouble("jDirectionIncrementInDegrees");

	grib_get_double_array(grib.id(),"pl",pl,&res);

	double *data = new double[nb];

	size_t aux2 = size_t(nb);
	int nblon = 2*res;
	float width = east-west;
	float step = (width)/(nblon-1); 
	// We have to determine if the field is global! 
        bool global = east-west > 360 - 5*step;
        if (global) {
                Log::dev() << "YES THE FIELD IS GLOBAL" << endl;
                east = west + 360;
        }

	
	grib_get_double_array(grib.id(),"values",data,&aux2);
	int d = 0;
	for ( size_t i = 0; i < res; i++)
	{
		float lon = west;
	
		float lon1 = west;
		if ( pl[i] == 0 ) {
			// add missing data 
			for ( int x = 0; x < nblon; x++ ) 
				(*matrix)->push_back(missing);
		}
		else
		{
			unsigned int p1 = 0;
			unsigned int p2 = 1;
			vector<double> p;

			for ( int ii = 0; ii < pl[i]; ii++)
			{
				p.push_back(data[d]);
				d++;
			}
		
			float lon2 = lon1 + (width/(p.size()-1));

			for ( int x = 0; x < nblon; x++ )
			{
				float d1 = (lon2 - lon)/(lon2-lon1);
				float d2 = 1-d1;
				double val; 
				if (p[p1] == missing ||  p[p2] == missing)
					val = missing;
				else
					val = (p[p1] * d1) + (p[p2] * d2);
				(*matrix)->push_back(val);
				lon += step;
				if ( lon >= lon2 )
				{
					p1++;
					if ( p1==p.size() )
					{
						p1 = 0;
					}
					p2++;
					if ( p2 == p.size() )
					{
						p2 = 0;
					}
					lon1 = lon2;
					lon2 += (width)/(p.size()-1);
				}
			}
		}
	}
    
	delete [] data;

	float lon = (width) / (nblon-1);
//	float lat = (north - south) / (2 *res);
	for (int x = 0; x < nblon; x++)
	{
		(*matrix)->columnsAxis().push_back(west+(x*lon));
	}

	double y = north;
	for (long i = 0; i < nblat; i++)
	{
		(*matrix)->rowsAxis().push_back(y);
		y  += lat;
	}

	(*matrix)->setMapsAxis();
}


/*
 * Imported from Metview MvGrid...
 */ 


void GribRotatedInterpretor::print(ostream& out) const
{
	out << "GribRotatedInterpretor[]";
}


GeoPoint GribLambertAzimutalInterpretor::unrotate(double lat, double lon) const
{
	  
	  return GeoPoint(lon, lat);
}

pair<double, double> GribRotatedInterpretor::unrotate( double lat_y, double lon_x) const
{
	const double cToRadians         = M_PI/180.0;    
	double ZRADI  = 1./cToRadians; 
	double ZSYCEN = sin(cToRadians*(southPoleLat_+90.));  
	double ZCYCEN = cos(cToRadians*(southPoleLat_+90.));            
	double ZSXROT = sin(cToRadians*lon_x);                                       
	double ZCXROT = cos(cToRadians*lon_x);                                   
	double ZSYROT = sin(cToRadians*lat_y);                                    
	double ZCYROT = cos(cToRadians*lat_y);                                  
	double ZSYREG = ZCYCEN*ZSYROT + ZSYCEN*ZCYROT*ZCXROT;             
	ZSYREG = MAX( MIN(ZSYREG, +1.0), -1.0 );                                            
	double PYREG = asin(ZSYREG)*ZRADI;                                   
	double ZCYREG = cos(PYREG*cToRadians);                         
	double ZCXMXC = (ZCYCEN*ZCYROT*ZCXROT - ZSYCEN*ZSYROT)/ZCYREG;                           
	ZCXMXC = MAX( MIN(ZCXMXC, +1.0), -1.0 );                                                
	double ZSXMXC = ZCYROT*ZSXROT/ZCYREG;                                       
	double ZXMXC  = acos(ZCXMXC)*ZRADI;                                            
	if( ZSXMXC < 0.0)                                                                             
		ZXMXC = -ZXMXC;                                                          
	double PXREG = ZXMXC + southPoleLon_;                        
	return make_pair( PYREG, PXREG );                                     
}               

pair<double, double> GribRotatedInterpretor::rotate( double lat_y, double lon_x) const
{
  const double cToRadians         = M_PI/180.0;
  double ZRADI  = 1./cToRadians;
  double ZSYCEN = sin(cToRadians*(southPoleLat_+90.));
  double ZCYCEN = cos(cToRadians*(southPoleLat_+90.));

  double ZXMXC  = cToRadians*(lon_x - southPoleLon_);
  double ZSXMXC = sin(ZXMXC);
  double ZCXMXC = cos(ZXMXC);
  double ZSYREG = sin(cToRadians*lat_y);
  double ZCYREG = cos(cToRadians*lat_y);
  double ZSYROT = ZCYCEN*ZSYREG - ZSYCEN*ZCYREG*ZCXMXC;
  ZSYROT = MAX( MIN(ZSYROT, +1.0), -1.0 );

  double PYROT  = asin(ZSYROT)*ZRADI;

  double ZCYROT = cos(PYROT*cToRadians);
  double ZCXROT = (ZCYCEN*ZCYREG*ZCXMXC + ZSYCEN*ZSYREG)/ZCYROT;
  ZCXROT = MAX( MIN(ZCXROT, +1.0), -1.0 );
  double ZSXROT = ZCYREG*ZSXMXC/ZCYROT;

  double PXROT = acos(ZCXROT)*ZRADI;

  if( ZSXROT < 0.0)
    PXROT = -PXROT;

  return make_pair( PYROT, PXROT );
}


void GribLambertAzimutalInterpretor::interpretAsMatrix(const GribDecoder& grib, Matrix** matrix) const
{
	  long nblon = grib.getLong("numberOfPointsAlongXAxis");
	  long nblat = grib.getLong("numberOfPointsAlongYAxis");
	  
	  ProjectedMatrix* lambert = new ProjectedMatrix(nblat, nblon);
	  
	  if ( *matrix == 0 ) *matrix = lambert;
	  
	  size_t nb;
	  grib_get_size(grib.id(), "values", &nb);

	  

	  Log::dev() << "numberOfFieldValues[" << nb << "]" << "\n";
	  double missing = -std::numeric_limits<double>::max();
	  missing = grib.getDouble("missingValue");    
	  lambert->missing(missing);

	  double north = grib.getDouble("latitudeOfFirstGridPointInDegrees");
	  double west = grib.getDouble("longitudeOfFirstGridPointInDegrees");
	  
	  Log::dev() << "NewAPI---> area[" << west << ", " << north << "]" << "\n";
	  
	  //lambert->standardParallel(grib.getDouble("standardParallelInMicrodegrees")/1000000);
	 //lambert->centralLongitude(grib.getDouble("centralLongitudeInMicrodegrees")/1000000);
	  
	  
	  double lon = grib.getDouble("xDirectionGridLengthInMillimetres")/1000;
	  double lat =   grib.getDouble("xDirectionGridLengthInMillimetres")/1000;
	 // double x = lambert->x(west, north);
	  //double y = lambert->y(west, north);
	  //double newlat = lambert->real_row(y,x);
	  //double newlon = lambert->real_column(y,x);
	  Log::dev() << "New API--->GridIncrements[" << lon << ", " << lat << "]" << "\n"; 
	  Log::dev() << "New API--->GridDimensions[" << nblon << ", " << nblat << "]" << "\n";
	  //Log::dev() << "New API--->Grid[<" << west << ", " << north << "], [" << x << ", " << y << "], [" << newlon << ", " << newlat << "\n";
	   

	  double *data = new double[nb]; 
	  double *latitudes = new double[nb]; 
	  double *longitudes = new double[nb];
	  size_t aux = size_t(nb);
	  grib_get_double_array(grib.id(),"values",data,&aux); 
	  grib_get_double_array(grib.id(),"latitudes",latitudes,&aux); 
	  grib_get_double_array(grib.id(),"longitudes", longitudes,&aux); 
	  lambert->missing(missing);
	  
	  for (unsigned int i = 0; i < nb; i++)
	  {
	        lambert->push_back(data[i]);
	        if (longitudes[i] > 180) {
	        	longitudes[i] -= 360;
	        }
	        lambert->rowsAxis().push_back(latitudes[i]);   
	        lambert->columnsAxis().push_back(longitudes[i]);
	  }
	  
	  Log::dev() << "top-left[" << latitudes[0] << ", " << longitudes[0] << "]" << endl;
	  Log::dev() << "top-right[" << latitudes[nblon-1] << ", " <<  longitudes[nblon-1] << "]" << endl; 
	  Log::dev() << "top-right[" << latitudes[nb-1] << ", " <<  longitudes[nb-1] << "]" << endl;


	  delete [] data;  
	  delete [] latitudes;
	  delete [] longitudes;  
	  
	  lambert->setMapsAxis();
	  Log::dev() << "top-left[" << lambert->real_row(0,0) << ", " << lambert->real_column(0,0) << "]" << endl;
	  Log::dev() << "top-right[" << lambert->real_row(0,nblon-1) << ", " << lambert->real_column(0,nblon-1) << "]" << endl;
	  Log::dev() << "bottom-right[" << lambert->real_row(nblat-1, nblon-1) << ", " << lambert->real_column(nblat-1,nblon-1) << "]" << endl;
	  Log::dev() << "bottom-left[" << lambert->real_row(nblat-1, 0) << ", " << lambert->real_column(nblat-1, 0) << "]" << endl;

	  Log::dev() << **matrix<< endl;
}



void GribLambertAzimutalInterpretor::print(ostream& out) const
{
	out << "GribLambertAzimutalInterpretor[]";
}

void GribRotatedInterpretor::interpretAsMatrix(const GribDecoder& grib, Matrix** matrix) const
{
	long nblon = grib.getLong("numberOfPointsAlongAParallel");
	long nblat = grib.getLong("numberOfPointsAlongAMeridian");

	*matrix = new Matrix(nblat, nblon);

	Matrix* helper = new Matrix(nblat, nblon); // settup as the equivalent regular matrix! 

	size_t nb;
	grib_get_size(grib.id(), "values", &nb);

	Log::dev() << "numberOfFieldValues[" << nb << "]" << "\n";
	double missing = -std::numeric_limits<double>::max();
	grib.setDouble("missingValue", missing);    
	helper->missing(missing);
	(*matrix)->missing(missing);

	double north = grib.getDouble("latitudeOfFirstGridPointInDegrees");
	double west = grib.getDouble("longitudeOfFirstGridPointInDegrees");
	double south = grib.getDouble("latitudeOfLastGridPointInDegrees");; 
	double east = grib.getDouble("longitudeOfLastGridPointInDegrees");;
	longitudesSanityCheck(west, east);

	Log::dev() << "NewAPI---> area[" << west << ", " << north << ", " << east << ", " << south << "]" << "\n";

	double lon = (east-west)/(nblon-1);
	double lat =   (south-north)/(nblat-1);

	Log::dev() << "calcul -->" << lon << " (from->" << west << " to-->" << west + (nblon-1) *lon << ")" <<  endl;
    
	double y = north;
    	for (int i = 0; i < nblat; i++)
    	{
    	    
    		helper->rowsAxis().push_back(y);
    		y  = north + (i+1)*lat;
    		
    	}
    	
   
   

   
	double x = west;
  	for (int i = 0; i < nblon; i++)
  	{
  	    
  		helper->columnsAxis().push_back(x);
  		x  = west + (i+1)*lon;
  		
  	}
  
	helper->setMapsAxis();


	try { 
		helper->resize(nb);
		size_t aux = size_t(nb);
		grib_get_double_array(grib.id(),"values", &helper->front(),&aux); 
	} 
	catch (...) 
	{
		throw MagicsException("Not enough memory");
	}
   
  
	southPoleLat_  = grib.getDouble("latitudeOfSouthernPoleInDegrees"); 
	southPoleLon_  = grib.getDouble("longitudeOfSouthernPoleInDegrees");
	

	lon = west; 
	lat = north; 
	double steplon = (east-west)/(nblon-1);
	double steplat = (south-north)/(nblat-1);

	// Fisrt try to find the bounding box 
	vector<double>  rows;
	rows.reserve(nb);
	vector<double>  columns;
	columns.reserve(nb);
	vector<double>  values;
	values.reserve(nb);

	for ( int j = 0; j < nblat; j++ )
	{
		lon = west;
		
		for ( int i = 0; i < nblon; i++ )
		{
			pair<double, double> point = unrotate(lat,lon);
			pair<double, double> rpoint = rotate(point.first, point.second);
			
			
			
			rows.push_back(point.first);   
			columns.push_back(point.second);
			
			lon += steplon;
		}
		lat += steplat;
	}

	double minx = *std::min_element(columns.begin(), columns.end());
	double maxx = *std::max_element(columns.begin(), columns.end());
	double miny = *std::min_element(rows.begin(), rows.end());
	double maxy = *std::max_element(rows.begin(), rows.end());
	
	
	

	double stepx = (maxx - minx)/(nblon-1);
	double stepy = (maxy - miny)/(nblat-1);
	x = minx;
	y = miny;
	// Create the Axis for Regular Matrix..
	for ( int i = 0; i < nblon; i++)
	{
		(*matrix)->columnsAxis().push_back(x); 
		x += stepx;
	}
	for ( int j = 0; j < nblat; j++)
	{
		(*matrix)->rowsAxis().push_back(y); 
		y +=  stepy;
	}

	miny = std::min(north, south);
	maxy = std::max(north, south);
	minx = std::min(west, east);
	maxx = std::max(west, east);
	
	(*matrix)->setMapsAxis();
	for ( int j = 0; j < nblat; j++)
	{
		for ( int i = 0; i < nblon; i++)
		{
			
		    pair<double, double> point = rotate( (*matrix)->row(j, i), (*matrix)->column(j, i) );
		    if ( point.first >  miny && point.first < maxy && point.second > minx && point.second < maxx )		    {	
		    	(*matrix)->push_back(helper->interpolate(point.first, point.second));
		    	
		    }
		    else {
		    	(*matrix)->push_back(missing);
		    	
		    }
		}
	}
}
