/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file PointsHandler.h
    \brief Definition of the Template class PointsHandler.
    
    Magics Team - ECMWF 2004
    
    Started: Tue 20-Jan-2004
    
    Changes:
    
*/

#ifndef PointsHandler_H
#define PointsHandler_H

#include "magics.h"

#include "BasePointsHandler.h"
#include "Transformation.h"

namespace magics {

template <class P>
class PointsHandler: public AbstractPoints<P> {

public:
	PointsHandler(AbstractPoints<P>& handler): handler_(handler) {}
	virtual ~PointsHandler() {}
    //! Method to set the current position to the first point.(abstract)
    virtual void setToFirst() const { handler_.setToFirst(); }
    //! Method to test the end of collection.
    virtual bool more()    const    { return handler_.more(); }
    //! Method to return the current value
    virtual const P& current() const { return handler_.current(); }
    //! Method to advance
    virtual void advance() const  { handler_.advance(); }

protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream& out) const { out << "PointsHandler on " << handler_; }
     AbstractPoints<P>& handler_;

private:
    //! Copy constructor - No copy allowed
	PointsHandler(const PointsHandler&);
    //! Overloaded << operator to copy - No copy allowed
	PointsHandler& operator=(const PointsHandler&);
    
// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const PointsHandler<P>& p)
		{ p.print(s); return s; }

};
template <class P>
class BatchPointsHandler : public AbstractPoints<P>
{
public:
	BatchPointsHandler(AbstractPoints<P>& handler, int count): handler_(handler),  count_(count), more_(0), last_(false) {}
	virtual ~BatchPointsHandler() {}
    //! Method to set the current position to the first point.(abstract)
    virtual void setToFirst() const {}
    //! Method to test the end of collection.
    virtual bool more() const 
    {
         if ( last_ ) {
            last_ = false;
            return false;
        }
        //Log::debug() << "batch " << more_ << "\n";
        if (more_++ < count_) return handler_.more();  
       
        last_ = true;
        return true;
    }
    //! Method to return the current value
    virtual const P& current() const { return handler_.current(); }
    //! Method to advance
    virtual void advance()  const { handler_.advance(); }
  
protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream& out) const { out << "BatchPointsHandler(" << count_ << "items) on " << handler_; }
     AbstractPoints<P>& handler_;
     int count_;
     mutable int more_;
     mutable bool last_;
     
private:
    //! Copy constructor - No copy allowed
	BatchPointsHandler(const BatchPointsHandler<P>&);
    //! Overloaded << operator to copy - No copy allowed
	BatchPointsHandler& operator=(const BatchPointsHandler<P>&);
    
// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const BatchPointsHandler<P>& p)
		{ p.print(s); return s; }

};


template <class P>
class BoxPointsHandler : public AbstractPoints<P>
{
public:
	BoxPointsHandler(AbstractPoints<P>& handler, const Transformation& transformation) : 
		handler_(handler),
		transformation_(transformation) {}
	virtual ~BoxPointsHandler() {}
	
    //! Method to set the current position to the first point.(abstract)    
    virtual void setToFirst() const {  
    	handler_.setToFirst(); 
    	advance(); 
    	
    }
    
    //! Method to test the end of collection.
    virtual bool more() const 
    {
    	return more_;
        
    }
    
    //! Method to return the current value
    virtual const P& current() const { return current_; }
    
    //! Method to advance
    virtual void advance()  const { 
    	while ( handler_.more() ) {
    		current_ = handler_.current();
    		handler_.advance();
    		
    		if (  current_.in(transformation_.getMinX(), transformation_.getMaxX(), transformation_.getMinY(), transformation_.getMaxY()) ) {
    			more_ = true;
    			return;
    		}
    	
    	}
    	more_ = false;
    }
  
protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream& out) const { out << "BocPointsHandler() on " << handler_; }
     AbstractPoints<P>& handler_;
     const Transformation& transformation_;
     vector<P>             duplicates_;
     mutable P                current_;
     mutable bool more_;

};
    
} // namespace magics

#endif
