/*
 *   pam_abl - a PAM module and program for automatic blacklisting of hosts and users
 *
 *   Copyright (C) 2005 Andy Armstrong andy@hexten.net
 *   Copyright (C) 2009 Chris Tasma pam-abl@deksai.com
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __PAM_ABL_H
#define __PAM_ABL_H

#include <security/pam_appl.h>
#include <security/pam_modules.h>
#include <db.h>

#include <ctype.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>

#define PAM_SM_AUTH

#define DB_TIME "time"
#define DB_STATE "state"

/* DB_BUFFER_SMALL is a relatively new error code */
#ifdef DB_BUFFER_SMALL
#define TOOSMALL DB_BUFFER_SMALL
#else
#define TOOSMALL ENOMEM
#endif

#define DBPERM 0600

#define MODULE_NAME "pam_abl"
#define DATA_NAME   MODULE_NAME

#define HOURSECS    (60 * 60)
/* Host purge time in seconds */
#define HOST_PURGE  (HOURSECS * 24)
/* User purge time in seconds */
#define USER_PURGE  (HOURSECS * 24)
#define COMMAND_SIZE 1024

#define CONFIG "/etc/security/pam_abl.conf"

typedef enum { 
    FAIL,
    WHITELIST,
    CHECK,
    PURGE,
    SHOW
} commands;

typedef enum { 
    USER,
    HOST,
    BLOCKED,
    CLEAR
} state_enum;

typedef struct abl_string {
    struct abl_string *link;
} abl_string;

typedef struct {
    state_enum subject;
    state_enum state;
    int state_change;
    const char *user;
    const char *host;
    const char *service;
    DB   *utdb;
    DB   *usdb;
    DB   *htdb;
    DB   *hsdb;

} abl_info;

typedef struct {
    /* Session handle */
    pam_handle_t    *pamh;

    /* Standard args */
    int             debug;
    int             no_warn;
    int             use_first_pass;
    int             try_first_pass;
    int             use_mapped_pass;
    int             expose_account;

    /* Our args */
    const char      *host_db;
    const char      *host_rule;
    long            host_purge;
    const char      *host_blk_cmd;
    const char      *host_clr_cmd;
    const char      *user_db;
    const char      *user_rule;
    long            user_purge;
    const char      *user_blk_cmd;
    const char      *user_clr_cmd;

    /* Storage */
    abl_string      *strs;
} abl_args;

typedef struct abl_context {
    abl_args    *args;
    abl_info    *info;
} abl_context;

//int check(const abl_args *args, const char *dbname, const char *user, const char *service,
//                 const char *rule, const char *kv, time_t tm, int *rv);
/* functions from log.c */

void log_out(int pri, const char *format, ...);
#if !defined(TOOLS)
void log_pam_error(const abl_args *args, int err, const char *what);
#endif
void log_sys_error(int err, const char *what);
void log_info(const char *format, ...);
void log_warning(const abl_args *args, const char *format, ...);
void log_debug(const abl_args *args, const char *format, ...);
void log_db_error(int err, const char *what);

/* functions from config.c */

void config_clear(pam_handle_t *pamh, abl_args *args);
int config_parse_args(pam_handle_t *pamh, int argc, const char **argv, abl_args *args);
int config_parse_file(const char *name, abl_args *args);
void config_free(abl_args *args);
void dump_args(const abl_args *args);

/* functions from rule.c */

int rule_purge(DBT *rec, long maxage, time_t now);
int rule_parse_time(const char *p, long *t, long min);
int rule_test(const abl_args *args, const char *rule,
              const char *user, const char *service,
              const time_t *history, int histsz, time_t now);
int rule_matchperiods(const abl_args *args, const time_t *history, int histsz,
                        time_t now, const char **rp);


/* functions from common.c */
void die(const char *msg, ...);
int check(const abl_args *args, abl_info *info, time_t tm);
int check_user(const abl_args *args, abl_info *info, time_t tm); 
int check_host(const abl_args *args, abl_info *info, time_t tm);
int record(const abl_args *args, abl_info *info, time_t tm, long maxage);
int dbopen(const abl_args *args, const char *dbname, const char *dbfile, DB **db);
int dbget(DB *db, DBT *key, DBT *dbtdata);
int prepare_command(const abl_args *args, const char *cmd, const abl_info *info,
        char **string);
int run_command(const abl_args *args, const abl_info *info);
int update_status(const abl_args *args, abl_info *info);
void make_key(DBT *dbt, const char *key);

#endif /* __PAM_ABL_H */


