#include "osl/record/csaRecord.h"
#include "osl/record/csaIOError.h"
#include "osl/state/simpleState.h"
#include "osl/oslConfig.h"
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/foreach.hpp>
#include <iostream>
#include <fstream>
#include <stdexcept>
#include <cassert>
#include <string>
#include <sstream>

/* ------------------------------------------------------------------------- */

namespace osl
{
  namespace record
  {
    namespace
    {
      const SearchInfo makeInfo(const SimpleState& initial,
				const std::string& line,
				Move last_move)
      {
	std::istringstream is(line);
	SearchInfo info;
	is >> info.value;

	NumEffectState state(initial);
	std::string s;
	while (is >> s)
	{
	  if (s == csa::show(last_move)) // effective only if s is the first move in a comment
	    continue;
	  last_move = Move::INVALID();
	  try
	  {
	    const Move move = ((s == "%PASS" || /* gekisashi */ s == "<PASS>")
			       ? Move::PASS(state.turn())
			       : csa::strToMove(s, state));
	    if (move.isPass() || (move.isNormal() && state.isValidMove(move,false)))
	    {
	      info.moves.push_back(move);
	      state.makeMove(move);
	      continue;
	    }
	  }
	  catch(CsaIOError& e)
	  {
	    // fall through
	  }
	  std::cerr << "drop illegal move in comment " << s << std::endl;
	  break;
	}
	return info;
      }
      void csaParseLine(boost::shared_ptr<RecordVisitor>& rv, std::string s,
			CArray<bool,9>& board_parsed,
			bool parse_move_comment=true)
      {
	Record *rec=rv->getRecord();
	SimpleState* state=rv->getState();
	while (! s.empty() && isspace(s[s.size()-1])) // ignore trailing garbage
	  s.resize(s.size()-1);
	if (s.length()==0) 
	  return;
	switch(s.at(0)){
 	case '\'': /* コメント行 */
	  if (s.substr(1,2) == "* ")
          {
	    MoveRecord *mr = rv->getLastMove();
            if (mr)
              mr->addComment(s.substr(3));
          }
	  else if (s.substr(1,2) == "**" && parse_move_comment) 
	  {
	    MoveRecord *mr = rv->getLastMove();
            if (mr)
              mr->info = makeInfo(*state, s.substr(3), mr->getMove());
	  }
	  return;
	case '$': /* コメント行 */
	  if (s.find("$START_TIME:") == 0) {
	    const std::string YYMMDD = s.substr(12,10);
	    rec->setDate(YYMMDD);
	    return;
	  }
	  rec->addInitialComment(s.substr(1));
	  return;
	case 'V': /* バージョン番号 */
	  rec->setVersion(s.substr(1));
	  return;
	case 'N': /* 対局者名 */
	  switch(s.at(1)){
	  case '+':
	  case '-':
	    rec->setPlayer(csa::charToPlayer(s.at(1)),s.substr(2));
	    break;
	  default:
	    std::cerr << "Illegal csa line " << s << std::endl;
	    throw CsaIOError("illegal csa line "+s);
	  }
	  break;
	case 'P': /* 開始盤面 */
	  switch(s.at(1)){
	  case 'I': /* 平手初期配置 */
	    board_parsed.fill(true);
	    state->init(HIRATE);
	    break;
	  case '+': /* 先手の駒 */
	  case '-':{ /* 後手の駒 */
	    Player pl=csa::charToPlayer(s.at(1));
	    for(int i=2;i<=(int)s.length()-4;i+=4){
	      Square pos=csa::strToPos(s.substr(i,2));
	      if(s.substr(i+2,2) == "AL"){
		state->setPieceAll(pl);
	      }
	      else{
		Ptype ptype=csa::strToPtype(s.substr(i+2,2));
		state->setPiece(pl,pos,ptype);
	      }
	    }
	    break;
	  }
	  default:
	    if(isdigit(s.at(1))){
	      const int y=s.at(1)-'0';
	      board_parsed[y-1] = true;
	      for(unsigned int x=9,i=2;i<s.length();i+=3,x--){
		if (s.at(i) != '+' && s.at(i) != '-' && s.find(" *",i)!=i) {
		  if (OslConfig::inUnitTest())
		    throw CsaIOError("parse board error " + s);
		  else
		    std::cerr << "possible typo for empty square " << s << "\n";
		}   
		if (s.at(i) != '+' && s.at(i) != '-') continue;
		Player pl=csa::charToPlayer(s.at(i));
		Square pos(x,y);
		Ptype ptype=csa::strToPtype(s.substr(i+1,2));
		state->setPiece(pl,pos,ptype);
	      }
	    }
	  }
	  break;
	case '+':
	case '-':{
	  Player pl=csa::charToPlayer(s.at(0));
	  if(s.length()==1){
	    state->setTurn(pl);
	    rec->setInitialState(*state);
	    state->initPawnMask();
	  }
	  else{ // actual moves
	    const Move m = csa::strToMove(s,*state);
	    if (! state->isValidMove(m))
	    {
	      std::cerr << "Illegal move " << m << std::endl;
	      throw CsaIOError("illegal move "+s);
	    }
	    rv->addMoveAndAdvance(m);
	    return;
	  }
	  break;
	}
	case 'T':
        {
          MoveRecord *mr = rv->getLastMove();
          if (mr)
            mr->setTime(atoi(s.c_str()+1));
	  return;
        }
	case '%':
	  if (s.find("%TORYO") == 0 || s.find("%ILLEGAL_MOVE") == 0)
	    rec->setResult((state->turn() == BLACK) 
			   ? Record::WHITE_WIN : Record::BLACK_WIN);
	  else if (s.find("%SENNICHITE") == 0)
	    rec->setResult(Record::SENNNICHITE);
	  else if (s.find("%KACHI") == 0)
	    rec->setResult((state->turn() == BLACK) 
			   ? Record::BLACK_WIN : Record::WHITE_WIN);
	  else if (s.find("%JISHOGI") == 0 || s.find("%HIKIWAKE") == 0)
	    rec->setResult(Record::JISHOGI);
	  else if (s.find("%+ILLEGAL_ACTION") == 0)
	    rec->setResult(Record::WHITE_WIN);
	  else if (s.find("%-ILLEGAL_ACTION") == 0)
	    rec->setResult(Record::BLACK_WIN);
	  return;
	default:
	  throw CsaIOError("unknown character in csaParseLine "+s);
	}
      }
    } // anonymous namespace
  } // namespace record
} // namespace osl

osl::record::csa::
InputStream::InputStream(std::istream& is) 
  : is(is), 
    rv(boost::shared_ptr<record::RecordVisitor>(new record::RecordVisitor()))
{
  if (! is)
  {
    std::cerr << "InputStream::InputStream cannot read \n";
    abort();
  }
}
  
osl::record::csa::
InputStream::InputStream(std::istream& is, boost::shared_ptr<record::RecordVisitor> rv) 
  : is(is), rv(rv)
{
  if (! is)
  {
    std::cerr << "InputStream::InputStream cannot read \n";
    abort();
  }
}
  
osl::record::csa::
InputStream::~InputStream(){}
  
void osl::record::csa::
InputStream::load(Record* rec)
{
  //  rec->init();
  state.init();
  rv->setState(&state);
  rv->setRecord(rec);
  std::string line;
  CArray<bool, 9> board_parsed = {{ false }};
  while (std::getline(is, line)) 
  {
    // quick hack for \r
    if ((! line.empty())
	&& (line[line.size()-1] == 13))
      line.erase(line.size()-1);

    std::vector<std::string> elements;
    boost::algorithm::split(elements, line, boost::algorithm::is_any_of(","));
    BOOST_FOREACH(std::string e, elements) {
      boost::algorithm::trim(e);
      boost::algorithm::trim_left(e);
      csaParseLine(rv, e, board_parsed, !OslConfig::inUnitTest());
    }
  }
  if (*std::min_element(board_parsed.begin(), board_parsed.end()) == false)
    throw CsaIOError("incomplete position description in csaParseLine");
  assert(state.isConsistent());
}

osl::record::csa::
CsaFile::CsaFile(const std::string& fileName)
{
  std::ifstream ifs(fileName.c_str());
  if (! ifs)
  {
    const std::string msg = "CsaFile::CsaFile file cannot read ";
    std::cerr << msg << fileName << "\n";
    throw CsaIOError(msg + fileName);
  }
  InputStream irs(ifs);
  irs.load(&rec);
}

osl::record::csa::
CsaFile::~CsaFile()
{
}

const osl::record::Record& osl::record::csa::
CsaFile::getRecord() const
{
  return rec;
}

const osl::NumEffectState osl::record::csa::
CsaFile::getInitialState() const
{
  return NumEffectState(rec.getInitialState());
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
