#include "osl/move_classifier/kingOpenMove.h"
#include "osl/move_classifier/moveAdaptor.h"
#include "osl/move_action/store.h"
#include "osl/move_generator/allMoves.h"
#include "osl/record/record.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/container/moveVector.h"
#include "osl/state/numEffectState.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"

#include <iostream>
#include <fstream>

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
typedef osl::NumEffectState test_state_t;

class KingOpenMoveTest : public CppUnit::TestFixture {
  CPPUNIT_TEST_SUITE( KingOpenMoveTest );
  CPPUNIT_TEST( testIsKingOpenMove );
  CPPUNIT_TEST( testFile );
  CPPUNIT_TEST_SUITE_END();
public:
  void testIsKingOpenMove();
  void testFile();
};

CPPUNIT_TEST_SUITE_REGISTRATION(KingOpenMoveTest);

using namespace osl;
using namespace osl::move_action;
using namespace osl::move_generator;
using namespace osl::move_classifier;

void KingOpenMoveTest::testIsKingOpenMove(){
  {
    // 王手がかかった状態での開き王手
    NumEffectState state=CsaString(
"P1+NY+RY *  *  * -FU-OU-KE-KY\n"
"P2 *  *  *  *  * +GI-KI *  *\n"
"P3 * -FU *  *  *  * -KI-FU-FU\n"
"P4 * +FU * -FU *  *  *  *  *\n"
"P5 *  * -KE * +FU *  * +FU *\n"
"P6+KE *  * +FU+GI *  *  * +FU\n"
"P7 *  * -UM-KA *  *  *  *  *\n"
"P8 *  * +FU *  *  *  *  *  * \n"
"P9 * +OU * -GI *  *  *  * -NG\n"
"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU\n"
"P-00KI00KY00FU00FU\n"
"P-00AL\n"
"-\n"
).getInitialState();
    // 開き王手ではない
    CPPUNIT_ASSERT(!(KingOpenMove<WHITE>::isMember(state,GOLD,Position(3,2),Position(4,2))));
    // 開き王手
    CPPUNIT_ASSERT((KingOpenMove<WHITE>::isMember(state,PAWN,Position(4,1),Position(4,2))));
  }
  {
    // 
    NumEffectState state=CsaString(
"P1+NY+RY *  *  * -FU-OU-KE-KY\n"
"P2 *  *  *  *  *  * -KI *  *\n"
"P3 * -FU *  *  * +GI-KI-FU-FU\n"
"P4 * +FU * -FU *  *  *  *  *\n"
"P5 *  * -KE * +FU-KA * +FU *\n"
"P6+KE * +FU+FU+GI *  *  * +FU\n"
"P7 *  * -UM *  *  *  *  *  *\n"
"P8 *  *  *  *  *  *  *  *  * \n"
"P9 * +OU * -GI *  *  *  * -NG\n"
"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU\n"
"P-00KI00KY00FU00FU\n"
"P-00AL\n"
"+\n"
).getInitialState();
    // 開き王手ではない
    CPPUNIT_ASSERT(!(KingOpenMove<BLACK>::isMember(state,SILVER,Position(5,6),Position(6,7))));
  }
}
template<typename State> 
static bool isKingOpenMove(const State& state_org,Move move){
  State state = state_org;
  ApplyMoveOfTurn::doMove(state, move);  
  return state.hasEffectBy(alt(move.player()),state.getKingPosition(move.player()));
}

template<class State>
static void testMoveFile(const std::string& fileName){
  Record rec=CsaFile(fileName).getRecord();
  NumEffectState sState(rec.getInitialState());
  State state(sState);
  vector<osl::Move> moves=rec.getMoves();
  for(unsigned int i=0;i<moves.size();i++){
    MoveVector allMoves;
    Player turn=state.getTurn();
    // 敵から王手がかかっている時は扱わない
    // 本当は扱った方が良いが
    if(state.hasEffectBy(alt(turn),state.getKingPosition(turn)))
      continue;
    {
      Store store(allMoves);
      AllMoves<Store>::
	generate(turn,state,store);
    }
    for(size_t j=0;j<allMoves.size();j++){
      Move move=allMoves[j];
      // dropmoveは扱わない
      if (move.isDrop()) 
	  continue;
      // Kingが動く手は扱わない
      if (move.ptype()==KING) 
	  continue;
      CPPUNIT_ASSERT_EQUAL(isKingOpenMove(state,move),
			   PlayerMoveAdaptor<KingOpenMove>
			   ::isMember(state,move));
    }
    ApplyMoveOfTurn::doMove(state, moves[i]);
  }
}

void KingOpenMoveTest::testFile(){
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=50;
  if (isShortTest) 
    count=5;
  std::string fileName;
  while((ifs >> fileName) && ++i<count){
    if(fileName == "") 
      break;
    if (! isShortTest)
      std::cerr << fileName << " ";
    testMoveFile<test_state_t>(OslConfig::testCsaFile(fileName));
  }
}
