#include "osl/effect_util/neighboring8Direct.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/apply_move/applyMove.h"
#include "osl/state/numEffectState.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iostream>
#include <fstream>

using namespace osl;

class Neighboring8DirectTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(Neighboring8DirectTest);
  CPPUNIT_TEST(testTable);
  CPPUNIT_TEST(testEqual);
  CPPUNIT_TEST(testRook);
  CPPUNIT_TEST(testBishop);
  CPPUNIT_TEST(testLance);
  CPPUNIT_TEST(testFindNearest);
  CPPUNIT_TEST(testAdditional);
  CPPUNIT_TEST_SUITE_END();
public:
  void testTable();
  void testEqual();
  void testRook();
  void testBishop();
  void testLance();
  void testFindNearest();
  void testAdditional();
};

CPPUNIT_TEST_SUITE_REGISTRATION(Neighboring8DirectTest);

void Neighboring8DirectTest::testBishop()
{
  const NumEffectState state(CsaString(
			    "P1-KY+HI *  *  *  *  * -KE-KY\n"
			    "P2 *  * +UM-FU *  *  *  *  * \n"
			    "P3-KE-KI *  * +UM-GI-FU+FU-FU\n"
			    "P4 * -GI-OU * -FU *  * -FU * \n"
			    "P5-FU *  *  *  * +GI *  *  * \n"
			    "P6 * +FU-FU *  *  *  *  *  * \n"
			    "P7+FU *  *  *  *  * +FU * +FU\n"
			    "P8+KY+KI+KI *  *  *  *  *  * \n"
			    "P9+OU+KE *  *  *  *  * +KE+KY\n"
			    "P+00KI\n"
			    "P-00HI00GI00FU00FU00FU00FU00FU00FU\n"
			    "-\n").getInitialState());
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,PBISHOP), Position(7,2), Position(7,4)));
  
}


void Neighboring8DirectTest::testRook()
{
  const NumEffectState state(CsaString("P1-KY-KE *  *  *  *  * -KE-KY\n"
				    "P2 *  *  *  *  *  * -KI * -OU\n"
				    "P3 *  * -FU *  * -KI *  * -KA\n"
				    "P4-FU-HI *  * -FU * -FU * -GI\n"
				    "P5 * -FU+FU * +GI-FU * -FU * \n"
				    "P6+FU * +HI *  *  *  *  * -FU\n"
				    "P7 * +FU * -TO * +FU+FU+FU * \n"
				    "P8+KY *  *  *  * +KI+KI+GI+KY\n"
				    "P9 * +KE *  * +KA *  * +KE+OU\n"
				    "P-00FU00FU00FU\n"
				    "P+00GI\n"
				    "+").getInitialState());
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,ROOK), Position(1,6), Position(1,9)));
  
}

void Neighboring8DirectTest::testLance()
{
  {
    const NumEffectState state(CsaString(
			      "P1-KY-KE *  *  *  *  *  * -KY\n"
			      "P2 *  *  *  *  *  * -KI *  * \n"
			      "P3 *  * -FU *  * -KI *  * -OU\n"
			      "P4-FU-HI *  * -FU * -FU *  * \n"
			      "P5 * -FU+FU * +GI-FU * -FU+KY\n"
			      "P6+FU *  *  *  *  * +KE *  * \n"
			      "P7 * +FU * -TO * +FU+FU+FU * \n"
			      "P8+KY *  *  *  * +KI+KI+GI * \n"
			      "P9 * +KE-UM * +KA *  * +KE+OU\n"
			      "P+00GI00FU\n"
			      "P-00HI00GI00FU00FU00FU\n"
			      "-\n").getInitialState());
    CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		   (state, newPtypeO(BLACK,LANCE), Position(1,5), Position(1,3)));
  }
  {
    const NumEffectState state(CsaString(
			      "P1-KY-KE *  *  *  *  *  * -KY\n"
			      "P2 *  *  *  *  *  * -KI *  * \n"
			      "P3 *  * -FU *  * -KI *  * -OU\n"
			      "P4-FU-HI *  * -FU * -FU *  * \n"
			      "P5 * -FU+FU * +GI-FU * -FU+KY\n"
			      "P6+FU *  *  *  *  * +KE *  * \n"
			      "P7 * +FU * -TO * +FU+FU+FU * \n"
			      "P8+KY *  *  *  * +KI+KI+GI * \n"
			      "P9 * +KE-UM * +KA *  * +KE+OU\n"
			      "P+00GI00FU\n"
			      "P-00HI00GI00FU00FU00FU\n"
			      "-\n").getInitialState());
    CPPUNIT_ASSERT(! Neighboring8Direct::hasEffect
		   (state, newPtypeO(BLACK,LANCE), Position(1,5), Position(1,6)));
  }
  {
    const NumEffectState state(CsaString(
			      "P1-KY-KE *  *  *  *  *  * -KY\n"
			      "P2 *  *  *  *  *  * -KI * -OU\n"
			      "P3 *  * -FU *  * -KI *  *  * \n"
			      "P4-FU-HI *  * -FU * -FU *  * \n"
			      "P5 * -FU+FU * +GI-FU * -FU+KY\n"
			      "P6+FU *  *  *  *  * +KE *  * \n"
			      "P7 * +FU * -TO * +FU+FU+FU * \n"
			      "P8+KY *  *  *  * +KI+KI+GI * \n"
			      "P9 * +KE-UM * +KA *  * +KE+OU\n"
			      "P+00GI00FU\n"
			      "P-00HI00GI00FU00FU00FU\n"
			      "-\n").getInitialState());
    CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		   (state, newPtypeO(BLACK,LANCE), Position(1,5), Position(1,2)));
  }
  {
    const NumEffectState state(CsaString(
			      "P1-KY-KE *  *  *  *  *  * -KY\n"
			      "P2 *  *  *  *  *  * -KI-OU * \n"
			      "P3 *  * -FU *  * -KI *  *  * \n"
			      "P4-FU-HI *  * -FU * -FU *  * \n"
			      "P5 * -FU+FU * +GI-FU * -FU+KY\n"
			      "P6+FU *  *  *  *  * +KE *  * \n"
			      "P7 * +FU * -TO * +FU+FU+FU * \n"
			      "P8+KY *  *  *  * +KI+KI+GI * \n"
			      "P9 * +KE-UM * +KA *  * +KE+OU\n"
			      "P+00GI00FU\n"
			      "P-00HI00GI00FU00FU00FU\n"
			      "-\n").getInitialState());
    CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		   (state, newPtypeO(BLACK,LANCE), Position(1,5), Position(2,2)));
  }
  
}

void Neighboring8DirectTest::testFindNearest()
{
  {
    const NumEffectState state(CsaString(
			      "P1-KY-KE-KI-GI * +HI-FU *  * \n"
			      "P2 * -OU-GI * -FU *  *  *  * \n"
			      "P3 *  * -FU-FU-KI *  *  * -FU\n"
			      "P4 *  *  *  *  *  *  *  *  * \n"
			      "P5-FU * +FU *  *  *  *  *  * \n"
			      "P6 *  *  *  * +UM *  *  *  * \n"
			      "P7+FU+FU+KE *  * +FU *  * +FU\n"
			      "P8 *  * +OU * +FU *  *  *  * \n"
			      "P9+KY * +GI+KI *  *  *  * +KY\n"
			      "P+00KA00KI00GI00FU00FU00FU00FU00FU\n"
			      "P-00HI00KE00KE00KY00FU\n"
			      "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(Position(8,3),
			 Neighboring8Direct::findNearest
			 (state, newPtypeO(BLACK,BISHOP), Position(6,5), Position(8,2)));
    CPPUNIT_ASSERT_EQUAL(Position::STAND(),
			 Neighboring8Direct::findNearest
			 (state, newPtypeO(BLACK,BISHOP), Position(3,6), Position(8,2)));
  }
}

void Neighboring8DirectTest::testTable() 
{
  const NumEffectState state((SimpleState(HIRATE)));
  // gold
  // direct
  const Position white_king = Position(5,1);
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,GOLD), Position(4,2), white_king));
  // 3x3
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,GOLD), Position(3,2), white_king));
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(WHITE,GOLD), Position(4,2), white_king));
  // out
  CPPUNIT_ASSERT(!Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,GOLD), Position(2,2), white_king));
  CPPUNIT_ASSERT(!Neighboring8Direct::hasEffect
		 (state, newPtypeO(WHITE,GOLD), Position(3,3), white_king));

  // black king
  const Position black_king = Position(5,9);
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,GOLD), Position(4,8), black_king));
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(WHITE,GOLD), Position(3,8), black_king));
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(WHITE,GOLD), Position(4,8), black_king));
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(WHITE,GOLD), Position(3,7), black_king));
  CPPUNIT_ASSERT(!Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,GOLD), Position(3,7), black_king));
  
  // lance
  CPPUNIT_ASSERT(! Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,LANCE), Position(4,7), black_king));
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(WHITE,LANCE), Position(4,7), black_king));
  CPPUNIT_ASSERT(! Neighboring8Direct::hasEffect
		 (state, newPtypeO(WHITE,LANCE), Position(4,6), black_king));

  // bishop
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,BISHOP), Position(6,3), white_king));
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,BISHOP), Position(8,3), white_king));
  CPPUNIT_ASSERT(!Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,BISHOP), Position(9,4), white_king));

  // rook
  CPPUNIT_ASSERT(Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,ROOK), Position(1,4), Position(1,7)));

  // nearestがはみだす
  CPPUNIT_ASSERT(!Neighboring8Direct::hasEffect
		 (state, newPtypeO(BLACK,BISHOP), Position(8,8), black_king));
}

void Neighboring8DirectTest::testEqual()
{
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int max=100;
  if (isShortTest)
      max=10;
  std::string filename;
  int i=0;
  while((ifs >> filename) && (++i<max)) 
  {
    if (! isShortTest)
      std::cerr << filename << " ";
    std::string full_filename = OslConfig::testCsaFile(filename);

    Record rec=CsaFile(full_filename).getRecord();
    NumEffectState state(rec.getInitialState());
    const vector<osl::Move> moves=rec.getMoves();
    for (size_t i=0; i<moves.size(); ++i)
    {
      const PtypeO ptypeo = moves[i].ptypeO();
      const Position from = moves[i].to();
      const Position to = state.getKingPosition(BLACK);

      const bool stable 
	= Neighboring8Direct::hasEffectNaive(state, ptypeo, from, to);
      const bool current 
	= Neighboring8Direct::hasEffect(state, ptypeo, from, to);
      if (stable != current)
      {
	// 以前の実装は to だけに利きをつける手を除いていた．例穴熊の銀の守り
	if (! state.hasEffectFromTo(ptypeo, from, to))
	{
	  std::cerr << state << moves[i] << "\n";
	  CPPUNIT_ASSERT_EQUAL(stable, current);
	}
      }
      ApplyMoveOfTurn::doMove(state, moves[i]);
    }
  }
}

void Neighboring8DirectTest::testAdditional()
{
  {
    const NumEffectState state(CsaString(
				 "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
				 "P2 * -HI *  *  *  *  * -KA * \n"
				 "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
				 "P4 *  *  *  *  *  *  *  *  * \n"
				 "P5 *  *  *  * +GI *  *  *  * \n"
				 "P6 *  *  *  *  *  *  *  *  * \n"
				 "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
				 "P8 *  *  *  *  *  *  * +HI * \n"
				 "P9+KY+KE * +KI+OU+KI+GI+KE+KY\n"
				 "P+00KA\n"
				 "+\n").getInitialState());
    CPPUNIT_ASSERT(Neighboring8Direct::hasEffectOrAdditional
		   (state, newPtypeO(BLACK,BISHOP), Position(6,6), Position(4,4)));
    CPPUNIT_ASSERT(Neighboring8Direct::hasEffectOrAdditional
		   (state, newPtypeO(BLACK,BISHOP), Position(6,6), Position(4,5)));
    CPPUNIT_ASSERT(Neighboring8Direct::hasEffectOrAdditional
		   (state, newPtypeO(BLACK,BISHOP), Position(6,6), Position(8,8)));

    CPPUNIT_ASSERT(Neighboring8Direct::hasEffectOrAdditional
		   (state, newPtypeO(BLACK,BISHOP), Position(6,6), Position(3,3)));
    CPPUNIT_ASSERT(! Neighboring8Direct::hasEffectOrAdditional
		   (state, newPtypeO(BLACK,BISHOP), Position(6,6), Position(2,3)));
    CPPUNIT_ASSERT(! Neighboring8Direct::hasEffectOrAdditional
		   (state, newPtypeO(BLACK,BISHOP), Position(6,6), Position(9,9)));
  }
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
