/* simpleState.h
 */
#ifndef _SIMPLE_STATE_H
#define _SIMPLE_STATE_H

#include "osl/misc/loki.h"
#include "osl/direction.h"
#include "osl/boardTable.h"
#include "osl/ptype.h"
#include "osl/ptypeTraits.h"
#include "osl/piece.h"
#include "osl/container/pieceMask.h"
#include "osl/container/bitXmask.h"
#include "osl/effectContent.h"
#include "osl/move.h"
#include "osl/player.h"
#include "osl/handicap.h"
#include "osl/misc/carray.h"
#include "osl/effect_action/pieceFilter.h"
#include "osl/apply_move/applyDoUndoXMove.h"
#include "osl/misc/fastCopier.h"
#include "osl/ptypeTable.h"

#include <iosfwd>

namespace osl
{
  namespace state
  {
    class SimpleState;
    std::ostream& operator<<(std::ostream& os,const SimpleState& state);
    /**
     * 盤上の駒のみを比較する（持ち駒は見ない）.
     * なお、駒番に非依存な局面比較をしたい場合は、osl::record::CompactBoardや
     * osl::hash::HashKeyを用いる.
     */
    bool operator==(const SimpleState& st1,const SimpleState& st2);

    class SimpleState
    {
    private:
      friend std::ostream& operator<<(std::ostream& os,const SimpleState& state);
      friend bool operator==(const SimpleState& st1,const SimpleState& st2);
      typedef SimpleState state_t;
    public:
      static const bool hasPawnMask=true;
    protected:
      CArray<Piece,Position::SIZE> board
#ifdef __GNUC__
      __attribute__((aligned(16)))
#endif
	;
      /**
       * 全てのpieceが登録されている
       */
      CArray<Piece,Piece::SIZE> pieces
#ifdef __GNUC__
      __attribute__((aligned(16)))
#endif
	;
      CArray<PieceMask,2> stand_mask;
      CArray<BitXmask,2> pawnMask;
      CArray<CArray<char,PTYPE_SIZE-PTYPE_BASIC_MIN>,2> stand_count;

      /** 手番 */
      Player turn;
      PieceMask used_mask;

      friend class osl::apply_move::ApplyDoUndoSimpleMove<BLACK,SimpleState>;
      friend class osl::apply_move::ApplyDoUndoSimpleMove<WHITE,SimpleState>;
      friend class osl::apply_move::ApplyDoUndoCaptureMove<BLACK,SimpleState>;
      friend class osl::apply_move::ApplyDoUndoCaptureMove<WHITE,SimpleState>;
      friend class osl::apply_move::ApplyDoUndoDropMove<BLACK,SimpleState>;
      friend class osl::apply_move::ApplyDoUndoDropMove<WHITE,SimpleState>;
      friend class osl::misc::FastCopier;
    public:
      // 生成に関するもの
      explicit SimpleState();
      explicit SimpleState(Handicap h);
      // public継承させるには，virtual destructorを定義する．
      virtual ~SimpleState();
      /** 盤面が空の状態に初期化 */
      void init();
      /** ハンディに応じた初期状態に初期化 */
      void init(Handicap h);
      // private:
      void initPawnMask();
    public:
      const Piece getPieceOf(int num) const{
	return pieces[num];
      }
      void setPieceOf(int num,Piece p) {
	pieces[num]=p;
      }
      template<Player P>
      const Piece getKingPiece() const{
	return getPieceOf(KingTraits<P>::index);
      }
      const Piece getKingPiece(Player P) const{
	assert(isValid(P));
	if (P==BLACK)
	  return getKingPiece<BLACK>();
	else
	  return getKingPiece<WHITE>();
      }
      template<Player P>
      Position getKingPosition() const{
	return getKingPiece<P>().position();
      }
      Position getKingPosition(Player player) const{
	assert(isValid(player));
	if (player==BLACK)
	  return getKingPosition<BLACK>();
	else
	  return getKingPosition<WHITE>();
      }

      void setBoard(Position pos,Piece piece)
      {
	board[pos.index()]=piece;
      }
    protected:
      PieceMask& standMask(Player p) {
	return stand_mask[p];
      }
    public:
      const PieceMask& standMask(Player p) const {
	return stand_mask[p];
      }
      const PieceMask& usedMask() const {return used_mask;}
      bool isOffBoard(int num) const{
	return standMask(BLACK).test(num) 
	  || standMask(WHITE).test(num);
      }
      // protected:
      /** (internal) */
      void clearPawn(Player pl,Position pos){
	pawnMask[pl].clear(pos);
      }
      /** (internal) */
      void setPawn(Player pl,Position pos){
	pawnMask[pl].set(pos);
      }
    public:      
      bool isPawnMaskSet(Player player, int x) const
      {
	return pawnMask[player].isSet(x);
      }

      template<Player P>
      bool isPawnMaskSet(int x)const {return isPawnMaskSet(P,x); }

      void setPiece(Player player,Position pos,Ptype ptype);
      void setPieceAll(Player player);

      /**
       * @param pos は isOnboardを満たす Position の12近傍(8近傍+桂馬の利き)
       * ! isOnBoard(pos) の場合は PIECE_EDGE を返す
       */
      const Piece getPieceAt(Position pos) const { return board[pos.index()];}
      const Piece* getPiecePtr(Position pos) const { return &board[pos.index()];}
      const Piece getPieceOnBoard(Position pos) const
      {
	assert(pos.isOnBoard());
	return getPieceAt(pos);
      }

      bool isOnBoard(int num) const {
	return getPieceOf(num).isOnBoard();
      }
      /**
       * 持駒の枚数を数える
       */
      int countPiecesOnStand(Player pl,Ptype ptype) const {
	assert(isBasic(ptype));
	return stand_count[pl][ptype-PTYPE_BASIC_MIN];
      }
      /** 後方互換 */
      template <Ptype Type>
      int countPiecesOnStand(Player pl) const {
	return countPiecesOnStand(pl, Type);
      }
      bool hasPieceOnStand(Player player,Ptype ptype) const{
	return countPiecesOnStand(player, ptype)!=0;
      }
      template<Ptype T>
      bool hasPieceOnStand(Player P) const {
	return countPiecesOnStand(P, T);
      }
    private:
      int countPiecesOnStandBit(Player pl,Ptype ptype) const {
	return (standMask(pl).getMask(Ptype_Table.getIndex(ptype))
		& Ptype_Table.getMaskLow(ptype)).countBit();
      }
    public:
      /**
       * diff方向にあるPiece を求める. 
       * @return 盤外ならPTYPE_EDGE
       */
      Piece nextPiece(Position cur, Offset diff) const
      {
	assert(! diff.zero());
	cur += diff;
	while (getPieceAt(cur) == Piece::EMPTY())
	  cur += diff;
	return getPieceAt(cur);
      }
    
      void setTurn(Player player) {
	turn=player;
      }
      Player getTurn() const{
	return turn;
      }
      /**
       * 手番を変更する
       */
      void changeTurn() {
	osl::changeTurn(turn);
      }
      void makeMovePass() { changeTurn(); }

      // check
      bool isConsistent(bool showError=true) const;
      /** エラー表示をするかどうかをtemplateパラメータにした高速化版 */
      template <bool showError>
      bool isAlmostValidMove(Move move) const;
      /**
       * 合法手かどうかを簡単に検査する．局面に依存するチェックのみ．
       * ルール上指せない手である可能性がある場合は，isValidMove を用いる．
       *
       * 局面に依存する検査でも，玉の素抜きや王手を防いでいるか，
       * 千日手，打歩詰かどうかは検査しない．
       */
      bool isAlmostValidMove(Move move,bool showError=true) const;
      /**
       * 合法手かどうかを検査する．
       * isValidMoveByRule, isAlmostValidMove をおこなう．
       * 玉の素抜きや王手を防いでいるか，
       * 千日手，打歩詰かどうかは検査しない．
       */
      bool isValidMove(Move move,bool showError=true) const;

      /**
       * 盤面以外の部分の反則のチェック
       *
       * 本来はstatic にできるが，急いでいるので後回し
       */
      bool isValidMoveByRule(Move move,bool showError) const;

      /**
       * @param from - マスの位置
       * @param to - マスの位置
       * @param offset - fromからtoへのshort offset
       * fromとtoがクイーンで利きがある位置関係にあるという前提
       * で，間が全部空白かをチェック
       * @param pieceExistsAtTo - toに必ず駒がある (toが空白でも動く)
       */
      bool isEmptyBetween(Position from, Position to,Offset offset,bool pieceExistsAtTo=false) const
      {
	assert(from.isOnBoard());
	assert(! offset.zero());
	assert(offset==Board_Table.getShortOffset(Offset32(to,from)));
	Position pos=from+offset;
	for (; getPieceAt(pos).isEmpty(); pos+=offset) {
	  if (!pieceExistsAtTo && pos==to) 
	    return true;
	}
	return pos==to;
      
      }
      /**
       * @param from - マスの位置
       * @param to - マスの位置
       * fromとtoがクイーンで利きがある位置関係にあるという前提
       * で，間が全部空白かをチェック
       */
      bool isEmptyBetween(Position from, Position to,bool noSpaceAtTo=false) const{
	assert(from.isOnBoard());
	Offset offset=Board_Table.getShortOffset(Offset32(to,from));
	assert(! offset.zero());
	return isEmptyBetween(from,to,offset,noSpaceAtTo);
      }

      /** dump: 自分を cerr に表示する。abort 前などにデバッグに使う */
      bool dump() const;
      void doSimpleMove(Position from, Position to, int promoteMask);
      void doDropMove(Position to,Ptype ptype);
      void doCaptureMove(Position from, Position to, Piece target,int promoteMask);
      /**
       * from で表現されたPieceをnew_ownerの持駒にした局面を作る.
       */
      const SimpleState emulateCapture(Piece from, Player new_owner) const;

      /**
       * from からto に ptypeの持駒を一枚渡した局面を作る.
       */
      const SimpleState emulateHandPiece(Player from, Player to, Ptype ptype) const;
      const SimpleState rotate180() const;
      const SimpleState flipHorizontal() const;
    };  
  } // namespace state
  using state::SimpleState;

  namespace apply_move
  {
    template<Player P>
    struct ApplyDoUndoSimpleMove<P,state::SimpleState>
    {
      static void prologue(state::SimpleState& s, 
			   Position from, Position to, int promoteMask,
			   Piece& oldPiece, int& num)
      {
	oldPiece=s.getPieceAt(from);
	Piece newPiece=oldPiece.promoteWithMask(promoteMask);
	newPiece+=(to-from);
	num=oldPiece.number();
	s.setPieceOf(num,newPiece);
	s.setBoard(to,newPiece);
	s.setBoard(from,Piece::EMPTY());
      }
      static void epilogue(state::SimpleState& s, Position from, Position to,
			   Piece oldPiece, int num)
      {
	// 自分自身の効きを外す
	s.setPieceOf(num,oldPiece);
	s.setBoard(from,oldPiece);
	s.setBoard(to,Piece::EMPTY());
      }
      template <typename F>
      static void doUndoSimpleMove(state::SimpleState& s, 
				   Position from, Position to, int promoteMask,F& func);
    };
  
    template<Player P>
    template <typename F>
    void ApplyDoUndoSimpleMove<P,state::SimpleState>::
    doUndoSimpleMove(state::SimpleState& s, 
		     Position from, Position to, int promoteMask,F& func)
    {
      Piece oldPiece;
      int num;
      prologue(s, from, to, promoteMask, oldPiece, num);
      if (promoteMask!=0 && num < PtypeTraits<PAWN>::indexLimit)
      {
	s.clearPawn(P,from);
	s.changeTurn();
	func(to);
	s.changeTurn();
	s.setPawn(P,from);
      }
      else{
	s.changeTurn();
	func(to);
	s.changeTurn();
      }
      epilogue(s, from, to, oldPiece, num);
    }

    template<Player P>
    struct ApplyDoUndoDropMove<P,state::SimpleState>
    {
      typedef state::SimpleState state_t;

      /**
       * OSL_WORDSIZE == 64の時は，numIndex, numLowよりも
       * standMaskの方を保存した方が良い
       */
      static void prologue(state::SimpleState& s, Ptype ptype, Position to,
			   Piece& oldPiece, int& num, int& numIndex, int& numLow)
      {
#if OSL_WORDSIZE == 64
	numIndex=0;
	const mask_t ownMochigoma=
	  s.standMask(P).getMask(0) & Ptype_Table.getMaskLow(ptype);
	assert(ownMochigoma.any() || (s.dump(), 0));
	num=numLow=ownMochigoma.bsf();
#elif OSL_WORDSIZE == 32
	numIndex=Ptype_Table.getIndex(ptype);
	const mask_t ownMochigoma=
	  s.standMask(P).getMask(numIndex) & Ptype_Table.getMaskLow(ptype);
	assert(ownMochigoma.any() || (s.dump(), 0));
	numLow=ownMochigoma.bsf();
	num=numLow|(numIndex<<5);
#endif
	oldPiece=s.getPieceOf(num);
	Piece p=oldPiece;
	p += to-Position::STAND();

	s.setPieceOf(num,p);
	s.setBoard(to,p);
	s.standMask(P).xorMask(numIndex,PieceMask::numToMask(numLow));
	s.stand_count[P][ptype-PTYPE_BASIC_MIN]--;
      }

      static void epilogue(state::SimpleState& s, Ptype ptype, Position to,
			   Piece oldPiece, int num,
			   int numIndex, int numLow)
      {
	s.standMask(P).xorMask(numIndex,PieceMask::numToMask(numLow));
	s.setBoard(to,Piece::EMPTY());
	s.setPieceOf(num,oldPiece);
	s.stand_count[P][ptype-PTYPE_BASIC_MIN]++;
      }
      template <typename F>
      static void doUndoDropMove(state::SimpleState& s, 
				 Position to, Ptype ptype,F& func);
    };
  
  
    template<Player P>
    template <typename F>
    void ApplyDoUndoDropMove<P,state::SimpleState>::
    doUndoDropMove(state::SimpleState& s, 
		   Position to, Ptype ptype,F& func)
    {
      Piece oldPiece;
      int num, numIndex, numLow;
      prologue(s, ptype, to, oldPiece, num, numIndex, numLow);

      if (ptype==PAWN)
      {
	s.setPawn(P,to);
	s.changeTurn();
	func(to);
	s.changeTurn();
	s.clearPawn(P,to);
      }
      else
      {
	s.changeTurn();
	func(to);
	s.changeTurn();
      }
      epilogue(s, ptype, to, oldPiece, num, numIndex, numLow);
    }

    template<Player P>
    struct ApplyDoUndoCaptureMove<P,state::SimpleState>
    {
      typedef state::SimpleState state_t;
      template <typename F>
      static void doUndoCaptureMove(state_t& s,
				    Position from,Position to, Piece p1, int promoteMask,F& func);

      static
      void prologue(state::SimpleState& s, 
		    Position from, Position to, Piece target, int promoteMask,
		    Ptype& capturePtype, Piece& oldPiece, int& num0, 
		    int& num1, int& num1Index, mask_t& num1Mask)
      {
	capturePtype=target.ptype();
	num1=target.number();
	num1Index=PieceMask::numToIndex(num1);
	num1Mask=PieceMask::numToMask(num1);

	s.standMask(P).xorMask(num1Index,num1Mask);
	oldPiece=s.getPieceAt(from);
	Piece newPiece=oldPiece.promoteWithMask(promoteMask);
	newPiece+=(to-from);
	num0=oldPiece.number();
	s.setPieceOf(num0,newPiece);
	s.setPieceOf(num1,target.captured());
	s.setBoard(to,newPiece);
	s.setBoard(from,Piece::EMPTY());
	s.stand_count[P][unpromote(capturePtype)-PTYPE_BASIC_MIN]++;
      }

      static 
      void epilogue(state::SimpleState& s, Position from, Position to, Piece target, 
		    Ptype capturePtype, Piece oldPiece, int num0, 
		    int num1, int num1Index, mask_t num1Mask)
      {
	s.standMask(P).xorMask(num1Index,num1Mask);
	s.setPieceOf(num0,oldPiece);
	s.setPieceOf(num1,target);
	s.setBoard(from,oldPiece);
	s.setBoard(to,target);
	s.stand_count[P][unpromote(capturePtype)-PTYPE_BASIC_MIN]--;
      }
    };
  
  
    template<Player P>
    template <typename F>
    void ApplyDoUndoCaptureMove<P,state::SimpleState>::
    doUndoCaptureMove(state_t& s,
		      Position from,Position to, Piece target, int promoteMask,F& func)
    {
      Piece oldPiece;
      int num0, num1;
      int num1Index;
      mask_t num1Mask;
      Ptype capturePtype;
      prologue(s, from, to, target, promoteMask,
	       capturePtype, oldPiece, num0, num1, num1Index, num1Mask);
      s.changeTurn();
      if (capturePtype==PAWN){
	s.clearPawn(PlayerTraits<P>::opponent,to);
	if (promoteMask!=0 && num0<PtypeTraits<PAWN>::indexLimit){
	  s.clearPawn(P,from);
	  func(to);
	  s.setPawn(P,from);
	}
	else{
	  func(to);
	}
	s.setPawn(PlayerTraits<P>::opponent,to);
      }
      else if (promoteMask!=0 && num0<PtypeTraits<PAWN>::indexLimit){
	s.clearPawn(P,from);
	func(to);
	s.setPawn(P,from);
      }
      else{
	func(to);
      }
      s.changeTurn();

      epilogue(s, from, to, target, capturePtype, oldPiece, num0, num1, num1Index, num1Mask);
    }
  } // namespace apply_move
} // namespace osl

#endif /* _SIMPLE_STATE_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
