/******************************************************************************************************************************************
 ctreeview.h

 CClass						CClass
 |--CSerialized					|--CSerialized
    |--CMetaModule				   |--CMetaModule
       |--CObject				      |--CObjectListener
          |--CComponent				         |
             |--CControl				 |
                |--CWidget				 |--CWidgetListener
                |  |--CContainer			    |
                |     |--CTreeView			    |--CTreeViewListener
                |--CTreeViewItem

 Like the gtk view representations, the gtkol tree view definition is based on a model. This model is specified as a metaclasses list of
 item field values instances (see citemfieldvalue.h). The gtkol tree view instance works on tree view items that own the field values of
 the specific handled tree view model. The tree view model must be defined before any tree view item is added to the tree view widget so
 that the tree view widget knows how to handle the field values of the owned items. The tree view item field values must map the tree view
 widget defined model or the owner affectation of the tree view item to another tree view item or to the tree view widget would fail 
 because of not corresponding to the expected model. The default model is the simpliest one i.e. a field value for each tree view item
 that is a CItemFieldValueString one and so displays just a string for each item.
 To pack multiple field values into one view column, just use the CItemFieldValuePack metaclass specification while defining the model.

 CTreeView ==> { CTreeViewItem }
		      |==> { CItemFieldValue  ( CItemFieldValueString, CItemFieldValueBoolean, CItemFieldValuePixbuf,
						CItemFieldValueCombo,  CItemFieldValueProgress ) }

******************************************************************************************************************************************/

#ifndef __CTREEVIEW_H__
#define __CTREEVIEW_H__

#include "ccontainer.h"
#include "citemfieldvalue.h"

// what's defined in this section
class CTreeView;
class CTreeViewListener;
class CTreeViewItem;

//-----------------------------------------------------------------------------------------------------------------------------------------
// define a tree view item buffer list
//-----------------------------------------------------------------------------------------------------------------------------------------
typedef NServices::TBuffer <CTreeViewItem *> CTreeViewItems;

//-----------------------------------------------------------------------------------------------------------------------------------------
// ctreeviewitem xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <ctreeviewitem>
//   <fields>
//     ...
//   </fields>
// </ctreeviewitem>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_TREEVIEWITEM_ELEMENT			("ctreeviewitem");
static CString XML_TREEVIEWITEM_FIELDS_ELEMENT		("fields");

//-----------------------------------------------------------------------------------------------------------------------------------------
// ctreeview xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <ctreeview selection="single|multiple|browse|none" headers-visible="true|false" rules-hint="true|false" selected="[UInt32[,UInt32]]" 
//  expanded="[UInt32[,UInt32]]">
//   <ctreeview-model>
//       <ctreeview-field name="string_type" tag="tag_type" [title="string"]>
//     [ <ctreeview-field name="string_type" tag="tag_type" [title="string"]> ]
//   </ctreeview-model>
//   ...
// </ctreeview>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_TREEVIEW_ELEMENT			("ctreeview");
static CString XML_TREEVIEW_ATTR_SELECTION_MODE		("selection");
static CString XML_TREEVIEW_ATTR_HEADERS_VISIBLE	("headers-visible");
static CString XML_TREEVIEW_ATTR_RULES_HINT		("rules-hint");
static CString XML_TREEVIEW_ATTR_SELECTION		("selected");
static CString XML_TREEVIEW_ATTR_EXPAND			("expanded");
static CString XML_TREEVIEW_MODEL_ELEMENT		("ctreeview-model");
static CString XML_TREEVIEW_FIELDCLASS_ELEMENT		("ctreeview-field");
static CString XML_TREEVIEW_FIELDCLASS_ATTR_NAME	("name");
static CString XML_TREEVIEW_FIELDCLASS_ATTR_TAG 	("tag");
static CString XML_TREEVIEW_FIELDCLASS_ATTR_TITLE	("title");

//-----------------------------------------------------------------------------------------------------------------------------------------
// CTreeViewItem class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CTreeViewItem : public CControl
{
	// instanciation section
	public :
	
		CTreeViewItem			(CTreeView *inOwner=NULL, const CItemFieldValues &inFieldValues=CItemFieldValues(),
						 const CObjectListener *inListener=NULL);
		CTreeViewItem			(CTreeViewItem *inOwner, const CItemFieldValues &inFieldValues=CItemFieldValues(),
						 const CObjectListener *inListener=NULL);
		virtual ~CTreeViewItem		();

	// CComponent redefinition
	public :

		// expected owner type : CTreeView or CTreeViewItem, expected children type : CTreeViewItem
		virtual CMetaClasses		OwnerMustBe			() const;
		virtual CMetaClasses		ChildMustBe			() const;

		// owner affectation, must be derived from CTreeView or CTreeViewItem
		virtual bool			SetOwner			(CComponent *inOwner, const SInt16 inIndex=-1);

	// CControl redefinition
	public :

		// get the tree view item bounds
		virtual TBounds			GetBounds			() const;

		// get the tree view item pixbuf representation, used for friendly d&d representations
		virtual CPixbuf *		GetControlPixbuf		() const;

	// gtkol tree view item / gtk tree view item correspondance
	public :
	
		// get the gtk tree iter associated to the gtkol tree item instance
		GtkTreeIter *			GetGtkTreeIter			() const;

		// get the gtkol tree view item of specified gtk tree iter if any
		static CTreeViewItem *		GetTreeViewItem			(const GtkTreeIter *inGtkTreeIter);

	// tree view item field values access
	public :

		// item field values access
		virtual Bool			SetItemFieldValues		(const CItemFieldValues &inItemFieldValues);
		CItemFieldValues		GetItemFieldValues		() const;

		// 0 indexed field value direct access, no check on range so be carefull
		CItemFieldValue &		operator []			(const size_t) const;

	// tree view item specific functions
	public :

		// select / unselect the tree view item (row)
		virtual void			Select				(const bool inSelect=true);
		virtual bool			IsSelected			() const;

		// select the item's row and set the cursor on the specified column index, starting editing it if editable
		virtual void			SetCursor			(const size_t inCol, const bool inEdit=true);

		// expand the tree view item if not in an expanded state yet and if able to do so, collapse a tree view item
		virtual void			Expand				(const bool inAll=false);
		virtual void			Collapse			();
		virtual Bool			IsExpanded			() const;

	// CSerialized redefinition
	public :

		// tree view item xml serialization
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;
	// protected attributes
	protected :

		// gtk tree iter handled attribute
		GtkTreeIter			m_GtkTreeIter;

		// tree view item field values
		CItemFieldValues		m_ItemFieldValues;

		// some friends of us...
		friend class			CTreeView;

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('trwi', CTreeViewItem, CControl);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CTreeViewListener class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CTreeViewListener : public CWidgetListener
{
	// instanciation section
	public :

		CTreeViewListener		();
		virtual ~CTreeViewListener	() =0;

	// tree view listening interface
	public :

		// called when the specified tree view item of the inSender tree view is about to be selected / unselected, the inCurrent
		// parameter specifies the current state of the tree view item to be changed
		virtual void			OnItemQuerySelect		(CObject *inSender, CObject *inItem, const Bool inCurrent,
										 Bool &ioDoToggle)					{ }

		// called when the selection of the specified inSender tree view has changed
		virtual void			OnItemSelectionChanged		(CObject *inSender)					{ }

		// called when the specified tree view item of the inSender tree view has been activated i.e. double clicked
		virtual void			OnItemActivated			(CObject *inSender, CObject *inItem)			{ }
		
		// called when the specified tree view item of the inSender tree view is about to be expanded
		virtual void			OnItemQueryExpand		(CObject *inSender, CObject *inItem, Bool &ioExpand)	{ }

		// called when the specified tree view item of the inSender tree view has been expanded
		virtual void			OnItemExpanded			(CObject *inSender, CObject *inItem)			{ }

		// called when the specified tree view item of the inSender tree view is about to be collapsed
		virtual void			OnItemQueryCollapse		(CObject *inSender, CObject *inItem, Bool &ioCollapse)	{ }
		
		// called when the specified tree view item of the inSender tree view has been collapsed
		virtual void			OnItemCollapsed			(CObject *inSender, CObject *inItem)			{ }

		// called when the specified indexed boolean field value of the inSender's tree view item is about to be toggled
		virtual void			OnItemToggled			(CObject *inSender, CObject *inItem, const size_t inIndex,
										 Bool &ioDoChange)					{ }

		// called when the specified indexed string field value of the inSender's tree view item is about to be modified 
		// after beeing edited
		virtual void			OnItemEdited			(CObject *inSender, CObject *inItem, const size_t inIndex,
										 CString &ioNewString, Bool &ioDoChange)		{ }

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('_trw', CTreeViewListener, CWidgetListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CTreeView class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CTreeView : public CContainer
{
	// instanciation section
	public :
		
		CTreeView			(CContainer *inOwner=NULL, const CMetaClasses &inModel=_IFVString_, 
						 const CTreeViewListener *inListener=NULL) THROWABLE;
		virtual ~CTreeView		();

	// protected specific gtk widget requests handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

	// CObject redefinition
	protected :

		// listener affectation, must be derived from CTreeViewListener
		virtual const CMetaClass *	ListenerMustBe			() const;

	// CComponent redefinition
	public :

		// expected owner types : CContainer, expected children types : CTreeViewItem
		virtual CMetaClasses		OwnerMustBe			() const;
		virtual CMetaClasses		ChildMustBe			() const;

	// CControl / CWidget redefinition
	public :

		// get the tree view item owned at the specified relative point
		virtual CControl *		GetControlAtPoint		(const TPoint &inRelativePoint) const;

	// specific tree view functions
	public :

		// tree view model affectation / access, the specified metaclasses should be the one derived from CItemFieldValue i.e.
		// CItemFieldValueString, CItemFieldValueBoolean and CItemFieldValuePixbuf
		Bool				SetModel			(const CMetaClasses &inModel);
		CMetaClasses			GetModel			() const;

		// set columns titles
		virtual void			SetColumnsTitle			(const CStrings &);
		CStrings			GetColumnsTitle			() const;

		// set rules hint
		virtual void			SetRulesHint			(const bool inRulesHint);
		Bool				GetRulesHint			() const;

		// header visible
		virtual void			SetHeadersVisible		(const bool inVisible);
		Bool				GetHeadersVisible		() const;

		// tree view selection mode access
		virtual void			SetSelectionMode		(const GtkSelectionMode inSelectionMode);
		GtkSelectionMode		GetSelectionMode		() const;

		// current tree view selection, returns the list of the currently selected items if any
		CTreeViewItems			GetSelection			() const;

	// CSerialized redefinition
	public :

		// tree view xml serialization handling
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;
	// protected attributes section
	protected :

		// tree view model representation (expected tree view items typed field values)
		CMetaClasses			m_Model;

		// columns title
		CStrings			m_ColumnsTitle;

	// protected static listening section
	protected :

		static void			OnItemEdited			(GtkCellRendererText *, const gchar *, const gchar *,
										 gpointer);
		static void			OnItemToggled			(GtkCellRendererToggle *, gchar *, gpointer);
		static void			OnItemActivated			(GtkTreeView *, GtkTreePath *, GtkTreeViewColumn *,
										 gpointer);
		static gboolean			OnItemQueryExpand		(GtkTreeView *, GtkTreeIter *, GtkTreePath *, gpointer);
		static gboolean			OnItemQueryCollapse		(GtkTreeView *, GtkTreeIter *, GtkTreePath *, gpointer);
		static void			OnItemExpanded			(GtkTreeView *, GtkTreeIter *, GtkTreePath *, gpointer);
		static void			OnItemCollapsed			(GtkTreeView *, GtkTreeIter *, GtkTreePath *, gpointer);
		static gboolean			OnItemQuerySelect		(GtkTreeSelection *, GtkTreeModel *, GtkTreePath *, 
										 gboolean, gpointer);
		static void			OnItemSelectionChanged		(GtkTreeSelection *, gpointer);

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('trvw', CTreeView, CContainer);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CTreeViewItemFieldValueAPIListener definition, internaly used, should not be used by the front end development
//-----------------------------------------------------------------------------------------------------------------------------------------
class CTreeViewItemFieldValueAPIListener : public CItemFieldValueAPIListener
{
	// public handling
	public :

		// called when the developper explicitely requests some modification on the item field value api
		virtual void			OnStateChange			(CObject *);

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('_tpi', CTreeViewItemFieldValueAPIListener, CItemFieldValueAPIListener);

#endif
