/*
 * @(#)VirtualWindowUTest.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.uicapture.v1;

import java.awt.Robot;

import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import net.sourceforge.groboutils.uicapture.v1.event.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;



/**
 * Tests the VirtualWindow class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2003/02/10 22:52:34 $
 * @since   Jan 6, 2002
 */
public class VirtualWindowUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = VirtualWindowUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public VirtualWindowUTest( String name )
    {
        super( name );
    }

    
    private VirtualWindow vw;
    
    
    public class MyCaptureListener implements ICaptureListener
    {
        ArrayList mwm = new ArrayList();
        ArrayList mm = new ArrayList();
        ArrayList mp = new ArrayList();
        ArrayList mr = new ArrayList();
        ArrayList kp = new ArrayList();
        ArrayList kr = new ArrayList();
        
        public void mouseWheelMoved( MouseWheelCaptureEvent ce )
        { mwm.add( ce ); }
        
        public void mouseMoved( MouseMovedCaptureEvent ce )
        { mm.add( ce ); }
        
        public void mousePressed( MousePressedCaptureEvent ce )
        { mp.add( ce ); }
        
        public void mouseReleased( MouseReleasedCaptureEvent ce )
        { mr.add( ce ); }
        
        public void keyPressed( KeyPressedCaptureEvent ce )
        { kp.add( ce ); }
        
        public void keyReleased( KeyReleasedCaptureEvent ce )
        { kr.add( ce ); }
    }
    
    

    //-------------------------------------------------------------------------
    // Tests
    
    public void testInstantiation1()
    {
        this.vw = createVirtualWindow();
        assertTrue(
            "Default glass pane state is true.",
            this.vw.isGlassEnabled() );
    }
    
    
    public void testInstantiation2()
    {
        this.vw = createVirtualWindow( null, true );
        assertTrue(
            "Manually set glass pane state to true.",
            this.vw.isGlassEnabled() );
    }
    
    
    public void testInstantiation3()
    {
        this.vw = createVirtualWindow( null, false );
        assertTrue(
            "Manually set glass pane state to false.",
            !this.vw.isGlassEnabled() );
    }
    
    
    public void testMouseWheelMoved1()
    {
        this.vw = createVirtualWindow();
        MyCaptureListener mcl1 = createCaptureListener();
        MyCaptureListener mcl2 = createCaptureListener();
        this.vw.addCaptureListener( mcl1 );
        this.vw.addCaptureListener( mcl2 );
        
        MouseWheelEvent mwe = new MouseWheelEvent( getUIComponent(),
            1, System.currentTimeMillis(), 0, 0, 0, 2, false,
            MouseWheelEvent.WHEEL_UNIT_SCROLL, 1, 1 );
        this.vw.mouseWheelMoved( mwe );
        
        assertEquals( "Did not fire a wheel event to listener 1.",
            1,
            mcl1.mwm.size() );
        assertEquals( "Did not fire a wheel event to listener 2.",
            1,
            mcl2.mwm.size() );
    }
    
    
    public void testMouseDragged1()
    {
        this.vw = createVirtualWindow();
        MyCaptureListener mcl1 = createCaptureListener();
        MyCaptureListener mcl2 = createCaptureListener();
        this.vw.addCaptureListener( mcl1 );
        this.vw.addCaptureListener( mcl2 );
        
        MouseEvent me = new MouseEvent( getUIComponent(),
            1, System.currentTimeMillis(), 0, 0, 0,
            0, // click count
            false, 0 );
        this.vw.mouseDragged( me );
    }
    
    
    public void testMouseMoved1()
    {
        this.vw = createVirtualWindow();
        MyCaptureListener mcl1 = createCaptureListener();
        MyCaptureListener mcl2 = createCaptureListener();
        this.vw.addCaptureListener( mcl1 );
        this.vw.addCaptureListener( mcl2 );
        
        MouseEvent me = new MouseEvent( getUIComponent(),
            1, System.currentTimeMillis(), 0, 0, 0,
            0, // click count
            false, 0 );
        this.vw.mouseMoved( me );
        
        assertEquals( "Did not fire a move event to listener 1.",
            1,
            mcl1.mm.size() );
        assertEquals( "Did not fire a move event to listener 2.",
            1,
            mcl2.mm.size() );
    }
    
    
    public void testMousePressed1()
    {
        this.vw = createVirtualWindow();
        MyCaptureListener mcl1 = createCaptureListener();
        MyCaptureListener mcl2 = createCaptureListener();
        this.vw.addCaptureListener( mcl1 );
        this.vw.addCaptureListener( mcl2 );
        
        MouseEvent me = new MouseEvent( getUIComponent(),
            1, System.currentTimeMillis(), 0, 0, 0,
            1, // click count
            false, 0 );
        this.vw.mousePressed( me );
        
        assertEquals( "Did not fire a mouse press event to listener 1.",
            1,
            mcl1.mp.size() );
        assertEquals( "Did not fire a mouse press event to listener 2.",
            1,
            mcl2.mp.size() );
    }
    
    
    public void testMouseReleased1()
    {
        this.vw = createVirtualWindow();
        MyCaptureListener mcl1 = createCaptureListener();
        MyCaptureListener mcl2 = createCaptureListener();
        this.vw.addCaptureListener( mcl1 );
        this.vw.addCaptureListener( mcl2 );
        
        MouseEvent me = new MouseEvent( getUIComponent(),
            1, System.currentTimeMillis(), 0, 0, 0,
            0, // click count
            false, 0 );
        this.vw.mouseReleased( me );
        
        assertEquals( "Did not fire a mouse release event to listener 1.",
            1,
            mcl1.mr.size() );
        assertEquals( "Did not fire a mouse release event to listener 2.",
            1,
            mcl2.mr.size() );
    }
    
    
    public void testKeyPressed1()
    {
        this.vw = createVirtualWindow();
        MyCaptureListener mcl1 = createCaptureListener();
        MyCaptureListener mcl2 = createCaptureListener();
        this.vw.addCaptureListener( mcl1 );
        this.vw.addCaptureListener( mcl2 );
        
        KeyEvent ke = new KeyEvent( getUIComponent(),
            1, System.currentTimeMillis(), 0, KeyEvent.VK_A, 'a', 0 );
        this.vw.keyPressed( ke );
        
        assertEquals( "Did not fire a key press event to listener 1.",
            1,
            mcl1.kp.size() );
        assertEquals( "Did not fire a key press event to listener 2.",
            1,
            mcl2.kp.size() );
    }
    
    
    public void testKeyReleased1()
    {
        this.vw = createVirtualWindow();
        MyCaptureListener mcl1 = createCaptureListener();
        MyCaptureListener mcl2 = createCaptureListener();
        this.vw.addCaptureListener( mcl1 );
        this.vw.addCaptureListener( mcl2 );
        
        KeyEvent ke = new KeyEvent( getUIComponent(),
            1, System.currentTimeMillis(), 0, KeyEvent.VK_A, 'a', 0 );
        this.vw.keyReleased( ke );
        
        assertEquals( "Did not fire a key release event to listener 1.",
            1,
            mcl1.kr.size() );
        assertEquals( "Did not fire a key release event to listener 2.",
            1,
            mcl2.kr.size() );
    }
    
    
    
    
    //-------------------------------------------------------------------------
    
    
    protected VirtualWindow createVirtualWindow()
    {
        try
        {
            return new VirtualWindow();
        }
        catch (java.awt.AWTException ae)
        {
            fail( "JDK implementation does not support low-level Robot "+
                "actions: "+ae );
            // unreachable
            return null;
        }
    }
    
    
    protected VirtualWindow createVirtualWindow( String title, boolean enable )
    {
        try
        {
            return new VirtualWindow( title, enable );
        }
        catch (java.awt.AWTException ae)
        {
            fail( "JDK implementation does not support low-level Robot "+
                "actions: "+ae );
            // unreachable
            return null;
        }
    }
    
    
    protected Component getUIComponent()
    {
        return this.vw.getWindow();
    }
    
    
    protected MyCaptureListener createCaptureListener()
    {
        return new MyCaptureListener();
    }
    
    
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected synchronized void tearDown() throws Exception
    {
        // tear ourself down
        if (this.vw != null)
        {
            this.vw.dispose();
            this.vw = null;
        }
        
        super.tearDown();
    }
}

