/* $Id: limo.c,v 1.12 2000/01/24 00:37:53 fraserm Exp $
   $Log: limo.c,v $
   Revision 1.12  2000/01/24 00:37:53  fraserm
   added 'e' chunk

   Revision 1.11  1999/12/22 02:07:02  fraserm
   added a missing break; in a default: clause

   Revision 1.10  1999/12/22 01:53:38  fraserm
   added -S

   Revision 1.9  1999/12/21 02:42:22  fraserm
   changes as per 0.3 in CHANGELOG

   Revision 1.8  1999/12/14 02:13:46  fraserm
   added new env variables

   Revision 1.7  1999/07/01 01:40:13  fraserm
   word 0, LIMO_STYLE

   Revision 1.6  1999/06/15 23:46:34  fraserm
   added ioctl and COLUMNS output_width checks

   Revision 1.5  1999/05/30 11:35:15  fraserm
   added support for multiple sort chunks

   Revision 1.4  1999/05/30 01:07:16  fraserm
   -v verbose now shows chunk meanings
   -h help lists chunk meanings

   Revision 1.3  1999/05/26 23:51:43  fraserm
   added custom command stuff

   Revision 1.2  1999/05/16 19:49:15  fraserm
   *** empty log message ***

   Revision 1.1  1999/04/28 00:34:44  fraserm
   Initial revision

*/

#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include "limo.h"

/* main module for custom ls - limo */

extern char *optarg;
extern int optind;

/* regrettably this must be global, so that the compare routine passed to
   qsort() can see it */
extern char *sort_chunks;
/* these don't have to be global, but I'm going to make them global anyway */
struct custcmd cmds[MAXCUSTCMDS]; /* custom commands */
int numcustcmds = 0; /* number of custom commands defined */
int euid, egid; /* current effective userid and groupid */

/* helpful hints and stuff */
void showhelp(char querychunk)
{
  int i, j;
  char chunks[] = {
    NAME, NAMEID, DESCSHORT, DESCLONG, SIZEBLOCKS, BYTESSHORT, BYTESLONG,
    MODESHORT, MODELONG, EFFECTIVE_SHORT,
    OWNERNUMBER, OWNERNAME, GROUPNUMBER, GROUPNAME,
    INODE, LINKS, ACCESSSHORT, ACCESSLONG, MODIFYSHORT, MODIFYLONG,
    CHANGESHORT, CHANGELONG, RELATIVE, RAWTIME, LINK, '\0'
  };
  char *chunkhelp[] = {
    D_NAME, D_NAMEID, D_DESCSHORT, D_DESCLONG, D_SIZEBLOCKS, D_BYTESSHORT,
    D_BYTESLONG, D_MODESHORT, D_MODELONG, D_EFFECTIVE_SHORT,
    D_OWNERNUMBER, D_OWNERNAME,
    D_GROUPNUMBER, D_GROUPNAME, D_INODE, D_LINKS, D_ACCESSSHORT, D_ACCESSLONG,
    D_MODIFYSHORT, D_MODIFYLONG, D_CHANGESHORT, D_CHANGELONG, D_RELATIVE,
    D_RAWTIME, D_LINK
  };

  if (querychunk == HELP) {
    printf("\nChunks are:\n");
  }
  for (i = 0, j = 0; chunks[i] != '\0'; ++i) {
    if (querychunk == HELP) {
      printf("%c - %-24s%s", chunks[i], chunkhelp[i],
	     (j++ % 2) ? "\n" : "");
    }
    else if (querychunk == chunks[i]) {
      printf(":%c=%s", chunks[i], chunkhelp[i]);
    }
  }
  if (j % 2) {
    printf("\n");
  }
}

/* show usage */
void usage(int argc, char *argv[])
{
  printf("This is Limo (LIst files with Many Options), version %s.\n\nThis is free software distributed under the terms of the GNU General Public\nLicence. You are welcome to distribute copies of it under certain conditions\noutlined in the file LICENCE, supplied with the distribution.\nThere is no warranty for this software, not even the implied warranty of\nmerchantability or fitness for a particular purpose.\n\n", VERSION);
  fprintf(stderr, "%s [-o<sortopt>] [-p<style>] [-f<output-format>] [opts] [file ...]\n\tsortopt = + or - followed by sort code, default '%s'\n\tstyle = d (down) / a (across) / , (comma separated), default '%c'\n\toutput-format = list of chunks to show for each file, default '%s'\n",
	  argv[0], SORT_CHUNKS_DEFAULT,
	  STYLE_DEFAULT, DEFAULTCHUNKS);
  showhelp(HELP);
  fprintf(stderr, "For full information, type \"man limo\"\n");
}

int set_sort(int argc, char *argv[], char *s)
{
  char *i;

  for (i = s; *i != '\0'; ++i) {
    if (*i != SORT_DIR_ASC && *i != SORT_DIR_DESC
	&& index(SORT_ALLOWED_CHUNKS, *i) == NULL) {
      fprintf(stderr, "%s: '%c' in '%s' is not a valid sort chunk character\n",
	      argv[0], *i, s);
      return ERROR_ARGS;
    }
  }
  sort_chunks = s;
  return 0;
}

int main(int argc, char *argv[])
{
  int o;
  char print_style;
  char *s_print_style; /* text version from environment */
  char defsepchar = DEFSEPCHAR; /* default separator character */
  int output_width = WIDTH_DEFAULT;
  int tmp_output_width;
  char *txtwidth; /* just used to get the width from the environment */
  int options = OPTIONS_DEFAULT;
  char *chunks, *format, *fi, *fj; /* which chunks are requested for each file,
				      and the computed output format, plus a
				      loop variables */
  char prevchunk; /* using when building a format from a chunk list */
  struct flist deffilelst; /* list of files given on the command
			      line that must be examined */
  struct flist defdirlst; /* ditto for directories, listed after the
			     files - unless -d in effect, in which
			     case, empty */
  unsigned long i;
  struct fnode *tmp;
  time_t now;
  struct stat cfile;
  char *j;

  /* get defaults from environment variables, if any */
  if ((chunks = getenv(LIMO_FORMAT)) == NULL || *chunks == '\0') {
    chunks = DEFAULTCHUNKS;
  }
  if ((sort_chunks = getenv(LIMO_ORDER)) == NULL || *sort_chunks == '\0') {
    sort_chunks = SORT_CHUNKS_DEFAULT;
  }
  if ((s_print_style = getenv(LIMO_STYLE)) == NULL
      || strlen(s_print_style) == 0) {
    print_style = STYLE_DEFAULT;
  }
  else {
    print_style = s_print_style[0];
  }
  /* override the screen width with the terminal parameters, if possible */
#ifdef TIOCGWINSZ
  {
    struct winsize ws;

    if (ioctl (1, TIOCGWINSZ, &ws) != -1 && ws.ws_col != 0)
      output_width = ws.ws_col;
  }
#endif
  /* override the screen width with the environment, if it exists */
  if ((txtwidth = getenv(LIMO_WIDTH)) != NULL && *txtwidth != '\0') {
    if (sscanf(txtwidth, "%d", &tmp_output_width) != 1
	|| tmp_output_width < 1) {
      fprintf(stderr, "%s: warning: env variable %s contains invalid output width '%s'\n", argv[0], LIMO_WIDTH, txtwidth);
    }
    else {
      output_width = tmp_output_width;
    }
  }
  /* get the current time; this time is overridden by the -c (compare file)
     option */
  if ((now = time(NULL)) == (time_t)-1) {
    fprintf(stderr, "%s: time: %s\n", argv[0], strerror(errno));
    exit(ERROR_PROC);
  }
  while ((o = getopt(argc, argv, "c:o:p:w:W:f:e:S:aAdhlqRrstv/")) != EOF) {
    /* parse command line options */
    switch (o) {
    case 'c': /* compare file - this file is used as the value for "now"
		 for the file age options */
      /* lstat() the file in question */
      if (lstat(optarg, &cfile) != 0) { /* stat failed, just bomb */
	fprintf(stderr, "%s: can't compare with file %s: %s\n",
		argv[0], optarg, strerror(errno));
	exit(ERROR_ARGS);
      }
      options |= USE_COMPARE_FILE;
      break;
    case 'o': /* set the sort chunks */
      if (set_sort(argc, argv, optarg) == ERROR_ARGS) {
	exit(ERROR_ARGS);
      }
      break;
    case 'p':
      switch (*optarg) {
      case COMMA_JUST:
      case COLUMN_DOWN:
      case COLUMN_ACROSS:
	print_style = *optarg;
	break;
      default:
	fprintf(stderr, "%s: '%c' is not a valid print style character\n",
		argv[0], *optarg);
	exit(ERROR_ARGS);
	break;
      }
      break;
    case 'w':
    case 'W':
      if (sscanf(optarg, "%d", &output_width) != 1
	  || output_width < 1) {
	fprintf(stderr, "%s: invalid output width '%s'\n", argv[0], optarg);
	exit(ERROR_ARGS);
      }
      if (o == 'W') {
	output_width = -output_width;
      }
      break;
    case 'S': /* default chunk separator character */
      defsepchar = *optarg;
      break;
    case 'f': /* output format - set chunks */
      chunks = optarg;
      if (*chunks == '\0') {
	  fprintf(stderr, "%s: empty list of chunks\n", argv[0]);
	  exit(ERROR_ARGS);
      }
      break;
    case 'e': /* define a custom command */
      if (numcustcmds >= MAXCUSTCMDS) {
	fprintf(stderr, "%s: too many custom commands, '%s' ignored\n",
		argv[0], optarg);
      }
      else {
	int cmdindex;
	char *cmd;

	if (sscanf(optarg, "%d:%d:%n",
		   &cmds[numcustcmds].linenum,
		   &cmds[numcustcmds].wordnum,
		   &cmdindex)
	    < 2
	    || cmds[numcustcmds].linenum < 1
	    || cmds[numcustcmds].wordnum < -1) {
	  fprintf(stderr, "%s: custom command must be: line-num:word-num:command\n\t('%s')\n",
		  argv[0], optarg);
	  exit(ERROR_ARGS);
	}
	else { /* custom command is in proper format - now parse it */
	  cmd = optarg + cmdindex;
	  if (strstr(cmd, "'%s'") != NULL) { /* user has kindly quoted it
						themselves, no reformatting
						necessary */
	    if ((cmds[numcustcmds].cmd = (char *) malloc(strlen(cmd)+1))
		== NULL) {
	      fprintf(stderr, "%s: malloc: %s\n", argv[0], strerror(errno));
	      exit(ERROR_ARGS);
	    }
	    strcpy(cmds[numcustcmds].cmd, cmd);
	  }
	  else if (strstr(cmd, "%s") != NULL) { /* user just put a %s, we must
						  quote it to be safe */
	    if ((cmds[numcustcmds].cmd = (char *) malloc(strlen(cmd)+3))
		== NULL) { /* +3 = 1 for the terminating \0 and 2 for the
			      extra 'quotes' */
	      fprintf(stderr, "%s: malloc: %s\n", argv[0], strerror(errno));
	      exit(ERROR_ARGS);
	    }
	    sprintf(cmds[numcustcmds].cmd, cmd, "'%s'");
	  }
	  else { /* user didn't put a %s at all, we better add one */
	    if ((cmds[numcustcmds].cmd = (char *) malloc(strlen(cmd)+6))
		== NULL) { /* +6 = 1 for the terminating \0 and 5 for the
			      extra " '%s'" */
	      fprintf(stderr, "%s: malloc: %s\n", argv[0], strerror(errno));
	      exit(ERROR_ARGS);
	    }
	    strcpy(cmds[numcustcmds].cmd, cmd);
	    strcat(cmds[numcustcmds].cmd, " '%s'");
	  }
	  ++numcustcmds;
	}
      }
      break;
    case 'a':
      options |= SHOW_ALL;
      break;
    case 'A':
      options |= SHOW_ALL_LIMITED;
      break;
    case 'd':
      options |= DIRS_AS_FILES;
      break;
    case 'h': /* help - print usage information */
      usage(argc, argv);
      exit(ERROR_SUCCESS);
      break;
    case 'l': /* simulate ls -l */
      /* check first if we have a modified -l */
      if ((chunks = getenv(LIMO_FORMAT_L)) == NULL || *chunks == '\0') {
	chunks = LONGCHUNKS;
      }
      output_width = -1; /* force single column */
      break;
    case 'q':
      options |= QUOTE_UNPRINTABLE;
      break;
    case 'R': /* recurse into directories */
      options |= (RECURSE | SHOW_DIR_HEAD);
      break;
    case 'r': /* don't recurse out of the current filesystem */
      options |=  (RECURSE | SHOW_DIR_HEAD | SAME_FILESYSTEM);
      break;
    case 's': /* simulate ls -s */
      /* check first if we have a modified -l */
      if ((chunks = getenv(LIMO_FORMAT_S)) == NULL || *chunks == '\0') {
	chunks = SHORTCHUNKS;
      }
      break;
    case 't': /* show totals */
      options |= SHOW_TOTALS;
      break;
    case 'v': /* verbose */
      options |= VERBOSE;
      break;
    case '/': /* full pathnames */
      options |= FULL_PATHNAMES;
      break;
    case '?':
      exit(ERROR_ARGS);
      break;
    default:
      fprintf(stderr, "%s: internal error; fell out of option switch with option '%c'\n",
	      argv[0], o);
      exit(ERROR_ARGS);
      break;
    }
  }
  /* right, got all the options, down to the nitty-gritty */
  /* display directory header lines? */
  if (argc - optind > 1) { /* more than one argument */
    options |= SHOW_DIR_HEAD;
  }
  /* if chunks has a % in it, copy it to format and compute the raw
     chunks; else if it has none, compute the format */
  if (index(chunks, OPT_PREFIX) == NULL) { /* no %, build a format */
    if ((format = malloc(strlen(chunks) * 3)) == NULL) {
      fprintf(stderr, "%s: unable to allocate memory to build output format\n",
	      argv[0]);
      exit(ERROR_PROC);
    }
    for (fi = chunks, fj = format; *fi != '\0'; ++fi) {
      if (fi != chunks && index(OPT_LETTERS, prevchunk) != NULL) {
	*fj++ = defsepchar; /* add spacer if not the first chunk
			       and previous was a letter */
      }
      *fj++ = OPT_PREFIX; /* add the option character */
      *fj++ = *fi; /* and the chunk */
      prevchunk = *fi;
    }
    *fj = '\0'; /* terminate the string */
  }
  else { /* we have a %, build chunks from the format */
    format = chunks;
    /* we allocate a string the same size as the format string (well, one
       less); since the chunk list will be shorter than the format, this is
       inefficient of memory, but quick and easy */
    if ((chunks = malloc(strlen(format))) == NULL) {
      fprintf(stderr, "%s: unable to allocate memory to build chunk list\n",
	      argv[0]);
      exit(ERROR_PROC);
    }
    fi = format; fj = chunks;
    while (*fi != '\0') {
      if (*fi == OPT_PREFIX) {
	++fi;
	if (*fi == '\0') { /* ended too soon! */
	  fprintf(stderr, "%s: illegal format string \"%s\" (%c at end)\n",
		  argv[0], format, OPT_PREFIX);
	  exit(ERROR_ARGS);
	}
	*fj++ = *fi;
      }
      ++fi;
    }
    *fj = '\0';
  }
  /* for each file, get the user and group names? */
  if (index(sort_chunks, OWNERNAME) != NULL) {
    options |= GET_OWNER;
  }
  if (index(sort_chunks, GROUPNAME) != NULL) {
    options |= GET_GROUP;
  }
  for (j = chunks; *j != '\0'; ++j) {
    switch (*j) {
    case OWNERNAME:
      options |= GET_OWNER;
      break;
    case GROUPNAME:
      options |= GET_GROUP;
      break;
    case LINK:
      options |= GET_LINK_TARGET;
      break;
    default:
      /* nothing */
      break;
    }
  }
  /* get the effective user and group ids - these calls never fail */
  euid = geteuid();
  egid = getegid();
  /* clear the lists */
  flist_clear(argc, argv, &defdirlst);
  flist_clear(argc, argv, &deffilelst);
  if (optind == argc) { /* no actual files given on the command line, so just
			   do the currect directory */
    if ((tmp = newfnode(argc, argv, ".", options))
	== NULL) { /* this creates the record and does all the stat()ing
		      and stuff */
      exit(ERROR_PROC);
    }
    if (flist_append(argc, argv, &defdirlst, tmp) == 0) {
      exit(ERROR_PROC);
    }
  }
  else {
    for (; optind < argc; ++optind) {
      if ((tmp = newfnode(argc, argv, argv[optind], options))
	  == NULL) { /* this creates the record and does all the stat()ing
			and stuff */
	exit(ERROR_PROC);
      }
      if (tmp->s != NULL) { /* check that the stat succeeded, if not, ignore
			       this file */
	if (S_ISDIR(tmp->s->st_mode) && !(options & DIRS_AS_FILES)) {
	  if (flist_append(argc, argv, &defdirlst, tmp) == 0) {
	    exit(ERROR_PROC);
	  }
	}
	else {
	  if (flist_append(argc, argv, &deffilelst, tmp) == 0) {
	    exit(ERROR_PROC);
	  }
	}
      }
    }
  }
  /* okay, now we have everything that we want to show in one list or
     another */
  if (options & VERBOSE) {
    printf("%s: output format: \"%s\" order: \"%s\" style: \"%c\" line width: %d\n",
	   argv[0], chunks, sort_chunks, print_style, output_width);
    printf("Format chunks are, in order:\n");
    for (j = chunks; *j != '\0'; ++j) {
      showhelp(*j);
    }
    printf(":\n");
  }
  display_file_list(argc, argv, &deffilelst, print_style, output_width,
		    format, &cfile, options, now);
  flist_sort(argc, argv, &defdirlst);
  for (i = 0; i < defdirlst.size; ++i) {
    display_directory(argc, argv, defdirlst.fa[i], print_style,
		      output_width, format, &cfile, options, now);
  }
  return ERROR_SUCCESS;
}
