/***********************************************************************************

	Copyright (C) 2007-2012 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "entry_parser.hpp"


using namespace LIFEO;


void
EntryParser::reset( Glib::ustring::size_type start, Glib::ustring::size_type end )
{
    pos_start = start;
    pos_end = end;
    pos_current = pos_word = pos_regular = start;

    char_last = CC_NONE;
    char_req = CC_ANY;
    word_last = "";
    int_last = 0;
    date_last = 0;
    id_last = 0;
    lookingfor.clear();
    m_appliers.clear();
    if( start == 0 )
    {
        lookingfor.push_back( LF_IGNORE ); // to prevent formatting within title
        m_applier_nl = &EntryParser::apply_heading_end;
        apply_heading();
    }
    else
    {
        lookingfor.push_back( LF_NOTHING );
        m_applier_nl = NULL;
    }
}

void
EntryParser::parse( Glib::ustring::size_type start, Glib::ustring::size_type end )
{
    reset( start, end );

    for( ; pos_current < pos_end; ++pos_current )
    {
        char_current = get_char_at( pos_current );

        if( m_search_str.size() > 0 )
        {
            if( m_search_str[ i_search ] == Glib::Unicode::tolower( char_current ) )
            {
                if( i_search == 0 )
                    pos_search = pos_current;
                if( i_search == i_search_end )
                {
                    apply_match();
                    i_search = 0;
                }
                else
                    i_search++;
            }
            else
            {
                i_search = 0;
            }
        }

        // MARKUP PARSING
        switch( char_current )
        {
            case 0:     // should never be the case
            case '\n':
            case '\r':
                process_char( LF_NEWLINE,
                              LF_NUM_SPCIM_CHK|LF_ALPHA|LF_FORMATCHAR|
                              LF_SLASH|LF_DOTDATE|LF_MORE|LF_TAB|LF_IGNORE,
                              0, NULL,
                              CC_NEWLINE );
                break;
            case ' ':
                process_char( LF_SPACE,
                              LF_ALPHA|LF_NUMBER|LF_SLASH|LF_DOTDATE|LF_CHECKBOX,
                              LF_NOTHING, &EntryParser::trigger_subheading,
                              CC_SPACE );
                break;
            case '*':
                process_char( LF_ASTERISK,
                              LF_NUM_SPCIM_CHK|LF_ALPHA|LF_SLASH|LF_DOTDATE,
                              LF_NOTHING, &EntryParser::trigger_bold,
                              CC_SIGN );
                break;
            case '_':
                process_char( LF_UNDERSCORE,
                              LF_NUM_SPCIM_CHK|LF_SLASH|LF_DOTDATE,
                              LF_NOTHING, &EntryParser::trigger_italic,
                              CC_SIGN );
                break;
            case '=':
                process_char( LF_EQUALS,
                              LF_NUM_SPCIM_CHK|LF_ALPHA|LF_SLASH|LF_DOTDATE,
                              LF_NOTHING, &EntryParser::trigger_strikethrough,
                              CC_SIGN );
                break;
            case '#':
                process_char( LF_HASH,
                              LF_NUM_SPCIM_CHK|LF_ALPHA|LF_SLASH|LF_DOTDATE,
                              LF_NOTHING, &EntryParser::trigger_highlight,
                              CC_SIGN );
                break;
            case '[':
                process_char( LF_SBB,
                              LF_NUM_SPCIM_CHK|LF_ALPHA|LF_SLASH|LF_DOTDATE,
                              LF_NOTHING, &EntryParser::trigger_comment,
                              CC_SIGN );
                break;
            case ']':
                process_char( LF_SBE,
                              LF_NUM_SPCIM_CHK|LF_ALPHA|LF_SLASH|LF_DOTDATE,
                              0, NULL,
                              CC_SIGN );
                break;
            case '0': case '1': case '2': case '3': case '4':
            case '5': case '6': case '7': case '8': case '9':
                handle_number();   // calculates numeric value
                process_char( LF_NUMBER,
                              LF_SLASH|LF_ALPHA|LF_DOTDATE|LF_CHECKBOX,
                              LF_NOTHING, &EntryParser::trigger_link_date,
                              CC_NUMBER );
                break;
            case '.':
                process_char( LF_DOTDATE,
                              LF_NUM_SPCIM_CHK|LF_ALPHA|LF_SLASH,
                              LF_NOTHING, &EntryParser::trigger_ignore,
                              CC_SIGN );
                break;
            case '/':
                process_char( LF_SLASH|LF_DOTDATE,
                              LF_NUM_SPCIM_CHK|LF_ALPHA,
                              0, NULL,
                              CC_SIGN );
                break;
            case ':':
                process_char( LF_PUNCTUATION_RAW,
                              LF_NUM_SPCIM_CHK|LF_ALPHA|LF_SLASH|LF_DOTDATE,
                              LF_NOTHING, &EntryParser::trigger_link,
                              CC_SIGN );
                break;
            case '@':
                process_char( LF_AT,
                              LF_NUM_SPCIM_CHK|LF_ALPHA|LF_SLASH|LF_DOTDATE,
                              LF_NOTHING, &EntryParser::trigger_link_at,
                              CC_SIGN );
                break;
            case '>':
                process_char( LF_MORE,
                              LF_NUM_SPCIM_CHK|LF_ALPHA|LF_SLASH|LF_DOTDATE,
                              0, NULL,
                              CC_SIGN );
                break;
            case '\t':
                process_char( LF_TAB,
                              LF_NUM_SPCIM_SLH|LF_ALPHA|LF_DOTDATE,
                              LF_NOTHING, &EntryParser::trigger_list,
                              CC_TAB );
                break;
            // LIST CHARS
            case L'☐':
            case L'☑':
            case L'☒':
                process_char( LF_CHECKBOX,
                              LF_NUM_SPCIM_SLH|LF_ALPHA|LF_DOTDATE,
                              0, NULL,
                              CC_SIGN );
                break;
            default:
                process_char( LF_ALPHA,
                              LF_NUM_SPCIM_CHK|LF_DOTDATE|LF_SLASH,
                              0, NULL,
                              CC_ALPHA );   // most probably :)
                break;
        }
    }
    // end of the text -treated like new line
    process_char( LF_NEWLINE,
                  LF_NUM_SPCIM_CHK|LF_ALPHA|LF_FORMATCHAR|
                  LF_SLASH|LF_DOTDATE|LF_MORE|LF_TAB,
                  LF_EOT, NULL,
                  CC_NEWLINE );
}

inline void
EntryParser::process_char( unsigned int satisfies, unsigned int breaks,
              unsigned int triggers, FPtr_void parsing_triggerer,
              CharClass cc )
{
    unsigned int lf( lookingfor.front() );
    if( lf & satisfies )
    {
        if( ( lf & LF_APPLY ) && ( char_last & char_req ) )
        {
            lookingfor.clear();
            lookingfor.push_back( LF_NOTHING );

            ( this->*m_appliers.front() )();    // lookingfor has to be cleared beforehand

            if( satisfies & LF_NEWLINE )
                process_newline();
        }
        else
        if( satisfies & LF_NEWLINE )
        {
            lookingfor.clear();
            lookingfor.push_back( LF_NOTHING );

            process_newline();
        }
        else
        if( lf & LF_JUNCTION )
        {
            ( this->*m_appliers.front() )();
        }
        else
        {
            lookingfor.pop_front();
            if( lf & triggers )
            {
                ( this->*parsing_triggerer )();
            }
        }
    }
    else
    if( triggers & LF_EOT )
    {
        if( m_applier_nl == NULL)
        {
            pos_start = pos_current + 1;
            apply_regular();
        }
        process_newline();
    }
    else
    if( ( lf & breaks ) || ( lf & LF_IMMEDIATE ) || ( satisfies & LF_NEWLINE ) )
    {
        lookingfor.clear();
        lookingfor.push_back( LF_NOTHING );
        if( triggers & LF_NOTHING )
            ( this->*parsing_triggerer )();
        if( satisfies & LF_NEWLINE )
            process_newline();
    }
    else
    if( lf & triggers )
    {
        ( this->*parsing_triggerer )();
    }

    // SET NEW CHAR CLASS & ADJUST WORD_LAST ACOORDINGLY
    if( cc & CC_SEPARATOR )
        word_last.clear();
    else
    {
        if( word_last.empty() )
            pos_word = pos_current;
        word_last += char_current;
    }
    char_last = cc;
}

inline void
EntryParser::process_newline( void )
{
    if( m_applier_nl != NULL )
    {
        ( this->*m_applier_nl )();
        m_applier_nl = NULL;
    }
}

void
EntryParser::set_search_str( const Glib::ustring &str )
{
    m_search_str = str;
    i_search = 0;
    i_search_end = str.size() - 1;
}

// TRIGGERERS ======================================================================================
void
EntryParser::trigger_subheading( void )
{
    if( char_last == CC_NEWLINE )
    {
        lookingfor.clear();
        lookingfor.push_back( LF_NONSPACE|LF_APPLY );
        char_req = CC_ANY;
        pos_start = pos_current;
        m_appliers.clear();
        m_appliers.push_back( &EntryParser::apply_subheading_0 );
    }
}

void
EntryParser::apply_subheading_0( void )
{
    m_applier_nl = &EntryParser::apply_subheading_end;
    apply_subheading();
}

void
EntryParser::trigger_bold( void )
{
    if( char_last & CC_NOT_SEPARATOR )
        return;

    lookingfor.clear();
    lookingfor.push_back( LF_NONSPACE - LF_ASTERISK );
    lookingfor.push_back( LF_ASTERISK|LF_APPLY );
    char_req = CC_NOT_SEPARATOR;
    pos_start = pos_current;
    m_appliers.clear();
    m_appliers.push_back( &EntryParser::apply_bold );
}

void
EntryParser::trigger_italic( void )
{
    if( char_last & CC_NOT_SEPARATOR )
        return;

    lookingfor.clear();
    lookingfor.push_back( LF_NONSPACE - LF_UNDERSCORE );
    lookingfor.push_back( LF_UNDERSCORE|LF_APPLY );
    char_req = CC_NOT_SEPARATOR;
    pos_start = pos_current;
    m_appliers.clear();
    m_appliers.push_back( &EntryParser::apply_italic );
}

void
EntryParser::trigger_strikethrough( void )
{
    if( char_last & CC_NOT_SEPARATOR )
        return;

    lookingfor.clear();
    lookingfor.push_back( LF_NONSPACE - LF_EQUALS );
    lookingfor.push_back( LF_EQUALS|LF_APPLY );
    char_req = CC_NOT_SEPARATOR;
    pos_start = pos_current;
    m_appliers.clear();
    m_appliers.push_back( &EntryParser::apply_strikethrough );
}

void
EntryParser::trigger_highlight( void )
{
    if( char_last & CC_NOT_SEPARATOR )
        return;

    lookingfor.clear();
    lookingfor.push_back( LF_NONSPACE - LF_HASH );
    lookingfor.push_back( LF_HASH|LF_APPLY );
    char_req = CC_NOT_SEPARATOR;
    pos_start = pos_current;
    m_appliers.clear();
    m_appliers.push_back( &EntryParser::apply_highlight );
}

void
EntryParser::trigger_comment( void )
{
    lookingfor.clear();
    lookingfor.push_back( LF_SBB|LF_IMMEDIATE );
    lookingfor.push_back( LF_SBE );
    lookingfor.push_back( LF_SBE|LF_IMMEDIATE|LF_APPLY );
    char_req = CC_ANY;
    pos_start = pos_current;
    m_appliers.clear();
    m_appliers.push_back( &EntryParser::apply_comment );
}

void
EntryParser::trigger_link( void )
{
    PRINT_DEBUG( "word_last: " + word_last );
    m_flag_hidden_link = word_last[ 0 ] == '<';
    if( m_flag_hidden_link )
        word_last.erase( 0, 1 );

    char_req = CC_ANY;

    if( word_last == "http" || word_last == "https" ||
        word_last == "ftp" || word_last == "file" )
    {
        lookingfor.clear();
        lookingfor.push_back( LF_SLASH );
        lookingfor.push_back( LF_SLASH );
        if( word_last == "file" )
        {
            lookingfor.push_back( LF_SLASH );
            lookingfor.push_back( LF_NONSPACE );
        }
        else
            lookingfor.push_back( LF_ALPHA|LF_NUMBER ); // TODO: add dash
    }
    else
    if( word_last == "mailto" )
    {
        lookingfor.clear();
        lookingfor.push_back( LF_UNDERSCORE|LF_ALPHA|LF_NUMBER );
        lookingfor.push_back( LF_AT );
        lookingfor.push_back( LF_ALPHA|LF_NUMBER ); // TODO: add dash
    }
    else
    if( word_last == "deid" && m_flag_hidden_link )
    {
        lookingfor.clear();
        lookingfor.push_back( LF_NUMBER );
        lookingfor.push_back( LF_TAB|LF_JUNCTION );
        lookingfor.push_back( LF_NONSPACE - LF_MORE );
        lookingfor.push_back( LF_MORE|LF_APPLY );
        pos_start = pos_word;
        m_appliers.clear();
        m_appliers.push_back( &EntryParser::junction_link_hidden_tab );
        m_appliers.push_back( &EntryParser::apply_link_id );
        return;
    }
    else
        return;

    if( m_flag_hidden_link )
    {
        lookingfor.push_back( LF_TAB|LF_JUNCTION );
        lookingfor.push_back( LF_NONSPACE - LF_MORE );
        lookingfor.push_back( LF_MORE|LF_APPLY );
        m_appliers.clear();
        m_appliers.push_back( &EntryParser::junction_link_hidden_tab );
        m_appliers.push_back( &EntryParser::apply_link );
    }
    else
    {
        lookingfor.push_back( LF_TAB|LF_NEWLINE|LF_SPACE|LF_APPLY );
        m_appliers.clear();
        m_appliers.push_back( &EntryParser::apply_link );
    }
    pos_start = pos_word;
}

void
EntryParser::trigger_link_at( void )
{
    PRINT_DEBUG( "word_last [@]: " + word_last );
    if( char_last & CC_SEPARATOR )
        return;

    m_flag_hidden_link = false;
    word_last.insert( 0, "mailto:" );
    lookingfor.clear();
    lookingfor.push_back( LF_ALPHA|LF_NUMBER ); // TODO: add dash
    lookingfor.push_back( LF_TAB|LF_NEWLINE|LF_SPACE|LF_APPLY );
    char_req = CC_ANY;
    pos_start = pos_word;
    m_appliers.clear();
    m_appliers.push_back( &EntryParser::apply_link );
}

void
EntryParser::trigger_link_date( void )
{
    char_req = CC_ANY;
    lookingfor.clear();
    lookingfor.push_back( LF_NUMBER );
    lookingfor.push_back( LF_NUMBER );
    lookingfor.push_back( LF_NUMBER );
    lookingfor.push_back( LF_DOTYM|LF_JUNCTION );
    lookingfor.push_back( LF_NUMBER );
    lookingfor.push_back( LF_NUMBER );
    lookingfor.push_back( LF_DOTMD|LF_JUNCTION );
    lookingfor.push_back( LF_NUMBER );

    m_appliers.clear();
    m_appliers.push_back( &EntryParser::junction_date_dotym );
    m_appliers.push_back( &EntryParser::junction_date_dotmd );

    m_flag_hidden_link = ( word_last == "<" );
    if( m_flag_hidden_link )
    {
        lookingfor.push_back( LF_NUMBER );
        lookingfor.push_back( LF_TAB|LF_JUNCTION );
        lookingfor.push_back( LF_NONSPACE );
        lookingfor.push_back( LF_MORE|LF_APPLY );
        pos_start = pos_current - 1;
        m_appliers.push_back( &EntryParser::junction_link_hidden_tab );
    }
    else
    {
        lookingfor.push_back( LF_NUMBER|LF_APPLY );
        pos_start = pos_current;
    }
    m_appliers.push_back( &EntryParser::apply_link_date );
}

void
EntryParser::trigger_list( void )
{
    if( char_last != CC_NEWLINE )
        return;

    lookingfor.clear();
    lookingfor.push_back( LF_CHECKBOX|LF_JUNCTION );
    char_req = CC_ANY;
    pos_start = pos_current;
    m_appliers.clear();
    m_appliers.push_back( &EntryParser::junction_list );
}

void
EntryParser::trigger_ignore( void )
{
    if( char_last == CC_NEWLINE )
    {
        lookingfor.clear();
        lookingfor.push_back( LF_TAB|LF_IMMEDIATE|LF_JUNCTION );
        char_req = CC_ANY;
        pos_start = pos_current;
        m_appliers.clear();
        m_appliers.push_back( &EntryParser::junction_ignore );
    }
}

void
EntryParser::junction_link_hidden_tab( void )
{
    lookingfor.pop_front();
    pos_tab = pos_current + 1;
    m_appliers.pop_front();
    id_last = int_last;     // if not id link assignment is in vain
}

void
EntryParser::junction_list( void )
{
    lookingfor.front() = LF_SPACE|LF_IMMEDIATE|LF_APPLY;
    char_req = CC_ANY;

    switch( char_current )
    {
        case L'☐':
            m_appliers.front() = &EntryParser::apply_check_unf;
            break;
        case L'☑':
            m_appliers.front() = &EntryParser::apply_check_fin;
            break;
        case L'☒':
            m_appliers.front() = &EntryParser::apply_check_ccl_0;
            break;
        //default: // should never occur
    }
}

void
EntryParser::junction_date_dotym( void )
{
    if( int_last >= Date::YEAR_MIN && int_last <= Date::YEAR_MAX )
    {
        date_last.set_year( int_last );
        lookingfor.pop_front();
        m_appliers.pop_front();
    }
    else
    {
        lookingfor.clear();
        lookingfor.push_back( LF_NOTHING );
    }
}

void
EntryParser::junction_date_dotmd( void )
{
    if( int_last >= 1 && int_last <= 12 )
    {
        date_last.set_month( int_last );
        lookingfor.pop_front();
        m_appliers.pop_front();
    }
    else
    {
        lookingfor.clear();
        lookingfor.push_back( LF_NOTHING );
    }
}

void
EntryParser::junction_ignore( void )
{
//    m_applier_nl = &EntryParser::apply_ignore_end;
    lookingfor.front() = LF_IGNORE;
    m_appliers.pop_front();
    apply_ignore();
}

// TODO: this is a temporary solution until begin and end appliers, and junction functions will
// be bundled in a class also eliminating the need for separate deques for all of them:
void
EntryParser::apply_check_ccl_0( void )
{
    m_applier_nl = &EntryParser::apply_check_ccl_end;
    apply_check_ccl();
}

// HELPERS =========================================================================================
inline void
EntryParser::handle_number( void )
{
    if( char_last == CC_NUMBER )
    {
        int_last *= 10;
        int_last += ( char_current - '0' );
    }
    else
        int_last = ( char_current - '0' );
}
