/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iomanip>

#include "lifeobase.hpp"
#include "panel_diary.hpp"
#include "view_entry.hpp"
#include "widget_textview.hpp"


using namespace LIFEO;

PanelDiary::PanelDiary( void )
{
	try
	{
		Lifeobase::builder->get_widget( "calendar_main", m_calendar );
		Lifeobase::builder->get_widget_derived( "treeview_main", m_treeview_entries );
		Lifeobase::builder->get_widget_derived( "entry_filter", m_entry_filter );
		Lifeobase::builder->get_widget_derived( "entry_replace", m_entry_replace );
		Lifeobase::builder->get_widget( "tbutton_match_prev", m_button_match_prev );
		Lifeobase::builder->get_widget( "tbutton_match_next", m_button_match_next );
		Lifeobase::builder->get_widget( "tbutton_replace", m_toolbutton_replace );
		Lifeobase::builder->get_widget( "tbutton_replace_all", m_toolbutton_replace_all );
		Lifeobase::builder->get_widget( "toolbar_replace", m_toolbar_replace );

		Lifeobase::builder->get_widget( "menuitem_sort_by_date", m_menuitem_sort_by_date );
		Lifeobase::builder->get_widget( "menuitem_sort_by_size", m_menuitem_sort_by_size );
		Lifeobase::builder->get_widget( "menuitem_filter_favorites", m_menuitem_filter_favorites );
	}
	catch( ... ) { }

	// FILTER & REPLACE
	m_entry_filter->set_idle_text( _( "Filter" ) );
	m_entry_replace->set_idle_text( _( "Replace" ) );

	// SIGNALS
	m_calendar->signal_day_selected_double_click().connect(
			sigc::mem_fun( this, &PanelDiary::handle_calendar_doubleclicked ) );
	m_calendar->signal_month_changed().connect(
			sigc::mem_fun( this, &PanelDiary::handle_calendar_monthchanged ) );
	m_calendar->signal_day_selected().connect(
			sigc::mem_fun( this, &PanelDiary::handle_calendar_dayselected ) );
	m_calendar->signal_button_press_event().connect_notify(
			sigc::mem_fun( this, &PanelDiary::handle_calendar_button_pressed ) );

	m_entry_filter->signal_changed().connect(
			sigc::mem_fun( this, &PanelDiary::handle_entry_filter_changed ) );
	m_entry_filter->signal_activate().connect(
			sigc::mem_fun( this, &PanelDiary::go_next_match ) );

	m_entry_replace->signal_activate().connect(
			sigc::mem_fun( this, &PanelDiary::replace_match ) );
	m_toolbutton_replace->signal_clicked().connect(
			sigc::mem_fun( this, &PanelDiary::replace_match ) );
	m_toolbutton_replace_all->signal_clicked().connect(
			sigc::mem_fun( this, &PanelDiary::replace_all_matches ) );

	m_treeview_entries->signal_row_expanded().connect(
			sigc::mem_fun( this, &PanelDiary::handle_treeview_row_expanded ) );
	m_treeview_entries->signal_row_collapsed().connect(
			sigc::mem_fun( this, &PanelDiary::handle_treeview_row_expanded ) );

	m_menuitem_sort_by_date->signal_activate().connect(
			sigc::bind(
					sigc::mem_fun( m_treeview_entries,
								   &WidgetEntryList::handle_sorting_criteria_changed ),
					SC_DATE ) );
	m_menuitem_sort_by_size->signal_activate().connect(
			sigc::bind(
					sigc::mem_fun( m_treeview_entries,
								   &WidgetEntryList::handle_sorting_criteria_changed ),
					SC_SIZE ) );
	m_menuitem_filter_favorites->signal_activate().connect(
			sigc::mem_fun( this, &PanelDiary::handle_filter_favorites_toggled ) );
}

void
PanelDiary::add_chapter_category_to_list( const CategoryChapters *chapters )
{
	Gtk::TreeRow	row_container;
	Chapter			*chapter = NULL;

	for( CategoryChapters::const_iterator iter_chapter = chapters->begin();
		 iter_chapter != chapters->end();
		 ++iter_chapter )
	{
		chapter = iter_chapter->second;
		if( ! chapter->is_initialized() )
			continue;

		// show all folders even when they are empty:
		row_container = * m_treeview_entries->m_treestore_entries->append( l_row_diary.children() );
		row_container[ ListData::colrec->ptr ] = chapter;	// ptr must be first:
		row_container[ ListData::colrec->info ] = chapter->get_list_str();
		row_container[ ListData::colrec->icon ] = chapter->get_icon();
		chapter->m_list_data->treepath = m_treeview_entries->m_treestore_entries->get_path( row_container );

		chapter->reset_size();

		if( l_entry == NULL )
			continue;

		while( l_entry->get_date() >= chapter->get_date() )
		{
			chapter->increase_size();
			if( l_entry->get_filtered_out() == false )
			{
				Lifeobase::m_entrycount++;
				l_row_entry = * m_treeview_entries->m_treestore_entries->append( row_container.children() );
				l_row_entry[ ListData::colrec->ptr ] = l_entry;
				l_row_entry[ ListData::colrec->info ] = l_entry->get_list_str();
				l_row_entry[ ListData::colrec->icon ] = l_entry->get_icon();

				l_entry->m_list_data->treepath = m_treeview_entries->m_treestore_entries->get_path( l_row_entry );
			}

			++l_itr_entry;
			if( l_itr_entry != Diary::d->m_entries.end() )
				l_entry = l_itr_entry->second;
			else
			{
				l_entry = NULL;
				break;
			}
		}

		if( chapter->get_expanded() )
			m_treeview_entries->expand_element( chapter );
	}
}

void
PanelDiary::update_entry_list( void )
{
	PRINT_DEBUG( "update_entry_list()" );
	Lifeobase::m_internaloperation++;

	l_itr_entry = Diary::d->m_entries.begin();
	l_entry = ( l_itr_entry != Diary::d->m_entries.end() ? l_itr_entry->second : NULL );

	Lifeobase::base->sync_entry();

	m_treeview_entries->m_treestore_entries->clear();
	Lifeobase::m_entrycount = 0;

	// DIARY ITEM
	l_row_diary = * m_treeview_entries->m_treestore_entries->append();
	l_row_diary[ ListData::colrec->ptr ] = Diary::d; // ptr must be first
	l_row_diary[ ListData::colrec->info ] =
			Glib::ustring::compose( "<b>%1</b>", Diary::d->get_name() );
	l_row_diary[ ListData::colrec->icon ] = Lifeobase::icons->diary_16;

	Diary::d->m_list_data->treepath =
			m_treeview_entries->m_treestore_entries->get_path( l_row_diary );

	// CHAPTERS
	//CategoryChapters *chapters = Diary::d->get_current_chapter_ctg();

	if( Diary::d->get_sorting_criteria() == SC_DATE )
	{
		add_chapter_category_to_list( Diary::d->get_topics() );
		add_chapter_category_to_list( Diary::d->get_current_chapter_ctg() );
	}

	for( ;
		 l_itr_entry != Diary::d->m_entries.end();
		 ++l_itr_entry )
	{
		l_entry = ( *l_itr_entry ).second;
		if( l_entry->get_filtered_out() == false )
		{
			Lifeobase::m_entrycount++;

			l_row_entry = *m_treeview_entries->m_treestore_entries->append(
					l_row_diary.children() );
			l_row_entry[ ListData::colrec->ptr ] = l_entry; // ptr must be first
			l_row_entry[ ListData::colrec->info ] = l_entry->get_list_str();
			l_row_entry[ ListData::colrec->icon ] = l_entry->get_icon();

			l_entry->m_list_data->treepath =
					m_treeview_entries->m_treestore_entries->get_path( l_row_entry );
		}
	}
	// always expand diary
	m_treeview_entries->expand_element( Diary::d );

	// UPDATE DIARY ELEM IF LAST ONE IS NOT IN THE LIST ANYMORE
	DiaryElement *elem = Lifeobase::base->get_cur_elem();
	bool flag_show_elem = true;
	if( Lifeobase::m_entrycount <= 0 )
		elem = Diary::d;
	else
	if( elem != NULL )
	{
		if( elem->get_filtered_out() )
			elem = Diary::d->get_entry_first();
		else
			flag_show_elem = false;
	}
	else	// the case at the start-up
		elem = Diary::d->get_startup_elem();

	if( flag_show_elem )
		elem->show();
	/*else
	if( elem->get_type() >= DiaryElement::IT_DIARY )	// only entry list elements
		m_treeview_entries->present_row( elem->m_list_data->treepath );*/

	Diary::d->set_filtering_status_applied();

	Lifeobase::m_internaloperation--;
}

void
PanelDiary::update_calendar( void )
{
	guint year, month, day;

	m_calendar->get_date( year, month, day );
	month++;	// fix month
	Entry *entry;
	m_calendar->clear_marks();

	for( Entryiter itr_entry = Diary::d->get_entries().begin();
		 itr_entry != Diary::d->get_entries().end();
		 ++itr_entry )
	{
		entry = itr_entry->second;
		if( entry->get_filtered_out() || entry->get_date().is_ordinal() )
			continue;

		if( entry->get_date().get_year() == year )
		{
			if( entry->get_date().get_month() == month )
			{
				m_calendar->mark_day( entry->get_date().get_day() );
			}
			else
			if( entry->get_date().get_month() < month )
				break;
		}
		else
		if( entry->get_date().get_year() < year )
			break;
	}
}

void
PanelDiary::handle_login( void )
{
	if( Diary::d->get_sorting_criteria() == SC_DATE )
		m_menuitem_sort_by_date->set_active();
	else
		m_menuitem_sort_by_size->set_active();

	// not called automatically because of m_internaloperation:
	m_treeview_entries->set_sorting_criteria();
	update_entry_list();
}

void
PanelDiary::handle_logout( void )
{
	m_entry_filter->set_text( "" );
	m_entry_replace->set_text( "" );
	m_menuitem_filter_favorites->set_active( false );
	m_treeview_entries->m_treestore_entries->clear();
}

void
PanelDiary::show( const DiaryElement *element )
{
	if( element->get_type() >= DiaryElement::IT_DIARY )
		m_treeview_entries->present_element( element );
	else
		m_treeview_entries->get_selection()->unselect_all();

	if( element->get_type() > DiaryElement::IT_DIARY )
		select_date_in_calendar( element->get_date() );
	else
		m_calendar->select_day( 0 );
}

void
PanelDiary::show_prev_session_elem( void )
{
	DiaryElement *elem( Diary::d->get_prev_session_elem() );
	if( elem != NULL )
		elem->show();
	else
		print_info( "Previous sesion element cannot be found!" );
}

void
PanelDiary::present_current_row( void )
{
	Gtk::TreeIter iter = m_treeview_entries->get_selection()->get_selected();
	if( iter == m_treeview_entries->m_treemodelsort_entries->children().end() )
		return;
	Gtk::TreePath path = m_treeview_entries->m_treemodelsort_entries->get_path( iter );

	m_treeview_entries->scroll_to_row( path );
}

void
PanelDiary::handle_entry_filter_changed( void )
{
	if( Lifeobase::m_internaloperation ) return;
	// current entry must be closed here or else it loses...
	// ...changes since it was selected:
	Lifeobase::base->sync_entry();

	const std::string filterstring( m_entry_filter->get_text().lowercase() );

	Diary::d->set_filter_text( filterstring );
	update_entry_list();

	if( filterstring.size() > 0 )
	{
		if( Lifeobase::m_entrycount > 0 )
		{
			TextviewDiary::m_buffer->set_searchstr( filterstring );
			if( Lifeobase::base->get_cur_elem_type() == DiaryElement::IT_ENTRY )
				TextviewDiary::m_buffer->select_searchstr_next();
		}

		m_entry_filter->set_tooltip_markup(
				Glib::ustring::compose(
						_( "Found in <b>%1</b> of <b>%2</b> entrie(s)" ),
						Lifeobase::m_entrycount, Diary::d->get_size() ) );
	}
	else
	{
		TextviewDiary::m_buffer->set_searchstr( "" );
		m_entry_filter->set_tooltip_text( _( "Enter text to be filtered" ) );
	}

	update_calendar();

	if( Lifeobase::m_entrycount < 1 || filterstring.size() < 1 )
	{
		m_button_match_prev->hide();
		m_button_match_next->hide();
		m_toolbar_replace->hide();
	}
	else
	{
		m_button_match_prev->show();
		m_button_match_next->show();
		m_toolbar_replace->show();
	}
}

void
PanelDiary::handle_filter_favorites_toggled( void )
{
	if( Lifeobase::m_internaloperation ) return;

	Diary::d->toggle_filter_favorites();	// set filter
	update_entry_list();					// apply the filter
}

inline bool
PanelDiary::make_path_deeper_last( Gtk::TreePath &path )
{
	if( ! m_treeview_entries->m_treemodelsort_entries->
					get_iter( path )->children().empty() )
	{
		path.push_back( m_treeview_entries->m_treemodelsort_entries->
				get_iter( path )->children().size() - 1 );
		return true;
	}
	return false;
}

inline bool
PanelDiary::move_path_next( Gtk::TreePath &path )
{
	if( unsigned( path.back() ) < ( m_treeview_entries->m_treemodelsort_entries->
				get_iter( path )->parent()->children().size() - 1 ) )
	{
		path.next();
		return true;
	}
	else
		return false;
}

void
PanelDiary::go_up( bool flag_entry_operation )
{
	// flag_entry: if true, goes to an entry item even when the current...
	// ...item is a folder
	Gtk::TreeIter iter( m_treeview_entries->get_selection()->get_selected() );
	if( iter == m_treeview_entries->m_treemodelsort_entries->children().end() )
		return;
	Gtk::TreePath path( m_treeview_entries->m_treemodelsort_entries->get_path( iter ) );
	DiaryElement *elem;

	do
	{
		if( ! path.prev() )
		{
			if( path.size() > 1 )
				path.up();
			else	// diary
			if( make_path_deeper_last( path ) )
				make_path_deeper_last( path ); // go still deeper if possible:
		}
		else
			make_path_deeper_last( path );

		elem = ( * m_treeview_entries->m_treemodelsort_entries->
						get_iter( path ) )[ ListData::colrec->ptr ];
	}
	// repeat until an entry is found if that is what we are looking for
	// BEWARE: do not set this flag when there is no entry in the list
	while( flag_entry_operation && elem->get_type() != DiaryElement::IT_ENTRY );

	PRINT_DEBUG( "previous path: " + path.to_string() );

	Gtk::TreeRow row = * m_treeview_entries->m_treemodelsort_entries->get_iter( path );
	DiaryElement *element( row[ ListData::colrec->ptr ] );
	element->show();
}

void
PanelDiary::go_down( bool flag_entry_operation )
{
	// flag_entry: if true, goes to an entry item even when the current...
	// ...item is a folder
	Gtk::TreeIter iter( m_treeview_entries->get_selection()->get_selected() );
	if( iter == m_treeview_entries->m_treemodelsort_entries->children().end() )
		return;
	Gtk::TreePath path( m_treeview_entries->m_treemodelsort_entries->get_path( iter ) );
	DiaryElement *elem;

	do
	{
		if( ! m_treeview_entries->m_treemodelsort_entries->
					get_iter( path )->children().empty() )
			path.down();
		else
		if( ! move_path_next( path ) )
		{
			while( path.size() > 1 )
			{
				path.up();
				if( move_path_next( path ) )
					break;
			}
		}

		elem = ( * m_treeview_entries->m_treemodelsort_entries->
						get_iter( path ) )[ ListData::colrec->ptr ];
	}
	// repeat until an entry is found if that is what we are looking for
	// BEWARE: do not set this flag when there is no entry in the list
	while( flag_entry_operation && elem->get_type() != DiaryElement::IT_ENTRY );

	PRINT_DEBUG( "next path: " + path.to_string() );

	Gtk::TreeRow row = * m_treeview_entries->m_treemodelsort_entries->get_iter( path );
	DiaryElement *listitem = row[ ListData::colrec->ptr ];
	listitem->show();
}

void
PanelDiary::handle_calendar_doubleclicked( void )
{
	guint year, month, day;

	m_calendar->get_date( year, month, day );

	Entry *entry = Diary::d->create_entry( Date( year, month + 1, day ) );

	// FIXME: replace these with more direct functions like add_entry()
	// (after treeview becomes a class of its own)
	update_entry_list();
	update_calendar();

	entry->show();

	// this is the only way i could find to give focus to textview
	//FIXME
//	Glib::signal_timeout().connect_once(
//			sigc::mem_fun( m_textviewdiary, &Gtk::Widget::grab_focus ), 200 );
}

void
PanelDiary::handle_calendar_dayselected( void )
{
	if( Lifeobase::m_internaloperation ) return;

	guint year, month, day;
	m_calendar->get_date( year, month, day );
	Date date( year, month + 1, day );
	Entry *entry;
	if( date.get_pure() == ( Lifeobase::base->get_cur_elem()->get_date().get_pure() ) )
	{
		//date = m_panel_main->get_cur_elem()->get_date();
		entry = Diary::d->get_entry_nextinday( date );
	}
	else
	{
		entry = Diary::d->get_entry( date );
	}

	if( entry )
		entry->show();
}

void
PanelDiary::handle_calendar_monthchanged( void )
{
	if( Lifeobase::m_internaloperation ) return;

	update_calendar();
}

void
PanelDiary::create_chapter( const Date &date )
{
	// CAUTION: nohing is checked here as everything should have been checked before

	Chapter *c = Diary::d->get_current_chapter_ctg()->create_chapter(
			_( "Untitled chapter" ), date );

	Lifeobase::base->update_entry_list();
	c->show();
}

void
PanelDiary::handle_calendar_button_pressed( GdkEventButton *event )
{
	if( event->button != 3 )
		return;

	static Gtk::Menu *m_menu( NULL );

	if( m_menu )
	{
		delete m_menu;
		m_menu = NULL;
	}

	m_menu = new Gtk::Menu;

	m_menu->items().push_back(
			Gtk::Menu_Helpers::MenuElem( _( "Create an _Entry" ),
			sigc::mem_fun( this, &PanelDiary::handle_calendar_doubleclicked ) ) );

	CategoryChapters *cc = Diary::d->get_current_chapter_ctg();
	if( cc != NULL )
	{
		guint year, month, day;
		m_calendar->get_date( year, month, day );
		Date date_selected( year, month + 1, day );
		if( cc->get_chapter( date_selected ) == NULL )
		{
			m_menu->items().push_back(
					Gtk::Menu_Helpers::MenuElem( _( "Create a _Chapter" ),
					sigc::bind( sigc::mem_fun( this, &PanelDiary::create_chapter ),
								date_selected ) ) );
		}
	}

	m_menu->accelerate( *m_calendar );
	m_menu->popup( event->button, event->time );

	return;
}

void
PanelDiary::select_date_in_calendar( const Date &date )
{
	++Lifeobase::m_internaloperation;

	// CALENDAR
	// the current day has to be deselected first to avoid gtk's complaints
	m_calendar->select_day( 0 );
	if( ! date.is_ordinal() )
	{
		m_calendar->select_month( date.get_month() - 1, date.get_year() );
		update_calendar();
		if( date.get_day() )
			m_calendar->select_day( date.get_day() );
	}

	--Lifeobase::m_internaloperation;
}

void
PanelDiary::go_prev_match( void )
{
	if( Lifeobase::m_entrycount < 1 )
		return;

	// TODO: move to Entry
	if( Lifeobase::base->get_cur_elem_type() == DiaryElement::IT_ENTRY )
		if( TextviewDiary::m_buffer->select_searchstr_previous() )
			return;

	int i = 0;
	do
	{
		go_up( true );
		// put cursor to the end:
		TextviewDiary::m_buffer->select_range( TextviewDiary::m_buffer->end(),
											   TextviewDiary::m_buffer->end() );
		i++;
	}
	while( ! TextviewDiary::m_buffer->select_searchstr_previous() &&
		   i < Lifeobase::m_entrycount );
}

void
PanelDiary::go_next_match( void )
{
	if( Lifeobase::m_entrycount < 1 )
		return;

	// TODO: move to Entry
	if( Lifeobase::base->get_cur_elem_type() == DiaryElement::IT_ENTRY )
		if( TextviewDiary::m_buffer->select_searchstr_next() )
			return;

	int i = 0;
	do
	{
		go_down( true );
		i++;
	}
	while( ! TextviewDiary::m_buffer->select_searchstr_next() &&
		   i < Lifeobase::m_entrycount );
}

void
PanelDiary::replace_match( void )
{
	if( ! TextviewDiary::m_buffer->get_has_selection() )
		go_next_match();
	if( TextviewDiary::m_buffer->get_has_selection() )
	{
		TextviewDiary::m_buffer->erase_selection();
		TextviewDiary::m_buffer->insert_at_cursor( m_entry_replace->get_text() );
		go_next_match();
	}
}

void
PanelDiary::replace_all_matches( void )
{
	Lifeobase::base->sync_entry();

	Diary::d->replace_text( m_entry_replace->get_text() );
	// update current entry:
	if( Lifeobase::base->get_cur_elem_type() == DiaryElement::IT_ENTRY )
	{
		Entry *entry = dynamic_cast< Entry* >( Lifeobase::base->get_cur_elem() );
		TextviewDiary::m_buffer->set_richtext( entry );
	}
}

Gtk::TreeRow
PanelDiary::get_row( const Gtk::TreePath &path )
{
	return( * m_treeview_entries->m_treestore_entries->get_iter( path ) );
	// TODO: check validity
}

void
PanelDiary::handle_treeview_row_expanded( const Gtk::TreeIter &iter,
										  const Gtk::TreePath &path )
{
	//if( Lifeobase::m_internaloperation )
		//return;

	DiaryElement *element = ( * iter )[ ListData::colrec->ptr ];
	if( element->get_type() == DiaryElement::IT_CHAPTER )
	{
		Chapter *chapter = dynamic_cast< Chapter* >( element );
		chapter->set_expanded( m_treeview_entries->row_expanded( path ) );
	}
}
