/***********************************************************************************

	Copyright (C) 2007-2009 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cmath>

#include "view_entry.hpp"
#include "widget_textview.hpp"


using namespace LIFEO;


// STATIC MEMBERS
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_heading;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_subheading;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_match;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_markup;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_bold;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_italic;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_strikethrough;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_highlight;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_link;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_link_broken;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_checkbox;

Gdk::Color				TextbufferDiary::m_theme_color_match( "green" );
Gdk::Color				TextbufferDiary::m_theme_color_markup( "gray" );
Gdk::Color				TextbufferDiary::m_theme_color_link( "navy" );
Gdk::Color				TextbufferDiary::m_theme_color_linkbroken( "red" );

Gtk::TextBuffer					*UndoEdit::m_ptr2buffer;

// LINK
Link::Link(	const Glib::RefPtr< Gtk::TextMark > &start,
			const Glib::RefPtr< Gtk::TextMark > &end )
	:	m_mark_start( start ), m_mark_end( end )
{

}

Link::~Link()
{
	// TODO: is this necessary?
	Glib::RefPtr< Gtk::TextBuffer > buffer = m_mark_start->get_buffer();
	if ( buffer )
	{
		buffer->delete_mark( m_mark_start );
		buffer->delete_mark( m_mark_end );
	}
}

// LINK TO ENTRY
// static variable:
LinkEntry::Signal_void_Date LinkEntry::m_signal_activated;

LinkEntry::LinkEntry(	const Glib::RefPtr< Gtk::TextMark > &start,
						const Glib::RefPtr< Gtk::TextMark > &end,
						Date date,
						unsigned int order )
	:	Link( start, end ), m_date( date ), m_order( order )
{
}

void
LinkEntry::go( void )
{
	m_signal_activated.emit( m_date );
}

// LINK TO URI
Gtk::TextBuffer *LinkUri::m_ptr2buffer;

LinkUri::LinkUri(	const Glib::RefPtr< Gtk::TextMark > &start,
					const Glib::RefPtr< Gtk::TextMark > &end,
					const std::string& url )
	:	Link( start, end ), m_url( url )
{
}

void
LinkUri::go( void )
{
	GError *err = NULL;
	gtk_show_uri (NULL, m_url.c_str(), GDK_CURRENT_TIME, &err);
}

// LINK CHECKBOX
const Glib::ustring		LinkCheck::boxes = "☐☑☒";

LinkCheck::LinkCheck( const Glib::RefPtr< Gtk::TextMark > &start,
					  const Glib::RefPtr< Gtk::TextMark > &end,
					  unsigned int state_index )
	:	Link( start, end ), m_state_index( state_index )
{
}

void
LinkCheck::go( void )
{
	PRINT_DEBUG( "LinkCheck::go()" );
	m_state_index = ( m_state_index + 1 ) % 3;

	// link should be preserved
	TextviewDiary::m_buffer->m_flag_ongoingoperation = true;
	TextviewDiary::m_buffer->insert(
			m_mark_start->get_iter(), boxes.substr( m_state_index, 1) );

	Gtk::TextIter iter_start = m_mark_start->get_iter();
	iter_start++;
	TextviewDiary::m_buffer->erase( iter_start, m_mark_end->get_iter() );

	iter_start = m_mark_start->get_iter();
	Gtk::TextIter iter_end = m_mark_end->get_iter();
	TextviewDiary::m_buffer->apply_tag(
					TextbufferDiary::m_tag_checkbox, iter_start, iter_end );

	iter_start = m_mark_end->get_iter();
	iter_end.forward_line();
	TextviewDiary::m_buffer->remove_all_tags( iter_start, iter_end );
	switch( m_state_index )
	{
		case 1:
			TextviewDiary::m_buffer->apply_tag(
					TextbufferDiary::m_tag_highlight, iter_start, iter_end );
			break;
		case 2:
			TextviewDiary::m_buffer->apply_tag(
					TextbufferDiary::m_tag_strikethrough, iter_start, iter_end );
			break;
	}

	TextviewDiary::m_buffer->m_flag_ongoingoperation = false;
}

// TEXTBUFFERDIARY
TextbufferDiary::TextbufferDiary( void )
	:	Gtk::TextBuffer(),
	parser_date_link( 0 ), m_option_search( false ),
	m_flag_settextoperation( false ), m_flag_ongoingoperation( false ),
	m_flag_parsing( false ),
	m_ptr2textview( NULL ), m_ptr2spellobj( NULL )
{
	LinkUri::m_ptr2buffer = this;
	UndoEdit::m_ptr2buffer = this;
	Glib::RefPtr< TagTable > tag_table = get_tag_table();

	m_tag_heading = Tag::create( "heading" );
	m_tag_heading->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_heading->property_scale() = 1.5;
	tag_table->add( m_tag_heading );

	m_tag_subheading = Tag::create( "subheading" );
	m_tag_subheading->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_subheading->property_scale() = 1.2;
	tag_table->add( m_tag_subheading );

	m_tag_match = Tag::create( "match" );
	tag_table->add( m_tag_match );

	m_tag_markup = Tag::create( "markup" );
	//m_tag_markup->property_invisible() = true;	// for now visible is the better default
	tag_table->add( m_tag_markup );

	m_tag_bold = Tag::create( "bold" );
	m_tag_bold->property_weight() = Pango::WEIGHT_BOLD;
	tag_table->add( m_tag_bold );

	m_tag_italic = Tag::create( "italic" );
	m_tag_italic->property_style() = Pango::STYLE_ITALIC;
	tag_table->add( m_tag_italic );

	m_tag_strikethrough = Tag::create( "strikethrough" );
	m_tag_strikethrough->property_strikethrough() = true;
	tag_table->add( m_tag_strikethrough );

	m_tag_highlight = Tag::create( "highlight" );
	tag_table->add( m_tag_highlight );

	m_tag_link = Tag::create( "link" );
	m_tag_link->property_underline() = Pango::UNDERLINE_SINGLE;
	tag_table->add( m_tag_link );

	m_tag_link_broken = Tag::create( "link.broken" );
	m_tag_link_broken->property_underline() = Pango::UNDERLINE_SINGLE;
	tag_table->add( m_tag_link_broken );

	m_tag_checkbox = Tag::create( "checkbox" );
	m_tag_checkbox->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_checkbox->property_scale() = 1.2;
	tag_table->add( m_tag_checkbox );

	//PLANS
	m_plan_date.push_back( LF_NUMBER|LF_APPLY_NUMBER );
	m_plan_date.push_back( LF_NUMBER|LF_APPLY_NUMBER );
	m_plan_date.push_back( LF_NUMBER|LF_APPLY_NUMBER );
	m_plan_date.push_back( LF_DOTYM|LF_APPLY_DOTYM );
	m_plan_date.push_back( LF_NUMBER|LF_APPLY_NUMBER );
	m_plan_date.push_back( LF_NUMBER|LF_APPLY_NUMBER );
	m_plan_date.push_back( LF_DOTMD|LF_APPLY_DOTMD );
	m_plan_date.push_back( LF_NUMBER|LF_APPLY_NUMBER );
	m_plan_date.push_back( LF_NUMBER|LF_APPLY_LINK_DATE );

	m_plan_subheading.push_back( LF_NONSPACE );
	m_plan_subheading.push_back( LF_NEWLINE|LF_APPLY_SUBHEADING );

	m_plan_bold.push_back( LF_NONSPACE );
	m_plan_bold.push_back( LF_ASTERISK|LF_APPLY_MARKUP );

	m_parsers[ LF_NEWLINE|LF_APPLY_HEADING ] = &TextbufferDiary::parsing_applier_heading;
	m_parsers[ LF_NEWLINE|LF_APPLY_SUBHEADING ] = &TextbufferDiary::parsing_applier_subheading;
	m_parsers[ LF_ASTERISK|LF_APPLY_MARKUP ] = &TextbufferDiary::parsing_applier_bold;
	m_parsers[ LF_UNDERSCORE|LF_APPLY_MARKUP ] = &TextbufferDiary::parsing_applier_italic;
	m_parsers[ LF_EQUALS|LF_APPLY_MARKUP ] = &TextbufferDiary::parsing_applier_strikethrough;
	m_parsers[ LF_HASH|LF_APPLY_MARKUP ] = &TextbufferDiary::parsing_applier_highlight;
	m_parsers[ LF_SPACE|LF_APPLY_LINK ] = &TextbufferDiary::parsing_applier_link;
	m_parsers[ LF_NUMBER|LF_APPLY_LINK_DATE ] = &TextbufferDiary::parsing_applier_link_date;

	m_parsers[ LF_NUMBER|LF_APPLY_NUMBER ] = &TextbufferDiary::parsing_applier_number;
	m_parsers[ LF_DOTYM|LF_APPLY_DOTYM ] = &TextbufferDiary::parsing_applier_dotym;
	m_parsers[ LF_DOTMD|LF_APPLY_DOTMD ] = &TextbufferDiary::parsing_applier_dotmd;

	m_parsers[ LF_CHECKBOX|LF_APPLY ] = &TextbufferDiary::parsing_junction_list;
	m_parsers[ LF_NEWLINE|LF_APPLY_CHECK_CAN ] = &TextbufferDiary::parsing_applier_check_can;
	m_parsers[ LF_NEWLINE|LF_APPLY_CHECK_UNF ] = &TextbufferDiary::parsing_applier_check_unf;
	m_parsers[ LF_NEWLINE|LF_APPLY_CHECK_FIN ] = &TextbufferDiary::parsing_applier_check_fin;
}

void
TextbufferDiary::handle_login( void )
{
	// SPELL CHECKING
	set_spellcheck( Diary::d->get_spellcheck() );
}

void
TextbufferDiary::handle_logout( void )
{
	m_flag_settextoperation = true;

	// reset search str ( do not call set_searchstr() because it reparse()s )
	m_searchstr = "";
	m_option_search = false;

	set_spellcheck( false );

	m_flag_settextoperation = false;
}

void
TextbufferDiary::set_searchstr( const Glib::ustring &searchstr )
{
	m_searchstr = searchstr;
	// disable search if searchstr is empty:
	m_option_search = searchstr.size() > 0;

	reparse();
}

bool
TextbufferDiary::select_searchstr_previous( void )
{
	if( ! m_option_search )
		return false;

	Gtk::TextIter iter_start, iter_end, iter_bound_start, iter_bound_end;
	get_selection_bounds( iter_bound_start, iter_bound_end );

	if( iter_bound_start.is_start() )
		return false;

	iter_bound_end = iter_bound_start;
	
	if( ! iter_bound_start.backward_to_tag_toggle( m_tag_match ) )
		return false;
	iter_bound_start.backward_to_tag_toggle( m_tag_match );
	iter_bound_start.backward_to_tag_toggle( m_tag_match );

	if( ! iter_bound_end.backward_search(	m_searchstr,
											Gtk::TextSearchFlags( 0 ),
											iter_start,
											iter_end,
											iter_bound_start ) )
		return false;

	select_range( iter_start, iter_end );
	m_ptr2textview->scroll_to( iter_start );
	return true;
}

bool
TextbufferDiary::select_searchstr_next( void )
{
	if( ! m_option_search )
		return false;

	Gtk::TextIter iter_start, iter_end, iter_bound_start, iter_bound_end;
	get_selection_bounds( iter_bound_start, iter_bound_end );

	if( iter_bound_end.is_end() )
		return false;

	iter_bound_start = iter_bound_end;

	if( ! iter_bound_end.forward_to_tag_toggle( m_tag_match ) )
		return false;
	iter_bound_end.forward_to_tag_toggle( m_tag_match );
	if( iter_bound_end.has_tag( m_tag_match ) )
		iter_bound_end.forward_to_tag_toggle( m_tag_match );

	if( ! iter_bound_start.forward_search(	m_searchstr,
													Gtk::TextSearchFlags( 0 ),
													iter_start,
													iter_end,
													iter_bound_end ) )
		return false;

	select_range( iter_start, iter_end );
	m_ptr2textview->scroll_to( iter_start );
	return true;
}

void
TextbufferDiary::parse( unsigned int pos_start, unsigned int pos_end )
{
	m_flag_parsing = true;

	// BEGINNING & END
	parser_iter_start = get_iter_at_offset( pos_start );
	parser_iter_current = parser_iter_start;
	parser_iter_word = parser_iter_start;
	Gtk::TextIter	const iter_end = get_iter_at_offset( pos_end );
	//PRINT_DEBUG( "text to be processed: " + get_text( parser_iter_start, iter_end ) );
	// remove all tags:
	remove_all_tags( parser_iter_start, iter_end );
	clear_links( pos_start, pos_end );

	parser_char_last = CC_NONE;
	Glib::ustring title( "" );
	parser_word_last = "";
	parser_int_last = 0;
	parser_date_link = 0;
	parser_lookingfor.clear();

	// SEARCHING
	gunichar		char_search = 0;
	Gtk::TextIter	iter_search = parser_iter_start;
	size_t			pos_search = 0;
	const size_t	pos_search_end = m_searchstr.size();


	if( pos_start == 0 )
		parser_lookingfor.push_back( LF_NEWLINE|LF_APPLY_HEADING );
	else
		parser_lookingfor.push_back( LF_NOTHING );

	for( ; ; ++parser_iter_current )
	{
		parser_char_current = parser_iter_current.get_char();
		// SEARCH WORD MATCHING
		// TODO: eliminate partial duplication of effort with diary filter functions
		if( m_option_search )
		{
			if( pos_search == pos_search_end )
			{
				apply_tag( m_tag_match, iter_search, parser_iter_current );
				pos_search = 0;
			}
			char_search = m_searchstr[ pos_search ];
			if( char_search == Glib::Unicode::tolower( parser_char_current ) )
			{
				if( pos_search == 0 )
					iter_search = parser_iter_current;
				pos_search++;
			}
			else
			{
				pos_search = 0;
			}
		}

		// MARKUP PARSING
		switch( parser_char_current )
		{
			case 0:		// treat end of the buffer like a line end
			case '\n':
			case '\r':
				process_char( LF_NEWLINE|LF_SPACE,
							  LF_NUM_SPCIM_CHK|LF_NONSPACE|LF_FORMATCHAR|LF_SLASH|LF_DOTDATE,
							  0, NULL,
							  CC_NEWLINE );
				break;
			case ' ':
				process_char( LF_SPACE|LF_SPACE_IMMEDIATE,
							  LF_NUMBER|LF_SLASH|LF_DOTDATE|LF_CHECKBOX,
							  LF_NOTHING, &TextbufferDiary::parsing_triggerer_subheading,
							  CC_SPACE );
				break;
			case '*':
				process_char( LF_NONSPACE|LF_ASTERISK,
							  LF_NUM_SPCIM_CHK|LF_SLASH|LF_DOTDATE,
							  LF_NOTHING, &TextbufferDiary::parsing_triggerer_bold,
							  CC_SIGN );
				break;
			case '_':
				process_char( LF_NONSPACE|LF_UNDERSCORE,
							  LF_NUM_SPCIM_CHK|LF_SLASH|LF_DOTDATE,
							  LF_NOTHING, &TextbufferDiary::parsing_triggerer_italic,
							  CC_SIGN );
				break;
			case '=':
				process_char( LF_NONSPACE|LF_EQUALS,
							  LF_NUM_SPCIM_CHK|LF_SLASH|LF_DOTDATE,
							  LF_NOTHING, &TextbufferDiary::parsing_triggerer_strikethrough,
							  CC_SIGN );
				break;
			case '#':
				process_char( LF_NONSPACE|LF_HASH,
							  LF_NUM_SPCIM_CHK|LF_SLASH|LF_DOTDATE,
							  LF_NOTHING, &TextbufferDiary::parsing_triggerer_highlight,
							  CC_SIGN );
				break;
			case '0': case '1': case '2': case '3': case '4':
			case '5': case '6': case '7': case '8': case '9':
				process_char( LF_NONSPACE|LF_NUMBER,
							  LF_SPACE_IMMEDIATE|LF_SLASH|LF_DOTDATE|LF_CHECKBOX,
							  LF_NOTHING, &TextbufferDiary::parsing_triggerer_link_date,
							  CC_NUMBER );
				break;
			case '.':
				process_char( LF_NONSPACE|LF_DOTDATE,
							  LF_NUM_SPCIM_CHK|LF_SLASH,
							  0, NULL,
							  CC_SIGN );
				break;
			case '/':
				process_char( LF_NONSPACE|LF_SLASH|LF_DOTDATE,
							  LF_NUM_SPCIM_CHK,
							  0, NULL,
							  CC_SIGN );
				break;
			case ':':
				process_char( LF_NONSPACE,
							  LF_NUM_SPCIM_CHK|LF_SLASH|LF_DOTDATE,
							  LF_NOTHING, &TextbufferDiary::parsing_triggerer_link,
							  CC_SIGN );
				break;
			case '@':
				process_char( LF_NONSPACE|LF_AT,
							  LF_NUM_SPCIM_CHK|LF_SLASH|LF_DOTDATE,
							  LF_NOTHING, &TextbufferDiary::parsing_triggerer_link_at,
							  CC_SIGN );
				break;
			case '\t':
				process_char( LF_TAB|LF_SPACE,
							  LF_NUM_SPCIM_SLH|LF_NONSPACE|LF_DOTDATE,
							  LF_NOTHING, &TextbufferDiary::parsing_triggerer_list,
							  CC_TAB );
				break;
			// LIST CHARS
			case L'☐':
			case L'☑':
			case L'☒':
				process_char( LF_CHECKBOX,
							  LF_NUMBER|LF_SPACE_IMMEDIATE|LF_SLASH|LF_DOTDATE,
							  0, NULL,
							  CC_SIGN );
				break;
			default:
				process_char( LF_NONSPACE,
							  LF_NUMBER|LF_DOTDATE|LF_SLASH|LF_CHECKBOX,
							  0, NULL,
							  CC_ALPHA );	// most probably :)
				break;
		}
		// this is here instead of being in the for() to...
		// ...continue for one more cycle after iter_end is reached:
		if( parser_iter_current == iter_end )
			break;
    }

	m_flag_parsing = false;
}

// HELPER FUNCTIONS FOR PARSING
void
TextbufferDiary::parsing_triggerer_subheading( void )
{
	if( parser_char_last == CC_NEWLINE )
	{
		parser_lookingfor = m_plan_subheading;
		parser_iter_start = parser_iter_current;
	}
}

void
TextbufferDiary::parsing_triggerer_bold( void )
{
	parser_lookingfor = m_plan_bold;
	parser_iter_start = parser_iter_current;
}

void
TextbufferDiary::parsing_triggerer_italic( void )
{
	parser_lookingfor.clear();
	parser_lookingfor.push_back( LF_NONSPACE );
	parser_lookingfor.push_back( LF_UNDERSCORE|LF_APPLY_MARKUP );
	parser_iter_start = parser_iter_current;
}

void
TextbufferDiary::parsing_triggerer_strikethrough( void )
{
	parser_lookingfor.clear();
	parser_lookingfor.push_back( LF_NONSPACE );
	parser_lookingfor.push_back( LF_EQUALS|LF_APPLY_MARKUP );
	parser_iter_start = parser_iter_current;
}

void
TextbufferDiary::parsing_triggerer_highlight( void )
{
	parser_lookingfor.clear();
	parser_lookingfor.push_back( LF_NONSPACE );
	parser_lookingfor.push_back( LF_HASH|LF_APPLY_MARKUP );
	parser_iter_start = parser_iter_current;
}

void
TextbufferDiary::parsing_triggerer_link( void )
{
	PRINT_DEBUG( "word_last: " + parser_word_last );
	if( parser_word_last == "http" || parser_word_last == "https" ||
	    parser_word_last == "ftp" || parser_word_last == "file" )
	{
		parser_lookingfor.clear();
		parser_lookingfor.push_back( LF_SLASH );
		parser_lookingfor.push_back( LF_SLASH );
		// should we really disable relative links?
		if(	parser_word_last == "file" )
			parser_lookingfor.push_back( LF_SLASH );
		parser_lookingfor.push_back( LF_NONSPACE );
		parser_lookingfor.push_back( LF_SPACE|LF_APPLY_LINK );
		parser_iter_start = parser_iter_word;
	}
	else
	if( parser_word_last == "mailto" )
	{
		parser_lookingfor.clear();
		parser_lookingfor.push_back( LF_NONSPACE );
		parser_lookingfor.push_back( LF_AT );
		parser_lookingfor.push_back( LF_NONSPACE );
		parser_lookingfor.push_back( LF_SPACE|LF_APPLY_LINK );
		parser_iter_start = parser_iter_word;
	}
}

void
TextbufferDiary::parsing_triggerer_link_at( void )
{
	PRINT_DEBUG( "word_last [@]: " + parser_word_last );
	if( parser_char_last & CC_SEPARATOR )
		return;

	parser_word_last.insert( 0, "mailto:" );
	parser_lookingfor.clear();
	parser_lookingfor.push_back( LF_NONSPACE );
	parser_lookingfor.push_back( LF_SPACE|LF_APPLY_LINK );
	parser_iter_start = parser_iter_word;
}

void
TextbufferDiary::parsing_triggerer_link_date( void )
{
	parser_int_last = ( parser_char_current - '0' );
	parser_lookingfor = m_plan_date;
	parser_iter_start = parser_iter_current;
}

void
TextbufferDiary::parsing_triggerer_list( void )
{
	if( parser_char_last != CC_NEWLINE )
		return;

	parser_lookingfor.clear();
	parser_lookingfor.push_back( LF_CHECKBOX|LF_APPLY );
}

// PARSING APPLIERS
void
TextbufferDiary::parsing_applier_heading( void )
{
	apply_tag( m_tag_heading, parser_iter_start, parser_iter_current );

	if( ! m_flag_settextoperation )
	{
		// TODO: what about storing title in entry for faster later use
		Lifeobase::base->handle_entry_title_changed(
				m_ptr2entry,
				( parser_char_current == 0 && parser_iter_current.get_offset() == 0 ) ?
						_( "<empty entry>" )	:
						get_slice( parser_iter_start, parser_iter_current ) );
	}
	parser_lookingfor.front() = LF_NOTHING;
}

void
TextbufferDiary::parsing_applier_subheading( void )
{
	apply_tag( m_tag_subheading, parser_iter_start, parser_iter_current );
	parser_lookingfor.front() = LF_NOTHING;
}

void
TextbufferDiary::parsing_applier_bold( void )
{
	apply_markup( m_tag_bold );
	parser_lookingfor.front() = LF_NOTHING;
}

void
TextbufferDiary::parsing_applier_italic( void )
{
	apply_markup( m_tag_italic );
	parser_lookingfor.front() = LF_NOTHING;
}

void
TextbufferDiary::parsing_applier_strikethrough( void )
{
	apply_markup( m_tag_strikethrough );
	parser_lookingfor.front() = LF_NOTHING;
}

void
TextbufferDiary::parsing_applier_highlight( void )
{
	apply_markup( m_tag_highlight );
	parser_lookingfor.front() = LF_NOTHING;
}

void
TextbufferDiary::parsing_applier_link( void )
{
	PRINT_DEBUG( "url: " + parser_word_last );
	Gtk::TextIter iter_end = parser_iter_current;
	iter_end--;
	m_list_links.push_back( new LinkUri( create_mark( parser_iter_start ),
										 create_mark( iter_end ),
										 parser_word_last ) );

	apply_tag( m_tag_link, parser_iter_start, parser_iter_current );
	parser_lookingfor.front() = LF_NOTHING;
}

void
TextbufferDiary::parsing_applier_link_date( void )
{
	parsing_applier_number();
	parser_date_link.set_day( parser_int_last );

	if( parser_date_link.is_valid() )
	{
		LinkStatus status;
		Entry *ptr2entry = Diary::d->get_entry( parser_date_link );
		if( ptr2entry == NULL )
			status = LS_ENTRY_UNAVAILABLE;
		else
		if( parser_date_link.get_pure() == m_ptr2entry->get_date().get_pure() )
			status = Diary::d->get_day_has_multiple_entries( parser_date_link ) ? LS_OK : LS_CYCLIC;
		else
			status = LS_OK;

		if( status < LS_INVALID )
		{
			Gtk::TextIter iter_tagend = parser_iter_current;
			++iter_tagend;
			m_list_links.push_back(
					new LinkEntry(	create_mark( parser_iter_start ),
									create_mark( parser_iter_current ),
									parser_date_link ) );

			if( status == LS_OK )
				apply_tag( m_tag_link, parser_iter_start, iter_tagend );
			else
			if( status == LS_ENTRY_UNAVAILABLE )
				apply_tag(	m_tag_link_broken, parser_iter_start, iter_tagend );
		}
	}

	parser_lookingfor.front() = LF_NOTHING;
}

void
TextbufferDiary::parsing_applier_number( void )
{
	if( parser_char_last == CC_NUMBER )
	{
		parser_int_last *= 10;
		parser_int_last += ( parser_char_current - '0' );
	}
	else
		parser_int_last = ( parser_char_current - '0' );
	parser_lookingfor.pop_front();
}

void
TextbufferDiary::parsing_applier_dotym( void )
{
	parser_date_link.set_year( parser_int_last );
	parser_lookingfor.pop_front();
}

void
TextbufferDiary::parsing_applier_dotmd( void )
{
	parser_date_link.set_month( parser_int_last );
	parser_lookingfor.pop_front();
}

void
TextbufferDiary::parsing_junction_list( void )
{
	parser_iter_start = parser_iter_current;

	parser_lookingfor.front() = LF_NONSPACE;

	switch( parser_char_current )
	{
		case L'☐':
			parser_lookingfor.push_back( LF_NEWLINE|LF_APPLY_CHECK_UNF );
			break;
		case L'☑':
			parser_lookingfor.push_back( LF_NEWLINE|LF_APPLY_CHECK_FIN );
			break;
		case L'☒':
			parser_lookingfor.push_back( LF_NEWLINE|LF_APPLY_CHECK_CAN );
			break;
		//default: // should never occur
	}
}

void
TextbufferDiary::parsing_applier_check_unf( void )
{
	Gtk::TextIter iter2 = parser_iter_start;
	iter2++;
	m_list_links.push_back( new LinkCheck( create_mark( parser_iter_start ),
										   create_mark( iter2 ),
										   0 ) );
	apply_tag( m_tag_checkbox, parser_iter_start, iter2 );
	parser_lookingfor.front() = LF_NOTHING;
}

void
TextbufferDiary::parsing_applier_check_fin( void )
{
	Gtk::TextIter iter2 = parser_iter_start;
	iter2++;
	m_list_links.push_back( new LinkCheck( create_mark( parser_iter_start ),
										   create_mark( iter2 ),
										   1 ) );
	apply_tag( m_tag_checkbox, parser_iter_start, iter2 );
	apply_tag( m_tag_highlight, iter2, parser_iter_current );
	parser_lookingfor.front() = LF_NOTHING;
}

void
TextbufferDiary::parsing_applier_check_can( void )
{
	Gtk::TextIter iter2 = parser_iter_start;
	iter2++;
	m_list_links.push_back( new LinkCheck( create_mark( parser_iter_start ),
										   create_mark( iter2 ),
										   2 ) );
	apply_tag( m_tag_checkbox, parser_iter_start, iter2 );
	apply_tag( m_tag_strikethrough, iter2, parser_iter_current );
	parser_lookingfor.front() = LF_NOTHING;
}


//#define CALL_MEMBER_FN(object,ptrToMember)  ((object).*(ptrToMember))

inline void
TextbufferDiary::process_char( unsigned int targeted, unsigned int broken,
			  unsigned int triggering, ParsingApplierFn parsing_triggerer,
			  CharClass cc )
{
	unsigned int lf = parser_lookingfor.front();
	if( lf & targeted )
	{
		if( lf & LF_APPLY )
		{
			ParsingApplierFn f = m_parsers.find( parser_lookingfor.front() )->second;
			( this->*f )();
		}
		else // appliers handle converting current LookingFor item
		{
			parser_lookingfor.pop_front();
			if( lf & triggering )
			{
				( this->*parsing_triggerer )();
			}
		}
	}
	else
	if( lf & broken )
	{
		parser_lookingfor.clear();
		parser_lookingfor.push_back( LF_NOTHING );
	}
	else
	if( lf & triggering )
	{
		( this->*parsing_triggerer )();
	}

	// SET NEW CHAR CLASS & ADJUST WORD_LAST ACOORDINGLY
	if( cc & CC_SEPARATOR )
		parser_word_last.clear();
	else
	{
		if( parser_word_last.empty() )
			parser_iter_word = parser_iter_current;
		parser_word_last += parser_char_current;
	}
	parser_char_last = cc;
}

inline void
TextbufferDiary::apply_markup( const Glib::RefPtr<Tag> &tag)
{
	if( parser_char_last & CC_SEPARATOR )
		return;

	Gtk::TextIter iter2 = parser_iter_start;
	iter2++;
	apply_tag( m_tag_markup, parser_iter_start, iter2 );
	apply_tag( tag, iter2, parser_iter_current );
	iter2 = parser_iter_current;
	iter2++;
	apply_tag( m_tag_markup, parser_iter_current, iter2 );
}


void
TextbufferDiary::set_richtext( Entry *entry )
{
	m_flag_settextoperation = true;
	clear_links();
	UndoManager::m->clear();
	UndoManager::m->freeze();
	m_ptr2entry = entry;
	set_theme( entry->get_theme() );
	Gtk::TextBuffer::set_text( entry->get_text() );
	place_cursor( begin() );
	UndoManager::m->thaw();
	m_flag_settextoperation = false;
}

void
TextbufferDiary::reparse( void )
{
	place_cursor( begin() );
	m_flag_settextoperation = true;
	parse( 0, get_char_count() );
	m_flag_settextoperation = false;
}

void
TextbufferDiary::on_insert(	const Gtk::TextIter& iterator,
							const Glib::ustring& text,
							int bytes )
{
	const Glib::ustring::size_type offset_itr = iterator.get_offset();

	// UNDO
	if( ! UndoManager::m->is_freezed() )
	{
		UndoInsert * undo_insert = new UndoInsert( offset_itr, text );
		UndoManager::m->add_action( undo_insert );
	}
	else
	{
		Gtk::TextIter iter_scroll( iterator );	// to remove constness
		m_ptr2textview->scroll_to( iter_scroll );
	}

	Gtk::TextBuffer::on_insert( iterator, text, text.bytes() );

	if( m_flag_ongoingoperation )
		return;

	// PARSING LIMITS
	Glib::ustring::size_type pos_start = 0;
	Glib::ustring::size_type pos_end = get_char_count();
	if( m_flag_settextoperation == false )
	{
		if( offset_itr > 0 )
		{
			pos_start = get_text().rfind( '\n', offset_itr - 1 );
			if( pos_start == Glib::ustring::npos )
				pos_start = 0;
		}

		if( offset_itr < pos_end )
		{
			pos_end = get_text().find( '\n', offset_itr + 1 );
			if( pos_end == Glib::ustring::npos )
				pos_end = get_char_count();
		}
	}

	parse( pos_start, pos_end );
}

void
TextbufferDiary::process_space( void )
{
	Gtk::TextIter iter_end = get_iter_at_mark( get_insert() );
	const Glib::ustring::size_type	pos_cursor = iter_end.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list == Glib::ustring::npos )
		return;
	++pos_list;	// get rid of the new line char

	Gtk::TextIter iter_start = get_iter_at_offset( pos_list );
	Glib::ustring line = get_text( iter_start, iter_end );
	char lookingfor = '\t';
	unsigned int size = pos_cursor - pos_list;

	for( unsigned int i = 0; i < size; i++ )
	{
		switch( line[ i ] )
		{
			case '\t':
				if( lookingfor == '\t' )
					lookingfor = 'A';	// any list char like [ or *
				else
				if( lookingfor != 'A' )	// multiple tabs are possible (indentation)
					return;
				pos_list++;	// indentation level
				break;
			case '[':
				if( lookingfor != 'A' )
					return;
				lookingfor = ']';
				break;
			case ']':
				if( lookingfor != ']' )
					return;
				//lookingfor = 'C';	// checkbox
				PRINT_DEBUG( "time to insert a checkbox" );
				iter_start = get_iter_at_offset( pos_list );	// get rid of the tab chars
				m_flag_ongoingoperation = true;
				erase( iter_start, iter_end );
				m_flag_ongoingoperation = false;
				iter_start = get_iter_at_offset( pos_list );	// refresh the iterator
				insert( iter_start, "☐" );
				break;
			case '*':
				if( lookingfor != 'A' )
					return;
				iter_start = get_iter_at_offset( pos_list );	// get rid of the tab chars
				m_flag_ongoingoperation = true;
				erase( iter_start, iter_end );
				m_flag_ongoingoperation = false;
				iter_start = get_iter_at_offset( pos_list );	// refresh the iterator
				insert( iter_start, "•" );
				break;
			default:
				return;
		}
	}
}

bool
TextbufferDiary::process_new_line( void )
{
	Gtk::TextIter iter_end = get_iter_at_mark( get_insert() );
	const Glib::ustring::size_type	pos_cursor = iter_end.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list != Glib::ustring::npos &&
		pos_list > 0 &&					// previous line should not be the first
		pos_cursor - pos_list > 3 )		// previous line should be at least 3 chars long
	{
		++pos_list;	// get rid of the new line char
		Gtk::TextIter iter_start = get_iter_at_offset( pos_list );
		Glib::ustring line = get_text( iter_start, iter_end );

		if( line[ 0 ] == '\t' )
		{
			Glib::ustring text( "\n\t" );
			int value = 0;
			char lookingfor = '*';

			for( unsigned int i = 1; i < pos_cursor - pos_list; i++ )
			{
				switch( line[ i ] )
				{
					// BULLETED LIST
					case L'•':
						if( lookingfor != '*' )
							return false;
						lookingfor = ' ';
						text += "• ";
						break;
					// CHECK LIST
					case L'☐':
					case L'☑':
					case L'☒':
						if( lookingfor != '*' )
							return false;
						lookingfor = ' ';
						text += "☐ ";
						break;
					// NUMBERED LIST
					case '0': case '1': case '2': case '3': case '4':
					case '5': case '6': case '7': case '8': case '9':
						if( lookingfor != '*' && lookingfor != '1' )
							return false;
						lookingfor = '1';
						value *= 10;
						value += line[ i ] - '0';
						break;
					case '-':
					case '.':
					case ')':
						if( lookingfor != '1' )
							return false;
						lookingfor = ' ';
						text += Glib::ustring::compose( "%1%2 ", ++value, line.substr( i, 1 ) );
						break;
					case '\t':
						if( lookingfor != '*' )
							return false;
						text += '\t';
						break;
					case ' ':
						if( lookingfor != ' ' )
							return false;
						if( i == pos_cursor - pos_list - 1 )
						{
							erase( iter_start, iter_end );
							iter_start = get_iter_at_offset( pos_list );
							insert( iter_start, "\n" );
							return true;
						}
						else
						{
							insert( iter_end, text );
							return true;
						}
						break;
					default:
						return false;
				}
			}
		}
	}
	return false;
}

void
TextbufferDiary::on_erase(	const Gtk::TextIter& iter_start,
							const Gtk::TextIter& iter_end )
{
	if( ! UndoManager::m->is_freezed() )
	{
		UndoErase * undo_erase = new UndoErase(
				iter_start.get_offset(), get_slice( iter_start, iter_end ) );
		UndoManager::m->add_action( undo_erase );
	}
	else
	{
		Gtk::TextIter iter_scroll( iter_start );	// to remove constness
		m_ptr2textview->scroll_to( iter_scroll );
	}

	Gtk::TextBuffer::on_erase( iter_start, iter_end );

	// set_text() calls on_erase too:
	if( m_flag_settextoperation == false && m_flag_ongoingoperation == false )
	{
		Glib::ustring::size_type pos_start = iter_start.get_offset();
		Glib::ustring::size_type pos_end = iter_end.get_offset();

		if( pos_start > 0 )
		{
			pos_start = get_text().rfind( '\n', pos_start - 1 );
			if( pos_start == Glib::ustring::npos )
				pos_start = 0;
		}

		pos_end = get_text().find( '\n', pos_end );
		if( pos_end == Glib::ustring::npos )
			pos_end = get_char_count();

		parse( pos_start, pos_end );
	}
}

void
TextbufferDiary::on_apply_tag(	const Glib::RefPtr< TextBuffer::Tag >& tag,
								const Gtk::TextIter& iter_begin,
								const Gtk::TextIter& iter_end )
{
	// do not check spelling of links:
	if( iter_begin.has_tag( m_tag_link ) && tag->property_name() == "gtkspell-misspelled" )
		return;
	else
		Gtk::TextBuffer::on_apply_tag( tag, iter_begin, iter_end );
}

void
TextbufferDiary::on_remove_tag(	const Glib::RefPtr< TextBuffer::Tag >& tag,
								const Gtk::TextIter& iter_begin,
								const Gtk::TextIter& iter_end )
{
	// do not remove gtkspell tags while parsing:
	if( m_flag_parsing && tag->property_name() == "gtkspell-misspelled" )
		return;
	else
		Gtk::TextBuffer::on_remove_tag( tag, iter_begin, iter_end );
}

Link*
TextbufferDiary::get_link( int offset ) const
{
	for (	ListLinks::const_iterator iter = m_list_links.begin();
			iter != m_list_links.end();
			++iter )
	{
		if(	offset >= ( *iter )->m_mark_start->get_iter().get_offset() &&
			offset <= ( *iter )->m_mark_end->get_iter().get_offset() )
			return( *iter );
	}

	return NULL;
}

void
TextbufferDiary::clear_links( int pos_start, int pos_end )
{
	ListLinks::iterator iter_tmp;
	for(	ListLinks::iterator iter = m_list_links.begin();
			iter != m_list_links.end(); )
		if(	pos_start <= ( *iter )->m_mark_start->get_iter().get_offset() &&
			pos_end >= ( *iter )->m_mark_end->get_iter().get_offset() )
		{
			iter_tmp = iter;
			++iter;
			delete( *iter_tmp );
			m_list_links.erase( iter_tmp );
		}
		else
			++iter;
}

void
TextbufferDiary::clear_links( void )
{
	for(	ListLinks::iterator iter = m_list_links.begin();
			iter != m_list_links.end();
			++iter )
	{
		delete( *iter );
	}
	m_list_links.clear();
}

void
TextbufferDiary::handle_menu( Gtk::Menu *menu )
{
	Gtk::MenuItem *menuitem_format = Gtk::manage(
			new Gtk::MenuItem( _( "Format" ) ) );

	Gtk::CheckMenuItem *menuitem_spell = Gtk::manage(
			new Gtk::CheckMenuItem( _( "Check Spelling" ) ) );

	Gtk::SeparatorMenuItem *separator = Gtk::manage( new Gtk::SeparatorMenuItem );

	menuitem_spell->set_active( Diary::d->get_spellcheck() );

	menuitem_spell->signal_toggled().connect(
			sigc::bind(	sigc::mem_fun( *this, &TextbufferDiary::set_spellcheck ),
						! Diary::d->get_spellcheck() ) );

	menuitem_format->show();
	menuitem_spell->show();
	separator->show();

	// FORMATTING
	Gtk::Menu *menu_format = new Gtk::Menu;
	
	Gtk::MenuItem *menuitem_bold = create_menuitem_markup(
			Glib::ustring::compose( "*<b>%1</b>*", _( "Bold" ) ),
			sigc::mem_fun( *this, &TextbufferDiary::toggle_bold ) );
	Gtk::MenuItem *menuitem_italic = create_menuitem_markup(
			Glib::ustring::compose( "_<i>%1</i>_", _( "Italic" ) ),
			sigc::mem_fun( *this, &TextbufferDiary::toggle_italic ) );
	Gtk::MenuItem *menuitem_strikethrough = create_menuitem_markup(
			Glib::ustring::compose( "=<s>%1</s>=", _( "Strikeout" ) ),
			sigc::mem_fun( *this, &TextbufferDiary::toggle_strikethrough ) );
	Gtk::MenuItem *menuitem_highlight = create_menuitem_markup(
			Glib::ustring::compose(	"#<span foreground=\"%2\" background=\"%3\">%1</span>#",
					_( "Highlight" ),
					convert_gdkcolor_to_html(
							m_ptr2textview->get_style()->get_text( Gtk::STATE_NORMAL ) ),
					convert_gdkcolor_to_html( m_tag_highlight->property_background_gdk() ) ),
			sigc::mem_fun( *this, &TextbufferDiary::toggle_highlight ) );

	Gtk::MenuItem *separator1 = Gtk::manage( new Gtk::SeparatorMenuItem );
	Gtk::CheckMenuItem *menuitem_hide_markup = Gtk::manage(
			new Gtk::CheckMenuItem( _( "Hide Markup Characters" ) ) );

	menuitem_hide_markup->set_active( m_tag_markup->property_invisible() );

	menuitem_hide_markup->signal_toggled().connect(
			sigc::mem_fun( *this, &TextbufferDiary::toggle_hide_markup ) );

	menu_format->append( *menuitem_bold );
	menu_format->append( *menuitem_italic );
	menu_format->append( *menuitem_strikethrough );
	menu_format->append( *menuitem_highlight );
	menu_format->append( *separator1 );
	menu_format->append( *menuitem_hide_markup );
	menu_format->show_all_children();

	menuitem_format->set_submenu( *menu_format );

	menu->prepend( *separator );
	menu->prepend( *menuitem_spell );
	menu->prepend( *menuitem_format );
}

void
TextbufferDiary::toggle_format(	Glib::RefPtr< Tag > tag, const Glib::ustring &markup )
{
	Gtk::TextIter iter_start, iter_end;
	if( get_has_selection() )
	{
		get_selection_bounds( iter_start, iter_end );
		iter_end--;
		int pos_start = -1, pos_end = -1;

		for( ; ; iter_start++ )
		{
			if( iter_start.has_tag( tag ) )
				return;
			switch( iter_start.get_char() )
			{
				// do nothing if selection spreads over more than one line:
				case '\n':
					return;
				case ' ':
				case '\t':
					break;
				case '*':
				case '_':
				case '#':
				case '=':
					if( iter_start.get_char() == markup[ 0 ] )
						break;
				default:
					if( pos_start < 0 )
						pos_start = iter_start.get_offset();
					pos_end = iter_start.get_offset();
					break;
			}
			if( iter_start == iter_end )
				break;
		}
		// add markup chars to the beginning and end:
		if( pos_start >= 0 )
		{
			pos_end += 2;
			insert( get_iter_at_offset( pos_start ), markup );
			insert( get_iter_at_offset( pos_end ), markup );
			place_cursor( get_iter_at_offset( pos_end ) );
		}
	}
	else	// no selection case
	{
		Glib::RefPtr< Gtk::TextMark > mark = get_insert();
		iter_start = mark->get_iter();
		if( Glib::Unicode::isspace( iter_start.get_char() ) ||
			iter_start.has_tag( TextbufferDiary::m_tag_markup ) )
		{
			if( ! iter_start.starts_line() )
				iter_start--;
			else
				return;
		}
		if( iter_start.has_tag( tag ) )
		{
			m_flag_ongoingoperation = true;

			// find start and end points of formatting:
			if( iter_start.starts_word() )
				iter_start++;	// necessary when cursor is between a space char
								// and non-space char
			iter_start.backward_to_tag_toggle( tag );
			backspace( iter_start );

			iter_end = mark->get_iter();
			if( iter_end.has_tag( tag ) )
				iter_end.forward_to_tag_toggle( TextbufferDiary::m_tag_markup );

			m_flag_ongoingoperation = false;

			backspace( ++iter_end );
		}
		else
		if( iter_start.get_tags().empty() )		// nested tags are not supported atm
		{
			// find word boundaries:
			if( !( iter_start.starts_word() || iter_start.starts_line() ) )
				iter_start.backward_word_start();
			insert( iter_start, markup );

			iter_end = mark->get_iter();
			if( !( iter_end.ends_word() || iter_end.ends_line() ) )
			{
				iter_end.forward_word_end();
				insert( iter_end, markup );
			}
			else
			{
				int offset = iter_end.get_offset();
				insert( iter_end, markup );
				place_cursor( get_iter_at_offset( offset ) );

			}
		}
	}

}

void
TextbufferDiary::toggle_bold( void )
{
	toggle_format( m_tag_bold, "*" );
}

void
TextbufferDiary::toggle_italic( void )
{
	toggle_format( m_tag_italic, "_" );
}

void
TextbufferDiary::toggle_strikethrough( void )
{
	toggle_format( m_tag_strikethrough, "=" );
}

void
TextbufferDiary::toggle_highlight( void )
{
	toggle_format( m_tag_highlight, "#" );
}

void
TextbufferDiary::toggle_hide_markup( void )
{
	m_tag_markup->property_invisible() = ! m_tag_markup->property_invisible();
}

void
TextbufferDiary::handle_indent( void )
{
	Gtk::TextIter iter = get_iter_at_mark( get_insert() );
	const Glib::ustring::size_type	pos_cursor = iter.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list == Glib::ustring::npos )
		return;
	++pos_list;	// get rid of the new line char

	iter = get_iter_at_offset( pos_list );
	insert( iter, "\t" );
}

void
TextbufferDiary::handle_unindent( void )
{
	Gtk::TextIter iter = get_iter_at_mark( get_insert() );
	const Glib::ustring::size_type	pos_cursor = iter.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list == Glib::ustring::npos || ( pos_cursor - pos_list ) < 2 )
		return;
	++pos_list;	// get rid of the new line char

	iter = get_iter_at_offset( pos_list );
	if( iter.get_char() != '\t' )
		return;
	Gtk::TextIter iter_end = get_iter_at_offset( pos_list + 1 );
	erase( iter, iter_end );
}

void
TextbufferDiary::add_bullet( void )
{
	Gtk::TextIter iter = get_iter_at_mark( get_insert() );
	if( iter.get_line() < 1 )	// no list in the first line
		return;
	const Glib::ustring::size_type	pos_cursor = iter.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list == Glib::ustring::npos )
		return;
	++pos_list;	// get rid of the new line char

	iter = get_iter_at_offset( pos_list );
	Gtk::TextIter iter_erase_begin( iter );
	char lookingfor( 't' );	// tab
	while( true )
	{
		switch( iter.get_char() )
		{
			case L'•':	// remove bullet
				if( lookingfor == 'b' )
					lookingfor = 's';	// space
				else
					return;
				break;
			case ' ':
				if( lookingfor == 's' )
					erase( iter_erase_begin, ++iter );
				return;
			case L'☐':
			case L'☑':
			case L'☒':
				// TODO: convert to bullet
				return;
			case '\t':
				lookingfor = 'b';	// bullet
				iter_erase_begin = iter;
				break;
			case 0:	// end
			default:
				if( lookingfor != 's' )
					insert( iter, "\t• " );
				return;
		}
		++iter;
	}
}

void
TextbufferDiary::add_checkbox( void )
{
		Gtk::TextIter iter = get_iter_at_mark( get_insert() );
	if( iter.get_line() < 1 )	// no list in the first line
		return;
	const Glib::ustring::size_type	pos_cursor = iter.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list == Glib::ustring::npos )
		return;
	++pos_list;	// get rid of the new line char

	iter = get_iter_at_offset( pos_list );
	Gtk::TextIter iter_erase_begin( iter );
	char lookingfor( 't' );	// tab
	while( true )
	{
		switch( iter.get_char() )
		{
			case L'☐':
			case L'☑':
			case L'☒':	// remove checkbox
				if( lookingfor == 'c' )
					lookingfor = 's';	// space
				else
					return;
				break;
			case ' ':
				if( lookingfor == 's' )
					erase( iter_erase_begin, ++iter );
				return;
			case L'•':
				// TODO: convert to checkbox
				return;
			case '\t':
				lookingfor = 'c';	// checkbox
				iter_erase_begin = iter;
				break;
			case 0:	// end
			default:
				if( lookingfor != 's' )
					insert( iter, "\t☐ " );
				return;
		}
		++iter;
	}
}

void
TextbufferDiary::set_spellcheck( bool option_spell )
{
	if( option_spell )
	{
		if( ! m_ptr2spellobj )
			m_ptr2spellobj = gtkspell_new_attach( m_ptr2textview->gobj(), NULL, NULL );
	}
	else
	if( m_ptr2spellobj )
	{
		gtkspell_detach( m_ptr2spellobj );
		m_ptr2spellobj = NULL;
	}

	Diary::d->set_spellcheck( option_spell );
}

void
TextbufferDiary::set_theme( const Theme *theme0 )
{
	const Theme *theme = ( bool( theme0 ) ? theme0 : Diary::d->get_default_theme() );

	m_ptr2textview->modify_font( theme->font );
	m_ptr2textview->modify_base( Gtk::STATE_NORMAL, theme->color_base );
	m_ptr2textview->modify_text( Gtk::STATE_NORMAL, theme->color_text );
	m_tag_heading->property_foreground_gdk() = theme->color_heading;
	m_tag_subheading->property_foreground_gdk() = theme->color_subheading;
	m_tag_highlight->property_background_gdk() = theme->color_highlight;

	m_tag_match->property_background_gdk() = contrast(
			theme->color_base, m_theme_color_match );
	m_tag_markup->property_foreground_gdk() = midtone(
			theme->color_base, theme->color_text );
	m_tag_link->property_foreground_gdk() = contrast(
			theme->color_base, m_theme_color_link );
	m_tag_link_broken->property_foreground_gdk() = contrast(
			theme->color_base, m_theme_color_linkbroken );
}

// TEXTVIEW ========================================================================================
// STATIC MEMBERS
TextbufferDiary			*TextviewDiary::m_buffer;

TextviewDiary::TextviewDiary( BaseObjectType *cobject, const Glib::RefPtr<Gtk::Builder>& )
	:	Gtk::TextView( cobject ),
	m_cursor_hand( Gdk::HAND2 ), m_cursor_xterm( Gdk::XTERM ),
	m_ptr2cursor_last( &m_cursor_xterm ), m_link_hovered( NULL )
{
	set_buffer( static_cast< Glib::RefPtr< TextbufferDiary > >( m_buffer ) );
	m_buffer->set_textview( this );
	set_wrap_mode( Gtk::WRAP_WORD );
	set_left_margin( 10 );

	signal_populate_popup().connect(
			sigc::mem_fun( *m_buffer, &TextbufferDiary::handle_menu ) );
}

inline void
TextviewDiary::update_link( void )
{
	Gtk::TextIter		iter;
	const Gdk::Cursor	*ptr2cursor = &m_cursor_xterm;
	int					pointer_x, pointer_y;
	int					buffer_x, buffer_y;
    Gdk::ModifierType	modifiers;

    Gtk::Widget::get_window()->get_pointer( pointer_x, pointer_y, modifiers );
	window_to_buffer_coords(	Gtk::TEXT_WINDOW_WIDGET,
								pointer_x, pointer_y,
								buffer_x, buffer_y );
	get_iter_at_location( iter, buffer_x, buffer_y );
	m_link_hovered = m_buffer->get_link( iter.get_offset() );

	if( m_link_hovered != NULL )
	{
		if( !( modifiers & Gdk::CONTROL_MASK ) )
			ptr2cursor = &m_cursor_hand;
	}

	if( ptr2cursor != m_ptr2cursor_last )
	{
		m_ptr2cursor_last = ptr2cursor;
		get_window( Gtk::TEXT_WINDOW_TEXT )->set_cursor( *ptr2cursor );
	}
}

bool
TextviewDiary::on_motion_notify_event( GdkEventMotion *event )
{
	update_link();
	return Gtk::TextView::on_motion_notify_event( event );
}

bool
TextviewDiary::on_button_release_event( GdkEventButton *event )
{
	if( m_link_hovered != NULL )
		if( ( event->state & Gdk::CONTROL_MASK ) != Gdk::CONTROL_MASK )
			m_link_hovered->go();

	return Gtk::TextView::on_button_release_event( event );
}

bool
TextviewDiary::on_key_press_event( GdkEventKey *event )
{
	if( ( event->state & Gdk::CONTROL_MASK ) == Gdk::CONTROL_MASK )
	{
		switch( event->keyval )
		{
			case GDK_b:
				m_buffer->toggle_bold();
				break;
			case GDK_i:
				m_buffer->toggle_italic();
				break;
			case GDK_s:
				m_buffer->toggle_strikethrough();
				break;
			case GDK_h:
				m_buffer->toggle_highlight();
				break;
			case GDK_t:
				Lifeobase::base->focus_tag();
				break;
		}
	}
	else
	if( ( event->state & Gdk::MOD1_MASK ) == Gdk::MOD1_MASK )
	{
		switch( event->keyval )
		{
			case GDK_i:
				m_buffer->handle_indent();
				break;
			case GDK_u:
				m_buffer->handle_unindent();
				break;
		}
	}
	else
	if( event->state == 0 )
	{
		switch( event->keyval )
		{
			case GDK_space:
				m_buffer->process_space();
				break;
			case GDK_Return:
				if( m_buffer->process_new_line() )
					return true;
				break;
			case GDK_Control_L:
			case GDK_Control_R:
				if( m_link_hovered )
					update_link();
				break;
		}
	}

	return Gtk::TextView::on_key_press_event( event );
}

//void
//TextviewDiary::on_style_changed( const Glib::RefPtr< Gtk::Style > &style_prev )
//{
//	Gtk::TextView::on_style_changed( style_prev );
////	TextbufferDiary = get_pango_context()->get_font_description();
//	TextbufferDiary::m_theme_font = get_style()->get_font();
//}

bool
TextviewDiary::on_key_release_event( GdkEventKey *event )
{
	if( event->keyval == GDK_Control_L || event->keyval == GDK_Control_R )
		if( m_link_hovered )
			update_link();
		
	return Gtk::TextView::on_key_release_event( event );
}
