/***********************************************************************************

	Copyright (C) 2007-2009 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_HELPERS_HEADER
#define LIFEOGRAPH_HELPERS_HEADER


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <fstream>

#include <gcrypt.h>

#include <gtkmm.h>

#include <libintl.h>

// DEFINITIONS FOR LIBGETTEXT
#define _(String)               gettext(String)
#define gettext_noop(String)    String
#define N_(String)              gettext_noop(String)
// END OF LIBGETTEXT DEFINITIONS


namespace LIFEO
{

class Error
{
	public:
						Error( const Glib::ustring& );
};

const unsigned int	MAX_RECENT_FILE_COUNT( 8 );

// RESPONSE IDS
static const int		RESPONSE_CANCEL		= 0;
static const int		RESPONSE_GO			= 1;

// DATE ========================================================================
// order: 10 bits
// day:	   5 bits
// month:  4 bits
// year:  13 bits
class Date
{
	public:
		typedef unsigned long date_t;

		void operator=( const Date &date )
		{ m_date = date.m_date; }
		void operator=( const Date::date_t &date )
		{ m_date = date; }

		bool operator>( const Date& date_r )
		{ return( m_date > date_r.m_date ); }
		bool operator>=( const Date& date_r )
		{ return( m_date >= date_r.m_date ); }
		bool operator<( const Date& date_r )
		{ return( m_date < date_r.m_date ); }

		bool operator==( const Date &date )
		{ return( m_date == date.m_date ); }
		bool operator==( const Date::date_t &date )
		{ return( m_date == date ); }

		bool operator!=( const Date &date )
		{ return( m_date != date.m_date ); }

		explicit					Date( date_t date )
			:	m_date( date ) {}
									Date( unsigned int y, unsigned int m, unsigned int d,
										  unsigned int o = 0 )
			:	m_date( ( y << 19 ) + ( m << 15 ) + ( d << 10 ) + o ) {}

		date_t						get_pure( void ) const
		{ return( m_date & 0xFFFFFC00 ); }

		date_t						get_yearmonth( void ) const
		{ return( m_date & 0xFFFF8000 ); }

		unsigned int				get_year( void ) const
		{ return ( ( m_date & 0xFFF80000 ) >> 19 ); }

		Glib::ustring				get_year_str( void ) const;
		Glib::ustring				get_month_str( void ) const;
		Glib::ustring				get_day_str( void ) const;
		Glib::ustring				get_weekday_str( void ) const;

		unsigned int				get_month( void ) const
		{ return( ( m_date & 0x78000 ) >> 15 ); }

		Glib::Date::Month			get_month_glib( void ) const;

		unsigned int				get_day( void ) const
		{ return( ( m_date & 0x7C00 ) >> 10 ); }

		unsigned int				get_order( void ) const
		{ return( m_date & 0x3FF ); }

		Glib::Date					get_glib( void ) const
		{ return Glib::Date( get_day(), get_month_glib(), get_year() ); }
		time_t						get_ctime( void ) const;

		void						set_year( unsigned int y )
		{
			m_date &= 0x74444;
			m_date |= ( y << 19 );
		}
		void						set_month( unsigned int m )
		{
			m_date &= 0xFFF87FFF;
			m_date |= ( m << 15 );
		}
		void						set_day( unsigned int d )
		{
			m_date &= 0xFFFF83FF;
			m_date |= ( d << 10 );
		}
		void						reset_order( void )
		{
			m_date &= 0xFFFFFC00;
		}

		Glib::ustring				format_string( void ) const;
		static Glib::ustring		format_string( const time_t* );

		bool						is_valid( void ) const
		{
			return( Glib::Date::get_days_in_month( get_month_glib(), get_year() ) >= get_day() );
		}

		void						forward_months( int months )
		{
			months += ( ( m_date & 0x78000 ) >> 15 ); // get month
			m_date &= 0xFFF80000;	// truncate months
			const int mod_months = months % 12;
			if( mod_months == 0 )
			{
				m_date += make_year( ( months / 12 ) - 1);
				m_date |= 0x60000;	// make month 12
			}
			else
			{
				m_date += make_year( months / 12 );
				m_date |= make_month( mod_months );
			}
		}

		void						forward_month( void )
		{
			int months = get_month() + 1;
			m_date &= 0xFFF80000;	// truncate months
			const int mod_months = months % 12;
			if( mod_months == 0 )
			{
				m_date += make_year( ( months / 12 ) - 1);
				m_date |= 0x60000;	// make month 12
			}
			else
			{
				m_date += make_year( months / 12 );
				m_date |= make_month( mod_months );
			}
		}

		static date_t				make_year( unsigned int y )
		{ return( y << 19 ); }
		static date_t				make_month( unsigned int m )
		{ return( m << 15 ); }
		static date_t				make_day( unsigned int m )
		{ return( m << 10 ); }

	//protected:
		date_t						m_date;
};

/*
inline  int
subtract_months( Date d1, Date d2 )
{
	return ( ( isolate_month( d1 ) - isolate_month( d2 ) ) +
			 ( ( isolate_year( d1 ) - isolate_year( d2 ) ) * 12 ) );
}
*/

inline bool
compare_dates( const Date::date_t &date_l, const Date::date_t &date_r )
{
	return( date_l > date_r );
}

typedef bool( *FuncCompareDates )( const Date::date_t&, const Date::date_t& ) ;

// GLOBAL FUNCTIONS
void				print_error( const Glib::ustring& );
void				print_info( const Glib::ustring& );

#ifdef LIFEOGRAPH_DEBUG_BUILD
#define PRINT_DEBUG( message )	std::cout << "DEBUG: " << message << std::endl
#else
#define PRINT_DEBUG( message )	;
#endif

long				convert_string( const std::string& );

Gdk::Color			contrast( const Gdk::Color&, const Gdk::Color& );
Gdk::Color			midtone( const Gdk::Color&, const Gdk::Color& );

inline Glib::ustring
convert_gdkcolor_to_html( const Gdk::Color &gdkcolor )
{
	// this function's source of inspiration is Geany
	char buffer[ 8 ];

	g_snprintf(	buffer, 8, "#%02X%02X%02X",
			gdkcolor.get_red() >> 8,
			gdkcolor.get_green() >> 8,
			gdkcolor.get_blue() >> 8 );
	return buffer;
}

std::ios::pos_type	get_file_size( std::ifstream& );

void				open_url( Gtk::AboutDialog&, const Glib::ustring& );
void				mail_to( Gtk::AboutDialog&, const Glib::ustring& );

Gtk::MenuItem*		create_menuitem_markup( const Glib::ustring&,
											const Glib::SignalProxy0< void >::SlotType& );

// COMMON SIGNALS
typedef sigc::signal< void >		SignalVoid;
typedef std::list< std::string >	ListPaths;

// MEMORY POOL
template< class T >
class Pool : public std::list< T* >
{
	public:
		virtual					~Pool()
		{ clear(); }

		void					clear( void )
		{
			for(	typename std::list< T* >::const_iterator iter = this->begin();
					iter != this->end();
					++iter )
			{
				delete *iter;
			}

			std::list< T* >::clear();
		}
};


class Cipher
{
	public:
		static const int	cCIPHER_ALGORITHM	= GCRY_CIPHER_AES256;
		static const int	cCIPHER_MODE		= GCRY_CIPHER_MODE_CFB;
		static const int	cIV_SIZE			= 16; // = 128 bits
		static const int	cSALT_SIZE			= 16; // = 128 bits
		static const int	cKEY_SIZE			= 32; // = 256 bits
		static const int	cHASH_ALGORITHM		= GCRY_MD_SHA256;

		static bool			init( void );

		static void			create_iv(	unsigned char** );
		static void			expand_key(	char const*,
										const unsigned char*,
										unsigned char** );
		static void			create_new_key(	char const*,
											unsigned char**,
											unsigned char** );
		static void			encrypt_buffer(	unsigned char*,
											size_t&,
											const unsigned char*,
											const unsigned char* );
		static void			decrypt_buffer(	unsigned char*,
											size_t&,
											const unsigned char*,
											const unsigned char * );

	protected:

	private:

};

// ENTRY WIDGET WITH TIP TEXT WHEN IDLE
class EntryIdletext : public Gtk::Entry
{
	public:
							EntryIdletext( BaseObjectType*, const Glib::RefPtr<Gtk::Builder>& );
		explicit			EntryIdletext( const Glib::ustring& );

		void				set_idle_text( const Glib::ustring &idle_text )
		{ m_idletext = idle_text; }

	protected:
		void				handle_clear_icon( Gtk::EntryIconPosition, const GdkEventButton* );
		virtual void		on_map( void );
		virtual void		on_changed( void );
		virtual bool		on_expose_event( GdkEventExpose* );
		virtual bool		on_focus_in_event( GdkEventFocus* );
		virtual bool		on_key_release_event( GdkEventKey* );

		Glib::ustring		m_idletext;
		bool				m_flag_empty;
};

class Menu2 : public Gtk::Menu
{
	public:
		Menu2() { }
		Menu2( BaseObjectType* cobject, const Glib::RefPtr<Gtk::Builder>& )
		:	Gtk::Menu( cobject ) { }

	protected:

	private:

	friend class Menubutton;
};

class Menubutton : public Gtk::ToggleButton
{
	public:
							Menubutton(	const Gtk::StockID&,
										const Glib::ustring&,
										Gtk::ReliefStyle = Gtk::RELIEF_NONE,
										Gtk::IconSize = Gtk::ICON_SIZE_BUTTON,
										Menu2* = NULL );
							Menubutton(	const Glib::RefPtr< Gdk::Pixbuf >*,
										const Glib::ustring&,
										Gtk::ReliefStyle = Gtk::RELIEF_NONE,
										Menu2* = NULL );
		virtual				~Menubutton( void );

		Menu2*				get_menu( void );
		bool				clear_menu( void );
		void				release( void );

		void				set_label( const Glib::ustring& );
		void				get_menu_position( int&, int&, bool& );

	protected:
		bool				on_button_press_event( GdkEventButton* );

		Menu2				*m_menu;
		Gtk::Label			*m_label;
		Gtk::Image			*m_icon;
		Gtk::HBox			*m_hbox;
		Gtk::VBox			*m_vbox;

	private:

	friend class Menutoolbutton;
};

class Menutoolbutton : public Gtk::ToolItem
{
	public:
		Menutoolbutton( const Gtk::StockID&, const std::string&,
						Menu2* = NULL, bool = false );

		Menu2*			get_menu( void );

	protected:
		void			on_toolbar_reconfigured( void );

		bool			m_flag_important;
		Menubutton		m_button;
};

// FILE FILTERS
class FilefilterAny : public Gtk::FileFilter
{
	public:
		FilefilterAny()
		{
			set_name( _("All Files") );
			add_pattern( "*" );
		}
};

class FilefilterDiary : public Gtk::FileFilter
{
	public:
		FilefilterDiary()
		{
			set_name( _("Diary Files (*.diary)") );
			add_pattern( "*.diary" );
		}
};

class MenuitemRecent : public Gtk::MenuItem
{
	public:
		typedef sigc::signal< void, const std::string& >
									Signal_void_string;

									MenuitemRecent( const std::string& );

		Signal_void_string			signal_removerecent( void )
		{ return m_signal_removerecent; }

	protected:
		virtual bool				on_motion_notify_event( GdkEventMotion* );
		virtual bool				on_leave_notify_event( GdkEventCrossing* );
		virtual bool				on_button_release_event( GdkEventButton* );
		virtual bool				on_button_press_event( GdkEventButton* );

		Gtk::Image					m_icon_remove;
		std::string					m_path;
		bool						m_flag_deletepressed;

		Signal_void_string			m_signal_removerecent;
};

class FilebuttonRecent : public Menubutton
{
	public:
									FilebuttonRecent( const Glib::RefPtr< Gdk::Pixbuf >&,
													  ListPaths*, const Glib::ustring& );

		std::list< std::string >*	get_recentlist( void )
		{
			return m_list_recent;
		}
		std::string					get_filename( void ) const;
		std::string					get_filename_recent( int ) const;
		void						set_filename( const std::string& );
		void						update_filenames( void );

		bool						add_recent( const std::string& );
		void						remove_recent( const std::string& );

		void						show_filechooser( void );

		SignalVoid					signal_selection_changed( void );
		SignalVoid					signal_create_file( void );

		bool						handle_motion_notify( GdkEventMotion* );

	protected:
		virtual void				on_create_file( void );
		virtual void				on_size_allocate( Gtk::Allocation& );
		virtual void				on_drag_data_received(
											const Glib::RefPtr<Gdk::DragContext>&,
											int, int,
											const Gtk::SelectionData&, guint, guint );

	private:
		Gtk::FileChooserDialog		*m_filechooserdialog;
		Gtk::Image					m_icon_new;
		Gtk::Image					m_icon_browse;
		ListPaths					*m_list_recent;
		SignalVoid					m_signal_selectionchanged;
		SignalVoid					m_signal_createfile;
};

} // end of namespace LIFEO

#endif

