/***********************************************************************************

	Copyright (C) 2007-2010 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "entry.hpp"
#include "diary.hpp"
#include "lifeobase.hpp"


using namespace LIFEO;


// STATIC MEMBERS
Entry::Shower						*Entry::shower = NULL;

Entry::Entry( const Date &date, const Glib::ustring &text, bool flag_favored )
:	m_date( date ), m_date_created( date.get_ctime() ), m_date_changed( date.get_ctime() ),
	m_text( text ), m_ptr2theme( NULL ),
	m_option_favored( flag_favored ),
	m_option_spellcheck( true ), m_flag_filteredout( false )
{
}

Entry::Entry( const Date &date, bool flag_favored )
:	m_date( date ), m_date_created( date.get_ctime() ), m_date_changed( date.get_ctime() ),
	m_text( "" ), m_ptr2theme( NULL ),
	m_option_favored( flag_favored ),
	m_option_spellcheck( true ), m_flag_filteredout( false )
{
}

Glib::RefPtr< Gdk::Pixbuf >&
Entry::get_icon( void ) const
{
	return( m_option_favored ?
			Lifeobase::icons->entry_favorite_16 : Lifeobase::icons->entry_16 );
}

Glib::ustring
Entry::get_title( void ) const
{
	if( m_text.size() < 1 )
		// TRANSLATORS: title of an empty diary entry
		return( _( "<empty entry>" ) );
	unsigned int l_pos = m_text.find( '\n', 0 );
	if( l_pos == Glib::ustring::npos )
		return m_text;
	else
		return m_text.substr( 0, l_pos );
}

Glib::ustring
Entry::get_2nd_str( void ) const
{
	return get_title();
}

Glib::ustring
Entry::get_date_created_str( void ) const
{
	return Date::format_string( &m_date_created );
}

Glib::ustring
Entry::get_date_changed_str( void ) const
{
	return Date::format_string( &m_date_changed );
}

void
Entry::set_tags( const Tagset &tagset )
{
	m_tags = tagset;
}

const Tagset&
Entry::get_tags( void ) const
{
	return m_tags;
}

Tagset&
Entry::get_tags( void )
{
	return m_tags;
}

bool
Entry::add_tag( Tag *tag )
{
	if( m_tags.add( tag ) )
	{
		tag->get_items()->insert( this );
		return true;
	}
	else
		return false;
}

bool
Entry::remove_tag( Tag *tag )
{
	if( tag == NULL )
		return false;

	m_tags.erase( tag );
	tag->get_items()->erase( this );

	return true;
}

bool
Entry::get_filtered_out( void )
{
	 Diary::FilteringStatus filtering_status = Diary::d->get_filtering_status();
	 
	if( filtering_status & Diary::FS_NEW )	// recalculate
	{
		m_flag_filteredout = false;

		if( filtering_status == Diary::FS_NEW )	// FS_NEW alone means not applied clear
			return false;

		// cheapest first: favorites filter
		if( filtering_status & Diary::FS_FILTER_FAVORITES )
			if( ( m_flag_filteredout = ! m_option_favored ) )
				return true; // no need to continue, already filtered out

		if( filtering_status & Diary::FS_FILTER_TAG )
			if( ( m_flag_filteredout = ! m_tags.checkfor_member( Diary::d->get_filter_tag() ) ) )
				return true; // no need to continue, already filtered out

		if( filtering_status & Diary::FS_FILTER_TEXT )
			m_flag_filteredout =
					( m_text.lowercase().find( Diary::d->get_filter_text() ) == std::string::npos );
	}

	return m_flag_filteredout;
}

void
Entry::show( void )
{
	if( shower != NULL )
		shower->show( *this );
	else
		PRINT_DEBUG( "Entry has no graphical data!" );
}

void
Entry::prepare_for_hiding( void )
{
	if( shower != NULL )
		shower->prepare_for_hiding( *this );
	else
		PRINT_DEBUG( "Entry has no graphical data!" );
}

// ENTRY SET ===================================================================
PoolEntries::~PoolEntries()
{
	for( iterator iter = begin(); iter != end(); ++iter )
		delete iter->second;
}

void
PoolEntries::clear( void )
{
	for( iterator iter = begin(); iter != end(); ++iter )
		delete iter->second;

	std::map< Date::date_t, Entry*, FuncCompareDates >::clear();
}
