/***********************************************************************************

	Copyright (C) 2007-2010 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cstdio>	// for file operations
#include <string>
#include <sstream>
#include <iostream>
#include <fstream>
#include <gcrypt.h>

#include "diary.hpp"
#include "helpers.hpp"
#include "lifeobase.hpp"


using namespace LIFEO;


// STATIC MEMBERS
Diary				*Diary::d;
Diary::Shower		*Diary::shower;

Diary::Diary( const std::string &path )
:	m_path( path ), m_passphrase( "" ),
	m_ptr2chapter_ctg_cur( NULL ), m_default_theme( &Theme::system_theme ),
	m_option_spellcheck( false ), m_option_sorting_criteria( SC_DATE ),
	m_filter_text( "" ), m_filter_tag( NULL ), m_filtering_status( FS_CLEAR )
{
	// "touch" new lock
	std::string path_lock = m_path + LOCK_SUFFIX;
	FILE *fp = fopen( path_lock.c_str(), "a+" );
	if( fp )
		fclose( fp );
}

// derived classes should use this
Diary::Diary( void )
:	m_path( "" ), m_passphrase( "" ),
	m_ptr2chapter_ctg_cur( NULL ), m_default_theme( &Theme::system_theme ),
	m_option_spellcheck( false ), m_option_sorting_criteria( SC_DATE ),
	m_filter_text( "" ), m_filter_tag( NULL ), m_filtering_status( FS_CLEAR )
{
}

Diary::~Diary( void )
{
	if( ! m_path.empty() )
	{
		std::string path_lock = m_path + LOCK_SUFFIX;
		remove( path_lock.c_str() );
	}
}

Glib::RefPtr< Gdk::Pixbuf >&
Diary::get_icon( void ) const
{
	return Lifeobase::icons->diary_16;
}

void
Diary::set_path( const std::string &path )
{
	std::string path_lock;

	if( ! m_path.empty() )
	{
		path_lock = m_path + LOCK_SUFFIX;
		remove( path_lock.c_str() );
	}

	m_path = path;

	// "touch" new lock
	// BEWARE: Diary( const std::string& ) also uses this algorithm
	path_lock = m_path + LOCK_SUFFIX;
	FILE *fp = fopen( path_lock.c_str(), "a+" );
	if( fp )
		fclose( fp );
}

const std::string&
Diary::get_path( void ) const
{
	return m_path;
}

bool
Diary::is_path_set( void ) const
{
	return ( (bool) m_path.size() );
}

bool
Diary::set_passphrase( const std::string &passphrase )
{
	if( passphrase.size() >= PASSPHRASE_MIN_SIZE )
	{
		m_passphrase = passphrase;
		return true;
	}
	else
		return false;
}

void
Diary::clear_passphrase( void )
{
	m_passphrase.clear();
}

std::string
Diary::get_passphrase( void ) const
{
	return m_passphrase;
}

bool
Diary::compare_passphrase( const std::string &passphrase ) const
{
	return( m_passphrase == passphrase );
}

bool
Diary::is_passphrase_set( void ) const
{
	return (bool) m_passphrase.size();
}

LIFEO::Result
Diary::read_header()
{
	std::ifstream file( m_path.c_str() );

	if( ! file.is_open() )
	{
		print_error( "failed to open diary file " + m_path );
		return LIFEO::COULD_NOT_START;
	}
	std::string line;

	getline( file, line );

	if( line != LIFEO::DB_FILE_HEADER )
	{
		file.close();
		return LIFEO::CORRUPT_FILE;
	}

	while( getline( file, line ) )
	{
		switch( line[ 0 ] )
		{
			case 'V':
			{
				int version = convert_string( line.substr( 2 ) );
				if( version < LIFEO::DB_FILE_VERSION_INT_MIN ||
					version > LIFEO::DB_FILE_VERSION_INT )
				{
					file.close();
					return LIFEO::INCOMPATIBLE_FILE;
				}
				m_flag_v56 = ( version == 56 );
				break;
			}
			case 'E':
				if( line[ 2 ] != 'n' )
					// passphrase is set to a dummy value to indicate that diary
					// is an encrypted one until user enters the real passphrase
					m_passphrase = " ";
				else
					m_passphrase.clear();
				break;
			case 0: // end of header
				BodyPosition = file.tellg();
				file.close();
				return LIFEO::SUCCESS;
			default:
				print_error( "unrecognized header line: " + line );
				break;
		}
	}

	file.close();
	m_path = "";
	return LIFEO::CORRUPT_FILE;
}

LIFEO::Result
Diary::read_body()
{
	if( m_passphrase.empty() )
		return read_plain();
	else
		return read_encrypted();
}

LIFEO::Result
Diary::read_plain()
{
	std::ifstream file( m_path.c_str() );

	if( ! file.is_open() )
	{
		print_error( "failed to open diary file " + m_path );
		return LIFEO::COULD_NOT_START;
	}

	file.seekg( 0, std::ios::end );

	int readsize = file.tellg() - BodyPosition - 1;

	if( readsize <= 0 )
	{
		print_info( "empty diary" );
		file.close();
		return LIFEO::EMPTY_DATABASE;
	}

	file.seekg( BodyPosition );

	char *readbuffer = new char[ readsize + 1 ];
	// TODO: check validity

	file.read( readbuffer, readsize );

	readbuffer[ readsize ] = 0;   // terminating zero

	file.close();

	std::stringstream stream( readbuffer );
	LIFEO::Result result = m_flag_v56 ? parse_db_body_text_56( stream ) :
										parse_db_body_text( stream );
	delete[] readbuffer;

	return result;
}

LIFEO::Result
Diary::read_encrypted()
{
	std::ifstream file( m_path.c_str(), std::ios::in | std::ios::binary );

	if( ! file.is_open() )
	{
		print_error( "Failed to open diary file" + m_path );
		return LIFEO::COULD_NOT_START;
	}
	file.seekg( BodyPosition );

	std::string line, content;
	unsigned char *buffer;
	try {
		// Allocate memory for salt
		unsigned char *salt = new unsigned char[ LIFEO::Cipher::cSALT_SIZE ];
		if( ! salt )
			throw LIFEO::Error( "Unable to allocate memory for salt" );
		// Read salt value
		file.read( (char*) salt, LIFEO::Cipher::cSALT_SIZE );

		unsigned char *iv = new unsigned char[ LIFEO::Cipher::cIV_SIZE ];
		if( ! iv )
			throw LIFEO::Error( "Unable to allocate memory for IV" );
		// Read IV
		file.read( (char*) iv, LIFEO::Cipher::cIV_SIZE );

		unsigned char *key;
		LIFEO::Cipher::expand_key( m_passphrase.c_str(), salt, &key );

		// Calulate bytes of data in file
		size_t size = LIFEO::get_file_size( file ) - file.tellg();
		if( size <= 3 )
		{
			file.close();
			return LIFEO::EMPTY_DATABASE;
		}
		buffer = new unsigned char[ size + 1 ];
		// TODO(<0.4.0): ex handling here!!!

		file.read( (char*) buffer, size );
		LIFEO::Cipher::decrypt_buffer( buffer, size, key, iv );
		file.close();

		// passphrase check
		if( buffer[ 0 ] != m_passphrase[ 0 ] && buffer[ 1 ] != '\n' )
		{
			return LIFEO::WRONG_PASSWORD;
		}

		buffer[ size ] = 0;   // terminating zero
	}
	catch( ... )
	{
		return LIFEO::COULD_NOT_START;
	}

	std::stringstream stream;
	buffer += 2;    // ignore first two chars which are for passphrase checking
	stream << buffer;
	buffer -= 2;    // restore pointer to the start of the buffer before deletion
	delete[] buffer;

	return( m_flag_v56 ? parse_db_body_text_56( stream ) :
						 parse_db_body_text( stream ) );
}

LIFEO::Result
Diary::parse_db_body_text( std::stringstream &stream )
{
	std::string			line( "" );
	Entry				*entry_new = NULL;
	CategoryChapters	*ptr2chapter_ctg = NULL;
	Chapter				*ptr2chapter = NULL;
	CategoryTags		*ptr2tag_ctg = NULL;
	Theme				*ptr2theme = NULL;

	// TAG DEFINITIONS & CHAPTERS
	while( getline( stream, line ) )
	{
		if( line[ 0 ] == 0 )	// end of section
			break;
		else
		if( line.size() < 3 )
			continue;
		else
		{
			switch( line[ 0 ] )
			{
				case 'O':	// options
					if( line.size() < 4 )
						break;
					m_option_spellcheck = ( line[ 2 ] == 's' );
					m_option_sorting_criteria = line[ 3 ];
					break;
				case 'T':	// tag category
					ptr2tag_ctg = create_tag_ctg( line.substr( 2 ) );
					ptr2tag_ctg->set_expanded( line[ 1 ] == 'e' );
					break;
				case 't':	// tag
					m_tags.create( line.substr( 2 ), ptr2tag_ctg );
					break;
				case 'C':	// chapter category
					ptr2chapter_ctg = create_chapter_ctg( line.substr( 2 ) );
					if( line[ 1 ] == 'c' )
						m_ptr2chapter_ctg_cur = ptr2chapter_ctg;
					break;
				case 'c':	// chapter
					if( ptr2chapter_ctg == NULL )
					{
						print_error( "No chapter category defined" );
						break;
					}
					ptr2chapter = ptr2chapter_ctg->add_chapter( line.substr( 2 ) );
					ptr2chapter->set_expanded( line[ 1 ] == 'e' );
					break;
				case 'd':	// chapter begin date
					if( ptr2chapter != NULL )	// chapter begin date
					{
						Date date( LIFEO::convert_string( line.substr( 2 ) ) );
						ptr2chapter_ctg->set_chapter_date( ptr2chapter, date );
					}
					break;
				case 'M':
					ptr2theme = create_theme( line.substr( 2 ) );
					if( line[ 1 ] == 'd' )
						m_default_theme = ptr2theme;
					break;
				case 'm':
					if( ptr2theme == NULL )
					{
						print_error( "No theme declared" );
						break;
					}
					switch( line[ 1 ] )
					{
						case 'f':	// font
							ptr2theme->font = Pango::FontDescription( line.substr( 2 ) );
							break;
						case 'b':	// base color
							ptr2theme->color_base.set( line.substr( 2 ) );
							break;
						case 't':	// text color
							ptr2theme->color_text.set( line.substr( 2 ) );
							break;
						case 'h':	// heading color
							ptr2theme->color_heading.set( line.substr( 2 ) );
							break;
						case 's':	// subheading color
							ptr2theme->color_subheading.set( line.substr( 2 ) );
							break;
						case 'l':	// highlight color
							ptr2theme->color_highlight.set( line.substr( 2 ) );
							break;
					}
					break;
				default:
					print_error( "unrecognized line:\n" + line );
					return LIFEO::FAILURE;
			}
		}
	}

	// ENTRIES
	while( getline( stream, line ) )
	{
		if( line.size() < 2 )
			continue;

		switch( line[ 0 ] )
		{
			case 'E':	// new entry
				entry_new = new Entry( Date( LIFEO::convert_string( line.substr( 2 ) ) ),
									   line[ 1 ] == 'f' );
				// chapters
				place_entry_in_chapters( entry_new );
				// TODO: per entry spellcheck options
				//entry_new->set_spellcheck( opt_spell );
				m_entries[ entry_new->m_date.m_date ] = entry_new;
				break;
			case 'D':	// creation & change dates (optional)
				if( entry_new == NULL )
				{
					print_error( "No entry declared" );
					break;
				}
				if( line[ 1 ] == 'r' )
					entry_new->m_date_created = LIFEO::convert_string( line.substr( 2 ) );
				else	// it should be 'h'
					entry_new->m_date_changed = LIFEO::convert_string( line.substr( 2 ) );
				break;
			case 'M':	// theme
				if( entry_new == NULL )
					print_error( "No entry declared" );
				else
				{
					PoolThemes::iterator iter_theme( m_themes.find( line.substr( 2 ) ) );
					if( iter_theme != m_themes.end() )
						entry_new->set_theme( iter_theme->second );
					else
						print_error( "Reference to undefined theme: " + line.substr( 2 ) );
				}
				break;
			case 'T':	// tag
				if( entry_new == NULL )
					print_error( "No entry declared" );
				else
				{
					PoolTags::iterator iter_tag( m_tags.find( line.substr( 2 ) ) );
					if( iter_tag != m_tags.end() )
						entry_new->add_tag( iter_tag->second );
					else
						print_error( "Reference to undefined tag: " + line.substr( 2 ) );
				}
				break;
			case 'P':    // paragraph
				if( entry_new == NULL )
				{
					print_error( "No entry declared" );
					break;
				}
				if( ! entry_new->m_text.empty() )
					entry_new->m_text += "\n";
				entry_new->m_text += line.substr( 2 );
				break;
			default:
				print_error( "unrecognized line:\n" + line );
				return LIFEO::FAILURE;
				break;
		}
	}

	if( m_entries.size() )
	{
		return LIFEO::SUCCESS;
	}
	else
	{
		print_info( "empty diary" );
		return LIFEO::EMPTY_DATABASE;
	}
}

LIFEO::Result
Diary::parse_db_body_text_56( std::stringstream &stream )
{
	std::string			line( "" );
	Glib::ustring		content( "" );
	Date				date( 0 );
	bool				flag_favorite( false );
	Entry				*entry_new = NULL;
	CategoryChapters	*ptr2chapter_ctg = NULL;
	Chapter				*ptr2chapter = NULL;
	std::vector< Tag* >	tmp_vector_tags;
	std::string			str_tags;

	// TAG DEFINITIONS & CHAPTERS
	while( getline( stream, line ) )
	{
		if( line[ 0 ] == 0 )
			break;
		else
		if( line.size() < 3 )
			continue;
		else
		{
			switch( line[ 0 ] )
			{
				case 'T':	// tag
					try
					{
						Tag *tmp_tag = m_tags.create( line.substr( 2 ) );
						tmp_vector_tags.push_back( tmp_tag );
						// TODO: this is a temporary solution. will be fixed in new diary version
					}
					catch( std::exception &ex )
					{
						throw ex;
					}
					break;
				case 'L':	// current entryliststyle
					// not used anymore...
					break;
				case 'S':	// chapterset
					try
					{
						ptr2chapter_ctg = create_chapter_ctg( line.substr( 2 ) );
						if( m_ptr2chapter_ctg_cur == NULL )
							m_ptr2chapter_ctg_cur = ptr2chapter_ctg;
					}
					catch( std::exception &ex )
					{
						throw ex;
					}
					break;
				case 'C':	// chapter
					if( m_chapter_categories.size() > 0 )
					{
						try
						{
							ptr2chapter = ptr2chapter_ctg->add_chapter(
									line.substr( 2 ) );
						}
						catch( std::exception &ex )
						{
							throw ex;
						}
					}
					break;
				case 'd':	// begin date
					if( ptr2chapter != NULL )	// chapter begin date
					{
						ptr2chapter_ctg->set_chapter_date( ptr2chapter, line.substr( 2 ) );
					}
					break;
				case 'O':	// options
					m_option_spellcheck = ( line[2] == 's' );
					break;
				case 'M':
					if( m_default_theme == &Theme::system_theme )
					{
						m_default_theme = create_theme( _( "theme" ) );
					}

					switch( line[1] )
					{
						case 'f':	// font
							m_default_theme->font = Pango::FontDescription( line.substr( 2 ) );
							break;
						case 'b':	// base color
							m_default_theme->color_base.set( line.substr( 2 ) );
							break;
						case 't':	// text color
							m_default_theme->color_text.set( line.substr( 2 ) );
							break;
						case 'h':	// heading color
							m_default_theme->color_heading.set( line.substr( 2 ) );
							break;
						case 's':	// subheading color
							m_default_theme->color_subheading.set( line.substr( 2 ) );
							break;
						case 'l':	// highlight color
							m_default_theme->color_highlight.set( line.substr( 2 ) );
							break;
					}
					break;
				default:
					print_error( "unrecognized line:\n" + line );
					return LIFEO::FAILURE;
			}
		}
	}

	// ENTRIES
	while( getline( stream, line ) )
	{
		switch( line[ 0 ] )
		{
			case 'A':	// date
			{	// FIXME: temporary (for compatibility)
				Date::date_t date_new = LIFEO::convert_string( line.substr( 2 ) );
				if( date_new == ( date.m_date >> 10 ) )
					date.m_date++;
				else
					date.m_date = ( date_new << 10 );
				break;
			}
			case 'C':    // content
				// when writing a diary file by hand, one can easily forget to put
				// a blank space after C, so we have to check
				if( line.size() < 2 )
					continue;

				content += line.substr( 2 );
				content += "\n";
				break;
			case 'T':    // tags
				str_tags = line;
			break;
			case '}':    // end of Entry
			{
				// erase redundant new line:
				content.erase( content.size() - 1 );

				entry_new = new Entry( date, content, flag_favorite );

				// tags
				std::vector< Tag* >::const_iterator iter_tag = tmp_vector_tags.begin();
				for( std::string::size_type i = 2; i < str_tags.size(); i++ )
				{
					if( iter_tag == tmp_vector_tags.end() )
						//TODO: issue a warning
						break;
					else
					if( str_tags[ i ] == '#' )
						entry_new->add_tag( *iter_tag );
					else
					if( str_tags[ i ] == ' ' )
						continue;

					++iter_tag;
				}

				// chapters
				place_entry_in_chapters( entry_new );
				// TODO: per entry spellcheck options
				//entry_new->set_spellcheck( opt_spell );
				m_entries[ date.m_date ] = entry_new;

				content.clear();
				str_tags.clear();
				break;
			}
			case '{':
				flag_favorite = false;
				break;
			case '[':
				flag_favorite = true;
				break;
			case 0:
			case '\n':
				break;
			default:
				print_error( "unrecognized line:\n" + line );
				return LIFEO::FAILURE;
				break;
		}
	}

	if( m_entries.size() )
	{
		return LIFEO::SUCCESS;
	}
	else
	{
		print_info( "empty diary" );
		return LIFEO::EMPTY_DATABASE;
	}
}

LIFEO::Result
Diary::write( void )
{
	std::string path_old( m_path + ".~previousversion~" );

	if( Glib::file_test( m_path, Glib::FILE_TEST_EXISTS ) )
		rename( m_path.c_str(), path_old.c_str() );

	Result result;
	if( m_passphrase.empty() )
		result = write_plain( m_path );
	else
		result = write_encrypted( m_path );

// DAILY BACKUP SAVES
#ifdef LIFEOGRAPH_DEBUG_BUILD
	if( m_entries.empty() )
		return result;

	Result result_daily;
	std::string path_daily( m_path + "." + m_entries.begin()->second->get_date().format_string() );

	if( m_passphrase.empty() )
		result_daily = write_plain( path_daily );
	else
		result_daily = write_encrypted( path_daily );

	if( result_daily == SUCCESS )
		PRINT_DEBUG( "daily backup has been written successfully" );
#endif

	return result;
}

LIFEO::Result
Diary::write( const std::string& path )
{
	Result result;
	if( m_passphrase.empty() )
		result = write_plain( path );
	else
		result = write_encrypted( path );

	return result;
}

LIFEO::Result
Diary::write_copy( const std::string &path, const std::string &passphrase )
{
	Result result;

	if( passphrase.empty() )
		result = write_plain( path );
	else
	{	
		std::string passphrase_actual( m_passphrase );
		m_passphrase = passphrase;
		result = write_encrypted( path );
		m_passphrase = passphrase_actual;
	}

	return result;
}

LIFEO::Result
Diary::write_txt( const std::string &path )
{
	std::ofstream file( path.c_str(), std::ios::out | std::ios::trunc );
	if( ! file.is_open() )
	{
		print_error( "i/o error: " + path );
		return LIFEO::COULD_NOT_START;
	}

	// ENTRIES
	for(	Entryiter itr_entry = m_entries.begin();
			itr_entry != m_entries.end();
			itr_entry++ )
	{
		Entry *entry = itr_entry->second;
		// purge empty entries:
		if( entry->m_text.size() < 1 ) continue;

		file << ( "---------------------------------------------\n\n\n" );

		// DATE AND FAVOREDNESS
		if( entry->is_favored() )
			file << "*** ";
		file << entry->get_date().format_string();
		if( entry->is_favored() )
			file << " ***";

		// TAGS
		const Tagset &tagset = entry->get_tags();
		for(	Tagset::const_iterator itr_tags = tagset.begin();
				itr_tags != tagset.end();
				++itr_tags )
		{
			if( itr_tags == tagset.begin() )
				file << "\n>>> "; // FIXME: use a more meaningful sign
			else
				file << ", ";

			file << ( *itr_tags )->get_name();
		}

		// CONTENT
		file << "\n\n" << entry->get_text();

		file << "\n\n\n";
	}
	file.close();
	return SUCCESS;
}

LIFEO::Result
Diary::write_plain( const std::string &path, bool flag_headeronly )
{
	std::ofstream file( path.c_str(), std::ios::out | std::ios::trunc );
	if( ! file.is_open() )
	{
		print_error( "i/o error: " + path );
		return LIFEO::COULD_NOT_START;
	}

	std::stringstream output;
	create_db_header_text( output, flag_headeronly );
	// header only mode is for encrypted diaries
	if( ! flag_headeronly )
	{
		create_db_body_text( output );
	}

	file << output.str();
	file.close();

	return LIFEO::SUCCESS;
}

LIFEO::Result
Diary::write_encrypted( const std::string &path )
{
	// writing header:
	write_plain( path, true );
	std::ofstream file( path.c_str(), std::ios::out | std::ios::app | std::ios::binary );
	if( ! file.is_open() )
	{
		print_error( "i/o error: " + path );
		return LIFEO::COULD_NOT_START;
	}
	std::stringstream output;
	// first char of passphrase for validity checking
	output << m_passphrase[ 0 ] << '\n';
	create_db_body_text( output );

	// encryption
	try {
		size_t size =  output.str().size() + 1;

		unsigned char *key, *salt;
		LIFEO::Cipher::create_new_key( m_passphrase.c_str(), &salt, &key );

		unsigned char *iv;
		LIFEO::Cipher::create_iv( &iv );

		unsigned char *buffer = new unsigned char[ size ];
		memcpy( buffer, output.str().c_str(), size );

		LIFEO::Cipher::encrypt_buffer( buffer, size, key, iv );

		file.write( (char*) salt, LIFEO::Cipher::cSALT_SIZE );
		file.write( (char*) iv, LIFEO::Cipher::cIV_SIZE );
		file.write( (char*) buffer, size );

	}
	catch( ... )
	{
		return LIFEO::FAILURE;
	}

	file.close();

	return LIFEO::SUCCESS;
}

bool
Diary::create_db_header_text( std::stringstream &output, bool encrypted )
{
	output << LIFEO::DB_FILE_HEADER;
	output << "\nV " << LIFEO::DB_FILE_VERSION_INT;
	output << "\nE " << ( encrypted ? "yes" : "no" );
	output << "\n\n"; // end of header

	return true;
}

bool
Diary::create_db_body_text( std::stringstream &output )
{
	std::string::size_type	pt_start, pt_end;
	std::string				content_std;

	// OPTIONS
	output << "O " << ( m_option_spellcheck ? 's' : '-' ) <<
					  m_option_sorting_criteria << '\n';

	// ROOT TAGS
	for( PoolTags::const_iterator iter = m_tags.begin();
		 iter != m_tags.end();
		 ++iter )
	{
		if( iter->second->get_category() == NULL )
			output << "t " << ( *iter ).first << '\n';
	}
	// CATEGORIZED TAGS
	for( PoolCategoriesTags::const_iterator iter = m_tag_categories.begin();
		 iter != m_tag_categories.end();
		 ++iter )
	{
		output << "T" <<
				( iter->second->get_expanded() ? 'e' : ' ' ) <<
				( *iter ).first << '\n';	// tag category
		CategoryTags *category( iter->second );
		for( CategoryTags::const_iterator iter_tag = category->begin();
			 iter_tag != category->end();
			 ++iter_tag )
		{
			output << "t " << ( *iter_tag )->get_name() << '\n';
		}
	}
	// CHAPTERS
	for( PoolCategoriesChapters::iterator iter = m_chapter_categories.begin();
		 iter != m_chapter_categories.end();
		 ++iter )
	{
		output << "C" <<
				( iter->second == m_ptr2chapter_ctg_cur ? 'c' : ' ' ) <<
				iter->first << '\n';
		CategoryChapters *category( iter->second );
		for( CategoryChapters::iterator iter_chapter = category->begin();
			 iter_chapter != category->end();
			 iter_chapter++ )
		{
			output << "c" <<
					( ( *iter_chapter )->get_expanded() ? 'e' : ' ' ) <<
					( *iter_chapter )->get_name() << '\n';
			if( ( *iter_chapter )->is_initialized() )
				output << "d " << ( *iter_chapter )->get_date().m_date << '\n';
		}
	}
	// THEMES
	for( PoolThemes::const_iterator iter = m_themes.begin();
		 iter != m_themes.end();
		 ++iter )
	{
		Theme *theme( iter->second );
		output << "M" << ( theme == m_default_theme ? 'd' : ' ' ) <<
						 theme->m_name << '\n';
		output << "mf" << theme->font.to_string() << '\n';
		output << "mb" << theme->color_base.to_string() << '\n';
		output << "mt" << theme->color_text.to_string() << '\n';
		output << "mh" << theme->color_heading.to_string() << '\n';
		output << "ms" << theme->color_subheading.to_string() << '\n';
		output << "ml" << theme->color_highlight.to_string() << '\n';
	}

	output << '\n'; // end of section

	// ENTRIES
	for( Entryiter iter_entry = m_entries.begin();
		 iter_entry != m_entries.end();
		 ++iter_entry )
	{
		Entry *entry = ( *iter_entry ).second;

		// PURGE EMPTY ENTRIES IF THEY HAVE NO TAGS EITHER
		if( entry->m_text.size() < 1 && entry->m_tags.empty() ) continue;

		// ENTRY DATE
		output << "E" <<
				( entry->m_option_favored ? 'f' : ' ' ) <<
				entry->m_date.m_date << '\n';
		output << "Dr" << entry->m_date_created << '\n';
		output << "Dh" << entry->m_date_changed << '\n';

		// THEME
		if( entry->m_ptr2theme != NULL )
			output << "M " << entry->m_ptr2theme->get_name() << '\n';

		// TAGS
		for( Tagset::const_iterator iter_tag = entry->m_tags.begin();
			 iter_tag != entry->m_tags.end();
			 ++iter_tag )
			output << "T " << ( *iter_tag )->get_name() << '\n';
		// CONTENT
		// NOTE: for some reason, implicit conversion from Glib:ustring...
		// ...fails while substr()ing when LANG=C
		// we might reconsider storing text of entries as std::string.
		// for now we convert entry text to std::string here:
		content_std = entry->m_text;
		pt_start = 0;

		while( true )
		{
			pt_end = content_std.find( '\n', pt_start );
			if( pt_end == std::string::npos )
			{
				pt_end = content_std.size();
				output	<< "P "
						<< content_std.substr( pt_start, pt_end - pt_start );
				break; // end of while( true )
			}
			else
			{
				pt_end++;
				output	<< "P "
						<< content_std.substr( pt_start, pt_end - pt_start );
				pt_start = pt_end;
			}
		}

		output << "\n\n";
	}

	return true;
}

void
Diary::clear( void )
{
	if( ! m_path.empty() )
	{
		std::string path_lock = m_path + LOCK_SUFFIX;
		remove( path_lock.c_str() );
		m_path.clear();
	}
	m_entries.clear();
	m_tags.clear();
	m_tag_categories.clear();
	m_chapter_categories.clear();
	m_ptr2chapter_ctg_cur = NULL;
	m_themes.clear();
	m_default_theme = &Theme::system_theme;

	m_passphrase.clear();
	m_filter_text.clear();
	m_filter_tag = NULL;
	m_filtering_status = FS_CLEAR;

	// NOTE: only reset body options here:
	m_option_spellcheck = true;
	m_option_sorting_criteria = SC_DATE;
}

// ENTRIES =====================================================================
Entry*
Diary::add_today( void )
{
	Glib::Date date;
	date.set_time_current();

	return create_entry( Date( date.get_year(),
							   date.get_month(),
							   date.get_day() ) );
}

Entry*
Diary::get_entry( const Date &date )
{
	for( Entryiter iter = m_entries.find( date.m_date ); iter != m_entries.end(); ++iter )
	{
		if( iter->second->get_date().get_pure() != date.get_pure() )
			break;
		else
		if( iter->second->get_filtered_out() == false )
			return iter->second;
	}
	return NULL;
}

Entry*
Diary::get_entry_today( void )
{
	time_t t = time( NULL );
	struct tm *ti = localtime( &t );
	Date date( ti->tm_year + 1900, ti->tm_mon + 1, ti->tm_mday );

	return get_entry( date );
}

Entryvector*
Diary::get_entries( Date::date_t date )	// takes pure date
{
	Entryvector *entries = new Entryvector;

	Entryiter iter = m_entries.find( date );
	if( iter == m_entries.end() )
		return entries;	// return empty vector

	for( ; ; --iter )
	{
		if( iter->second->get_date().get_pure() == date )
		{
			if( iter->second->get_filtered_out() == false )
				entries->push_back( iter->second );
		}
		else
			break;
		if( iter == m_entries.begin() )
			break;
	}
	return entries;
}

bool
Diary::get_day_has_multiple_entries( const Date &date_impure )
{
	Date::date_t date = date_impure.get_pure();
	Entryiter iter = m_entries.find( date );
	if( iter == m_entries.end() || iter == m_entries.begin() )
		return false;

	do
	{
		--iter;
		if( iter->second->get_date().get_pure() == date )
		{
			if( iter->second->get_filtered_out() == false )
				return true;
		}
		else
			break;
	}
	while( iter != m_entries.begin() );

	return false;
}

Entry*
Diary::get_entry_nextinday( const Date &date )
{
	Entryiter entry_1st = m_entries.find( date.m_date + 1 );
	if( entry_1st != m_entries.end() )
		return entry_1st->second;
	else
	if( date.get_order() > 0 )
	{
		entry_1st = m_entries.find( date.get_pure() );
		if( entry_1st != m_entries.end() )
			return entry_1st->second;
		else
			return NULL;
	}
	else
		return NULL;
	/* TODO MAP
	Entryiter	iter_last			= m_entries.end();
	Entryiter	iter_firstinday		= iter_last;
	bool		flag_found			= false;

	for( Entryiter iter = m_entries.begin(); iter != iter_last; ++iter )
	{
		if( ( *iter )->get_date() == date )
		{
			if( iter_firstinday == iter_last )
				iter_firstinday = iter;
			if( *iter == entry )
				flag_found = true;
			else
			if( flag_found )
				return( *iter );
		}
		else
		if( iter_firstinday != iter_last )
			break;
	}

	if( iter_firstinday != iter_last && *iter_firstinday != entry )
		return( * iter_firstinday );
	else
		return NULL;*/
}

Entry*
Diary::get_entry_first( void )
{
	// return first unfiltered entry
	for( Entryiter iter = m_entries.begin(); iter != m_entries.end(); ++iter )
	{
		Entry *entry = iter->second;
		if( entry->get_filtered_out() == false )
			return( entry );
	}
	return NULL;
}

bool
Diary::is_first( Entry const*const entry ) const
{
	return( entry == m_entries.begin()->second );
}

bool
Diary::is_last( Entry const*const entry ) const
{
	return( entry == m_entries.rbegin()->second );
}

Entry*
Diary::create_entry( Date date, const Glib::ustring &content, bool flag_favorite )
{
	// make it the last entry of its day:
	date.reset_order();
	while( m_entries.find( date.m_date ) != m_entries.end() )
		++date.m_date;

	Entry *entry = new Entry( date, content, flag_favorite );
	entry->m_date_created = time( NULL );
	entry->m_date_changed = entry->m_date_created;

	m_entries[ date.m_date ] = entry;
	place_entry_in_chapters( entry );

	return( entry );
}

bool
Diary::dismiss_entry( Entry *entry )
{
	Date::date_t date = entry->get_date().m_date;

	// remove from chapters:
	remove_entry_from_chapters( entry );

	// remove from tags:
	for( Tagset::iterator iter = entry->get_tags().begin();
		 iter != entry->get_tags().end();
		 ++iter )
		( *iter )->get_items()->erase( entry );

	// erase entry from map:
	m_entries.erase( date );

	// fix entry order:
	int i = 1;
	for( Entryiter iter = m_entries.find( date + i );
		 iter != m_entries.end();
		 iter = m_entries.find( date + i ) )
	{
		Entry *entry2fix = iter->second;
		m_entries.erase( iter );
		entry2fix->get_date().m_date--;
		m_entries[ entry2fix->get_date().m_date ] = entry2fix;
		++i;
	}

	delete entry;
	return true;
}

int
Diary::list_entries( void )
{
	int number_of_entries( 0 );
	for( Entryiter t1_itr = m_entries.begin(); t1_itr != m_entries.end(); t1_itr++ )
	{
		std::cout << t1_itr->second->get_date().format_string() << std::endl;
		number_of_entries++;
	}
	std::cout << number_of_entries << " entries total." << std::endl;
	return( number_of_entries );
}

void
Diary::set_filter_text( const Glib::ustring &filter )
{
	m_filter_text = filter;

	if( filter.empty() )
	{
		if( m_filtering_status & FS_FILTER_TEXT )
			m_filtering_status -= FS_FILTER_TEXT;
	}
	else
		m_filtering_status |= FS_FILTER_TEXT;

	m_filtering_status |= FS_NEW;
}

void
Diary::set_filter_tag( const Tag *tag )
{
	m_filter_tag = tag;

	if( tag == NULL )
	{
		if( m_filtering_status & FS_FILTER_TAG )
			m_filtering_status -= FS_FILTER_TAG;
	}
	else
		m_filtering_status |= FS_FILTER_TAG;

	m_filtering_status |= FS_NEW;
}

void
Diary::toggle_filter_favorites( void )
{
	if( m_filtering_status & FS_FILTER_FAVORITES )
		m_filtering_status -= FS_FILTER_FAVORITES;
	else
		m_filtering_status |= FS_FILTER_FAVORITES;

	m_filtering_status |= FS_NEW;
}

int
Diary::replace_text( const Glib::ustring &newtext )
{
	Glib::ustring::size_type iter_str;
	const int chardiff = newtext.size() - m_filter_text.size();

	Entry *entry;

	for(	Entryiter iter_entry = m_entries.begin();
			iter_entry != m_entries.end();
			iter_entry++ )
	{
		entry = iter_entry->second;
		if( entry->get_filtered_out() )
			continue;

		Glib::ustring entrytext = entry->get_text().lowercase();
		iter_str = 0;
		int count = 0;
		while(	( iter_str = entrytext.find( m_filter_text, iter_str ) ) !=
				std::string::npos )
		{
			entry->get_text().erase(
				iter_str + ( chardiff * count ), m_filter_text.size() );
			entry->get_text().insert(
				iter_str + ( chardiff * count ), newtext );
			count++;
			iter_str += m_filter_text.size();
		}
	}
	
	return 0;	// reserved
}

// TAGS ========================================================================
void
Diary::dismiss_tag( Tag *tag )
{
	// remove from entries:
	for( SetDiaryElements::const_iterator iter = tag->get_items()->begin();
		 iter != tag->get_items()->end();
		 ++ iter )
	{
		Entry *entry = dynamic_cast< Entry* >( *iter );
		entry->remove_tag( tag );
	}

	// remove from category if any:
	if( tag->get_category() != NULL )
		tag->get_category()->erase( tag );

	// clear filter if necessary:
	if( tag == m_filter_tag )
		m_filter_tag = NULL;

	m_tags.erase( tag->get_name() );
	delete tag;
}

CategoryTags*
Diary::create_tag_ctg( void )
{
	Glib::ustring name = create_unique_name_for_map( m_tag_categories, _( "New category" ) );
	CategoryTags *new_category = new CategoryTags( name );
	m_tag_categories.insert( PoolCategoriesTags::value_type( name, new_category ) );

	return new_category;
}

CategoryTags*
Diary::create_tag_ctg( const Glib::ustring &name )	// used while reading diary file
{
	try
	{
		CategoryTags *new_category = new CategoryTags( name );
		m_tag_categories.insert( PoolCategoriesTags::value_type( name, new_category ) );
		return new_category;
	}
	catch( std::exception &ex )
	{
		PRINT_DEBUG( ex.what() );
		throw ex;
	}
}

void
Diary::dismiss_tag_ctg( CategoryTags *category )
{
	// fix entries
	for( CategoryTags::const_iterator iter = category->begin();
		 iter != category->end();
		 ++iter )
	{
		( * iter )->set_category( NULL );
	}
	// remove from the list and delete
	m_tag_categories.erase( category->get_name() );
	delete category;
}

// CHAPTERS ====================================================================
CategoryChapters*
Diary::create_chapter_ctg( void )
{
	Glib::ustring name = create_unique_name_for_map( m_chapter_categories, _( "New category" ) );
	CategoryChapters *category = new CategoryChapters( name );
	m_chapter_categories.insert(
			PoolCategoriesChapters::value_type( name, category ) );

	return category;
}

CategoryChapters*
Diary::create_chapter_ctg( const Glib::ustring &name )
{
	// TODO: check string's availability
	try
	{
		CategoryChapters *category = new CategoryChapters( name );
		m_chapter_categories.insert(
				PoolCategoriesChapters::value_type( name, category ) );
		return category;
	}
	catch( std::exception &ex )
	{
		throw ex;
	}
}

void
Diary::dismiss_chapter_ctg( CategoryChapters *category )
{
	if( category == m_ptr2chapter_ctg_cur )
		m_ptr2chapter_ctg_cur = NULL;

	delete category;
	m_chapter_categories.erase( category->get_name() );
}

bool
Diary::rename_chapter_ctg( CategoryChapters *category, const Glib::ustring &name )
{
	if( m_chapter_categories.count( name ) > 0 )
		return false;

	m_chapter_categories.erase( category->get_name() );
	category->set_name( name );
	m_chapter_categories.insert(
			PoolCategoriesChapters::value_type( name, category ) );

	return true;
}

void
Diary::place_entry_in_chapters( Entry *entry )
{
	for( PoolCategoriesChapters::const_iterator iter_set = m_chapter_categories.begin();
		 iter_set != m_chapter_categories.end();
		 ++iter_set )
	{
		for( CategoryChapters::const_iterator iter_chapter = iter_set->second->begin();
			 iter_chapter != iter_set->second->end();
			 ++iter_chapter )
		{
			if( ( * iter_chapter )->is_initialized() )
				if( entry->get_date() >= ( * iter_chapter )->get_date() )
				{
					( * iter_chapter )->get_items()->insert( entry );
					break;
				}
		}
	}
}

void
Diary::update_entries_in_chapter_ctg( CategoryChapters *set )
{
	Entryiter	iter_last = m_entries.end();
	bool		flag_early;

	// CLEAR ALL SETS
	for(	CategoryChapters::const_iterator iter_chapter = set->begin();
			iter_chapter != set->end();
			++iter_chapter )
	{
		( * iter_chapter )->clear();
	}
	//set->get_earlier()->clear();

	// INSERT ENTRIES AGAIN
	for(	Entryiter iter_entry = m_entries.begin();
			iter_entry != iter_last;
			++iter_entry )
	{
		flag_early = true;
		// TODO: efficiency: do not start form the beginning each time:
		for(	CategoryChapters::const_iterator iter_chapter = set->begin();
				iter_chapter != set->end();
				++iter_chapter )
		{
			if( ( * iter_chapter )->is_initialized() )
			{
				if( iter_entry->second->get_date().get_pure()
					>= ( * iter_chapter )->get_date().m_date )
				{
					( * iter_chapter )->get_items()->insert( iter_entry->second );
					flag_early = false;
					break;
				}
			}
		}
		//if( flag_early )
			//set->get_earlier()->get_items()->insert( iter_entry->second );
	}
}

void
Diary::remove_entry_from_chapters( Entry *entry )
{
	for( PoolCategoriesChapters::const_iterator iter_set = m_chapter_categories.begin();
		 iter_set != m_chapter_categories.end();
		 ++iter_set  )
	{
		for( CategoryChapters::const_iterator iter_chapter = iter_set->second->begin();
			 iter_chapter != iter_set->second->end();
			 ++iter_chapter )
		{
			( * iter_chapter )->get_items()->erase( entry );
		}
	}
}

// THEMES ======================================================================
Theme*
Diary::create_theme( const Glib::ustring& name )
{
	PoolThemes::iterator iter = m_themes.find( name );
	if( iter != m_themes.end() )
		return( iter->second );
	Theme *theme = new Theme( name );
	m_themes.insert( PoolThemes::value_type( name, theme ) );
	return theme;
}

Theme*
Diary::duplicate_theme( const Theme &theme )
{
	Glib::ustring name = create_unique_name_for_map( m_themes, _( "New theme" ) );
	Theme *new_theme = new Theme( theme );	// duplication
	new_theme->m_name = name;
	m_themes.insert( PoolThemes::value_type( name, new_theme ) );

	return new_theme;
}

void
Diary::dismiss_theme( Theme *theme )
{
	// change default theme if necessary
	if( m_default_theme == theme )
	{
		m_default_theme = &Theme::system_theme;
	}
	// fix entries
	for( Entryiter iter = m_entries.begin(); iter != m_entries.end(); ++iter )
		if( iter->second->m_ptr2theme == theme )
			iter->second->m_ptr2theme = NULL;
	// remove from the list and delete
	m_themes.erase( theme->get_name() );
	delete theme;
}

void
Diary::show( void )
{
	if( shower != NULL )
		shower->show( *this );
}

// IMPORTING ===================================================================
bool
Diary::import_tag( Tag *tag )
{
	return( m_tags.create( tag->get_name() ) != NULL );
}

bool
Diary::import_entryset( const PoolEntries &set,
						bool flag_import_tags,
						const Glib::ustring &tag_all_str )
{
	Entry *entry_new, *entry_ext;
	Tag *tag_all = NULL;

	if( ! tag_all_str.empty() )
		tag_all = m_tags.create( tag_all_str );

	for( Entryciter iter = set.begin();
		 iter != set.end();
		 ++iter )
	{
		entry_ext = iter->second;
		entry_new = new Entry( entry_ext->m_date,
							   entry_ext->m_text,
							   entry_ext->m_option_favored );

		// fix order:
		entry_new->m_date.reset_order();
		while( m_entries.find( entry_new->m_date.m_date ) != m_entries.end() )
			entry_new->m_date.m_date++;

		// copy dates:
		entry_new->m_date_created = entry_ext->m_date_created;
		entry_new->m_date_changed = entry_ext->m_date_changed;

		// insert it into the diary:
		m_entries[ entry_new->m_date.m_date ] = entry_new;
		place_entry_in_chapters( entry_new );

		if( flag_import_tags )
		{
			Tagset &tags = entry_ext->get_tags();
			Tag *tag;

			for( Tagset::const_iterator iter = tags.begin(); iter != tags.end(); ++iter )
			{
				tag = ( m_tags.find( ( *iter )->get_name() ) )->second;
				entry_new->add_tag( tag );
			}
		}
		if( tag_all )
			entry_new->add_tag( tag_all );
	}

	return true;
}
