//
// C++ Implementation: kpgcompletionbox
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kpgcompletionbox.h"

// include files for Qt
#include <qapplication.h>

// include files for KDE
#include <kdebug.h>

KPGCompletionBox::KPGCompletionBox(QWidget *pParent, const char *szName)
 : KListBox(pParent, szName, WType_Popup)
{
    m_parent        = pParent;
    tabHandling     = true;
    down_workaround = false;
    upwardBox       = false;
    emitSelected    = true;

    setColumnMode( 1 );
    setLineWidth( 1 );
    setFrameStyle( QFrame::Box | QFrame::Plain );

    if ( pParent )
        setFocusProxy( pParent );
    else
        setFocusPolicy( NoFocus );

    setVScrollBarMode( Auto );
    setHScrollBarMode( AlwaysOff );

    connect( this, SIGNAL( doubleClicked( QListBoxItem * )), SLOT( slotActivated( QListBoxItem * )) );
    connect( this, SIGNAL( currentChanged( QListBoxItem * )), SLOT( slotCurrentChanged() ));
    connect( this, SIGNAL( clicked( QListBoxItem * )), SLOT( slotItemClicked( QListBoxItem * )) );
}


KPGCompletionBox::~KPGCompletionBox()
{
}

QStringList KPGCompletionBox::items() const
{
    QStringList list;

    const QListBoxItem* currItem = firstItem();

    while (currItem) {
        list.append(currItem->text());
        currItem = currItem->next();
    }

    return list;
}

void KPGCompletionBox::slotActivated( QListBoxItem *item )
{
    if ( !item )
        return;

    hide();
    emit activated( item->text() );
}

bool KPGCompletionBox::eventFilter( QObject *o, QEvent *e )
{
    int type = e->type();

    if ( o == m_parent ) {
        if ( isVisible() ) {
            if ( type == QEvent::KeyPress ) {
                QKeyEvent *ev = static_cast<QKeyEvent *>( e );
                switch ( ev->key() ) {
                    case Key_BackTab:
                        if ( tabHandling && (ev->state() == NoButton ||
                             (ev->state() & ShiftButton)) ) {
                            up();
                            ev->accept();
                            return true;
                        }
                        break;
                    case Key_Tab:
                        if ( tabHandling && (ev->state() == NoButton) ) {
                            down(); // Only on TAB!!
                            ev->accept();
                            return true;
                        }
                        break;
                    case Key_Down:
                        down();
                        ev->accept();
                        return true;
                    case Key_Up:
                        // If there is no selected item and we've popped up above
                        // our parent, select the first item when they press up.
                        if ( selectedItem() ||
                             mapToGlobal( QPoint( 0, 0 ) ).y() >
                             m_parent->mapToGlobal( QPoint( 0, 0 ) ).y() )
                            up();
                        else
                            down();
                        ev->accept();
                        return true;
                    case Key_Prior:
                        pageUp();
                        ev->accept();
                        return true;
                    case Key_Next:
                        pageDown();
                        ev->accept();
                        return true;
                    case Key_Escape:
                        canceled();
                        ev->accept();
                        return true;
                    case Key_Enter:
                    case Key_Return:
                        {
							/*if ( ev->state() & ShiftButton ) {
								hide();
								ev->accept();  // Consume the Enter event
								return true;
							}*/
							kdDebug() << "Enter !" << endl;
							int nCurrentItem = currentItem();
							if(nCurrentItem >= 0)
							{
								slotItemClicked(item(nCurrentItem));
								ev->accept();  // Consume the Enter event
								return true;
							}
                        }
                        break;
                    case Key_End:
                        if ( ev->state() & ControlButton )
                        {
                            end();
                            ev->accept();
                            return true;
                        }
                    case Key_Home:
                        if ( ev->state() & ControlButton )
                        {
                            home();
                            ev->accept();
                            return true;
                        }
                    default:
                        break;
                }
            }
            else if ( type == QEvent::AccelOverride ) {
                // Override any acceleartors that match
                // the key sequences we use here...
                QKeyEvent *ev = static_cast<QKeyEvent *>( e );
                switch ( ev->key() ) {
                    case Key_Down:
                    case Key_Up:
                    case Key_Prior:
                    case Key_Next:
                    case Key_Escape:
                    case Key_Enter:
                    case Key_Return:
                      ev->accept();
                      return true;
                      break;
                    case Key_Tab:
                    case Key_BackTab:
                        if ( ev->state() == NoButton ||
                            (ev->state() & ShiftButton))
                        {
                            ev->accept();
                            return true;
                        }
                        break;
                    case Key_Home:
                    case Key_End:
                        if ( ev->state() & ControlButton )
                        {
                            ev->accept();
                            return true;
                        }
                        break;
                    default:
                        break;
                }
            }

            // parent loses focus or gets a click -> we hide
            else if ( type == QEvent::FocusOut || type == QEvent::Resize ||
                      type == QEvent::Close || type == QEvent::Hide ||
                      type == QEvent::Move ) {
                hide();
            }
        }
    }

    // any mouse-click on something else than "this" makes us hide
    else if ( type == QEvent::MouseButtonPress ) {
        QMouseEvent *ev = static_cast<QMouseEvent *>( e );
        if ( !rect().contains( ev->pos() )) // this widget
            hide();

        if ( !emitSelected && currentItem() && !::qt_cast<QScrollBar*>(o) )
        {
          emit highlighted( currentText() );
          hide();
          ev->accept();  // Consume the mouse click event...
          return true;
        }
    }

    return KListBox::eventFilter( o, e );
}

void KPGCompletionBox::popup()
{
    if ( count() == 0 )
        hide();
    else {
        ensureCurrentVisible();
        bool block = signalsBlocked();
        blockSignals( true );
        setCurrentItem( 0 );
        blockSignals( block );
        clearSelection();
        if ( !isVisible() )
            show();
        else if ( size().height() != sizeHint().height() )
            calculateSize();
    }
}

void KPGCompletionBox::calculateSize()
{
    //int currentGeom = height();
    QPoint currentPos = pos();
    QRect geom = calculateGeometry();
    resize( geom.size() );
}

void KPGCompletionBox::show()
{
    upwardBox = false;
    if ( m_parent ) 
    {
        calculateSize();
        qApp->installEventFilter( this );
    }

    qApp->sendPostedEvents();
    KListBox::show();
}

void KPGCompletionBox::hide()
{
    if ( m_parent )
        qApp->removeEventFilter( this );
    cancelText = QString::null;
    KListBox::hide();
}

QRect KPGCompletionBox::calculateGeometry() const
{
    int x = 0, y = 0;
    int ih = itemHeight();
    int h = QMIN( 15 * ih, (int) count() * ih ) + 2*frameWidth();

    int w QMAX( KListBox::minimumSizeHint().width(), 200 );
    
    return QRect(x, y, w, h);
}

QSize KPGCompletionBox::sizeHint() const
{
    return calculateGeometry().size();
}

void KPGCompletionBox::down()
{
    int i = currentItem();

    if ( i == 0 && down_workaround ) {
        down_workaround = false;
        setCurrentItem( 0 );
        setSelected( 0, true );
        emit highlighted( currentText() );
    }

    else if ( i < (int) count() - 1 )
        setCurrentItem( i + 1 );
}

void KPGCompletionBox::up()
{
    if ( currentItem() > 0 )
        setCurrentItem( currentItem() - 1 );
}

void KPGCompletionBox::pageDown()
{
    int i = currentItem() + numItemsVisible();
    i = i > (int)count() - 1 ? (int)count() - 1 : i;
    setCurrentItem( i );
}

void KPGCompletionBox::pageUp()
{
    int i = currentItem() - numItemsVisible();
    i = i < 0 ? 0 : i;
    setCurrentItem( i );
}

void KPGCompletionBox::home()
{
    setCurrentItem( 0 );
}

void KPGCompletionBox::end()
{
    setCurrentItem( count() -1 );
}

void KPGCompletionBox::setTabHandling( bool enable )
{
    tabHandling = enable;
}

bool KPGCompletionBox::isTabHandling() const
{
    return tabHandling;
}

void KPGCompletionBox::setCancelledText( const QString& text )
{
    cancelText = text;
}

QString KPGCompletionBox::cancelledText() const
{
    return cancelText;
}

void KPGCompletionBox::canceled()
{
    if ( !cancelText.isNull() )
        emit userCancelled( cancelText );
    if ( isVisible() )
        hide();
}

/*
void KPGCompletionBox::insertItems( const QStringList& items, int index )
{
    bool block = signalsBlocked();
    blockSignals( true );
    insertStringList( items, index );
    blockSignals( block );
    down_workaround = true;
}
*/

void KPGCompletionBox::setItems(const QStringList& items, const KPGOidNameList &listOfCodeCompletionObjects)
{
    bool block = signalsBlocked();
    blockSignals( true );

    clear();

    QListBoxItem * pAfter = 0;
    for(QStringList::ConstIterator cit = items.begin(); cit != items.end(); ++cit)
    {
        // For each item, get it's icon in listOfCodeCompletionObjects
        const KPGOidName * pOidName = listOfCodeCompletionObjects.getConstItemByName(*cit);
        if(pOidName != 0)
        {
            if(pAfter)
                pAfter = new KPGCompletionBoxItem(this, * pOidName->icon(), (*cit), pAfter);
            else
                pAfter = new KPGCompletionBoxItem(this, * pOidName->icon(), (*cit));
        }
    }

    if ( isVisible() && size().height() != sizeHint().height() )
        calculateSize();

    blockSignals( block );
    down_workaround = true;
}

// This same as above for list with aliases
void KPGCompletionBox::setItems(const QStringList& items, const KPGOidNameAliasesList &listOfCodeCompletionObjects)
{
    bool block = signalsBlocked();
    blockSignals( true );

    clear();

    QListBoxItem * pAfter = 0;
    for(QStringList::ConstIterator cit = items.begin(); cit != items.end(); ++cit)
    {
        // For each item, get it's icon in listOfCodeCompletionObjects
        const KPGOidNameAliases * pOidNameAliases = listOfCodeCompletionObjects.getConstItemByName(*cit);
        if(pOidNameAliases != 0)
        {
            if(pAfter)
                pAfter = new KPGCompletionBoxItem(this, * pOidNameAliases->icon(), (*cit), pAfter);
            else
                pAfter = new KPGCompletionBoxItem(this, * pOidNameAliases->icon(), (*cit));
        }
    }

    if ( isVisible() && size().height() != sizeHint().height() )
        calculateSize();

    blockSignals( block );
    down_workaround = true;
}

void KPGCompletionBox::setItems(const QStringList& items, const KPGListTableColumns &listOfCodeCompletionTableColumns)
{
    bool block = signalsBlocked();
    blockSignals( true );

    clear();

    QListBoxItem * pAfter = 0;
    //kdDebug() << "------------------" << endl;
    for(QStringList::ConstIterator citItems = items.begin(); citItems != items.end(); ++citItems)
    {
        for(KPGListTableColumns::const_iterator citColumns = listOfCodeCompletionTableColumns.begin(); citColumns != listOfCodeCompletionTableColumns.end(); ++citColumns)
        {
           KPGTableColumnWizInfo column = (*citColumns);
           if((*citItems) == column.name())
           {
                //kdDebug() << (*citItems) << endl;
                
                if(pAfter)
                    pAfter = new KPGCompletionBoxItem(this, column.icon(), column.name(), pAfter);
                else
                    pAfter = new KPGCompletionBoxItem(this, column.icon(), column.name());
                
                break;
            }
        }
    }
    //kdDebug() << "------------------" << endl;

    if ( isVisible() && size().height() != sizeHint().height() )
        calculateSize();

    blockSignals( block );
    down_workaround = true;
}

/* Old way with hock to reduce flickring
void KPGCompletionBox::setItems( const QStringList& items, const KPGOidNameList &listOfCodeCompletionObjects )
{
    bool block = signalsBlocked();
    blockSignals( true );

    QListBoxItem* item = firstItem();
    if ( !item ) 
    {
        QListBoxItem * pAfter = 0;
        for(QStringList::ConstIterator cit = items.begin(); cit != items.end(); ++cit)
        {
            const KPGOidName * oidName = listOfCodeCompletionObjects.getItemByName(*cit);
            if(oidName != 0)
            {
                if(pAfter)
                    pAfter = new KPGCompletionBoxItem(this, * oidName->icon(), (*cit), pAfter);
                else
                    pAfter = new KPGCompletionBoxItem(this, * oidName->icon(), (*cit));
            }
        }
    }
    
    else 
    {
        //Keep track of whether we need to change anything,
        //so we can avoid a repaint for identical updates,
        //to reduce flicker
        bool dirty = false;
        for (QStringList::ConstIterator cit = items.constBegin(); it != items.constEnd(); ++it) 
        {
            if(item) 
            {
                const bool changed = ((KPGCompletionBoxItem*)item)->reuse(*cit);
                dirty = dirty || changed;
                item = item->next();
            }
            else 
            {
                dirty = true;
                
                //Inserting an item is a way of making this dirty                
                insertItem( new KPGCompletionBoxItem(*cit) );
                
            }
        }

        //If there is an unused item, mark as dirty -> less items now
        if ( item ) 
        {
            dirty = true;
        }

        QListBoxItem* tmp = item;
        while ( (item = tmp ) ) 
        {
            tmp = item->next();
            delete item;
        }

        if (dirty)
            triggerUpdate( false );
    }

    if ( isVisible() && size().height() != sizeHint().height() )
        calculateSize();

    blockSignals( block );
    down_workaround = true;
}
*/


void KPGCompletionBox::slotCurrentChanged()
{
    down_workaround = false;
}

void KPGCompletionBox::slotItemClicked( QListBoxItem *item )
{
    if ( item )
    {
        if ( down_workaround ) 
        {
            down_workaround = false;
            emit highlighted( item->text() );
        }

        hide();
        emit activated( item->text() );
    }
}

void KPGCompletionBox::setActivateOnSelect(bool state)
{
    emitSelected = state;
}

bool KPGCompletionBox::activateOnSelect() const
{
    return emitSelected;
}

void KPGCompletionBox::virtual_hook( int id, void* data )
{ KListBox::virtual_hook( id, data ); }

