/*
    collectionaccess.h - Access to the collection of ROMs

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#ifndef KAMEFUCOLLECTIONACCESS_H
#define KAMEFUCOLLECTIONACCESS_H

// Qt includes
#include <qobject.h>
#include <qvaluelist.h>

// KDE includes
#include <kdemacros.h>
#include <kurl.h>

class QSqlQuery;
class QStringList;

namespace Kamefu
{

class RomMetaInformation;
/**
 * @brief Singleton class access to database.
 *
 * This class handle all transactions between the application and the database.
 * Do not create this class directly, use CollectionAccess::self() instead.
 *
 * @author Michaël Larouche
 */
class KDE_EXPORT CollectionAccess : public QObject
{
	Q_OBJECT
public:
	/**
	 * This enum help to set the table mode.
	 * TablePermanent: Use the permanent tables.
	 * TableTemporary: Use the temporary tables.
	 */
	enum TableMode { TablePermanent = 0, TableTemporary = 1};

	/**
	 * d-tor.
	 */
	~CollectionAccess();

	/**
	 * Singleton access to this class.
	 */
	static CollectionAccess *self();

	/**
	 * Execute a SQL query, emit databaseError(const QString&) if the query fails.
	 * @param query SQL query to execute.
	 * @param errorMessage Error Message to display to the user.
	 * @return Executed SQL query as QSqlQuery, used to retrive entries if the query was a SELECT for example.
	 */
	QSqlQuery executeQuery(const QString &query, const QString &errorMessage = QString::null);

	/**
	 * Retrive the full url(path) of a rom item from the rom name.
	 * @param romName The name of the rom.
	 * @return the URL for the first ROM name found.
	 */
	QString getRomUrlFromRomName(const QString &romName);
	
	/**
	 * Retrieve ROM metainformation for the given ROM name.
	 * @deprecated Use getRometadataFromRomUrl instead, romName can result multiple results.
	 * @param romName The name of the ROM.
	 * @return the RomMetaInformation.
	 */
	Kamefu::RomMetaInformation getRomMetadataFromRomName(const QString &romName) KDE_DEPRECATED;
	
	/**
	 * Retrieve ROM metainformation for the given ROM URL.
	 * @param romUrl The URL of the ROM.
	 * @return the RomMetaInformation
	 */
	Kamefu::RomMetaInformation getRomMetadataFromRomUrl(const QString &romUrl);

	/**
	 * Retrieve system name from rom url.
	 * @param romUrl The rom url.
	 * @return the system name.
	 */
	QString getSystemNameFromRomUrl(const QString &romUrl);

	/**
	 * @brief Check if the current ROM url is in database.
	 *
	 * This method is used to do incremental build of the database.
	 * A transaction must be beginned to work property.
	 * If the method found the romUrl in database, it insert it into the rom_temp table.
	 *
	 * @param romUrl rom url to check.
	 * @return true if the the given URL is in database.
	 */
	bool isRomUrlInDatabase(const QString &romUrl);

	/**
	 * @brief If a transaction is in progress.
	 * @return true if a transaction is in progress.
	 */
	bool isTransactionInProgress() const;

public slots:
	/**
	 * Start the transaction process of updating the ROM database.
	 */
	void beginRomTransaction();
	/**
	 * End the transaction process. Commit the changes to database.
	 * @param commit True by default, Commit the changes to database.
	 */
	void endRomTransaction(bool commit = true);

	/**
	 * Initialize the database connection.
	 * If the database is empty, the tables will be created.
	 * @return false if anything goes wrong.
	 */ 
	bool init();

	/**
	 * @brief Insert a new ROM into the database.
	 * You must call beginRomTransaction() before calling this one.
	 * @param newRom RomMetaInformation to insert in database.
	 */
	void insertNewRom(const Kamefu::RomMetaInformation &newRom);
	
	/**
	 * @brief Modify an existing ROM in the database.
	 * Do nothing if a transaction is in progress.
	 * @param existingRom RomMetaInformation to modifiy in database.
	 */
	void modifyExistingRom(const Kamefu::RomMetaInformation &existingRom);

	/**
	 * @brief Modify common information for multiple ROMs
	 * Update multiple ROMs at time using common information(ex: same Developer, same Genre, etc...)
	 * Do nothing if a transaction is in progress.
	 * @param commonInformation RomMetaInformation that contains common information.
	 * @param romUrlList List of ROM url to modify.
	 */
	void modifyRoms(const Kamefu::RomMetaInformation &commonInformation, const KURL::List &romUrlList);
	/**
	 * @brief Retrieve all ROM from database (async)
	 *
	 * This method is asynchronous. Listen to 
	 * gettingRomMetaInformation() signal to actually 
	 * get a single RomMetaInformation.
	 */
	void retrieveAllRoms();
	
	/**
	 * @brief Retrieve ROM from database for a given query. (async)
	 *
	 * This method is asynchronous. Listen to 
	 * gettingRomMetaInformation() signal to actually 
	 * get a single RomMetaInformation.
	 * @param query ROM query to execute.
	 */
	void retrieveRoms(const QString &query);

	/**
	 * @brief Get the ROMs from database in a list. (sync)
	 * @return the list of all available ROM from database.
	 */
	QValueList<Kamefu::RomMetaInformation> getAllRomsList();
	
	/**
	 * @brief Get the metainformations for the given list of urls (sync)
	 * @param romUrlList List of ROM url to retrieve meta information.
	 * @return the list of metainformation retrieved.
	 */
	QValueList<Kamefu::RomMetaInformation> getRomList(const KURL::List &romUrlList);

	/**
	 * @brief Retrieve table entries
	 * 
	 * DO NOT USE THIS METHOD TO RETRIEVE A LIST OF ROMS.
	 * Query a table and return his entries.
	 * 
	 * The table must contains a field named "table"_name.
	 * @param table table name in lowercase.
	 * @return The table entries as QStringList.
	 */
	QStringList retrieveTableEntries(const QString &table);
	
	/**
	 * @brief Retrieve table entries with a filter
	 * 
	 * DO NOT USE THIS METHOD TO RETRIEVE A LIST OF ROMS.
	 * Query a table with a filter and return his entries.
	 * A filter will be defined like this: "table_id = (SELECT table_id FROM table WHERE table_name LIKE ('Value%')"
	 * and the filter will be added to the WHERE clause of the query.
	 * 
	 * The table must contains a field named "table"_name.
	 * @param table table name in lowercase.
	 * @param filterValue Filter on this value.
	 * @return The table entries as QStringList.
	 */
	QStringList retrieveTableEntriesFiltered(const QString &table, const QString &filterValue);
	
	/**
	 * @brief Remove a single ROM from the database
	 *
	 * This method will do nothing if a transaction is in progress.
	 * WARNING: You can't undo this operation.
	 * @param romUrl URL of ROM you want to remove from database
	 */
	void removeRomFromDatabase(const QString &romUrl);

signals:
	/**
	 * Emitted when a database error occurs.
	 * @param message Error message.
	 */
	void databaseError(const QString &message);

	/**
	 * This signal is emitted when a single Rom is
	 * retrived from the main SELECT query.
	 * @param rom RomMetaInformation retrived from SELECT.
	 * TODO: Maybe create a signal to send a list of RomMetaInformation instead of item by item.
	 */
	void gettingRomMetaInformation(const Kamefu::RomMetaInformation &rom);

	/**
	 * This signal is emitted when a async rom listing operationg is complete.
	 */
	void romListingFinished();

private:
	/**
	 * Disallow creation of this class. Use CollectionAccess::self() instead.
	 */
	CollectionAccess(QObject *parent = 0, const char *name = 0);

	/**
	 * @internal Create the tables. 
	 * If temp is true, it will create temp tables needed for transaction.
	 * @param tableMode Table mode. See TableMode enum.
	 */
	void createTables(int tableMode = 0);
	/**
	 * Clean the tables for all their entries.
	 * @param tableMode Table mode. See TableMode enum.
	 **/
	void cleanTables(int tableMode = 0);
	/**
	 * Drop(remove) the tables.
	 * @param tableMode Table mode. See TableMode enum.
	 */
	void dropTables(int tableMode = 0);

	/**
	 * Create the database connection using DatabasePreferences.
	 * @return false if anything goes wrong.
	 */
	bool createConnection();

	/**
	 * @internal
	 * This method exists because CREATE INDEX require a different syntax depending of the database driver.
	 * @param indexName name to give to the index.
	 * @param tableName Create the index for this table.
	 * @param fieldName Field in table to create a index for.
	 * @param indexLength Length of the index if required. (MySQL need it)
	 */
	void createIndex(const QString &indexName, const QString &tableName, const QString &fieldname, int indexLength = 0);
	/**
	 * @internal
	 * Helper method to create a new entry in foreign tables in the form id-name.
	 * @param fieldName This is the field to create a new value for it.
	 * @param tableName Table where the field is located.
	 */
	int createNewIndex(const QString &fieldName, const QString &tableName);
	/**
	 * @internal
	 * Retrive a ID from a foreign table based on researchFieldName and researchValue.
	 * Create a new entry in the give table if it can't found a ID.
	 *
	 * @param key This is the field name for the primary key(ID), what we are looking for.
	 * @param table Given table to look into.
	 * @param researchFieldName This is the fieldName used to do the research.
	 * @param researchValue Value to look for.
	 */ 
	int getForeignId(const QString &key, const QString &table, const QString &researchFieldName, const QString &researchValue);

private:
	static CollectionAccess *s_self;

	class Private;
	Private *d;

};

}

#endif
