/*
    kfile_gbx.cpp - GBX(Game Boy/Game Boy Color) Rom image KFile Plugin

    Copyright (c) 2006      by Michaël Larouche       <michael.larouche@kdemail.net>

     *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#include "kfile_gbx.h"

#include <kgenericfactory.h>
#include <kdebug.h>
#include <kmdcodec.h>

#include <qfile.h>
#include <qdatastream.h>

#include <nintendomakerlist.h>

// TODO: GB specific information.

typedef KGenericFactory<KGbxFilePlugin> kgbxfileFactory;

K_EXPORT_COMPONENT_FACTORY(kfile_gbx, kgbxfileFactory( "kfile_gbx" ))

KGbxFilePlugin::KGbxFilePlugin(QObject *parent, const char *name, const QStringList& args)
	: KFilePlugin(parent, name, args)
{
	//add the mimetype here:
	KFileMimeTypeInfo *info = addMimeTypeInfo( "application/x-rom-gbx" );

	KFileMimeTypeInfo::GroupInfo *groupGeneral = 0L;
	groupGeneral = addGroupInfo(info, "romGeneralInfo", i18n("General"));
	
	// General Group
	addItemInfo(groupGeneral, "internalName", i18n("Internal Name"), QVariant::String);
	addItemInfo(groupGeneral, "country", i18n("Country"), QVariant::String);
	addItemInfo(groupGeneral, "editor", i18n("Developer"), QVariant::String);
	addItemInfo(groupGeneral, "md5Hash", i18n("MD5 Hash"), QVariant::String);
}

bool KGbxFilePlugin::readInfo( KFileMetaInfo& info, uint /*what*/)
{
	KFileMetaInfoGroup groupGeneral = appendGroup(info, "romGeneralInfo");

	QByteArray internalName(15), makerCode(2);
	Q_UINT8 oldMakerCode, countryFlag;

	QString md5Hash, countryName(i18n("Unknown")), editorName(i18n("Unknown"));

	QFile gbx_file(info.path());

	if(gbx_file.open(IO_ReadOnly))
	{
		QDataStream stream(&gbx_file);
		stream.setByteOrder(QDataStream::LittleEndian);
		
		// Go to header information.
		gbx_file.at(0x134);

		// Read the game title (internalName)
		stream.readRawBytes(internalName.data(), internalName.size());
		// Dummy getch to advance from one byte.
		gbx_file.getch();
		// Read the maker code
		stream.readRawBytes(makerCode.data(), makerCode.size());

		// Go to country flag.
		gbx_file.at(0x14a);
		stream >> countryFlag;
		// Read old maker code;
		stream >> oldMakerCode;

		gbx_file.reset(); // Make sure that the md5 hash read the entire ROM.

		// Generate the MD5 Hash
		KMD5 context(0L);
		context.update(gbx_file);
		md5Hash = context.hexDigest();
	
		// Get the information
		// Get the editor name from the maker code
		int result=0;
		if( makerCode[0] != 0 )
		{
			kdDebug() << k_funcinfo << "Read makerCode: " << QString(makerCode) << endl;
			bool ok;
			QString sMaker(makerCode);
			result = sMaker.toInt(&ok, 16);
		}
		else
		{
			result = oldMakerCode;
		}	
		kdDebug() << k_funcinfo << "Convert editorIndex: " << QString::number(result, 16) << endl;

		result = (result >> 4) * 36 + (result & 0x0f);
		if(result > 0 || result <= nintendoMakerListLength)
			editorName = nintendoMakerList[result];

		kdDebug() << k_funcinfo << "Editor name: " << editorName << endl;

		if(countryFlag)
		{
			countryName = i18n("America/Europe");
		}
		else
		{
			countryName = i18n("Japan");
		}

		gbx_file.close();
	}

	// General group
	appendItem(groupGeneral, "internalName", QString(internalName));
	appendItem(groupGeneral, "country", countryName);
	appendItem(groupGeneral, "editor", editorName);
	appendItem(groupGeneral, "md5Hash", md5Hash);
	
	return true;
}

#include "kfile_gbx.moc"
