/*
    kamefumainwindow.cpp - Kamefu Application Main Window

    Copyright (c) 2005-2006 by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/

#include "kamefumainwindow.h"
#include "kamefuview.h"

// Qt includes
#include <qeventloop.h>
#include <qapplication.h>
#include <qlabel.h>
#include <qstringlist.h>
#include <qtimer.h>
#include <qvaluelist.h>
#include <qfile.h>

// KDE includes
#include <kmainwindow.h>
#include <kdebug.h>
#include <klocale.h>
#include <kaction.h>
#include <kiconloader.h>
#include <kstdaction.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <kfiledialog.h>
#include <kio/netaccess.h>
#include <kprogress.h>
#include <kstatusbar.h>
#include <kdirselectdialog.h>
#include <khtmlview.h>

// For configuration dialog.
#include <ksettings/dialog.h>
#include <kwin.h>
#include <kcmultidialog.h>

// Kamefu include
#include <collectionaccess.h>
#include <collectionbuilder.h>
#include <systemmanager.h>
#include <kamefupreferences.h>
#include <gamelauncher.h>

#include "kamefuapplication.h"
#include "firsttimewizard.h"

class KamefuMainWindow::Private
{
public:
	Private()
		: m_view(0), progressBuildCollection(0), cBuilder(0), launcher(0), buildProgress(0), totalItems(0)
	{}

	KamefuView *m_view;
	KProgress *progressBuildCollection;
	Kamefu::CollectionBuilder *cBuilder;
	Kamefu::GameLauncher *launcher;

	int buildProgress;
	int totalItems;

	// NOTE: This is in Privte for a reason, this is a personal hack for me.
	// Will be made as a plugin in 0.2
	void updateImageInDatabase(const KURL &imageDirectory, int type);
};

KamefuMainWindow::KamefuMainWindow()
    : KMainWindow(0, "KamefuMainWindow"), d(new Private())
{
	// Init Collection acces.
	Kamefu::CollectionAccess::self()->init();

	// Set the XML GUI file.
	setXMLFile("kamefuui.rc");
	
	// Create the main view.
	d->m_view = new KamefuView(this, "m_view");
	
	setCentralWidget(d->m_view);

	// Create the KProgress in statusBar.
	d->progressBuildCollection = new KProgress(statusBar());
	d->progressBuildCollection->hide();
	
	// Create the CollectionBuilder.
	d->cBuilder = new Kamefu::CollectionBuilder(this);
	// Create the Game Launcher.
	d->launcher = new Kamefu::GameLauncher(this);
	
	// Setup the actions.
	setupActions();
	// Setup the Qt signals/slots connection
	setupConnections();

	// Restore the size from settings.
	readSettings();

	// Check if it's the first launch of Kamefu. Build the initial collection.
	checkFirstTime();
}

KamefuMainWindow::~KamefuMainWindow()
{
	kdDebug() << k_funcinfo << endl;

	delete d;
}

void KamefuMainWindow::setupActions()
{
	KStdAction::open(this, SLOT(fileOpen()), actionCollection());
	KStdAction::quit(kapp, SLOT(quit()), actionCollection());

	setStandardToolBarMenuEnabled(true);
	createStandardStatusBarAction();

	KStdAction::preferences(this, SLOT(showPreferences()), actionCollection());

	new KAction(i18n("First time Wizard"), SmallIcon("wizard"), 0, this, SLOT(showFirstTimeWizard()), actionCollection(), "collectionWizard");
	new KAction(i18n("Update Collection"), SmallIcon("reload"), 0, this, SLOT(buildCollection()), actionCollection(), "collectionUpdate");
	// TODO: 0.2 Make this a plugin
	new KAction(i18n("Look for titlescreens..."), SmallIcon("image"), 0, this, SLOT(collectionTitlescreenLookup()), actionCollection(), "collectionTitlescreenLookup");
	new KAction(i18n("Look for screenshots.."), SmallIcon("image"), 0, this, SLOT(collectionScreenshotLookup()), actionCollection(), "collectionScreenshotLookup");

	createGUI();
}

void KamefuMainWindow::setupConnections()
{
	connect(Kamefu::CollectionAccess::self(), SIGNAL(databaseError(const QString& )), this, SLOT(displayError(const QString&)));
	connect(Kamefu::CollectionAccess::self(), SIGNAL(gettingRomMetaInformation(const Kamefu::RomMetaInformation& )), d->m_view, SLOT(addRomListViewItem(const Kamefu::RomMetaInformation& )));
	connect(Kamefu::CollectionAccess::self(), SIGNAL(romListingFinished()), this, SLOT(updateCollectionTotal()));

	connect(d->m_view, SIGNAL(romExecuted(const QString& )), this, SLOT(executeRom(const QString& )));
	connect(d->m_view, SIGNAL(romsDeletedOrTrashed()), this, SLOT(updateCollectionTotal()));

	connect(d->cBuilder, SIGNAL(itemListed()), this, SLOT(updateBuildProgress()));
	connect(d->cBuilder, SIGNAL(beginListDirectory( int )), this, SLOT(buildBegin( int )));
	connect(d->cBuilder, SIGNAL(collectionBuildFinished()), this, SLOT(fillRomList()));
	connect(d->cBuilder, SIGNAL(collectionBuildFinished()), d->m_view, SLOT(fillFilters()));
	connect(d->cBuilder, SIGNAL(collectionBuildFinished()), this, SLOT(buildComplete()));
}

bool KamefuMainWindow::queryExit()
{
	kdDebug() << k_funcinfo << endl;

	saveSettings();
	d->m_view->saveSettings();

	Kamefu::SystemManager::self()->shutdown();

	// Always delete this main window.
	deleteLater();

	return true;
}

void KamefuMainWindow::checkFirstTime()
{
	if( KamefuPreferences::self()->kamefuFirstTime() )
	{
		showFirstTimeWizard();
	}
	else
	{
		show();
		fillRomList();
	}
}

void KamefuMainWindow::showFirstTimeWizard()
{
	FirstTimeWizard *wizard = new FirstTimeWizard(this, "firsttime_wizard");
	wizard->show();
	connect(wizard, SIGNAL(accepted()), this, SLOT(wizardAccepted()));
}

void KamefuMainWindow::wizardAccepted()
{
	kdDebug() << k_funcinfo << endl;

	show();

	if( KamefuPreferences::self()->kamefuFirstTime() )
	{
		KamefuPreferences::self()->setKamefuFirstTime(false);
		KamefuPreferences::self()->writeConfig();
	}

	// Don't start the collection building now, wait a little to show the main window.
	QTimer::singleShot(1000, this, SLOT(buildCollection()));
}

void KamefuMainWindow::fileOpen()
{
	// Get all available mime types from System manager
	QStringList romMimeType = Kamefu::SystemManager::self()->getRegistredMimeTypes();
	romMimeType.append( "application/x-zip" ); // Add ZIP file.
	KURL url = KFileDialog::getOpenURL(QString::null, romMimeType.join(" "), this, i18n("Open Location"));
	
	if(url.isEmpty())
		return;

	if(url.isLocalFile())
	{
		executeRom( url.path() );
	}
	else
	{
		KMessageBox::queuedMessageBox(this, KMessageBox::Error, i18n("It doesn't support loading ROM from remote file at the moment."));
	}
}

void KamefuMainWindow::showPreferences()
{
	KSettings::Dialog *settings = new KSettings::Dialog( KSettings::Dialog::Static, this );
	settings->show();
}

void KamefuMainWindow::displayError(const QString &errorMessage)
{
	KMessageBox::queuedMessageBox(this, KMessageBox::Error, errorMessage, i18n("Ooops!"));
}

void KamefuMainWindow::executeRom(const QString &romUrl)
{
	if( !d->launcher->launchGame(romUrl) )
	{
		// TODO: Display ROM name from database instead.
		KMessageBox::queuedMessageBox(this, KMessageBox::Error, i18n("An error occured when trying to launch this rom:\n%1").arg( KURL::fromPathOrURL(romUrl).fileName() ), i18n("Kamefu"));
	}
}

void KamefuMainWindow::fillRomList()
{
	d->m_view->clearRomList();
	Kamefu::CollectionAccess::self()->retrieveAllRoms();
}

void KamefuMainWindow::buildBegin(int numberItems)
{
	kdDebug() << k_funcinfo << QString::number(numberItems) << endl;
	
	d->totalItems += numberItems;
	d->progressBuildCollection->setTotalSteps(d->totalItems);
}

void KamefuMainWindow::updateBuildProgress()
{
	d->progressBuildCollection->setProgress(++d->buildProgress);
	KApplication::eventLoop()->processEvents( QEventLoop::ExcludeUserInput );
}

void KamefuMainWindow::buildComplete()
{
	kdDebug() << k_funcinfo << endl;
	
	d->progressBuildCollection->setProgress(0);

	statusBar()->removeWidget(d->progressBuildCollection);
	d->progressBuildCollection->hide();
	statusBar()->removeItem(0);
	statusBar()->clear();
}

void KamefuMainWindow::buildCollection()
{
	kdDebug() << k_funcinfo << "Building collection." << endl;

	d->buildProgress = 0;
	d->totalItems = 0;

	statusBar()->insertItem(i18n("Building collection..."), 0);

	statusBar()->addWidget(d->progressBuildCollection);
	d->progressBuildCollection->show();
	
	// HACK: Display the statusBar each time.
	KApplication::eventLoop()->processEvents( QEventLoop::ExcludeUserInput );

	// Fill KURL list with selected Directories from Preferences.
	KURL::List urlList;
	QStringList selectedDirectories = KamefuPreferences::self()->collectionDirectories();
	QStringList::const_iterator it;
	QStringList::const_iterator itEnd = selectedDirectories.end();

	for( it = selectedDirectories.begin(); it != itEnd; ++it )
	{
		urlList.append( KURL(*it) );
	}

	d->cBuilder->setDirectoryList( urlList );
	d->cBuilder->startBuild( KamefuPreferences::self()->collectionRecursive() ); // recursive
}

void KamefuMainWindow::readSettings()
{
	if( !KamefuPreferences::self()->mainWindowSize().isEmpty() )
		resize( KamefuPreferences::self()->mainWindowSize() );
}

void KamefuMainWindow::saveSettings()
{
	// Save size preferences
	KamefuPreferences::self()->setMainWindowSize( size() );
	KamefuPreferences::self()->writeConfig();
}

void KamefuMainWindow::updateCollectionTotal()
{
	if( statusBar()->hasItem(1) )
	{
		statusBar()->changeItem( i18n( "%n ROM", "%n ROMs", d->m_view->collectionSize() ), 1 );
	}
	else
	{
		statusBar()->insertItem( i18n( "%n ROM", "%n ROMs", d->m_view->collectionSize() ) , 1, 0, true );
	}
}

// TODO: 0.2 Make this a plugin.
void KamefuMainWindow::collectionTitlescreenLookup()
{
	KURL titlescreenDir = KDirSelectDialog::selectDirectory(QString::null, true, this, i18n("Select titlescreen image directory."));
	if( !titlescreenDir.isEmpty() )
	{
		d->updateImageInDatabase(titlescreenDir, 0); // 0 = Titlescreen
	}
}

// TODO: 0.2 Make this a plugin.
void KamefuMainWindow::collectionScreenshotLookup()
{
	KURL screenshotDir = KDirSelectDialog::selectDirectory(QString::null, true, this, i18n("Select screenshot image directory."));
	if( !screenshotDir.isEmpty() )
	{
		d->updateImageInDatabase(screenshotDir, 1); // 1 = Screenshot
	}
}

// TODO: 0.2 Make this a plugin
// NOTE: Yes I know this is hacky and bad written(not follow my code standards), but this is on purpose. Will be rewritten better as a plugin.
void KamefuMainWindow::Private::updateImageInDatabase(const KURL &imageDir, int type)
{
	QValueList<Kamefu::RomMetaInformation> romList = Kamefu::CollectionAccess::self()->getAllRomsList();
	QValueList<Kamefu::RomMetaInformation>::iterator it, itEnd = romList.end();
	for( it = romList.begin(); it != itEnd; ++it )
	{
		Kamefu::RomMetaInformation tempRomInfo = *it;
		KURL romUrl(tempRomInfo.url());
		QString filename = romUrl.fileName();
		QString pngFilename = imageDir.path() + "/" + filename.replace(filename.findRev(".")+1, filename.length() -filename.findRev(".") -1, "png");

		if( QFile::exists(pngFilename) )
		{
			switch(type)
			{
				// Titlescreen
				case 0:
					tempRomInfo.setTitlescreenPath(pngFilename);
					break;
				// Screenshot
				case 1:
					tempRomInfo.setScreenshotPath(pngFilename);
					break;
				default:
					continue;
			}
			Kamefu::CollectionAccess::self()->modifyExistingRom( tempRomInfo );
		}
	}
}

#include "kamefumainwindow.moc"
