/*
    collectionviewer.cpp - Show items from collection.

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#include "collectionviewer.h"

// Qt includes
#include <qlayout.h>
#include <qptrlist.h>

// KDE includes
#include <kdebug.h>
#include <kurl.h>
#include <klocale.h>
#include <klistviewsearchline.h>

// Kamefu includes
#include "collectionlistview.h"
#include <romlistviewitem.h>

class CollectionViewer::Private
{
public:
	Private()
	{}

	CollectionListView *collectionListView;
	KListViewSearchLineWidget *searchWidget;
	RomListViewItem *latestSelectedItem;
};

CollectionViewer::CollectionViewer(QWidget *parent, const char *name)
	: QWidget(parent, name), d(new Private)
{
	QVBoxLayout *layout = new QVBoxLayout(this);
	
	d->collectionListView = new CollectionListView(this, "collectionListView");

	connect(d->collectionListView, SIGNAL(romDoubleClicked(const QString&)), this, SLOT(collectionListExecuted(const QString&)));
	connect(d->collectionListView, SIGNAL(romClicked(const QString&)), this, SLOT(collectionListSelected(const QString&)));
	// Dispatch these menu calls.
	connect(d->collectionListView, SIGNAL(editRoms(const KURL::List&)), this, SIGNAL(itemEditMetadata(const KURL::List&)));
	connect(d->collectionListView, SIGNAL(deleteRoms(const KURL::List&)), this, SIGNAL(itemDeleteRoms( const KURL::List& )));
	connect(d->collectionListView, SIGNAL(trashRoms(const KURL::List&)), this, SIGNAL(itemTrashRoms(const KURL::List&)));
	
	d->searchWidget = new KListViewSearchLineWidget(d->collectionListView, this, "searchWidget");

	layout->addWidget(d->searchWidget);
	layout->addWidget(d->collectionListView);
}


CollectionViewer::~CollectionViewer()
{
	delete d;
}

void CollectionViewer::clear()
{
	d->collectionListView->clear();
}

void CollectionViewer::addItem(const Kamefu::RomMetaInformation &romMetaInfo)
{
	new RomListViewItem(d->collectionListView, romMetaInfo);
}

void CollectionViewer::updateSelectedRomInformation(const QString &romUrl)
{
	d->latestSelectedItem = static_cast<RomListViewItem*>(d->collectionListView->selectedItems().first());
	if(d->latestSelectedItem && d->latestSelectedItem->romUrl() == romUrl)
	{
		kdDebug() << k_funcinfo << "Updating listview item for ROM: " << romUrl << endl;
		d->latestSelectedItem->refreshInfo();
	}
}

void CollectionViewer::updateSelectedRoms()
{
	QPtrList<QListViewItem> roms = d->collectionListView->selectedItems();

	QPtrListIterator<QListViewItem> itr(roms);
	QListViewItem *rom;
	while( (rom = itr.current()) != 0 )
	{
		RomListViewItem *romItem = static_cast<RomListViewItem*>(rom);
		if(romItem)
		{
			romItem->refreshInfo();
		}
		++itr;
	}
}

void CollectionViewer::collectionListExecuted(const QString &url)
{
	d->latestSelectedItem = static_cast<RomListViewItem*>(d->collectionListView->selectedItems().first());
	emit itemExecuted(url);
}

void CollectionViewer::collectionListSelected(const QString &url)
{
	d->latestSelectedItem = static_cast<RomListViewItem*>(d->collectionListView->selectedItems().first());
	emit itemSelected(url);
}

void CollectionViewer::removeSelectedItems()
{
	QPtrList<QListViewItem> roms = d->collectionListView->selectedItems();

	QPtrListIterator<QListViewItem> itr(roms);
	QListViewItem *rom;
	while( (rom = itr.current()) != 0 )
	{
		delete rom;
		++itr;
	}

	d->collectionListView->triggerUpdate();
}

int CollectionViewer::collectionSize() const
{
	return d->collectionListView->childCount();
}
#include "collectionviewer.moc"
