#ifndef GADU_H
#define GADU_H

#include <qhostaddress.h>
#include <qvaluelist.h>

#include "libgadu.h"
#include "protocol.h"
#include "status.h"
#include "usergroup.h"
#include "userlist.h"

typedef uin_t UinType;

class QTimer;
class GaduSocketNotifiers;

// ------------------------------------
//            UinsList
// ------------------------------------

class UinsList : public QValueList<UinType>
{
	public:
		/**
			konstruuje obiekt UinsList
		**/
		UinsList();

		/**
			konstruuje obiekt UinsList, inicjujc go uinem
		**/
		UinsList(UinType uin);

		/**
			konstruuje obiekt UinsList na podstawie acucha "uins" skadajcego
			si z oddzielonych przecinkami Uinw
		**/
		UinsList(const QString &uins);

		/**
			konstruuje obiekt UinsList na podstawie "list"
			przeksztaacajc kady element do typy UinType
		**/
		UinsList(const QStringList &list);

		bool equals(const UinsList &uins) const;
		void sort();
		QStringList toStringList() const;
};

// ------------------------------------
//             UserStatus
// ------------------------------------

class GaduStatus : public UserStatus
{
//	Q_OBJECT

	public:
		GaduStatus();
		virtual ~GaduStatus();

		GaduStatus &operator = (const UserStatus &copyMe);

		virtual QPixmap pixmap(eUserStatus status, bool has_desc, bool mobile) const;
		virtual QString pixmapName(eUserStatus status, bool has_desc, bool mobile) const;

		int toStatusNumber() const;
		static int toStatusNumber(eUserStatus status, bool has_desc);

		void fromStatusNumber(int statusNumber, const QString &description);

		virtual UserStatus *copy() const;
		virtual QString protocolName() const;
};

// ------------------------------------
//              Search
// ------------------------------------

struct SearchResult
{
	QString Uin;
	QString First;
	QString Last;
	QString Nick;
	QString Born;
	QString City;
	QString FamilyName;
	QString FamilyCity;
	int Gender;
	GaduStatus Stat;

	SearchResult();
	SearchResult(const SearchResult &);
	void setData(const char *uin, const char *first, const char *last, const char *nick, const char *born,
		const char *city, const char *familyName, const char *familyCity, const char *gender, const char *status);
};

typedef QValueList<SearchResult> SearchResults;

struct SearchRecord
{
	int Seq;
	int FromUin;
	QString Uin;
	QString FirstName;
	QString LastName;
	QString NickName;
	QString City;
	QString BirthYearFrom;
	QString BirthYearTo;
	int Gender;
	bool Active;
	bool IgnoreResults;

	SearchRecord();
	virtual ~SearchRecord();

	void reqUin(const QString& uin);
	void reqFirstName(const QString& firstName);
	void reqLastName(const QString& lastName);
	void reqNickName(const QString& nickName);
	void reqCity(const QString& city);
	void reqBirthYear(const QString& birthYearFrom, const QString& birthYearTo);
	void reqGender(bool female);
	void reqActive();

	void clearData();
};


enum GaduError
{
	ConnectionServerNotFound,
	ConnectionCannotConnect,
	ConnectionNeedEmail,
	ConnectionInvalidData,
	ConnectionCannotRead,
	ConnectionCannotWrite,
	ConnectionIncorrectPassword,
	ConnectionTlsError,
	ConnectionIntruderError,
	ConnectionUnavailableError,
	ConnectionUnknow,
	ConnectionTimeout,
	Disconnected
};

// ------------------------------------
//			GaduFormater
// ------------------------------------
class GaduFormater
{
	private:
		struct attrib_formant
		{
			QString name;
			QString value;
			attrib_formant() : name(), value() {}
		};

		struct richtext_formant
		{
			struct gg_msg_richtext_format format;
			struct gg_msg_richtext_color color;
			struct gg_msg_richtext_image image;
		};

		static QString stripHTMLFromGGMessage(const QString &msg);

		/**
		 * Translates QValueList with formants into flat buffer on heap
		 *
		 * Precondition - formats_length must contain valid length of result buffer
		 */
		static unsigned char *allocFormantBuffer(const QValueList<struct richtext_formant> &formants, unsigned int &formats_length);

	public:
		static QString formatGGMessage(const QString &msg, unsigned int formats_length, void *formats, UinType sender);
		static QString unformatGGMessage(const QString &msg, unsigned int &formats_length, unsigned char *&formats);
};

// ------------------------------------
//            GaduProtocol
// ------------------------------------

/**
	Klasa do obsugi protokou Gadu-Gadu
**/
class GaduProtocol : public Protocol
{
	Q_OBJECT

	private:

		/**
			@enum Mode
			@short Tryb pracy obiektu
			@see TokenSocketNotifiers
			@see gotToken
			@see needTokenValue
			@see register
			@see doRegister
			@see unregister
			@see doUnregister
			@see remindPassword
			@see doRemindPassword
			@see changePassword
			@see doChangePassword

			Kilka operacji w protokole Gadu-Gadu wymaga od klienta wysania do serwera, oprcz
			standardowych pakietw, tak zwanego 'tokena'. Najpierw pobierany jest z serwera identyfikator
			tokena oraz obrazek. Nastpnie z obrazka odczytywana jest wartoc tokena (w zaoeniach
			miao to zapobiega rejestrowaniu nowych uytkownikw przez automaty.

			Sloty register, unregister, remindPassword i changePassword inicjuj pobieranie tokena
			i ustalaj warto pola Mode. Pobieranie obrazka realizowane jest przez klas
			TokenSocketNotifiers. Po pobraniu wywoywany jest slot gotToken, ktry na podstawie wartoci
			pola Mode wywouje jedn z funkcji doRegister, doUnregister, doRemindPassword i doChangePassword.
		**/
		enum
		{
			/** Rejestrowanie nowego uytkownika **/
			Register,
			/** Wyrejestrowywanie istniejcego uytkownika **/
			Unregister,
			/** Przypominanie hasa **/
			RemindPassword,
			/** Zmienianie hasa **/
			ChangePassword
		} Mode;

		/** Identyfikator uytkownika **/
		UinType DataUin;
		/** e-mail uytkownika **/
		QString DataEmail;
		/** stare haso uytkownika **/
		QString DataPassword;
		/** nowe haso uytkownika **/
		QString DataNewPassword;
		/** identyfikator tokena **/
		QString TokenId;
		/** warto tokena **/
		QString TokenValue;

		/** Serwery, z ktrymi aczy si obiekt. **/
		static QValueList<QHostAddress> ConfigServers;
		/** Numer serwera, do ktrego obiekt ostatnio prbowa si podczy. **/
		unsigned int ServerNr;
		/** Adres serwera, do ktrego obiekt jest podczony. **/
		QHostAddress ActiveServer;

		/** IP serwera, do ktrego udao si podczy ostatnim razem **/
		QHostAddress lastServerIP;

		/** port serwera, do ktrego udao si podczy ostatnim razem **/
		int lastServerPort;

		/** czy w procedurze czenia mamy korzysta z informacji o IP/portu ostatniego serwera? **/
		bool useLastServer;

		int lastTriedServerPort;

		/** Parametry logowania - wymagane przez bibliotek libgadu **/
		struct gg_login_params LoginParams;

		/** Sesja poczenia - wymagane przez bibliotek libgadu **/
		gg_session* Sess;

		/** liczba da obrazkw wysanych w cigu ostatniej minuty**/
		unsigned int sendImageRequests;

		/** numer sekwencyjny ostatnio wysanej wiadomoci **/
		int seqNumber;

		/** czy jestemy w trakcie czenia si z serwerem **/
		bool whileConnecting;

		QHostAddress DccExternalIP;

		/**
			Klasa gniazdek czca si z serwerem. Wysya sygnay po wystpieniu zdarzenia protokou
			(poczenie, zerwanie poczenia, nowa wiadomo).

			@see GaduSocketNotifiers
		**/
		GaduSocketNotifiers *SocketNotifiers;

		/**
			Zegar pingujcy serwer.
		**/
		QTimer* PingTimer;

		QTimer* SendUserListTimer;

		/**
			Zmienna ustawiana w zalenoci od tego, czy wysyamy list kontaktw na serwer
			czy te usuwamy j z tego serwera. Zakoczenie obydwu tych czynnoci wywouje
			sygna podpity do slotu userListReplyReceived, ktry w zalenoci od wartoci
			tego pola wywouje userListCleared albo userListExported.

			@see userListReplyReceived
			@see userListCleared
			@see userListExported
		**/
		bool UserListClear;

		/**
			Lista uytkownikw pobrana z serwera w postaci acucha. Warto ustalana w slocie
			userListReplyReceived.

			@see userListReplyReceived
		**/
		QString ImportReply;

		/**
			Ustawianie parametrw poczenia proxy. Metoda wywoywana podczas logowania.

			@see login
		**/
		void setupProxy();

		/**
			Za pomoc klasy TokenSocketNotifiers metoda pobiera z serwera GaduGadu token wraz
			z identyfikatorem. Pobrany token jest obsugiwany za pomoc slota gotToken,
			ktry pobiera warto tokena emitujc sygna needTokenValue i nastpnie wywoujc
			jedn z metod  doRegisterAccount, doUnregisterAccount(), doRemindPassword(),
			doChangePassword() na podstawie wartoci pola Mode.

			@see TokenSocketNotifiers
			@see gotToken
			@see doRegisterAccount
			@see doUnregisterAccount
			@see doRemindPassword
			@see doChangePassword
			@see Mode
		**/
		void getToken();

		/**
			Rejestruje nowe konto. Wywoywane przez gotToken (ktre jest wywoane porednio przez
			registerAccount). Korzysta z pomocy PubdirSocketNotifiers oraz slotu registerDone,
			ktry emituje sygna registered.

			@see registerAccount
			@see registered
			@see registerDone
			@see gotToken
		**/
		void doRegisterAccount();

		/**
			Wyrejestrowuje konto. Wywoywane przez gotToken (ktre jest wywoane porednio przez
			unregisterAccount). Korzysta z pomocy PubdirSocketNotifiers oraz slotu unregisterDone,
			ktry emituje sygna unregistered.

			@see unregisterAccount
			@see unregistered
			@see unregisterDone
			@see gotToken
		**/
		void doUnregisterAccount();

		/**
			Przypomina haso. Wywoywane przez gotToken (ktre jest wywoane porednio przez
			remindPassword). Korzysta z pomocy PubdirSocketNotifiers oraz slotu remindDone,
			ktry emituje sygna reminded.

			@see remindPassword
			@see reminded
			@see remindDone
			@see gotToken
		**/
		void doRemindPassword();

		/**
			Zmienia haso. Wywoywane przez gotToken (ktre jest wywoane porednio przez
			changePassword). Korzysta z pomocy PubdirSocketNotifiers oraz slotu changePasswordDone,
			ktry emituje sygna passwordChanged.

			@see changePassword
			@see passwordChanged
			@see changePasswordDone
			@see gotToken
		**/
		void doChangePassword();

		GaduProtocol(const GaduProtocol &) : Protocol(QString::null, QString::null) {}
		GaduProtocol &operator=(const GaduProtocol &){return *this;}

	private slots:

		/**
			Loguje si do serwera Gadu-Gadu. Po uruchomieniu emituje sygna connecting. Parametry
			logowania odczytuje z konfiguracji, status logowania pobiera z pola NextStatus.
			czc si, wybiera kolejne serwery (w przypadku nieudanego poczenia) wykorzystujc
			pola ConfigServers i i ServerNr.

			Po poprawnym zalogowaniu wywoywany jest slot connectedSlot, ktry emituje sygna
			GaduProtocol::connected

			Metod mona uruchomi porednio poprzez wywoanie typu gadu->status().setOnline(),
			ktre wywoa slot iWantToGoOnline, ktry z kolei (gdy stwierdzi, e nie jestemy zalogowani)
			wywoa procedur.

			@see connecting
			@see connected
			@see connectedSlot
			@see NextStatus
			@see ConfigServers
			@see ServerNr
		**/
		void login();

		/**
			Wywoywany po zarejestrowaniu konta. Emituje registered/

			@see registerAccount
			@see doRegisterAccount
			@see registered
		**/
		void registerDone(bool ok, struct gg_http *);

		/**
			Wywoywany po wyrejestrowaniu konta. Emituje unregistered.

			@see unregisterAccount
			@see doUnregisterAccount
			@see unregistered
		**/
		void unregisterDone(bool ok, struct gg_http *);

		/**
			Wywoywany po przypomnieniu hasa. Emituje reminded.

			@see remindPassword
			@see doRemindPassword
			@see reminded
		**/
		void remindDone(bool ok, struct gg_http *);

		/**
			Wywoywany po zmianie hasa. Emituje passwordChanged.

			@see changePassword
			@see doChangePassword
			@see passwordChanged
		**/
		void changePasswordDone(bool ok, struct gg_http *);

		/**
			Slot wywoywany, gdy pobieranie tokena si nie udao.

			@see getToken
		**/
		void tokenError();
		/**
			Slot wywoywany, gdy pobieranie tokena si powiodo. Emituje needTokenValue

			@see getToken
			@see needTokenValue
		**/
		void gotToken(QString, QPixmap);

		/**
			Slot wywoywany po poczeniu z serwerem. Emituje connected i wcza pingowanie
			serwera.

			@see connected
		**/
		void connectedSlot();

		/**
			Slot wywoywany po rozczeniu z serwerem. Emituje disconnected i wycza pingowanie
			serwera.

			@see disconnected
		**/
		void disconnectedSlot();

		/**
			Slot wywoywany po zerwaniu poczenia przez serwer. Wywouje disconnectedSlot(),
			oraz dodatkowo zmienia NextStatus.
		**/
		void socketDisconnectedSlot();


		/**
			Slot wywoywany po przekroczeniu czasu poczenia. Prbuje poaczy ponownie.
		**/
		void connectionTimeoutTimerSlot();

		/**
			Slot wywoywane po wystpieniu bdu poczenia. Emituje disconnected i error.

			@see error
			@see disconnected
		**/
		void errorSlot(GaduError);

		/**
			Slot wywoywany po otrzymaniu obrazka od serwera. Emituje imageReceivedAndSaved

			@see imageReceivedAndSaved
		**/
		void imageReceived(UinType sender, uint32_t size, uint32_t crc32,
			const QString &filename, const char *data);

		/**
			Slot wywoywany po otrzymaniu proby o obrazek od serwera. Wysya obrazek.
		**/
		void imageRequestReceivedSlot(UinType, uint32_t, uint32_t);

		/**
			Slot wywoywany po otrzymaniu wiadomoci od serwera.
		**/
		void messageReceivedSlot(int, UserListElements, QCString& msg, time_t, QByteArray &formats);

		/**
			Wykonuje zadania co minut - pinguje sie i zeruje licznik
			odebranych obrazkw (jeli jestemy poczeni).
		**/
		void everyMinuteActions();

		/**
			Obsuguje otrzymanie nowych wynikw wyszukiwania z serwera. Emituje newSearchResults.

			@see newSearchResults
		**/
		void newResults(gg_pubdir50_t res);

		/**
			Nowa wiadomo od serwera. Emituje systemMessageReceived

			@see systemMessageReceived
		**/
		void systemMessageReceived(QString &, QDateTime &, int, void *);

		/**
			Pobrano list uytkownikw z serwera. Emituje userStatusChanged dla kadego
			otrzymanego kontaktu oraz userListChanged przed zakoczeniem dziaania.

			@see userStatusChanged
			@see userListChanged
		**/
		void userListReceived(const struct gg_event *);

		/**
			Odpowied od serwera na temat operacji na licie uytkownikw. Emituje, w zalenoci
			od trybu dziaania: userListCleared, userListExported, userListImported.

			@see userListCleared
			@see userListExported
			@see userListImported
		**/
		void userListReplyReceived(char, char *);

		/**
			Informacja o zmianie statusu kontaktu. Emituje userStatusChanged oraz userListChanged.

			@see userStatusChanged
			@see userListChanged
		**/
		void userStatusChanged(const struct gg_event *);

		/**
			Kto wykona gadu.status().setOnline(). czymy z serwerem, jeeli jeszcze tego nie
			zrobilimy, i zmieniamy status.

			@see CurrentStatus
			@see NextStatus
			@see login
		**/
		void iWantGoOnline(const QString &);

		/**
			Kto wykona gadu.status().setBusy(). czymy z serwerem, jeeli jeszcze tego nie
			zrobilimy, i zmieniamy status.

			@see CurrentStatus
			@see NextStatus
			@see login
		**/
		void iWantGoBusy(const QString &);

		/**
			Kto wykona gadu.status().setInvisible(). czymy z serwerem, jeeli jeszcze tego nie
			zrobilimy, i zmieniamy status.

			@see CurrentStatus
			@see NextStatus
			@see login
		**/
		void iWantGoInvisible(const QString &);

		/**
			Kto wykona gadu.status().setOffline(). Rozczamy si z serwerem i ustawiamy opis (jeeli
			bylimy poczeni).

			@see CurrentStatus
			@see NextStatus
			@see logout
		**/
		void iWantGoOffline(const QString &);

		/**
			Przysza informacja o dostarczeniu (lub nie) wiadomoci.
			Na podstawie statusu emituje odpowiednie sygnay message*

			@see messageBlocked
			@see messageDelivered
			@see messageQueued
			@see messageBoxFull
			@see messageNotDelivered
			@see messageAccepted
			@see messageRejected
		**/
		void ackReceived(int seq, uin_t uin, int status);

		void currentStatusChanged(const UserStatus &status, const UserStatus &oldStatus);

	public:
		static void initModule();
		static void closeModule();

		GaduProtocol(const QString &id, QObject *parent = NULL, const char *name = NULL);
		virtual ~GaduProtocol();

		gg_session * session() { return Sess; }

		void changeID(const QString &id);
		/**
			Zwraca serwer z ktrym jestemy poczeni lub do ktrego si wanie czymy.
			isNull() = hub.
		**/
		QHostAddress activeServer();

		/**
			Konwertuje list uytkownikw do postaci acucha.

			acuch wynikowy ma posta:
			<code>
				opis_uytkownika<CR><LF>
				opis_uytkownika<CR><LF>
				...
				opis_uytkownika<CR><LF>
			</code>

			opis_uytkownika ma posta:
			<code>
				firstName;lastName;nickName;altNick;mobile;grupy;uin;email;0;;0;
			</code>

			grupy maj posta:
			<code>
				grupa_1;grupa_2;grupa_3
			</code>

			@param userList lista uytkownikw, ktra zostanie skonwertowana
			@return acuch reprezentujcy list uytkownikw
			@see stringToUserList
			@see streamToUserList
		**/
		QString userListToString(const UserList &userList) const;

		/**
			Konwertuj acuch do listy uytkownikw.

			Format acucha jest anologiczny do tego z funkcji userListToString.

			@param source auch, bdcy reprezentacj listy uytkownikw
			@see userListToString
			@see streamToUserList
		**/
		QValueList<UserListElement> stringToUserList(const QString &source) const;

		/**
			Odczytuje ze strumienia acuch reprezentujcy list uytkownikw i konwertuje
			go go postaci obiektu UserList.

			Format acucha jest anologiczny do tego z funkcji userListToString.

			@param source strumie, z ktrego odczytane zostan dane
			@see userListToString
			@see stringToUserList
		**/
		QValueList<UserListElement> streamToUserList(QTextStream &source) const;

		/**
			Po jedno sekundowym opnieniu wykonuje prb poczenia.
		**/
		void connectAfterOneSecond();

		/**
			Ustawia adres IP ktry ma by przekazany do serwera GG
			jako adres komputera, ktry przekierowuje do nas port DCC.
			Zmiana jest uwzgldniana dopiero przy ponownym poczeniu
			z serwerem.
		**/
		void setDccExternalIP(const QHostAddress& ip);
		int seqNum() { return seqNumber; }
		bool validateUserID(QString& uid);

		virtual UserStatus *newStatus() const;

		// --------------------
		//  DCC
		// --------------------

		void dccRequest(UinType);
		void setDccIpAndPort(unsigned long dcc_ip, int dcc_port);

	public slots:
		/**
			Wysya wiadomo bez formatowania tekstu. Jeli adresatw jest wicej ni jeden, to wysyana
			jest wiadomo konferencyjna. Zwracany jest numer sekwencyjny wiadomoci, jeli
			przypadkiem bymy chcieli ledzi jej potwierdzenie.
			@param users lista uytkownikw, do ktrych wysyamy wiadomo
			@param mesg wiadomo, ktr wysyamy - kodowanie zmieniane wewntrz
		**/
		QString sendMessage(UserListElements users, const QString &mesg);
		/**
			Wysya prob o przysanie obrazka z danymi parametrami.

			@param user uytkownik, od ktrego chcemy obrazek
			@param size rozmiar obrazka w bajtach
			@param crc32 crc32 pliku
			@todo powinno by sendImageRequest(uniqId uint32_t) - info o obrazku zapisywa gdzie w rodku
		**/
		bool sendImageRequest(UserListElement user, int size, uint32_t crc32);

		/**
			Wywya obrazek o podanych parametrach.

			@param user uytkownik, ktremu wysyamy obrazek
			@param file_name nazwa pliku obrazka
			@param size rozmiar obrazka w bajtach
			@param data zawarto pliku
			@todo usun parametry size i data - moemy to chyba sami wyznaczy
		**/
		bool sendImage(UserListElement user, const QString &file_name, uint32_t size, const char *data);

		/**
			Rejetrujemy nowe konto. Odpowied przychodzi poprzez sygna registered. Moe
			zosta take wywoany sygna needTokenValue.

			@param mail nasz email, pole nieobowizkowe
			@param password nasze haso, pole obowizkowe
			@see registered
			@see needTokenValue
			@see unregisterAccount
		**/
		void registerAccount(const QString &mail, const QString &password);

		/**
			Wyrejestrowujemy stare konto. Odpowied przychodzi poprzez sygna unregistered. Moe
			zosta take wywoany sygna needTokenValue.

			@param uin nasz uin
			@param password nasze haso
			@todo parametr uin naprawd potrzebny?
		**/
		void unregisterAccount(UinType uin, const QString &password);

		/**
			Wysya haso na email. Odpowied przychodzi poprzez sygna reminded. Moe
			zosta take wywoany sygna needTokenValue.

			@param uin nasz uin
			@param email nasz email (musi by taki sam jak podczas rejestracji)
			@todo parametr uin naprawd potrzebny?
		**/
		void remindPassword(UinType uin, const QString& mail);

		/**
			Zmienia nasze haso. Odpowied przychodzi poprzez sygna passwordChanged. Moe
			zosta take wywoany sygna needTokenValue.

			@param uin nasz uin
			@param mail nasz email, jaki podalimy przy rejestracji
			@param password stare haso
			@param newPassword nowe haso
			@todo parametr uin naprawd potrzebny?
		**/
		void changePassword(UinType uin, const QString &mail, const QString &password,
			const QString &newPassword);

		/**
			Wysya list uytkownikw na serwer. Odpowied przychodzi przez sygna userListExported.

			@return false, jeli operacja si nie powioda
			@param userList lista do wysania
			@see userListExported
			@todo usun warto zwracan
		**/
		bool doExportUserList(const UserList &userList);

		/**
			Usuwa list uytkownikw z serwera. Odpowied przychodzi przez sygna userListCleared.

			@return false, jeli operacja si nie powioda
			@see userListCleared
			@todo usun warto zwracan
		**/
		bool doClearUserList();

		/**
			Importuje list uytkownikw z serwera. Odpowied przychodzi przez sygna userListImported.

			@return false, jeli operacja si nie powioda
			@see userListImported
			@todo usun warto zwracan
		**/
		bool doImportUserList();

		/**
			Wysya nasz list uytkownikw na serwer. Uwaga: nie ma to nic wsplnego z importem/eksportem.
		**/
		void sendUserList();

		void sendUserListLater();

		/**
			Szuka ludzi w katalogu publicznym. Wyniki przychodz za pomoca sygnau newSearchResults.

			@param searchRecord dane do szukania
			@see newSearchResults
			@see searchNextInPubdir
		**/
		void searchInPubdir(SearchRecord& searchRecord);

		/**
			Szuka ludzi w katalogu publicznym. Wyniki przychodz za pomoca sygnau newSearchResults.

			@param searchRecord dane do szukania
			@see newSearchResults
			@see searchInPubdir
		**/
		void searchNextInPubdir(SearchRecord& searchRecord);

		/**
			Zatrzymuje wyszukiwanie.

			@param searchRecord dane uyte w wyszukiwaniu
			@see searchInPubdir
		**/
		void stopSearchInPubdir(SearchRecord& searchRecord);

		/**
			Pobiera informacje o danych osobowych z katalogu publicznego.

			@todo jak to w ogle dziaa, bo zapomniaem??
		**/
		void getPersonalInfo(SearchRecord& searchRecord);

		/**
			Ustawia informacje o danych osobowych w katalogu publicznym.

			@todo jak to w ogle dziaa, bo zapomniaem??
		**/
		void setPersonalInfo(SearchRecord& searchRecord, SearchResult& newData);


	protected slots:
		/* sloty podczane do sygnaw z klasy UserList */
		void protocolUserDataChanged(QString protocolName, UserListElement elem,
							QString name, QVariant oldValue, QVariant currentValue,
							bool massively, bool last);
		void userDataChanged(UserListElement elem, QString name, QVariant oldValue,
							QVariant currentValue, bool massively, bool last);
		void userAdded(UserListElement elem, bool massively, bool last);

		void removingUser(UserListElement elem, bool massively, bool last);

		void protocolAdded(UserListElement elem, QString protocolName, bool massively, bool last);

		void removingProtocol(UserListElement elem, QString protocolName, bool massively, bool last);

	signals:
		/**
			wiadomo zostaa zablokowana przez serwer
		**/
		//void messageBlocked(int seq, UinType uin);

		/**
			wiadomo dostarczono
		**/
		void messageDelivered(int seq, UinType uin);

		/**
			wiadomo zakolejkowano
		**/
		void messageQueued(int seq, UinType uin);

		/**
			skrzynka odbiorcza na serwerze jest pena
		**/
		//void messageBoxFull(int seq, UinType uin);


		//void messageNotDelivered(int seq, UinType uin);


		//void messageAccepted(int seq, UinType uin);
		/**
			wiadomo zostaa odrzucona przez serwer
		**/
		void messageRejected(int seq, UinType uin);

		/**
			wystpi bd poczenia
			@param err przyczyna
			@see connectionError
		**/
		void error(GaduError err);

		/**
			dostalimy prob o przysanie obrazka
			@param sender od kogo
			@param size rozmiar pliku
			@param crc32 jego suma kontrolna obliczana przy pomocy crc32
		**/
		void imageRequestReceived(UinType sender, uint32_t size, uint32_t crc32);

		/**
			Otrzymano dane obrazka i zapisano go do pliku.
			@param sender od kogo
			@param size rozmiar pliku
			@param crc32 jego suma kontrolna obliczana przy pomocy crc32
			@param path cieka do zapisanego pliku
		**/
		void imageReceivedAndSaved(UinType sender, uint32_t size, uint32_t crc32, const QString &path);

		/**
			dostalimy od serwera informacj o zmianie statusu dla kontaktu,
			ktrego nie mamy na licie
		**/
		void userStatusChangeIgnored(UinType uin);

		/**
			otrzymana wiadomo systemow
			@param message tre wiadomoci wraz z dat
		**/
		void systemMessageReceived(const QString &message);

		/**
			Otrzymano wiadomo CTCP.
			Kto nas prosi o poczenie dcc, poniewa
			jestemy za NAT-em
			@param user wywoujcy
			@todo zmieni nazw
		**/
		void dccConnectionReceived(const UserListElement &user);

		/**
			otrzymano nowe wyniki wyszukiwania w katalogu publicznym
			@param searchResults wyniki
			@param seq numer sekwencyjny
			@param lastUin ?
		**/
		void newSearchResults(SearchResults& searchResults, int seq, int lastUin);

		/**
			operacja rejestracji nowego konta zostaa zakoczona
			@param ok powodzenie operacji
			@param uin nowy numer
			@see doRegisterAccount
		**/
		void registered(bool ok, UinType uin);

		/**
			operacja wyrejestrowania konta zostaa zakoczona
			@param ok powodzenie operacji
			@see doUnregisterAccount
		**/
		void unregistered(bool ok);

		/**
			operacja przypomnienia hasa zostaa zakoczona
			@param ok powodzenie operacji
			@see doRemindPassword
		**/
		void reminded(bool ok);

		/**
			operacja zmiany hasa zostaa zakoczona
			@param ok powodzenie operacji
			@see doChangePassword
		**/
		void passwordChanged(bool ok);

		/**
			operacja eksportu listy kontaktw na serwer zostaa zakoczona
			@param ok powodzenie operacji
			@see doExportUserList
		**/
		void userListExported(bool ok);

		/**
			operacja usunicia listy kontaktw z serwera zostaa zakoczona
			@param ok powodzenie operacji
			@see doClearUserList
		**/
		void userListCleared(bool ok);

		/**
			operacja importu listy kontaktw z serwera zostaa zakoczona
			@param ok powodzenie operacji
			@param list jeeli operacja si powioda, to zaimportowana lista
			@see doImportUserList
		**/
		void userListImported(bool ok, QValueList<UserListElement> list);

		/**
			Sygna daje mozliwo operowania na wiadomoci
			ktra przysza z serwera jeszcze w jej oryginalnej
			formie przed konwersj na unicode i innymi zabiegami.
			Tre wiadomoci mona zmieni grzebic w buforze msg,
			ale uwaga: mona zepsu formatowanie tekstu zapisane
			w formats. Oczywicie je rwnie mona zmienia wedug
			opisu protokou GG ;)
			Mona te przerwa dalsz obrbk wiadomoci ustawiajc
			stop na true.
		**/
		void rawGaduReceivedMessageFilter(Protocol *protocol, UserListElements senders, QCString &msg, QByteArray &formats, bool &ignore);

		/**
			Wywoywane, gdy chcemy odczyta token z obrazka
		**/
		void needTokenValue(QPixmap in, QString &out);

		void dcc7New(struct gg_dcc7 *);
		void dcc7Accepted(struct gg_dcc7 *);
		void dcc7Rejected(struct gg_dcc7 *);

};

extern GaduProtocol* gadu;

#endif
