/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kapplication.h>

#include "debug.h"
#include "../mediaplayer/mediaplayer.h"
#include "amarok.h"

#define MODULE_AMAROK_MEDIAPLAYER_VERSION 1.0

AmarokMediaPlayer* amarok;

// Kadu initializing functions
extern "C" int amarok_mediaplayer_init()
{
	amarok = new AmarokMediaPlayer();
	bool res = mediaplayer->registerMediaPlayer(amarok, amarok);
	return res ? 0 : 1;
}

extern "C" void amarok_mediaplayer_close()
{
	mediaplayer->unregisterMediaPlayer();
	delete amarok;
	amarok = NULL;
}

AmarokMediaPlayer::AmarokMediaPlayer()
{
	kdebugf();
	client = new DCOPClient();
	client->attach();
}

AmarokMediaPlayer::~AmarokMediaPlayer()
{
	kdebugf();
	client->detach();
	delete client;
	client = 0;
}

QString AmarokMediaPlayer::getString(QString obj, QString func)
{
	if (!isActive())
		return "";

	QByteArray replyData, data;
	QCString replyType;
	if (!client->call("amarok", obj.latin1(), func.latin1(), data, replyType, replyData))
		return "";

	QDataStream reply(replyData, IO_ReadOnly);
	QString str;
	reply >> str;
	return str.simplifyWhiteSpace();
}

QStringList AmarokMediaPlayer::getStringList(QString obj, QString func)
{
	if (!isActive())
		return QStringList();

	QByteArray replyData, data;
	QCString replyType;
	if (!client->call("amarok", obj.latin1(), func.latin1(), data, replyType, replyData))
		return QStringList();

	QDataStream reply(replyData, IO_ReadOnly);
	QStringList list;
	reply >> list;
	return list;
}

int AmarokMediaPlayer::getInt(QString obj, QString func)
{
	if (!isActive())
		return 0;

	QByteArray replyData, data;
	QCString replyType;
	if (!client->call("amarok", obj.latin1(), func.latin1(), data, replyType, replyData))
		return 0;

	QDataStream reply(replyData, IO_ReadOnly);
	int i;
	reply >> i;
	return i;
}

uint AmarokMediaPlayer::getUint(QString obj, QString func)
{
	if (!isActive())
		return 0;

	QByteArray replyData, data;
	QCString replyType;
	if (!client->call("amarok", obj.latin1(), func.latin1(), data, replyType, replyData))
		return 0;

	QDataStream reply(replyData, IO_ReadOnly);
	uint i;
	reply >> i;
	return i;
}

void AmarokMediaPlayer::send(QString obj, QString func, int arg)
{
	if (!isActive())
		return;

	QByteArray data, replyData;
	QCString replyType;
	if (arg != -1)
	{
		QDataStream str(data, IO_WriteOnly);
		str << arg;
	}
	client->call("amarok", obj.latin1(), func.latin1(), data, replyType, replyData);
}

// PlayerInfo

QString AmarokMediaPlayer::getPlayerName()
{
	return "amaroK";
}

QString AmarokMediaPlayer::getPlayerVersion()
{
	kdebugf();
	return getString("player", "version()");
}

QStringList AmarokMediaPlayer::getPlayListTitles()
{
	kdebugf();
	// No API for titles list, only file names.
	return getStringList("playlist", "filenames()");
}

QStringList AmarokMediaPlayer::getPlayListFiles()
{
	kdebugf();
	return getStringList("playlist", "filenames()");
}

uint AmarokMediaPlayer::getPlayListLength()
{
	kdebugf();
	return getUint("playlist", "getTotalTrackCount()");
}

QString AmarokMediaPlayer::getTitle(int position)
{
	kdebugf();
	return getString("player", "title()");
}

QString AmarokMediaPlayer::getAlbum(int position)
{
	kdebugf();
	return getString("player", "album()");
}

QString AmarokMediaPlayer::getArtist(int position)
{
	kdebugf();
	return getString("player", "artist()");
}

QString AmarokMediaPlayer::getFile(int position)
{
	kdebugf();
	return getString("player", "path()");
}

int AmarokMediaPlayer::getLength(int position)
{
	kdebugf();
	return getInt("player", "trackTotalTime()") * 1000;
}

int AmarokMediaPlayer::getCurrentPos()
{
	kdebugf();
	return getInt("player", "trackCurrentTime()") * 1000;
}

// PlayerCommands

void AmarokMediaPlayer::nextTrack()
{
	kdebugf();
	send("player", "next()");
}

void AmarokMediaPlayer::prevTrack()
{
	kdebugf();
	send("player", "prev()");
}

void AmarokMediaPlayer::play()
{
	kdebugf();
	send("player", "play()");
}

void AmarokMediaPlayer::stop()
{
	kdebugf();
	send("player", "stop()");
}

void AmarokMediaPlayer::pause()
{
	kdebugf();
	send("player", "pause()");
}

void AmarokMediaPlayer::setVolume(int vol)
{
	kdebugf();
	send("player", "setVolume(int)", vol);
}

void AmarokMediaPlayer::incrVolume()
{
	kdebugf();
	int vol = getInt("player", "getVolume");
	if (vol < 100)
		vol += 2;

	if (vol > 100)
		vol = 100;

	send("player", "setVolume(int)", vol);
}

void AmarokMediaPlayer::decrVolume()
{
	kdebugf();
	int vol = getInt("player", "getVolume");
	if (vol > 0)
		vol -= 2;

	if (vol < 0)
		vol = 0;

	send("player", "setVolume(int)", vol);
}

bool AmarokMediaPlayer::isPlaying()
{
	kdebugf();
	bool b;
	QByteArray replyData, data;
	QCString replyType;
	if (!client->call("amarok", "player", "isPlaying()", data, replyType, replyData))
		return false;

	QDataStream reply(replyData, IO_ReadOnly);
	reply >> b;
	return b;
}

bool AmarokMediaPlayer::isActive()
{
	kdebugf();
	QByteArray replyData, data;
	QCString replyType;
	return ((client->call("amarok", "player", "isPlaying()", data, replyType, replyData)) ? true : false);
}
