
/*
 * dialogs.C -- written for Juice
 *	Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef DIALOGS_DIALOGS_C
#define DIALOGS_DIALOGS_C

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <curses.h>
#include <ctype.h>
#include <sys/stat.h>

#include "typedefs.h"
#include "windows.h"
#include "vfs.h"
#include "dialogs.h"
#include "utils.h"

Dialogs::Dialogs ()
{
   hotkeys = NULL;
   abortflag = FALSE;
   hotkeyflag = HOTKEY_NONE;
}

Dialogs::~Dialogs ()
{
   HotkeyType *tmp;
   while (hotkeys != NULL)
	 {
		if (hotkeys->next == NULL)
		  {
			 free (hotkeys);
			 hotkeys = NULL;
		  }
		else
		  {
			 tmp = hotkeys;
			 while (tmp->next->next != NULL) tmp = tmp->next;
			 free (tmp->next);
			 tmp->next = NULL;
		  }
	 }
}

void Dialogs::operator = (const Windows &parent)
{
   window = parent.window;
}

void Dialogs::SetAbort (int enabled)
{
   abortflag = enabled;
}

bool Dialogs::Confirm (int tlx,int tly,const char *title,int attr,int fg,int bg)
{
   bool choice = FALSE,finished = FALSE;
   chtype ch;
   WINDOW *curwin;
   Add (28,3,tlx,tly,title,attr,fg,bg);
   curwin = GetWindow ();
   mvwaddstr (curwin,2,3,"Are you sure? [Y/N]: ");
   Update ();
   keypad (curwin,TRUE);
   do
	 {
		ch = wgetch (curwin);
		if (tolower (ch) == 'y')
		  {
			 finished = TRUE;
			 choice = TRUE;
			 mvwaddstr (curwin,2,24,"Yes");
			 wrefresh (curwin);
			 Update ();
		  }
		else if (tolower (ch) == 'n')
		  {
			 finished = TRUE;
			 choice = FALSE;
			 mvwaddstr (curwin,2,24,"No");
			 wrefresh (curwin);
			 Update ();
		  }
		else if (IsHotkey (ch))
		  {
			 execfunc (ch);
			 Update ();
			 if (abortflag)
			   {
				  finished = TRUE;
				  choice = FALSE;
			   }
		  }
		else if (ch == XCTRL ('l'))
		  {
			 Refresh ();
		  }
		else
		  {
			 beep ();
			 wrefresh (curwin);
			 Update ();
		  }
	 }
   while (!finished);
   keypad (curwin,FALSE);
   flushinp ();
   usleep (SLEEPTIME);
   Remove ();
   Update ();
   return choice;
}

void Dialogs::Menu (const MenuRec *options,const char *title,int fg,int bg)
{
   int nopts = 0,x = 0,hilite = 0,xmax = 0,ymax = 0,active = 1;
   bool finished = FALSE;
   WINDOW *curwin;
   while (options[nopts].txt != NULL) nopts++;
   for (int i = 0; i < nopts; i++)
	 if (x < (int) strlen (options[i].txt)) x = strlen (options[i].txt);
   Add (x + 4,nopts + 2,(COLS - x - 4) >> 1,(LINES - nopts - 2) >> 1,title,A_BOLD,fg,bg);
   curwin = GetWindow ();
   getmaxyx (curwin,ymax,xmax);
   switch (bg)
	 {
	  case BLACK: hilite = GREEN; break;
	  case RED: hilite = BLUE; break;
	  case GREEN: hilite = YELLOW; break;
	  case YELLOW: hilite = BLACK; break;
	  case BLUE: hilite = GREEN; break;
	  case MAGENTA: hilite = GREEN; break;
	  case CYAN: hilite = YELLOW; break;
	  case WHITE: hilite = CYAN;
	 }
   keypad (curwin,TRUE);
   do
	 {
		for (int i = 0; i < nopts; i++)
		  {
			 if (options[i].id != active) setcolor (curwin,fg,bg,A_BOLD); else setcolor (curwin,hilite,hilite,A_BOLD);
			 mvwaddstr (curwin,i + 2,(xmax - strlen (options[i].txt)) >> 1,options[i].txt);
		  }
		setcolor (curwin,fg,bg,A_BOLD);
		wrefresh (curwin);
		Update ();
		chtype ch = wgetch (curwin);
		if (ch == KEY_UP)
		  active = active > 1 ? active - 1 : nopts;
		else if (ch == KEY_DOWN)
		  active = active < nopts ? active + 1 : 1;
		else if (ch == '\n')
		  {
			 flushinp ();
			 (options[active - 1].function)();
			 finished = abortflag;
		  }
		else if (IsHotkey (ch))
		  {
			 execfunc (ch);
			 finished = abortflag;
		  }
		else if (ch == XCTRL ('l'))
		  {
			 Refresh ();
		  }
		else beep ();
		flushinp ();
	 }
   while (!finished);
   keypad (curwin,FALSE);
   Remove ();
   Update ();
}

void Dialogs::Select (const MenuRec *options,const char *title,int fg,int bg)
{
   int nopts = 0,x = 0,hilite = 0,xmax = 0,ymax = 0,active = 1;
   bool finished = FALSE;
   WINDOW *curwin;
   while (options[nopts].txt != NULL) nopts++;
   for (int i = 0; i < nopts; i++)
	 if (x < (int) strlen (options[i].txt)) x = strlen (options[i].txt);
   Add (x + 4,nopts + 2,(COLS - x - 4) >> 1,(LINES - nopts - 2) >> 1,title,A_BOLD,fg,bg);
   curwin = GetWindow ();
   getmaxyx (curwin,ymax,xmax);
   switch (bg)
	 {
	  case BLACK: hilite = GREEN; break;
	  case RED: hilite = BLUE; break;
	  case GREEN: hilite = YELLOW; break;
	  case YELLOW: hilite = BLACK; break;
	  case BLUE: hilite = GREEN; break;
	  case MAGENTA: hilite = GREEN; break;
	  case CYAN: hilite = YELLOW; break;
	  case WHITE: hilite = CYAN;
	 }
   keypad (curwin,TRUE);
   do
	 {
		for (int i = 0; i < nopts; i++)
		  {
			 if (options[i].id != active) setcolor (curwin,fg,bg,A_BOLD); else setcolor (curwin,hilite,hilite,A_BOLD);
			 mvwaddstr (curwin,i + 2,(xmax - strlen (options[i].txt)) >> 1,options[i].txt);
		  }
		setcolor (curwin,fg,bg,A_BOLD);
		wrefresh (curwin);
		Update ();
		chtype ch = wgetch (curwin);
		if (ch == KEY_UP)
		  active = active > 1 ? active - 1 : nopts;
		else if (ch == KEY_DOWN)
		  active = active < nopts ? active + 1 : 1;
		else if (ch == '\n')
		  {
			 flushinp ();
			 (options[active - 1].function)();
			 finished = TRUE;
		  }
		else if (IsHotkey (ch))
		  {
			 execfunc (ch);
			 finished = abortflag;
		  }
		else if (ch == XCTRL ('l'))
		  {
			 Refresh ();
		  }
		else beep ();
		flushinp ();
	 }
   while (!finished);
   keypad (curwin,FALSE);
   Remove ();
   Update ();
}

void Dialogs::Input (char *buffer,int buflen,int tlx,int tly,
					 const char *title,int attr,int fg,int bg)
{
   bool finished = FALSE;
   WINDOW *curwin;
   chtype ch;
   char *tmp;
   tmp = (char *) malloc (buflen + 1);
   strcpy (tmp,buffer);
   if (buflen > COLS - tlx - 4) buflen = COLS - tlx - 4;
   tmp[buflen] = '\0';
   Add (buflen + 2,1,tlx,tly,title,attr,fg,bg);
   curwin = GetWindow ();
   mvwaddstr (curwin,1,2,tmp);
   keypad (curwin,TRUE);
   curs_set (CURSOR_NORMAL);
   Update ();
   do
	 {
		flushinp ();
		ch = wgetch (curwin);
		if (isalnum (ch) || ispunct (ch) || (ch == ' '))
		  {
			 if ((int) strlen (tmp) >= buflen - 1) beep (); else
			   {
				  int i = strlen (tmp);
				  tmp[i] = ch;
				  i++;
				  tmp[i] = '\0';
			   }
		  }
		else if (ch == KEY_BACKSPACE)
		  {
			 if (strlen (tmp) == 0) beep (); else
			   {
				  int j,i = strlen (tmp) - 1;
				  tmp[i] = '\0';
				  getyx (curwin,j,i);
				  i--;
				  mvwaddch (curwin,j,i,' ');
			   }
		  }
		else if (IsHotkey (ch))
		  {
			 curs_set (CURSOR_INVISIBLE);
			 flushinp ();
			 execfunc (ch);
			 if (abortflag)
			   {
				  finished = TRUE;
				  strcpy (tmp,buffer);
			   }
			 curs_set (CURSOR_NORMAL);
		  }
		else if (ch == XCTRL ('l'))
		  {
			 Refresh ();
		  }
		else if (ch == '\n') finished = TRUE; else beep ();
		mvwaddstr (curwin,1,2,tmp);
		Update ();
	 }
   while (!finished);
   curs_set (CURSOR_INVISIBLE);
   keypad (curwin,FALSE);
   flushinp ();
   Remove ();
   strcpy (buffer,tmp);
   free (tmp);
   Update ();
}

void Dialogs::scrollframe (PANEL *curpan,bool single,bool visible,int frametype)
{
   int x = 0,y1 = 0,y2 = 0;
   switch (frametype)
	 {
	  case FRAME_FILEDIR:
		x = 69;
		y1 = 0;
		y2 = 18;
		break;
	  case FRAME_SAVE:
		x = 69;
		y1 = 0;
		y2 = 16;
		break;
	  case FRAME_HELP:
		x = 69;
		y1 = 0;
		y2 = 20;
		break;
	  case FRAME_MIDDLE:
		x = 34;
		y1 = 0;
		y2 = 16;
		break;
	  default:
		break;
		/* This shouldn't happen */
	 }
   if (visible) vertline (curpan,x,y1,y2,single,TRUE,SIDE_TNORMAL,SIDE_TNORMAL); else
	 {
		vertline (curpan,x,y1 + 1,y2 - 1,single,FALSE,SIDE_NORMAL,SIDE_NORMAL);
		WINDOW *curwin = panel_window (curpan);
		if (single)
		  {
			 mvwaddch (curwin,y1,x,lcs_horz);
			 mvwaddch (curwin,y2,x,lcs_horz);
		  }
		else
		  {
			 mvwaddch (curwin,y1,x,lcd_horz);
			 mvwaddch (curwin,y2,x,lcd_horz);
		  }
	 }
}

void Dialogs::scrollbutton (PANEL *curpan,int current,int total,bool visible,int frametype)
{
   float multiplier = 0;
   int x = 0,y,difference = 0,fg1 = 0,bg1 = 0,fg2 = 0,bg2 = 0;
   WINDOW *curwin = panel_window (curpan);
   switch (frametype)
	 {
	  case FRAME_FILEDIR:
		multiplier = 17;
		difference = 0;
		x = 70;
		fg1 = YELLOW;
		bg1 = bg2 = BLUE;
		fg2 = WHITE;
		break;
	  case FRAME_SAVE:
		multiplier = 15;
		difference = 0;
		x = 70;
		fg1 = YELLOW;
		bg1 = bg2 = BLUE;
		fg2 = WHITE;
		break;
	  case FRAME_HELP:
		multiplier = 20;
		difference = 17;
		x = 70;
		fg1 = WHITE;
		bg1 = fg2 = bg2 = CYAN;
		break;
	  case FRAME_MIDDLE:
		multiplier = 15;
		difference = 0;
		x = 35;
		fg1 = YELLOW;
		bg1 = bg2 = BLUE;
		fg2 = WHITE;
		break;
	  default:
		break;
		/* This shouldn't happen */
	 }
   y = (int) (((((float) current * multiplier) / (float) (total - difference))) + 1);
   if ((frametype == FRAME_HELP) && (y > 19)) y = 19;
   setcolor (curwin,fg1,bg1,A_BOLD);
   if (visible) mvwaddch (curwin,y,x,sc_scrlbut); else mvwaddch (curwin,y,x,' ');
   setcolor (curwin,fg2,bg2,A_BOLD);
}

void Dialogs::getfiledir (char *dirname,char *filename)
{
   int i = strlen (filename) - 1;
   while (filename[i] != '/') i--;
   strcpy (dirname,filename);
   dirname[i] = '\0';
}

void Dialogs::SelectDirectory (char *dirname,int maxdirlen,const char *title)
{
   WINDOW *curwin;
   PANEL *curpan;
   VirtualFS vfs;
   char txt[70],*curdir,*tmp;
   bool top = TRUE,finished = FALSE;
   int i,j,k,topchoice = 0,active = 0;
   chtype ch;
   Add (70,19,(COLS - 70) >> 1,(LINES - 19) >> 1,title,A_BOLD,WHITE,BLUE);
   disableframes ();
   curwin = GetWindow ();
   curpan = GetPanel ();
   horzline (curpan,0,71,18,FALSE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
   vfs.CreateDirListing (dirname,65);
   vfs.SortEntries (SORT_NAME);
   if (vfs.NumEntries () > 15) scrollframe (curpan,FALSE,TRUE,FRAME_FILEDIR);
   txt[65] = '\0';
   keypad (curwin,TRUE);
   do
	 {
		if (top) setcolor (curwin,WHITE,BLUE,A_BOLD); else setcolor (curwin,GREEN,GREEN,A_BOLD);
		mvwaddstr (curwin,19,32," Select ");
		k = topchoice + 15;
		if (vfs.NumEntries () < k) k = vfs.NumEntries ();
		setcolor (curwin,WHITE,BLUE,A_BOLD);
		for (i = 0; i < 65; i++) txt[i] = ' ';
		for (i = 2; i < 17; i++) mvwaddstr (curwin,i,2,txt);
		for (i = topchoice; i < k; i++)
		  {
			 strcpy (txt,vfs.EntryName (i));
			 for (j = strlen (txt); j < 65; j++) txt[j] = ' ';
			 if ((i == active) && top) setcolor (curwin,GREEN,GREEN,A_BOLD); else setcolor (curwin,WHITE,BLUE,A_BOLD);
			 mvwaddstr (curwin,i - topchoice + 2,2,txt);
		  }
        if (vfs.NumEntries () > 15)
		  scrollbutton (curpan,active,vfs.NumEntries (),TRUE,FRAME_FILEDIR);
		Update ();
		ch = wgetch (curwin);
		if (vfs.NumEntries () > 15)
		  scrollbutton (curpan,active,vfs.NumEntries (),FALSE,FRAME_FILEDIR);
		if (IsHotkey (ch))
		  {
			 bool visible = vfs.NumEntries () > 15;
			 flushinp ();
			 singleframe (curpan);
			 showtitle (curpan,title);
			 horzline (curpan,0,71,18,TRUE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
			 scrollframe (curpan,TRUE,visible,FRAME_FILEDIR);
			 execfunc (ch);
			 doubleframe (curpan);
			 showtitle (curpan,title);
			 horzline (curpan,0,71,18,FALSE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
			 scrollframe (curpan,FALSE,visible,FRAME_FILEDIR);
			 finished = abortflag;
		  }
		else if (ch == XCTRL ('l'))
		  {
			 Refresh ();
		  }
		else if (ch == '\n')
		  {
             if (top)
			   {
				  tmp = vfs.EntryFilename (active);
				  int dirlen = strlen (tmp) + 1 < PATH_MAX ? PATH_MAX : strlen (tmp) + 1;
				  curdir = (char *) malloc (dirlen);
				  strcpy (curdir,tmp);
				  vfs.RemoveEntries ();
				  canonicalize_pathname (curdir,dirlen);
				  vfs.CreateDirListing (curdir,65);
				  setcolor (curwin,WHITE,BLUE,A_BOLD);
				  if (vfs.NumEntries () > 15) scrollframe (curpan,FALSE,TRUE,FRAME_FILEDIR); else
					scrollframe (curpan,FALSE,FALSE,FRAME_FILEDIR);
				  topchoice = active = 0;
				  free (curdir);
			   }
			 else
			   {
				  char *curentry;
				  tmp = vfs.EntryFilename (active);
				  curentry = (char *) malloc (strlen (tmp) + 1);
				  strcpy (curentry,tmp);
				  curdir = (char *) malloc (strlen (tmp) + 1);
				  getfiledir (curdir,curentry);
				  strncpy (dirname,curdir,maxdirlen);
				  free (curentry);
				  free (curdir);
				  finished = TRUE;
			   }
		  }
		else if (ch == '\t')
		  top = (!top);
		else if (ch == KEY_UP)
		  {
			 if (!top) beep (); else
			   {
				  if (active == 0) beep (); else
					{
					   if (active > topchoice) active--; else
						 {
							active--;
							topchoice--;
						 }
					}
			   }
		  }
		else if (ch == KEY_DOWN)
		  {
			 if (!top) beep (); else
			   {
				  if (active == vfs.NumEntries () - 1) beep (); else
					{
					   if (active < k - 1) active++; else
						 {
							active++;
							topchoice++;
						 }
					}
			   }
		  }
		else if (ch == KEY_NPAGE)		/* Page Down */
		  {
			 if (!top) beep (); else
			   {
				  if (active == vfs.NumEntries () - 1) beep (); else
					{
					   if (active < vfs.NumEntries () - 16)
						 {
							topchoice += 15;
							active += 15;
							while (topchoice > vfs.NumEntries () - 16)
							  {
								 topchoice--;
								 active--;
							  }
						 }
					   else
						 {
							active = vfs.NumEntries () - 1;
							topchoice = active - 14;
							while (topchoice < 0) topchoice++;
						 }
					}
			   }
		  }
		else if (ch == KEY_PPAGE)		/* Page Up */
		  {
			 if (!top) beep (); else
			   {
				  if (active == 0) beep (); else
					{
					   if (active > 14)
						 {
							active -= 15;
							topchoice -= 15;
							if (topchoice < 0) topchoice = 0;
						 }
					   else active = topchoice = 0;
					}
			   }
		  }
		else if (ch == KEY_HOME)
		  {
			 if (!top) beep (); else topchoice = active = 0;
		  }
		else if (ch == KEY_END)
		  {
			 if (!top) beep (); else
			   {
				  active = vfs.NumEntries () - 1;
				  topchoice = active - 14;
				  while (topchoice < 0) topchoice++;
			   }
		  }
		else beep ();
		flushinp ();
	 }
   while (!finished);
   keypad (curwin,FALSE);
   vfs.RemoveEntries ();
   Remove ();
   Update ();
}

void Dialogs::SelectFile (char *filename,int maxfilelen,char *filespec,const char *title)
{
   WINDOW *curwin;
   PANEL *curpan;
   VirtualFS vfs;
   char txt[70],*curdir,*tmp;
   bool top = TRUE,finished = FALSE;
   int i,j,k,topchoice = 0,active = 0;
   chtype ch;
   Add (70,19,(COLS - 70) >> 1,(LINES - 19) >> 1,title,A_BOLD,WHITE,BLUE);
   disableframes ();
   curwin = GetWindow ();
   curpan = GetPanel ();
   horzline (curpan,0,71,18,FALSE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
   curdir = (char *) malloc (strlen (filename) + 1);
   if (isdir (filename)) strcpy (curdir,filename); else getfiledir (curdir,filename);
   vfs.CreateFileListing (curdir,filespec,64);
   vfs.SortEntries (SORT_EXT);
   free (curdir);
   if (vfs.NumEntries () > 15) scrollframe (curpan,FALSE,TRUE,FRAME_FILEDIR);
   for (i = 0; i < vfs.NumEntries (); i++) if (strcmp (filename,vfs.EntryFilename (i)) == 0)
	 {
		active = i;
		if (active > topchoice + 15) topchoice = active - 14;
		break;
	 }
   txt[65] = '\0';
   keypad (curwin,TRUE);
   do
	 {
		if (top) setcolor (curwin,WHITE,BLUE,A_BOLD); else setcolor (curwin,GREEN,GREEN,A_BOLD);
		mvwaddstr (curwin,19,32," Select ");
		k = topchoice + 15;
		if (vfs.NumEntries () < k) k = vfs.NumEntries ();
		setcolor (curwin,WHITE,BLUE,A_BOLD);
		for (i = 0; i < 65; i++) txt[i] = ' ';
		for (i = 2; i < 17; i++) mvwaddstr (curwin,i,2,txt);
		for (i = topchoice; i < k; i++)
		  {
			 strcpy (txt,vfs.EntryName (i));
			 if (vfs.EntryType (i) == FT_DIR) strcat (txt,"/");
			 for (j = strlen (txt); j < 65; j++) txt[j] = ' ';
			 if ((i == active) && top) setcolor (curwin,GREEN,GREEN,A_BOLD); else
			   {
				  setcolor (curwin,WHITE,BLUE,A_BOLD);
				  if (vfs.EntryType (i) == FT_FILE) wattroff (curwin,/*A_NORMAL*/A_BOLD);
			   }
			 mvwaddstr (curwin,i - topchoice + 2,2,txt);
			 wattron (curwin,A_BOLD);
		  }
        if (vfs.NumEntries () > 15)
		  scrollbutton (curpan,active,vfs.NumEntries (),TRUE,FRAME_FILEDIR);
		Update ();
		ch = wgetch (curwin);
		if (vfs.NumEntries () > 15)
		  scrollbutton (curpan,active,vfs.NumEntries (),FALSE,FRAME_FILEDIR);
		if (IsHotkey (ch))
		  {
			 bool visible = vfs.NumEntries () > 15;
			 flushinp ();
			 singleframe (curpan);
			 showtitle (curpan,title);
			 horzline (curpan,0,71,18,TRUE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
			 scrollframe (curpan,TRUE,visible,FRAME_FILEDIR);
			 execfunc (ch);
			 doubleframe (curpan);
			 showtitle (curpan,title);
			 horzline (curpan,0,71,18,FALSE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
			 scrollframe (curpan,FALSE,visible,FRAME_FILEDIR);
			 finished = abortflag;
		  }
		else if (ch == XCTRL ('l'))
		  {
			 Refresh ();
		  }
		else if (ch == '\n')
		  {
			 if (top)
			   {
				  if (vfs.EntryType (active) == FT_FILE) beep (); else
					{
					   tmp = vfs.EntryFilename (active);
					   int dirlen = strlen (tmp) + 1 < PATH_MAX ? PATH_MAX : strlen (tmp) + 1;
					   curdir = (char *) malloc (dirlen);
					   strcpy (curdir,tmp);
					   vfs.RemoveEntries ();
					   canonicalize_pathname (curdir,dirlen);
					   vfs.CreateFileListing (curdir,filespec,64);
					   vfs.SortEntries (SORT_EXT);
					   setcolor (curwin,WHITE,BLUE,A_BOLD);
					   if (vfs.NumEntries () > 15)
						 scrollframe (curpan,FALSE,TRUE,FRAME_FILEDIR);
					   else
						 scrollframe (curpan,FALSE,FALSE,FRAME_FILEDIR);
					   topchoice = active = 0;
					   free (curdir);
					}
			   }
			 else if (vfs.EntryType (active) == FT_FILE)
			   {
				  strncpy (filename,vfs.EntryFilename (active),maxfilelen);
				  finished = TRUE;
			   }
			 else beep ();
		  }
		else if (ch == '\t')
		  top = (!top);
		else if (ch == KEY_UP)
		  {
			 if (!top) beep (); else
			   {
				  if (active == 0) beep (); else
					{
					   if (active > topchoice) active--; else
						 {
							active--;
							topchoice--;
						 }
					}
			   }
		  }
		else if (ch == KEY_DOWN)
		  {
			 if (!top) beep (); else
			   {
				  if (active == vfs.NumEntries () - 1) beep (); else
					{
					   if (active < k - 1) active++; else
						 {
							active++;
							topchoice++;
						 }
					}
			   }
		  }
		else if (ch == KEY_NPAGE)		/* Page Down */
		  {
			 if (!top) beep (); else
			   {
				  if (active == vfs.NumEntries () - 1) beep (); else
					{
					   if (active < vfs.NumEntries () - 16)
						 {
							topchoice += 15;
							active += 15;
							while (topchoice > vfs.NumEntries () - 16)
							  {
								 topchoice--;
								 active--;
							  }
						 }
					   else
						 {
							active = vfs.NumEntries () - 1;
							topchoice = active - 14;
							while (topchoice < 0) topchoice++;
						 }
					}
			   }
		  }
		else if (ch == KEY_PPAGE)		/* Page Up */
		  {
			 if (!top) beep (); else
			   {
				  if (active == 0) beep (); else
					{
					   if (active > 14)
						 {
							active -= 15;
							topchoice -= 15;
							if (topchoice < 0) topchoice = 0;
						 }
					   else active = topchoice = 0;
					}
			   }
		  }
		else if (ch == KEY_HOME)
		  {
			 if (!top) beep (); else topchoice = active = 0;
		  }
		else if (ch == KEY_END)
		  {
			 if (!top) beep (); else
			   {
				  active = vfs.NumEntries () - 1;
				  topchoice = active - 14;
				  while (topchoice < 0) topchoice++;
			   }
		  }
		else beep ();
		flushinp ();
	 }
   while (!finished);
   keypad (curwin,FALSE);
   vfs.RemoveEntries ();
   Remove ();
   Update ();
}

void Dialogs::Help (const char *filename,const char *title)
{
   WINDOW *curwin;
   PANEL *curpan;
   chtype ch;
   FILE *handle = fopen (filename,"r");
   int i,k,lines = 0,topline = 0;
   char **buffer,line[80];
   bool finished = FALSE;
   buffer = NULL;
   while (!feof (handle))
	 {
		fgets (line,79,handle);
		lines++;
		buffer = (char **) realloc (buffer,lines * sizeof (char *));
		buffer[lines - 1] = (char *) malloc (strlen (line) + 1);
		strcpy (buffer[lines - 1],line);
		buffer[lines - 1][strlen (buffer[lines - 1]) - 1] = '\0';
	 }
   fclose (handle);
   Add (70,19,(COLS - 70) >> 1,(LINES - 19) >> 1,title,A_BOLD,CYAN,CYAN);
   disableframes ();
   curwin = GetWindow ();
   curpan = GetPanel ();
   if (lines > 17) scrollframe (curpan,FALSE,TRUE,FRAME_HELP);
   keypad (curwin,TRUE);
   do
	 {
		k = topline + 17;
		if (lines < k) k = lines;
		setcolor (curwin,CYAN,CYAN,A_BOLD);
		for (i = 0; i < 65; i++) line[i] = ' ';
		line[65] = '\0';
		for (i = 2; i < 20; i++) mvwaddstr (curwin,i,2,line);
        for (i = topline; i < k; i++)
		  {
			 if (strlen (buffer[i]) == 0)
			   {
				  strcpy (line," ");
				  setcolor (curwin,CYAN,CYAN,A_BOLD);
			   }
			 else if (buffer[i][0] == '%')
			   {
				  strcpy (line,buffer[i] + 1);
				  setcolor (curwin,RED,CYAN,A_BOLD);
			   }
			 else if (buffer[i][0] == ' ')
			   {
				  strcpy (line,buffer[i]);
				  setcolor (curwin,CYAN,CYAN,A_BOLD);
			   }
			 else
			   {
				  strcpy (line,buffer[i]);
				  setcolor (curwin,GREEN,CYAN,A_BOLD);
			   }
			 mvwaddstr (curwin,i - topline + 2,4,line);
		  }
		if (lines > 17) scrollbutton (curpan,topline,lines,TRUE,FRAME_HELP);
		Update ();
		ch = wgetch (curwin);
		if (lines > 17) scrollbutton (curpan,topline,lines,FALSE,FRAME_HELP);
		if (IsHotkey (ch))
		  {
             flushinp ();
			 singleframe (curpan);
			 showtitle (curpan,title);
			 scrollframe (curpan,TRUE,TRUE,FRAME_HELP);
			 execfunc (ch);
			 doubleframe (curpan);
			 showtitle (curpan,title);
			 scrollframe (curpan,FALSE,TRUE,FRAME_HELP);
			 finished = abortflag;
		  }
		else if (ch == XCTRL ('l'))
		  {
			 Refresh ();
		  }
		else if (ch == '\n')
		  finished = TRUE;
		else if (ch == KEY_UP)
		  {
			 if (topline == 0) beep (); else topline--;
		  }
		else if (ch == KEY_DOWN)
		  {
			 if ((topline + 17 == lines) || (lines < 17)) beep (); else topline++;
		  }
		else if (ch == KEY_HOME)
		  {
			 if (topline == 0) beep (); else topline = 0;
		  }
		else if (ch == KEY_END)
		  {
			 if ((topline + 17 == lines) || (lines < 17)) beep (); else topline = lines - 17;
		  }
		else if (ch == KEY_NPAGE)		/* PGDN */
		  {
			 if ((topline + 17 == lines) || (lines < 17)) beep (); else
			   {
				  topline += 17;
				  while (topline + 17 > lines) topline--;
			   }
		  }
		else if (ch == KEY_PPAGE)		/* PGUP */
		  {
			 if (topline == 0) beep (); else
			   {
				  topline -= 17;
				  while (topline < 0) topline++;
			   }
		  }
		else beep ();
		flushinp ();
	 }
   while (!finished);
   keypad (curwin,FALSE);
   Remove ();
   for (i = 0; i < lines; i++) free (buffer[i]);
   free (buffer);
   Update ();
}

void Dialogs::Info (int tlx,int tly,const char *txt,int attr,int fg,int bg)
{
   Add (strlen (txt) + 4,3,tlx,tly,"",attr,fg,bg);
   doubleframe (GetPanel ());
   mvwaddstr (GetWindow (),2,3,txt);
   Update ();
   usleep (SLEEPTIME * 7);
   Remove ();
}

bool Dialogs::Question (int tlx,int tly,const char *question,int attr,int fg,int bg)
{
   bool choice = FALSE,finished = FALSE;
   chtype ch;
   WINDOW *curwin;
   PANEL *curpan;
   char *tmp = (char *) malloc (strlen (question) + 9);
   int xpos;
   strcpy (tmp,question);
   strcat (tmp," [Y/N]: ");
   xpos = strlen (tmp) + 3;
   Add (strlen (tmp) + 7,3,tlx,tly,"",attr,fg,bg);
   curwin = GetWindow ();
   curpan = GetPanel ();
   disableframes ();
   doubleframe (curpan);
   mvwaddstr (curwin,2,3,tmp);
   free (tmp);
   Update ();
   keypad (curwin,TRUE);
   do
	 {
		ch = wgetch (curwin);
		if (tolower (ch) == 'y')
		  {
			 finished = TRUE;
			 choice = TRUE;
			 mvwaddstr (curwin,2,xpos,"Yes");
			 wrefresh (curwin);
			 Update ();
		  }
		else if (tolower (ch) == 'n')
		  {
			 finished = TRUE;
			 choice = FALSE;
			 mvwaddstr (curwin,2,xpos,"No");
			 wrefresh (curwin);
			 Update ();
		  }
		else if (IsHotkey (ch))
		  {
			 singleframe (curpan);
			 execfunc (ch);
			 doubleframe (curpan);
			 Update ();
			 if (abortflag)
			   {
				  finished = TRUE;
				  choice = FALSE;
			   }
		  }
		else if (ch == XCTRL ('l'))
		  {
			 Refresh ();
		  }
		else
		  {
			 beep ();
			 wrefresh (curwin);
			 Update ();
		  }
	 }
   while (!finished);
   keypad (curwin,FALSE);
   flushinp ();
   usleep (SLEEPTIME);
   Remove ();
   Update ();
   return choice;
}

bool Dialogs::isdir (char *name)
{
   struct stat filestat;
   stat (name,&filestat);
   return (S_ISDIR (filestat.st_mode));
}

void Dialogs::SaveFile (char *filename,int maxfilelen,char *filespec,const char *title)
{
   WINDOW *curwin;
   PANEL *curpan;
   VirtualFS vfs;
   char txt[PATH_MAX],curfile[PATH_MAX],*curdir,*tmp;
   bool top = TRUE,finished = FALSE,modified = FALSE;
   int i,j,k,topchoice = 0,active = 0;
   chtype ch;
   Add (70,19,(COLS - 70) >> 1,(LINES - 19) >> 1,title,A_BOLD,WHITE,BLUE);
   disableframes ();
   curwin = GetWindow ();
   curpan = GetPanel ();
   horzline (curpan,0,71,16,FALSE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
   horzline (curpan,0,71,18,FALSE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
   curdir = (char *) malloc (strlen (filename) + 1);
   if (isdir (filename)) strcpy (curdir,filename); else getfiledir (curdir,filename);
   vfs.CreateFileListing (curdir,filespec,64);
   vfs.SortEntries (SORT_EXT);
   free (curdir);
   if (vfs.NumEntries () > 13) scrollframe (curpan,FALSE,TRUE,FRAME_SAVE);
   if (!isdir (filename))
	 {
		for (i = 0; i < vfs.NumEntries (); i++) if (strcmp (filename,vfs.EntryFilename (i)) == 0)
		  {
			 active = i;
			 if (active > topchoice + 13) topchoice = active - 12;
			 break;
		  }
	 }
   txt[65] = '\0';
   strcpy (curfile,vfs.EntryFilename (active));
   keypad (curwin,TRUE);
   curs_set (CURSOR_NORMAL);
   do
	 {
		if (top) setcolor (curwin,WHITE,BLUE,A_BOLD); else setcolor (curwin,GREEN,GREEN,A_BOLD);
		mvwaddstr (curwin,19,33," Save ");
		k = topchoice + 13;
		if (vfs.NumEntries () < k) k = vfs.NumEntries ();
		setcolor (curwin,WHITE,BLUE,A_BOLD);
		for (i = 0; i < 65; i++) txt[i] = ' ';
		for (i = 2; i < 15; i++) mvwaddstr (curwin,i,2,txt);
		for (i = topchoice; i < k; i++)
		  {
			 strcpy (txt,vfs.EntryName (i));
			 if (vfs.EntryType (i) == FT_DIR) strcat (txt,"/");
			 for (j = strlen (txt); j < 65; j++) txt[j] = ' ';
			 if ((i == active) && top) setcolor (curwin,GREEN,GREEN,A_BOLD);
			 else
			   {
				  setcolor (curwin,WHITE,BLUE,A_BOLD);
				  if (vfs.EntryType (i) == FT_FILE) wattroff (curwin,/*A_NORMAL*/A_BOLD);
			   }
			 mvwaddstr (curwin,i - topchoice + 2,2,txt);
			 wattron (curwin,A_BOLD);
		  }
        if (vfs.NumEntries () > 13)
		  scrollbutton (curpan,active,vfs.NumEntries (),TRUE,FRAME_SAVE);
		/* *** */
		setcolor (curwin,WHITE,BLUE);
		if (strlen (curfile) > 61)
		  {
			 strcpy (txt,"...");
			 strcat (txt,curfile + strlen (curfile) - 61);
		  }
		else strcpy (txt,curfile);
		if ((vfs.EntryType (active) == FT_DIR) && (!modified))
		  {
			 canonicalize_pathname (txt,PATH_MAX);
			 if (strcmp (txt,"/") != 0) strcat (txt,"/");
		  }
		for (i = strlen (txt); i < 68; i++) mvwaddch (curwin,17,i,' ');
		mvwaddstr (curwin,17,2,txt);
		txt[65] = '\0';
		/* *** */
		Update ();
		ch = wgetch (curwin);
		if (vfs.NumEntries () > 13)
		  scrollbutton (curpan,active,vfs.NumEntries (),FALSE,FRAME_SAVE);
		if (IsHotkey (ch))
		  {
			 bool visible = vfs.NumEntries () > 13;
			 flushinp ();
			 singleframe (curpan);
			 showtitle (curpan,title);
			 horzline (curpan,0,71,16,TRUE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
			 horzline (curpan,0,71,18,TRUE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
			 scrollframe (curpan,TRUE,visible,FRAME_SAVE);
			 execfunc (ch);
			 doubleframe (curpan);
			 showtitle (curpan,title);
			 horzline (curpan,0,71,16,FALSE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
			 horzline (curpan,0,71,18,FALSE,TRUE,SIDE_TNORMAL,SIDE_TNORMAL);
			 scrollframe (curpan,FALSE,visible,FRAME_SAVE);
			 finished = abortflag;
		  }
		else if (ch == XCTRL ('l'))
		  {
			 Refresh ();
		  }
		else if (ch == '\n')
		  {
			 if (top)
			   {
				  if (vfs.EntryType (active) == FT_FILE) beep (); else
					{
					   tmp = vfs.EntryFilename (active);
					   int dirlen = strlen (tmp) + 1 < PATH_MAX ? PATH_MAX : strlen (tmp) + 1;
					   curdir = (char *) malloc (dirlen);
					   strcpy (curdir,tmp);
					   vfs.RemoveEntries ();
					   canonicalize_pathname (curdir,dirlen);
					   vfs.CreateFileListing (curdir,filespec,64);
					   vfs.SortEntries (SORT_EXT);
					   setcolor (curwin,WHITE,BLUE,A_BOLD);
					   if (vfs.NumEntries () > 13)
						 scrollframe (curpan,FALSE,TRUE,FRAME_SAVE);
					   else
						 scrollframe (curpan,FALSE,FALSE,FRAME_SAVE);
					   topchoice = active = 0;
					   free (curdir);
					}
			   }
			 else if ((vfs.EntryType (active) == FT_FILE) ||
					  ((vfs.EntryType (active) == FT_DIR) && modified))
			   {
				  strncpy (filename,curfile,maxfilelen);
				  finished = TRUE;
			   }
			 else beep ();
		  }
		else if (ch == '\t')
		  top = (!top);
		else if (ch == KEY_UP)
		  {
			 if (!top) beep (); else
			   {
				  if (active == 0) beep (); else
					{
					   if (active > topchoice) active--; else
						 {
							active--;
							topchoice--;
						 }
					}
				  strcpy (curfile,vfs.EntryFilename (active));
				  modified = FALSE;
			   }
		  }
		else if (ch == KEY_DOWN)
		  {
			 if (!top) beep (); else
			   {
				  if (active == vfs.NumEntries () - 1) beep (); else
					{
					   if (active < k - 1) active++; else
						 {
							active++;
							topchoice++;
						 }
					}
				  strcpy (curfile,vfs.EntryFilename (active));
				  modified = FALSE;
			   }
		  }
		else if (ch == KEY_NPAGE)		/* Page Down */
		  {
			 if (!top) beep (); else
			   {
				  if (active == vfs.NumEntries () - 1) beep (); else
					{
					   if (active < vfs.NumEntries () - 14)
						 {
							topchoice += 13;
							active += 13;
							while (topchoice > vfs.NumEntries () - 14)
							  {
								 topchoice--;
								 active--;
							  }
						 }
					   else
						 {
							active = vfs.NumEntries () - 1;
							topchoice = active - 12;
							while (topchoice < 0) topchoice++;
						 }
					}
				  strcpy (curfile,vfs.EntryFilename (active));
				  modified = FALSE;
			   }
		  }
		else if (ch == KEY_PPAGE)		/* Page Up */
		  {
			 if (!top) beep (); else
			   {
				  if (active == 0) beep (); else
					{
					   if (active > 12)
						 {
							active -= 13;
							topchoice -= 13;
							if (topchoice < 0) topchoice = 0;
						 }
					   else active = topchoice = 0;
					}
				  strcpy (curfile,vfs.EntryFilename (active));
				  modified = FALSE;
			   }
		  }
		else if (ch == KEY_HOME)
		  {
			 if (!top) beep (); else topchoice = active = 0;
			 strcpy (curfile,vfs.EntryFilename (active));
			 modified = FALSE;
		  }
		else if (ch == KEY_END)
		  {
			 if (!top) beep (); else
			   {
				  active = vfs.NumEntries () - 1;
				  topchoice = active - 12;
				  while (topchoice < 0) topchoice++;
			   }
			 strcpy (curfile,vfs.EntryFilename (active));
			 modified = FALSE;
		  }
		else if (isalnum (ch) || ispunct (ch) || (ch == ' '))
		  {
			 if ((vfs.EntryType (active) == FT_DIR) && (!modified))
			   {
				  canonicalize_pathname (curfile,PATH_MAX);
				  if (strcmp (curfile,"/") != 0) strcat (curfile,"/");
				  modified = TRUE;
			   }
			 i = strlen (curfile);
			 curfile[i] = ch;
			 i++;
			 curfile[i] = '\0';
		  }
		else if (ch == KEY_BACKSPACE)
		  {
			 if ((vfs.EntryType (active) == FT_DIR) && (!modified)) beep (); else
			   {
				  getfiledir (txt,curfile);
				  j = strlen (txt) + 1;
				  if ((int) strlen (curfile) == j) beep (); else
					{
					   i = strlen (curfile) - 1;
					   curfile[i] = '\0';
					}
				  strcpy (txt,vfs.EntryFilename (active));
				  strcat (txt,"/");
				  if (strcmp (curfile,txt) == 0) modified = FALSE;
				  txt[65] = '\0';
			   }
		  }
		else beep ();
		flushinp ();
	 }
   while (!finished);
   curs_set (CURSOR_INVISIBLE);
   keypad (curwin,FALSE);
   vfs.RemoveEntries ();
   Remove ();
   Update ();
}

bool Dialogs::IsHotkey (int hotkey)
{
   HotkeyType *tmp;
   bool finished = FALSE;
   tmp = hotkeys;
   if (tmp == NULL) return FALSE;
   while ((tmp/*->next*/ != NULL) && (!finished))
	 {
		if (tmp->hotkey == hotkey) finished = TRUE; else tmp = tmp->next;
	 }
//   if ((tmp->next == NULL) && (tmp->hotkey == hotkey)) finished = TRUE;
   return finished;
}

void Dialogs::AddHotkey (int hotkey,void (*function)())
{
   HotkeyType *nw,*tmp;
   nw = (HotkeyType *) malloc (sizeof (HotkeyType));
   nw->next = NULL;
   nw->hotkey = hotkey;
   nw->function = function;
   if (hotkeys == NULL) hotkeys = nw; else
	 {
		tmp = hotkeys;
		while (tmp->next != NULL) tmp = tmp->next;
		tmp->next = nw;
	 }
}

void Dialogs::RemoveHotkey (int hotkey)
{
   HotkeyType *current,*previous;
   bool finished = FALSE;
   current = hotkeys;
   previous = NULL;
   if (current == NULL) return;
   while ((current != NULL) && (!finished))
     {
		if (current->hotkey == hotkey)
		  {
			 if ((previous != NULL) && (current->next != NULL)) previous->next = current->next;
			 else if (previous == NULL) hotkeys = current->next;
			 else previous->next = NULL;
			 free (current);
			 finished = TRUE;
		  }
		else
		  {
			 previous = current;
			 current = current->next;
		  }
	 }
}

void Dialogs::execfunc (int hotkey)
{
   HotkeyType *tmp;
   bool finished = FALSE;
   tmp = hotkeys;
   if (tmp == NULL) return;
   while ((tmp->next != NULL) && (!finished))
	 {
		if (tmp->hotkey == hotkey) finished = TRUE; else tmp = tmp->next;
	 }
   if ((tmp->next == NULL) && (tmp->hotkey == hotkey)) finished = TRUE;
   if (finished)
	 {
		(tmp->function)();
		hotkeyflag = tmp->hotkey;
	 }
}

void Dialogs::ResetHotkeyFlag ()
{
   hotkeyflag = HOTKEY_NONE;
}

int Dialogs::LastHotkey ()
{
   return hotkeyflag;
}

#endif
