#include <jmp-config.h>

#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>

#include <hash.h>
#include <jmp.h>
#include <cls.h>
#include <comparators.h>
#include <stats_context.h>
#include <stats_personality.h>

#include <ui_gtk.h>
#include <ui_gtk_prefs.h>
#include <ui_gtk_gtkutils.h>
#include <ui_gtk_classlist_menu.h>
#include <ui_gtk_class_window.h>
#include <ui_gtk_heapgraph.h>

static stats_context_t stats_context;

static GtkWidget* class_window;

static GtkListStore *object_list = NULL;
static int object_list_size = 0;
static GtkWidget *class_statusbar;

static int max_class_rows = 100;

int get_class_rows () {
    return max_class_rows;
}

void set_class_rows (int rows) {
    max_class_rows = rows;
}


/** array of the class compare functions... */
static int ((*cls_comprs[])(const void* v1, const void* v2)) = { cls_compr_name,
								 cls_compr_instance,
								 cls_compr_max_instance,
								 cls_compr_size,
								 cls_compr_instance_gc,
								 cls_compr_tenure};

static int ((*cls_comprs_r[])(const void* v1, const void* v2)) = { cls_compr_name_r,
								 cls_compr_instance_r,
								 cls_compr_max_instance_r,
								 cls_compr_size_r,
								 cls_compr_instance_gc_r,
								 cls_compr_tenure_r};

static void cls_column_clicked (GtkWidget *treeviewcolumn, gpointer user_data) {
    int column = (int)user_data;
    if (cls_comprs[column] != NULL) {
        int (*old_func)(const void* v1, const void* v2);
        old_func = stats_context_get_compr(&stats_context);
        if(old_func == cls_comprs[column])
          stats_context_set_compr(&stats_context, cls_comprs_r[column]);
        else
          stats_context_set_compr(&stats_context, cls_comprs[column]);
        update_class_tree (get_classes ());
    } else {
	fprintf (stdout, "Sort order not yet implemented.\n");
    }
}

void toggle_class_window (void) {
  if(class_window == NULL)
    return;
  if(GTK_WIDGET_VISIBLE(class_window)) {
    gtk_widget_hide_all (class_window);
  } else {
    gtk_widget_show_all (class_window);
    update_class_tree (get_classes ());
  }
}

static void destroy (GtkWidget *widget, gpointer data) {
  /* NOP */
}

static void init () {
    stats_context_init (&stats_context, &stats_personality_class_usage);
    stats_context_set_compr (&stats_context, cls_compr_size);
}

static void done () {
    stats_context_end (&stats_context);
}

static void build_object_window () {
    GtkWidget *vbox;
    GtkWidget* scroller;
    GtkWidget* tree;
    GtkTreeSelection* select;

    GtkWidget* jmpwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_signal_connect (GTK_OBJECT (jmpwin), "delete-event",
			GTK_SIGNAL_FUNC (gtk_widget_hide_on_delete), NULL);
    gtk_signal_connect (GTK_OBJECT (jmpwin), "destroy",
			GTK_SIGNAL_FUNC (destroy), NULL);
    gtk_window_set_title (GTK_WINDOW (jmpwin), _("Java Memory Profiler - Objects"));
    scroller = gtk_scrolled_window_new (NULL, NULL);
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (jmpwin), vbox);
    gtk_box_pack_start (GTK_BOX (vbox), scroller, TRUE, TRUE, 0);

    object_list = gtk_list_store_new (ON_COLUMNS, G_TYPE_STRING, G_TYPE_LONG, 
				      G_TYPE_LONG, G_TYPE_STRING, G_TYPE_LONG, 
				      G_TYPE_DOUBLE, G_TYPE_POINTER);
    
    tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (object_list));
    add_column (tree, _("Class"), OCLASS_COLUMN, (gpointer)0, cls_column_clicked, 200, 0);
    add_column (tree, _("Instances"), OINSTANCE_COLUMN, (gpointer)1, cls_column_clicked, 80, 1);
    add_column (tree, _("Max instances"), OINSTANCE_MAX_COLUMN, (gpointer)2, cls_column_clicked, 80, 1);
    add_column (tree, _("Size"), OSIZE_COLUMN, (gpointer)3, cls_column_clicked, 80, 1);
    add_column (tree, _("#GC"), OGC_COLUMN, (gpointer)4, cls_column_clicked, 80, 1);
    add_column (tree, _("Tenure"), OTENURE_COLUMN, (gpointer)5, cls_column_clicked, 80, 1);
    gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (tree), TRUE);
    gtk_container_add (GTK_CONTAINER (scroller), tree);
    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree));
    g_signal_connect (G_OBJECT (select), "changed",
		      G_CALLBACK (olist_row_changed),
		      object_list);
    gtk_signal_connect (GTK_OBJECT(tree), "button_press_event",
			GTK_SIGNAL_FUNC (olist_button_handler), 
			NULL);

    class_statusbar = gtk_statusbar_new ();
    gtk_box_pack_start (GTK_BOX (vbox), class_statusbar, FALSE, FALSE, 0);

    gtk_window_set_default_size (GTK_WINDOW (jmpwin), 600, 200);
    ui_gtk_prefs_load_window (UI_GTK_PREFS_CLASS_WINDOW, ui_gtk_state(), GTK_WINDOW (jmpwin));
    class_window = jmpwin;
}

void setup_class_tracing () {
    if (class_window == NULL) {
      init ();
      build_object_window ();
    }
}

void quit_class_window () {
    if (class_window) {
        ui_gtk_prefs_save_window (UI_GTK_PREFS_CLASS_WINDOW, (GtkWindow *)class_window);
	gtk_widget_destroy (class_window);
	class_window = NULL;
        done ();
    }
}

static double get_tenure (cls* c) {
    double n = cls_get_instances (c);
    return 
	(get_current_gc_level () * n - cls_get_tenure (c)) / n;
}

/** Add a row to the gtk CList (update the ui). */
static void add_class_row_to_list (cls* c, int row, GtkTreeIter* iter) {
    int fast = 0;

    if (row < object_list_size) {
	cls* c2;
	gtk_tree_model_get (GTK_TREE_MODEL (object_list), iter, OOBJECT_COLUMN, &c2, -1);
	if (c == c2) {
	    if (!cls_check_modified (c)) {
		gtk_tree_model_iter_next (GTK_TREE_MODEL (object_list), iter);
		return;
	    }
	    fast = 1;
	}
    } else {
	gtk_list_store_append (object_list, iter);
    }
    
    if (fast) {
	gtk_list_store_set (object_list, iter,
			    OINSTANCE_COLUMN, cls_get_instances (c),
			    OINSTANCE_MAX_COLUMN, cls_get_max_instances (c),
			    OSIZE_COLUMN, format_num (cls_get_size (c)),
			    OGC_COLUMN, cls_get_total_gc (c),
			    OTENURE_COLUMN, get_tenure (c),
			    -1);
    } else {
	gtk_list_store_set (object_list, iter,
			    OCLASS_COLUMN, cls_get_name (c),
			    OINSTANCE_COLUMN, cls_get_instances (c), 
			    OINSTANCE_MAX_COLUMN, cls_get_max_instances (c),
			    OSIZE_COLUMN, format_num (cls_get_size (c)),
			    OGC_COLUMN, cls_get_total_gc (c),
			    OTENURE_COLUMN, get_tenure (c),
			    OOBJECT_COLUMN, c,
			    -1);
    }
    cls_set_modified (c, 0);
    gtk_tree_model_iter_next (GTK_TREE_MODEL (object_list), iter);
}

void update_class_tree (hashtab* classes) {
    char buf[64];

    /* make sure that we are initialized correctly. */
    setup_class_tracing ();
    
#if 0	/* needs to run for heapdump stats */
    if (class_window == NULL || !GTK_WIDGET_VISIBLE (class_window))
        return;
#endif    

    /* Single pass filter, count and copy (with total) to expandable arraylist datums */
    stats_context_calc (&stats_context, classes);

    add_heap_size_value (stats_context.p.class_usage.total[1].totalAlloced, 
			 stats_context.p.class_usage.total[0].totalAlloced, 
			 current_heap_size ());

    update_tree (object_list, stats_context.resultlist_count, max_class_rows, (void**)stats_context.resultlist, 
		 (add_row_func)add_class_row_to_list, object_list_size);

    object_list_size = 
	max_class_rows < stats_context.resultlist_count ? max_class_rows : stats_context.resultlist_count;
    snprintf (buf, sizeof(buf), _("Showing %d classes out of %d"), 
	      object_list_size, stats_context.resultlist_total);
    set_status_internal (class_statusbar, buf);
}

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
