#include <jmp-config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
 #include <unistd.h>
#endif
#ifdef HAVE_PTHREAD_H
 #include <pthread.h>
#endif
#ifdef HAVE_WIN32COMPAT_H
 #include <win32compat.h>
#endif
#include <jni.h>
#include <jvmpi.h>

#include <jmp.h>
#include <jmp-debug.h>
#include <time.h>
#include <hash.h>
#include <arena.h>
#include <cls.h>
#include <obj.h>
#include <method.h>
#include <jmpthread.h>
#include <timerstack.h>
#include <objectstore.h>
#include <heap_dump.h>
#include <filter.h>
#include <jmptime.h>
#include <deadlock_detector.h>
#include <string_dumper.h>
#include "jmp_idle_thread.h"

#include <ui.h>
#ifdef CONFIG_UI_GTK
 #include <ui_gtk.h>
#endif

#ifdef linux
/* back trace handling only works in linux for what I know. */
#include <execinfo.h>
#endif /* linux */

/* forward references */
void notifyEvent (JVMPI_Event *e);
void notifyEventSink (JVMPI_Event *e);
static void enable_events (void);


/** This is the interface into the jvm */
JavaVM *JavaVM_jvm;
static JVMPI_Interface* jvmpi;
static JNIEnv *jni;	/* TODO: research longevity of this thread */

/** are we showing any graphical user interface? */
static int doUI = 0;
static int isUIstarted = 0;
/** are we profiling classes. */
static int class_profiling = 0;
/** are we profiling method calls? */
static int method_profiling = 1;
/** are we tracing objects? */
static int object_profiling = 1;
/** Do object allocations follow the filter? */
static int alloc_follow_filter = 0;
/** are we tracing monitors. */
static int monitor_profiling = 1;
/** do we allow dump on signal? */
static int dump_enabled = 0;
/** assume jmp is running in a simulator and don't do jni calls. */
static int simulator = 0;

static int sink_events = 0;

static int verbose = 1;

/** Information about the vm we are running in */
static char* vm_version;
static char* vm_vendor;
static char* vm_name;

/** Use absolute times instead of thread times in method profiling? */
static int absolute_times;

/** These are the hashes where we store our different objects. */
static hashtab* arenas;
static hashtab* classes;
static hashtab* objects;
static hashtab* methods;
static hashtab* threads;

/** The current reset level. */
static int current_reset_level = 0;
/** The minimum level needed when getting objects. */
static int minimum_reset_level = 0;
/** The current gc level. */
static int current_gc_level = 0; 

/** This is our objectstore. */
static objectstore* osp;

/** The names of the primitive array types. */
static char* L = "[L";
static cls* clsLP = 0;
static char* Z = "[Z";
static cls* clsZP = 0;
static char* B = "[B";
static cls* clsBP = 0;
static char* C = "[C";
static cls* clsCP = 0;
static char* S = "[S";
static cls* clsSP = 0;
static char* I = "[I";
static cls* clsIP = 0;
static char* J = "[J";
static cls* clsJP = 0;
static char* F = "[F";
static cls* clsFP = 0;
static char* D = "[D";
static cls* clsDP = 0;

/** Counters for methods. */
static long c_class_load = 0;
static long c_class_unload = 0;
static long c_object_alloc = 0;
static long c_object_move = 0;
static long c_object_free = 0;
static long c_thread_start = 0;
static long c_thread_end = 0;
static long c_thread_end_unknown = 0;
static long c_thread_start_system = 0;
static long c_thread_end_system = 0;
static long c_thread_start_jmp = 0;
static long c_thread_end_jmp = 0;
static long c_method_entry = 0;
static long c_method_exit = 0;

/** Some counter.. */
static long this_gc_object_move = 0;
static long this_gc_object_free = 0;
static jlong this_gc_time = 0;
static jlong heap_size = 0;

/** Where should we dump data files? */
static char* dumpdir = NULL;

static int down = 0;   /** since we get multiple shutdowns... */
static int do_gc_on_window_close = 1;
static int do_gc_on_shutdown = 0;
static char gc_status_description[128];

static hashtab* last_monitor_dump;

/** we need to make sure that we call the right set_status during gc.*/
static int gc_requested = 0;

#ifdef JNI_VERSION_1_4
 static jint jni_version_supported = JNI_VERSION_1_4;
#elif JNI_VERSION_1_2
 static jint jni_version_supported = JNI_VERSION_1_2;
#else
 static jint jni_version_supported = JNI_VERSION_1_1;
#endif
static jint jni_version_using = 0;

#ifdef JVMPI_VERSION_1_2
 static jint jvmpi_version_supported = JVMPI_VERSION_1_2;
#elif JVMPI_VERSION_1_1
 static jint jvmpi_version_supported = JVMPI_VERSION_1_1;
#else
 static jint jvmpi_version_supported = JVMPI_VERSION_1;
#endif
static jint jvmpi_version_using = 0;

static const char *JMP_THREAD_NAME_GTK = "jmp-gtk";
static const char *JMP_THREAD_NAME_NOUI = "jmp-noui";
#ifndef SUPPORT_PTHREAD
static const char *MONITOR_NAME_JVM_SHUTDOWN = "_jmp_jvm_shutdown";
#endif

#ifdef linux
void get_backtrace () {
    int i;
    void *array[10];
    size_t size;
    char **strings;
    size = backtrace (array, 10);
    strings = backtrace_symbols (array, size);
    printf ("Obtained %d stack frames.\n", size);
    
    for (i = 0; i < size; i++)
	printf ("%s\n", strings[i]);	
    free (strings);
}
#endif /* linux */

JVMPI_RawMonitor jvmpi_create_monitor (char *name) {
    return jvmpi->RawMonitorCreate (name);
}

void jvmpi_delete_monitor (JVMPI_RawMonitor m) {
    jvmpi->RawMonitorDestroy (m);
}

void jvmpi_lock_monitor (JVMPI_RawMonitor m) {
    jvmpi->RawMonitorEnter (m);
}

void jvmpi_unlock_monitor (JVMPI_RawMonitor m) {
    jvmpi->RawMonitorExit (m);
}

void jvmpi_monitor_wait (JVMPI_RawMonitor m, jlong milis) {
    jvmpi->RawMonitorWait (m, milis);
}

void jvmpi_monitor_notify_all (JVMPI_RawMonitor m) {
    jvmpi->RawMonitorNotifyAll (m);
}

static void lock_all (void) {
    jmphash_lock (threads, MONITOR_AGENT);
    jmphash_lock (arenas, MONITOR_AGENT);
    jmphash_lock (classes, MONITOR_AGENT);
    jmphash_lock (methods, MONITOR_AGENT);
    jmphash_lock (objects, MONITOR_AGENT);
}

static void unlock_all (void) {
    jmphash_unlock (objects, MONITOR_AGENT);
    jmphash_unlock (methods, MONITOR_AGENT);
    jmphash_unlock (classes, MONITOR_AGENT);
    jmphash_unlock (arenas, MONITOR_AGENT);
    jmphash_unlock (threads, MONITOR_AGENT);
}

/** Test if jmp is currently tracing objects. */
INLINE int tracing_objects () {
    return object_profiling;
}

/** Test if object allocations follows filter. */
INLINE int allocs_follow_filter () {
    return alloc_follow_filter;
}

/** Test if jmp is currently tracing method calls. */
INLINE int tracing_methods () {
    return method_profiling;
}

INLINE int tracing_monitors () {
    return monitor_profiling;
}

/** Test if jmp is using its GUI */
INLINE int usingUI () {
    return doUI;
}


/** Reset the counter on all classes and method. */
void reset_counters () {
    jmphash_lock (classes, MONITOR_AGENT);
    jmphash_for_each ((jmphash_iter_f)cls_reset_count, classes);
    current_reset_level++;
    minimum_reset_level = current_reset_level;
    jmphash_unlock (classes, MONITOR_AGENT);
    jmphash_lock (methods, MONITOR_AGENT);
    jmphash_for_each ((jmphash_iter_f)method_reset_count, methods);
    jmphash_unlock (methods, MONITOR_AGENT);
}


/** Restore the counter to theire full values. */
void restore_counters () {
    jmphash_lock (classes, MONITOR_AGENT);    
    minimum_reset_level = 0;
    jmphash_for_each ((jmphash_iter_f)cls_restore_count, classes);
    jmphash_unlock (classes, MONITOR_AGENT);
    jmphash_lock (methods, MONITOR_AGENT);
    jmphash_for_each ((jmphash_iter_f)method_restore_count, methods);
    jmphash_unlock (methods, MONITOR_AGENT);
}

static void object_free (jobjectID);

/** Run the garbage collector. */
void run_GC () {
    gc_requested = 1;
    jvmpi->RunGC ();    
    gc_requested = 0;
}

/** Get the size of the heap. */
jlong current_heap_size () {
    return heap_size;
}

static void free_the_object (void* v) {
    /* it is not safe to call object_free here, it will mess up the 
     * hashs internal structures. /robo
     */
    obj* op = (obj*)v;
    cls* cp = get_class (obj_get_class_id (op));
    if (cp != NULL) {
	jint os = obj_get_size (op);
	cls_object_free_not_gc (cp, os);
    }
    objectstore_obj_free (osp, op);
}

static void release_object (obj* o) {
    objectstore_obj_free (osp, o);
}

/** Load a primitive array class. */
static cls* load_array_class (char* cls_name) {
    cls* c;
    c = cls_new (cls_name, cls_name, (jobjectID)cls_name, 0, 0, NULL, 0, NULL);
    if (classes) {
	if (c)
	    jmphash_insert (c, classes);
	else 
	    fprintf (stderr, "failed to allocate array class: '%s'\n", cls_name);
    }
    return c;
}

/** Load all of the primitive array classes. */
static void load_array_classes () {
    clsLP = load_array_class (L);
    clsZP = load_array_class (Z);
    clsBP = load_array_class (B);
    clsCP = load_array_class (C);
    clsSP = load_array_class (S);
    clsIP = load_array_class (I);
    clsJP = load_array_class (J);
    clsFP = load_array_class (F);
    clsDP = load_array_class (D);
}

/** Perform a heao dump with the specified level.
 */
static void run_heap_dump_arg (int level) {
    JVMPI_HeapDumpArg heap_arg;    
    heap_arg.heap_dump_level = level;
    lock_all ();
    if ((level == JVMPI_DUMP_LEVEL_0) || (!tracing_objects ())) {
	/* clear out old incorrect information. */
	jmphash_for_each ((jmphash_iter_f)free_the_object, objects);
	jmphash_clear (objects);
    }
    fprintf (stderr, "requesting heap dump: %d\n", level);
    jvmpi->RequestEvent (JVMPI_EVENT_HEAP_DUMP, &heap_arg);
    fprintf (stderr, "heap dumping done: %d\n", level);
    if (class_profiling == 0) {
	/* clear classes, objects and methods... */
	jmphash_for_each ((jmphash_iter_f)release_object, objects);
	jmphash_clear (objects);
	jmphash_for_each ((jmphash_iter_f)method_free, methods); 
	jmphash_clear (methods);
	jmphash_for_each ((jmphash_iter_f)cls_free, classes); 
	jmphash_clear (classes);
	/* and fill in with standard information... */
	load_array_classes ();    
    }
    unlock_all ();
}

void run_heap_dump () {
    run_heap_dump_arg (JVMPI_DUMP_LEVEL_2);
}

void run_object_dump () {
    run_heap_dump_arg (JVMPI_DUMP_LEVEL_0);
}

void run_string_dump () {
    dump_strings ();
}

hashtab* run_monitor_dump () {
    jvmpi->RequestEvent (JVMPI_EVENT_MONITOR_DUMP, NULL);
    detect_deadlock (last_monitor_dump);
    return last_monitor_dump;
}

void cleanup_monitor_information () {
    if (last_monitor_dump) {
	jmphash_for_each ((jmphash_iter_f)free_monitor_info, last_monitor_dump); 
	jmphash_free (last_monitor_dump);
    }
    last_monitor_dump = NULL;
}

/** Ask the jvm about a specific object allocation.
 */
void get_object_alloc (jobjectID obj_id) {
    jint res = jvmpi->RequestEvent (JVMPI_EVENT_OBJECT_ALLOC, obj_id);
    if (res != JVMPI_SUCCESS)
	fprintf (stderr, "failed to get object (%p), error: %d\n", 
		 obj_id, PRINTF_JINT_CAST res);
}

/** Ask the jvm about a specified class load.
 */
void get_class_load (jobjectID class_id) {
    jint res;
    if (class_id == 0) {
        return;
    }

    jmphash_lock_nested (classes, MONITOR_AGENT);
    res = jvmpi->RequestEvent (JVMPI_EVENT_CLASS_LOAD, class_id);
    jmphash_unlock_nested (classes, MONITOR_AGENT);
    if (res != JVMPI_SUCCESS) 
	fprintf (stderr, "failed to get class (%p), error: %d\n", 
		 class_id, PRINTF_JINT_CAST res);
}

/** Ask the jvm about a specified arena.
 *  Note: this method is not available in the SUN jvm.
 */
void get_arena_load (jint arena_id) {
    jint res;

    jmphash_lock_nested (arenas, MONITOR_AGENT);
    res = jvmpi->RequestEvent (JVMPI_EVENT_ARENA_NEW, &arena_id);
    jmphash_unlock_nested (arenas, MONITOR_AGENT);
    if (res != JVMPI_SUCCESS) 
	fprintf (stderr, "failed to get arena (%d), error: %d\n", 
		 PRINTF_JINT_CAST arena_id, PRINTF_JINT_CAST res);
}

/** Do initial setup. */
static int setup () {
    trace ("JMP setup ()\n");
    arenas = jmphash_new (11, arena_jmphash_func, arena_cmp_func, "arenas");
    classes = jmphash_new (2000, cls_jmphash_func, cls_cmp_func, "classes");
    methods = jmphash_new (10000, method_jmphash_func, method_cmp_func, "methods");
    threads = jmphash_new (50, jmpthread_jmphash_func, jmpthread_cmp_func, "threads");
    objects = jmphash_new (100000, obj_jmphash_func, obj_cmp_func, "objects");
    trace ("allocating object store\n");
    osp = objectstore_new ();
    trace ("loading array classes\n");
    load_array_classes ();    
    trace ("setup finished\n");
    return 0;
}

static void wait_for_ui () {
    if (isUIstarted)
	quit_ui ();
}

/** free *p in a safe way, make sure it is unlocked 
 *  after we have set *p to NULL.
 */
static void cleanup_hash (hashtab** p) {
    hashtab* h = *p;
    *p = NULL;
    /* we should always know the lockstate in every possible execution context,
     *  so this must be unesssary and dangerous - dlm */
    /*jmphash_unlock (h);*/
    jmphash_free (h);
}

/** free all elements in *p and free p. 
 * @param p the hashtab to clean up
 * @param f the function to use to free the elements in p.
 */
static void free_and_cleanup_hash (hashtab** p, jmphash_iter_f f) {
    if (*p) {
	jmphash_for_each (f, *p);
	cleanup_hash (p);
    }
}


/** Get a class pointer from its id. */
cls* get_class (const jobjectID class_id) {
    cls c;
    cls* cp = NULL;
    cls_set_class_id (&c, class_id);
    cp = jmphash_search (&c, classes);
    return cp;
}

/** Get the hashtable of all loaded classes. */
hashtab* get_classes () {
    return classes;
}

/** Get an object pointer from its id. */
obj* get_object (const jobjectID obj_id) {
    obj o;
    obj* op = NULL;
    obj_set_object_id (&o, obj_id);
    op = jmphash_search (&o, objects);
    return op;
}

/** Get a method object pointer from its id. */
method* get_method (const jmethodID method_id) {
    method m;
    method* mp = NULL;
    method_set_method_id (&m, method_id);
    mp = jmphash_search (&m, methods);
    return mp;
}

/** Get the hashtable of all methods in the system. */
hashtab* get_methods () {
    return methods;
}

/** Get an jmpthread pointer from its id. */
jmpthread* get_jmpthread (const JNIEnv* env_id) {
    jmpthread t;
    jmpthread* tp = NULL;
    jmpthread_set_env_id (&t, (JNIEnv*)env_id);
    tp = jmphash_search (&t, threads);
    return tp;
}

arena* get_arena (const jint arena_id) {
    arena a;
    arena* ap = NULL;
    arena_set_arena_id (&a, arena_id);
    ap = jmphash_search (&a, arenas);
    return ap;
}

/** get the timerstack for the given thread. */
timerstack* get_timerstack (JNIEnv* env_id) {
    return (timerstack*)jvmpi->GetThreadLocalStorage (env_id);    
}

hashtab* get_threads () {
    return threads;
}

/** Get the given system property str and put the value int dest, 
 */
static void get_string (char** dest, char* str, JNIEnv* env, jclass clz, jmethodID jmethod) {
    jstring j_str;
    const char *utf8;
    j_str = (*env)->NewStringUTF (env, str);
    j_str = (*env)->CallStaticObjectMethod (env, clz, jmethod, j_str);
    utf8 = (*env)->GetStringUTFChars (env, j_str, NULL);
    *dest = strdup (utf8);
    trace3 ("    %s = %s\n", str, utf8);
    (*env)->ReleaseStringUTFChars (env, j_str, utf8);
}

static const char *jvm_jni_version_string (jint vers) {
  switch (vers) {
  case JNI_VERSION_1_1:
    return "1.1";

#ifdef JNI_VERSION_1_2
  case JNI_VERSION_1_2:
    return "1.2";
#endif

#ifdef JNI_VERSION_1_4
  case JNI_VERSION_1_4:
    return "1.4";
#endif
  }
  return "unknown";
}

static JNIEnv *jvm_detect_jni_version (JavaVM *jvm) {
    JNIEnv *jni_this;
    JNIEnv **jni_this_p;
    jni_this_p = &jni_this;

#ifdef JNI_VERSION_1_4
    if (jni_version_using == 0) {
        if ((*jvm)->GetEnv(jvm, (void **) jni_this_p, JNI_VERSION_1_4) == 0)
            jni_version_using = JNI_VERSION_1_4;
    }
#endif

#ifdef JNI_VERSION_1_2
    if (jni_version_using == 0) {
        if ((*jvm)->GetEnv(jvm, (void **) jni_this_p, JNI_VERSION_1_2) == 0)
            jni_version_using = JNI_VERSION_1_2;
    }
#endif

    if (jni_version_using == 0) {
        if ((*jvm)->GetEnv(jvm, (void **) jni_this_p, JNI_VERSION_1_1) == 0)
            jni_version_using = JNI_VERSION_1_1;
    }

    if (jni_version_using)
        jni = jni_this;

    return jni_this;
}

static const char *jvm_jvmpi_version_string (jint vers) {
  switch (vers) {
  case JVMPI_VERSION_1:
    return "1.0";

#ifdef JVMPI_VERSION_1_1
  case JVMPI_VERSION_1_1:
    return "1.1";
#endif

#ifdef JVMPI_VERSION_1_2
  case JVMPI_VERSION_1_2:
    return "1.2";
#endif
  }
  return "unknown";
}

static jint jvm_detect_jvmpi_version (JavaVM *jvm) {
    JVMPI_Interface** jvmpi_p = &jvmpi;
#ifdef JVMPI_VERSION_1_2
    if (jvmpi_version_using == 0) {
        if ((*jvm)->GetEnv(jvm, (void **) jvmpi_p, JVMPI_VERSION_1_2) == 0)
            jvmpi_version_using = JVMPI_VERSION_1_2;
    }
#endif

#ifdef JVMPI_VERSION_1_1
    if (jvmpi_version_using == 0) {
        if ((*jvm)->GetEnv(jvm, (void **) jvmpi_p, JVMPI_VERSION_1_1) == 0)
            jvmpi_version_using = JVMPI_VERSION_1_1;
    }
#endif

    if (jvmpi_version_using == 0) {
        if ((*jvm)->GetEnv(jvm, (void **) jvmpi_p, JVMPI_VERSION_1) == 0)
            jvmpi_version_using = JVMPI_VERSION_1;
    }

    return jvmpi_version_using;
}

static int o_jmp_started;
static jobject o_jmp_idle_ref;
static const char *JMP_THREAD_NAME_IDLE = "jmp-shutdown-holder";
static const char *JMP_PACKAGE_CLASS = "jmp/JMPIdleThread";
#ifdef SUPPORT_PTHREAD
static PTHREAD_MUTEX_T(jvm_shutdown_thread_mutex);
#else
static JVMPI_RawMonitor jvm_shutdown_thread_monitor;
#endif

/* TODO: We want the jmp user threads to below to their own threadgroup
 *  so appliations using the main threadgroup wont affect us.
 *  new ThreadGroup();
 *  Thread.currentThread().getThreadGroup().getParent();
 *
 * Special care is taken here as jvm_shutdown_thread_stop() maybe called
 *  from either a JVM system thread or from out GTK GUI thread.  This is
 *  to make certain the JMPIdleThread is blown away for sure otherwise we
 *  risk a hung JVM.
 *
 */
int jvm_shutdown_thread_stop (JNIEnv* env) {
    jclass c_jmp_idle_class;
    jmethodID m_stop;
    jmethodID m_jmp_idle_waitForShutdown;
    int rv;

    PTHREAD_MUTEX_LOCK (jvm_shutdown_thread_mutex);
    if (o_jmp_started == 0) {
	PTHREAD_MUTEX_UNLOCK (jvm_shutdown_thread_mutex);
        return 0;
    }
    o_jmp_started = 0;
    PTHREAD_MUTEX_UNLOCK (jvm_shutdown_thread_mutex);

    rv = -1;

    c_jmp_idle_class = (*env)->FindClass (env, JMP_PACKAGE_CLASS);
    if(c_jmp_idle_class == 0)
        goto done;

    m_stop = (*env)->GetMethodID (env, c_jmp_idle_class, "stop", "()V");
    if (m_stop == 0)
        goto done;

    (*env)->CallVoidMethod (env, o_jmp_idle_ref, m_stop);

    /* jmpIdle.waitForShutdown(); */
    m_jmp_idle_waitForShutdown = (*env)->GetMethodID (env, c_jmp_idle_class, "waitForShutdown", "()V");
    if (m_jmp_idle_waitForShutdown == 0)
        goto done;

    (*env)->CallVoidMethod (env, o_jmp_idle_ref, m_jmp_idle_waitForShutdown);

    /* We can still access it here */

    rv = 0;

done:
    /* Keep it around, even when it dies */
    if (o_jmp_idle_ref != 0) {
        (*env)->DeleteGlobalRef (env, o_jmp_idle_ref);
        o_jmp_idle_ref = 0;
    }

    return rv;
}

static int jvm_shutdown_thread_start (JNIEnv* env) {
    jobject o_loader;
    jclass c_class_loader;
    jmethodID m_get_system_class_loader;
    jclass c_jmp_idle_class;

    PTHREAD_MUTEX_LOCK (jvm_shutdown_thread_mutex);
    if (o_jmp_started) {
	/* We have already created the thread */
	PTHREAD_MUTEX_UNLOCK (jvm_shutdown_thread_mutex);
        return 0;
    }
    o_jmp_started = 1;
    PTHREAD_MUTEX_UNLOCK (jvm_shutdown_thread_mutex);

    // ClassLaoder o_loader = java.lang.ClassLoader.getSystemClassLoader()
    c_class_loader = (*env)->FindClass (env, "java/lang/ClassLoader");
    if(c_class_loader == 0)
        goto error;
    //fprintf(stderr, "c_class_loader=%p\n", c_class_loader);

    m_get_system_class_loader = (*env)->GetStaticMethodID (env, c_class_loader, 
							   "getSystemClassLoader", 
							   "()Ljava/lang/ClassLoader;");
    if(m_get_system_class_loader == 0)
        goto error;
    //fprintf(stderr, "m_get_system_class_loader=%p\n", m_get_system_class_loader);

    o_loader = (*env)->CallStaticObjectMethod (env, c_class_loader, m_get_system_class_loader);
    if(o_loader == 0)
        goto error;
    //fprintf(stderr, "o_loader=%p\n", o_loader);

    // JMPIdleThread jmpIdle = new jmp.JMPIdleThread()
    c_jmp_idle_class = (*env)->DefineClass (env, JMP_PACKAGE_CLASS, o_loader, 
					    (const jbyte*)jmp_idle_thread_data, 
					    jmp_idle_thread_len);
    if(c_jmp_idle_class == 0)
        goto error;
    //fprintf(stderr, "c_jmp_idle_class=%p\n", c_jmp_idle_class);

    {
	jmethodID m_jmp_idle_ctor;
	jobject o_jmp_idle;
	jclass c_thread;
	jmethodID m_thread_ctor;
	jobject o_thread;
	jmethodID m_start;
	jmethodID m_get_thread_id;
	jlong thread_id;
	jstring st_jmp_thread_name;
	jmethodID m_jmp_idle_waitForStartup;
	
	m_jmp_idle_ctor = (*env)->GetMethodID (env, c_jmp_idle_class, "<init>", "()V");
	if(m_jmp_idle_ctor == 0)
	    goto error;
	//fprintf(stderr, "m_jmp_idle_ctor=%p\n", m_jmp_idle_ctor);
	

	o_jmp_idle = (*env)->NewObject (env, c_jmp_idle_class, m_jmp_idle_ctor);
	if(o_jmp_idle == 0)
	    goto error;
	//fprintf(stderr, "o_jmp_idle=%p\n", o_jmp_idle);
	
	m_get_thread_id = (*env)->GetMethodID (env, c_jmp_idle_class, "getThreadId", "()J");
	if(m_get_thread_id == 0)
	    goto error;
	//fprintf(stderr, "m_get_thread_id=%p\n", m_get_thread_id);
	
	// thread = new Thread(jmpIdle, String);
	c_thread = (*env)->FindClass (env, "java/lang/Thread");
	if(c_thread == 0)
        goto error;
	//fprintf(stderr, "c_thread=%p\n", c_thread);

	m_thread_ctor = (*env)->GetMethodID (env, c_thread, "<init>", "(Ljava/lang/Runnable;Ljava/lang/String;)V");
	if(m_thread_ctor == 0)
	    goto error;
	//fprintf(stderr, "m_thread_ctor=%p\n", m_thread_ctor);

	st_jmp_thread_name = (*env)->NewStringUTF (env, JMP_THREAD_NAME_IDLE);
	if(st_jmp_thread_name == 0)
	    goto error;
	//fprintf(stderr, "st_jmp_thread_name=%p\n", st_jmp_thread_name);

	o_thread = (*env)->NewObject (env, c_thread, m_thread_ctor, o_jmp_idle, st_jmp_thread_name);
	if(o_thread == 0)
	    goto error;
	//fprintf(stderr, "o_thread=%p\n", o_thread);

	// thread.start()
	m_start = (*env)->GetMethodID (env, c_thread, "start", "()V");
	if(m_start == 0)
	    goto error;
	//fprintf(stderr, "m_start=%p\n", m_start);

	(*env)->CallVoidMethod (env, o_thread, m_start);
	//fprintf(stderr, "Started!\n");

	// jmpIdle.waitForStartup();
	m_jmp_idle_waitForStartup = (*env)->GetMethodID (env, c_jmp_idle_class, "waitForStartup", "()V");
	if(m_jmp_idle_waitForStartup == 0)
	    goto error;
	//fprintf(stderr, "m_jmp_idle_waitForStartup=%p\n", m_jmp_idle_waitForStartup);

	(*env)->CallVoidMethod (env, o_jmp_idle, m_jmp_idle_waitForStartup);

	thread_id = (*env)->CallLongMethod (env, o_jmp_idle, m_get_thread_id);
	//fprintf(stderr, "thread_id=%lld WAITED\n", thread_id);

    /* Keep it around, even when it dies */
	o_jmp_idle_ref = (*env)->NewGlobalRef (env, o_jmp_idle);
	if (o_jmp_idle_ref == 0)
	    goto error;
    //fprintf(stderr, "o_jmp_idle_ref=%p\n", o_jmp_idle_ref);
    }

    return 0;

error:
    return -1;
}

/** Perform additional setup that cant be done in setup () 
 *  This will mean that we get the jvm vendor/version/name to correctly 
 *  select a timing function (maybee remove all time captured up to this point?). 
 */
static void jvm_init_done (JNIEnv* env) {
    jint tstatus;
    trace ("jvm_init_done.\nContinuing with JMP setup...\n");

    sink_events = 0;

    if (doUI || get_dump_timer () > 0) {
        const char *thread_name;

	if (doUI)
	    thread_name = JMP_THREAD_NAME_GTK;
	else
	    thread_name = JMP_THREAD_NAME_NOUI;
	
	tstatus = jvmpi->CreateSystemThread((char *)thread_name, 
					    JVMPI_NORMAL_PRIORITY, 
					    gtkthread);
	if (tstatus == JNI_ERR)
	    fprintf (stdout, 
		     "JMP worker thread create status: %d (ok: %d, bad: %d)\n", 
		     PRINTF_JINT_CAST tstatus, 
		     JNI_OK, 
		     JNI_ERR);

        start_ui ();

        if (doUI)
    jvm_shutdown_thread_start (env);

        isUIstarted = 1;	/* must be set before thread started */
    }
    
    trace ("JMP setup complete...\n");    
} 

static int run_data_dump_unlocked () {
    int ret = -1;
    if (objects && classes && methods) 
	ret = dump_data (dumpdir, classes, methods, threads);
    return ret;
}

int
run_data_dump () {
    int ret = -1;
    
    trace ("run_data_dump ()\n");
    lock_all ();
    ret = run_data_dump_unlocked ();
    unlock_all ();    
    return ret;
}

#ifdef CONFIG_UI_GTK
void
run_show_objects_alloced_by_method (method* m) {
    trace2 ("run_show_objects_alloced_by_method (%s)\n", m->jmpname);
    if (!down && objects)
	show_objects_alloced_by_method (objects, m, minimum_reset_level);
}


void
run_show_objects_alloced_by_class (cls* c) {
    trace2 ("run_show_objects_alloced_by_class (%s)\n", c->name);
    if (!down && objects)
	show_objects_alloced_by_class (objects, c, minimum_reset_level);
}


void
run_find_owners (cls* c) {
    lock_all ();
    show_instance_owners (objects, c);
    unlock_all ();
}

void run_find_instance_owners (obj* o) {
    lock_all ();
    show_owner_for_object (objects, o);
    unlock_all ();
}

void
run_owners_statistics (cls* c) {
    lock_all ();
    show_owners_statistics (objects, c);
    unlock_all ();
}
#endif


typedef struct string_callback_data {
    string_callback sc;
    void* data;
} string_callback_data;

static void filter_strings (obj* o, string_callback_data* scd) {
    if (obj_get_class (o) == clsCP && minimum_reset_level <= obj_get_reset_level (o))
	scd->sc (o, scd->data);
}

void for_each_string (string_callback sc, void* data) {
    string_callback_data scd; 
    scd.sc = sc;
    scd.data = data;
    jmphash_lock (objects, MONITOR_AGENT);
    jmphash_for_each_with_arg ((jmphash_iter_fa)filter_strings, objects, &scd);
    jmphash_unlock (objects, MONITOR_AGENT);
}

/** Get the super class of c
 *  NOTE: call this with gc disabled (heap dump has gc disabled, so 
 *  only needed if you add another path into this method).
 * 
 * @param c the cls we want to get the super class for.
 */
cls* get_super_class (cls* c) {
    if (c->super)
	return c->super;
    /* jvm does not like our fake ids. */
    if (c == clsLP || c == clsZP || c == clsBP || c == clsCP || c == clsSP || 
	c == clsIP || c == clsJP || c == clsFP || c == clsDP)
	return NULL;
    jvmpi->RequestEvent (JVMPI_EVENT_OBJECT_DUMP, cls_get_class_id (c));
    return c->super;
}

/** Call get_last_down_link after this to get the object links.
 */
void get_instance_info (obj* o) {
    jvmpi->DisableGC ();
    jvmpi->RequestEvent (JVMPI_EVENT_OBJECT_DUMP, obj_get_object_id (o));
    jvmpi->EnableGC ();
}

void check_objects (size_t cnt) {
    int diff = jmphash_cardinal (objects) - cnt;
    fprintf (stderr, "objects has: %u, cnt is %u (%u)\n", jmphash_cardinal (objects), cnt, diff);
}

static void data_reset () {
    trace ("data_reset ()\n");
}

static void disable_gc_events () {
    jvmpi->DisableEvent (JVMPI_EVENT_GC_START, NULL);
    jvmpi->DisableEvent (JVMPI_EVENT_GC_FINISH, NULL);
}

static void disable_arena_events () {
    jvmpi->DisableEvent (JVMPI_EVENT_ARENA_NEW, NULL);
    jvmpi->DisableEvent (JVMPI_EVENT_ARENA_DELETE, NULL);
}

static void disable_class_events () {
    jvmpi->DisableEvent (JVMPI_EVENT_CLASS_LOAD, NULL);
    jvmpi->DisableEvent (JVMPI_EVENT_CLASS_UNLOAD, NULL);
    jvmpi->DisableEvent (JVMPI_EVENT_OBJECT_MOVE, NULL);
}

void disable_object_events () {
    if (object_profiling) {
	/* Disable object tracing events. */
	jvmpi->DisableEvent (JVMPI_EVENT_OBJECT_ALLOC, NULL);
	jvmpi->DisableEvent (JVMPI_EVENT_OBJECT_FREE, NULL);
	object_profiling = 0;
    }
}

static void disable_thread_events (int keep_threads) {
    if (!keep_threads) {
        jvmpi->DisableEvent (JVMPI_EVENT_THREAD_START, NULL);
	jvmpi->DisableEvent (JVMPI_EVENT_THREAD_END, NULL);
    }
}

void disable_method_events () {
    if (method_profiling) {
	/* Disable method profiling events. */
	jvmpi->DisableEvent (JVMPI_EVENT_METHOD_ENTRY, NULL);
	jvmpi->DisableEvent (JVMPI_EVENT_METHOD_EXIT, NULL);
	method_profiling = 0;
    }
}

static void disable_dump_events () {
    if (dump_enabled) {
	jvmpi->DisableEvent (JVMPI_EVENT_DATA_DUMP_REQUEST, NULL);
	jvmpi->DisableEvent (JVMPI_EVENT_DATA_RESET_REQUEST, NULL);
	dump_enabled = 0;
    }
}

void disable_monitor_events () {
    if (monitor_profiling) {
	jvmpi->DisableEvent (JVMPI_EVENT_MONITOR_CONTENDED_ENTER, NULL);
	jvmpi->DisableEvent (JVMPI_EVENT_MONITOR_CONTENDED_ENTERED, NULL);
	jvmpi->DisableEvent (JVMPI_EVENT_MONITOR_CONTENDED_EXIT , NULL);
	jvmpi->DisableEvent (JVMPI_EVENT_MONITOR_WAIT, NULL);
	jvmpi->DisableEvent (JVMPI_EVENT_MONITOR_WAITED , NULL);
	
	jvmpi->DisableEvent (JVMPI_EVENT_RAW_MONITOR_CONTENDED_ENTER, NULL);
	jvmpi->DisableEvent (JVMPI_EVENT_RAW_MONITOR_CONTENDED_ENTERED, NULL);
	jvmpi->DisableEvent (JVMPI_EVENT_RAW_MONITOR_CONTENDED_EXIT, NULL);

	monitor_profiling = 0;
    }
}

jint get_thread_state (const jmpthread* jt) {
    return jvmpi->GetThreadStatus (jt->env_id);
}

int get_current_gc_level () {
    return current_gc_level;
}

static void disable_events (int keep_threads) {
    /* Disable standard events */
    jvmpi->DisableEvent (JVMPI_EVENT_JVM_INIT_DONE, NULL);
    jvmpi->DisableEvent (JVMPI_EVENT_JVM_SHUT_DOWN, NULL);
    disable_gc_events ();
    disable_arena_events ();
    disable_class_events ();
    disable_object_events ();
    disable_thread_events (keep_threads);
    disable_method_events ();
    disable_dump_events ();
    disable_monitor_events ();
}

static void platform_end (void) {
    PTHREAD_MUTEX_DESTROY (jvm_shutdown_thread_mutex);
    return;
}

static int platform_init (void) {
    int res;
    PTHREAD_MUTEX_INIT (jvm_shutdown_thread_mutex, NULL, MONITOR_NAME_JVM_SHUTDOWN, &res);
    if (res < 0)
        return -1;
    return 0;
}

/** Tear down all allocations we have made. 
 */
static int teardown () {
    trace ("JMP teardown ()\n");    
    if (verbose > 1) {
      fprintf (stdout, _("teardown called, freeing jmp-data..\n"));
      fflush (stdout);
    }

    down = 1;

    /* suspend input event processing */
    sink_events = 1;
    /* FIXME, we need a way to barrier() the change above and wait
     *  until all threads are out of notifyEvent(), the pureist might
     *  stick an atomic inuse counter on the handler.
     */

    wait_for_ui ();

    free_last_down_link ();
    cleanup_monitor_information ();
    remove_owners_information ();
    if (objects) {
	/*fprintf (stdout, _("freeing objects: %d..\n"), jmphash_cardinal (objects));*/
	/* freeing the objectstore will free all objects in it.. */
	objectstore_free (osp);
	cleanup_hash (&objects);
    }
    free_and_cleanup_hash (&methods, (jmphash_iter_f)method_free);
    free_and_cleanup_hash (&classes, (jmphash_iter_f)cls_free);
    free_and_cleanup_hash (&arenas, (jmphash_iter_f)arena_free);
    free_and_cleanup_hash (&threads, (jmphash_iter_f)jmpthread_free);
    if (verbose > 1) {
      fprintf (stdout, _("done freeing jmp-data..\n"));    
      fflush (stdout);
    }
    if (dumpdir != NULL) {
      free (dumpdir); /* NULL or malloc'ed */
      dumpdir = NULL;
    }

    end_ui ();

    platform_end ();

    if (verbose > 1) {
      fprintf (stdout, _("teardown complete.\n"));
      fflush (stdout);
    }
    return 0;
}

/** The jvm is shutting down so clean up our mess.
 */
static void jvm_shut_down (JNIEnv* env) {
    if (down)
	return;
    trace ("jvm_shut_down ()\n");
    fprintf (stdout, "jvm_shut_down.\n");
    fflush (stdout);

    jvm_shutdown_thread_stop (env);

    notify_jvm_shutdown_ui ();

    if (isUIstarted)
        stop_ui ();

    if (do_gc_on_shutdown) {
        fprintf (stdout, "Running GC on shutdown\n");
        fflush (stdout);
        run_GC ();
        fprintf (stdout, "%s\n", gc_status_description);
        fflush (stdout);
    }

    disable_events (1);

    if (vm_version)
	free (vm_version);
    if (vm_vendor)
	free (vm_vendor);
    if (vm_name)
	free (vm_name);

    lock_all ();

    /* print statistics. */
    fprintf (stdout, "c_class_load: %ld\n", c_class_load);
    fprintf (stdout, "c_class_unload: %ld\n", c_class_unload);
    fprintf (stdout, "c_object_alloc: %ld\n", c_object_alloc);
    fprintf (stdout, "c_object_move: %ld\n", c_object_move);
    fprintf (stdout, "c_object_free: %ld (diff %ld)\n", c_object_free, c_object_alloc - c_object_free);
    fprintf (stdout, "c_thread_start: %ld\n", c_thread_start);
    fprintf (stdout, "c_thread_end: %ld\n", c_thread_end);
    if (c_thread_end_unknown != 0)
        fprintf (stdout, "c_thread_end_unknown: %ld\n", c_thread_end_unknown);
    fprintf (stdout, "c_thread_start_system: %ld\n", c_thread_start_system);
    fprintf (stdout, "c_thread_end_system: %ld\n", c_thread_end_system);
    fprintf (stdout, "c_thread_start_jmp: %ld\n", c_thread_start_jmp);
    fprintf (stdout, "c_thread_end_jmp: %ld\n", c_thread_end_jmp);
    fprintf (stdout, "c_method_entry: %ld\n", c_method_entry);
    fprintf (stdout, "c_method_exit: %ld\n", c_method_exit);
    fflush (stdout);
    
    trace ("running data dump\n");
    run_data_dump_unlocked ();

    unlock_all ();

    trace ("tearing down jmp data structures\n");
    teardown ();

    /* Unlocking here should be done, except that 
     * all hashes are gone, so we cant do it... 
     * unlock_all ();
    */
}

/** grab the locks needed during gc.
 */
static void gc_start () {
    trace2 ("gc_start (), %d\n", gc_requested);
    if (down)
	return;
    lock_all ();
    set_status (_("Running garbage collection..."));
    this_gc_object_move = 0;
    this_gc_object_free = 0;
    this_gc_time = get_absolute_time (jvmpi);
}

/** Releas the locks needed taken during gc. Show some info.
 */
static void gc_finish (jlong used_objects,
		       jlong used_object_space,
		       jlong total_object_space) {
    jlong td, jsec, jnsec;
    td = get_absolute_time (jvmpi) - this_gc_time;
    jmptime_set_time_values (td, &jsec, &jnsec);
    if (down)
	return;
    remove_owners_information ();
    trace2 ("gc_finish (), %d\n", gc_requested);
    snprintf (gc_status_description, sizeof(gc_status_description), 
	      _("Garbage collection completed: %ld objects moved, "
		"%ld objects freed in %lld.%06lld seconds"), 
	      this_gc_object_move, this_gc_object_free, jsec, jnsec);
    set_status (gc_status_description); 
    heap_size = total_object_space;
    current_gc_level++;
    unlock_all ();
}


/** A new arena has arrived.
 */
static void jvmpi_arena_new (jint arena_id, const char* arena_name) {
    arena* a;
    char buffer[80];
    
    snprintf (buffer, sizeof(buffer), _("Arena %d, %s created."), 
	      PRINTF_JINT_CAST arena_id, arena_name);
    fprintf (stderr, "arena_new: %d, %s\n", PRINTF_JINT_CAST arena_id, arena_name);
    set_status (buffer);
    trace3 ("arena_new: (%d, %s)\n", PRINTF_JINT_CAST arena_id, arena_name);
    if (!arenas)
	return;
    
    a = arena_new (arena_id, arena_name);
    if (a == NULL) {
	fprintf (stderr, "arena_new: failed to allocate arena %d, '%s'\n",
		 PRINTF_JINT_CAST arena_id, arena_name);
    } else {
	jmphash_lock (arenas, MONITOR_AGENT);
	jmphash_insert (a, arenas);
	jmphash_unlock (arenas, MONITOR_AGENT);
    }
}

/** Free an object if it is in the arena that is being destroyed.. */
void free_object_if_in_arena (obj* o, jint* arena_id) {
    if (o->arena_id == *arena_id)
	object_free (o->obj_id);
}

/** An arena is destroyed, every object in arena are also deallocated so remove
 *  them (we do not get an explicit object_freed for them).
 */
static void arena_delete(jint arena_id) {
    /* The JVMPI_EVENT_ARENA_DELETE is issued in
       the thread suspended mode.  So we should
       NOT use malloc() here.  That's the reason
       for this arena-initialization by hand
       instead of using arena_new().  Change it
       if somebody has a better idea. */
    /* The locks are aquired in the gc_start()
       already! */
    arena a;
    arena *ah = NULL;
    
    trace2 ("arena_delete (%d)\n", arena_id);
    arena_set_arena_id (&a, arena_id);
    
    ah = jmphash_search (&a, arenas);
    if (!ah) {
	/* I don't think that's good either (may malloc), but
	   since that's only an error message, I hope
	   it's ok for now. */
	fprintf (stderr, "arena_delete: failed to find freed arena: %d.\n",
		 PRINTF_JINT_CAST arena_id);
    } else {
	jmphash_remove (ah, arenas);
	arena_free (ah);
    }
    
    jmphash_for_each_with_arg ((jmphash_iter_fa)free_object_if_in_arena, 
			    objects, &arena_id);
}

#if DLMDEBUG
static jobjectID dlm_global_object_cls_id = 0;	/* DEBUGGING REMOVE ME */
static jmethodID dlm_global_object_method_id_0 = 0; /* DEBUGGING REMOVE ME */
static jmethodID dlm_global_object_method_id_1 = 0; /* DEBUGGING REMOVE ME */
static jmethodID dlm_global_object_method_id_2 = 0; /* DEBUGGING REMOVE ME */
#endif

/** A class has been loaded. 
 *  Locks will only be taken if requested = 0.
 *  Requested class_load are assumed to be locked already..
 */
static void class_load (JNIEnv* env,
			const char *class_name, 
			const char *source_name, 
			jint num_interfaces,
			jint num_methods, 
			JVMPI_Method *meths, 
			jint num_static_fields,
			JVMPI_Field *statics, 
			jint num_instance_fields, 
			JVMPI_Field *instances, 
			jobjectID class_id,
			jint requested) {
    cls* c;
    method* m;
    int i;

    trace3 ("class_load: (%s, %p ...)\n", class_name, class_id);

    if (down || !classes)
	return;

    c_class_load++;
    
    /* Store the class. */
    if (!requested)
	jmphash_lock_nested (classes, MONITOR_AGENT);

    c = get_class (class_id);
    if (c == NULL) {
	c = cls_new (class_name, source_name, class_id, num_interfaces, 
		     num_static_fields, statics, 
		     num_instance_fields, instances);
	if (c == NULL) {
	    fprintf (stderr, "class_load: failed to allocate cls: %s, %s, %p.\n", 
		     class_name, source_name, class_id);
	} else {
	    jmphash_insert (c, classes);
#if DLMDEBUG
//fprintf(stderr, "**** Object=%s %s c=%p cid=%p\n", class_name, source_name, c, class_id);
#endif
	    if (!strcmp ("java/lang/Object", class_name) || !strcmp ("java.lang.Object", class_name)) {
		cls_print (c);
#if DLMDEBUG
//fprintf(stderr, "XXXX Object=%s %s c=%p cid=%p\n", class_name, source_name, c, class_id);
dlm_global_object_cls_id = class_id;
#endif
            }
	}
	
	/* and its methods */
	if (!requested)
	    jmphash_lock_nested (methods, MONITOR_AGENT);
	
	for (i = 0; i < num_methods; i++) {
	    JVMPI_Method *jmethod;
	    jmethod = meths + i;
	    m = get_method (jmethod->method_id);
	    if (m == NULL) {
#if DLMDEBUG
if (!strcmp ("java/lang/Object", class_name) || !strcmp ("java.lang.Object", class_name)) {
  if (strcmp("wait", jmethod->method_name) == 0) {
   fprintf(stderr, "XXXX Object=%s sig=%s mid=%p\n", jmethod->method_name, jmethod->method_signature, jmethod->method_id);
   if (strcmp("()V", jmethod->method_signature) == 0)
     dlm_global_object_method_id_0 = jmethod->method_id;
   else if (strcmp("(JI)V", jmethod->method_signature) == 0)
     dlm_global_object_method_id_1 = jmethod->method_id;
   else if (strcmp("(J)V", jmethod->method_signature) == 0)
     dlm_global_object_method_id_2 = jmethod->method_id;
  }
}
#endif

		m = method_new (jmethod->method_name, jmethod->method_signature,
				jmethod->start_lineno, jmethod->end_lineno, 
				jmethod->method_id, c);
		if (m == NULL) {
		    fprintf (stderr, "class_load: failed to allocate method: %s, %p, %s, %s.\n",
			     class_name, class_id, jmethod->method_name, jmethod->method_signature);
		} else {
		    jmphash_insert (m, methods);
		}
	    }
	}

	if (!requested)
	    jmphash_unlock_nested (methods, MONITOR_AGENT);
    }
    
    if (!requested)
	jmphash_unlock_nested (classes, MONITOR_AGENT);
}


/** A class has been unloaded.
 * TODO evaluate what should happen to methods for this class, we
 * probably want to keep them. 
 */
static void class_unload (jobjectID class_id) {
    cls* cp = NULL;

    trace2 ("class_unload (%p)\n", class_id);
    if (down || !classes)
	return;
    c_class_unload++;
    cp = get_class (class_id);
    if (!cp) {
	fprintf (stderr, "class_unload: failed to find unloaded class: %p.\n", class_id);
    } else {
	/* It should be marked as unloaded but there
	   are still interesting statistics. So it
	   should not be removed! */
	/* dont remove it if we have objects that have not been GC:ed... */
	/* jmphash_remove (cp, classes);
	   cls_free (cp);*/
    }
}

static cls* get_class_pointer (jint is_array, jobjectID class_id) {
    cls* cp = NULL;
    switch (is_array) {
    case JVMPI_NORMAL_OBJECT:	
	cp = get_class (class_id);
	break;
    case JVMPI_CLASS:
	/* The documentation say that class_id is null always for [L... to bad.*/
	cp = clsLP;
	break;
    case JVMPI_BOOLEAN:
	cp = clsZP;
	break;
    case JVMPI_BYTE:
	cp = clsBP; 
	break;
    case JVMPI_CHAR:
	cp = clsCP; 
	break;
    case JVMPI_SHORT:
	cp = clsSP; 
	break;
    case JVMPI_INT:
	cp = clsIP; 
	break;
    case JVMPI_LONG:
	cp = clsJP; 
	break;
    case JVMPI_FLOAT:
	cp = clsFP; 
	break;
    case JVMPI_DOUBLE:
	cp = clsDP; 
	break;
    } 
    return cp;
}

/** Create a timerstack for the given thread.
 */
static INLINE timerstack* setup_thread_local_storage (JNIEnv* thread_env_id) {
    timerstack* s;
    s = (timerstack*)jvmpi->GetThreadLocalStorage (thread_env_id);
    if (s)
	return s;

    /* Ok, since we have the GC disabled this will deadlock
     * if we try anything fancy like: 
     * jobjectID thread_id = jvmpi->GetThreadObject (env_id);
     * jvmpi->RequestEvent (JVMPI_EVENT_THREAD_START, thread_id);
     * 
     * We can not get stack frames for a thread that has not started.
     */

    /* we do not need to lock, we are the thread that handle this storage. */
    trace2 ("setup_thread_local_storage %p\n", thread_env_id);
    s = timerstack_new (100);
    if (s == NULL) 
	fprintf (stderr, "thread_start: failed to allocate thread local stoarge.\n");
    jvmpi->SetThreadLocalStorage (thread_env_id, s);
    return s;
}

/** A java object is being allocated...
 */
static void object_alloc (jint arena_id,
			  jobjectID class_id,
			  jint is_array,
			  jint size,
			  jobjectID obj_id,
			  jint requested,
			  JNIEnv* env) {
    cls* cp = NULL;
    obj* op = NULL;
    method* mp = NULL;	    
    
    if (down || !objects)
	return;

    trace3 ("object_alloc (%p, %p)\n", class_id, obj_id);
    c_object_alloc++;

    /* First some quick manipulation of the class */
    jmphash_lock (classes, MONITOR_AGENT);
//if(dlm_global_object_cls_id == class_id)
// fprintf(stderr, "AAAA Alloc cid=%p oid=%p\n", class_id, obj_id);
    cp = get_class_pointer (is_array, class_id);
    if (!cp) {
        /* Ok, we dont know about this class yet, request it */
        jmphash_unlock (classes, MONITOR_AGENT);
        get_class_load (class_id);
        jmphash_lock (classes, MONITOR_AGENT);
        cp = get_class (class_id);	    
    }
    if (cp) 
	cls_object_alloc (cp, size, current_gc_level);
    jmphash_unlock (classes, MONITOR_AGENT);

    /* We dont change cp from now on,
     * only use the pointer which never changes. 
     */
    if (cp) {
	if (method_profiling) {
	    methodtime* mt;
	    timerstack* t = setup_thread_local_storage (env);
	    if (t) {
		/* SUN's jvm seems evil. It creates a lot of
		   objects before sending thread start.
		   Calling getCallTrace on these will give 0
		   frames back (to bad). which means that we
		   wont get any method here...
		*/
		/* object alloc is the owner of this thread, 
		 * no possible way to have a method_(exit|entry)
		 * so we do not need to lock. /robo
		 * 
		 * Actually we do, the thread window can be 
		 * showing the thread. /robo
		 */
		timerstack_lock (t);
		if (t->top > 0) {
		    mt = t->times + (t->top - 1);
		    if (allocs_follow_filter ())
			mp = mt->filtered_method;
		    else 
			mp = mt->method;
		    mp->allocated_objects++;
		    mp->allocated_memory += size;
		    mp->modified = 1;
		}
		timerstack_unlock (t);
	    }
	}

	jmphash_lock (objects, MONITOR_AGENT);
	/* we use the lock on objects to ensure that 
	 * objectstore_obj_new is thread safe 
	 */
	op = objectstore_obj_new (osp, arena_id, cp, is_array, size, 
				  obj_id, mp, current_reset_level, 
				  current_gc_level);
	jmphash_insert (op, objects);
	jmphash_unlock (objects, MONITOR_AGENT);
    } else {
	fprintf (stderr, "failed to find class(%p) for object_alloc (%p)"
		 "(probably java.lang.Class?)\n", class_id, obj_id);
    }
}


static void object_move (jint arena_id,
			 jobjectID obj_id, 
			 jint new_arena_id,
			 jobjectID new_obj_id) {
    /* This is executed in the thread suspended
       mode.  We should not call any malloc() etc
       in here.  The jmphash_insert()'s below are
       fine because the jmphash_remove()'s will free
       up an element in the same hash tables.
       BUT this has to be remembered iff the hash
       implementation ever changes! */
    obj* op = NULL;
    obj* hop = NULL;
    cls* cp = NULL;

    trace3 ("object_move (%p, %p)\n", obj_id, new_obj_id);
    this_gc_object_move++;
    c_object_move++;
    
    if (objects && object_profiling) {
	op = get_object (obj_id);
	if (op != NULL) {
	    hop = jmphash_remove (op, objects);
	    obj_set_arena_id (op, new_arena_id);
	    obj_set_object_id (op, new_obj_id);
	    jmphash_insert (op, objects);
	}
    }

    if (classes) {
	/* try to get class */
	cp = get_class (obj_id);
	if (cp) {
	    /* have to have a good hash table since new 
	     * classes will reuse the old object  id 
	     */
	    jmphash_remove (cp, classes);
	    cls_set_class_id (cp, new_obj_id);
	    jmphash_insert (cp, classes);
	}
    }    
}


static void object_free (jobjectID obj_id) {
    /* This is executed in the thread suspended
       mode.  We should not call any malloc() etc
       in here.
    */
    obj* op = NULL;
    obj* hop = NULL;
    cls* cp = NULL;
    
    trace2 ("object_free (%p)\n", obj_id);
    this_gc_object_free++;
    c_object_free++;
    if (!objects)
	return;
    
    op = get_object (obj_id);
    if (op != NULL) { /* we ignore objects that we know nothing about. */
	hop = jmphash_remove (op, objects);
	cp = get_class (obj_get_class_id (hop));
	if (cp != NULL) {
	    jint os = obj_get_size (hop);
	    jint ol = obj_get_gc_level (hop);
	    cls_object_free (cp, os, ol);
	}
	objectstore_obj_free (osp, hop);
    }
}


/* If this gets any more complicated with other JVMs we should setup a
 *  JVM quirks table and auto-detection of the JVM in which we're running
 *  it might also be helpful for support feedback to have a diagnostic tool
 *  to lets user dump interesting data about their JVM.
 *
 * Under Sun JVM 1.5 Update 5:
 *  System threads has group_name=="system" and parent_name==NULL
 *  First application thead has group_name=="main" and parent_name=="system"
 *
 * Return 1 if we consider this thread a system thread not a user thread.
 */
static int thread_is_system (const char *parent_name, const char *group_name) {
    if (parent_name == NULL && group_name != NULL && strcasecmp (group_name, "system") == 0)
        return 1;
    return 0;
}

/* When all of the non-daemonized application threads have temninated the JVM
 *  created a cleanup thread and executes it from the main threadgroup.
 * For all intents and purpose we consider this a system thread as the JMP user
 *  genreally does not care about it for profiling purposes.
 *
 * Under Sun JVM 1.5 Update 5:
 *  parent_name=system; group_name=main; thread_name=DestroyJavaVM
 */
static int thread_is_cleanup (const char *parent_name, const char *group_name, const char *thread_name) {
    if (parent_name != NULL && strcasecmp (parent_name, "system") == 0 &&
     group_name != NULL && strcasecmp (group_name, "main") == 0 &&
     thread_name != NULL && strcasecmp (thread_name, "DestroyJavaVM") == 0) {
        return 1;
    }
    return 0;
}

/*
 * The first application thread Under Sun JVM 1.5 Update 5:
 * parent_name=system; group_name=main; thread_name=main;
 */
static int thread_is_main (const char *parent_name, const char *group_name, const char *thread_name) {
    if (parent_name != NULL && strcasecmp (parent_name, "system") == 0 &&
     group_name != NULL && strcasecmp (group_name, "main") == 0 &&
     thread_name != NULL && strcasecmp (thread_name, "main") == 0) {
        return 1;
    }
    return 0;
}

/*
 * JMP threads Under Sun JVM 1.5 Update 5:
 * parent_name=<undef>; group_name=system; thread_name=jmp-gtk;	jmp-gtk
 * parent_name=system; group_name=main; thread_name=jmp-shutdown-holder; JMPIdleThread
 * TODO: Fix this so we're not in thread group "main"
 */
static int thread_is_jmp (const char *parent_name, const char *group_name, const char *thread_name) {
    if ((parent_name == NULL || strcmp(parent_name, "<unknown>") == 0) &&
     group_name != NULL && strcasecmp (group_name, "system") == 0 &&
     thread_name != NULL && strcasecmp (thread_name, JMP_THREAD_NAME_GTK) == 0) {
        return 1;
    }
    if ((parent_name == NULL || strcmp(parent_name, "<unknown>") == 0) &&
     group_name != NULL && strcasecmp (group_name, "system") == 0 &&
     thread_name != NULL && strcasecmp (thread_name, JMP_THREAD_NAME_NOUI) == 0) {
        return 1;
    }
    if (parent_name != NULL && strcasecmp (parent_name, "system") == 0 &&
     group_name != NULL /*&& strcasecmp (group_name, "main") == 0*/ &&
     thread_name != NULL && strcasecmp (thread_name, JMP_THREAD_NAME_IDLE) == 0) {
        return 1;
    }
    return 0;
}

/* DEBUGGING AID */
#if 0
static void print_thread (const char *label, const char *parent_name, const char *group_name, const char *thread_name) {
    fprintf(stdout, "%sparent_name=%s; group_name=%s; thread_name=%s\n", label ? label : "", parent_name, group_name, thread_name);
    fflush(stdout);
}
#endif

static void thread_start_nolock (char* thread_name, 
				 char* group_name,
				 char* parent_name,
				 jobjectID thread_id,
				 JNIEnv* thread_env_id, 
				 jint requested) {
    jmpthread* t;
    unsigned char mode;
    static int first = 1;	/* TODO: beter way, this does not allow for multiple windows */
    //print_thread ("START: ", parent_name, group_name, thread_name);
    if (thread_is_jmp (parent_name, group_name, thread_name)) {
        c_thread_start_jmp++;
        mode = THREAD_T_JMP;
    } else if (thread_is_system (parent_name, group_name) ||
	       thread_is_cleanup (parent_name, group_name, thread_name)) {
        c_thread_start_system++;
        mode = THREAD_T_SYSTEM;
    } else {
        c_thread_start++;
        mode = THREAD_T_APPLICATION;
        mode |= THREAD_F_INTERESTED;	/* TODO: Make a thread filter to configure this */
    }

    if (!down && threads) {
	t = get_jmpthread (thread_env_id);
	if (t == NULL) {
	    timerstack *s = setup_thread_local_storage (thread_env_id);
	    t = jmpthread_new (thread_name, group_name, parent_name, 
			     thread_id, thread_env_id, s, mode);
	    if (t == NULL) {
		fprintf (stderr, 
			 "thread_start: failed to allocate jmpthread:"
			 "%s, %s, %s, %p, %p\n",
			 thread_name, group_name, parent_name, 
			 thread_id, thread_env_id);
	    } else {
		jmphash_insert (t, threads);
	    }
	}
    }
    if((mode & THREAD_F_INTERESTED) != 0) {
        /* This event is significant, however in future we need to be a lot more
         *   flexible over what events might start our profiling runtime window.
         */
        if(first) {
            char buffer[80];
            snprintf (buffer, sizeof(buffer),  _("Thread \"%s\" started"), thread_name);
            set_status (buffer);
            notify_profile_window_begin_ui ();
            first = 0;
        }
    }
}


static void thread_start (char* thread_name, 
			  char* group_name,
			  char* parent_name,
			  jobjectID thread_id,
			  JNIEnv* thread_env_id, 
			  jint requested) {
    
    trace4 ("thread_start (%s,%s,%s,", thread_name, group_name, parent_name);
    trace3 ("%p,%p)\n", thread_id, thread_env_id);
    jmphash_lock (threads, MONITOR_AGENT);
    thread_start_nolock (thread_name, group_name, parent_name, thread_id,
			 thread_env_id, requested);
    jmphash_unlock (threads, MONITOR_AGENT);
}


static void thread_end (JNIEnv* env) {
    jmpthread* t;
    timerstack* s;
    int shutdown = 0;
    char thread_name[128];

    trace2 ("thread_end (%p)\n", env);
    if (!threads)
	return;

    thread_name[0] = '\0';
    if (!down) {
	jmphash_lock (threads, MONITOR_AGENT);
	t = get_jmpthread (env);
	if (!t) {
	    fprintf (stderr, "failed to find thread that ended: %p\n", env);
	    c_thread_end_unknown++;
	} else {
	    //print_thread ("ENDED: ", t->parent_name, t->group_name, t->thread_name);
            if (thread_is_jmp (t->parent_name, t->group_name, t->thread_name)) {
 	        c_thread_end_jmp++;
            } else if (thread_is_system (t->parent_name, t->group_name) ||
                     thread_is_cleanup (t->parent_name, t->group_name, t->thread_name)) {
		c_thread_end_system++;
            } else {
	        c_thread_end++;
            }
	    if (thread_is_main (t->parent_name, t->group_name, t->thread_name)) {
		/* This event is significant, however in future we need to be a lot more
		 *   flexible over what events might finish our profiling runtime window.
		 */
	    }
	    if (c_thread_start == c_thread_end)
	        shutdown = 1;

	    /* Do we really want to remove the dead threads? */
	    /* No we probably only want to mark them dead and move them
	     * to a dead storage, maybe free the timerstack though /robo 
	     */
	    jmphash_remove (t, threads);
 	    strncpy(thread_name, t->thread_name, sizeof(thread_name));
 	    thread_name[sizeof(thread_name)-1] = '\0';
	    jmpthread_free (t);
	}
	
	s = (timerstack*)jvmpi->GetThreadLocalStorage (env);
	jvmpi->SetThreadLocalStorage (env, NULL);
	timerstack_free (s);
	jmphash_unlock (threads, MONITOR_AGENT);
    }

    {
        char buffer[80];
        snprintf (buffer, sizeof(buffer),  _("Thread \"%s\" ended"), thread_name);
        set_status (buffer);
        notify_profile_window_end_ui ();	/* TODO: This should not be here */
    }
    if (shutdown) {	/* This can stop profiling. May be bogous. */
	notify_profile_window_close_ui ();
	
	if (do_gc_on_window_close) {
            fprintf (stdout, "Running GC on profile window close\n");
            fflush (stdout);
            run_GC ();
        }        
    }
}

/** only call this with methods locked. */
static method* get_unknown_method (jmethodID method_id) {
    /* ok, try really hard to get it... */
    /* MW: I don't think this will work! */
    /* This seems to work ok for me, it also seems to 
     * happen on NT/4 frequently (according to some bug
     * reports Ive got. To try this: simply disable 
     * class_load and class_unload events. /robo 
     */
    jobjectID class_id;
    cls* c;
    method* m;
    trace2 ("method_id %p is unknown, trying to find class...\n", 
	    method_id);
    jvmpi->DisableGC ();
    class_id = jvmpi->GetMethodClass (method_id);
    get_class_load (class_id);
    c = get_class (class_id);
    m = get_method (method_id);
    trace3 ("method_id %p => %p...\n", method_id, m);
    jvmpi->EnableGC ();
    /* Got this 20040201, after a bit of running with full tracing...
    tried to get unknown method: 0xca4b9a0 => class: 0x90099e8 
        (sun.reflect.GeneratedSerializationConstructorAccessor122) => (nil)
    */
    if (m == NULL) {
	fprintf (stderr, 
		 "tried to get unknown method: %p => class: %p (%s) => %p\n",
		 method_id, c, (c ? cls_get_name (c) : "?"), m);
    }
    return m;
}

/** A method is being called. */
static void method_entry (jmethodID method_id, JNIEnv *env) {
    timerstack* s;
    jlong tval;
    method *m;

    trace3 ("method_entry (%p, %p)\n", method_id, env);
    /* Since we don't move or delete the method
       struct's, it's safe to use the pointer in
       the timerstack */
    if (down)
	return;
    jmphash_lock (methods, MONITOR_AGENT);
    m = get_method (method_id);
    if (m == NULL)
	m = get_unknown_method (method_id);
    jmphash_unlock (methods, MONITOR_AGENT);
    c_method_entry++;

    if (m) {
	s = setup_thread_local_storage (env);
	tval = get_thread_time (jvmpi);    
#if DLMDEBUG
if (dlm_global_object_method_id_0 == method_id || dlm_global_object_method_id_1 == method_id || dlm_global_object_method_id_2 == method_id)
  fprintf(stderr, "XXXX ENTRY mid=%p tval=%lld\n", method_id, tval);
#endif
	jmpthread_method_entry (s, m, tval);
    }
    trace ("method_entry (...) done\n");
}

/** A method is being called. 
 *  Note that method_entry2 is called with GC disabled, this can make some things
 *  hard. Normally we do not care about this method since we do not need obj_id.
 */
static void method_entry2 (jmethodID method_id, jobjectID obj_id, JNIEnv *env) {
    timerstack* s;
    jlong tval;
    method *m;

    trace4 ("method_entry2 (%p, %p, %p)\n", method_id, obj_id, env);
    /* Since we don't move or delete the method
       struct's, it's safe to use the pointer in
       the timerstack */
    if (down)
	return;
    jmphash_lock (methods, MONITOR_AGENT);
    m = get_method (method_id);
    if (m == NULL)
	m = get_unknown_method (method_id);
    jmphash_unlock (methods, MONITOR_AGENT);
    c_method_entry++;
    
    if (m) {
	s = setup_thread_local_storage (env);
	tval = get_thread_time (jvmpi);    
#if DLMDEBUG
if (dlm_global_object_method_id_0 == method_id || dlm_global_object_method_id_1 == method_id || dlm_global_object_method_id_2 == method_id)
  fprintf(stderr, "XXXX ENTRY2 mid=%p tval=%lld\n", method_id, tval);
#endif
	jmpthread_method_entry (s, m, tval);
    }
    trace ("method_entry (...) done\n");
}

/** A method has exited. 
 */
static void method_exit (jmethodID method_id, JNIEnv* env) {
    timerstack* s;
    jlong tval;
    
    trace3 ("method_exit (%p, %p)\n", method_id, env);
    if (down)
	return;
    c_method_exit++;
    s = setup_thread_local_storage (env);
    tval = get_thread_time (jvmpi);
    
#if DLMDEBUG
if (dlm_global_object_method_id_0 == method_id || dlm_global_object_method_id_1 == method_id || dlm_global_object_method_id_2 == method_id)
  fprintf(stderr, "XXXX EXIT mid=%p tval=%lld\n", method_id, tval);
#endif

    jmpthread_method_exit (s, method_id, tval, env);
    trace ("method_exit (...) done\n");
}


static void heap_dump (int dump_level,
		       char* begin,
		       char* end,
		       jint num_traces,
		       JVMPI_CallTrace* traces) {
    trace4 ("heap dump dump_level: %d, begin: %p, end: %p, ", dump_level, begin, end);
    trace3 ("num_traces: %d, traces: %p\n", num_traces, traces);    
    switch (dump_level) {
    case JVMPI_DUMP_LEVEL_0:
	heap_dump_0 (dump_level, begin, end, num_traces, traces);
	break;
    case JVMPI_DUMP_LEVEL_1:
	heap_dump_1 (dumpdir, dump_level, begin, end, num_traces, traces);
	break;
    case JVMPI_DUMP_LEVEL_2:
	heap_dump_2 (dumpdir, dump_level, begin, end, num_traces, traces);
	break;
    }
}

typedef void (*contend_func)(timerstack*, obj*, jlong);

static void monitor_contend_handler (JNIEnv* env_id, jobjectID object, 
				     char* trace, contend_func func) {
    jlong tval;
    obj* o;
    timerstack* s;

    o = get_object (object);
    if (!o) {
	jvmpi->RequestEvent(JVMPI_EVENT_OBJECT_ALLOC, object);
	o = get_object (object);
    }
    trace4 ("%s %p, %s\n", trace, env_id, 
	    cls_get_name (obj_get_class (o)));
    s = setup_thread_local_storage (env_id);
    tval = get_absolute_time (jvmpi);
    func (s, o, tval);
}

static obj* get_object_hard (jobjectID obj_id) {
    obj* o = get_object (obj_id);
    if (o == NULL) {
	get_object_alloc (obj_id);
	o = get_object (obj_id);
    }
    return o;
}

static void monitor_contended_enter (JNIEnv* env_id, jobjectID object) {
    timerstack* s = setup_thread_local_storage (env_id);
    /* Since we don't want to update ts->waiting on object_move we use the obj* instead */
    s->waiting = get_object_hard (object);
    s->timeout = 0;
    monitor_contend_handler (env_id, object, "monitor contended enter", 
			     (contend_func)jmpthread_contenation_enter);
}

static void monitor_contended_entered (JNIEnv* env_id, jobjectID object) {
    timerstack* s = setup_thread_local_storage (env_id);
    s->waiting = NULL;
    s->timeout = 0;
    monitor_contend_handler (env_id, object, "monitor contended entered", 
			     (contend_func)jmpthread_contenation_entered);
}

static INLINE void monitor_contended_exit (JNIEnv* env_id, jobjectID object) {
    timerstack* ts;
    trace ("monitor_contended_exit\n");
    ts = setup_thread_local_storage (env_id);
    /*
    jmpthread* jt = get_jmpthread (env_id);
    obj* o = get_object (object);
    fprintf (stderr, "JVMPI_EVENT_MONITOR_CONTENDED_EXIT   : env: %p => %s, obj: %p => %s\n", 
	     env_id,jmpthread_get_thread_name (jt),  object, cls_get_name (obj_get_class (o)));
    */
    ts->waiting = NULL;
    ts->timeout = 0;
}

static void monitor_wait (JNIEnv* env_id, jobjectID object, jlong timeout) {
    timerstack* ts;
    trace ("monitor_wait\n");
    ts = setup_thread_local_storage (env_id);
    /*
    jmpthread* jt = get_jmpthread (env_id);
    obj* o = get_object (object);
    fprintf (stderr, "JVMPI_EVENT_MONITOR_WAIT  : env: %p => %s, obj: %p => %s, %lld\n", 
	     env_id, jmpthread_get_thread_name (jt), object, cls_get_name (obj_get_class (o)), timeout);
    */
    /* Since we don't want to update ts->waiting on object_move we use the obj* instead */
    ts->waiting = get_object_hard (object);
    ts->timeout = timeout;    
}

static void monitor_waited (JNIEnv* env_id, jobjectID object, jlong timeout) {
    timerstack* ts;
    trace ("monitor_waited\n");
    ts = setup_thread_local_storage (env_id);
    /*
    jmpthread* jt = get_jmpthread (env_id);
    obj* o = get_object (object);
    fprintf (stderr, "JVMPI_EVENT_MONITOR_WAITED: env: %p => %s, obj: %p => %s, %lld\n", 
	     env_id, jmpthread_get_thread_name (jt), object, cls_get_name (obj_get_class (o)), timeout);
    */
    ts->waiting = NULL;
    ts->timeout = 0;
}

void notifyEventSink (JVMPI_Event *e) {
    trace4 ("e->event_type: %d%s, e->env: %p: SUNK\n",
      (e->event_type & ~JVMPI_REQUESTED_EVENT),
      ((e->event_type & JVMPI_REQUESTED_EVENT) ? "|REQ" : ""), e->env_id);
#ifdef DLMDEBUG
fprintf(stderr, "e->event_type: %d%s, e->env: %p: SUNK\n",
      (e->event_type & ~JVMPI_REQUESTED_EVENT),
      ((e->event_type & JVMPI_REQUESTED_EVENT) ? "|REQ" : ""), e->env_id);
#endif

    switch (e->event_type) {
    case JVMPI_EVENT_JVM_INIT_DONE:
	jvm_init_done (e->env_id);
	break;
    case JVMPI_EVENT_JVM_SHUT_DOWN:
	jvm_shut_down (e->env_id);
	break;
    }

    return;
}

#define THREAD_FILTER(evt) do {					\
        int mode = jmpthread_get_mode_by_env_id((evt)->env_id);	\
        if((mode & THREAD_F_INTERESTED) == 0)			\
            return;	/* filtered out */			\
} while(0)

/** Switch on the event_type and call the right method. 
 */
void notifyEvent (JVMPI_Event *e) {
    static int ratelimit = 3;

    if (sink_events) {
       notifyEventSink (e);
       return;
    }

    trace4 ("e->event_type: %d%s, e->env: %p: ",
      (e->event_type & ~JVMPI_REQUESTED_EVENT),
      ((e->event_type & JVMPI_REQUESTED_EVENT) ? "|REQ" : ""), e->env_id);

    /** Events that are tested with the jvmsimulator are marked with "tested" 
     *  That mark means that valgrind does not find any errors.
     */
    switch(e->event_type) {
    case JVMPI_EVENT_JVM_INIT_DONE:  /* tested */
	jvm_init_done (e->env_id);
	break;
    case JVMPI_EVENT_JVM_SHUT_DOWN:  /* tested */
	jvm_shut_down (e->env_id);
	break;
    case JVMPI_EVENT_GC_START:       /* tested */
	gc_start ();
	break;
    case JVMPI_EVENT_GC_FINISH:      /* tested */
	gc_finish (e->u.gc_info.used_objects, 
		   e->u.gc_info.used_object_space,
		   e->u.gc_info.total_object_space);		
	break;
    case JVMPI_EVENT_ARENA_NEW:      /* tested */
	jvmpi_arena_new (e->u.new_arena.arena_id, 
			 e->u.new_arena.arena_name);
	break;
    case JVMPI_EVENT_ARENA_DELETE:   /* tested */
	arena_delete (e->u.delete_arena.arena_id);
	break;
    case JVMPI_EVENT_CLASS_LOAD:     /* tested */
        THREAD_FILTER(e);
    case JVMPI_EVENT_CLASS_LOAD | JVMPI_REQUESTED_EVENT:
	class_load (e->env_id,
		    e->u.class_load.class_name, e->u.class_load.source_name,
		    e->u.class_load.num_interfaces, e->u.class_load.num_methods,
		    e->u.class_load.methods, e->u.class_load.num_static_fields,
		    e->u.class_load.statics, e->u.class_load.num_instance_fields, 
		    e->u.class_load.instances, e->u.class_load.class_id, 
		    e->event_type & JVMPI_REQUESTED_EVENT);
	break;
    case JVMPI_EVENT_CLASS_UNLOAD:   /* tested */
        THREAD_FILTER(e);
	class_unload (e->u.class_unload.class_id);
	break;	
    case JVMPI_EVENT_OBJECT_ALLOC:   /* tested */
        THREAD_FILTER(e);
    case JVMPI_EVENT_OBJECT_ALLOC | JVMPI_REQUESTED_EVENT:
	object_alloc (e->u.obj_alloc.arena_id,
		      e->u.obj_alloc.class_id,
		      e->u.obj_alloc.is_array,
		      e->u.obj_alloc.size,
		      e->u.obj_alloc.obj_id,
		      e->event_type & JVMPI_REQUESTED_EVENT,
		      e->env_id);
	break;
    case JVMPI_EVENT_OBJECT_MOVE:    /* tested */
        //THREAD_FILTER(e);
	object_move (e->u.obj_move.arena_id, e->u.obj_move.obj_id, 
		     e->u.obj_move.new_arena_id, e->u.obj_move.new_obj_id);
	break;
    case JVMPI_EVENT_OBJECT_FREE:    /* tested */
        //THREAD_FILTER(e);
	object_free (e->u.obj_free.obj_id);	
	break;
    case JVMPI_EVENT_THREAD_START:   /* tested */
    case JVMPI_EVENT_THREAD_START | JVMPI_REQUESTED_EVENT:
	thread_start (e->u.thread_start.thread_name, 
		      e->u.thread_start.group_name, 
		      e->u.thread_start.parent_name,
		      e->u.thread_start.thread_id, 
		      e->u.thread_start.thread_env_id,
		      e->event_type & JVMPI_REQUESTED_EVENT);
	break;
    case JVMPI_EVENT_THREAD_END:     /* tested */
	thread_end (e->env_id);
	break;
    case JVMPI_EVENT_METHOD_ENTRY:   /* tested */
        THREAD_FILTER(e);
	method_entry (e->u.method.method_id, e->env_id);
	break;
    case JVMPI_EVENT_METHOD_ENTRY2:
        THREAD_FILTER(e);
	method_entry2 (e->u.method_entry2.method_id, e->u.method_entry2.obj_id, e->env_id);
	break;
    case JVMPI_EVENT_METHOD_EXIT:    /* tested */
        THREAD_FILTER(e);
	method_exit (e->u.method.method_id, e->env_id);
	break;
    case JVMPI_EVENT_DATA_DUMP_REQUEST:   /* tested */
	run_data_dump ();
	break;
    case JVMPI_EVENT_DATA_RESET_REQUEST:  /* tested */
	data_reset ();
	break;	
    case JVMPI_EVENT_HEAP_DUMP:
    case JVMPI_EVENT_HEAP_DUMP | JVMPI_REQUESTED_EVENT:
	heap_dump (e->u.heap_dump.dump_level,
		   e->u.heap_dump.begin,
		   e->u.heap_dump.end,
		   e->u.heap_dump.num_traces,
		   e->u.heap_dump.traces);
	break;
    case JVMPI_EVENT_OBJECT_DUMP:
    case JVMPI_EVENT_OBJECT_DUMP | JVMPI_REQUESTED_EVENT:
	object_dump (e->u.object_dump.data_len,
		     e->u.object_dump.data);
	break;
    case JVMPI_EVENT_MONITOR_CONTENDED_ENTER:   /* tested */
	monitor_contended_enter (e->env_id, e->u.monitor.object);
	break;
    case JVMPI_EVENT_MONITOR_CONTENDED_ENTERED: /* tested */
	monitor_contended_entered (e->env_id, e->u.monitor.object);
	break;
    case JVMPI_EVENT_MONITOR_CONTENDED_EXIT:    /* tested */
	monitor_contended_exit (e->env_id, e->u.monitor.object);
	break;
    case JVMPI_EVENT_MONITOR_DUMP:
    case JVMPI_EVENT_MONITOR_DUMP | JVMPI_REQUESTED_EVENT:
	last_monitor_dump = monitor_dump (e->u.monitor_dump.begin,
					  e->u.monitor_dump.end,
					  e->u.monitor_dump.num_traces,
					  e->u.monitor_dump.traces,
					  e->u.monitor_dump.threads_status);
	break;
    case JVMPI_EVENT_MONITOR_WAIT:    /* tested */
	monitor_wait (e->env_id, e->u.monitor_wait.object, e->u.monitor_wait.timeout);
	break;
    case JVMPI_EVENT_MONITOR_WAITED:  /* tested */
	monitor_waited (e->env_id, e->u.monitor_wait.object, e->u.monitor_wait.timeout);
	break;

    case JVMPI_EVENT_RAW_MONITOR_CONTENDED_ENTER:
	/*
	if (e->u.raw_monitor.id != jmphash_get_monitor (methods)) 
	    fprintf (stderr, "raw_monitor_contended_enter  :%p, %p => %s\n", 
		     e->env_id, e->u.raw_monitor.id, e->u.raw_monitor.name);
	*/
	break;
    case JVMPI_EVENT_RAW_MONITOR_CONTENDED_ENTERED:
	/*
	if (e->u.raw_monitor.id != jmphash_get_monitor (methods)) 
	    fprintf (stderr, "raw_monitor_contended_entered :d%p, %p => %s\n", 
		     e->env_id, e->u.raw_monitor.id, e->u.raw_monitor.name);
	*/
	break;
    case JVMPI_EVENT_RAW_MONITOR_CONTENDED_EXIT:
	/*
	if (e->u.raw_monitor.id != jmphash_get_monitor (methods)) 
	    fprintf (stderr, "raw_monitor_contended_exit   :%p, %p => %s\n", 
		     e->env_id, e->u.raw_monitor.id, e->u.raw_monitor.name);
	*/
	break;
    default:
        while (ratelimit > 0) {
           ratelimit--;
           fprintf (stderr, "unhandled event=%p please report.\n", (void *)e->event_type);
        }
	trace ("unhandled.\n");
    }
}

static void display_help () {
    fprintf (stdout, "java -Xrunjmp[:[options]] package.Class\n");
    fprintf (stdout, "options is a comma separated list and may include:\n");
    fprintf (stdout, "help      - to show this text.\n");
    fprintf (stdout, "nomethods - to disable method profiling.\n");
    fprintf (stdout, "noobjects - to disable object profiling.\n");
    fprintf (stdout, "nomonitors - to disable monitor profiling.\n");
    fprintf (stdout, "allocfollowsfilter - to group object allocations into filtered methods.\n");    
#ifndef CONFIG_UI_NONE
    fprintf (stdout, "nogui     - to run jmp without the user interface.\n");
#endif
    fprintf (stdout, "dodump    - to allow to be called with signals.\n");
    fprintf (stdout, "dumpdir=<directoryr> - to specify where the dump-/heapdumpfiles go.\n");
    fprintf (stdout, "dumptimer=<n> - to specify automatic dump every n:th second.\n");
    fprintf (stdout, "filter=<somefilter> - to specify an initial recursive filter.\n");
    fprintf (stdout, "threadtime - to specify that timing of methods and monitors \n"
	     "           should use thread cpu time instead of absolute time.\n"); 
    fprintf (stdout, "simulator - to specify that jmp should not perform any jni tricks.\n"
	     "           probably only useful if you debug jmp.\n");
    fprintf (stdout, "nogconexit - disable run GC during shutdown.\n");
    fprintf (stdout, "gconexit   - run GC during shutdown.\n");
#ifndef CONFIG_UI_NONE
    fprintf (stdout, "nojvmhold  - don't keep JVM alive after java class main() exit.\n");
    fprintf (stdout, "jvmhold    - keep JVM alive after java class main() exit.\n");
#endif
    fprintf (stdout, "\nAn example may look like this:\n"); 
    fprintf (stdout, "java -Xrunjmp:nomethods,dumpdir=/tmp/jmpdump/ rabbit.proxy.Proxy\n"); 
}

static size_t get_size_of_option (char* option) {
    size_t t;
    char* n;
    n = strstr (option, ",");
    if (n != NULL) 
	t = n - option;
    else 
	t = strlen (option);
    return t;
}

static char* setup_filter (char* m) {
    char* filter;
    char* typeEnd;
    char* mem;
    static char *filterTypeStr[] = {"class", "package", 
				    "recursive", "all"};
    int filterMode = FILTER_INCLUDE;
    int filterType = FILTER_MATCH_RECURSIVE;

    size_t t = get_size_of_option (m + 7);
    mem = malloc (t + 1);
    filter = mem;
    strncpy (filter, m + 7, t);
    filter[t] = '\0';

    typeEnd = strchr (filter, ':');
    if (typeEnd != NULL && typeEnd <= filter + t) {
	int i;
	char* typestr = filter;

	*typeEnd = '\0';
	filter = typeEnd + 1;
	if (typestr[0] == '+') {
	    filterMode = FILTER_INCLUDE;
	    typestr++;
	} else if (typestr[0] == '-') {
	    filterMode = FILTER_EXCLUDE;
	    typestr++;
	}
	filterType = -1;
	for (i=0; i < sizeof (filterTypeStr) / sizeof (char*); i++) {
	    if (strcmp (typestr, filterTypeStr[i]) == 0) {
		filterType = i;
	    }
	}

	if (filterType == -1) {
	    fprintf (stdout, 
		     "strange filter type (%s) value specified, ignored\n",
		     typestr);
	}
    }

    if (filterType != -1) {
	fprintf (stdout, 
		 "    adding %s filter for %s with matching mode %s\n", 
		 (filterMode == FILTER_INCLUDE ? 
		  "inclusive" : "exclusive"),
		 filter, filterTypeStr[filterType]);                              
	filter_add_filter (filterMode, filter, filterType);
    }
    free (mem);
    return m + 7 + t + 1;
}

static void parse_options (char* options) {
    char* m = NULL;
    char* co;

    /** Set some defaults */
#ifndef CONFIG_UI_NONE
    doUI = 1;
#endif

    /** Parse the options we got. */
    if (options != NULL) {
	m = strstr (options, "help");
	if (m != NULL) {
	    fprintf (stdout, "help wanted..\n");
	    display_help ();
	    exit (0);
	}

	m = strstr (options, "nomethods");
	method_profiling = (m == NULL);
	
	m = strstr (options, "noobjects");
	object_profiling = (m == NULL);
	
	m = strstr (options, "nomonitors");
	monitor_profiling = (m == NULL);

#ifndef CONFIG_UI_NONE
	m = strstr (options, "nogui");
	doUI = (m == NULL);
#endif
	
	m = strstr (options, "dodump");
	dump_enabled = (m != NULL);

	m = strstr (options, "simulator");
	simulator = (m != NULL);
	
	m = strstr (options, "dumpdir=");
	if (m != NULL) {
	    size_t t = get_size_of_option (m + 8);	    
	    dumpdir = malloc (t + 1);
	    strncpy (dumpdir, m + 8, t);
	    dumpdir[t] = '\0';
	}
	
	m = strstr (options, "dumptimer=");
	if (m != NULL) {
	    char *e;
	    size_t t = get_size_of_option (m + 10); 
	    long l = strtol (m + 10, &e, 0);
	    if (e != m + 10 + t) {
		fprintf (stdout, "strange dumptimer (%ld) value specified, ignored: %p, %p...\n", 
			 l, (m + 10), (m + 10 + t));
	    } else {
		 set_dump_timer (l);
	    }		
	}

	m = strstr (options, "allocfollowsfilter");
	if (m != NULL) {
	    alloc_follow_filter = 1;
	}
	
	co = options;
	while (co) {
	    m = strstr (co, "filter=");
	    if (m != NULL) {
		co = setup_filter (m);
	    } else {
		co = NULL;
	    }
	}
	
	absolute_times = strstr (options, "threadtime") == NULL;

	m = strstr (options, "nogconexit");
	if (m != NULL)
            do_gc_on_shutdown = (m == NULL);

	m = strstr (options, "gconexit");
	if (m != NULL)
            do_gc_on_shutdown = (m != NULL);

#ifndef CONFIG_UI_NONE
	m = strstr (options, "nojvmhold");
	if (m != NULL) {
            exit_on_jvm_shutdown = 1;
	} else {
	    m = strstr (options, "jvmhold");
	    if (m != NULL)
		exit_on_jvm_shutdown = 0;
	}
#endif
    } else {
	/* to be consistent with above... */
	absolute_times = 1;
    }
}

static void enable_gc_events () {
    jvmpi->EnableEvent (JVMPI_EVENT_GC_START, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_GC_FINISH, NULL);
}

static void enable_arena_events () {
    jvmpi->EnableEvent (JVMPI_EVENT_ARENA_NEW, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_ARENA_DELETE, NULL);
}

static void enable_class_events () {
    class_profiling = 1;
    jvmpi->EnableEvent (JVMPI_EVENT_CLASS_LOAD, NULL); 
    jvmpi->EnableEvent (JVMPI_EVENT_CLASS_UNLOAD, NULL);
    /* classes are moved so we need this here.. */
    jvmpi->EnableEvent (JVMPI_EVENT_OBJECT_MOVE, NULL);
}

static void ensure_class_profiling () {
    if (class_profiling == 0)
	enable_class_events ();
}

void enable_object_events () {
    ensure_class_profiling ();
    object_profiling = 1;
    /* Enable object tracing events. */
    jvmpi->EnableEvent (JVMPI_EVENT_OBJECT_ALLOC, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_OBJECT_FREE, NULL);
}

static void enable_thread_events () {
    jvmpi->EnableEvent (JVMPI_EVENT_THREAD_START, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_THREAD_END, NULL);
}

void get_call_trace (jmpthread* t) {
    JNIEnv* tid = jmpthread_get_env_id (t);
    get_call_trace_env (tid);
}

void get_call_trace_env (JNIEnv* tid) {
    int i;
    timerstack* ts;
    JVMPI_CallTrace ct;
    JVMPI_CallFrame* cf;
    jint depth = 10;
    cf = malloc (sizeof (*cf) * depth);
    /* can only get calltrace for suspended threads.. */

    /* Currently there seems to be a bug(?) in the SUN jdk, it will
     * always return 0 call frames for threads that are in sleep ()
     * This is not a big problem, will give us one stack underflow 
     * later on which will make us get the stack from that thread
     * and that works.
     * reported and got an internal review ID of: 179429, 
     * probably not visible yet /robo
     * External id should now be: 4860655.
     */
    jvmpi->SuspendThread (tid);
    ts = get_timerstack (tid);
    if (ts) {
	jlong tval;
	timerstack_lock (ts);
	tval = get_thread_time (jvmpi);
	ct.frames = cf;
	ct.env_id = tid;
	ts->top = 0;
	ts->last_contentation = -1;
	jvmpi->GetCallTrace (&ct, depth);
	trace3 ("got: %d methods for %p...\n", ct.num_frames, tid); 
	for (i = ct.num_frames - 1; i >= 0; i--) {
	    method* m;
	    JVMPI_CallFrame* f = ct.frames + i;
	    m = get_method (f->method_id);
	    if (m == NULL) 
		m = get_unknown_method (f->method_id);
	    if (m) {
		/** We should always have m since we always trace classes.
		 *  In case we change our mind in the future /robo
		 */
		jmpthread_method_entry (ts, m, tval);
	    }
	}
	timerstack_unlock (ts);
    }
    jvmpi->ResumeThread (tid);
    free (cf);
}

static void enable_method_events_and_stacks (int get_stacks) {
    ensure_class_profiling ();

    /* Ok, the thread stacks are bogous, clear them all and try to get 
     * real stacks, timing will be a bit bogous, but not to much...
     */
    
    if (get_stacks) {
	int locks;
	jmphash_lock (threads, MONITOR_AGENT);
	jmphash_lock (methods, MONITOR_AGENT);
	locks = timerstacks_get_need_locks ();
	timerstacks_set_need_locks (1);

	/*
	  WARNING! due to a bug inside SUN's jvm
	  method profiling may cause the jvm to crash see:
	  http://developer.java.sun.com/developer/bugParade/bugs/4652208.html
	  for more info
	*/

	/* Enable method profiling events under lock, so we 
	 * don't get odd method entries. 
	 */
	jvmpi->EnableEvent (JVMPI_EVENT_METHOD_ENTRY, NULL);
	jvmpi->EnableEvent (JVMPI_EVENT_METHOD_EXIT, NULL);
	
	jvmpi->DisableGC ();
	jmphash_for_each ((jmphash_iter_f)get_call_trace, threads);
	jvmpi->EnableGC ();
	timerstacks_set_need_locks (locks);
	jmphash_unlock (methods, MONITOR_AGENT);
	jmphash_unlock (threads, MONITOR_AGENT);
    } else {
	jvmpi->EnableEvent (JVMPI_EVENT_METHOD_ENTRY, NULL);
	jvmpi->EnableEvent (JVMPI_EVENT_METHOD_EXIT, NULL);
    }
    method_profiling = 1;
}

void enable_method_events () {
    enable_method_events_and_stacks (1);
}

static void enable_dump_events () {
    dump_enabled = 1;
    jvmpi->EnableEvent (JVMPI_EVENT_DATA_DUMP_REQUEST, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_DATA_RESET_REQUEST, NULL);
}

void enable_monitor_events () {
    jvmpi->EnableEvent (JVMPI_EVENT_MONITOR_CONTENDED_ENTER, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_MONITOR_CONTENDED_ENTERED, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_MONITOR_CONTENDED_EXIT , NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_MONITOR_WAIT, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_MONITOR_WAITED , NULL);
    
    jvmpi->EnableEvent (JVMPI_EVENT_RAW_MONITOR_CONTENDED_ENTER, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_RAW_MONITOR_CONTENDED_ENTERED, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_RAW_MONITOR_CONTENDED_EXIT, NULL);
}

static void enable_events () {
    /* Enable standard events */
    jvmpi->EnableEvent (JVMPI_EVENT_JVM_INIT_DONE, NULL);
    jvmpi->EnableEvent (JVMPI_EVENT_JVM_SHUT_DOWN, NULL);    
    enable_gc_events ();
    enable_arena_events ();
    /* we dont have to enable class events yet, object 
     * and/or method profiling will do that for us.
     * enable_class_events ();
     */
    if (object_profiling)
	enable_object_events ();
    enable_thread_events ();
    if (method_profiling)     
	enable_method_events_and_stacks (0);
    if (dump_enabled)
	enable_dump_events ();
    if (monitor_profiling)
	enable_monitor_events ();
}

#ifdef JVMPI_VERSION_1_1
jobject jvmpi_jobjectID2jobject (jobjectID jid) {
	return (*jvmpi->jobjectID2jobject)(jid);
}

jobjectID jvmpi_jobject2jobjectID (jobject jobj) {
	return (*jvmpi->jobject2jobjectID)(jobj);
}
#endif

#ifdef JVMPI_VERSION_1_2
void jvmpi_SuspendThreadList (jint reqCount, JNIEnv **reqList, jint *results) {
	(*jvmpi->SuspendThreadList)(reqCount, reqList, results);
}

void jvmpi_ResumeThreadList (jint reqCount, JNIEnv **reqList, jint *results) {
	(*jvmpi->ResumeThreadList)(reqCount, reqList, results);
}
#endif

JNIEnv * JavaVM_AttachCurrentThread () {
    JNIEnv *env;
    JNIEnv **envp;
    jint res;
    envp = &env;
    res = (*JavaVM_jvm)->AttachCurrentThread(JavaVM_jvm, (void **)envp, NULL);
    if (res < 0)
        return NULL;
    return env;
}

int JavaVM_DetachCurrentThread () {
    jint res = (*JavaVM_jvm)->DetachCurrentThread(JavaVM_jvm);
    return res;
}

/** This method is the one that is called by the jvm on startup. 
 */
JNIEXPORT jint JNICALL JVM_OnLoad (JavaVM *jvm, char *options, void *reserved) {
#ifndef PACKAGE_VERSION
#define PACKAGE_VERSION "unknown"
#endif
    JNIEnv *jni_this;

    JavaVM_jvm = jvm;

    fprintf (stdout, "jmp/%s initializing: (%s):... build flags:", PACKAGE_VERSION, (options ? options : ""));
#ifdef CONFIG_UI_NONE
    fprintf (stdout, " noui");
#endif
#ifdef CONFIG_UI_GTK
    fprintf (stdout, " gtk");
#endif
    fprintf (stdout, "\n");
    parse_options (options);
    fprintf (stdout, "    tracing objects: %s\n", (object_profiling ? "true" : "false"));
    fprintf (stdout, "    tracing methods: %s\n", (method_profiling ? "true" : "false"));
    fprintf (stdout, "    tracing monitors: %s\n", (monitor_profiling ? "true" : "false"));
#ifndef CONFIG_UI_NONE
        fprintf (stdout, "    showing gui: %s\n", (doUI ? "true" : "false"));
#endif
    fprintf (stdout, "    dump/reset by signal allowed: %s\n", (dump_enabled ? "true" : "false"));
    fflush (stdout);

    /* get jni interface pointer */
    if ((jni_this = jvm_detect_jni_version (jvm)) == NULL) {
	fprintf (stderr, "jmp: error in obtaining JNI interface pointer\n");
	return JNI_ERR;
    }

    /* get jvmpi interface pointer */
    if (jvm_detect_jvmpi_version (jvm) == 0) {
	fprintf (stderr, "jmp: error in obtaining JVMPI interface pointer\n");
	return JNI_ERR;
    }


    trace ("Going to evaluate vm version\n");
    if (jni_this == NULL) {
	fprintf (stderr, "Warning: jvm_init_done with JNIEnv == NULL, assuming simulator\n");
	simulator = 1;
    }

    if (!simulator) {
        jclass c_system;
        jmethodID m_get_property;

	c_system = (*jni_this)->FindClass (jni_this, "java/lang/System");
	m_get_property =
	    (*jni_this)->GetStaticMethodID (jni_this, c_system, "getProperty",
				       "(Ljava/lang/String;)Ljava/lang/String;");
	
	get_string (&vm_version, "java.vm.version", jni, c_system, m_get_property);
	get_string (&vm_vendor, "java.vm.vendor", jni, c_system, m_get_property);
	get_string (&vm_name, "java.vm.name", jni, c_system, m_get_property);
    } else {
	vm_version = strdup ("jmp");
	vm_vendor = strdup ("jmp");
	vm_name = strdup ("jmp");
    }

    if (verbose) {
      fprintf (stdout, "JVM Vendor    : %s (%s) %s\n", vm_vendor, vm_version, vm_name);
      if (jni_version_using != jni_version_supported) {
        fprintf (stdout, "JNI Version   : JVM supports %s, JMP supports %s, using %s\n",
             jvm_jni_version_string (jni_version_using),
             jvm_jni_version_string (jni_version_supported),
             jvm_jni_version_string (jni_version_using));
      } else {
        fprintf (stdout, "JNI Version   : %s\n",
             jvm_jni_version_string (jni_version_using));
      }
      if (jvmpi_version_using != jvmpi_version_supported) {
        fprintf (stdout, "JVMPI Version: JVM supports %s, JMP supports %s, using %s\n",
             jvm_jvmpi_version_string (jvmpi_version_using),
             jvm_jvmpi_version_string (jvmpi_version_supported),
             jvm_jvmpi_version_string (jvmpi_version_using));
      } else {
        fprintf (stdout, "JVMPI Version : %s\n",
             jvm_jvmpi_version_string (jvmpi_version_using));
      }
      fflush (stdout);
    }
    
    jmptime_init (absolute_times, vm_version, vm_vendor, vm_name);

    jvmpi->NotifyEvent = notifyEvent;

    if (platform_init())
        return JNI_ERR;

    if (setup ())
	return JNI_ERR;

#ifdef LC_ALL
    fprintf (stdout, "jmp: Enabling localization.\n");
    fflush (stdout);
    /* Enable internationalization */
    setlocale (LC_ALL, "");
    bindtextdomain (PACKAGE_NAME, LOCALEDIR);
    bind_textdomain_codeset (PACKAGE_NAME, "UTF-8");
    textdomain (PACKAGE_NAME);
#endif
    fprintf (stdout, _("jmp: Loaded and registered correctly.\n"));
    fflush (stdout);

    /* initialize jvmpi interface */
    enable_events ();

    /* initialize the ui - called from here and not jvm_init_done() due to
     *  necessary event queue initialization.
     */
    init_ui ();

    return JNI_OK;    
}

    

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
