/*
A small wrapper utility to load indicators and put them as menu items
into the gnome-panel using it's applet interface.

Copyright 2009 Canonical Ltd.

Authors:
    Ted Gould <ted@canonical.com>

This program is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License version 3, as published 
by the Free Software Foundation.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranties of 
MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along 
with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gio/gio.h>

#include "gen-me-service.xml.h"
#include "dbus-shared-names.h"
#include "me-service-dbus.h"

static void status_service_dbus_class_init (StatusServiceDbusClass *klass);
static void status_service_dbus_init       (StatusServiceDbus *self);
static void status_service_dbus_dispose    (GObject *object);
static void status_service_dbus_finalize   (GObject *object);
static GVariant * get_icon (StatusServiceDbus * service);
static GVariant * get_username (StatusServiceDbus * service);
static GVariant * status_service_dbus_watch (StatusServiceDbus * service);
static void bus_get_cb (GObject * object, GAsyncResult * res, gpointer user_data);
static void bus_method_call (GDBusConnection * connection, const gchar * sender, const gchar * path, const gchar * interface, const gchar * method, GVariant * params, GDBusMethodInvocation * invocation, gpointer user_data);


	
/**
	StatusServiceDbusPrivate:
	@dbus_registration: The handle for this object being registered
		on dbus.

	Structure to define the memory for the private area
	of the status service dbus instance.
*/
typedef struct _StatusServiceDbusPrivate StatusServiceDbusPrivate;
struct _StatusServiceDbusPrivate
{
	GDBusConnection * bus;
	GCancellable * bus_cancel;
	guint dbus_registration;
	gchar * name;
	gchar * icon;
};

#define STATUS_SERVICE_DBUS_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), STATUS_SERVICE_DBUS_TYPE, StatusServiceDbusPrivate))

/* GDBus Stuff */
static GDBusNodeInfo *      node_info = NULL;
static GDBusInterfaceInfo * interface_info = NULL;
static GDBusInterfaceVTable interface_table = {
       method_call:    bus_method_call,
       get_property:   NULL, /* No properties */
       set_property:   NULL  /* No properties */
};

/* GObject Boilerplate */
G_DEFINE_TYPE (StatusServiceDbus, status_service_dbus, G_TYPE_OBJECT);

static void
status_service_dbus_class_init (StatusServiceDbusClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (object_class, sizeof(StatusServiceDbusPrivate));

	object_class->dispose = status_service_dbus_dispose;
	object_class->finalize = status_service_dbus_finalize;

	/* Setting up the DBus interfaces */
	if (node_info == NULL) {
		GError * error = NULL;

		node_info = g_dbus_node_info_new_for_xml(_me_service, &error);
		if (error != NULL) {
			g_error("Unable to parse Me Service Interface description: %s", error->message);
			g_error_free(error);
		}
	}

	if (interface_info == NULL) {
		interface_info = g_dbus_node_info_lookup_interface(node_info, INDICATOR_ME_SERVICE_DBUS_INTERFACE);

		if (interface_info == NULL) {
			g_error("Unable to find interface '" INDICATOR_ME_SERVICE_DBUS_INTERFACE "'");
		}
	}

	return;
}

static void
status_service_dbus_init (StatusServiceDbus *self)
{
	StatusServiceDbusPrivate * priv = STATUS_SERVICE_DBUS_GET_PRIVATE(self);
	priv->name = NULL;
	priv->icon = NULL;

	priv->bus = NULL;
	priv->bus_cancel = NULL;
	priv->dbus_registration = 0;

	priv->bus_cancel = g_cancellable_new();
	g_bus_get(G_BUS_TYPE_SESSION,
		priv->bus_cancel,
		bus_get_cb,
		self);

	return;
}

static void
bus_get_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;
	GDBusConnection * connection = g_bus_get_finish(res, &error);

	if (error != NULL) {
		g_error("OMG! Unable to get a connection to DBus: %s", error->message);
		g_error_free(error);
		return;
	}

	StatusServiceDbusPrivate * priv = STATUS_SERVICE_DBUS_GET_PRIVATE(user_data);

	g_warn_if_fail(priv->bus == NULL);
	priv->bus = connection;

	if (priv->bus_cancel != NULL) {
		g_object_unref(priv->bus_cancel);
		priv->bus_cancel = NULL;
	}

	/* Now register our object on our new connection */
	priv->dbus_registration = g_dbus_connection_register_object(connection,
                                                                   INDICATOR_ME_SERVICE_DBUS_OBJECT,
                                                                   interface_info,
                                                                   &interface_table,
                                                                   user_data,
                                                                   NULL,
                                                                   &error);

	if (error != NULL) {
		g_error("Unable to register the object to DBus: %s", error->message);
		g_error_free(error);
		return;
	}

	return; 
}

/* A method has been called from our dbus inteface.  Figure out what it
   is and dispatch it. */
static void
bus_method_call (GDBusConnection * connection, const gchar * sender,
                 const gchar * path, const gchar * interface,
                 const gchar * method, GVariant * params,
                 GDBusMethodInvocation * invocation, gpointer user_data)
{
	StatusServiceDbus * service = STATUS_SERVICE_DBUS(user_data);
	GVariant * retval = NULL;

	if (g_strcmp0(method, "PrettyUserName") == 0) {
		retval = get_username(service);
	} else if (g_strcmp0(method, "StatusIcons") == 0) {
		retval = get_icon(service);
	} else if (g_strcmp0(method, "Watch") == 0) {
		retval = status_service_dbus_watch(service);
	} else {
		g_warning("Calling method '%s' on the indicator service and it's unknown", method);
	}

	g_dbus_method_invocation_return_value(invocation, retval);
	return;
}


static void
status_service_dbus_dispose (GObject *object)
{
	StatusServiceDbusPrivate * priv = STATUS_SERVICE_DBUS_GET_PRIVATE(object);

	if (priv->dbus_registration != 0) {
		g_dbus_connection_unregister_object(priv->bus, priv->dbus_registration);
		/* Don't care if it fails, there's nothing we can do */
		priv->dbus_registration = 0;
	}

	if (priv->bus != NULL) {
		g_object_unref(priv->bus);
		priv->bus = NULL;
	}

	if (priv->bus_cancel != NULL) {
		g_cancellable_cancel(priv->bus_cancel);
		g_object_unref(priv->bus_cancel);
		priv->bus_cancel = NULL;
	}

	G_OBJECT_CLASS (status_service_dbus_parent_class)->dispose (object);
	return;
}

static void
status_service_dbus_finalize (GObject *object)
{

	G_OBJECT_CLASS (status_service_dbus_parent_class)->finalize (object);
	return;
}

static GVariant *
status_service_dbus_watch (StatusServiceDbus * service)
{
	return g_variant_new("(b)", TRUE);
}


static GVariant *
get_username (StatusServiceDbus * service)
{
	StatusServiceDbusPrivate * priv = STATUS_SERVICE_DBUS_GET_PRIVATE(service);
        return g_variant_new("(s)", priv->name);
}

static GVariant *
get_icon (StatusServiceDbus * service)
{
	StatusServiceDbusPrivate * priv = STATUS_SERVICE_DBUS_GET_PRIVATE(service);
        return g_variant_new("(s)", priv->icon);
}

void
status_service_dbus_set_status (StatusServiceDbus * self, const gchar * icon)
{
	g_return_if_fail(IS_STATUS_SERVICE_DBUS(self));

	g_debug("Setting icon to: %s", icon);

	StatusServiceDbusPrivate * priv = STATUS_SERVICE_DBUS_GET_PRIVATE(self);

	GError * error = NULL;

	if (priv->icon != NULL) {
		g_free(priv->icon);
	}
	priv->icon = g_strdup(icon);

	if (priv->bus == NULL) {
		return;
	}

	g_dbus_connection_emit_signal (priv->bus,
                                       NULL,
                                       INDICATOR_ME_SERVICE_DBUS_OBJECT,
                                       INDICATOR_ME_SERVICE_DBUS_INTERFACE,
                                       "StatusIconsChanged",
                                       g_variant_new("(s)", icon),
                                       &error);

	if (error != NULL) {
		g_error("Unable to send StatusIconsChanged signal: %s", error->message);
		g_error_free(error);
		return;
	}

	return;
}

void
status_service_dbus_set_username (StatusServiceDbus * self, const gchar * username)
{
	g_return_if_fail(IS_STATUS_SERVICE_DBUS(self));

	g_debug("Setting username to: %s", username);

	StatusServiceDbusPrivate * priv = STATUS_SERVICE_DBUS_GET_PRIVATE(self);

	GError * error = NULL;

	if (priv->name != NULL) {
		g_free(priv->name);
	}
	priv->name = g_strdup(username);

	if (priv->bus == NULL) {
		return;
	}

	g_dbus_connection_emit_signal (priv->bus,
                                       NULL,
                                       INDICATOR_ME_SERVICE_DBUS_OBJECT,
                                       INDICATOR_ME_SERVICE_DBUS_INTERFACE,
                                       "UserChanged",
                                       g_variant_new("(s)", username),
                                       &error);

	if (error != NULL) {
		g_warning("Unable to send UserChanged signal: %s", error->message);
		g_error_free(error);
		return;
	}

	return;
}
