// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         Hash wrapper functions for uniform hash interface
// ****************************************************************************

// Copyright 2008, 2009, 2010 Guy Voncken
//
// This file is part of guymager.
//
// guymager is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// guymager is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with guymager. If not, see <http://www.gnu.org/licenses/>.

#include "common.h"

#include <QtCore>
#include "hash.h"

// ------------------------------------
//             MD5 Functions
// ------------------------------------

APIRET HashMD5Init (t_pHashContextMD5 pContext)
{
   #ifdef USE_MD5_FROM_OPENSSL
      (void) MD5_Init (pContext);    // strange, man page says that MD5_Init returns void, but header file says it's int
   #else
      md5_init (pContext);
   #endif

   return NO_ERROR;
}

APIRET HashMD5Append (t_pHashContextMD5 pContext, const void *pData, int DataLen)
{
   #ifdef USE_MD5_FROM_OPENSSL
      (void) MD5_Update (pContext, pData, (unsigned long) DataLen); // Same remark as for MD5_Init
   #else
      md5_append (pContext, (md5_byte_t *)pData, DataLen);
   #endif

   return NO_ERROR;
}

APIRET HashMD5Digest (t_pHashContextMD5 pContext, t_pHashMD5Digest pDigest)
{
   #ifdef USE_MD5_FROM_OPENSSL
      (void) MD5_Final (&pDigest->Buff[0], pContext); // Same remark as for MD5_Init
   #else
      md5_finish (pContext, (md5_byte_t *)&pDigest->Buff[0]);
   #endif

   return NO_ERROR;
}

APIRET HashMD5DigestStr (t_pHashMD5Digest pDigest, QString &Str)
{
   Str.clear();
   for (int i=0; i<HASH_MD5_DIGEST_LENGTH; i++)
      Str += QString ("%1") .arg ((int)(pDigest->Buff[i]), 2, 16, QChar('0'));
   return NO_ERROR;
}

bool HashMD5Match (t_pHashMD5Digest pDigest1, t_pHashMD5Digest pDigest2)
{
   return (memcmp (pDigest1, pDigest2, sizeof(t_HashMD5Digest)) == 0);
}

// ------------------------------------
//           SHA256 Functions
// ------------------------------------

APIRET HashSHA256Init (t_pHashContextSHA256 pContext)
{
   sha256_starts (pContext);
   return NO_ERROR;
}

APIRET HashSHA256Append (t_pHashContextSHA256 pContext, void *pData, int DataLen)
{
   sha256_update (pContext, (unsigned char *)pData, DataLen);
   return NO_ERROR;
}

APIRET HashSHA256Digest (t_pHashContextSHA256 pContext, t_pHashSHA256Digest pDigest)
{
   sha256_finish (pContext, (unsigned char *)pDigest);
   return NO_ERROR;
}

APIRET HashSHA256DigestStr (t_pHashSHA256Digest pDigest, QString &Str)
{
   Str.clear();
   for (int i=0; i<HASH_SHA256_DIGEST_LENGTH; i++)
      Str += QString ("%1") .arg ((int)(pDigest->Buff[i]), 2, 16, QChar('0'));
   return NO_ERROR;
}

bool HashSHA256Match (t_pHashSHA256Digest pDigest1, t_pHashSHA256Digest pDigest2)
{
   return (memcmp (pDigest1, pDigest2, sizeof(t_HashMD5Digest)) == 0);
}


// ------------------------------------
//       Module initialisation
// ------------------------------------

APIRET HashInit (void)
{
   CHK (TOOL_ERROR_REGISTER_CODE (ERROR_HASH_))

   return NO_ERROR;
}

APIRET HashDeInit (void)
{
   return NO_ERROR;
}

