/* gt_shapp.c - Shared application definition management
 *
 * Copyright (C) 1997 Free Software Foundation
 * Copyright (C) 1996, 1997 Eric M. Ludlam
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 *   This file manages and starts shared applications.  A shared
 * application is a program which knows how to communicate with other
 * remote users, and uses etalk as a gateway to initialize the network
 * connections.  These routines store their definition and start them
 * based on proxies, and program requirements.
 *   
 * $Log: gt_shapp.c,v $
 * Revision 1.9  1997/12/14 19:17:07  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.8  1997/07/22 13:21:48  zappo
 * Changed to use the list library.
 *
 * Revision 1.7  1997/03/23 14:49:00  zappo
 * Make sure user->remote is not set when done creating shared app structure.
 *
 * Revision 1.6  1997/02/23 03:22:25  zappo
 * API change on GT_clean_dev changes.
 *
 * Revision 1.5  1997/02/01 14:22:58  zappo
 * Changed USER_send and USER_read to also take Ctxt
 *
 * Revision 1.4  1997/01/28  03:20:36  zappo
 * changed calls to FORK_save_pid
 *
 * Revision 1.3  1997/01/26  15:31:48  zappo
 * Fixed parameters to FORK_save_pid so that it made more sense.
 *
 * Revision 1.2  1996/02/26  00:12:25  zappo
 * Added more control of keyboard controlled sub-processes
 *
 * Revision 1.1  1996/02/01  02:31:02  zappo
 * Initial revision
 *
 * History:
 * zappo   1/21/96    Created
 *
 * Tokens: ::Header:: gtproc.h
 */

#include "gtalklib.h"
#include "gtalkc.h"
#include "gtproc.h"
#include "sitecnfg.h"

static MakeList(list);


/*
 * Function: SHAPP_add
 *
 *   Adds a shared application definition into the shapp database.
 *
 * Returns:     int  - 
 * Parameters:  Ctxt    - Context
 *              alias   - Alias given to this application
 *              command - Command line used to start application
 *              proxy   - Proxy method code
 *              dispreq - display requirements code
 * History:
 * zappo   12/21/95   Created
 */
int SHAPP_add(Ctxt, alias, command, proxy, dispreq)
     struct TalkContext *Ctxt;
     char *alias;
     char *command;
     int   proxy;
     int   dispreq;
{
  struct shapp_node *new;

  if(!alias || !command)
    {
      DISP_message(Ctxt, "Null pointer passed to SHAPP_add");
      return Fail;
    }

  new = (struct shapp_node *)LIST_alloc(&list, sizeof (struct shapp_node));

  new->name = strdup(alias);
  new->commandline = strdup(command);
  if(!new->name || !new->commandline)
    {
      gtalk_shutdown("strdup failed in SHAPP_add");
    }
  if((proxy >= 0) && (proxy <= PROXY_MAX))
    new->proxy = proxy;
  else
    {
      DISP_message(Ctxt, "Bad proxy type in SHAPP_add");
    }
  if((dispreq >= 0) && (dispreq <= DISPREQ_MAX))
    new->dispreq = dispreq;
  else
    {
      DISP_message(Ctxt, "Bad display requirement type in SHAPP_add");
    }
  new->trapkeys = NULL;

  return Success;
}

/*
 * Function: SHAPP_verify
 *
 *   Check for occurance of ALIAS, and make sure proxy and dispreq
 * fields also match.  Return 0 on failure, 1 on success.
 *
 * Returns:     int  - 
 * Parameters:  alias   - Pointer to the alias
 *              proxy   - Id of the proxy
 *              dispreq - Id of the display requirements
 * History:
 * zappo   1/7/96     Created
 */
static unsigned char MatchName(shapp, name)
     struct shapp_node *shapp;
     char              *name;
{
  return !strcmp(shapp->name, name);
}
int SHAPP_verify(alias, proxy, dispreq)
     char *alias;
     int   proxy;
     int   dispreq;
{
  struct shapp_node  *amatch;

  /* Second, find the alias */
  amatch = (struct shapp_node *)LIST_find(&list, MatchName, alias);

  return (amatch && (amatch->proxy == proxy) && (amatch->dispreq == dispreq));
}

/*
 * Function: SHAPP_number
 *
 *   Returns the number of active shared applications stored
 *
 * Returns:     int  - 
 * Parameters:  None
 *
 * History:
 * zappo   12/21/95   Created
 */
int SHAPP_number()
{
  return LIST_map(&list, NULL, NULL);
}


/*
 * Function: SHAPP_print, SHAPP_print_node
 *
 *   Prints out the list of currently active shared applications.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   12/21/95   Created
 */
void SHAPP_print_node(shapp, Ctxt)
     struct shapp_node  *shapp;
     struct TalkContext *Ctxt;
{
  static char *proxyname[] = { "NONE", "TO_ONE", "TO_N"};
  static char *dispreqname[] = { "NONE", "KEYBRD", "TTY", "X" };
  char buffer[200];
  char keysbuff[100];

  if(shapp->trapkeys)
    {
      int l;
      keysbuff[0] = 0;
      for(l=0; shapp->trapkeys[l]; l++)
	{
	  strcat(keysbuff, GTC_c2str(shapp->trapkeys[l]));
	}
    }
  else
    {
      if(shapp->dispreq == DISPREQ_KEYBOARD)
	strcpy(keysbuff, "[unknown]");
      else
	strcpy(keysbuff, "[none]");
    }
  sprintf(buffer, "%s\t%s\t%s\t%-15s\t%s",
	  shapp->name, proxyname[shapp->proxy], dispreqname[shapp->dispreq], 
	  keysbuff, shapp->commandline);
  DISP_message(Ctxt, buffer);
}
void SHAPP_print(Ctxt)
     struct TalkContext *Ctxt;
{
  DISP_message(Ctxt, "Alias\t\tProxy\tDpy Req\tTrap Keys\tCommandline");
  LIST_map(&list, SHAPP_print_node, Ctxt);
}

/*
 * Function: piperead, interp_output
 *
 *   Locally defined function which reads pipes from child processes
 *
 * Returns:     static void  - 
 * Parameters:  Ctxt - Context
 *              io   - Pointer to io
 * History:
 * zappo   1/7/96     Created
 */
static void interp_output(Ctxt, str, shapp)
     struct TalkContext *Ctxt;
     char *str;
     struct shapp_node *shapp;
{
  char *msgbuff;

  if(shapp->dispreq == DISPREQ_KEYBOARD)
    {
      if(!shapp->trapkeys)
	{
	  /* Here the output is a kbd trap */
	  shapp->trapkeys = strdup(str);
	  if(!shapp->trapkeys)
	    gtalk_shutdown("strdup failed in SHAPP readpipe");
	}
      else
	{
	  msgbuff = (char *)malloc(strlen(str) + 2);
	  sprintf(msgbuff, "\03%s", str);
	  DISP_message(Ctxt, msgbuff);
	}
    }
  else
    {
      /* Must be a message of some sort */
      msgbuff = (char *)malloc(strlen(str) + 2);
      sprintf(msgbuff, "\03%s", str);
      DISP_message(Ctxt, msgbuff);
    }
}
static void piperead(Ctxt, io)
     struct TalkContext *Ctxt;
     struct InputDevice *io;
{
  char buffer[1000];		/* capture output */
  char *s, *l;
  int rval;
  struct UserObject *uo;
  struct shapp_node *shapp;
  
  /* Find our associations */
  uo = USER_iofind(io);

  if((rval = GT_recv(io, buffer, sizeof(buffer))) == Fail)
    {
      DISP_message(Ctxt, "Pipe to shared application has closed.");
      /* We have to clean both pipes */
      GT_clean_dev(uo->inpipe, Ctxt->cleanup);
      GT_clean_dev(uo->outpipe, Ctxt->cleanup);
      uo->state = USER_CLOSED;	/* mark this userstruct as closed */
      if(Ctxt->cleanup == AutoClean) USER_clean();
      return;
    }

  /* Terminate the buffer since it's not guaranteed */
  buffer[rval] = 0;

  /* Learn more about this process.  For instance, this may be a message
   * describing a keyboard input to be captured.  This user structs' 
   * name field should match (by pointer) the name of a shared app struct.
   */
  shapp = (struct shapp_node *)LIST_find(&list, MatchName, uo->name);

  if(!shapp)
    {
      DISP_message(Ctxt, "Error finding pipe association!");
      return;
    }

  /* Pipes only flush after a specific call, or a CR.  We don't need
   * to take up a collection so far as subprocesses are kind.
   *
   * Update this to be more robust at a future date!
   */
  s = l = buffer;

  while(*s)
    {
      if((*s == 10) || (*s == 13))
	{
	  *s = 0;

	  if(strlen(l) > 0)
	    interp_output(Ctxt, l, shapp);
	  
	  /* update word start */
	  l = s+1;
	}
      s++;
    }

  if(strlen(l) > 0)
    {
      interp_output(Ctxt, l, shapp);
    }
}

/*
 * Function: shapp_get_pipes
 *
 *   Locally defined function which gets two pipes in preparation for
 * forking a controlled sub-process.  Results are placed in uo's fields
 *
 * Returns:     void -
 * Parameters:  None
 *
 * History:
 * zappo   1/7/96     Created
 */
static int shapp_get_pipes(uo, app, sin, sout)
     struct UserObject  *uo;
     char *app;
     int *sin, *sout;
{
  int piped1[2], piped2[2];

  /* Create the two pipes we need. */
  if(pipe(piped1) || pipe(piped2))
    {
      gtalk_shutdown("shapp_get_pipes: pipe()");
    }

  /* Prepare our write, subprocess read */
  uo->inpipe = GT_pipe(piped1[0], app, "from");/* our reader */
  uo->outpipe = GT_pipe(piped2[1], app, "to"); /* our writer */

  uo->inpipe->readme = piperead; /* setup the reader */

  *sout = piped1[1];		/* pass out thier writer/reader */
  *sin = piped2[0];
  
  return Success;
}

/*
 * Function: SHAPP_fork_shared
 *
 *   Forks shared process specified by ALIAS to user UO.
 * (Multi-connect apps specified by ALIAS should find a running copy
 * and attach to that via proxy.)
 *
 * Returns:     Nothing
 * Parameters:  Ctxt  - Context
 *              alias - Pointer toCharacter of alias
 *              uo    - Pointer to uo
 *              io    - Pointer to io
 * History:
 * zappo   1/6/96     Created
 */
int SHAPP_fork_shared(Ctxt, alias, uo)
     struct TalkContext *Ctxt;
     char               *alias;
     struct UserObject  *uo;
{
  struct shapp_node  *loop;
  union app_def       app;	/* app struct to pass to save_pid */
  struct InputDevice *newtcp;
  struct UserObject  *newuo;
  int rin, rout;
  pid_t kid;
  unsigned char answercode;
  char block[1000];
  char *rp;

  /* First, make sure we can contact them */
  if(!((uo->type == GNUTALK) || 
       ((uo->type == ETALK) && (((uo->ver == 0) && (uo->num >= 11)) ||
				(uo->ver > 0)))))
    {
      DISP_message(Ctxt, "Selected user client does not support shared apps");
      return Fail;
    }

  /* Second, find the alias */
  loop = (struct shapp_node *)LIST_find(&list, MatchName, alias);
  
  if(!loop)
    {
      char buff[100];
      sprintf(buff, "Unknown shared app %s", alias);
      DISP_message(Ctxt, buff);
      return Fail;
    }

  newtcp = DATA_open_new_connection(Ctxt, uo, NEWLINK_APP);

  GT_send(newtcp, &loop->dispreq, 1); /* display requirement */
  GT_send(newtcp, &loop->proxy, 1); /* proxy type */
  GT_send(newtcp, loop->name, strlen(loop->name)); /* the shared alias */
  GT_send(newtcp, "\n", 1);	/* terminate header */

  /* wait for response */
  if(GT_recv(newtcp, &answercode, 1) == Fail)
    {
      DISP_message(Ctxt, "\03Failed to read answer code.");
      return Fail;
    }

  /* check the code */
  if(answercode == 2)
    {
      int msgsize;
      msgsize = GT_recv(newtcp, block+1, sizeof(block));
      block[msgsize] = 0;
      block[0] = ETALK_ESCAPE;
      DISP_message(Ctxt, block); /* report the error */
      GT_clean_dev(newtcp, Ctxt->cleanup);
      return Fail;
    }

  /* Since we have a proxy, create a new userstruct to handle it */
  newuo = USER_alloc();
  newuo->name = strdup(loop->name);
  newuo->datauser = uo;		/* point backwards */
  if(!newuo->name)
    {
      gtalk_shutdown("strdup failed in SHAPP_fork_shared");
    }
  /*newuo->remote = newtcp; will this ever be needed? */
  newuo->editkeys[0] = 0;
  newuo->editkeys[1] = 0;
  newuo->editkeys[2] = 0;

  /* prepare pipes */
  if(shapp_get_pipes(newuo, loop->name, &rin, &rout) == Fail)
    {
      DISP_message(Ctxt, "Failed to load pipes!");
      newuo->state = USER_CLOSED;
      if(Ctxt->cleanup == AutoClean) USER_clean();
      return Fail;
    }

  /* build a new commandline - there should be a %d representing the
   * way we pass in the fd for the proxied socket
   */
  strcpy(block, loop->commandline);
  rp = strstr(block, "**");
  if(rp)
    {
      int offset;
      offset = rp - block;
      sprintf(rp, "%ld", (long)newtcp->fd);
      strcpy(rp + 2, block + offset + 2);
    }

  /* And launch */
  kid = FORK_try(Ctxt, block, NULL, newtcp, rin, rout);
  if(!kid)
    {
      newuo->state = USER_CLOSED;
      if(Ctxt->cleanup == AutoClean) USER_clean();
      return Fail;
    }
  
  app.shapp = loop;
  FORK_save_pid(kid, APP_SHAPP, app, NULL, newuo);

  /* Clean up after it (we don't need some stuff.. maybe) */
  GT_clean_dev(newtcp, Ctxt->cleanup);

  newuo->state = USER_CONNECTED;

  return Success;
}


/*
 * Function: SHAPP_fork_shared_unsolicited
 *
 *   Fork a shared app based on an existing IO which was unsolicited.
 *
 * Returns:     int  - 
 * Parameters:  Ctxt  - Context
 *              alias - Pointer to Character of alias
 *              uo    - Pointer to user object
 *              io    - Pointer to Input device
 * History:
 * zappo   1/7/96     Created
 */
int SHAPP_fork_shared_unsolicited(Ctxt, alias, uo, io)
     struct TalkContext *Ctxt;
     char               *alias;
     struct UserObject  *uo;
     struct InputDevice *io;
{
  struct shapp_node *loop;
  union app_def      app;
  int rin, rout;
  pid_t kid;
  char block[1000];
  char *rp;

  /* Find the alias -- assume pre-checked */
  loop = (struct shapp_node *)LIST_find(&list, MatchName, alias);

  if(!loop) return Fail;

  /* store the app name */
  uo->name = strdup(loop->name);
  if(!uo->name)
    gtalk_shutdown("strdup failed in SHAPP_fork_shared_unsolicited");

  /* prepare pipes */
  if(shapp_get_pipes(uo, loop->name, &rin, &rout) == Fail) 
    {
      DISP_message(Ctxt, "Failed to load pipes!");
      return Fail;
    }
  /* build a new commandline - there should be a %d representing the
   * way we pass in the fd for the proxied socket
   */
  strcpy(block, loop->commandline);
  rp = strstr(block, "**");
  if(rp)
    {
      int offset;
      offset = rp - block;
      sprintf(rp, "%ld", (long)io->fd);
      strcpy(rp + 2, block + offset + 2);
    }

  /* Send the success signal if we made it this far.  We must send it
   * BEFORE we start child, or the child's data may conflict.  We can
   * re-nedge later if necessary.
   */
  GT_send(io, "\01", 1);

  /* And launch */
  kid = FORK_try(Ctxt, block, NULL, io, rin, rout);
  if(!kid)
    {
      return Fail;
    }

  /* If we have no proxy, delete the user struct (which was created to
   * handle the proxy part.   Must start AFTER the fork, in case FORK
   * fails, we need UO to be active for failure message.
   */
#ifdef OOPS_I_NEED_THIS
  /* It turns out that this UO is still needed */
  if(loop->proxy == PROXY_NONE)
    {
      uo->remote = NULL;	/* clear this field to protect it */
      uo->state = USER_CLOSED;	/* mark this userstruct as closed */
      if(Ctxt->cleanup == AutoClean) USER_clean();
    }
  else
#endif
    uo->state = USER_CONNECTED;
  
  app.shapp = loop;
  FORK_save_pid(kid, APP_SHAPP, app, NULL, uo);

  /* Clean up after it (we don't need some stuff.. maybe) */
  uo->remote = NULL;
  GT_clean_dev(io, Ctxt->cleanup);

  return Success;
}
