/* GStreamer yuv (mjpegtools) runtime wrapper
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * gsty4mtrans.c: gstreamer yuv tool runtime wrapper
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-y4mparse
 *
 * <refsect2>
 * <para>
 * Parses a YUV4MPEG stream into video frames (by means of the proper
 * ffmpeg demuxer)
 * </para>
 * </refsect2>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>

#include <gst/video/video.h>

#define GST_TYPE_Y4M_PARSE \
  gst_y4m_parse_get_type ()
#define GST_Y4M_PARSE(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), GST_TYPE_Y4M_PARSE, GstY4mParse))
#define GST_Y4M_PARSE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), GST_TYPE_Y4M_PARSE, GstY4mParseClass))
#define GST_IS_Y4M_PARSE(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), GST_TYPE_Y4M_PARSE))
#define GST_IS_Y4M_PARSE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), GST_TYPE_Y4M_PARSE))

typedef struct _GstY4mParse GstY4mParse;
typedef struct _GstY4mParseClass GstY4mParseClass;

struct _GstY4mParse
{
  GstBin parent;

  GstElement *dec;
  gulong signal_id;
};

struct _GstY4mParseClass
{
  GstBinClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (y4m_parse_debug);
#define GST_CAT_DEFAULT y4m_parse_debug

static GstStaticPadTemplate sink_template = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("application/x-yuv4mpeg, " "y4mversion = (int) 2 ")
    );

static GstStaticPadTemplate src_template = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, Y41B, Y42B, Y444 }")
        "; " "video/x-raw-gray, bpp = (int) 8, depth = (int) 8")
    );

/* properties */
enum
{
  PROP_0,
  PROP_ARGS
};

static GstStateChangeReturn gst_y4m_parse_change_state (GstElement * element,
    GstStateChange transition);

/* properties */
static void gst_y4m_parse_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_y4m_parse_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

static void new_pad (GstElement * element, GstPad * pad, GstY4mParse * parse);

GST_BOILERPLATE (GstY4mParse, gst_y4m_parse, GstBin, GST_TYPE_BIN);

static void
gst_y4m_parse_base_init (gpointer klass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details_simple (element_class, "YUV4MPEG video parser",
      "Codec/Demuxer/Video", "Parses a YUV4MPEG stream into raw YUV video",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&sink_template));
}

static void
gst_y4m_parse_class_init (GstY4mParseClass * klass)
{
  GstElementClass *element_class;
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  element_class = GST_ELEMENT_CLASS (klass);

  GST_DEBUG_CATEGORY_INIT (y4m_parse_debug, "y4mtrans", 0, "y4m wrapper");

  gobject_class->set_property = gst_y4m_parse_set_property;
  gobject_class->get_property = gst_y4m_parse_get_property;

  element_class->change_state = GST_DEBUG_FUNCPTR (gst_y4m_parse_change_state);
}

static void
gst_y4m_parse_init (GstY4mParse * parse, GstY4mParseClass * klass)
{

  parse->dec = gst_element_factory_make ("ffdemux_yuv4mpegpipe",
      "ffdemux_yuv4mpegpipe");
  if (!parse->dec)
    g_warning
        ("Can't find ffdemux_yuv4mpegpipe element, y4mparse will not work");
  else {
    GstPad *pad, *gpad;

    if (!gst_bin_add (GST_BIN (parse), parse->dec)) {
      g_warning
          ("Could not add ffdemux_yuv4mpegpipe element, y4mparse will not work");
      gst_object_unref (parse->dec);
      parse->dec = NULL;
      return;
    }

    /* get the sinkpad */
    pad = gst_element_get_static_pad (parse->dec, "sink");

    /* ghost the sink pad to ourself */
    gpad = gst_ghost_pad_new ("sink", pad);
    gst_element_add_pad (GST_ELEMENT (parse), gpad);

    gst_object_unref (pad);

    /* create some ghostpad */
    gpad = gst_ghost_pad_new_no_target_from_template ("src",
        gst_static_pad_template_get (&src_template));
    gst_element_add_pad (GST_ELEMENT (parse), gpad);

    /* connect a signal to find out when the demuxer created its pad */
    parse->signal_id = g_signal_connect (G_OBJECT (parse->dec), "pad-added",
        G_CALLBACK (new_pad), parse);
  }
}

static void
new_pad (GstElement * element, GstPad * pad, GstY4mParse * parse)
{
  GstPad *gpad;

  gpad = gst_element_get_static_pad (GST_ELEMENT (parse), "src");
  gst_ghost_pad_set_target (GST_GHOST_PAD (gpad), pad);
  gst_object_unref (gpad);

  g_signal_handler_disconnect (parse->dec, parse->signal_id);
}

static void
gst_y4m_parse_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstY4mParse *trans;

  g_return_if_fail (GST_IS_Y4M_PARSE (object));

  trans = GST_Y4M_PARSE (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_y4m_parse_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstY4mParse *trans;

  g_return_if_fail (GST_IS_Y4M_PARSE (object));

  trans = GST_Y4M_PARSE (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstStateChangeReturn
gst_y4m_parse_change_state (GstElement * element, GstStateChange transition)
{
  GstStateChangeReturn ret;

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    goto done;

  switch (transition) {
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      break;
    default:
      break;
  }

done:
  return ret;
}
