/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * VirtualDub filter:
 * Copyright (C) 1999-2000 Donald A. Graft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-msharpen
 *
 * <refsect2>
 * <para>
 * This elements implements an  unusual concept in spatial sharpening.
 * Although designed specifically for anime, it also works well
 * with normal video.  The filter  is very effective at sharpening
 * important edges without amplifying noise.
 * <itemizedlist>
 * <listitem>
 * <para>
 * <link linkend="GstMsharpen--strength">strength</link> indicates
 * the strength of the sharpening to be applied to  the  edge  detail areas.
 * It is applied only to  the  edge detail areas as determined by
 * <link linkend="GstMsharpen--threshold">threshold</link>.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * <link linkend="GstMsharpen--threshold">threshold</link> indicates how
 * close a pixel must be to the brightest or dimmest pixel to be mapped.
 * This determines what is detected as edge detail and thus sharpened.
 * To see what edge detail areas will be sharpened, use
 * <link linkend="GstMsharpen--mask">mask</link>.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * If <link linkend="GstMsharpen--mask">mask</link> is set to true,
 * the areas to be sharpened are shown in white  against a black  background.
 * Use this to set the level of detail to be sharpened.
 * This function also makes a basic edge detection filter.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * <link linkend="GstMsharpen--highq">highq</link> lets you tradeoff
 * speed for quality  of  detail  detection.  Set it to true for the best
 * detail detection.  Set it to false for maximum speed.
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * VirtualDub msharpen filter [Donald A. Graft]
 * </listitem>
 * <listitem>
 * Also available in transcode (msharpen filter)
 * </listitem>
 * <listitem>
 * Also available in avidemux (MSharpen)
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-vd.h"

#include <string.h>
#include <stdlib.h>


#define GST_TYPE_MSHARPEN \
  (gst_msharpen_get_type())
#define GST_MSHARPEN(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_MSHARPEN,GstMsharpen))
#define GST_MSHARPEN_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_MSHARPEN,GstMsharpenClass))
#define GST_IS_MSHARPEN(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_MSHARPEN))
#define GST_IS_MSHARPEN_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_MSHARPEN))


typedef struct _GstMsharpen GstMsharpen;
typedef struct _GstMsharpenClass GstMsharpenClass;

struct _GstMsharpen
{
  GstVideoFilter videofilter;

  gint width, height;

  guint threshold, strength;
  gboolean highq, mask;

  gpointer work, blur;
};


struct _GstMsharpenClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (msharpen_debug);
#define GST_CAT_DEFAULT msharpen_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_THRESHOLD,
  PROP_STRENGTH,
  PROP_HIGHQ,
  PROP_MASK
      /* FILL ME */
};

#define DEFAULT_THRESHOLD   10
#define DEFAULT_STRENGTH   100
#define DEFAULT_HIGHQ     TRUE
#define DEFAULT_MASK     FALSE

static GstStaticPadTemplate gst_msharpen_src_template =
    GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_RGBx " ; " GST_VIDEO_CAPS_BGRx)
    );

static GstStaticPadTemplate gst_msharpen_sink_template =
    GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_RGBx " ; " GST_VIDEO_CAPS_BGRx)
    );


static gboolean gst_msharpen_hook_caps (GstMsharpen * filter,
    GstCaps * incaps, GstCaps * outcaps);
static GstFlowReturn gst_msharpen_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static gboolean gst_msharpen_start (GstBaseTransform * btrans);
static gboolean gst_msharpen_stop (GstBaseTransform * btrans);

static void gst_msharpen_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_msharpen_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstMsharpen, gst_msharpen, GstVideoFilter,
    GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL (GstMsharpen, gst_msharpen,
    gst_msharpen_hook_caps);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_msharpen);

static void
gst_msharpen_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Msharpen",
      "Filter/Effect/Video",
      "Performs sharpening limited to edge areas of the frame",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Donald A. Graft");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_msharpen_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_msharpen_src_template));
}

static void
gst_msharpen_class_init (GstMsharpenClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (msharpen_debug, "msharpen", 0, "msharpen");

  gobject_class->set_property = gst_msharpen_set_property;
  gobject_class->get_property = gst_msharpen_get_property;

  g_object_class_install_property (gobject_class, PROP_THRESHOLD,
      g_param_spec_uint ("threshold", "Threshold",
          "Threshold for pixel to be mapped",
          0, 256, DEFAULT_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_STRENGTH,
      g_param_spec_uint ("strength", "Strength",
          "Strength/weight of modification of a mapped pixel",
          0, 256, DEFAULT_STRENGTH,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_HIGHQ,
      g_param_spec_boolean ("highq", "High Quality",
          "Tradoff speed for quality detection",
          DEFAULT_HIGHQ, G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_MASK,
      g_param_spec_boolean ("mask", "Mask",
          "Show areas to be sharpened in white (basic edge detector)",
          DEFAULT_MASK, G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_msharpen_set_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_msharpen_get_unit_size);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_msharpen_transform);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_msharpen_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_msharpen_stop);
}

static void
gst_msharpen_init (GstMsharpen * filter, GstMsharpenClass * g_class)
{
  filter->threshold = DEFAULT_THRESHOLD;
  filter->strength = DEFAULT_STRENGTH;
  filter->highq = DEFAULT_HIGHQ;
  filter->mask = DEFAULT_MASK;

  filter->work = NULL;
  filter->blur = NULL;
}

static gboolean
gst_msharpen_hook_caps (GstMsharpen * filter, GstCaps * incaps,
    GstCaps * outcaps)
{
  GstStructure *structure;

  structure = gst_caps_get_structure (incaps, 0);

  g_free (filter->work);
  g_free (filter->blur);
  filter->work = g_malloc (filter->width * filter->height * 4);
  filter->blur = g_malloc (filter->width * filter->height * 4);

  return TRUE;
}

static GstFlowReturn
gst_msharpen_transform (GstBaseTransform * btrans, GstBuffer * in,
    GstBuffer * out)
{
  GstMsharpen *filter;
  guint8 *src, *dst;
  GstFlowReturn ret = GST_FLOW_OK;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_MSHARPEN (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);
  dst = (guint8 *) GST_BUFFER_DATA (out);

  guint width = filter->width;
  guint height = filter->height;

  guint strength = filter->strength;
  guint invstrength = 256 - strength;
  guint threshold = filter->threshold;

  guint8 *work = filter->work;
  guint8 *blur = filter->blur;

  const guint pitch = 4 * width;
  guint8 *srcpp, *srcp, *srcpn, *workp, *blurp, *blurpn, *dstp;
  gint r1, r2, r3, r4, g1, g2, g3, g4, b1, b2, b3, b4;
  gint x, y, max;

  /* Blur the source image prior to detail detection.
   * Separate dimensions for speed. */

  /* Vertical. */
  srcpp = src;
  srcp = srcpp + pitch;
  srcpn = srcp + pitch;
  workp = work + pitch;
  for (y = 1; y < height - 1; y++) {
    for (x = 0; x < pitch; x++) {
      workp[x] = (srcpp[x] + srcp[x] + srcpn[x]) / 3;
    }
    srcpp += pitch;
    srcp += pitch;
    srcpn += pitch;
    workp += pitch;
  }

  /* Horizontal. */
  workp = work;
  blurp = blur;
  for (y = 0; y < height; y++) {
    for (x = 4; x < pitch - 4; x++) {
      blurp[x] = (workp[x - 4] + workp[x] + workp[x + 4]) / 3;
    }
    workp += pitch;
    blurp += pitch;
  }

  /* Fix up blur frame borders. */
  srcp = src;
  blurp = blur;
  memcpy (blurp, srcp, pitch);
  memcpy (blurp + (height - 1) * pitch, srcp + (height - 1) * pitch, pitch);
  for (y = 0; y < height; y++) {
    *((guint32 *) (&blurp[0])) = *((guint32 *) (&srcp[0]));
    *((guint32 *) (&blurp[pitch - 4])) = *((guint32 *) (&srcp[pitch - 4]));
    srcp += pitch;
    blurp += pitch;
  }

  /* Diagonal detail detection. */
  blurp = blur;
  blurpn = blurp + pitch;
  workp = work;
  for (y = 0; y < height - 1; y++) {
    b1 = blurp[0];
    g1 = blurp[1];
    r1 = blurp[2];
    b3 = blurpn[0];
    g3 = blurpn[1];
    r3 = blurpn[2];
    for (x = 0; x < pitch - 4; x += 4) {
      b2 = blurp[x + 4];
      g2 = blurp[x + 5];
      r2 = blurp[x + 6];
      b4 = blurpn[x + 4];
      g4 = blurpn[x + 5];
      r4 = blurpn[x + 6];
      if ((abs (b1 - b4) >= threshold) || (abs (g1 - g4) >= threshold)
          || (abs (r1 - r4) >= threshold)
          || (abs (b2 - b3) >= threshold) || (abs (g2 - g3) >= threshold)
          || (abs (g2 - g3) >= threshold)) {
        *((guint32 *) (&workp[x])) = 0xffffffff;
      } else {
        *((guint32 *) (&workp[x])) = 0x0;
      }
      b1 = b2;
      b3 = b4;
      g1 = g2;
      g3 = g4;
      r1 = r2;
      r3 = r4;
    }
    workp += pitch;
    blurp += pitch;
    blurpn += pitch;
  }

  if (filter->highq) {
    /* Vertical detail detection. */
    for (x = 0; x < pitch; x += 4) {
      blurp = blur;
      blurpn = blurp + pitch;
      workp = work;
      b1 = blurp[x];
      g1 = blurp[x + 1];
      r1 = blurp[x + 2];
      for (y = 0; y < height - 1; y++) {
        b2 = blurpn[x];
        g2 = blurpn[x + 1];
        r2 = blurpn[x + 2];
        if (abs (b1 - b2) >= threshold || abs (g1 - g2) >= threshold
            || abs (r1 - r2) >= threshold) {
          *((guint32 *) (&workp[x])) = 0xffffffff;
        }
        b1 = b2;
        g1 = g2;
        r1 = r2;
        workp += pitch;
        blurp += pitch;
        blurpn += pitch;
      }
    }

    /* Horizontal detail detection. */
    blurp = blur;
    workp = work;
    for (y = 0; y < height; y++) {
      b1 = blurp[0];
      g1 = blurp[1];
      r1 = blurp[2];
      for (x = 0; x < pitch - 4; x += 4) {
        b2 = blurp[x + 4];
        g2 = blurp[x + 5];
        r2 = blurp[x + 6];
        if (abs (b1 - b2) >= threshold || abs (g1 - g2) >= threshold
            || abs (r1 - r2) >= threshold) {
          *((guint32 *) (&workp[x])) = 0xffffffff;
        }
        b1 = b2;
        g1 = g2;
        r1 = r2;
      }
      workp += pitch;
      blurp += pitch;
    }
  }

  /* Fix up detail map borders. */
  memset (work + (height - 1) * pitch, 0, pitch);
  workp = work;
  for (y = 0; y < height; y++) {
    *((guint32 *) (&workp[pitch - 4])) = 0;
    workp += pitch;
  }

  if (filter->mask) {
    /* only need the edge info, copy and return */
    memcpy (dst, work, GST_BUFFER_SIZE (out));

    return ret;
  }

  /* Fix up output frame borders. */
  srcp = src;
  dstp = dst;
  memcpy (dstp, srcp, pitch);
  memcpy (dstp + (height - 1) * pitch, srcp + (height - 1) * pitch, pitch);
  for (y = 0; y < height; y++) {
    *((guint32 *) (&dstp[0])) = *((guint32 *) (&srcp[0]));
    *((guint32 *) (&dstp[pitch - 4])) = *((guint32 *) (&srcp[pitch - 4]));
    srcp += pitch;
    dstp += pitch;
  }

  /* Now sharpen the edge areas and we're done! */
  srcp = src + pitch;
  dstp = dst + pitch;
  workp = work + pitch;
  blurp = blur + pitch;
  for (y = 1; y < height - 1; y++) {
    for (x = 4; x < pitch - 4; x += 4) {
      gint xplus1 = x + 1, xplus2 = x + 2;

      if (workp[x]) {
        b4 = (4 * ((gint) srcp[x]) - 3 * blurp[x]);
        g4 = (4 * ((gint) srcp[x + 1]) - 3 * blurp[x + 1]);
        r4 = (4 * ((gint) srcp[x + 2]) - 3 * blurp[x + 2]);

        if (b4 < 0)
          b4 = 0;
        if (g4 < 0)
          g4 = 0;
        if (r4 < 0)
          r4 = 0;
        max = b4;
        if (g4 > max)
          max = g4;
        if (r4 > max)
          max = r4;
        if (max > 255) {
          b4 = (b4 * 255) / max;
          g4 = (g4 * 255) / max;
          r4 = (r4 * 255) / max;
        }
        dstp[x] = (strength * b4 + invstrength * srcp[x]) >> 8;
        dstp[xplus1] = (strength * g4 + invstrength * srcp[xplus1]) >> 8;
        dstp[xplus2] = (strength * r4 + invstrength * srcp[xplus2]) >> 8;
      } else {
        dstp[x] = srcp[x];
        dstp[xplus1] = srcp[xplus1];
        dstp[xplus2] = srcp[xplus2];
      }
    }
    srcp += pitch;
    dstp += pitch;
    workp += pitch;
    blurp += pitch;
  }


  return ret;
}


static gboolean
gst_msharpen_start (GstBaseTransform * btrans)
{
  GstMsharpen *filter = GST_MSHARPEN (btrans);

  filter->work = NULL;
  filter->blur = NULL;

  return TRUE;
}

static gboolean
gst_msharpen_stop (GstBaseTransform * btrans)
{
  GstMsharpen *filter = GST_MSHARPEN (btrans);

  g_free (filter->work);
  filter->work = NULL;
  g_free (filter->blur);
  filter->blur = NULL;

  return TRUE;
}

static void
gst_msharpen_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstMsharpen *src;

  g_return_if_fail (GST_IS_MSHARPEN (object));
  src = GST_MSHARPEN (object);

  switch (prop_id) {
    case PROP_THRESHOLD:
      src->threshold = g_value_get_uint (value);
      break;
    case PROP_STRENGTH:
      src->strength = g_value_get_uint (value);
      break;
    case PROP_HIGHQ:
      src->highq = g_value_get_boolean (value);
      break;
    case PROP_MASK:
      src->mask = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_msharpen_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstMsharpen *src;

  g_return_if_fail (GST_IS_MSHARPEN (object));
  src = GST_MSHARPEN (object);

  switch (prop_id) {
    case PROP_THRESHOLD:
      g_value_set_uint (value, src->threshold);
      break;
    case PROP_STRENGTH:
      g_value_set_uint (value, src->strength);
      break;
    case PROP_HIGHQ:
      g_value_set_boolean (value, src->highq);
      break;
    case PROP_MASK:
      g_value_set_boolean (value, src->mask);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
