/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * transcode filter:
 * Copyright (C) Chad Page - October 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-smooth
 *
 * <refsect2>
 * <para>
 * Performs "single-frame" smoothing, that is, it only works with the current
 * frame, it does not need the next or the previous frame.
 * Usually smoothing is done by taking the data of previous frames into account
 * to see which parts of the picture can be "safely" smoothed, this filter only
 * needs one frame.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * transcode smooth filter [Chad Page]
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <stdlib.h>

#define GST_TYPE_SMOOTH \
  (gst_smooth_get_type())
#define GST_SMOOTH(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_SMOOTH,GstSmooth))
#define GST_SMOOTH_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_SMOOTH,GstSmoothClass))
#define GST_IS_SMOOTH(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_SMOOTH))
#define GST_IS_SMOOTH_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_SMOOTH))

typedef struct _GstSmooth GstSmooth;
typedef struct _GstSmoothClass GstSmoothClass;


struct _GstSmooth
{
  GstVideoFilter videofilter;

  gint width, height;

  /* denoise parameters */
  gfloat blend_factor;
  guint chroma_diff, luma_diff, search_range;
};

struct _GstSmoothClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (smooth_debug);
#define GST_CAT_DEFAULT smooth_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_BLEND_FACTOR,
  PROP_CHROMA_DIFF,
  PROP_LUMA_DIFF,
  PROP_SEARCH_RANGE
      /* FILL ME */
};

#define DEFAULT_BLEND_FACTOR    0.25
#define DEFAULT_CHROMA_DIFF        6
#define DEFAULT_LUMA_DIFF          8
#define DEFAULT_SEARCH_RANGE       4
#define MAX_BLEND_FACTOR         1.0
#define MAX_CHROMA_DIFF           16
#define MAX_LUMA_DIFF             16
#define MAX_SEARCH_RANGE          16

static GstStaticPadTemplate gst_smooth_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_smooth_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstFlowReturn gst_smooth_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static gboolean gst_smooth_start (GstBaseTransform * btrans);
static gboolean gst_smooth_stop (GstBaseTransform * btrans);

static void gst_smooth_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_smooth_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstSmooth, gst_smooth, GstVideoFilter, GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE (GstSmooth, gst_smooth);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_smooth);

static void
gst_smooth_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Smooth",
      "Filter/Effect/Video", "Smoothing (single frame)",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Chad Page");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_smooth_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_smooth_src_template));
}

static void
gst_smooth_class_init (GstSmoothClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (smooth_debug, "smooth", 0, "smooth");

  gobject_class->set_property = gst_smooth_set_property;
  gobject_class->get_property = gst_smooth_get_property;

  g_object_class_install_property (gobject_class, PROP_BLEND_FACTOR,
      g_param_spec_float ("blend-factor", "Blend Factor", "Blend Factor",
          0, MAX_BLEND_FACTOR, DEFAULT_BLEND_FACTOR,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_CHROMA_DIFF,
      g_param_spec_uint ("chroma-diff", "Chroma Difference",
          "Maximum Chroma Difference", 0, MAX_CHROMA_DIFF, DEFAULT_CHROMA_DIFF,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_LUMA_DIFF,
      g_param_spec_uint ("luma-diff", "Luma Difference",
          "Maximum Luma Difference", 0, MAX_LUMA_DIFF, DEFAULT_LUMA_DIFF,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_SEARCH_RANGE,
      g_param_spec_uint ("search-range", "Search Range", "Search Range",
          0, MAX_SEARCH_RANGE, DEFAULT_SEARCH_RANGE,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_smooth_set_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_smooth_get_unit_size);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_smooth_transform);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_smooth_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_smooth_stop);
}

static void
gst_smooth_init (GstSmooth * filter, GstSmoothClass * g_class)
{
  filter->blend_factor = DEFAULT_BLEND_FACTOR;
  filter->chroma_diff = DEFAULT_CHROMA_DIFF;
  filter->luma_diff = DEFAULT_LUMA_DIFF;
  filter->search_range = DEFAULT_SEARCH_RANGE;
}

static void
gst_smooth (guint8 * src, guint8 * buf, gint width, gint height, gint maxdiff,
    gint maxldiff, gint maxdist, gfloat level)
{
  gint x, y, pu, cpu, cdiff;
  gint xa, ya, oval, ldiff;
  guint8 *bufcr, *bufcb;
  guint8 *tbufcr, *tbufcb, *ltbuf;
  gfloat dist, ratio, nval;

  ltbuf = src;
  tbufcr = ltbuf + GST_VIDEO_I420_U_OFFSET (width, height);
  tbufcb = ltbuf + GST_VIDEO_I420_V_OFFSET (width, height);

  bufcr = buf + GST_VIDEO_I420_U_OFFSET (width, height);
  bufcb = buf + GST_VIDEO_I420_V_OFFSET (width, height);

  /* First pass - horizontal */
  for (y = 0; y < (height); y++) {
    for (x = 0; x < width; x++) {
      pu = ((y >> 1) * (width >> 1)) + (x >> 1);
      nval = ((gfloat) buf[x + (y * width)]);
      oval = buf[x + (y * width)];
      for (xa = x - maxdist; (xa <= (x + maxdist)) && (xa < width); xa++) {
        if (xa < 0)
          xa = 0;
        if ((xa == x) && (xa < width - 1))
          xa++;
        cpu = ((y >> 1) * (width >> 1)) + (xa >> 1);
        cdiff = abs (tbufcr[pu] - tbufcr[cpu]);
        cdiff += abs (tbufcb[pu] - tbufcb[cpu]);

        /* If color difference not too great,
         * average the pixel according to distance */
        ldiff = abs (ltbuf[xa + (y * width)] - oval);
        if ((cdiff < maxdiff) && (ldiff < maxldiff)) {
          dist = abs (xa - x);
          ratio = level / dist;
          nval = nval * (1 - ratio);
          nval += ((gfloat) ltbuf[xa + (y * width)]) * ratio;
        }
      }
      buf[x + (y * width)] = (guint8) (nval + 0.5);
    }
  }

  /* Second pass - vertical lines */
  for (y = 0; y < (height); y++) {
    for (x = 0; x < width; x++) {
      pu = ((y >> 1) * (width >> 1)) + (x >> 1);
      nval = ((gfloat) buf[x + (y * width)]);
      oval = buf[x + (y * width)];
      for (ya = y - maxdist; (ya <= (y + maxdist)) && (ya < height); ya++) {
        if (ya < 0)
          ya = 0;
        if ((ya == y) && (ya < height - 1))
          ya++;
        cpu = ((ya >> 1) * (width >> 1)) + (x >> 1);
        cdiff = abs (tbufcr[pu] - tbufcr[cpu]);
        cdiff += abs (tbufcb[pu] - tbufcb[cpu]);

        /* If color difference not too great,
         * average the pixel according to distance */
        ldiff = abs (ltbuf[x + (ya * width)] - oval);
        if ((cdiff < maxdiff) && (ldiff < maxldiff)) {
          dist = abs (ya - y);
          ratio = level / dist;
          nval = nval * (1 - ratio);
          nval += ((gfloat) ltbuf[x + (ya * width)]) * ratio;
        }
      }
      buf[x + (y * width)] = (guint8) (nval + 0.5);
    }
  }
}

static GstFlowReturn
gst_smooth_transform (GstBaseTransform * btrans, GstBuffer * in,
    GstBuffer * out)
{
  GstSmooth *filter;
  guint8 *src, *dest;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_SMOOTH (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);
  dest = (guint8 *) GST_BUFFER_DATA (out);

  oil_memcpy (dest, src, GST_VIDEO_I420_SIZE (filter->width, filter->height));

  gst_smooth (src, dest, GST_VIDEO_I420_Y_ROWSTRIDE (filter->width),
      filter->height, filter->chroma_diff, filter->luma_diff,
      filter->search_range, filter->blend_factor);

  return GST_FLOW_OK;
}

static gboolean
gst_smooth_start (GstBaseTransform * btrans)
{

  return TRUE;
}

static gboolean
gst_smooth_stop (GstBaseTransform * btrans)
{

  return TRUE;
}

static void
gst_smooth_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstSmooth *src;

  g_return_if_fail (GST_IS_SMOOTH (object));
  src = GST_SMOOTH (object);

  switch (prop_id) {
    case PROP_BLEND_FACTOR:
      src->blend_factor = g_value_get_float (value);
      break;
    case PROP_CHROMA_DIFF:
      src->chroma_diff = g_value_get_uint (value);
      break;
    case PROP_LUMA_DIFF:
      src->luma_diff = g_value_get_uint (value);
      break;
    case PROP_SEARCH_RANGE:
      src->search_range = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_smooth_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstSmooth *src;

  g_return_if_fail (GST_IS_SMOOTH (object));
  src = GST_SMOOTH (object);

  switch (prop_id) {
    case PROP_BLEND_FACTOR:
      g_value_set_float (value, src->blend_factor);
      break;
    case PROP_CHROMA_DIFF:
      g_value_set_uint (value, src->chroma_diff);
      break;
    case PROP_LUMA_DIFF:
      g_value_set_uint (value, src->luma_diff);
      break;
    case PROP_SEARCH_RANGE:
      g_value_set_uint (value, src->search_range);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
