/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * transcode filter:
 * Copyright (C) Gerhard Monzel - November 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-dnr
 *
 * <refsect2>
 * <title>Operation</title>
 * <para>
 * [translated from German transcode documentation]
 * </para>
 * <para>
 * The filter operates best on material in YUV/I420 format,
 * but can also be used for RGB material.
 * </para>
 * <para>
 * It operates only on one image when no scene-change is recognized.
 * This is determined by the proportion of pixels in an image that have
 * exceeded all threshold values (so when this propertion exceeds the
 * prescribed <link linkend="GstDnr--scene-change">scene-change</link>
 * value).
 * </para>
 * <para>
 * The filter examines each pixel of a picture separately according to
 * luma and chroma portions (YUV mode) and/or R/G/B portions (RGB mode),
 * so that both brightness and color noise can be treated separately.
 * </para>
 * <para>
 * The threshold values <link linkend="GstDnr--luma-threshold">luma-threshold</link>
 * and <link linkend="GstDnr--chroma-threshold">chroma-threshold</link>
 * indicate to what extent of difference between a pixel of the previous
 * image and one of the current image a "cross fade" of pixels is applied.
 * The former is used on the luma (or R) component, the latter on
 * chroma (or G/B) components.
 * </para>
 * <para>
 * The threshold values <link linkend="GstDnr--luma-lock">luma-lock</link>
 * and <link linkend="GstDnr--chroma-lock">chroma-lock</link>
 * indicate to what extent of difference between a pixel of the previous
 * image and one of the current image a "replacement" by a previous pixel
 * is performed, that is, whether the pixel value is kept locked.
 * The former is used on the luma (or R) component, the latter on
 * chroma (or G/B) components.
 * In any event, a pixel will be "refreshed" after at most 30 locks.
 * </para>
 * <para>
 * In principle the threshold values should be larger than the lock values,
 * so that a balanced relationship between "locking" and "cross-fading" ensues.
 * The threshold values are compared with the differences (biased) of pixels,
 * i.e. larger threshold values strengthen the filter effect,
 * because also larger differences (differences) are thus filtered.
 * The default parameter values should yield a usable result in most cases.
 * The filter has (then) altogether beside that noise reduction a soft-drawing
 * effect however without "streak-effect" with movement.
 * After filtering noise-afflicted material, (e.g. MPEG) compression can
 * save up to 25% bitrate (this can however clearly be more or less depending
 * on material).
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * transcode dnr filter [Gerhard Monzel]
 * </listitem>
 * <listitem>
 * Also available in avidemux (Denoise)
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <math.h>
#include <string.h>

#define GST_TYPE_DNR \
  (gst_dnr_get_type())
#define GST_DNR(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_DNR,GstDnr))
#define GST_DNR_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_DNR,GstDnrClass))
#define GST_IS_DNR(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_DNR))
#define GST_IS_DNR_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_DNR))

typedef struct _GstDnr GstDnr;
typedef struct _GstDnrClass GstDnrClass;

typedef struct t_dnr_filter_ctx T_DNR_FILTER_CTX;

struct _GstDnr
{
  GstVideoFilter videofilter;

  gint width, height;
  gboolean is_rgb;

  /* denoise parameters */
  guint luma_blend, luma_lock;
  guint chroma_blend, chroma_lock;
  guint scene_change;

  /* dnr context; see further on */
  T_DNR_FILTER_CTX *fctx;
};

struct _GstDnrClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (dnr_debug);
#define GST_CAT_DEFAULT dnr_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_LUMA_BLEND,
  PROP_LUMA_LOCK,
  PROP_CHROMA_BLEND,
  PROP_CHROMA_LOCK,
  PROP_SCENE_CHANGE
      /* FILL ME */
};

#define DEFAULT_LUMA_BLEND     DEFAULT_LT
#define DEFAULT_LUMA_LOCK      DEFAULT_LL
#define DEFAULT_CHROMA_BLEND   DEFAULT_CT
#define DEFAULT_CHROMA_LOCK    DEFAULT_CL
#define DEFAULT_SCENE_CHANGE   DEFAULT_SC

#define DEFAULT_LT 10
#define DEFAULT_LL  4
#define DEFAULT_CT 16
#define DEFAULT_CL  8
#define DEFAULT_SC 30

static GstStaticPadTemplate gst_dnr_src_template =
    GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }") " ; "
        GST_VIDEO_CAPS_BGR)
    );

static GstStaticPadTemplate gst_dnr_sink_template =
    GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }") " ; "
        GST_VIDEO_CAPS_BGR)
    );

static gboolean gst_dnr_hook_caps (GstDnr * filter, GstCaps * incaps,
    GstCaps * outcaps);
static GstFlowReturn gst_dnr_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);
static gboolean gst_dnr_start (GstBaseTransform * btrans);
static gboolean gst_dnr_stop (GstBaseTransform * btrans);

static void gst_dnr_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_dnr_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstDnr, gst_dnr, GstVideoFilter, GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL (GstDnr, gst_dnr, gst_dnr_hook_caps);

static void
gst_dnr_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Dnr",
      "Filter/Effect/Video", "Dynamic noise reduction",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Gerhard Monzel");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_dnr_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_dnr_src_template));
}

static void
gst_dnr_class_init (GstDnrClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (dnr_debug, "dnr", 0, "dnr");

  gobject_class->set_property = gst_dnr_set_property;
  gobject_class->get_property = gst_dnr_get_property;

  g_object_class_install_property (gobject_class, PROP_LUMA_BLEND,
      g_param_spec_uint ("luma-threshold", "Luma Blend Threshold",
          "Threshold to blend luma/red",
          1, 128, DEFAULT_LUMA_BLEND,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_LUMA_LOCK,
      g_param_spec_uint ("luma-lock", "Luma Lock Threshold",
          "Threshold to lock luma/red",
          1, 128, DEFAULT_LUMA_LOCK,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_CHROMA_BLEND,
      g_param_spec_uint ("chroma-threshold", "Chroma Blend Threshold",
          "Threshold to blend chroma/green+blue",
          1, 128, DEFAULT_CHROMA_BLEND,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_CHROMA_LOCK,
      g_param_spec_uint ("chroma-lock", "Chroma Lock Threshold",
          "Threshold to lock chroma/green+blue",
          1, 128, DEFAULT_CHROMA_LOCK,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_SCENE_CHANGE,
      g_param_spec_uint ("scene-change", "Scene Change",
          "Percentage of picture difference (scene change)",
          1, 90, DEFAULT_SCENE_CHANGE,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_dnr_set_caps);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_dnr_transform_ip);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_dnr_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_dnr_stop);
}

static void
gst_dnr_init (GstDnr * filter, GstDnrClass * g_class)
{
  filter->luma_blend = DEFAULT_LUMA_BLEND;
  filter->luma_lock = DEFAULT_LUMA_LOCK;
  filter->chroma_blend = DEFAULT_CHROMA_BLEND;
  filter->chroma_lock = DEFAULT_CHROMA_LOCK;
  filter->scene_change = DEFAULT_SCENE_CHANGE;
}

/* ---- begin filter legacy code ---- */
/* only minor changes; some code compliance, use oil_memcpy and alike */

typedef unsigned char T_PIXEL;

struct t_dnr_filter_ctx
{
  int is_first_frame;
  int pPartial;
  int pThreshold;
  int pThreshold2;
  int pPixellock;
  int pPixellock2;
  int pScene;

  int isYUV;
  T_PIXEL *lastframe;
  T_PIXEL *origframe;
  int gu_ofs, rv_ofs;

  unsigned char lookup[256][256];
  unsigned char *lockhistory;

  T_PIXEL *src_data;
  T_PIXEL *undo_data;
  long src_h, src_w;
  int img_size;
  int hist_size;
  int pitch;
  int line_size_c;
  int line_size_l;
  int undo;

};


static int
gst_dnr_run (T_DNR_FILTER_CTX * fctx, T_PIXEL * data)
{
  T_PIXEL *RY1, *RY2, *RY3, *GU1, *GU2, *GU3, *BV1, *BV2, *BV3;
  int rl, rc, w, h, update_needed, totpixels;
  int threshRY, threshGU = 0, threshBV = 0;
  int ry1, ry2, gu1 = 0, gu2 = 0, bv1 = 0, bv2 = 0;
  long totlocks = 0;
  unsigned char *lockhistory = fctx->lockhistory;

  //-- get data into account --
  fctx->src_data = data;

  //-- if we are dealing with the first --
  //-- frame, just make a copy.         --
  if (fctx->is_first_frame) {
    oil_memcpy (fctx->lastframe, fctx->src_data, fctx->img_size);
    fctx->undo_data = fctx->lastframe;
    fctx->is_first_frame = 0;

    return 0;
  }
  //-- make sure to preserve the existing frame --
  //-- in case this is a scene change           --
  oil_memcpy (fctx->origframe, fctx->src_data, fctx->img_size);

  if (fctx->isYUV) {
    RY1 = fctx->src_data;
    GU1 = RY1 + fctx->gu_ofs;
    BV1 = RY1 + fctx->rv_ofs;

    RY2 = fctx->lastframe;
    GU2 = RY2 + fctx->gu_ofs;
    BV2 = RY2 + fctx->rv_ofs;

    RY3 = fctx->src_data;
    GU3 = RY3 + fctx->gu_ofs;
    BV3 = RY3 + fctx->rv_ofs;
  } else {
    BV1 = fctx->src_data;
    GU1 = BV1 + fctx->gu_ofs;
    RY1 = BV1 + fctx->rv_ofs;

    BV2 = fctx->lastframe;
    GU2 = BV2 + fctx->gu_ofs;
    RY2 = BV2 + fctx->rv_ofs;

    BV3 = fctx->src_data;
    GU3 = BV3 + fctx->gu_ofs;
    RY3 = BV3 + fctx->rv_ofs;
  }

  h = fctx->src_h;
  do {
    w = fctx->src_w;
    rl = rc = 0;
    do {
      update_needed = 1;

      //-- on every row get (luma) actual/locked pixels --
      //-- -> calculate thresold (biased diff.)         --
      //--------------------------------------------------
      ry1 = RY1[rl];
      ry2 = RY2[rl];
      threshRY = fctx->lookup[ry1][ry2];

      //-- in YUV-Mode on every even row  (RGB every --
      //-- row) get (chroma) actual/locked pixels    --
      //-- -> calculate thresold (biased diff.)      --
      //-----------------------------------------------
      if (!fctx->isYUV || !(rl & 0x01)) {
        gu1 = GU1[rc];
        bv1 = BV1[rc];
        gu2 = GU2[rc];
        bv2 = BV2[rc];
        threshGU = fctx->lookup[gu1][gu2];
        threshBV = fctx->lookup[bv1][bv2];
      }
      //-- PARTIAL --
      //-------------
      if (fctx->pPartial) {
        // we're doing a full pixel lock since we're --
        // under all thresholds in a couple of time  --
        //---------------------------------------------
        if ((threshRY < fctx->pPixellock) &&
            (threshGU < fctx->pPixellock2) && (threshBV < fctx->pPixellock2)) {

          //-- if we've locked more than 30 times at --
          //-- this point, let's refresh the pixel.  --
          if (*lockhistory > 30) {
            *lockhistory = 0;

            ry1 = (ry1 + ry2) / 2;
            gu1 = (gu1 + gu2) / 2;
            bv1 = (bv1 + bv2) / 2;
          } else {
            *lockhistory = *lockhistory + 1;

            //-- take locked pixels --
            ry1 = ry2;
            gu1 = gu2;
            bv1 = bv2;
          }
        }
        //-- If the luma is within pixellock, and the chroma is within   --
        //-- blend, lets blend the chroma and lock the luma.
        //-----------------------------------------------------------------
        else if ((threshRY < fctx->pPixellock) &&
            (threshGU < fctx->pThreshold2) && (threshBV < fctx->pThreshold2)) {
          *lockhistory = 0;

          ry1 = ry2;
          gu1 = (gu1 + gu2) / 2;
          bv1 = (bv1 + bv2) / 2;
        }
        //-- We are above pixellock in luma and chroma, but     --
        //-- below the blend thresholds in both, so let's blend --
        //--------------------------------------------------------
        else if ((threshRY < fctx->pThreshold) &&
            (threshGU < fctx->pThreshold2) && (threshBV < fctx->pThreshold2)) {
          *lockhistory = 0;

          ry1 = (ry1 + ry2) / 2;
          gu1 = (gu1 + gu2) / 2;
          bv1 = (bv1 + bv2) / 2;
        }
        //-- if we are above all thresholds, --
        //-- just leave the output untouched --
        //-------------------------------------
        else {
          *lockhistory = 0;
          update_needed = 0;
          totlocks++;
        }
      }
      //-- nonPARTIAL --
      //----------------
      else {
        //-- beneath pixellock so lets keep   --
        //-- the existing pixel (most likely) --
        //--------------------------------------
        if ((threshRY < fctx->pPixellock) &&
            (threshGU < fctx->pPixellock2) && (threshBV < fctx->pPixellock2)) {
          // if we've locked more than 30 times at this point,
          // let's refresh the pixel
          if (*lockhistory > 30) {
            *lockhistory = 0;

            ry1 = (ry1 + ry2) / 2;
            gu1 = (gu1 + gu2) / 2;
            bv1 = (bv1 + bv2) / 2;
          } else {
            *lockhistory = *lockhistory + 1;

            ry1 = ry2;
            gu1 = gu2;
            bv1 = bv2;
          }
        }
        //-- we are above pixellock, but below the -- 
        //-- blend threshold so we want to blend   --
        //-------------------------------------------
        else if ((threshRY < fctx->pThreshold) &&
            (threshGU < fctx->pThreshold2) && (threshBV < fctx->pThreshold2)) {
          *lockhistory = 0;

          ry1 = (ry1 + ry2) / 2;
          gu1 = (gu1 + gu2) / 2;
          bv1 = (bv1 + bv2) / 2;
        }
        //-- it's beyond the thresholds, just leave it alone --
        //-----------------------------------------------------
        else {
          *lockhistory = 0;
          update_needed = 0;
          totlocks++;
        }
      }

      //-- set destination --
      //---------------------
      if (update_needed) {
        RY3[rl] = ry1;
        GU3[rc] = gu1;
        BV3[rc] = bv1;
      }
      //-- refresh locked pixels --
      //---------------------------
      if (*lockhistory == 0) {
        RY2[rl] = ry1;
        GU2[rc] = gu1;
        BV2[rc] = bv1;
      }

      lockhistory++;

      rl += fctx->pitch;
      rc = (fctx->isYUV) ? (rl >> 1) : rl;
    } while (--w);

    //-- next line ... --
    RY1 += fctx->line_size_l;
    RY2 += fctx->line_size_l;
    RY3 += fctx->line_size_l;

    //-- ... in YUV-Mode for chromas, only on odd luma-lines --
    if (!fctx->isYUV || (h & 0x01)) {
      GU1 += fctx->line_size_c;
      BV1 += fctx->line_size_c;

      GU2 += fctx->line_size_c;
      BV2 += fctx->line_size_c;

      GU3 += fctx->line_size_c;
      BV3 += fctx->line_size_c;
    }
  } while (--h);

  totpixels = fctx->src_h * fctx->src_w;
  totpixels *= fctx->pScene;
  totpixels /= 100;

  // If more than the specified percent of pixels have exceeded all thresholds
  // then we restore the saved frame.  (this doesn't happen very often
  // hopefully)  We also set the pixellock history to 0 for all frames

  if (totlocks > totpixels) {
    T_PIXEL *ptmp = fctx->lastframe;

    fctx->lastframe = fctx->origframe;
    fctx->undo_data = fctx->lastframe;
    fctx->origframe = ptmp;
    fctx->undo = 1;

    memset (fctx->lockhistory, 0, fctx->hist_size);
  } else {
    fctx->undo_data = fctx->src_data;
    fctx->undo = 0;
  }

  return 0;
}

static void
gst_dnr_cleanup (T_DNR_FILTER_CTX * fctx)
{
  if (fctx->lastframe)
    g_free (fctx->lastframe);
  if (fctx->origframe)
    g_free (fctx->origframe);
  if (fctx->lockhistory)
    g_free (fctx->lockhistory);

  fctx->lastframe = NULL;
  fctx->origframe = NULL;
  fctx->lockhistory = NULL;

  g_free (fctx);
}

static T_DNR_FILTER_CTX *
gst_dnr_setup (int src_w, int src_h, int isYUV)
{
  double low1, low2;
  double high1, high2;
  int a, b, dif1, dif2;

  T_DNR_FILTER_CTX *fctx = g_malloc (sizeof (T_DNR_FILTER_CTX));

  //-- PARAMETERS --
  fctx->pThreshold = DEFAULT_LT;        // threshold to blend luma/red (default 10)
  fctx->pPixellock = DEFAULT_LL;        // threshold to lock luma/red (default 4)
  fctx->pThreshold2 = DEFAULT_CT;       // threshold to blend croma/green+blue (default 16)
  fctx->pPixellock2 = DEFAULT_CL;       // threshold to lock croma/green+blue (default 8)
  fctx->pScene = DEFAULT_SC;    // percentage of picture difference
  // to interpret as a new scene (default 30%)
  fctx->pPartial = 0;           // operating mode [0,1] (default 0)
  //----------------

  fctx->isYUV = isYUV;
  fctx->is_first_frame = 1;
  fctx->lastframe = (T_PIXEL *) g_malloc0 (src_h * src_w * 3);
  fctx->origframe = (T_PIXEL *) g_malloc0 (src_h * src_w * 3);
  fctx->lockhistory = (unsigned char *) g_malloc0 (src_h * src_w);
  fctx->src_h = src_h;
  fctx->src_w = src_w;
  fctx->hist_size = src_h * src_w;
  fctx->undo = 0;

  if (isYUV) {
    fctx->img_size = (fctx->hist_size * 3) / 2;
    fctx->gu_ofs = fctx->hist_size;
    fctx->rv_ofs = (fctx->hist_size * 5) / 4;
    fctx->pitch = 1;

    fctx->line_size_c = (src_w >> 1);
    fctx->line_size_l = src_w;
  } else {
    fctx->img_size = fctx->hist_size * 3;
    fctx->gu_ofs = 1;
    fctx->rv_ofs = 2;
    fctx->pitch = 3;

    fctx->line_size_c = src_w * 3;
    fctx->line_size_l = src_w * 3;
  }

  if (!fctx->lastframe || !fctx->origframe || !fctx->lockhistory) {
    gst_dnr_cleanup (fctx);
    return NULL;
  }
  // setup a biased thresholding difference matrix
  // this is an expensive operation we only want to to once
  for (a = 0; a < 256; a++) {
    for (b = 0; b < 256; b++) {
      // instead of scaling linearly
      // we scale according to the following formulas
      // val1 = 256 * (x / 256) ^ .9
      // and
      // val2 = 256 * (x / 256) ^ (1/.9)
      // and we choose the maximum distance between two points
      // based on these two scales
      low1 = a;
      low2 = b;
      low1 = low1 / 256;
      low1 = 256 * pow (low1, .9);
      low2 = low2 / 256;
      low2 = 256 * pow (low2, .9);

      // the low scale should make all values larger
      // and the high scale should make all values smaller
      high1 = a;
      high2 = b;
      high1 = high1 / 256;
      high2 = high2 / 256;
      high1 = 256 * pow (high1, 1.0 / .9);
      high2 = 256 * pow (high2, 1.0 / .9);
      dif1 = (int) (low1 - low2);
      if (dif1 < 0)
        dif1 *= -1;
      dif2 = (int) (high1 - high2);
      if (dif2 < 0)
        dif2 *= -1;
      dif1 = (dif1 > dif2) ? dif1 : dif2;
      fctx->lookup[a][b] = dif1;
    }
  }

  return fctx;
}

/* ---- end legacy code ---- */

static void
gst_dnr_setprops (GstDnr * filter)
{
  if (filter->fctx) {
    filter->fctx->pThreshold = filter->luma_blend;
    filter->fctx->pPixellock = filter->luma_lock;
    filter->fctx->pThreshold2 = filter->chroma_blend;
    filter->fctx->pPixellock2 = filter->chroma_lock;
    filter->fctx->pScene = filter->scene_change;
  }
}

static gboolean
gst_dnr_hook_caps (GstDnr * filter, GstCaps * incaps, GstCaps * outcaps)
{
  GstStructure *structure;

  structure = gst_caps_get_structure (incaps, 0);

  /* check the colorspace we are using */
  filter->is_rgb = gst_structure_has_name (structure, "video/x-raw-rgb");
  /* and re-init to allow for intermediate caps changes */
  filter->fctx = gst_dnr_setup (filter->width, filter->height, !filter->is_rgb);

  return TRUE;
}

static GstFlowReturn
gst_dnr_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  GstDnr *filter;
  guint8 *src;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_DNR (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);

  gst_dnr_run (filter->fctx, src);

  if (filter->fctx->undo)
    oil_memcpy (src, filter->fctx->undo_data, filter->fctx->img_size);

  return GST_FLOW_OK;
}

static gboolean
gst_dnr_start (GstBaseTransform * btrans)
{
  GstDnr *filter;

  filter = GST_DNR (btrans);

  filter->fctx = gst_dnr_setup (filter->width, filter->height, !filter->is_rgb);
  /* now we have a context, we can give it the settings */
  gst_dnr_setprops (filter);

  return TRUE;
}

static gboolean
gst_dnr_stop (GstBaseTransform * btrans)
{
  GstDnr *filter;

  filter = GST_DNR (btrans);

  gst_dnr_cleanup (filter->fctx);
  filter->fctx = NULL;

  return TRUE;
}

static void
gst_dnr_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstDnr *src;
  gboolean changed = FALSE;
  guint tmp;

  g_return_if_fail (GST_IS_DNR (object));
  src = GST_DNR (object);

  switch (prop_id) {
    case PROP_LUMA_BLEND:
      tmp = g_value_get_uint (value);
      if (G_UNLIKELY (tmp != src->luma_blend)) {
        src->luma_blend = tmp;
        changed = TRUE;
      }
      break;
    case PROP_LUMA_LOCK:
      tmp = g_value_get_uint (value);
      if (G_UNLIKELY (tmp != src->luma_lock)) {
        src->luma_lock = tmp;
        changed = TRUE;
      }
      break;
    case PROP_CHROMA_BLEND:
      tmp = g_value_get_uint (value);
      if (G_UNLIKELY (tmp != src->chroma_blend)) {
        src->chroma_blend = tmp;
        changed = TRUE;
      }
      break;
    case PROP_CHROMA_LOCK:
      tmp = g_value_get_uint (value);
      if (G_UNLIKELY (tmp != src->chroma_lock)) {
        src->chroma_lock = tmp;
        changed = TRUE;
      }
      break;
    case PROP_SCENE_CHANGE:
      tmp = g_value_get_uint (value);
      if (G_UNLIKELY (tmp != src->scene_change)) {
        src->scene_change = tmp;
        changed = TRUE;
      }
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  /* props might change mid-stream, you never know ... */
  if (changed)
    gst_dnr_setprops (src);
}

static void
gst_dnr_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstDnr *src;

  g_return_if_fail (GST_IS_DNR (object));
  src = GST_DNR (object);

  switch (prop_id) {
    case PROP_LUMA_BLEND:
      g_value_set_uint (value, src->luma_blend);
      break;
    case PROP_LUMA_LOCK:
      g_value_set_uint (value, src->luma_lock);
      break;
    case PROP_CHROMA_BLEND:
      g_value_set_uint (value, src->chroma_blend);
      break;
    case PROP_CHROMA_LOCK:
      g_value_set_uint (value, src->chroma_lock);
      break;
    case PROP_SCENE_CHANGE:
      g_value_set_uint (value, src->scene_change);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
