/* GStreamer Filter
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * transcode filter
 * Copyright (C) Thomas Oestreich - June 2001
 * *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-detectinter
 *
 * <refsect2>
 * <para>
 * This filter checks for interlaced video frames and marks these
 * by preceding them with a custom downstream event
 * (simply an empty structure called 'detectinter').
 * Subsequent (dynamic/adaptive) de-interlacing can then be done by
 * a downstream de-interlacing filter that recognizes this custom mark,
 * such as <link linkend="GstFields">fields</link>.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * transcode 32detect filter [Thomas Oestreich]
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <string.h>
#include <stdlib.h>

#define GST_TYPE_DETECT_INTER \
  (gst_detect_inter_get_type())
#define GST_DETECT_INTER(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_DETECT_INTER,GstDetectInter))
#define GST_DETECT_INTER_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_DETECT_INTER,GstDetectInterClass))
#define GST_IS_DETECT_INTER(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_DETECT_INTER))
#define GST_IS_DETECT_INTER_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_DETECT_INTER))


typedef struct _GstDetectInter GstDetectInter;
typedef struct _GstDetectInterClass GstDetectInterClass;

struct _GstDetectInter
{
  GstVideoFilter videofilter;

  gint width, height;
  gboolean is_rgb;

  /* properties */
  guint threshold, chroma_threshold;
  guint threshold_eq, chroma_eq;
  guint diff, chroma_diff;
};


struct _GstDetectInterClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (detect_inter_debug);
#define GST_CAT_DEFAULT detect_inter_debug


/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_THRESHOLD,
  PROP_CHROMA_THRESHOLD,
  PROP_THRESHOLD_EQ,
  PROP_CHROMA_EQ,
  PROP_DIFF,
  PROP_CHROMA_DIFF,
  PROP_REQ_DEINTER
      /* FILL ME */
};

#define DEFAULT_THRESHOLD             9
#define DEFAULT_CHROMA_THRESHOLD     DEFAULT_THRESHOLD / 2
#define DEFAULT_THRESHOLD_EQ         10
#define DEFAULT_CHROMA_EQ            DEFAULT_THRESHOLD_EQ / 2
#define DEFAULT_DIFF                 30
#define DEFAULT_CHROMA_DIFF          DEFAULT_DIFF / 2

static GstStaticPadTemplate gst_detect_inter_src_template =
    GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }")
        " ; " GST_VIDEO_CAPS_RGB " ; " GST_VIDEO_CAPS_BGR)
    );

static GstStaticPadTemplate gst_detect_inter_sink_template =
    GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }")
        " ; " GST_VIDEO_CAPS_RGB " ; " GST_VIDEO_CAPS_BGR)
    );

static gboolean gst_detect_inter_hook_caps (GstDetectInter * filter,
    GstCaps * incaps, GstCaps * outcaps);
static GstFlowReturn gst_detect_inter_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);
static gboolean gst_detect_inter_start (GstBaseTransform * btrans);
static gboolean gst_detect_inter_stop (GstBaseTransform * btrans);

static void gst_detect_inter_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_detect_inter_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstDetectInter, gst_detect_inter, GstVideoFilter,
    GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL (GstDetectInter, gst_detect_inter,
    gst_detect_inter_hook_caps);

static void
gst_detect_inter_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "DetectInter",
      "Filter/Effect/Video", "Detect interlaced frames",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Thomas Oestreich");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_detect_inter_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_detect_inter_src_template));
}

static void
gst_detect_inter_class_init (GstDetectInterClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (detect_inter_debug, "detect_inter", 0,
      "detect_inter");

  gobject_class->set_property = gst_detect_inter_set_property;
  gobject_class->get_property = gst_detect_inter_get_property;

  g_object_class_install_property (gobject_class, PROP_THRESHOLD,
      g_param_spec_uint ("threshold", "Threshold",
          "Interlace detection threshold",
          0, G_MAXUINT8, DEFAULT_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_CHROMA_THRESHOLD,
      g_param_spec_uint ("chroma-threshold", "Chroma Threshold",
          "Interlace detection chroma threshold",
          0, G_MAXUINT8, DEFAULT_CHROMA_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_THRESHOLD_EQ,
      g_param_spec_uint ("threshold-eq", "Threshold Equal",
          "Threshold for equal colors",
          0, G_MAXUINT8, DEFAULT_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_CHROMA_EQ,
      g_param_spec_uint ("chroma-eq", "Chroma Equal",
          "Threshold for equal chroma",
          0, G_MAXUINT8, DEFAULT_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_DIFF,
      g_param_spec_uint ("diff", "Difference",
          "Threshold for different colors",
          0, G_MAXUINT8, DEFAULT_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_CHROMA_DIFF,
      g_param_spec_uint ("chroma-diff", "Chroma Difference",
          "Threshold for different chroma",
          0, G_MAXUINT8, DEFAULT_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_detect_inter_set_caps);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_detect_inter_transform_ip);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_detect_inter_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_detect_inter_stop);
}

static void
gst_detect_inter_init (GstDetectInter * filter, GstDetectInterClass * g_class)
{
  GstBaseTransform *btrans;

  btrans = GST_BASE_TRANSFORM (filter);
  btrans->passthrough = TRUE;

  filter->threshold = DEFAULT_THRESHOLD;
  filter->chroma_threshold = DEFAULT_CHROMA_THRESHOLD;
  filter->threshold_eq = DEFAULT_THRESHOLD_EQ;
  filter->chroma_eq = DEFAULT_CHROMA_EQ;
  filter->diff = DEFAULT_DIFF;
  filter->chroma_diff = DEFAULT_CHROMA_DIFF;
}

static gboolean
gst_detect_inter_hook_caps (GstDetectInter * filter, GstCaps * incaps,
    GstCaps * outcaps)
{
  GstStructure *structure;

  structure = gst_caps_get_structure (incaps, 0);

  /* check the colorspace we are using */
  filter->is_rgb = gst_structure_has_name (structure, "video/x-raw-rgb");

  return TRUE;
}

static gboolean
gst_interlace_test (GstDetectInter * filter, guint8 * video_buf, gint width,
    gint height, gint thres, gint eq, gint diff)
{

  gint j, n, off, block, cc_1, cc_2, cc;
  guint16 s1, s2, s3, s4;
  gboolean flag;

  cc_1 = 0;
  cc_2 = 0;

  block = width;

  flag = FALSE;

  for (j = 0; j < block; ++j) {

    off = 0;

    for (n = 0; n < (height - 4); n += 2) {
      s1 = (video_buf[off + j] & 0xff);
      s2 = (video_buf[off + j + block] & 0xff);
      s3 = (video_buf[off + j + 2 * block] & 0xff);
      s4 = (video_buf[off + j + 3 * block] & 0xff);

      if ((abs (s1 - s3) < eq) && (abs (s1 - s2) > diff))
        ++cc_1;

      if ((abs (s2 - s4) < eq) && (abs (s2 - s3) > diff))
        ++cc_2;

      off += 2 * block;
    }
  }

  /* compare results */
  cc = (int) ((cc_1 + cc_2) * 1000.0 / (width * height));
  flag = (cc > thres) ? 1 : 0;

  GST_INFO_OBJECT (filter, "frame: "
      "(1) = %5d | (2) = %5d | (3) = %3d | interlaced = %s\n", cc_1, cc_2, cc,
      ((flag) ? "yes" : "no"));

  return flag;
}


static GstFlowReturn
gst_detect_inter_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  GstDetectInter *filter;
  guint8 *src, *cubuf, *cvbuf;
  guint width, height, stride;
  gboolean interlaced;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_DETECT_INTER (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);

  width = filter->width;
  height = filter->height;

  stride = GST_VIDEO_I420_Y_ROWSTRIDE (width);

  cubuf = src + GST_VIDEO_I420_U_OFFSET (width, height);
  cvbuf = src + GST_VIDEO_I420_V_OFFSET (width, height);

  if (filter->is_rgb) {
    interlaced = gst_interlace_test (filter, src, 3 * width, height,
        filter->threshold, filter->threshold_eq, filter->diff);
  } else {
    interlaced = gst_interlace_test (filter, src, stride, height,
        filter->threshold, filter->threshold_eq, filter->diff);
    interlaced |= gst_interlace_test (filter, cubuf,
        GST_VIDEO_I420_U_ROWSTRIDE (width), height / 2,
        filter->chroma_threshold, filter->chroma_eq, filter->chroma_diff);
    interlaced |= gst_interlace_test (filter, cvbuf,
        GST_VIDEO_I420_V_ROWSTRIDE (width), height / 2,
        filter->chroma_threshold, filter->chroma_eq, filter->chroma_diff);
  }

  if (interlaced) {
    /* signal this by sending a custom serialized, down-stream event */
    gst_pad_push_event (btrans->srcpad,
        gst_event_new_custom (GST_EVENT_CUSTOM_DOWNSTREAM,
            gst_structure_empty_new ("detectinter")));
  }

  return GST_FLOW_OK;
}


static gboolean
gst_detect_inter_start (GstBaseTransform * btrans)
{
  return TRUE;
}

static gboolean
gst_detect_inter_stop (GstBaseTransform * btrans)
{
  return TRUE;
}

static void
gst_detect_inter_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstDetectInter *src;

  g_return_if_fail (GST_IS_DETECT_INTER (object));
  src = GST_DETECT_INTER (object);

  switch (prop_id) {
    case PROP_THRESHOLD:
      src->threshold = g_value_get_uint (value);
      break;
    case PROP_CHROMA_THRESHOLD:
      src->chroma_threshold = g_value_get_uint (value);
      break;
    case PROP_THRESHOLD_EQ:
      src->threshold_eq = g_value_get_uint (value);
      break;
    case PROP_CHROMA_EQ:
      src->chroma_eq = g_value_get_uint (value);
      break;
    case PROP_DIFF:
      src->diff = g_value_get_uint (value);
      break;
    case PROP_CHROMA_DIFF:
      src->chroma_diff = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_detect_inter_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstDetectInter *src;

  g_return_if_fail (GST_IS_DETECT_INTER (object));
  src = GST_DETECT_INTER (object);

  switch (prop_id) {
    case PROP_THRESHOLD:
      g_value_set_uint (value, src->threshold);
      break;
    case PROP_CHROMA_THRESHOLD:
      g_value_set_uint (value, src->chroma_threshold);
      break;
    case PROP_THRESHOLD_EQ:
      g_value_set_uint (value, src->threshold_eq);
      break;
    case PROP_CHROMA_EQ:
      g_value_set_uint (value, src->chroma_eq);
      break;
    case PROP_DIFF:
      g_value_set_uint (value, src->diff);
      break;
    case PROP_CHROMA_DIFF:
      g_value_set_uint (value, src->chroma_diff);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
