/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-stamp
 *
 * <refsect2>
 * <para>
 * Re-sequence timestamps on buffers and optionally provide progress info.
 * </para>
 * <para>
 * The <link linkend="GstIdentity">identity</link> element has a
 * similar <link linkend="GstIdentity--single-segment">single-segment</link>
 * functionality which timestamps buffers according to
 * the running time, which should typically then result into a perfect
 * stream of sequential timestamps.  It can do this on the basis
 * of segment information or based on bytes per second given by a
 * <link linkend="GstIdentity--datarate">datarate</link>, provided to
 * it externally.
 * However, this datarate is typically available within the stream itself,
 * by means of caps (e.g. framerate, samplerate), which is extracted by
 * this element and automagically used to perform (re)timestamping.
 * Again, the result is a single-segment <quote>time-perfect</quote>
 * stream.  Since the re-sequencing is performed based on datarate,
 * a/v synchronization might be affected if incoming timestamps are irregular
 * (with respect to datarate).  Use e.g. <link linkend="GstVideoRate">videorate</link>
 * to aid with that.
 * </para>
 * <para>
 * The <link linkend="GstStamp--silent">silent</link> and
 * <link linkend="GstStamp--progress">progress</link> properties allow
 * also using this element to obtain some intermittent progress
 * updates (re-assuring the pipeline is still carrying on).
 * </para>
 * <para>
 * If <link linkend="GstStamp--sync-margin">sync-margin</link> is non-zero
 * when part of a video stream, then it performs synchronization w.r.t. clock
 * (typically provided by audio src or sink).  Specifically,
 * buffers passing through will either be duplicated or dropped
 * as necessary to ensure that the (calculated) timestamp does not
 * deviate more than <link linkend="GstStamp--sync-margin">sync-margin</link>
 * frames.  A check for this is performed either each time or every
 * <link linkend="GstStamp--sync-interval">sync-interval</link> frames
 * (if non-zero).
 *
 * The properties <link linkend="GstStamp--duplicate">duplicate</link> and
 * <link linkend="GstStamp--drop">drop</link> can be read to obtain
 * information about number of dropped frames
 * (i.e. the number of unused input frames) and duplicated frames (i.e. the
 *  number of times an input frame was duplicated, beside being used normally).
 *
 * An input stream that needs no adjustments will thus never have dropped or
 * duplicated frames.
 *
 * When the <link linkend="GstStamp--silent">silent</link> property is FALSE,
 * a GObject property notification
 * will be emitted whenever one of the "duplicate" or "drop" values changes.
 * This can potentially cause performance degradation.
 * Note that property notification will happen from the streaming thread, so
 * applications should be prepared for this.
 *
 * If <link linkend="GstStamp--allow-segments">allow-segments</link> is FALSE,
 * <emphasis>all</emphasis> segment updates will be discarded.  Otherwise,
 * some moderately intelligent decision will allow crucial ones to pass
 * (e.g. time synchronization for subtitle streams).
 * </para>
 * <para>
 * It can be noted that the "sync" functionality is somewhat related to
 * <link linkend="GstVideoRate">videorate</link> element and should probably
 * be merged into the latter element at some point.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * sync functionality based on transcode's import_v4l2 [Erik Slagter]
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-entrans.h"

#define GST_TYPE_STAMP \
  (gst_stamp_get_type())
#define GST_STAMP(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_STAMP,GstStamp))
#define GST_STAMP_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_STAMP,GstStampClass))
#define GST_IS_STAMP(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_STAMP))
#define GST_IS_STAMP_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_STAMP))

typedef struct _GstStamp GstStamp;
typedef struct _GstStampClass GstStampClass;

struct _GstStamp
{
  GstBaseTransform basetransform;

  /* total time seen so far, dropped so far */

  /* properties */
  /* status reporting */
  gboolean silent;
  guint progress;
  gchar *last_message;
  GstClockTime last_stamp;
  /* sync */
  guint64 drop, duplicate;
  guint sync_interval, sync_margin;
  gboolean allow_segments;
  gboolean invalidate;

  /* info for cutting */
  guint type;
  /* if audio */
  gint rate, sample;
  /* if video */
  gint fps_num, fps_denom;

  /* buffers/bytes seen so far */
  guint64 count;
};

enum
{
  TYPE_UNKNOWN,
  TYPE_VIDEO,
  TYPE_AUDIO,
  TYPE_SUB,
  TYPE_OTHER
};

struct _GstStampClass
{
  GstBaseTransformClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (stamp_debug);
#define GST_CAT_DEFAULT stamp_debug

/* GstStamp signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_PROGRESS,
  PROP_SILENT,
  PROP_LAST_MESSAGE,
  PROP_SYNC_MARGIN,
  PROP_SYNC_INTERVAL,
  PROP_DROP,
  PROP_DUP,
  PROP_ALLOW_SEGMENTS,
  PROP_INVALIDATE
      /* FILL ME */
};

#define DEFAULT_PROGRESS      2 * 1000000
#define MAX_PROGRESS            G_MAXUINT
#define DEFAULT_SILENT               TRUE
#define DEFAULT_SYNC_MARGIN             0
#define DEFAULT_SYNC_INTERVAL          25
#define DEFAULT_ALLOW_SEGMENTS       TRUE
#define DEFAULT_INVALIDATE          FALSE

static GstStaticPadTemplate gst_stamp_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

static GstStaticPadTemplate gst_stamp_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

static gboolean gst_stamp_start (GstBaseTransform * trans);
static gboolean gst_stamp_stop (GstBaseTransform * trans);
static gboolean gst_stamp_event (GstBaseTransform * btrans, GstEvent * event);
static gboolean gst_stamp_setcaps (GstBaseTransform * btrans, GstCaps * caps,
    GstCaps * outcaps);
static GstFlowReturn gst_stamp_transform_ip (GstBaseTransform * btrans,
    GstBuffer * buf);

static void gst_stamp_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_stamp_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstStamp, gst_stamp, GstBaseTransform,
    GST_TYPE_BASE_TRANSFORM);

static void
gst_stamp_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Stamp",
      "Generic", "(Time)stamp buffers, thus re-sequencing",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_stamp_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_stamp_src_template));
}

static void
gst_stamp_class_init (GstStampClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (stamp_debug, "stamp", 0, "stamp");

  gobject_class->set_property = gst_stamp_set_property;
  gobject_class->get_property = gst_stamp_get_property;

  g_object_class_install_property (gobject_class, PROP_PROGRESS,
      g_param_spec_uint ("progress", "Report Interval",
          "Microseconds between progress status info",
          0, MAX_PROGRESS, DEFAULT_PROGRESS, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_SILENT,
      g_param_spec_boolean ("silent", "silent", "silent",
          DEFAULT_SILENT, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_LAST_MESSAGE,
      g_param_spec_string ("last-message", "last-message", "last-message",
          NULL, G_PARAM_READABLE));

  g_object_class_install_property (gobject_class, PROP_SYNC_MARGIN,
      g_param_spec_uint ("sync-margin", "Sync Margin",
          "Margin beyond which to enforce sync [0 = disabled]", 0, G_MAXUINT,
          DEFAULT_SYNC_MARGIN, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_SYNC_INTERVAL,
      g_param_spec_uint ("sync-interval", "Sync Interval",
          "Interval to check sync", 0, G_MAXUINT,
          DEFAULT_SYNC_INTERVAL, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_DUP,
      g_param_spec_uint64 ("duplicate", "Duplicate",
          "Number of duplicated frames", 0, G_MAXUINT64, 0, G_PARAM_READABLE));

  g_object_class_install_property (gobject_class, PROP_DROP,
      g_param_spec_uint64 ("drop", "Drop",
          "Number of dropped frames", 0, G_MAXUINT64, 0, G_PARAM_READABLE));

  g_object_class_install_property (gobject_class, PROP_ALLOW_SEGMENTS,
      g_param_spec_boolean ("allow-segments", "Allow Segments",
          "Allow (intelligent) passing of updating TIME segments"
          " (e.g. for stream time synchronization)",
          DEFAULT_ALLOW_SEGMENTS, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_INVALIDATE,
      g_param_spec_boolean ("invalidate", "Invalidate",
          "Invalidate buffer time and duration",
          DEFAULT_INVALIDATE, G_PARAM_READWRITE));

  /* not setting passthrough assures (metadata) writable buffers */
  trans_class->start = GST_DEBUG_FUNCPTR (gst_stamp_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_stamp_stop);
  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_stamp_setcaps);
  trans_class->event = GST_DEBUG_FUNCPTR (gst_stamp_event);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_stamp_transform_ip);
}

static void
gst_stamp_init (GstStamp * stamp, GstStampClass * g_class)
{
  stamp->progress = DEFAULT_PROGRESS;
  stamp->silent = DEFAULT_SILENT;
  stamp->sync_margin = DEFAULT_SYNC_MARGIN;
  stamp->sync_interval = DEFAULT_SYNC_INTERVAL;
  stamp->type = TYPE_UNKNOWN;
  stamp->allow_segments = DEFAULT_ALLOW_SEGMENTS;
  stamp->invalidate = DEFAULT_INVALIDATE;
}

static gboolean
gst_stamp_setcaps (GstBaseTransform * btrans, GstCaps * caps, GstCaps * ocaps)
{
  GstStamp *filter = GST_STAMP (btrans);
  GstStructure *structure;
  const gchar *mime;
  gboolean ret = TRUE;

  structure = gst_caps_get_structure (caps, 0);
  mime = gst_structure_get_name (structure);

  if (g_strrstr (mime, "text/") || g_strrstr (mime, "subpicture")) {
    filter->type = TYPE_SUB;
  } else if (g_strrstr (mime, "audio/")) {
    gint width, channels;

    filter->type = TYPE_AUDIO;
    ret &= gst_structure_get_int (structure, "rate", &filter->rate);
    ret &= gst_structure_get_int (structure, "width", &width);
    ret &= gst_structure_get_int (structure, "channels", &channels);
    filter->sample = channels * width / 8;
  } else if (g_strrstr (mime, "video/")) {
    const GValue *fps;

    filter->type = TYPE_VIDEO;
    fps = gst_structure_get_value (structure, "framerate");
    if (fps) {
      g_return_val_if_fail (GST_VALUE_HOLDS_FRACTION (fps), FALSE);
      filter->fps_num = gst_value_get_fraction_numerator (fps);
      filter->fps_denom = gst_value_get_fraction_denominator (fps);
      /* can happen if there's no real framerate */
      if (!filter->fps_num)
        ret = FALSE;
    } else {
      filter->type = TYPE_OTHER;
      ret = TRUE;
    }
  }

  GST_DEBUG_OBJECT (filter, "received caps, determined filter type %d",
      filter->type);

  return ret;
}

static GstEvent *
translate_outgoing_new_segment (GstBaseTransform * object, GstEvent * event)
{
  GstEvent *event2 = NULL;
  gboolean update;
  gdouble rate;
  GstFormat format;
  gint64 start, stop, stream;
  guint64 nstream, nstart;

  /* only modify the start and streamtime */
  gst_event_parse_new_segment (event, &update, &rate, &format,
      &start, &stop, &stream);

  GST_DEBUG_OBJECT (object,
      "Got NEWSEGMENT %" GST_TIME_FORMAT " -- %" GST_TIME_FORMAT " // %"
      GST_TIME_FORMAT, GST_TIME_ARGS (start), GST_TIME_ARGS (stop),
      GST_TIME_ARGS (stream));

  if (format != GST_FORMAT_TIME) {
    GST_WARNING_OBJECT (object,
        "Can't translate newsegments with format != GST_FORMAT_TIME");
    return event;
  }

  /* in our setting, stream = media-time is not really expected to be used
     anywhere, and basically provided equal to start */
  nstream = gst_segment_to_running_time (&object->segment, GST_FORMAT_TIME,
      stream);
  nstart = gst_segment_to_running_time (&object->segment, GST_FORMAT_TIME,
      start);
  if (!GST_CLOCK_TIME_IS_VALID (nstream) || !GST_CLOCK_TIME_IS_VALID (nstart)) {
    GST_DEBUG_OBJECT (object, "Ignoring newsegment");
    goto exit;
  }

  if (nstream > G_MAXINT64 || nstart > G_MAXINT64)
    GST_WARNING_OBJECT (object, "Return value too big...");

  GST_DEBUG_OBJECT (object,
      "Sending NEWSEGMENT %" GST_TIME_FORMAT " -- %" GST_TIME_FORMAT " // %"
      GST_TIME_FORMAT, GST_TIME_ARGS (nstart), GST_TIME_ARGS (-1),
      GST_TIME_ARGS (nstream));
  event2 =
      gst_event_new_new_segment (update, rate, format, nstart, -1,
      (gint64) nstream);
exit:
  gst_event_unref (event);

  return event2;
}

static gboolean
gst_stamp_event (GstBaseTransform * btrans, GstEvent * event)
{
  GstStamp *filter = GST_STAMP (btrans);
  gboolean ret = TRUE;
  GstFormat format;
  gboolean update = FALSE;

  /* as in identity element */
  if (GST_EVENT_TYPE (event) == GST_EVENT_NEWSEGMENT) {
    gst_event_parse_new_segment (event, &update, NULL, &format, NULL, NULL,
        NULL);

    if (btrans->have_newsegment == FALSE) {
      GstEvent *news;

      /* This is the first newsegment, send out a (0, -1) newsegment */
      news = gst_event_new_new_segment (FALSE, 1.0, format, 0, -1, 0);

      if (!(gst_pad_event_default (btrans->sinkpad, news)))
        return FALSE;
    }
  }

  /* prevent segment duration calculation due to updates,
   * otherwise intermediate updates following by a segment closing update
   * will have duration counted double */
  if (!update)
    GST_BASE_TRANSFORM_CLASS (parent_class)->event (btrans, event);

  if (GST_EVENT_TYPE (event) == GST_EVENT_NEWSEGMENT) {
    /* eat up segments ... */
    ret = FALSE;
    /* ... except for other (e.g. subtitle) segment updates */
    if (filter->allow_segments) {
      GstEvent *news;
      gboolean update;
      GstFormat format;

      gst_event_parse_new_segment (event, &update, NULL, &format, NULL, NULL,
          NULL);
      if (format == GST_FORMAT_TIME && (filter->type >= TYPE_SUB || update)) {
        if ((news = translate_outgoing_new_segment (btrans, event)))
          gst_pad_event_default (btrans->sinkpad, news);
      }
    }
  }

  return ret;
}

static GstFlowReturn
gst_stamp_transform_ip (GstBaseTransform * btrans, GstBuffer * buf)
{
  GstStamp *filter;
  GstFlowReturn res = GST_FLOW_OK;

  filter = GST_STAMP (btrans);

  GST_LOG_OBJECT (filter,
      "Received buffer of time %" GST_TIME_FORMAT ", duration %" GST_TIME_FORMAT
      ", size %d", GST_TIME_ARGS (GST_BUFFER_TIMESTAMP (buf)),
      GST_TIME_ARGS (GST_BUFFER_DURATION (buf)), GST_BUFFER_SIZE (buf));

  if (filter->invalidate) {
    GST_BUFFER_TIMESTAMP (buf) = GST_CLOCK_TIME_NONE;
    GST_BUFFER_DURATION (buf) = GST_CLOCK_TIME_NONE;
    goto exit;
  }

  /* timestamp */
  switch (filter->type) {
    case TYPE_VIDEO:
      if (G_UNLIKELY (filter->sync_margin)) {
        if (filter->count
            && (!filter->sync_interval
                || filter->count % filter->sync_interval == 0)) {
          guint64 buffer_count;

          buffer_count = gst_util_uint64_scale (GST_BUFFER_TIMESTAMP (buf),
              filter->fps_num, filter->fps_denom * GST_SECOND);
          GST_DEBUG_OBJECT (filter, "buffer %" G_GUINT64_FORMAT
              " video %" G_GUINT64_FORMAT, buffer_count, filter->count);
          if (filter->count > buffer_count + filter->sync_margin) {
            filter->drop++;
            if (!filter->silent)
              g_object_notify (G_OBJECT (filter), "drop");
            res = GST_BASE_TRANSFORM_FLOW_DROPPED;
            break;
          } else if (buffer_count > filter->count + filter->sync_margin) {
            GstBuffer *dupbuf;

            filter->duplicate++;
            if (!filter->silent)
              g_object_notify (G_OBJECT (filter), "duplicate");
            dupbuf = gst_buffer_make_metadata_writable (gst_buffer_ref (buf));
            GST_BUFFER_TIMESTAMP (dupbuf) =
                gst_util_uint64_scale (filter->count,
                filter->fps_denom * GST_SECOND, filter->fps_num);
            GST_BUFFER_DURATION (dupbuf) =
                gst_util_uint64_scale (1, filter->fps_denom * GST_SECOND,
                filter->fps_num);
            filter->count++;
            res = gst_pad_push (btrans->srcpad, dupbuf);
            GST_LOG_OBJECT (filter, "pushed duplicate");
          }
        }
      }
      GST_BUFFER_TIMESTAMP (buf) = gst_util_uint64_scale (filter->count,
          filter->fps_denom * GST_SECOND, filter->fps_num);
      GST_BUFFER_DURATION (buf) = gst_util_uint64_scale (1,
          filter->fps_denom * GST_SECOND, filter->fps_num);
      filter->count++;
      break;
    case TYPE_AUDIO:
      GST_BUFFER_TIMESTAMP (buf) =
          gst_util_uint64_scale (filter->count / filter->sample, GST_SECOND,
          filter->rate);
      GST_BUFFER_DURATION (buf) =
          gst_util_uint64_scale (GST_BUFFER_SIZE (buf) / filter->sample,
          GST_SECOND, filter->rate);
      filter->count += GST_BUFFER_SIZE (buf);
      break;
    default:
      if (btrans->segment.format == GST_FORMAT_TIME &&
          GST_BUFFER_TIMESTAMP_IS_VALID (buf)) {
        /* special semantics for subtitle buffers */
        if (filter->type == TYPE_SUB) {
          /* clip at start */
          if (GST_BUFFER_TIMESTAMP (buf) < btrans->segment.start) {
            GST_LOG_OBJECT (filter, "clipping sub start");
            GST_BUFFER_DURATION (buf)
                = GST_BUFFER_TIMESTAMP (buf) - btrans->segment.start;
            GST_BUFFER_TIMESTAMP (buf) = btrans->segment.start;
          }
          /* clip at end */
          if (GST_CLOCK_TIME_IS_VALID (btrans->segment.stop) &&
              GST_BUFFER_DURATION_IS_VALID (buf) &&
              GST_BUFFER_TIMESTAMP (buf) + GST_BUFFER_DURATION (buf) >=
              btrans->segment.stop) {
            GST_LOG_OBJECT (filter, "clipping sub end");
            GST_BUFFER_DURATION (buf) =
                btrans->segment.stop - GST_BUFFER_TIMESTAMP (buf);
          }
        }
        GST_BUFFER_TIMESTAMP (buf) =
            gst_segment_to_running_time (&btrans->segment, GST_FORMAT_TIME,
            GST_BUFFER_TIMESTAMP (buf));
      }
      filter->count++;
      break;
  }

  /* update stats */
  if (!filter->silent && filter->progress && GST_BUFFER_TIMESTAMP_IS_VALID (buf)
      && GST_BUFFER_TIMESTAMP (buf) - filter->last_stamp >=
      filter->progress * GST_USECOND) {
    GST_OBJECT_LOCK (filter);
    g_free (filter->last_message);
    filter->last_message =
        g_strdup_printf ("timestamp %" GST_TIME_FORMAT ", count so far: %"
        G_GUINT64_FORMAT, GST_TIME_ARGS (GST_BUFFER_TIMESTAMP (buf)),
        filter->count);
    GST_OBJECT_UNLOCK (filter);
    g_object_notify (G_OBJECT (filter), "last-message");
    filter->last_stamp = GST_BUFFER_TIMESTAMP (buf);
  }

exit:
  return res;
}

static gboolean
gst_stamp_start (GstBaseTransform * trans)
{
  GstStamp *stamp = GST_STAMP (trans);

  stamp->count = 0;
  stamp->duplicate = 0;
  stamp->drop = 0;
  stamp->type = TYPE_UNKNOWN;
  stamp->rate = 0;
  stamp->sample = 0;
  stamp->fps_num = stamp->fps_denom = -1;
  stamp->last_stamp = 0;

  return TRUE;
}

static gboolean
gst_stamp_stop (GstBaseTransform * trans)
{
  GstStamp *stamp = GST_STAMP (trans);

  g_free (stamp->last_message);
  stamp->last_message = NULL;

  return TRUE;
}

static void
gst_stamp_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstStamp *stamp;

  g_return_if_fail (GST_IS_STAMP (object));
  stamp = GST_STAMP (object);

  switch (prop_id) {
    case PROP_PROGRESS:
      stamp->progress = g_value_get_uint (value);
      break;
    case PROP_SILENT:
      stamp->silent = g_value_get_boolean (value);
      break;
    case PROP_SYNC_MARGIN:
      stamp->sync_margin = g_value_get_uint (value);
      break;
    case PROP_SYNC_INTERVAL:
      stamp->sync_interval = g_value_get_uint (value);
      break;
    case PROP_ALLOW_SEGMENTS:
      stamp->allow_segments = g_value_get_boolean (value);
      break;
    case PROP_INVALIDATE:
      stamp->invalidate = g_value_get_boolean (value);
      break;
    default:
      break;
  }
}

static void
gst_stamp_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstStamp *stamp;

  g_return_if_fail (GST_IS_STAMP (object));
  stamp = GST_STAMP (object);

  switch (prop_id) {
    case PROP_PROGRESS:
      g_value_set_uint (value, stamp->progress);
      break;
    case PROP_DUP:
      g_value_set_uint64 (value, stamp->duplicate);
      break;
    case PROP_DROP:
      g_value_set_uint64 (value, stamp->drop);
      break;
    case PROP_SYNC_MARGIN:
      g_value_set_uint (value, stamp->sync_margin);
      break;
    case PROP_SYNC_INTERVAL:
      g_value_set_uint (value, stamp->sync_interval);
      break;
    case PROP_SILENT:
      g_value_set_boolean (value, stamp->silent);
      break;
    case PROP_LAST_MESSAGE:
      GST_OBJECT_LOCK (stamp);
      g_value_set_string (value, stamp->last_message);
      GST_OBJECT_UNLOCK (stamp);
      break;
    case PROP_ALLOW_SEGMENTS:
      g_value_set_boolean (value, stamp->allow_segments);
      break;
    case PROP_INVALIDATE:
      g_value_set_boolean (value, stamp->invalidate);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
