//*****************************************************************************
//                               NbkNgSpice.cpp                               *
//                              ----------------                              *
//  Started     : 08/05/2004                                                  *
//  Last Update : 13/10/2009                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/NbkNgSpice.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( NbkNgSpice, wxNotebook )

  EVT_BUTTON( PnlAnaBase::ID_BTN_OPTIONS, NbkNgSpice::OnBtnOptions )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Arguments :
//   poParent - The parent window
//   oWinID   - The display object ID
//   roPosn   - The notebook position
//   roSize   - The notebook size

NbkNgSpice::NbkNgSpice( wxWindow * poParent, wxWindowID oWinID ) :
                      NbkSimrBase( poParent, oWinID ), m_oDlgCfgOPT( poParent )
{
  // Set the simulator engine type specifier
  m_eSimEng = eSIMR_NGSPICE;

  // Create the various display objects
  m_poPnlNgSpiceDC = new PnlNgSpiceDC( this );
  m_poPnlNgSpiceAC = new PnlNgSpiceAC( this );
  m_poPnlNgSpiceTR = new PnlNgSpiceTR( this );

  // Add the display objects to the note book
  AddPage( m_poPnlNgSpiceDC, wxT( "     DC     " ) );
  AddPage( m_poPnlNgSpiceAC, wxT( "     AC     " ) );
  AddPage( m_poPnlNgSpiceTR, wxT( " Transient  " ) );

  // Specify the default page to be displayed
  SetSelection( 0 );
}

//*****************************************************************************
// Default constructor.
// (Used for two stage creation ie. must call Create( ) method.

NbkNgSpice::NbkNgSpice( void ) : NbkSimrBase( ), m_oDlgCfgOPT( this )
{
  // Set the simulator engine type specifier
  m_eSimEng = eSIMR_NGSPICE;
}

//*****************************************************************************
// Destructor.

NbkNgSpice::~NbkNgSpice( )
{
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NbkNgSpice::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear the base class
  if( ! NbkSimrBase     ::bClear( ) ) bRtn = FALSE;

  if( ! m_poPnlNgSpiceDC->bClear( ) ) bRtn = FALSE;
  if( ! m_poPnlNgSpiceAC->bClear( ) ) bRtn = FALSE;
  if( ! m_poPnlNgSpiceTR->bClear( ) ) bRtn = FALSE;

  if( ! m_oDlgCfgOPT     .bClear( ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List :
//   roSimn - The simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NbkNgSpice::bLoad( const SimnBase & roSimn )
{
  bool  bRtn=TRUE;

  if( roSimn.eGetSimEng( ) != eSIMR_NGSPICE )            return( FALSE );

  SimnNgSpice & roSimnNgs = (SimnNgSpice &) roSimn;

  if( ! bSetPage( roSimnNgs.eGetAnaType( ) ) )           bRtn = FALSE;

  if( ! m_poPnlNgSpiceDC->bLoad( roSimnNgs ) )           bRtn = FALSE;
  if( ! m_poPnlNgSpiceAC->bLoad( roSimnNgs ) )           bRtn = FALSE;
  if( ! m_poPnlNgSpiceTR->bLoad( roSimnNgs ) )           bRtn = FALSE;

  if( ! m_oDlgCfgOPT.bSetValues( roSimnNgs.m_oCmdOPT ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Save information to a simulation object.
//
// Argument List :
//   roSimultn - The simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NbkNgSpice::bSave( SimnBase & roSimn )
{
  wxString  os1;

  // Only proceed if this is an NG-Spice simulation object
  if( roSimn.eGetSimEng( ) != eSIMR_NGSPICE ) return( FALSE );

  SimnNgSpice & roSimnNgs = (SimnNgSpice &) roSimn;

  // Clear the error string
  NbkSimrBase::m_osErrMsg.Empty( );

  // Create the analysis command/s
  switch( GetSelection( ) )
  {
    case 0 : // DC analysis
      if( ! m_poPnlNgSpiceDC->bSave( roSimnNgs ) )
        SetErrMsg( m_poPnlNgSpiceDC->rosGetErrMsg( ) );
      break;

    case 1 : // AC analysis
      if( ! m_poPnlNgSpiceAC->bSave( roSimnNgs ) )
        SetErrMsg( m_poPnlNgSpiceAC->rosGetErrMsg( ) );
      break;

    case 2 : // TR analysis
      if( ! m_poPnlNgSpiceTR->bSave( roSimnNgs ) )
        SetErrMsg( m_poPnlNgSpiceTR->rosGetErrMsg( ) );
      break;

    default:                                  return( FALSE );
  }
  if( ! bIsOk( ) )                            return( FALSE );

  // Create the OPTIONS command
  os1 = roSimnNgs.m_oCmdOPT.m_osTEMP;
  m_oDlgCfgOPT.bGetValues( roSimnNgs.m_oCmdOPT );
  roSimnNgs.m_oCmdOPT.m_osTEMP = os1;
  roSimnNgs.m_oCmdOPT.bSetWidth( roSimnNgs.m_oCmdPR.iGetParaCnt( ) );
  roSimnNgs.m_oCmdOPT.bFormat( );
  if( ! roSimnNgs.m_oCmdOPT.bIsValid( ) )
  {
    os1 = wxT("OPTIONS command fault :\n\n  ");
    SetErrMsg( os1 + roSimnNgs.m_oCmdOPT.rosGetErrMsg( ) );
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Set the page to be displayed.
//
// Argument List :
//   eAnalysis - The enumerated analysis type specifier
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NbkNgSpice::bSetPage( eCmdType eAnalysis )
{
  int  iPage;

  switch( eAnalysis )
  {
    case eCMD_NONE :
    case eCMD_OP   :
    case eCMD_DC   : iPage = 0; break;
    case eCMD_AC   : iPage = 1; break;
    case eCMD_TR   : iPage = 2; break;
    default        : return( FALSE );
  }

  SetSelection( iPage ); // Specify the page to be displayed

  return( TRUE );
}

//*****************************************************************************
// Set the page to be displayed.
//
// Argument List :
//   psAnalysis - The two letter analysis type specifier (case ignored)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NbkNgSpice::bSetPage( const wxChar * psAnalysis )
{
  eCmdType  eAnalysis=eCMD_NONE;
  wxString  osAnalysis;

  osAnalysis = wxString( psAnalysis ).Upper( );

  if( osAnalysis.Length( ) == 2 )
  {
    if(      osAnalysis == wxT("OP") ) eAnalysis = eCMD_DC;
    else if( osAnalysis == wxT("DC") ) eAnalysis = eCMD_DC;
    else if( osAnalysis == wxT("AC") ) eAnalysis = eCMD_AC;
    else if( osAnalysis == wxT("TR") ) eAnalysis = eCMD_TR;
  }

  return( bSetPage( eAnalysis ) );
}

//*****************************************************************************
// Get the two letter page specifier.
//
// Return Values :
//   Success - The two letter analysis type specifier (lower case)
//   Failure - An empty string

const wxString & NbkNgSpice::rosGetPage( void )
{
  static  wxString  osPage;

  switch( eGetPage( ) )
  {
    case eCMD_DC : osPage = wxT("dc"); break;
    case eCMD_AC : osPage = wxT("ac"); break;
    case eCMD_TR : osPage = wxT("tr"); break;
    default      : osPage.Empty( );
  }

  return( osPage );
}

//*****************************************************************************
// Get the enumerated page specifier.
//
// Return Values :
//   Success - The enumerated analysis specifier
//   Failure - Simulation::eANA_NONE

eCmdType  NbkNgSpice::eGetPage( void )
{
  eCmdType  ePage;

  switch( GetSelection( ) )
  {
    case  0 : ePage = eCMD_DC; break;
    case  1 : ePage = eCMD_AC; break;
    case  2 : ePage = eCMD_TR; break;
    default : ePage = eCMD_NONE;
  }

  return( ePage );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// .OPTIONS command setup button control event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  NbkNgSpice::OnBtnOptions( wxCommandEvent & roEvtCmd )
{
  PnlAnaBase * poPnlAna;
  wxString     os1;
  int          i1;

  // Set the temperature in the OPTIONS dialog
  poPnlAna = (PnlAnaBase *) GetPage( GetSelection( ) );
  if( poPnlAna->bIsCreated( PnlAnaBase::eCTLGRP_TEMP ) )
  {
    os1 = poPnlAna->m_oPnlTemp.rosGetValue( );
    m_oDlgCfgOPT.bSetValue( DlgNgsCfgOPT::ID_PNL_TEMP, os1 );
  }

  // Display the OPTIONS dialog
  m_oDlgCfgOPT.CenterOnParent( );
  i1 = m_oDlgCfgOPT.ShowModal( );

  // Set the temperature in the analysis panel
  if( poPnlAna->bIsCreated( PnlAnaBase::eCTLGRP_TEMP ) && i1==wxID_OK )
  {
    os1 = m_oDlgCfgOPT.rosGetValue( DlgNgsCfgOPT::ID_PNL_TEMP );
    poPnlAna->m_oPnlTemp.bSetValue( os1 );
  }
}

//*****************************************************************************
