/*
 *  frontend.c 
 *
 *  Copyright (c) 2005 Giansalvo Gusinu <giansalvo at gusinu.net>
 *  Copyright (C) 1999 Cory Lueninghoener (cluenin1@bigred.unl.edu)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <pthread.h>
#include <linux/types.h> 
#include <gtk/gtk.h>
#include <glib.h>
#include <linux/videodev.h>
#include <png.h>

#include "gqcam.h"
#include "frontend.h"
#include "preferences.h"
#include "save.h"
#include "color.h"
#include "greyscale.h"
#include "timer.h"


/* list of v4l palette recognized by gqcam */
static struct symbolic_list v4l_palette_plist[] = {
	{ VIDEO_PALETTE_GREY,   "GREY" },
	{ VIDEO_PALETTE_RGB24,  "RGB24" },
	{ VIDEO_PALETTE_YUV420, "YUV420" },
       	{ VIDEO_PALETTE_YUV420P,"YUV420P" },
	{ -1,NULL }
};

void closedialog(GtkWidget *widget, GtkWidget *dialog)
{
  gtk_widget_destroy(dialog);
}


void errordialog(char *message, struct Camera *camera)
{
  GtkWidget *errordialog;
  GtkWidget *button;
  GtkWidget *label;
  
  errordialog = gtk_dialog_new();

  button = gtk_button_new_with_label("Oops...");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(errordialog)->action_area), button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(button), "clicked", (GtkSignalFunc) closedialog, errordialog);
    
  gtk_widget_show (button);

  label = gtk_label_new(message);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(errordialog)->vbox), label, TRUE, TRUE, 8);
  gtk_widget_show (label);
  
  gtk_widget_show(errordialog);

  return;
}


void setbrightness(GtkAdjustment *adj, struct Camera *camera)
{
  pthread_mutex_lock(&camera->pref_mutex);
  camera->vid_pic.brightness = (adj->value)*256;
  camera->update_camera = 1;
  pthread_mutex_unlock(&camera->pref_mutex);
}

void setwhite(GtkAdjustment *adj, struct Camera *camera)
{
  pthread_mutex_lock(&camera->pref_mutex);
  camera->vid_pic.whiteness = (adj->value)*256;
  camera->update_camera = 1;
  pthread_mutex_unlock(&camera->pref_mutex);
}

void setcontrast(GtkAdjustment *adj, struct Camera *camera)
{
  pthread_mutex_lock(&camera->pref_mutex);
  camera->vid_pic.contrast = (adj->value)*256;
  camera->update_camera = 1;
  pthread_mutex_unlock(&camera->pref_mutex);
}

void sethue(GtkAdjustment *adj, struct Camera *camera)
{
  pthread_mutex_lock(&camera->pref_mutex);
  camera->vid_pic.hue = (adj->value)*256;
  camera->update_camera = 1;
  pthread_mutex_unlock(&camera->pref_mutex);
}

void setcolor(GtkAdjustment *adj, struct Camera *camera)
{
  pthread_mutex_lock(&camera->pref_mutex);
  camera->vid_pic.colour = (adj->value)*256;
  camera->update_camera = 1;
  pthread_mutex_unlock(&camera->pref_mutex);
}

void setsizefull(GtkWidget *widget, struct Camera *camera)
{
  pthread_mutex_lock(&camera->pref_mutex);
  camera->vid_win.width = camera->vid_caps.maxwidth;
  camera->vid_win.height = camera->vid_caps.maxheight;
  camera->update_camera = 1;
  pthread_mutex_unlock(&camera->pref_mutex);
}

void setsizehalf(GtkWidget *widget, struct Camera *camera)
{
  pthread_mutex_lock(&camera->pref_mutex);
  camera->vid_win.width = (camera->vid_caps.maxwidth)/2;
  camera->vid_win.height = (camera->vid_caps.maxheight)/2;
  camera->update_camera = 1;
  clearimg(camera);
  pthread_mutex_unlock(&camera->pref_mutex);
}

void setsizequarter(GtkWidget *widget, struct Camera *camera)
{
  pthread_mutex_lock(&camera->pref_mutex);
  camera->vid_win.width = (camera->vid_caps.maxwidth)/4;
  camera->vid_win.height = (camera->vid_caps.maxheight)/4;
  camera->update_camera = 1;
  clearimg(camera);
  pthread_mutex_unlock(&camera->pref_mutex);
}

void setdepth4(GtkWidget *widget, struct Camera *camera)
{
  pthread_mutex_lock(&camera->pref_mutex);
  camera->vid_pic.depth = 4;
  camera->update_camera = 1;
  pthread_mutex_unlock(&camera->pref_mutex);
}

void setdepth6(GtkWidget *widget, struct Camera *camera)
{
  pthread_mutex_lock(&camera->pref_mutex);
  camera->vid_pic.depth = 6;
  camera->update_camera = 1;
  pthread_mutex_unlock(&camera->pref_mutex);
}

void stop_capture(GtkWidget *widget, struct Camera *camera)
{
  pthread_mutex_lock( &camera->pref_mutex );
  if( camera->dev ) {
    camera->capture = 0;
    close_cam(camera, 0);
  }
  pthread_mutex_unlock( &camera->pref_mutex );
}

void start_capture(GtkWidget *widget, struct Camera *camera)
{
  pthread_mutex_lock( &camera->pref_mutex );
  open_cam(camera);
  get_cam_info(camera);
  camera->capture = 1;
  update_controls(camera);
  if( !camera->frozen ){
    clearimg(camera);
  }
  pthread_mutex_unlock( &camera->pref_mutex );
}

void en_dis_able(GtkWidget *disable, GtkWidget *enable)
{
  /* k-l-u-d-g-e */
  /* when you call this func w/ NULL, NULL it will go back to the orig state */
  /* I don't know how to ask GTK for object refs, so I put in this hack -jackd */
  static GtkWidget *a = NULL;
  static GtkWidget *b = NULL;
  if( a == NULL && b == NULL ) {
    b = disable;
    a = enable;
  }
  if( disable == NULL && enable == NULL ) {
    gtk_widget_set_sensitive(a, FALSE);
    gtk_widget_set_sensitive(b, TRUE);
  }
  else {
    gtk_widget_set_sensitive(disable, FALSE);
    gtk_widget_set_sensitive(enable, TRUE);
  }
}

void camera_info(GtkWidget *widget, struct Camera *camera)
{
  GtkWidget *dialog;
  GtkWidget *textbox;
  GtkWidget *button;
  GtkWidget *hbox;
  GtkWidget *vscrollbar;
  char buff[80];
  
  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW (dialog),"Gqcam - Camera Info");
  textbox = gtk_text_new(NULL, NULL);
  gtk_widget_set_usize(textbox, 400, 200);
  button = gtk_button_new_with_label("Close");
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), 
			     GTK_OBJECT(dialog));
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
		     hbox);
  vscrollbar = gtk_vscrollbar_new (GTK_TEXT(textbox)->vadj);

  gtk_container_add(GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     button);
  gtk_container_add (GTK_CONTAINER (hbox),
		     textbox);
  gtk_container_add (GTK_CONTAINER(hbox), vscrollbar);

  gtk_widget_show_all (dialog);

  sprintf(buff, "Name: %s\n", camera->vid_caps.name);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Type: %i\n", camera->vid_caps.type);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);

  if (camera->vid_caps.type & VID_TYPE_CAPTURE) {
    sprintf(buff, "\tCan capture\n");
    gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1); 
  }
  if (camera->vid_caps.type & VID_TYPE_TUNER) {
    sprintf(buff, "\tCan tune\n");
    gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  }
  if (camera->vid_caps.type & VID_TYPE_TELETEXT) {
    sprintf(buff, "\tDoes teletext\n");
    gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  }
  if (camera->vid_caps.type & VID_TYPE_OVERLAY) {
    sprintf(buff, "\tOverlay onto frame buffer\n");
    gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  }
  if (camera->vid_caps.type & VID_TYPE_CHROMAKEY) {
    sprintf(buff, "\tOverlay by chromakey\n");
    gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  }
  if (camera->vid_caps.type & VID_TYPE_CLIPPING) {
    sprintf(buff, "\tCan clip\n");
    gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  }
  if (camera->vid_caps.type & VID_TYPE_FRAMERAM) {
    sprintf(buff, "\tUses the frame buffer memory\n");
    gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  }
  if (camera->vid_caps.type & VID_TYPE_SCALES) {
    sprintf(buff, "\tScalable\n");
    gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  }
  if (camera->vid_caps.type & VID_TYPE_MONOCHROME) {
    sprintf(buff, "\tMonochrome only\n");
    gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  }
  if (camera->vid_caps.type & VID_TYPE_SUBCAPTURE) {
    sprintf(buff, "\tCan capture subareas of the image\n");
    gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  }

  sprintf(buff, "Channels: %i\n", camera->vid_caps.channels);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Audios: %i\n", camera->vid_caps.audios);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Maxwidth: %i\n", camera->vid_caps.maxwidth);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Maxheight: %i\n", camera->vid_caps.maxheight);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Minwidth: %i\n", camera->vid_caps.minwidth);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Minheight: %i\n", camera->vid_caps.minheight);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);

  sprintf(buff, "---------\n");
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  
  sprintf(buff, "X: %i\n", camera->vid_win.x);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Y: %i\n", camera->vid_win.y);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Width: %i\n", camera->vid_win.width);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Height: %i\n", camera->vid_win.height);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Chromakey: %i\n", camera->vid_win.chromakey);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Flags: %i\n", camera->vid_win.flags);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);

  sprintf(buff, "---------\n");
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  
  sprintf(buff, "Brightness:\t%i (%i)\n", camera->vid_pic.brightness, camera->vid_pic.brightness/256);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Hue:\t\t%i (%i)\n", camera->vid_pic.hue, camera->vid_pic.hue/256);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Color:\t\t%i (%i)\n", camera->vid_pic.colour, camera->vid_pic.colour/256);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Contrast:\t%i (%i)\n", camera->vid_pic.contrast, camera->vid_pic.contrast/256);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Whiteness:\t%i (%i)\n", camera->vid_pic.whiteness, camera->vid_pic.whiteness/256);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Depth:\t\t%i\n", camera->vid_pic.depth);
  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);
  sprintf(buff, "Palette:\t%i - %s\n",camera->vid_pic.palette, symbolic(v4l_palette_plist, camera->vid_pic.palette));

  gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, buff, -1);

  gtk_adjustment_set_value(GTK_ADJUSTMENT(GTK_TEXT(textbox)->vadj), 0);

}


void about_dialog(GtkWidget *widget, struct Camera *camera)
{
  GtkWidget *dialog, *label, *button;
  char buff[80];

  dialog = gtk_dialog_new();
  sprintf(buff, "gqcam %s\n\n(C)2000 Cory Lueninghoener\n(C)2005 Giansalvo Gusinu", VERSION);
  
  label = gtk_label_new (buff);
  button = gtk_button_new_with_label("OK");
     
  gtk_window_set_title(GTK_WINDOW (dialog),"Gqcam - About");
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), 
			     GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     button);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, FALSE, FALSE, 10);
  
  //gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
  //		     label);
  gtk_widget_show_all (dialog);
}

/*
void pref_dialog(GtkWidget *widget, struct Camera *camera)
{
  errordialog("Preferences dialog is not implemented yet...", camera);

}
*/

void open_dialog_ok(GtkWidget *widget, struct Camera *camera)
{
  stop_capture( NULL, camera );
  usleep(1);
  clearimg(camera);
  sprintf(camera->devname,"%s",gtk_file_selection_get_filename (GTK_FILE_SELECTION (widget->parent->parent->parent)));
  gtk_widget_destroy(widget->parent->parent->parent);  
  start_capture(NULL, camera);
  en_dis_able( NULL, NULL );
}

void open_dialog(GtkWidget *widget, struct Camera *camera)
{
  GtkWidget *opendialog;
  
  opendialog = gtk_file_selection_new ("Open Device");

  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (opendialog)->ok_button),
		      "clicked", (GtkSignalFunc) open_dialog_ok, camera);
  
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (opendialog)->cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (opendialog));
  
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(opendialog), camera->devname);
  
  gtk_widget_show(opendialog);
}

/* void close_dialog(GtkWidget *widget, struct Camera *camera)
   {
   fprintf( stderr, "close_dialog calling stop_capture\n" );
   stop_capture( NULL, camera );
   usleep(1);
   clearimg(camera);
   }*/

void update_controls(struct Camera *camera)
{
  gtk_widget_destroy(camera->controltop);

  if (camera->greyscale)
    create_controls_greyscale(camera);
  else
    create_controls_color(camera);
  if (camera->docked)
    gtk_box_pack_start(GTK_BOX(camera->controlcontainer), camera->controltop, TRUE, TRUE, 0);
  else
    gtk_container_add (GTK_CONTAINER(camera->controlwindow), camera->controltop);
}

void close_controls(GtkWidget *widget, GdkEvent *event, struct Camera *camera)
{
  dock_controls(widget, camera);
}


void dock_controls(GtkWidget *widget, struct Camera *camera)
{
  if (camera->docked){
    //    printf("Floating(%d)\n", camera->docked);
    camera->docked = 0;
    gtk_widget_destroy(camera->controltop);
    if (camera->greyscale)
      create_controls_greyscale(camera);
    else
      create_controls_color(camera);
    camera->controlwindow = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_container_add (GTK_CONTAINER(camera->controlwindow), camera->controltop);
    gtk_window_set_title(GTK_WINDOW (camera->controlwindow), "Gqcam - Control Panel");
    gtk_signal_connect (GTK_OBJECT (camera->controlwindow), "delete_event",
			GTK_SIGNAL_FUNC (close_controls), camera);
    gtk_widget_show(camera->controlwindow);
 
  }
  else {
    camera->docked = 1;
    gtk_widget_destroy(camera->controlwindow);
    if (camera->greyscale)
      create_controls_greyscale(camera);
    else
      create_controls_color(camera);  
    gtk_box_pack_start(GTK_BOX(camera->controlcontainer), camera->controltop, TRUE, TRUE, 0);
    //    printf("Docked\n");
  }
}


void clearimg(struct Camera *camera)
{
  GdkRectangle update_rec;

  configure_event(GTK_WIDGET (camera->drawing_area),NULL, camera);

  update_rec.x = 0;
  update_rec.y = 0;
  update_rec.width = camera->vid_caps.maxwidth;
  update_rec.height = camera->vid_caps.maxheight;
  gtk_widget_draw (camera->drawing_area, &update_rec);
  return;
}

static gint configure_event (GtkWidget *widget, GdkEventConfigure *event, struct Camera *camera)
{
  if (camera->pixmap)
    gdk_pixmap_unref(camera->pixmap);

  camera->pixmap = gdk_pixmap_new(widget->window,
				  widget->allocation.width,
				  widget->allocation.height,
				  -1);
  gdk_draw_rectangle (camera->pixmap,
		      widget->style->white_gc,
		      TRUE,
		      0, 0,
		      widget->allocation.width,
		      widget->allocation.height);
  return TRUE;
}

static gint expose_event (GtkWidget *widget, GdkEventExpose *event, struct Camera *camera)
{
  //  display(camera);
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  camera->pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);
  return FALSE;
}

void freeze(GtkWidget *widget, struct Camera *camera)
{
  /* correct our state */
  if( !camera->frozen && ( GTK_TOGGLE_BUTTON (widget)->active ) ) {
    GTK_TOGGLE_BUTTON (widget)->active;
  }
  /* Careful, if we aren't capturing we don't want to grab the freeze lock,
     or we will deadlock */
  /* same goes for saving */
  pthread_mutex_lock( &camera->pref_mutex );
  if( camera->capture && !camera->saving ) {
    if (GTK_TOGGLE_BUTTON (widget)->active){
      pthread_mutex_lock( &camera->freeze_mutex );
      camera->frozen=1;
    } else {
      camera->frozen=0;
      pthread_mutex_unlock( &camera->freeze_mutex );
    } 
  }
  else {  
    GTK_TOGGLE_BUTTON (widget)->active = !GTK_TOGGLE_BUTTON (widget)->active;
  }
  pthread_mutex_unlock( &camera->pref_mutex );
}

int create_frontend(struct Camera *camera)
{
  GtkWidget *window;
  GtkWidget *topbox;
  GtkWidget *outerbox;
  GtkWidget *innerbox;
  GtkWidget *scalebox;
  GtkWidget *frame;
  GtkObject *adjb;
  GtkWidget *scaleb;
  GtkObject *adjw;
  GtkWidget *scalew;
  GtkObject *adjc;
  GtkWidget *scalec;
  GtkWidget *radiobutton;
  GtkWidget *dockbutton;
  GSList *radiogroup;
  GtkWidget *button;
  GdkDrawable *drawable;
  GtkTooltips *tooltips;
  GtkWidget *statusbar;

  GtkWidget *menubar;
  GtkWidget *filemenu;
  GtkWidget *fileitem;
  GtkWidget *quititem;
  GtkWidget *saveitem;
  GtkWidget *openitem;
  GtkWidget *closeitem;
  GtkWidget *prefitem;
  GtkWidget *seperator;
  GtkWidget *cameramenu;
  GtkWidget *cameraitem;
  GtkWidget *infoitem;
  GtkWidget *updateitem;
  GtkWidget *startitem;
  GtkWidget *stopitem;
  GtkWidget *timeritem;
  GtkWidget *helpmenu;
  GtkWidget *helpitem;
  GtkWidget *aboutitem;

  guchar *gbuff;
  GdkColor c;
  char title[50] = "gqcam - ";
  struct video_clip vid_clips[32];
  int ib=130, ic=30, iw=150, i, count=32;

  strcat(title, camera->vid_caps.name);
/*
  for (i = 0; i < count; i++) {
    vid_clips[i].x      = 0;
    vid_clips[i].y      = 0;
    vid_clips[i].width  = 0;
    vid_clips[i].height = 0;
  }
  
  vid_win.clips = vid_clips;
  vid_win.clipcount = 0;

  ib = (vid_pic.brightness)/256;
  ic = (vid_pic.contrast)/256;
  iw = (vid_pic.whiteness)/256;
*/  
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), title);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, TRUE, TRUE);
  gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		      GTK_SIGNAL_FUNC (delete_event), camera);

  /* Create topbox */
  topbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), topbox);

  /* Create Menu */

  menubar = gtk_menu_bar_new ();
  filemenu = gtk_menu_new();

  openitem = gtk_menu_item_new_with_label("Open New Camera...");
  gtk_menu_append(GTK_MENU(filemenu), openitem);
  gtk_signal_connect(GTK_OBJECT (openitem), "activate",
		     GTK_SIGNAL_FUNC (open_dialog), camera);
  gtk_widget_show(openitem);

  saveitem = gtk_menu_item_new_with_label("Save Image...");
  gtk_menu_append(GTK_MENU(filemenu), saveitem);
  gtk_signal_connect(GTK_OBJECT (saveitem), "activate",
		     GTK_SIGNAL_FUNC (save_dialog), camera);
  gtk_widget_show(saveitem);

  /*   closeitem = gtk_menu_item_new_with_label("Close Camera");
       gtk_menu_append(GTK_MENU(filemenu), closeitem);
       gtk_signal_connect(GTK_OBJECT (closeitem), "activate",
       GTK_SIGNAL_FUNC (close_dialog), camera);  
       gtk_widget_show(closeitem); */

  seperator = gtk_menu_item_new();
  gtk_menu_append(GTK_MENU(filemenu), seperator);
  gtk_widget_show(seperator);

  prefitem = gtk_menu_item_new_with_label("Preferences...");
  gtk_menu_append(GTK_MENU(filemenu), prefitem);
  gtk_signal_connect(GTK_OBJECT (prefitem), "activate",
		     GTK_SIGNAL_FUNC (pref_dialog), camera);
  gtk_widget_show(prefitem);

  seperator = gtk_menu_item_new();
  gtk_menu_append(GTK_MENU(filemenu), seperator);
  gtk_widget_show(seperator);

  quititem = gtk_menu_item_new_with_label("Quit");
  gtk_menu_append(GTK_MENU(filemenu), quititem);
  gtk_signal_connect(GTK_OBJECT (quititem), "activate",
		     GTK_SIGNAL_FUNC (delete_event),
		     camera);
  gtk_widget_show(quititem);

  fileitem = gtk_menu_item_new_with_label("File");
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(fileitem), filemenu);
  gtk_menu_bar_append(GTK_MENU_BAR(menubar), fileitem);


  cameramenu = gtk_menu_new();
  infoitem = gtk_menu_item_new_with_label("Camera Info...");
  gtk_menu_append(GTK_MENU(cameramenu), infoitem);
  gtk_signal_connect(GTK_OBJECT (infoitem), "activate",
		     GTK_SIGNAL_FUNC (camera_info),
		     camera);
  gtk_widget_show(infoitem);

  startitem = gtk_menu_item_new_with_label("Start Capturing");
  gtk_signal_connect(GTK_OBJECT (startitem), "activate",
		     GTK_SIGNAL_FUNC (start_capture),
		     camera);
  gtk_menu_append(GTK_MENU(cameramenu), startitem);
  gtk_widget_show(startitem);
  
  stopitem = gtk_menu_item_new_with_label("Stop Capturing");
  gtk_signal_connect(GTK_OBJECT (stopitem), "activate",
		     GTK_SIGNAL_FUNC (stop_capture),
		     camera);
  gtk_signal_connect(GTK_OBJECT (stopitem), "activate",
		     GTK_SIGNAL_FUNC (en_dis_able),
		     startitem);
  gtk_signal_connect(GTK_OBJECT (startitem), "activate",
		     GTK_SIGNAL_FUNC (en_dis_able),
		     stopitem);
  gtk_menu_append(GTK_MENU(cameramenu), stopitem);
  gtk_widget_show(stopitem);

  if (camera->capture)
    gtk_widget_set_sensitive(startitem, FALSE);
  else
    gtk_widget_set_sensitive(stopitem, FALSE);

  timeritem = gtk_menu_item_new_with_label("Set Timer...");
  gtk_menu_append(GTK_MENU(cameramenu), timeritem);
  gtk_signal_connect(GTK_OBJECT (timeritem), "activate",
		     GTK_SIGNAL_FUNC (set_timer),
		     camera);
  gtk_widget_show(timeritem);

  /*
  updateitem = gtk_menu_item_new_with_label("Update Controls");
  gtk_menu_append(GTK_MENU(cameramenu), updateitem);
  gtk_signal_connect(GTK_OBJECT (updateitem), "activate",
		     GTK_SIGNAL_FUNC (update_controls), camera);
  gtk_widget_show(updateitem);
*/
  cameraitem = gtk_menu_item_new_with_label("Camera");
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(cameraitem), cameramenu);
  gtk_menu_bar_append(GTK_MENU_BAR(menubar), cameraitem);

  helpmenu = gtk_menu_new();
  aboutitem = gtk_menu_item_new_with_label("About");
  gtk_signal_connect(GTK_OBJECT (aboutitem), "activate",
		     GTK_SIGNAL_FUNC (about_dialog),
		     camera);
  gtk_menu_append(GTK_MENU(helpmenu), aboutitem);
  helpitem = gtk_menu_item_new_with_label("Help");
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(helpitem), helpmenu);
  gtk_menu_item_right_justify(GTK_MENU_ITEM(helpitem));
  gtk_menu_bar_append(GTK_MENU_BAR(menubar), helpitem);
  gtk_widget_show(aboutitem);

  gtk_widget_show(fileitem);
  gtk_widget_show(cameraitem);
  gtk_widget_show(helpitem);

  gtk_box_pack_start(GTK_BOX(topbox), menubar, FALSE, TRUE, 0);
  gtk_widget_show (menubar);


  /* Create the drawing area */
  frame = gtk_frame_new(NULL);
  gtk_box_pack_start (GTK_BOX(topbox), frame, FALSE, TRUE, 0);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
  gtk_widget_show(frame);

  camera->drawing_area = gtk_drawing_area_new();
  gtk_drawing_area_size (GTK_DRAWING_AREA (camera->drawing_area), camera->vid_caps.maxwidth, camera->vid_caps.maxheight);
  gtk_container_add(GTK_CONTAINER(frame), camera->drawing_area);
  gtk_widget_show (camera->drawing_area);
/*
  gtk_signal_connect (GTK_OBJECT (camera->drawing_area), "expose_event",
		      (GtkSignalFunc) display, camera);
*/
/*------ bp -------*/

  gtk_signal_connect (GTK_OBJECT (camera->drawing_area), "expose_event",
		      (GtkSignalFunc) expose_event, camera);
  gtk_signal_connect (GTK_OBJECT(camera->drawing_area),"configure_event",
		      (GtkSignalFunc) configure_event, camera);

/*-----------------*/

  dockbutton = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(topbox), dockbutton, FALSE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (dockbutton), "clicked",
		      (GtkSignalFunc) dock_controls, camera);
  gtk_widget_show(dockbutton);

  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip(tooltips, dockbutton, "Dock/Undock Controls", NULL);

  camera->controlcontainer = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start(GTK_BOX(topbox), camera->controlcontainer, FALSE, TRUE, 0);
  gtk_widget_show(camera->controlcontainer);

  if (camera->greyscale)
    create_controls_greyscale(camera);
  else
    create_controls_color(camera);

  gtk_box_pack_start(GTK_BOX(camera->controlcontainer), camera->controltop, FALSE, TRUE, 0);

  /* Create a few buttons */
  innerbox = gtk_hbox_new (TRUE, 0);
  gtk_box_pack_start( GTK_BOX (topbox), innerbox, FALSE, TRUE, 0);
  gtk_widget_show (innerbox);
  
  button = gtk_button_new_with_label ("Snap Picture");
  gtk_box_pack_start( GTK_BOX (innerbox), button, FALSE, TRUE, 0);
  gtk_widget_show(button);

  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (save_dialog), camera);

  button = gtk_toggle_button_new_with_label ("Freeze Picture");
  gtk_box_pack_start(GTK_BOX (innerbox), button, FALSE, TRUE, 0);
  gtk_widget_show(button);

  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) freeze, camera);

  button = gtk_button_new_with_label ("Quit");
  gtk_box_pack_start( GTK_BOX (innerbox), button, FALSE, TRUE, 0);
  gtk_widget_show(button);

  gtk_signal_connect (GTK_OBJECT (button), "clicked",
	 	      GTK_SIGNAL_FUNC (delete_event), camera);

  /* create the frames per second status */
  if( 1 ) {
    gint context_id;
    camera->statusbar = gtk_statusbar_new();
    gtk_box_pack_start( GTK_BOX (topbox), camera->statusbar, TRUE, TRUE, 0);
    gtk_widget_show( camera->statusbar );
    context_id = gtk_statusbar_get_context_id( GTK_STATUSBAR(camera->statusbar), "Statusbar example" );
    gtk_statusbar_push( GTK_STATUSBAR( camera->statusbar ), context_id, "Speed - Calculating" );
  }
  gtk_widget_show (topbox);
  gtk_widget_show (window);

  return 0;
}
