/* $Id: watchp.cc,v 1.1 2002/04/28 19:37:23 bergo Exp $ */

/*

    GPS - Graphical Process Statistics
    Copyright (C) 1999-2002 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "watchp.h"
#include "diefast.h"
#include "polling.h"
#include "netpolling.h"
#include "importglobals.h"

class WatchedProcess {
public:
  WatchedProcess() {
    cursor=0;
    dead=0;
    prepared=0;
    pend=0;
    wstart=time(0);
    timeout_id=0;
    window=0;
    ready=0;

    memset(ucpu,0,2048*sizeof(unsigned int));
    memset(kcpu,0,2048*sizeof(unsigned int));
    memset(ducpu,0,2048*sizeof(unsigned int));
    memset(dkcpu,0,2048*sizeof(unsigned int));
    memset(sizm,0,2048*sizeof(unsigned long));
    memset(rssm,0,2048*sizeof(unsigned long));
  }

  void shift() {
    int i;
    for(i=0;i<2047;i++) {
      ucpu[i]=ucpu[i+1];
      kcpu[i]=kcpu[i+1];
      ducpu[i]=ducpu[i+1];
      dkcpu[i]=dkcpu[i+1];
      rssm[i]=rssm[i+1];
      sizm[i]=sizm[i+1];
    }
  }

  long maxsize(int samples) {
    int i;
    long v=0;
    if (samples > cursor) samples=cursor;
    for(i=0;i<samples;i++)
      if (sizm[cursor-i] > v) v=sizm[cursor-i];
    return v;
  }

  int maxcpu(int samples) {
    int i,v=0;
    if (samples > cursor) samples=cursor;
    for(i=1;i<samples;i++) {
      if (ducpu[cursor-i]+dkcpu[cursor-i] > v)
	v=ducpu[cursor-i]+dkcpu[cursor-i];
    }
    return v;
  }

  int   pid;
  char  host[128];
  char  name[256];
  char  longname[256];
  int   dead, prepared, ready;
  
  int   cursor;
  unsigned int   ucpu[2048]; // usermode jiffies
  unsigned int   kcpu[2048]; // kernelmode jiffies

  unsigned int   ducpu[2048]; // delta
  unsigned int   dkcpu[2048]; // delta

  unsigned long  rssm[2048]; // rss 
  unsigned long  sizm[2048]; // size

  int   s_rss[2];   // min, max
  int   s_size[2];  // min, max

  time_t wstart, pend;
  char pstart[128];

  GtkWidget *window, *canvas;
  gint       timeout_id;
};

void
open_watch(gint thepid, char *thehost, int interval)
{
  GtkWidget *w,*v,*h,*b,*da;
  WatchedProcess *wp;
  char z[256];

  snprintf(z,255,"Process Watch: %d@%s (%1.2f sec polling)",
	   thepid, thehost, ((double)interval)/1000.0);
  z[255]=0;

  wp=new WatchedProcess();
  wp->pid=thepid;
  strncpy(wp->host, thehost, 127);
  wp->host[127]=0;

  w=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(w),z);
  gtk_window_set_default_size(GTK_WINDOW(w),500,350);
  gtk_container_set_border_width(GTK_CONTAINER(w), 6);

  wp->window = w;

  v=gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(w), v);

  da=gtk_drawing_area_new();
  gtk_widget_set_events(da, GDK_EXPOSURE_MASK);
  gtk_drawing_area_size(GTK_DRAWING_AREA(da), 500, 350);

  wp->canvas = da;

  gtk_box_pack_start(GTK_BOX(v), da, TRUE, TRUE, 0);

  h=gtk_hbox_new(FALSE,0);
  b=gtk_button_new_with_label("Dismiss");
  GTK_WIDGET_SET_FLAGS(b,GTK_CAN_DEFAULT);
  

  gtk_box_pack_start(GTK_BOX(v),h,FALSE,TRUE,2);
  gtk_box_pack_end(GTK_BOX(h),b,FALSE,FALSE,4);
  
  gtk_signal_connect(GTK_OBJECT(w),"destroy",
		     GTK_SIGNAL_FUNC(wp_destroy),(gpointer)wp);
  gtk_signal_connect(GTK_OBJECT(w),"delete_event",
		     GTK_SIGNAL_FUNC(wp_delete),0);
  gtk_signal_connect(GTK_OBJECT(b),"clicked",
		     GTK_SIGNAL_FUNC(wp_dismiss),(gpointer)wp);
  gtk_signal_connect(GTK_OBJECT(da),"expose_event",
		     GTK_SIGNAL_FUNC(wp_paint),(gpointer)wp);

  gtk_widget_show(b);
  gtk_widget_show(h);
  gtk_widget_show(da);
  gtk_widget_show(v);
  gtk_widget_show(w);

  wp->timeout_id=gtk_timeout_add(interval, wp_update, (gpointer) wp);
  wp_update((gpointer) wp);
}

void
wp_destroy(GtkWidget *w, gpointer data)
{
  WatchedProcess *wp;
  wp=(WatchedProcess*)data;
  if (wp->timeout_id != 0) {
    gtk_timeout_remove(wp->timeout_id);
    wp->timeout_id=0;
  }
  delete wp;
}

gboolean
wp_delete(GtkWidget *w, GdkEvent *e, gpointer data)
{
  return FALSE;
}

void
wp_dismiss(GtkWidget *w, gpointer data)
{
  WatchedProcess *wp;
  wp=(WatchedProcess*)data;
  if (wp->timeout_id != 0) {
    gtk_timeout_remove(wp->timeout_id);
    wp->timeout_id=0;
  }
  gtk_widget_destroy(wp->window);
}

gboolean
wp_update(gpointer data)
{
  WatchedProcess *wp;
  NetworkDetailsPoller *ndp;
  ProcessItem *pi;
  GList *pt;

  wp=(WatchedProcess*)data;

  if (wp->dead) {
    wp->timeout_id=0;
    return FALSE;
  }

  if (!strcmp(wp->host,this_host)) { // local process
    while(!enter_mutex()) usleep(13);
    details_poller->poll(wp->pid);
    pi=details_poller->item;
    exit_mutex();
  } else { // remote process
    pi=0;
    for(pt=detailswatch;pt!=NULL;pt=g_list_next(pt)) {
      ndp=(NetworkDetailsPoller *)(pt->data);
      if (ndp->isHost(wp->host)) {
	ndp->poll(wp->pid);
	pi=ndp->item;
	break;
      }      
    }
  }

  if (pi==0) {
    wp->dead=1;
    wp->pend=time(0);    
  } else {

    strncpy(wp->name,pi->name,255);
    wp->name[255]=0;
    strncpy(wp->longname,pi->longname,255);
    wp->longname[255]=0;

    strncpy(wp->pstart,pi->start,127);
    wp->pstart[127]=0;

    if (wp->cursor == 2048) {
      wp->shift();
      wp->cursor--;
    }

    wp->ucpu[wp->cursor] = atoi(pi->uj);
    wp->kcpu[wp->cursor] = atoi(pi->kj);

    if (wp->cursor == 0) {
      wp->ducpu[0]=0;
      wp->dkcpu[0]=0;
    } else {
      wp->ducpu[wp->cursor] = wp->ucpu[wp->cursor] - wp->ucpu[wp->cursor - 1];
      wp->dkcpu[wp->cursor] = wp->kcpu[wp->cursor] - wp->kcpu[wp->cursor - 1];
    }

    wp->rssm[wp->cursor] = atol(pi->rss);
    wp->sizm[wp->cursor] = atol(pi->size);

    /*
    printf("rss=%dK, size=%dK\n",
	   wp->rssm[wp->cursor]/1024,
	   wp->sizm[wp->cursor]/1024);
    */   

    if (!wp->prepared) {
      wp->s_rss[0]=wp->s_rss[1]=wp->rssm[wp->cursor];
      wp->s_size[0]=wp->s_size[1]=wp->sizm[wp->cursor];
      wp->prepared=1;
    } else {
      // min and max checking
      if (wp->rssm[wp->cursor] < wp->s_rss[0])
	wp->s_rss[0]=wp->rssm[wp->cursor];
      if (wp->rssm[wp->cursor] > wp->s_rss[1])
	wp->s_rss[1]=wp->rssm[wp->cursor];

      if (wp->sizm[wp->cursor] < wp->s_size[0])
	wp->s_size[0]=wp->sizm[wp->cursor];
      if (wp->sizm[wp->cursor] > wp->s_size[1])
	wp->s_size[1]=wp->sizm[wp->cursor];
    }
    
    wp->cursor++;  
  }

  wp->ready=1;
  gtk_widget_queue_resize(wp->canvas);

  return TRUE;
}


gboolean 
wp_paint(GtkWidget *widget,GdkEventExpose *ee, gpointer data)
{
  GdkGC *gc;
  GdkFont *fnt;
  WatchedProcess *wp;
  int w,h,gw;
  GdkPixmap *x;

  long mv, dv;
  int i,j,k,m;

  float mf, df;

  char z[128], y[128];

  wp=(WatchedProcess *)data;
  gc=gdk_gc_new(widget->window);

  w=widget->allocation.width;
  h=widget->allocation.height;

  fnt=gdk_font_load("-*-helvetica-medium-r-*-*-12-*-*-*-*-*-*-*");

  x=gdk_pixmap_new(widget->window,w,h,-1);

  gdk_rgb_gc_set_foreground(gc, 0);
  gdk_draw_rectangle(x, gc, TRUE, 0,0, w,h);

  gw = w - 100 - 10;
  if (gw < 50) goto not_enough_room;

  gdk_rgb_gc_set_foreground(gc, 0x8080ff);
  gdk_draw_string(x,fnt,gc,5,20,"Size");
  gdk_rgb_gc_set_foreground(gc, 0xffffff);
  gdk_draw_string(x,fnt,gc,35,20,"/");
  gdk_rgb_gc_set_foreground(gc, 0xffff40);
  gdk_draw_string(x,fnt,gc,45,20,"RSS");

  if (!wp->ready) goto not_enough_room;
  if (!wp->prepared) goto not_enough_room;

  mv=wp->maxsize(gw);
  dv=mv/64;
  if (dv==0) dv=1;

  if (wp->cursor) {
    k=wp->cursor;
    for(i=0;i<gw;i++) {
      --k;
      if (k<=0) break;
      
      gdk_rgb_gc_set_foreground(gc, 0x8080ff);
      gdk_draw_line(x,gc,10+gw-i,25+64, 10+gw-i, 25+64-(wp->sizm[k]/dv));
      gdk_rgb_gc_set_foreground(gc, 0xffff40);
      gdk_draw_line(x,gc,10+gw-i,25+64, 10+gw-i, 25+64-(wp->rssm[k]/dv));
    }
    
    gdk_rgb_gc_set_foreground(gc, 0xffffff);
    gdk_draw_rectangle(x, gc, FALSE, 10,25, gw,64);
    gdk_draw_line(x,gc,gw+10,25,gw+20,25);

    snprintf(z,127,"%luK", mv/1024);
    z[127]=0;
    gdk_draw_string(x,fnt,gc,gw+25,32,z);
    
    snprintf(z,127,"SIZE min=%luK max=%luK last=%luK",
	     wp->s_size[0]/1024,
	     wp->s_size[1]/1024,
	     wp->sizm[wp->cursor-1]/1024);
    z[127]=0;
    gdk_rgb_gc_set_foreground(gc, 0x8080ff);
    gdk_draw_string(x,fnt,gc,10,25+64+14,z);
    
    snprintf(z,127,"RSS min=%luK max=%luK last=%luK",
	     wp->s_rss[0]/1024,
	     wp->s_rss[1]/1024,
	     wp->rssm[wp->cursor-1]/1024);
    z[127]=0;
    gdk_rgb_gc_set_foreground(gc, 0xffff40);
    gdk_draw_string(x,fnt,gc,10,25+64+14+14,z);

    // cpu
    gdk_rgb_gc_set_foreground(gc, 0xffffff);
    gdk_draw_string(x,fnt,gc,10,155,"CPU");
    gdk_rgb_gc_set_foreground(gc, 0x00ff00);
    gdk_draw_string(x,fnt,gc,40,155,"(userland)");
    gdk_rgb_gc_set_foreground(gc, 0xff4080);
    gdk_draw_string(x,fnt,gc,100,155,"(kernelspace)");

    mf= (float) (wp->maxcpu(gw));
    df = mf/64.0;
    if (df==0.0) df=1.0;

    k=wp->cursor;
    for(i=0;i<gw;i++) {
      --k;
      if (k<=0) break;

      j=(int) (((float)(wp->dkcpu[k]))/df);
      m=(int) (((float)(wp->ducpu[k]))/df);
                
      gdk_rgb_gc_set_foreground(gc, 0xff4080);
      gdk_draw_line(x,gc,10+gw-i,160+64, 10+gw-i, 160+64-j);

      gdk_rgb_gc_set_foreground(gc, 0x00ff00);
      gdk_draw_line(x,gc,10+gw-i,160+64-j, 10+gw-i, 160+64-j-m);
    }

    gdk_rgb_gc_set_foreground(gc, 0xffffff);
    gdk_draw_rectangle(x, gc, FALSE, 10,160, gw,64);
    gdk_draw_line(x,gc,gw+10,160,gw+20,160);    

    snprintf(z,127,"%d jiffies", ((int)mf) );
    z[127]=0;
    gdk_draw_string(x,fnt,gc,gw+25,167,z);

    snprintf(z,127,"Userland: %d jiffies  Kernelspace: %d jiffies  Total: %d jiffies",
	     wp->ucpu[wp->cursor-1],
	     wp->kcpu[wp->cursor-1],
	     wp->ucpu[wp->cursor-1]+wp->kcpu[wp->cursor-1]);
    z[127]=0;
    gdk_rgb_gc_set_foreground(gc, 0xffffff);
    gdk_draw_string(x,fnt,gc,10,160+64+14,z);    

  }

  // "static" process info

  gdk_rgb_gc_set_foreground(gc, 0xffffff);
  snprintf(z,127,"Process Name: %s",wp->name);
  z[127]=0;
  gdk_draw_string(x,fnt,gc,10,160+64+14+20,z);

  snprintf(z,127,"Command Line: %s",wp->longname);
  z[127]=0;
  gdk_draw_string(x,fnt,gc,10,160+64+14+20+14,z);

  gdk_rgb_gc_set_foreground(gc, 0xffff80);

  snprintf(z,127,"Process Started: %s",wp->pstart);
  z[127]=0;
  gdk_draw_string(x,fnt,gc,10,160+64+14+20+14+14,z);

  strncpy(y,ctime(&(wp->wstart)),127);
  y[127]=0;
  if (y[strlen(y)-1]=='\n') y[strlen(y)-1]=0;

  snprintf(z,127,"Watch Started: %s",y);
  z[127]=0;
  gdk_draw_string(x,fnt,gc,10,160+64+14+20+14+14+14,z);

  if (wp->dead) {
    strncpy(y,ctime(&(wp->pend)),127);
    y[127]=0;
    snprintf(z,127,"Process Finished: %s",y);
    z[127]=0;
  } else {
    strcpy(z,"Process is still Running");
  }

  gdk_draw_string(x,fnt,gc,10,160+64+14+20+14+14+14+14,z);

 not_enough_room:
  gdk_gc_destroy(gc);
  gdk_font_unref(fnt);

  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  x,0,0,0,0,w,h);
  gdk_pixmap_unref(x);
  return FALSE;
}
