# -*- coding: utf-8 -*-
#
# gPodder - A media aggregator and podcast client
# Copyright (c) 2005-2009 Thomas Perl and the gPodder Team
#
# gPodder is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# gPodder is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""
Loads and executes user hooks.

Hooks are python scripts in the "Hooks" folder of $GPODDER_HOME. Each script
must define a class named "gPodderHooks", otherwise it will be ignored.

The hooks class defines several callbacks that will be called by the
gPodder application at certain points. See the methods defined below
for a list on what these callbacks are and the parameters they take.

For an example extension see examples/hooks.py
"""

import glob
import imp
import os
import functools

import gpodder

import logging
logger = logging.getLogger(__name__)


def call_hooks(func):
    """Decorator to create handler functions in HookManager

    Calls the specified function in all user extensions that define it.
    """
    method_name = func.__name__

    @functools.wraps(func)
    def handler(self, *args, **kwargs):
        result = None
        for filename, module in self.modules:
            try:
                callback = getattr(module, method_name, None)
                if callback is not None:
                    # If the results are lists, concatenate them to show all
                    # possible items that are generated by all hooks together
                    cb_res = callback(*args, **kwargs)
                    if isinstance(result, list) and isinstance(cb_res, list):
                        result.extend(cb_res)
                    elif cb_res is not None:
                        result = cb_res
            except Exception, e:
                logger.error('Error in %s, function %s: %s', filename,
                        method_name, e, exc_info=True)
        func(self, *args, **kwargs)
        return result

    return handler


class HookManager(object):
    # The class name that has to appear in a hook module
    HOOK_CLASS = 'gPodderHooks'

    def __init__(self):
        """Create a new hook manager"""
        self.modules = []

        for filename in glob.glob(os.path.join(gpodder.home, 'Hooks', '*.py')):
          try:
              module = self._load_module(filename)
              if module is not None:
                  self.modules.append((filename, module))
                  logger.info('Module loaded: %s', filename)
          except Exception, e:
              logger.error('Cannot load %s: %s', filename, e, exc_info=True)

    def register_hooks(self, obj):
        """
        Register an object that implements some hooks.
        """
        self.modules.append((None, obj))

    def unregister_hooks(self, obj):
        """
        Unregister a previously registered object.
        """
        if (None, obj) in self.modules:
            self.modules.remove((None, obj))
        else:
            logger.warn('Unregistered hook which was not registered.')

    def _load_module(self, filepath):
        """Load a Python module by filename

        Returns an instance of the HOOK_CLASS class defined
        in the module, or None if the module does not contain
        such a class.
        """
        basename, extension = os.path.splitext(os.path.basename(filepath))
        module = imp.load_module(basename, file(filepath, 'r'), filepath, (extension, 'r', imp.PY_SOURCE))
        hook_class = getattr(module, HookManager.HOOK_CLASS, None)

        if hook_class is None:
            return None
        else:
            return hook_class()

    # Define all known handler functions here, decorate them with the
    # "call_hooks" decorator to forward all calls to hook scripts that have
    # the same function defined in them. If the handler functions here contain
    # any code, it will be called after all the hooks have been called.

    @call_hooks
    def on_ui_initialized(self, model, update_podcast_callback,
            download_episode_callback):
        """Called when the user interface is initialized.

        @param model: A gpodder.model.Model instance
        @param update_podcast_callback: Function to update a podcast feed
        @param download_episode_callback: Function to download an episode
        """
        pass

    @call_hooks
    def on_podcast_subscribe(self, podcast):
        """Called when the user subscribes to a new podcast feed.

        @param podcast: A gpodder.model.PodcastChannel instance
        """
        pass

    @call_hooks
    def on_podcast_updated(self, podcast):
        """Called when a podcast feed was updated

        This hook will be called even if there were no new episodes.

        @param podcast: A gpodder.model.PodcastChannel instance
        """
        pass

    @call_hooks
    def on_podcast_update_failed(self, podcast, exception):
        """Called when a podcast update failed.

        @param podcast: A gpodder.model.PodcastChannel instance

        @param exception: The reason.
        """
        pass

    @call_hooks
    def on_podcast_save(self, podcast):
        """Called when a podcast is saved to the database

        This hooks will be called when the user edits the metadata of
        the podcast or when the feed was updated.

        @param podcast: A gpodder.model.PodcastChannel instance
        """
        pass

    @call_hooks
    def on_podcast_delete(self, podcast):
        """Called when a podcast is deleted from the database

        @param podcast: A gpodder.model.PodcastChannel instance
        """
        pass

    @call_hooks
    def on_episode_save(self, episode):
        """Called when an episode is saved to the database

        This hook will be called when a new episode is added to the
        database or when the state of an existing episode is changed.

        @param episode: A gpodder.model.PodcastEpisode instance
        """
        pass

    @call_hooks
    def on_episode_downloaded(self, episode):
        """Called when an episode has been downloaded

        You can retrieve the filename via episode.local_filename(False)

        @param episode: A gpodder.model.PodcastEpisode instance
        """
        pass

    @call_hooks
    def on_episodes_context_menu(self, episodes):
        """Called when the episode list context menu is opened

        You can add additional context menu entries here. You have to
        return a list of tuples, where the first item is a label and
        the second item is a callable that will get the episode as its
        first and only parameter.

        Example return value:

        [('Mark as new', lambda episodes: ...)]

        @param episode: A list of gpodder.model.PodcastEpisode instances
        """
        pass

    @call_hooks
    def on_episode_delete(self, episode, filename):
        """Called just before the episode's disk file is about to be
        deleted."""
        pass

    @call_hooks
    def on_episode_removed_from_podcast(self, episode):
        """Called just before the episode is about to be removed from
        the podcast channel, e.g., when the episode has not been
        downloaded and it disappears from the feed.

        @param podcast: A gpodder.model.PodcastChannel instance
        """
        pass
