/* *************************************************************************
                          gdlc.i.g 
the GDL interpreter
interprets the output of the treeparser/compiler
                             -------------------
    begin                : July 22 2002
    copyright            : (C) 2002 by Marc Schellens
    email                : m_schellens@hotmail.com
 ***************************************************************************/

/* *************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

header "pre_include_cpp" {
    // gets inserted before the antlr generated includes in the cpp file
#include "includefirst.hpp"
}

header "post_include_cpp" {
    // gets inserted after the antlr generated includes in the cpp file
#include "dinterpreter.hpp"
#include "prognodeexpr.hpp"

#include <cassert>

// tweaking ANTLR
#define ASTNULL          NULLProgNodeP
#define ProgNodeP( xxx ) NULL             /* ProgNodeP(antlr::nullAST) */
#define RefAST( xxx)     ConvertAST( xxx) /* antlr::RefAST( Ref type)  */
#define match( a, b)     /* remove from source */

using namespace std;
}

header {
    // antlr header

    // make sure it gets included before the 'tweak'
#include "GDLParser.hpp" 
#include "GDLTreeParser.hpp" 

#include <map>
#include <iomanip>
//#include <exception>

#include "datatypes.hpp"
#include "objects.hpp"
#include "dpro.hpp"
#include "accessdesc.hpp"
#include "initsysvar.hpp"
#include "gdljournal.hpp"

//class ProgNode;
//typedef ProgNode* ProgNodeP;

// tweaking ANTLR
#define RefAST( xxx)     ConvertAST( xxx) /* antlr::RefAST( Ref type)  */

// print out AST tree
//#define GDL_DEBUG
//#undef GDL_DEBUG
//#define GDL_DEBUG_HEAP

}

options {
	language="Cpp";
	genHashLines = false;
	namespaceStd="std";         // cosmetic option to get rid of long defines
	namespaceAntlr="antlr";     // cosmetic option to get rid of long defines
}	

// the GDL TreeParser  ****************************************
class GDLInterpreter extends TreeParser;

options {
  importVocab = GDL;	// use vocab generated by lexer
  buildAST = false;     // no modifying of AST anymore
    // no AST is created in the interpreter, hence we don't need ref counting
//  ASTLabelType = "RefDNode"; 
  ASTLabelType = "ProgNodeP"; 
//  defaultErrorHandler = true;
  defaultErrorHandler = false;
//  codeGenMakeSwitchThreshold = 2;
//  codeGenBitsetTestThreshold = 32;
}

{
private:
    // ASTNULL replacement
    static ProgNode  NULLProgNode;
    static ProgNodeP NULLProgNodeP;

    friend class BaseGDL;
    friend class ProgNode;
    friend class ARRAYDEFNode;
    friend class STRUCNode;
    friend class NSTRUCNode;
    friend class NSTRUC_REFNode;
    friend class ASSIGNNode;
    friend class ASSIGN_ARRAYEXPR_MFCALLNode;
    friend class ASSIGN_REPLACENode;
    friend class PCALL_LIBNode;//: public CommandNode
    friend class MPCALLNode;//: public CommandNode
    friend class MPCALL_PARENTNode;//: public CommandNode
    friend class PCALLNode;//: public CommandNode
    friend class RETFNode;
    friend class RETPNode;
    friend class FORNode;
    friend class FOR_LOOPNode;
    friend class FOREACHNode;
    friend class FOREACH_LOOPNode;
    friend class FOREACH_INDEXNode;
    friend class FOREACH_INDEX_LOOPNode;
    friend class FOR_STEPNode;
    friend class FOR_STEP_LOOPNode;
    
    friend class KEYDEFNode;
    friend class KEYDEF_REFNode;
    friend class KEYDEF_REF_CHECKNode;
    friend class KEYDEF_REF_EXPRNode;
    friend class REFNode;
    friend class REF_CHECKNode;
    friend class REF_EXPRNode;
    friend class ParameterNode;

public: 

//     RetCode returnCode;    
    ProgNodeP GetNULLProgNodeP() const { return NULLProgNodeP;}


    void SetRetTree( ProgNodeP rT)
    {
        this->_retTree = rT;
    }
//     void SetReturnCode( RetCode rC)
//     {
//         this->returnCode = rC;
//     }
    
//     enum RetCode {
//         RC_OK=0,
//         RC_BREAK,
//         RC_CONTINUE,
//         RC_RETURN, 
//         RC_ABORT, // checked as retCode >= RC_RETURN
//     };  

    // code in: dinterpreter.cpp
    static bool SearchCompilePro(const std::string& pro);
    static int GetFunIx( ProgNodeP);
    static int GetFunIx( const std::string& subName);
    static int GetProIx( ProgNodeP);//const std::string& subName);
    static int GetProIx( const std::string& subName);
    DStructGDL* ObjectStruct( BaseGDL* self, ProgNodeP mp);
    DStructGDL* ObjectStructCheckAccess( BaseGDL* self, ProgNodeP mp);

    // code in: dinterpreter.cpp
    static void SetFunIx( ProgNodeP f); // triggers read/compile


private: 

    static void SetProIx( ProgNodeP f); // triggers read/compile
    static void AdjustTypes( BaseGDL*&, BaseGDL*&);


protected:
    std::istringstream executeLine; // actual interactive executed line

//     std::vector<BaseGDL*> tmpList;
//     void ClearTmpList()
//     {
//         std::vector<BaseGDL*>::iterator i;
//         for(i = tmpList.begin(); i != tmpList.end(); ++i) 
//             { delete *i;}
//         tmpList.clear();
//     }

    class RetAllException 
    {
        public:
        enum ExCode {
            NONE=0, // normal RETALL
            RUN     // RETALL from .RUN command
        };  

        private:
        ExCode code;

        public:
        RetAllException( ExCode code_=NONE): code( code_) {}

        ExCode Code() { return code;}
    };
    
    // code in: dinterpreter.cpp
//    static bool CompleteFileName(std::string& fn); -> str.cpp

    BaseGDL*  returnValue;  // holding the return value for functions
    BaseGDL** returnValueL; // holding the return value for l_functions

    bool interruptEnable;

public:
    static bool CompileFile(const std::string& f, const std::string& untilPro=""); 

    typedef RefHeap<BaseGDL> RefBaseGDL;
    typedef RefHeap<DStructGDL> RefDStructGDL;

    typedef std::map<SizeT, RefBaseGDL> HeapT;
    typedef std::map<SizeT, RefDStructGDL> ObjHeapT;

protected:
//     typedef std::map<SizeT, BaseGDL*> HeapT;
//     typedef std::map<SizeT, DStructGDL*> ObjHeapT;

    // the following must be all static because several interpreter might be active
    // the heap for all dynamic variables
    // ease the handling, no memory leaks, gc possible
    static HeapT     heap; 
    static ObjHeapT  objHeap; 

    // index for newly allocated heap variables
    static SizeT objHeapIx;
    static SizeT heapIx;

    static EnvStackT  callStack; 

    static DLong stepCount;


// smuggle optimizations in
//#include "GDLInterpreterOptimized.inc"


public:
    // triggers read/compile/interpret
    DStructDesc* GetStruct(const std::string& name, const ProgNodeP cN); 

//     bool Called( std::string proName)
//     {
//         for( EnvStackT::reverse_iterator env = callStack.rbegin();
//             env != callStack.rend();
//             ++env)
//             {
//                 //std::cout <<  (*env)->GetPro()->ObjectFileName() << std::endl;
//                 if( proName == (*env)->GetPro()->ObjectFileName()) return true;
//             }
//         return false;
//     }

    // the New... functions 'own' their BaseGDL*
    SizeT NewObjHeap( SizeT n=1, DStructGDL* var=NULL)
    {
        SizeT tmpIx=objHeapIx;
        for( SizeT i=0; i<n; i++)
        objHeap.insert( objHeap.end(),
            std::pair<SizeT, RefDStructGDL>( objHeapIx++, (DStructGDL*)var));
        return tmpIx;
    }
    SizeT NewHeap( SizeT n=1, BaseGDL* var=NULL)
    {
        SizeT tmpIx=heapIx;
        for( SizeT i=0; i<n; i++)
        heap.insert( heap.end(),
            std::pair<SizeT, RefBaseGDL>( heapIx++, var));
        return tmpIx;
    }
    static void FreeObjHeap( DObj id)
    {
        if( id != 0)
        {       
            ObjHeapT::iterator it=objHeap.find( id);
            if  ( it != objHeap.end()) 
            { 
                delete (*it).second.get();
                objHeap.erase( id);
            }
        }
    }
    static void FreeObjHeapDirect( DObj id, ObjHeapT::iterator it)
    {
        delete (*it).second.get();
        objHeap.erase( id);
    }
    static void FreeHeap( DPtr id)
    {
        if( id != 0)
            {
                HeapT::iterator it=heap.find( id);
                if( it != heap.end()) 
                    { 
                        delete (*it).second.get();
                        heap.erase( id); 
                    }
            }
    }
    static void FreeHeapDirect( DPtr id, HeapT::iterator it)
    {
        delete (*it).second.get();
        // useless because of next: (*it).second.get() = NULL;
        heap.erase( id); 
    }

   static void FreeHeap( DPtrGDL* p)
    {
        SizeT nEl=p->N_Elements();
        for( SizeT ix=0; ix < nEl; ix++)
        {
            DPtr id= (*p)[ix];
            FreeHeap( id);
       }
    }

   static void DecRef( DPtr id)
    {
        if( id != 0)
            {
#ifdef GDL_DEBUG_HEAP
                std::cout << "-- <PtrHeapVar" << id << ">" << std::endl; 
#endif
                HeapT::iterator it=heap.find( id);
                if( it != heap.end()) 
                    { 
                        if( (*it).second.Dec())
                            {
#ifdef GDL_DEBUG_HEAP
                                std::cout << "Out of scope (garbage collected): <PtrHeapVar" << id 
                                          << ">"
                                          << " at: " << callStack.back()->GetProName()
                                          << "  line: " << callStack.back()->GetLineNumber()
                                          << std::endl; 
#endif
                                FreeHeapDirect( id, it);
                            }
#ifdef GDL_DEBUG_HEAP
                        else
std::cout << "<PtrHeapVar" << id << "> = " << (*it).second.Count() << std::endl; 
#endif
                    }
            }
    }
   static void DecRef( DPtrGDL* p)
    {
        SizeT nEl=p->N_Elements();
        for( SizeT ix=0; ix < nEl; ix++)
        {
            DPtr id= (*p)[ix];
            DecRef( id);
       }
    }
   static void DecRefObj( DObj id)
    {
        if( id != 0)
            {
#ifdef GDL_DEBUG_HEAP
std::cout << "-- <ObjHeapVar" << id << ">" << std::endl; 
#endif
                ObjHeapT::iterator it=objHeap.find( id);
                if( it != objHeap.end()) 
                    { 
                       if( (*it).second.Dec())
                           {
#ifdef GDL_DEBUG_HEAP
                               std::cout << "Out of scope (garbage collected): <ObjHeapVar" << id 
                                          << ">"
                                          << " at: " << callStack.back()->GetProName()
                                          << "  line: " << callStack.back()->GetLineNumber()
                                          << std::endl; 
#endif
                               callStack.back()->ObjCleanup( id);
//                             FreeObjHeapDirect( id, it);
                           }
#ifdef GDL_DEBUG_HEAP
                        else
std::cout << "<ObjHeapVar" << id << "> = " << (*it).second.Count() << std::endl; 
#endif
                        
                     }
            }
    }
   static void DecRefObj( DObjGDL* p)
    {
        SizeT nEl=p->N_Elements();
        for( SizeT ix=0; ix < nEl; ix++)
        {
            DObj id= (*p)[ix];
            DecRefObj( id);
       }
    }
   static void IncRef( DPtr id)
    {
        if( id != 0)
            {
#ifdef GDL_DEBUG_HEAP
std::cout << "++ <PtrHeapVar" << id << ">" << std::endl; 
#endif
                HeapT::iterator it=heap.find( id);
                if( it != heap.end()) 
                    { 
                        (*it).second.Inc(); 
#ifdef GDL_DEBUG_HEAP
std::cout << "<PtrHeapVar" << id << "> = " << (*it).second.Count() << std::endl; 
#endif
                    }
            }
    }
   static void AddRef( DPtr id, SizeT add)
    {
        if( id != 0)
            {
#ifdef GDL_DEBUG_HEAP
std::cout << add << " + <PtrHeapVar" << id << ">" << std::endl; 
#endif
                HeapT::iterator it=heap.find( id);
                if( it != heap.end()) 
                    { 
                        (*it).second.Add(add); 
                    }
            }
    }
   static void IncRef( DPtrGDL* p)
    {
        SizeT nEl=p->N_Elements();
        for( SizeT ix=0; ix < nEl; ix++)
        {
            DPtr id= (*p)[ix];
            IncRef( id);
       }
    }
   static void IncRefObj( DObj id)
    {
        if( id != 0)
            {
#ifdef GDL_DEBUG_HEAP
std::cout << "++ <ObjHeapVar" << id << ">" << std::endl; 
#endif
                ObjHeapT::iterator it=objHeap.find( id);
                if( it != objHeap.end()) 
                    { 
                        (*it).second.Inc(); 
                    }
            }
    }
   static void AddRefObj( DObj id, SizeT add)
    {
        if( id != 0)
            {
#ifdef GDL_DEBUG_HEAP
std::cout << add << " + <ObjHeapVar" << id << ">" << std::endl; 
#endif
                ObjHeapT::iterator it=objHeap.find( id);
                if( it != objHeap.end()) 
                    { 
                        (*it).second.Add(add); 
                    }
            }
    }
   static void IncRefObj( DObjGDL* p)
    {
        SizeT nEl=p->N_Elements();
        for( SizeT ix=0; ix < nEl; ix++)
        {
            DObj id= (*p)[ix];
            IncRefObj( id);
       }
    }

    class HeapException {};

    static BaseGDL*& GetHeap( DPtr ID)
    {
        HeapT::iterator it=heap.find( ID);
        if( it == heap.end()) throw HeapException();
        return it->second.get();
    }
    static DStructGDL*& GetObjHeap( DObj ID)
    {
        ObjHeapT::iterator it=objHeap.find( ID);
        if( it == objHeap.end()) throw HeapException();
        return it->second.get();
    }
    static DStructGDL*& GetObjHeap( DObj ID, ObjHeapT::iterator& it)
    {
//         ObjHeapT::iterator it=objHeap.find( ID);
        it=objHeap.find( ID);
        if( it == objHeap.end()) throw HeapException();
        return it->second.get();
    }

    static bool PtrValid( DPtr ID)
    {
        HeapT::iterator it=heap.find( ID);
        return  (it != heap.end());
    }

    static SizeT HeapSize()
    {
        return heap.size();
    }

    static DPtr FindInHeap( BaseGDL** p)
    {
        for( HeapT::iterator it=heap.begin(); it != heap.end(); ++it)
        {
            if( &it->second.get() == p)
                return it->first;
        }
        return 0;
    }
    static BaseGDL** GetPtrToHeap( BaseGDL* p)
    {
        for( HeapT::iterator it=heap.begin(); it != heap.end(); ++it)
        {
            if( it->second.get() == p)
                return &it->second.get();
        }
        return NULL;
    }
    static DPtrGDL* GetAllHeap()
    {
        SizeT nEl = heap.size();
        if( nEl == 0) return new DPtrGDL( 0);
        DPtrGDL* ret = new DPtrGDL( dimension( &nEl, 1), BaseGDL::NOZERO);
        SizeT i=0;
        for( HeapT::iterator it=heap.begin(); it != heap.end(); ++it)
        {
            IncRef( it->first);
            (*ret)[ i++] = it->first;
        }
        return ret;
    }

    // no ref counting here
    static std::vector<DPtr>* GetAllHeapSTL()
    {
        SizeT nEl = heap.size();
        if( nEl == 0) return new std::vector<DPtr>();
        std::vector<DPtr>* ret = new std::vector<DPtr>( nEl);
        SizeT i=0;
        for( HeapT::iterator it=heap.begin(); it != heap.end(); ++it)
        {
            (*ret)[ i++] = it->first;
        }
        return ret;
    }

    static bool ObjValid( DObj ID)
    {
        ObjHeapT::iterator it=objHeap.find( ID);
        return  (it != objHeap.end());
    }

    static SizeT ObjHeapSize()
    {
        return objHeap.size();
    }

//     static DObj FindInObjHeap( BaseGDL** p)
//     {
//         for( ObjHeapT::iterator it=objHeap.begin(); it != objHeap.end(); ++it)
//         {
//             if( &it->second == reinterpret_cast<DStructGDL**>(p))
//                 return it->first;
//         }
//         return 0;
//     }
    static DObjGDL* GetAllObjHeap()
    {
        SizeT nEl = objHeap.size();
        if( nEl == 0) return new DObjGDL( 0);
        DObjGDL* ret = new DObjGDL( dimension( &nEl, 1), BaseGDL::NOZERO);
        SizeT i=0;
        for( ObjHeapT::iterator it=objHeap.begin(); it != objHeap.end(); ++it)
        {
            IncRefObj( it->first);
            (*ret)[ i++] = it->first;
        }
        return ret;
    }

    // no ref counting here
    static std::vector<DObj>* GetAllObjHeapSTL()
    {
        SizeT nEl = objHeap.size();
        if( nEl == 0) return new std::vector<DObj>();
        std::vector<DObj>* ret = new std::vector<DObj>( nEl);
        SizeT i=0;
        for( ObjHeapT::iterator it=objHeap.begin(); it != objHeap.end(); ++it)
        {
            (*ret)[ i++] = it->first;
        }
        return ret;
    }

    // name of data
    static const std::string Name( BaseGDL* p) // const
    {
        return callStack.back()->GetString( p);
    }

    static const std::string Name( BaseGDL** p) // const
    {
        assert( *p == NULL);
        DPtr h = FindInHeap( p);
        if( h != 0) return std::string("<PtrHeapVar")+i2s(h)+">";
//         DObj o = FindInObjHeap( p);
//         if( o != 0) return std::string("<ObjHeapVar")+i2s(o)+">";
        return "<(ptr to undefined expression not found on the heap)>";
    }

    // compiler (lexer, parser, treeparser) def in dinterpreter.cpp
    static void ReportCompileError( GDLException& e, const std::string& file = "");

    // interpreter
    static void ReportError( GDLException& e, const std::string emsg, 
                             bool dumpStack=true)
    {
        DString msgPrefix = SysVar::MsgPrefix();

        std::cout << std::flush;
        if( dumpStack)
        if( e.Prefix())
        {
            std::cerr << msgPrefix << e.toString() << std::endl;
            lib::write_journal_comment(msgPrefix+e.toString());
        }
        else
        {
            std::cerr << e.toString() << std::endl;
            lib::write_journal_comment(e.toString());
        }

        std::cerr << msgPrefix << emsg << " " << 
        std::left << std::setw(16) << callStack.back()->GetProName();
        std::string file=callStack.back()->GetFilename();
        if( file != "")
        {
            SizeT line = e.getLine();
            if( line != 0)
            {       
                std::cerr << std::right << std::setw(6) << line;
            }
            else
            {
                std::cerr << std::right << std::setw(6) << "";
            }
            std::cerr << std::left << " " << file;
        }
        std::cerr << std::endl;
        
        if( dumpStack) DumpStack( emsg.size() + 1);
    }
    
    static void DumpStack( SizeT w)
    {
        DString msgPrefix = SysVar::MsgPrefix();

        EnvStackT::reverse_iterator upEnv = callStack.rbegin();
        //EnvStackT::reverse_iterator env = upEnv++;
        upEnv++;
        for(; 
            upEnv != callStack.rend();
            ++upEnv /*,++env*/)
        {
            std::cerr << msgPrefix << std::right << std::setw( w) << "";
            std::cerr << std::left << std::setw(16) << (*upEnv)->GetProName();

            std::string file = (*upEnv)->GetFilename();
            if( file != "")
            {              
//                 ProgNodeP cNode= (*env)->CallingNode();
//                 if( cNode != NULL)
//                 {       
//                     std::cerr << std::right << std::setw(6) << cNode->getLine();
//                 }
//                 else
//                 {
//                     std::cerr << std::right << std::setw(6) << "";
//                 }                

//                 ProgNodeP cNode= (*env)->CallingNode();
//                 if( cNode != NULL && cNode->getLine() != 0)
//                 {       
//                     (*upEnv)->SetLineNumber( cNode->getLine());
//                 }

                int lineNumber = (*upEnv)->GetLineNumber();
                if( lineNumber != 0)
                {       
                    std::cerr << std::right << std::setw(6) << lineNumber;
                }
                else
                {
                    std::cerr << std::right << std::setw(6) << "";
                }
                std::cerr << std::left << " " << file;
            }
            std::cerr << std::endl;
        }
    }

    static void DebugMsg( ProgNodeP _t, const std::string& msg)
    {    
        DString msgPrefix = SysVar::MsgPrefix();

        std::cout << std::flush;
        std::cerr << msgPrefix << msg
        << std::left << std::setw(16) << callStack.back()->GetProName();
        std::string file=callStack.back()->GetFilename();
        if( file != "")
        {
            ProgNodeP eNode = _t;
            if( eNode != NULL)
            {       
                std::cerr << std::right << std::setw(6) << eNode->getLine();
            }
            else
            {
                std::cerr << std::right << std::setw(6) << "";
            }
            std::cerr << std::left << " " << file;
        }
        std::cerr << std::endl;
    }

    static void RetAll( RetAllException::ExCode c=RetAllException::NONE)    
    {
        throw RetAllException( c);
    }

    static EnvStackT& CallStack() { return callStack;} // the callstack
    static EnvBaseT*  CallStackBack() { return callStack.back();} 
    
    std::string GetClearActualLine()
    {
        std::string ret = executeLine.str();
        executeLine.str("");
        return ret;
    }

    RetCode NewInterpreterInstance(SizeT lineOffset); // code in dinterpreter.cpp

    ~GDLInterpreter()
    {
    }
}

//***********************************************************************
// interpreter functions ************************************************
//***********************************************************************

// intercative usage
interactive returns[ RetCode retCode]
// {
// 	return interactive_statement_list(_t);
// }    
//     : retCode=statement_list
//     ;
// // used from interactive nulls line number
// interactive_statement_list returns[ RetCode retCode]
{
	for (; _t != NULL;) {

        //_t->setLine(0);
		retCode=statement(_t);
		_t = _retTree;
			
		if( retCode != RC_OK) break; // break out if non-regular
	}
	_retTree = _t;
	return retCode;
}
    : (retCode=statement
            {
                if( retCode != RC_OK) break; // break out if non-regular
            }
        )+
    ;

// execute statement
execute returns[ RetCode retCode]
{
//    RetCode retCode;
    ValueGuard<bool> guard( interruptEnable);
    interruptEnable = false;

	return statement_list(_t);
}
    : retCode=statement_list
    ;

// used to call functions
// same as statement list, but different behaviour for returncodes
call_fun returns[ BaseGDL* res]
{

    res = NULL;
    returnValue = NULL;
    RetCode retCode;

	for (; _t != NULL;) {

			retCode=statement(_t);
			
// 			if( retCode == RC_RETURN) 
			if( retCode >= RC_RETURN) 
			{
			res=returnValue;
			returnValue=NULL;
			
			break;
			}					

		_t = _retTree;
	}
	
	// default return value if none was set
	if( res == NULL) res = new DIntGDL( 0); 
	
	_retTree = _t;
	return res;
}
    : (retCode=statement
            {
//                if( retCode == RC_RETURN) 
                if( retCode >= RC_RETURN) 
                {
                    res=returnValue;
                    returnValue=NULL;
                    
                    break;
                }
            }
        )*
        {
            // default return value if none was set
            if( res == NULL) res = new DIntGDL( 0); 
        }
    ;

call_lfun returns[ BaseGDL** res]
{
    res = NULL;
    returnValueL = NULL;
    RetCode retCode;

	ProgNodeP in = _t;

	for (; _t != NULL;) {
			retCode=statement(_t);
			_t = _retTree;
			
//			if( retCode == RC_RETURN) 
			if( retCode >= RC_RETURN) 
			{
			res=returnValueL;
			returnValueL=NULL;
			break;
			}
			
	}
	
	// default return value if none was set
	if( res == NULL)
	throw GDLException( in, "Function "+
	callStack.back()->GetProName()+
	" must return a left-value in this context.",false,false);
	
	_retTree = _t;
	return res;
}
    : (retCode=statement
            {
//                if( retCode == RC_RETURN) 
                if( retCode >= RC_RETURN) 
                {
                    res=returnValueL;
                    returnValueL=NULL;
                    break;
                }
            }
        )*
        {
            // default return value if none was set
            if( res == NULL)
            throw GDLException( call_lfun_AST_in, "Function "+
                callStack.back()->GetProName()+
                " must return a left-value in this context.",false,false);
        }
    ;

// used to call procedures
call_pro
{
    RetCode retCode;

	for (; _t != NULL;) {
			retCode=statement(_t);
			_t = _retTree;
			
			// added RC_ABORT here
			if( retCode >= RC_RETURN) break;
	}
	_retTree = _t;
    return;
}
    : (retCode=statement
            {
                // added RC_ABORT here
                if( retCode >= RC_RETURN) break;
            }
        )*
    ;


// used on many occasions
statement_list returns[ RetCode retCode]
{
	for (; _t != NULL;) {

		retCode=statement(_t);
		_t = _retTree;
			
		if( retCode != RC_OK) break; // break out if non-regular
	}
	_retTree = _t;
	return retCode;
}
    : (retCode=statement
            {
                if( retCode != RC_OK) break; // break out if non-regular
            }
        )+
    ;

statement returns[ RetCode retCode]
{
//    ProgNodeP& actPos = statement_AST_in;
    assert( _t != NULL);
    ProgNodeP last;
    _retTree = _t;
//  if( callStack.back()->GetLineNumber() == 0) 
//  if( _t->getLine() != 0) 
//      callStack.back()->SetLineNumber( _t->getLine());
}
	:  
        {
            do {
//                 if( _t->getLine() != 0) 
//                     callStack.back()->SetLineNumber( _t->getLine());
                
                last = _retTree;

                // track actual line number
                callStack.back()->SetLineNumber( last->getLine());

                retCode = last->Run(); // Run() sets _retTree
                        
            }
            while( 
                    _retTree != NULL && 
                    retCode == RC_OK && 
                    !(sigControlC && interruptEnable) && 
                    (debugMode == DEBUG_CLEAR));

            if( _retTree != NULL) 
                last = _retTree;

            goto afterStatement;
}
        (
            // note: assignment must take care to update the owner of lvalue
            // a real copy must be performed (creating a new BaseGDL)  
            ASSIGN            
//            {a->Run();}
        |   ASSIGN_ARRAYEXPR_MFCALL
//            {ac->Run();}
        |   ASSIGN_REPLACE            
//            {ar->Run();}
            //            assignment
        |   PCALL_LIB
//            {pl->Run();}
        |   MPCALL
//            {m->Run();}
        |   MPCALL_PARENT
//            {mp->Run();}
        |   PCALL
//            {p->Run();}
            // procedure_call
            //        |   lib_procedure_call
        |   DEC
//            {d->Run();}
        |   INC
//            {in->Run();}
            // decinc_statement
        |   FOR
//            {f->Run(); _t = _retTree;}
        |   FOR_LOOP
//            {fl->Run(); _t = _retTree;}
        |   FOREACH
//            {fe->Run(); _t = _retTree;}
        |   FOREACH_LOOP
//            {fel->Run(); _t = _retTree;}
        |   FOREACH_INDEX
//            {fe->Run(); _t = _retTree;}
        |   FOREACH_INDEX_LOOP
//            {fel->Run(); _t = _retTree;}
        |   FOR_STEP
//            {fs->Run(); _t = _retTree;}
        |   FOR_STEP_LOOP
//            {fsl->Run(); _t = _retTree;}
        |   REPEAT
//            {r->Run(); _t = _retTree;}
        |   REPEAT_LOOP
//            {rl->Run(); _t = _retTree;}
        |   WHILE
//            {w->Run(); _t = _retTree;}
        |   IF
//            {i->Run(); _t = _retTree;}
        |   IF_ELSE
//            {ie->Run(); _t = _retTree;}
        |   CASE
//            {c->Run(); _t = _retTree;}
        |   SWITCH
//            {s->Run(); _t = _retTree;}
        |   BLOCK
//            {b->Run(); _t = _retTree;}
//        |   retCode=jump_statement
        |   LABEL
//            {l->Run(); _t = _retTree;}
        |   ON_IOERROR_NULL
//            {on->Run(); _t = _retTree;}
        |   ON_IOERROR
//            {o->Run(); _t = _retTree;}
        |   BREAK
//            {br->Run(); _t = _retTree;}
        |   CONTINUE
//            {co->Run(); _t = _retTree;}
        |   GOTO
//            {g->Run(); _t = _retTree;}
        |   RETF 
//             ( { !static_cast<EnvUDT*>(callStack.back())->LFun()}? e=expr // expr -> r value
//                 {
//                     delete returnValue;
//                     returnValue=e;
//                     retCode=RC_RETURN;
//                     callStack.back()->RemoveLoc( e); // steal e from local list
//                 }
//             | eL=l_ret_expr
//                 {
//                     // returnValueL is otherwise owned
//                     returnValueL=eL;
//                     retCode=RC_RETURN;
//                 }
//             )
//             ) 
        | RETP
//         {
//             retCode=RC_RETURN;
//         }
        )

        // control-c and debugging
        {
           afterStatement:;

           // possible optimization: make sigControlC a debugMode 
           if( interruptEnable && sigControlC)
            {
                DebugMsg( last, "Interrupted at: "); 

                sigControlC = false;

                retCode = NewInterpreterInstance( last->getLine());//-1);
            }
           else if( debugMode != DEBUG_CLEAR)
            {
                if( debugMode == DEBUG_STOP)
                {
                    DebugMsg( last, "Stop encountered: ");
                    if( !interruptEnable)
                        debugMode = DEBUG_PROCESS_STOP;
                }

                if( debugMode == DEBUG_STEP)
                    {
                        if( stepCount == 1)
                            {
                                stepCount = 0;
                                DebugMsg( last, "Stepped to: ");
                                
                                debugMode = DEBUG_CLEAR;
                
                                retCode = NewInterpreterInstance( last->getLine());//-1);
                            }
                        else
                            {
                            --stepCount;
#ifdef GDL_DEBUG
                            std::cout << "stepCount-- = " << stepCount << std::endl;
#endif
                            }
                    }
                else    
                if( interruptEnable)
                {
                    if( debugMode == DEBUG_PROCESS_STOP)
                    {
                        DebugMsg( last, "Stepped to: ");
                    }

                    debugMode = DEBUG_CLEAR;
                
                    retCode = NewInterpreterInstance( last->getLine());//-1);
                }   
                else
                {
                    retCode = RC_ABORT;
                }
            }
           return retCode;
        }
	;
    exception 
    catch [ GDLException& e] 
    { 
        if( dynamic_cast< GDLIOException*>( &e) != NULL)
            {
                // set the jump target - also logs the jump
                ProgNodeP onIOErr = 
                    static_cast<EnvUDT*>(callStack.back())->GetIOError();
                if( onIOErr != NULL)
                    {
                        SysVar::SetErr_String( e.getMessage());

                        _retTree = onIOErr;
                        return RC_OK;
                    }
            }

        EnvUDT* targetEnv = e.GetTargetEnv();
        if( targetEnv == NULL)
        {
            // initial exception, set target env
            // look if ON_ERROR is set somewhere
            for( EnvStackT::reverse_iterator i = callStack.rbegin();
                i != callStack.rend(); ++i)
            {
                DLong oE = -1;
                EnvUDT* envUD = dynamic_cast<EnvUDT*>(*i);
                if( envUD != NULL)
                    oE = envUD->GetOnError();
                
                if( oE != -1) 
                { // oE was set
                    
                    // 0 -> stop here
                    if( oE == 0) 
                    targetEnv = static_cast<EnvUDT*>(callStack.back()); 
                    // 1 -> $MAIN$
                    else if( oE == 1) 
                    {
                        EnvUDT* cS_begin = 
                        static_cast<EnvUDT*>(*callStack.begin());
                        targetEnv = cS_begin;  
                    }
                    // 2 -> caller of routine which called ON_ERROR
                    else if( oE == 2)
                    {
                        // set to caller, handle nested
                        while( static_cast<EnvUDT*>(*(++i))->GetOnError() == 2 
                               && i != callStack.rend());

                        if( i == callStack.rend())
                        {
                            EnvUDT* cS_begin = 
                            static_cast<EnvUDT*>(*callStack.begin());
                            targetEnv = cS_begin;
                        }
                        else
                        {
                            EnvUDT* iUDT = static_cast<EnvUDT*>(*i);
                            targetEnv = iUDT;
                        }
                    }   
                    // 3 -> routine which called ON_ERROR
                    else if( oE == 3)
                    {
                        EnvUDT* iUDT = static_cast<EnvUDT*>(*i);
                        targetEnv = iUDT;
                    }
                    
                    
                    // State where error occured
//                     if( e.getLine() == 0 && _t != NULL)
//                         e.SetLine( _t->getLine());
//                     if( e.getLine() == 0 && _retTree != NULL)
//                         e.SetLine( _retTree->getLine());
                    if( e.getLine() == 0 && last != NULL)
                        e.SetLine( last->getLine());

                    if( interruptEnable)
                        ReportError(e, "Error occurred at:");

                    // remeber where to stop
                    e.SetTargetEnv( targetEnv);
                    
                    if( targetEnv->GetLineNumber() != 0)
                        e.SetLine( targetEnv->GetLineNumber());                    

//                     ProgNodeP errorNodeP = targetEnv->CallingNode();
//                     e.SetErrorNodeP( errorNodeP);

                    // break on first occurence of set oE
                    break;
                }
            }
        }
        
        if( targetEnv != NULL && targetEnv != callStack.back())
        {
            throw e; // rethrow
        }
        lib::write_journal( GetClearActualLine());

        // many low level routines don't have errorNode info
        // set line number here in this case
//         if( e.getLine() == 0 && _t != NULL)
//             e.SetLine( _t->getLine());
//         if( e.getLine() == 0 && _retTree != NULL)
//             e.SetLine( _retTree->getLine());
//        if( e.getLine() == 0 && actPos != NULL)
//            e.SetLine( actPos->getLine());

        if( interruptEnable)
            {
                // tell where we are
                ReportError(e, "Execution halted at:", targetEnv == NULL); 

                retCode = NewInterpreterInstance(e.getLine());//-1);
            }    
        else
            {

                DString msgPrefix = SysVar::MsgPrefix();
                if( e.Prefix())
                    {
                        std::cerr << msgPrefix << e.toString() << std::endl;
                        lib::write_journal_comment(msgPrefix+e.toString());
                    }
                else
                    {
                        std::cerr << e.toString() << std::endl;
                        lib::write_journal_comment(e.toString());
                    }

                retCode = RC_ABORT;
            }

        return retCode;
    }

block returns[ RetCode retCode]
{
	match(antlr::RefAST(_t),BLOCK);
	_retTree = _t->getFirstChild();
    return RC_OK;
//     retCode = RC_OK;

// 	ProgNodeP block = _t;
// 	match(antlr::RefAST(_t),BLOCK);
// 	_t = _t->getFirstChild();
// 	if (_t != NULL)
// 		{

//             SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

//             retCode=statement_list(_t);

//             if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                 !block->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                 {
//                     // a jump (goto) occured out of this block
//                     return retCode;
//                 }
// 		}
// 	_retTree = block->getNextSibling();
// 	return retCode;
}
	: #(BLOCK (retCode=statement_list)?)
	;

switch_statement returns[ RetCode retCode]
{
    BaseGDL* e;
    retCode = RC_OK; // not set if no branch is executed
}
	: #(s:SWITCH e=expr 
            {
                auto_ptr<BaseGDL> e_guard(e);
                
//                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                ProgNodeP b=_t; // remeber block begin (block)

                bool hook=false; // switch executes everything after 1st match
                for( int i=0; i<s->numBranch; i++)
                {
                    if( b->getType() == ELSEBLK)
                    {
                        hook=true;

                        ProgNodeP sL = b->GetFirstChild(); // statement_list

                        if(sL != NULL )
                        {
                            _retTree = sL;
                            return RC_OK;
                        }

//                             // statement there
//                             retCode=statement_list( sL);
//                             if( retCode == RC_BREAK) 
//                             {
//                                 retCode = RC_OK;    
//                                 break;          // break
//                             }
//                             if( retCode >= RC_RETURN) break; // goto

//                             if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                                 !s->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                             {
//                                 // a jump (goto) occured out of this loop
//                                 return retCode;
//                             }

//                         }
                    }
                    else
                    {
                        ProgNodeP ex = b->GetFirstChild();  // EXPR
                        ProgNodeP bb = ex->GetNextSibling(); // statement_list

                        if( !hook)
                        {
//                            RefDNode ee_ = _t->GetFirstChild(); // expr

                            BaseGDL* ee=expr(ex);
                            // auto_ptr<BaseGDL> ee_guard(ee);

                            hook=e->Equal(ee); // Equal deletes ee
                        }
                            
                        if( hook)
                        {
                            // statement there
                            if(bb != NULL )
                                {
                                    _retTree = bb;
                                    return RC_OK;
                                }
                 
//                             _retTree = // find first non empty
//                             return RC_OK;
//                                 }

//                             retCode=statement_list(bb);
//                             if( retCode == RC_BREAK) 
//                             {
//                                 retCode = RC_OK;    
//                                 break;          // break
//                             }
//                             if( retCode >= RC_RETURN) break; // goto

//                             if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                                 !s->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                             {
//                                 // a jump (goto) occured out of this loop
//                                 return retCode;
//                             }
                        }
                        
                    }
                    b=b->GetNextSibling(); // next block
                }
                _retTree = s->GetNextSibling();
                return RC_OK;
                // finish or break
//                retCode=RC_OK; // clear RC_BREAK retCode
            }
        )
    ;

case_statement returns[ RetCode retCode]
{
    BaseGDL* e;
    retCode = RC_OK; // not set if no branch is executed
}
	: #(c:CASE e=expr
            {
                auto_ptr<BaseGDL> e_guard(e);

//                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                if( !e->Scalar())
                throw GDLException( _t, "Expression must be a"
                    " scalar in this context: "+Name(e),true,false);

                ProgNodeP b=_t; // remeber block begin

                for( int i=0; i<c->numBranch; ++i)
                {
                    if( b->getType() == ELSEBLK)
                    {
                        ProgNodeP sL = b->GetFirstChild(); // statement_list

                        if(sL != NULL )
                        {
                            _retTree = sL;
                            return RC_OK;
                        }
                        else
                        {
                            _retTree = c->GetNextSibling();
                            return RC_OK;
                        }

//                             // statement there
//                             retCode=statement_list(sL);
//                             //if( retCode == RC_BREAK) break; // break anyway
// //                            if( retCode >= RC_RETURN) return retCode; 
//                             if( retCode >= RC_RETURN) break;
                            
//                             if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                                 !c->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                             {
//                                 // a jump (goto) occured out of this loop
//                                 return retCode;
//                             }

//                         }
//                         retCode = RC_OK;
//                         break;
                    }
                    else
                    {
                        ProgNodeP ex = b->GetFirstChild();  // EXPR
                        ProgNodeP bb = ex->GetNextSibling(); // statement_list

                        BaseGDL* ee=expr(ex);
                        // auto_ptr<BaseGDL> ee_guard(ee);

                        bool equalexpr=e->Equal(ee); // Equal deletes ee

                        if( equalexpr)
                        {
                            if(bb != NULL )
                                {
                                    _retTree = bb;
                                    return RC_OK;
                                }
                            else
                                {
                                    _retTree = c->GetNextSibling();
                                    return RC_OK;
                                }
//                             if(bb != NULL)
//                             {
//                                 // statement there
//                                 retCode=statement_list(bb);
//                                 //if( retCode == RC_BREAK) break; // break anyway
// //                                if( retCode >= RC_RETURN) return retCode;
//                                 if( retCode >= RC_RETURN) break;

//                                 if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                                     !c->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                                 {
//                                     // a jump (goto) occured out of this loop
//                                     return retCode;
//                                 }

//                             }
//                             retCode = RC_OK;
//                             break;
                        }
                        
                    }
                    b=b->GetNextSibling(); // next block
                } // for
                // finish or break
//                retCode=RC_OK; // clear RC_BREAK retCode
                throw GDLException( c, "CASE statement found no match.",true,false);
            }

        )
	;



repeat_statement returns[ RetCode retCode]
	: #(r:REPEAT // block expr
            {
                // _t is REPEAT_LOOP, GetFirstChild() is expr, GetNextSibling is first loop statement
               _retTree = _t->GetFirstChild()->GetNextSibling();     // statement
               return RC_OK;
            }
        )
	;

repeat_loop_statement returns[ RetCode retCode]
// {
//     retCode = RC_OK; // not set if no branch is executed
// }
	: #(r:REPEAT_LOOP // block expr
            {
                auto_ptr<BaseGDL> eVal( expr(_t));
                if( eVal.get()->False())
                    {
                        _retTree = _t->GetNextSibling();     // 1st loop statement
                        if( _retTree == NULL)
                            throw GDLException(r,
                                               "Empty REPEAT loop entered (infinite loop).",
                                               true,false);
                        return RC_OK;
                    }
                else
                    {
                        _retTree = r->GetNextSibling();     // statement
                        return RC_OK;
                    }

//                 retCode=RC_OK; // clear RC_BREAK/RC_CONTINUE retCode
//                 SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

//                 // remember block and expr nodes
//                 ProgNodeP e =_t;
//                 ProgNodeP bb  = e->GetNextSibling();

// //                 ProgNodeP bb =_t;
// //                 ProgNodeP e  = bb->GetNextSibling();
// //                 bb = bb->GetFirstChild();
                
//                 auto_ptr<BaseGDL> eVal;
//                 do {
//                     if( bb != NULL)
//                     {
//                     retCode=statement_list(bb);

//                     if( retCode == RC_CONTINUE)
//                                 {
//                                 retCode = RC_OK;
//                                 continue;  
//                                 }
//                     if( retCode == RC_BREAK) 
//                     {
//                         retCode = RC_OK;
//                         break;        
//                     }
//                     if( retCode >= RC_RETURN) break;
//                     // if( retCode == RC_BREAK) break;        
//                     // if( retCode >= RC_RETURN) return retCode;

//                     if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                         !r->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                     {
//                         // a jump (goto) occured out of this loop
//                         return retCode;
//                     }
//                     }
//                     eVal.reset( expr(e));
//                 } while( eVal.get()->False());
                
//                 // retCode=RC_OK; // clear RC_BREAK/RC_CONTINUE retCode
            }
        )
	;

while_statement returns[ RetCode retCode]
{
    retCode = RC_OK;
}
	: #(w:WHILE // statement expr 
            {
//                 SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

//                 ProgNodeP e = _t; //->GetFirstChild();  // expr

                auto_ptr<BaseGDL> eVal( expr( _t));
                if( eVal.get()->True()) 
                    {
                        _retTree = _t->GetNextSibling();     // 1st loop statement
                        if( _retTree == NULL)
                            throw GDLException(w,
                                               "Empty WHILE loop entered (infinite loop).",
                                               true,false);
                        return RC_OK;
                    }
                else
                    {
                        _retTree = w->GetNextSibling();     // statement
                        return RC_OK;
                    }

//                 auto_ptr< BaseGDL> eVal( expr( e));
//                 while( eVal.get()->True()) {
//                     retCode=statement_list(s);

//                     if( retCode == RC_CONTINUE) 
//                                 {
//                                 retCode = RC_OK;
//                                 continue;  
//                                 }
//                     if( retCode == RC_BREAK) 
//                     {
//                         retCode = RC_OK;
//                         break;        
//                     }
//                     if( retCode >= RC_RETURN) break;
                    
//                     if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                         !w->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                     {
//                         // a jump (goto) occured out of this loop
//                         return retCode;
//                     }

//                     eVal.reset( expr( e));
//                 } 

                // retCode=RC_OK; // clear RC_BREAK/RC_CONTINUE retCode
            }
        )
	;

for_statement returns[ RetCode retCode]
{
    BaseGDL** v;
    BaseGDL* s;
    BaseGDL* e;
    BaseGDL* st;
    retCode = RC_OK;
}
    : #(f:FOR // #(FOR_LOOP (VAR|VARPTR) expr expr ...) 
            {
                _t = f->GetNextSibling()->GetFirstChild();

                EnvUDT* callStack_back = 
                static_cast<EnvUDT*>(callStack.back());

                ForLoopInfoT& loopInfo = 
                    callStack_back->GetForLoopInfo( f->forLoopIx);

                v=l_simple_var(_t);
                _t = _retTree;

                s=expr(_t);
                auto_ptr<BaseGDL> s_guard(s);
                _t = _retTree;

                delete loopInfo.endLoopVar;
                loopInfo.endLoopVar=expr(_t);

                ProgNodeP b = _retTree;
		
                s->ForCheck( &loopInfo.endLoopVar);

                // ASSIGNMENT used here also
                delete (*v);
                (*v)= s_guard.release(); // s held in *v after this

                if( (*v)->ForCondUp( loopInfo.endLoopVar))
                {
                    _retTree = b;
                    return RC_OK;
                }
                else
                {
                    // skip if initial test fails
                    _retTree = f->GetNextSibling()->GetNextSibling();
                    return RC_OK;
                }
            }
        ) 
    | #(fl:FOR_LOOP// (VAR|VARPTR) expr expr 
            {
                EnvUDT* callStack_back = 
                static_cast<EnvUDT*>(callStack.back());

                ForLoopInfoT& loopInfo = 
                    callStack_back->GetForLoopInfo( fl->forLoopIx);

                if( loopInfo.endLoopVar == NULL)
                    {
                        // non-initialized loop (GOTO)
                        _retTree = fl->GetNextSibling();
                        return RC_OK;
                    }

// // problem:
// // EXECUTE may call DataListT.loc.resize(), as v points to the
// // old sequence v might be invalidated -> segfault
// // note that the value (*v) is preserved by resize()

                v=l_simple_var(_t);
                _t = _retTree;

                (*v)->ForAdd();
                if( (*v)->ForCondUp( loopInfo.endLoopVar))
                {
                    _retTree = _t->GetNextSibling()->GetNextSibling();
                    return RC_OK;
                }
                else
                {
                    delete loopInfo.endLoopVar;
                    loopInfo.endLoopVar = NULL;
                    _retTree = fl->GetNextSibling();
                    return RC_OK;
                }

            }
        )
//                 for((*v)=s; (*v)->ForCondUp( e); 
//                     v=l_simple_var( sv), (*v)->ForAdd()) 
//                 {
// //                    retCode=block(b);
//                     if( b != NULL)
//                     {
//                         retCode=statement_list(b);
                    
//                         if( retCode != RC_OK) // optimization
//                         {
//                             if( retCode == RC_CONTINUE) 
//                                 {
//                                 retCode = RC_OK;
//                                 continue;  
//                                 }
//                             if( retCode == RC_BREAK) 
//                             {
//                                 retCode = RC_OK;
//                                 break;        
//                             }
//                             if( retCode >= RC_RETURN) break;
//                         }

//                         if( (callStack_back->NJump() != nJump) &&
//                             !f->LabelInRange( callStack_back->LastJump()))
//                         {
//                             // a jump (goto) occured out of this loop
//                             return retCode;
//                         }
//                     }
//                 }
// //                retCode=RC_OK; // clear RC_BREAK/RC_CONTINUE retCode
//             }
//         )
    | #(fs:FOR_STEP // (VAR|VARPTR) expr expr expr 
            {
                _t = fs->GetNextSibling()->GetFirstChild();

                EnvUDT* callStack_back = 
                static_cast<EnvUDT*>(callStack.back());

                ForLoopInfoT& loopInfo = 
                    callStack_back->GetForLoopInfo( fs->forLoopIx);

                v=l_simple_var(_t);
                _t = _retTree;

                s=expr(_t);
                auto_ptr<BaseGDL> s_guard(s);
                _t = _retTree;

                delete loopInfo.endLoopVar;
                loopInfo.endLoopVar=expr(_t);
                _t = _retTree;

                delete loopInfo.loopStepVar;
                loopInfo.loopStepVar=expr(_t);

                ProgNodeP b = _retTree;
		
                s->ForCheck( &loopInfo.endLoopVar, &loopInfo.loopStepVar);

                // ASSIGNMENT used here also
                delete (*v);
                (*v)= s_guard.release(); // s held in *v after this

                if( loopInfo.loopStepVar->Sgn() == -1) 
                    {
                    if( (*v)->ForCondDown( loopInfo.endLoopVar))
                        {
                            _retTree = b;
                            return RC_OK;
                        }
                    }
                else
                    {
                    if( (*v)->ForCondUp( loopInfo.endLoopVar))
                        {
                            _retTree = b;
                            return RC_OK;
                        }
                    }

                // skip if initial test fails
                _retTree = f->GetNextSibling()->GetNextSibling();
                return RC_OK;
            }
        )
    | #(fsl:FOR_STEP_LOOP // (VAR|VARPTR) expr expr expr 
            {
                EnvUDT* callStack_back = 
                static_cast<EnvUDT*>(callStack.back());

                ForLoopInfoT& loopInfo = 
                    callStack_back->GetForLoopInfo( fsl->forLoopIx);

                if( loopInfo.endLoopVar == NULL)
                    {
                        // non-initialized loop (GOTO)
                        _retTree = fsl->GetNextSibling();
                        return RC_OK;
                    }

// // problem:
// // EXECUTE may call DataListT.loc.resize(), as v points to the
// // old sequence v might be invalidated -> segfault
// // note that the value (*v) is preserved by resize()

                v=l_simple_var(_t);
                _t = _retTree;

                ProgNodeP b = _t->GetNextSibling()->GetNextSibling()->GetNextSibling();

                (*v)->ForAdd(loopInfo.loopStepVar);
                if( loopInfo.loopStepVar->Sgn() == -1) 
                    {
                    if( (*v)->ForCondDown( loopInfo.endLoopVar))
                        {
                            _retTree = b;
                            return RC_OK;
                        }
                    }
                else
                    {
                    if( (*v)->ForCondUp( loopInfo.endLoopVar))
                        {
                            _retTree = b;
                            return RC_OK;
                        }
                    }

               delete loopInfo.endLoopVar;
               loopInfo.endLoopVar = NULL;
               delete loopInfo.loopStepVar;
               loopInfo.loopStepVar = NULL;
               _retTree = fsl->GetNextSibling();
               return RC_OK;

            }
        )
//             {
//                 ProgNodeP sv = _t;
//             }
//             v=l_simple_var
//             s=expr e=expr st=expr
//             {
//                 auto_ptr<BaseGDL> s_guard(s);
//                 auto_ptr<BaseGDL> e_guard(e);
//                 auto_ptr<BaseGDL> st_guard(st);

//                 SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

//                 s->ForCheck( &e, &st);
//                 e_guard.release();
//                 e_guard.reset(e);
//                 st_guard.release();
//                 st_guard.reset(st);
                
//                 ProgNodeP bs=_t;
                
//                 // ASSIGNMENT used here also
//                 delete (*v);
                
//                 if( st->Sgn() == -1) 
//                 {
//                     s_guard.release();
//                     for((*v)=s; (*v)->ForCondDown( e); 
//                         v=l_simple_var( sv), (*v)->ForAdd(st))
//                     {
//                         if( bs != NULL)
//                         {
//                             retCode=statement_list(bs);
                            
//                             if( retCode == RC_CONTINUE)
//                                 {
//                                 retCode = RC_OK;
//                                 continue;  
//                                 }
//                             if( retCode == RC_BREAK) 
//                             {
//                                 retCode = RC_OK;
//                                 break;        
//                             }
//                             if( retCode >= RC_RETURN) break;
                            
//                             if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                                 !fs->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                             {
//                                 // a jump (goto) occured out of this loop
//                                 return retCode;
//                             }
//                         }
//                     }
//                 } 
//                 else
//                 {
//                     s_guard.release();
//                     for((*v)=s; (*v)->ForCondUp( e);
//                         v=l_simple_var( sv), (*v)->ForAdd(st))
//                         {
//                         if( bs != NULL)
//                         {
//                             retCode=statement_list(bs);
                        
//                             if( retCode == RC_CONTINUE)
//                                 {
//                                 retCode = RC_OK;
//                                 continue;  
//                                 }

//                             if( retCode == RC_BREAK) 
//                             {
//                                 retCode = RC_OK;
//                                 break;        
//                             }
//                             if( retCode >= RC_RETURN) break;
                            
//                             if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                                 !fs->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                             {
//                                 // a jump (goto) occured out of this loop
//                                 return retCode;
//                             }
//                         }
//                     }
//                 }
//             }
//         )
	;



foreach_statement returns[ RetCode retCode]
{
    BaseGDL** v; // loop variable
    BaseGDL** l; // loop index (or hash) variable
    BaseGDL* s;
    retCode = RC_OK;
}
    : #(f:FOREACH // #(FOREACH_LOOP (VAR|VARPTR) expr ...)
            {
                _t = f->GetNextSibling()->GetFirstChild();

                EnvUDT* callStack_back = 
                static_cast<EnvUDT*>(callStack.back());

                ForLoopInfoT& loopInfo = 
                    callStack_back->GetForLoopInfo( f->forLoopIx);

                v=l_simple_var(_t);
                _t = _retTree;

                delete loopInfo.endLoopVar;
                loopInfo.endLoopVar=expr(_t);

                ProgNodeP b = _retTree;
		
                loopInfo.foreachIx = 0;

                // currently there are no empty arrays
                //SizeT nEl = loopInfo.endLoopVar->N_Elements();

                // ASSIGNMENT used here also
                delete (*v);
                (*v) = loopInfo.endLoopVar->NewIx( 0);

                _retTree = b;
                return RC_OK;
            }
        )
    | #(fl:FOREACH_LOOP // (VAR|VARPTR) expr 
            {
                EnvUDT* callStack_back = 
                static_cast<EnvUDT*>(callStack.back());

                ForLoopInfoT& loopInfo = 
                    callStack_back->GetForLoopInfo( fl->forLoopIx);

                if( loopInfo.endLoopVar == NULL)
                    {
                        // non-initialized loop (GOTO)
                        _retTree = fl->GetNextSibling();
                        return RC_OK;
                    }

                v=l_simple_var(_t);
                _t = _retTree;

                // skip expr
                _t = _t->getNextSibling();

                ++loopInfo.foreachIx;

                SizeT nEl = loopInfo.endLoopVar->N_Elements();

                if( loopInfo.foreachIx < nEl)
                    {
                        // ASSIGNMENT used here also
                        delete (*v);
                        (*v) = loopInfo.endLoopVar->NewIx( loopInfo.foreachIx);

                        _retTree = _t;
                        return RC_OK;
                    }

                delete loopInfo.endLoopVar;
                loopInfo.endLoopVar = NULL;
                loopInfo.foreachIx = -1;
                _retTree = fl->GetNextSibling();
                return RC_OK;
            }
        )
    | #(fi:FOREACH_INDEX // #(FOREACH_INDEX_LOOP (VAR|VARPTR) expr (VAR|VARPTR) ...)
            {
                _t = fi->GetNextSibling()->GetFirstChild();

                EnvUDT* callStack_back = 
                static_cast<EnvUDT*>(callStack.back());

                ForLoopInfoT& loopInfo = 
                    callStack_back->GetForLoopInfo( fi->forLoopIx);

                v=l_simple_var(_t);
                _t = _retTree;

                delete loopInfo.endLoopVar;
                loopInfo.endLoopVar=expr(_t);

                // loop index (or hash) variable
                l=l_simple_var(_t);
                _t = _retTree;

                ProgNodeP b = _retTree;
		
                loopInfo.foreachIx = 0;

                // currently there are no empty arrays
                //SizeT nEl = loopInfo.endLoopVar->N_Elements();

                // ASSIGNMENT used here also
                delete (*v);
                (*v) = loopInfo.endLoopVar->NewIx( 0);

                _retTree = b;
                return RC_OK;
            }
        )
    | #(fil:FOREACH_INDEX_LOOP // (VAR|VARPTR) expr (VAR|VARPTR)
            {
                EnvUDT* callStack_back = 
                static_cast<EnvUDT*>(callStack.back());

                ForLoopInfoT& loopInfo = 
                    callStack_back->GetForLoopInfo( fil->forLoopIx);

                if( loopInfo.endLoopVar == NULL)
                    {
                        // non-initialized loop (GOTO)
                        _retTree = fil->GetNextSibling();
                        return RC_OK;
                    }

                v=l_simple_var(_t);
                _t = _retTree;

                // skip expr
                _t = _t->getNextSibling();

                // loop index variable
                l=l_simple_var(_t);
                _t = _retTree;

                ++loopInfo.foreachIx;

                SizeT nEl = loopInfo.endLoopVar->N_Elements();

                if( loopInfo.foreachIx < nEl)
                    {
                        // ASSIGNMENT used here also
                        delete (*v);
                        (*v) = loopInfo.endLoopVar->NewIx( loopInfo.foreachIx);

                        _retTree = _t;
                        return RC_OK;
                    }

                delete loopInfo.endLoopVar;
                loopInfo.endLoopVar = NULL;
                loopInfo.foreachIx = -1;
                _retTree = fil->GetNextSibling();
                return RC_OK;
            }
        )
// //          always copies (owned by caller)
//             s=expr
//             {
//                 EnvUDT* callStack_back = 
//                 static_cast<EnvUDT*>(callStack.back());
//                 SizeT nJump = callStack_back->NJump();

//                 ProgNodeP b=_t; //->getFirstChild();
                
//                 // ASSIGNMENT used here also


//                 SizeT nEl = s->N_Elements();
// // problem:
// // EXECUTE may call DataListT.loc.resize(), as v points to the
// // old sequence v might be invalidated -> segfault
// // note that the value (*v) is preserved by resize()
//                 for( SizeT i=0; i<nEl; ++i)
//                 {
// //                  retCode=block(b);
//                     v=l_simple_var( sv);
//                     delete (*v); 
//                     (*v) = s->NewIx( i);

//                     if( b != NULL)
//                     {
//                         retCode=statement_list(b);
                    
//                         if( retCode != RC_OK) // optimization
//                         {
//                             if( retCode == RC_CONTINUE) 
//                                 {
//                                 retCode = RC_OK;
//                                 continue;  
//                                 }
//                             if( retCode == RC_BREAK) 
//                             {
//                                 retCode = RC_OK;
//                                 break;        
//                             }
//                             if( retCode >= RC_RETURN) break;
//                         }

//                         if( (callStack_back->NJump() != nJump) &&
//                             !f->LabelInRange( callStack_back->LastJump()))
//                         {
//                             // a jump (goto) occured out of this loop
//                             return retCode;
//                         }
//                     }
//                 }
// //                retCode=RC_OK; // clear RC_BREAK/RC_CONTINUE retCode
//             }
//         )
    ;



if_statement returns[ RetCode retCode]
{
    BaseGDL* e;
//    retCode = RC_OK; // not set if not executed
}
	: #(i:IF e=expr
            { 
                auto_ptr<BaseGDL> e_guard(e);

//                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                if( e->True())
                {
                   _retTree = _t;
                   return RC_OK;
                }

                _retTree = i->GetNextSibling();
                return RC_OK;

//                     retCode=statement(_t);
// //                     if( retCode != RC_OK) return retCode;

//                         if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                             !i->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                         {
//                             // a jump (goto) occured out of this loop
//                             return retCode;
//                         }
            }
        )
	;   

if_else_statement returns[ RetCode retCode]
{
    BaseGDL* e;
    retCode = RC_OK; // not set if not executed
}
	: #(i:IF_ELSE e=expr
            { 
                auto_ptr<BaseGDL> e_guard(e);

//                SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

                if( e->True())
                {
                   _retTree = _t->GetFirstChild();
                   return RC_OK;
                }

                _retTree = _t->GetNextSibling();
                return RC_OK;
//             { 
//                 auto_ptr<BaseGDL> e_guard(e);

//                 SizeT nJump = static_cast<EnvUDT*>(callStack.back())->NJump();

//                 if( e->True())
//                 {
//                     retCode=statement(_t);
// // //                    if( retCode != RC_OK) return retCode;

//                     if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                         !i->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                     {
//                         // a jump (goto) occured out of this loop
//                         return retCode;
//                     }
//                 }
//                 else
//                 {
//                     _t=_t->GetNextSibling(); // jump over 1st statement
//                     retCode=statement(_t);
// // //                   if( retCode != RC_OK) return retCode;

//                     if( (static_cast<EnvUDT*>(callStack.back())->NJump() != nJump) &&
//                         !i->LabelInRange( static_cast<EnvUDT*>(callStack.back())->LastJump()))
//                     {
//                         // a jump (goto) occured out of this loop
//                         return retCode;
//                     }
//                 }
             }
        )
	;   



// ***************************************************************************
// the expressions ***********************************************************
// ***************************************************************************

l_deref returns [BaseGDL** res]
{
	ProgNodeP retTree = _t->getNextSibling();

    EnvBaseT* actEnv = callStack.back()->GetNewEnv();
    if( actEnv == NULL) actEnv = callStack.back();

    assert( actEnv != NULL);

    auto_ptr<BaseGDL> e1_guard;
    BaseGDL* e1;
    ProgNodeP evalExpr = _t->getFirstChild();
    if( NonCopyNode( evalExpr->getType()))
      {
            e1 = evalExpr->EvalNC();
      }
    else if( evalExpr->getType() ==  GDLTokenTypes::FCALL_LIB)
      {
		e1=lib_function_call(evalExpr);

		if( e1 == NULL) // ROUTINE_NAMES
			throw GDLException( evalExpr, "Undefined return value", true, false);
		
		if( !callStack.back()->Contains( e1)) 
			{
//                if( actEnv != NULL)
                    actEnv->Guard( e1); 
//                else
//                    e1_guard.reset( e1);
            }
      }
    else
      {
        e1 = evalExpr->Eval();

  //      if( actEnv != NULL)
            actEnv->Guard( e1); 
  //      else
  //          e1_guard.reset(e1);
      }

  if( e1 == NULL || e1->Type() != PTR)
    throw GDLException( evalExpr, "Pointer type required"
			" in this context: "+Name(e1),true,false);

  DPtrGDL* ptr=static_cast<DPtrGDL*>(e1);

//     _t = _t->getFirstChild();

//     BaseGDL* e1;

// 	auto_ptr<BaseGDL> e1_guard;

//     if( _t->getType() ==  GDLTokenTypes::FCALL_LIB)
//       {
// 		e1=lib_function_call(_t);

// 		if( e1 == NULL) // ROUTINE_NAMES
// 			throw GDLException( _t, "Undefined return value", true, false);
		
// 		if( !ProgNode::interpreter->callStack.back()->Contains( e1)) 
// 			e1_guard.reset( e1);
//       }
//     else
//       {
// 			e1=tmp_expr(_t);
// 			e1_guard.reset( e1);
//       }

// 	DPtrGDL* ptr=dynamic_cast<DPtrGDL*>(e1);
// 	if( ptr == NULL)
// 	throw GDLException( _t, "Pointer type required"
// 	" in this context: "+Name(e1),true,false);
	DPtr sc; 
	if( !ptr->Scalar(sc))
	throw GDLException( _t, "Expression must be a "
	"scalar in this context: "+Name(e1),true,false);
	if( sc == 0)
	throw GDLException( _t, "Unable to dereference"
	" NULL pointer: "+Name(e1),true,false);
	
	try{
        res = &GetHeap(sc);
	}
	catch( HeapException)
	{
        throw GDLException( _t, "Invalid pointer: "+Name(e1),true,false);
	}
	
	_retTree = retTree;
	return res;

}
    : #(DEREF e1=expr 
//             {
//                 auto_ptr<BaseGDL> e1_guard(e1);
                
//                 DPtrGDL* ptr=dynamic_cast<DPtrGDL*>(e1);
//                 if( ptr == NULL)
//                 throw GDLException( _t, "Pointer type required"
//                     " in this context: "+Name(e1),true,false);
//                 DPtr sc; 
//                 if( !ptr->Scalar(sc))
//                 throw GDLException( _t, "Expression must be a "
//                     "scalar in this context: "+Name(e1),true,false);
//                 if( sc == 0)
//                 throw GDLException( _t, "Unable to dereference"
//                     " NULL pointer: "+Name(e1),true,false);
                
//                 try{
//                     res = &GetHeap(sc);
//                 }
//                 catch( HeapException)
//                 {
//                     throw GDLException( _t, "Invalid pointer: "+Name(e1),true,false);
//                 }
//             }
        )
    ;

// return value from functions when used as l var
// used only from jump_statement and within itself
l_ret_expr returns [BaseGDL** res]
{
    BaseGDL*       e1;
}
    : res=l_deref
    | #(QUESTION e1=expr
            { 
                auto_ptr<BaseGDL> e1_guard(e1);
                if( e1->True())
                {
                    res=l_ret_expr(_t);
                }
                else
                {
                    _t=_t->GetNextSibling(); // jump over 1st expression
                    res=l_ret_expr(_t);
                }
            }
        ) // trinary operator
//    | #(EXPR res=l_ret_expr) // does not exist anymore
    | res=l_arrayexpr_mfcall_as_mfcall
    | res=l_function_call 
        { // here a local to the actual environment could be returned
            if( callStack.back()->IsLocalKW( res))
            throw GDLException( _t, 
                "Attempt to return indirectly a local variable "
                "from left-function.",true,false);
        }
    | varPtr:VARPTR // DNode.var   is ptr to common block variable
        {
            res=&varPtr->var->Data(); // returns BaseGDL* of var (DVar*) 
        }
    | var:VAR // DNode.varIx is index into functions/procedures environment
        {     // check if variable is non-local 
              // (because it will be invalid after return otherwise)
            if( !callStack.back()->GlobalKW(var->varIx))
            throw GDLException( _t, 
                "Attempt to return a non-global variable from left-function.",true,false);
            
            res=&callStack.back()->GetKW(var->varIx); 
        }
    | // here ASSIGN and ASSIGN_REPLACE are identical
      #(ASSIGN // can it occur at all?
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( e1=tmp_expr
                {
                    r_guard.reset( e1);
                }
            | e1=lib_function_call
                {
                    if( !callStack.back()->Contains( e1)) 
                        r_guard.reset( e1);
                }
            )
            res=l_ret_expr
            {
                if( e1 != (*res))
                    {
                    delete *res;
                    *res = e1;
                    }
                r_guard.release();
            }
        )
    | #(ASSIGN_ARRAYEXPR_MFCALL // here as return value of l_function
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( e1=tmp_expr
                {
                    r_guard.reset( e1);
                }
            | e1=lib_function_call
                {
                    if( !callStack.back()->Contains( e1)) 
                        r_guard.reset( e1);
                }
            )
            res=l_arrayexpr_mfcall_as_mfcall
            {
                if( e1 != (*res))
                    {
                    delete *res;
                    *res = e1;
                    }
                r_guard.release();
            }
        )
    | #(ASSIGN_REPLACE 
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( e1=tmp_expr
                {
                    r_guard.reset( e1);
                }
            | e1=lib_function_call
                {
                    if( !callStack.back()->Contains( e1)) 
                        r_guard.reset( e1);
                }
            )
            res=l_ret_expr
            {
                if( e1 != (*res))
                    {
                    delete *res;
                    *res = e1;
                    }
                r_guard.release();
            }
        )

    // the following are forbiden    
    | #(ARRAYEXPR
            {
                throw GDLException( _t, 
                    "Indexed expression not allowed as left-function"
                    " return value.",true,false);
            }
        )
    | #(DOT 
            {
                throw GDLException( _t, 
                    "Struct expression not allowed as left-function"
                    " return value.",true,false);
            }
        )
    | SYSVAR
        {
            throw GDLException( _t, 
                "System variable not allowed as left-function"
                " return value.",true,false);
        }
    | e1=r_expr
        {
            delete e1;
            throw GDLException( _t, 
                "Expression not allowed as left-function return value.",true,false);
        }
    | e1=constant_nocopy
        {
            throw GDLException( _t, 
                "Constant not allowed as left-function return value.",true,false);
        }
    ;

// l expressions for DEC/INC ********************************
// called from l_decinc_array_expr
l_decinc_indexable_expr [int dec_inc] returns [BaseGDL* res]
{
    BaseGDL** e;
}
//     : #(EXPR e = l_expr[ NULL])                       
//         {
//             res = *e;
//             if( res == NULL)
//             throw GDLException( _t, "Variable is undefined: "+Name(e));
//         }
//     | e=l_function_call
    : e=l_function_call
        {
            res = *e;
            if( res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(e),true,false);
        }
    | e=l_deref 
        {
            res = *e;
            if( res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(e),true,false);
        }
    | e=l_defined_simple_var { res = *e; } // no Dup here
    | e=l_sys_var { res = *e; }            // no Dup here
    ;

// called from l_decinc_expr
l_decinc_array_expr [int dec_inc] returns [BaseGDL* res]
{
    ArrayIndexListT* aL;
    BaseGDL*         e;
    ArrayIndexListGuard guard;

}
    : #(ARRAYEXPR 
            e=l_decinc_indexable_expr[ dec_inc]   
            aL=arrayindex_list)
        {
            guard.reset( aL); 
            aL->SetVariable( e);

            if( dec_inc == DECSTATEMENT) 
            {
                e->DecAt( aL); 
                res = NULL;
                break;
            }
            if( dec_inc == INCSTATEMENT)
            {
                e->IncAt( aL);
                res = NULL;
                break;
            }

            if( dec_inc == DEC) e->DecAt( aL); 
            else if( dec_inc == INC) e->IncAt( aL);
//
            res=e->Index( aL);

            if( dec_inc == POSTDEC) e->DecAt( aL);
            else if( dec_inc == POSTINC) e->IncAt( aL);
        }
    | e=l_decinc_indexable_expr[ dec_inc]
        {
            if( dec_inc == DECSTATEMENT) 
            {
                e->Dec(); 
                res = NULL;
                break;
            }
            if( dec_inc == INCSTATEMENT)
            {
                e->Inc();
                res = NULL;
                break;
            }

            if( dec_inc == DEC) e->Dec();
            else if( dec_inc == INC) e->Inc();
  //          
            res = e->Dup();
            
            if( dec_inc == POSTDEC) e->Dec();
            else if( dec_inc == POSTINC) e->Inc();
        }
    ;

// struct assignment
// MAIN function: called from l_decinc_expr
l_decinc_dot_expr [int dec_inc] returns [BaseGDL* res]
    : #(dot:DOT 
            { 
                SizeT nDot=dot->nDot;
                auto_ptr<DotAccessDescT> aD( new DotAccessDescT(nDot+1));
            } 
            l_dot_array_expr[ aD.get()] 
            (tag_array_expr[ aD.get()] /* nDot times*/ )+ 
        )         
        {
            if( dec_inc == DECSTATEMENT) 
            {
                aD->Dec(); 
                res = NULL;
            }
            else if( dec_inc == INCSTATEMENT)
            {
                aD->Inc();
                res = NULL;
            }
            else
            {
                if( dec_inc == DEC) aD->Dec(); //*** aD->Assign( dec_inc);
                else if( dec_inc == INC) aD->Inc();
//                
                res=aD->Resolve();
                
                if( dec_inc == POSTDEC) aD->Dec();
                else if( dec_inc == POSTINC) aD->Inc();
            }
        }
    ;

// l_decinc_expr is only used in dec/inc statements and within itself
l_decinc_expr [int dec_inc] returns [BaseGDL* res]
{
    BaseGDL*       e1;
    ProgNodeP startNode = _t;
}
    : #(QUESTION e1=expr
            { 
                auto_ptr<BaseGDL> e1_guard(e1);

                if( e1->True())
                {
                    res=l_decinc_expr(_t, dec_inc);
                }
                else
                {
                    _t=_t->GetNextSibling(); // jump over 1st expression
                    res=l_decinc_expr(_t, dec_inc);
                }
            }
        ) // trinary operator
    | #(ASSIGN 
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
//             ( e1=tmp_expr
//                 {
//                     r_guard.reset( e1);
//                 }
//             | e1=lib_function_call
//                 {
//                     if( !callStack.back()->Contains( e1)) 
//                         r_guard.reset( e1);
//                 }
//             )
            ( e1=indexable_expr
            | e1=indexable_tmp_expr { r_guard.reset( e1);}
            | e1=lib_function_call
                {
                    if( !callStack.back()->Contains( e1)) 
                        r_guard.reset( e1); // guard if no global data
                }
            )
            { 
                ProgNodeP l = _t;

                BaseGDL** tmp;
            } 
            tmp=l_expr[ e1] // assign
            {
                _t = l;
            }
            res=l_decinc_expr[ dec_inc]
        )
    | #(ASSIGN_ARRAYEXPR_MFCALL
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( e1=indexable_expr
            | e1=indexable_tmp_expr { r_guard.reset( e1);}
            | e1=lib_function_call
                {
                    if( !callStack.back()->Contains( e1)) 
                        r_guard.reset( e1); // guard if no global data
                }
            )
            { 
                ProgNodeP l = _t;

                BaseGDL** tmp;

                // try MFCALL
                try
                {
    
                    tmp=l_arrayexpr_mfcall_as_mfcall(l);
    
                    if( e1 != (*tmp))
                    {
                        delete *tmp;

                        if( r_guard.get() == e1)
                            *tmp = r_guard.release();
                        else          
                            *tmp = e1->Dup();
                    }

                    res=l_decinc_expr( l, dec_inc);
                }
                catch( GDLException& ex)
                {
                // try ARRAYEXPR
                    try
	                {
                        tmp=l_arrayexpr_mfcall_as_arrayexpr(l, e1);
	                }
                    catch( GDLException& ex2)
                    {
                        throw GDLException(ex.toString() + " or "+ex2.toString());
                    }

                    res=l_decinc_expr( l, dec_inc);
                }
            }
        )
    | #(ASSIGN_REPLACE 
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( e1=tmp_expr
                {
                    r_guard.reset( e1);
                }
            | e1=lib_function_call
                {
                    if( !callStack.back()->Contains( e1)) 
                        r_guard.reset( e1);
                }
            )
            { 
                ProgNodeP l = _t;

                BaseGDL** tmp;
            } 
//            tmp=l_expr[ e1] // assign
            (
              tmp=l_function_call   // FCALL_LIB, MFCALL, MFCALL_PARENT, FCALL
            | tmp=l_deref           // DEREF
            | tmp=l_simple_var      // VAR, VARPTR
            )
        {
            if( e1 != (*tmp))
            {
                delete *tmp;

                if( r_guard.get() == e1)
                  *tmp = r_guard.release();
                else  
                  *tmp = e1->Dup();
            }
        }
            {
                _t = l;
            }
            res=l_decinc_expr[ dec_inc]
        )
    | res=l_decinc_array_expr[ dec_inc]
    | #(ARRAYEXPR_MFCALL
        {
            ProgNodeP mark = _t;
            _t = _t->getNextSibling(); // step over DOT

            BaseGDL* self;
        }
            
        //BaseGDL** e = l_arrayexpr_mfcall_as_mfcall( _t);
        self=expr mp2:IDENTIFIER

        {  
                auto_ptr<BaseGDL> self_guard(self);
        
                EnvUDT* newEnv;

                try {
                    newEnv=new EnvUDT( self, mp2, "", true);
                    self_guard.release();
                }
                catch( GDLException& ex)
                {
                    _t = mark;

                    res=l_decinc_dot_expr(_t, dec_inc);

                    _retTree = startNode->getNextSibling();
                    return res;
                }   
        }    

        parameter_def[ newEnv]
        
        {
            // push environment onto call stack
            callStack.push_back(newEnv);
            
            // make the call
            BaseGDL** ee=call_lfun(static_cast<DSubUD*>(
                                  newEnv->GetPro())->GetTree());

            BaseGDL* e = *ee;
            if( e == NULL)
                throw GDLException( _t, "Variable is undefined: "+Name(ee),true,false);

            if( dec_inc == DECSTATEMENT) 
                {
                    e->Dec(); 
                    res = NULL;
                    _retTree = startNode->getNextSibling();
                    return res;
                }
            if( dec_inc == INCSTATEMENT)
                {
                    e->Inc();
                    res = NULL;
                    _retTree = startNode->getNextSibling();
                    return res;
                }

            if( dec_inc == DEC) e->Dec();
            else if( dec_inc == INC) e->Inc();
  //          
            res = e->Dup();
            
            if( dec_inc == POSTDEC) e->Dec();
            else if( dec_inc == POSTINC) e->Inc();

            _retTree = startNode->getNextSibling();
            return res;
        }   
        )
    | res=l_decinc_dot_expr[ dec_inc]
    | e1=r_expr
        {
            delete e1;
            throw GDLException( _t, 
                "Expression not allowed with decrement/increment operator.",true,false);
        }
    | e1=constant_nocopy
        {
            throw GDLException( _t, 
                "Constant not allowed with decrement/increment operator.",true,false);
        }
    ;

// l expressions for assignment *************************

// an indexable expression must be defined
l_indexable_expr returns [BaseGDL** res]
    : #(EXPR res=l_expr[ NULL]) // for l_dot_array_expr
        {
            if( *res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(res),true,false);
        }
    | res=l_function_call
        {
            if( *res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(res),true,false);
        }
    | res=l_arrayexpr_mfcall_as_mfcall 
        {
            if( *res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(res),true,false);
        }
    | res=l_deref
        {
            if( *res == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(res),true,false);
        }
    | res=l_defined_simple_var                         
    | res=l_sys_var 
    ;

// called from l_expr
l_array_expr [BaseGDL* right] returns [BaseGDL** res]
{
    ArrayIndexListT* aL;
    ArrayIndexListGuard guard;

	ProgNodeP astIn = _t;
// 	match(antlr::RefAST(_t),ARRAYEXPR);
	_t = _t->getFirstChild();
	res=l_indexable_expr(_t);
	_t = _retTree;
	aL=arrayindex_list(_t);

	guard.reset(aL);
	
	if( right == NULL)
	throw GDLException( astIn, 
	"Indexed expression not allowed in this context.",true,false);
	
	try {
	aL->AssignAt( *res, right);
	}
	catch( GDLException& ex)
	{
	ex.SetErrorNodeP( astIn);
	throw ex;
	}
	//             aL->AssignAt( *res, right);
	
	//             aL->SetVariable( *res);
	//             if( (*res)->EqType( right))
	//             {
	//                 (*res)->AssignAt( right, aL); // assigns inplace
	//             }
	//             else
	//             {
	//                 BaseGDL* rConv = right->Convert2( (*res)->Type(), BaseGDL::COPY);
	//                 auto_ptr<BaseGDL> conv_guard( rConv);
	//                 (*res)->AssignAt( rConv, aL); // assigns inplace
	//             }
	
	_retTree = astIn->getNextSibling();
	return res;
}
    : #(ARRAYEXPR res=l_indexable_expr aL=arrayindex_list { guard.reset(aL);})   //         {
//             if( right == NULL)
//             throw GDLException( _t, 
//                 "Indexed expression not allowed in this context.",true,false);

//             try {
//                 aL->AssignAt( *res, right);
//             }
//             catch( GDLException& ex)
//             {
//                 ex.SetLine( _t->getLine());
//                 throw ex;
//             }
// //             aL->AssignAt( *res, right);

// //             aL->SetVariable( *res);
// //             if( (*res)->EqType( right))
// //             {
// //                 (*res)->AssignAt( right, aL); // assigns inplace
// //             }
// //             else
// //             {
// //                 BaseGDL* rConv = right->Convert2( (*res)->Type(), BaseGDL::COPY);
// //                 auto_ptr<BaseGDL> conv_guard( rConv);
// //                 (*res)->AssignAt( rConv, aL); // assigns inplace
// //             }
//         }
    ;

l_dot_array_expr [DotAccessDescT* aD] // 1st
{
    ArrayIndexListT* aL;
    BaseGDL**        rP;
    DStructGDL*      structR;
    ArrayIndexListGuard guard;
    bool isObj = callStack.back()->IsObject();
}
    : #(ARRAYEXPR rP=l_indexable_expr aL=arrayindex_list { guard.reset(aL);})   
        {
            // check here for object and get struct
            structR=dynamic_cast<DStructGDL*>(*rP);
            if( structR == NULL)
            {
                if( isObj)
                {
                    DStructGDL* oStruct = ObjectStructCheckAccess( *rP, _t);

                    // oStruct cannot be "Assoc_"
                    aD->Root( oStruct, guard.release()); 
                }
                else
                {
                    throw GDLException( _t, "Expression must be a"
                        " STRUCT in this context: "+Name(*rP),true,false);
                }
            }
            else 
            {
                if( (*rP)->IsAssoc())
                throw GDLException( _t, "File expression not allowed "
                    "in this context: "+Name(*rP),true,false);
                
                aD->Root( structR, guard.release() /* aL */); 
            }
        }
    | rP=l_indexable_expr
        {
            // check here for object and get struct
            structR = dynamic_cast<DStructGDL*>(*rP);
            if( structR == NULL)
            {
                if( isObj) // member access to object?
                {
                    DStructGDL* oStruct = ObjectStructCheckAccess( *rP, _t);

                    // oStruct cannot be "Assoc_"
                    aD->Root( oStruct); 
                }
                else
                {
                    throw GDLException( _t, "Expression must be a"
                        " STRUCT in this context: "+Name(*rP),true,false);
                }
            }
            else
            {
                if( (*rP)->IsAssoc())
                {
                    throw GDLException( _t, "File expression not allowed "
                        "in this context: "+Name(*rP),true,false);
                }
                
                aD->Root(structR); 
            }
        }
    ;


// l_expr is only used in assignment and within itself
l_expr [BaseGDL* right] returns [BaseGDL** res]
{
    BaseGDL*       e1;
}
    : #(QUESTION e1=expr
            { 
                auto_ptr<BaseGDL> e1_guard(e1);

                if( e1->True())
                {
                    res=l_expr(_t, right);
                }
                else
                {
                    _t=_t->GetNextSibling(); // jump over 1st expression
                    res=l_expr(_t, right);
                }
            }
        ) // trinary operator
    | #(ASSIGN //???e1=expr
//             { 
//                 auto_ptr<BaseGDL> r_guard;
//             } 
//             ( e1=tmp_expr
//                 {
//                     r_guard.reset( e1);
//                 }
//             | e1=lib_function_call
//                 {
//                     if( !callStack.back()->Contains( e1)) 
//                         r_guard.reset( e1);
//                 }
//             )
            ( e1=indexable_expr
            | e1=indexable_tmp_expr { delete e1;}
//            | e1=indexable_tmp_expr { r_guard.reset( e1);}
            | e1=lib_function_call
                {
                    if( !callStack.back()->Contains( e1)) 
                        delete e1; // guard if no global data
//                        r_guard.reset( e1); // guard if no global data
                }
            )

            res=l_expr[ right]
//             {
//                 if( (*res) == e1 || callStack.back()->Contains( e1)) 
//                     r_guard.release();
//             }
        )
    | #(ASSIGN_ARRAYEXPR_MFCALL
            ( e1=indexable_expr
            | e1=indexable_tmp_expr { delete e1;}
            | e1=lib_function_call
                {
                    if( !callStack.back()->Contains( e1)) 
                        delete e1; 
                }
            )
            { 
                ProgNodeP l = _t;

                // try MFCALL
                try
                {
    
                    res=l_arrayexpr_mfcall_as_mfcall( l);
    
                    if( right != (*res))
                    {
                        delete *res;
                        *res = right->Dup();
                    }
                }
                catch( GDLException& ex)
                {
                // try ARRAYEXPR
                    try
	                {
                        res=l_arrayexpr_mfcall_as_arrayexpr(l, right);
	                }
                    catch( GDLException& ex2)
                    {
                        throw GDLException(ex.toString() + " or "+ex2.toString());
                    }
                }
            }
        )
    | #(ASSIGN_REPLACE //???e1=expr
//             { 
//                 auto_ptr<BaseGDL> r_guard;
//             } 
            ( e1=tmp_expr
                {
                    delete e1;
//                    r_guard.reset( e1);
                }
            | e1=lib_function_call
                {
                    if( !callStack.back()->Contains( e1)) 
                        delete e1;
//                        r_guard.reset( e1);
                }
            )
            (
              res=l_function_call   // FCALL_LIB, MFCALL, MFCALL_PARENT, FCALL
            | res=l_deref           // DEREF
            | res=l_simple_var      // VAR, VARPTR
            )
        {
            if( right != (*res))
//            if( e1 != (*res))
            {
                delete *res;
//
//                if( r_guard.get() == e1)
//                  *res = r_guard.release();
//                else  
                  *res = right->Dup();
            }
        }
        )
    | res=l_array_expr[ right]
    | { ProgNodeP sysVar = _t;} // for error reporting
        res=l_sys_var // sysvars cannot change their type
        {
            if( right == NULL)
            throw GDLException( sysVar, 
                "System variable not allowed in this context.",true,false);
            
           auto_ptr<BaseGDL> conv_guard; //( rConv);
           BaseGDL* rConv = right;
           if( !(*res)->EqType( right))
            {
                rConv = right->Convert2( (*res)->Type(), 
                                                  BaseGDL::COPY);
                conv_guard.reset( rConv);
            }
 
            if( right->N_Elements() != 1 && 
                ((*res)->N_Elements() != right->N_Elements()))
            {
                throw GDLException( _t, "Conflicting data structures: <"+
                    right->TypeStr()+" "+right->Dim().ToString()+">, !"+ 
                    sysVar->getText(),true,false);
            }
            
            (*res)->AssignAt( rConv); // linear copy
        }
//   | res=l_indexoverwriteable_expr 
     | // can be called via QUESTION
       ( res=l_function_call   // FCALL_LIB, MFCALL, MFCALL_PARENT, FCALL
       | res=l_deref           // DEREF
       | res=l_simple_var      // VAR, VARPTR
       )
     {
            if( right != NULL && right != (*res))
            {
                delete *res;
                *res = right->Dup();
            }
     }
//         {
//             if( right != NULL && right != (*res))
//             {
//                 // only here non-inplace copy is done
//                 delete *res;
//                 *res = right->Dup();
//             }
//         }
    | res=l_arrayexpr_mfcall[ right]
//    | res=l_dot_expr[ right]
    | #(dot:DOT  // struct assignment
            { 
                SizeT nDot=dot->nDot;
                auto_ptr<DotAccessDescT> aD( new DotAccessDescT(nDot+1));
            } 
            l_dot_array_expr[ aD.get()] 
            (tag_array_expr[ aD.get()] /* nDot times*/ )+ 
        )         
        {
            if( right == NULL)
            throw GDLException( _t, "Struct expression not allowed in this context.",true,false);
            
            aD->Assign( right);

            res=NULL;
        }
//    | { right == NULL}? res=l_function_call
    | e1=r_expr
        {
            delete e1;
            throw GDLException( _t, 
                "Expression not allowed as l-value.",true,false);
        }
    | e1=constant_nocopy
        {
            throw GDLException( _t, 
                "Constant not allowed as l-value.",true,false);
        }
    ;

// used in l_expr but also in parameter_def
l_simple_var returns [BaseGDL** res]
{
	assert( _t != NULL);

	_retTree = _t->getNextSibling();

	if( _t->getType() == VAR)
	{
		return &callStack.back()->GetKW(_t->varIx); 
//		ProgNodeP var = _t;
// 		match(antlr::RefAST(_t),VAR);
	}
	else
	{
		return &_t->var->Data(); // returns BaseGDL* of var (DVar*) 
//		ProgNodeP varPtr = _t;
// 		match(antlr::RefAST(_t),VARPTR);
	}
	return res;
}
    : VAR // DNode.varIx is index into functions/procedures environment
    | VARPTR // DNode.var   is ptr to common block variable
    ;

l_defined_simple_var returns [BaseGDL** res]
{
	if( _t->getType() == VAR)
	{
// 		match(antlr::RefAST(_t),VAR);
		res=&callStack.back()->GetKW(_t->varIx); 
		if( *res == NULL)
		throw GDLException( _t, "Variable is undefined: "+
		callStack.back()->GetString(_t->varIx),true,false);
		
	}
	else
	{
// 		match(antlr::RefAST(_t),VARPTR);
		res=&_t->var->Data(); // returns BaseGDL* of var (DVar*) 
		if( *res == NULL)
		throw GDLException( _t, "Variable is undefined: "+
		callStack.back()->GetString( *res),true,false);
		
	}
	_retTree = _t->getNextSibling();
	return res;
}
    : VAR // DNode.varIx is index into functions/procedures environment
//         {
//             res=&callStack.back()->GetKW(var->varIx); 
//             if( *res == NULL)
//             throw GDLException( _t, "Variable is undefined: "+
//                 callStack.back()->GetString(var->varIx),true,false);
//         }
    | VARPTR // DNode.var   is ptr to common block variable
//         {
//             res=&varPtr->var->Data(); // returns BaseGDL* of var (DVar*) 
//             if( *res == NULL)
//             throw GDLException( _t, "Variable is undefined: "+
//                 callStack.back()->GetString( *res),true,false);
//         }
    ;

l_sys_var returns [BaseGDL** res]
    : sysVar:SYSVAR  
        // ProgNodeP->getText() returns name which 
        // has to be searched 
        // in 'sysVarList' (objects.cpp) if ProgNodeP->var is NULL
        {
            if( sysVar->var == NULL) 
            {
                sysVar->var=FindInVarList(sysVarList,sysVar->getText());
                if( sysVar->var == NULL)		    
                throw GDLException( _t, "Not a legal system variable: !"+
                    sysVar->getText(),true,false);

                // note: this works, because system variables are never 
                //       passed by reference
                SizeT rdOnlySize = sysVarRdOnlyList.size();
                for( SizeT i=0; i<rdOnlySize; ++i)
                  if( sysVarRdOnlyList[ i] == sysVar->var)
                    throw GDLException( _t, 
                    "Attempt to write to a readonly variable: !"+
                    sysVar->getText(),true,false);
            }
            // system variables are always defined
            res=&sysVar->var->Data();
        }
    ;

// right expressions **********************************************************
// expressions which always only return a value
// expecting to delete any sub-expressions
r_expr returns [BaseGDL* res]
{
	switch ( _t->getType()) {
	case EXPR:
	case ARRAYDEF:
	case STRUC:
	case NSTRUC:
	case NSTRUC_REF:
	{
		res = _t->Eval(); 
		break;
	}
	case DEC:
	{
		res=l_decinc_expr( _t->getFirstChild(), DEC);
		break;
	}
	case INC:
	{
		res=l_decinc_expr( _t->getFirstChild(), INC);
		break;
	}
	case POSTDEC:
	{
		res=l_decinc_expr( _t->getFirstChild(), POSTDEC);
		break;
	}
//	case POSTINC:
	default:
	{
		res=l_decinc_expr( _t->getFirstChild(), POSTINC);
		break;
	}
// 	default:
// 	{
// 		throw antlr::NoViableAltException(antlr::RefAST(_t));
// 	}
	}
	_retTree = _t->getNextSibling();
	return res;
}
    : EXPR
    | ARRAYDEF
    | STRUC
    | NSTRUC
    | NSTRUC_REF
    |	#(DEC res=l_decinc_expr[ DEC])
    |	#(INC res=l_decinc_expr[ INC])
    |	#(POSTDEC res=l_decinc_expr[ POSTDEC])
    |	#(POSTINC res=l_decinc_expr[ POSTINC])
    ;

array_expr returns [BaseGDL* res]
{
    ArrayIndexListT* aL;
    BaseGDL* r;
    ArrayIndexListGuard guard;
    auto_ptr<BaseGDL> r_guard;

    ExprListT        exprList; // for cleanup
    IxExprListT      ixExprList;
    SizeT nExpr;
    BaseGDL* s;

	ProgNodeP retTree = _t->getNextSibling();
//	match(antlr::RefAST(_t),ARRAYEXPR);
	_t = _t->getFirstChild();
	
	switch ( _t->getType()) {
	case VAR:
	case CONSTANT:
	case DEREF:
	case SYSVAR:
	case VARPTR:
	{
        r=_t->EvalNC();
		//r=indexable_expr(_t);
		_t = _t->getNextSibling(); //_retTree;
		break;
	}
	case FCALL_LIB:
	{
		r=lib_function_call(_t);
		_t = _t->getNextSibling();
		
		if( !callStack.back()->Contains( r)) 
            r_guard.reset( r); // guard if no global data
		
		break;
	}
// 	case ASSIGN:
// 	case ASSIGN_REPLACE:
// 	case ASSIGN_ARRAYEXPR_MFCALL:
// 	case ARRAYDEF:
// 	case ARRAYEXPR:
// 	case ARRAYEXPR_MFCALL:
// 	case EXPR:
// 	case FCALL:
// 	case FCALL_LIB_RETNEW:
// 	case MFCALL:
// 	case MFCALL_PARENT:
// 	case NSTRUC:
// 	case NSTRUC_REF:
// 	case POSTDEC:
// 	case POSTINC:
// 	case STRUC:
// 	case DEC:
// 	case INC:
// 	case DOT:
// 	case QUESTION:
    default:
	{
		r=indexable_tmp_expr(_t);
		_t = _retTree;
		r_guard.reset( r);
		break;
	}
	}
	

	aL = _t->arrIxList;
	assert( aL != NULL);
	guard.reset(aL);
	
//    ax = _t
//	match(antlr::RefAST(_t),ARRAYIX);
	_t = _t->getFirstChild();
	
	nExpr = aL->NParam();
	
	if( nExpr == 0)
	{
        goto empty;
	}
	//                 if( nExpr > 1)
	//                 {
	//                     ixExprList.reserve( nExpr);
	//                     exprList.reserve( nExpr);
	//                 }
	//                if( nExpr == 0) goto empty;
	
	for (;;) {
//		if ((_tokenSet_1.member(_t->getType()))) {
			switch ( _t->getType()) {
			case VAR:
			case CONSTANT:
			case DEREF:
			case SYSVAR:
			case VARPTR:
			{
				s=_t->EvalNC();//indexable_expr(_t);
				_t = _t->getNextSibling();//_retTree;
				break;
			}
			case FCALL_LIB:
			{
				s=lib_function_call(_t);
				_t = _retTree;
				
				if( !callStack.back()->Contains( s)) 
				exprList.push_back( s);
				
				break;
			}
// 			case ASSIGN:
// 			case ASSIGN_REPLACE:
// 			case ASSIGN_ARRAYEXPR_MFCALL:
// 			case ARRAYDEF:
// 			case ARRAYEXPR:
// 			case ARRAYEXPR_MFCALL:
// 			case EXPR:
// 			case FCALL:
// 			case FCALL_LIB_RETNEW:
// 			case MFCALL:
// 			case MFCALL_PARENT:
// 			case NSTRUC:
// 			case NSTRUC_REF:
// 			case POSTDEC:
// 			case POSTINC:
// 			case STRUC:
// 			case DEC:
// 			case INC:
// 			case DOT:
// 			case QUESTION:
            default:
			{
				s=indexable_tmp_expr(_t);
				_t = _retTree;
				exprList.push_back( s);
				break;
			}
			} // switch
			
			ixExprList.push_back( s);
			if( ixExprList.size() == nExpr)
                break; // finish
			
		} // for
// 		else {
// 			assert( 0);//goto _loop106;
// 		}
		
	empty:
	//_retTree = ax;
	res = aL->Index( r, ixExprList);
	//                 aL->Init( ixExprList);
	//                 aL->SetVariable( r);
	//                 res=r->Index( aL);
	//                ClearTmpList();

    _retTree = retTree;
	return res;
}
    : #(ARRAYEXPR 
            ( r=indexable_expr
            | r=indexable_tmp_expr { r_guard.reset( r);}
            | r=lib_function_call
                {
                    if( !callStack.back()->Contains( r)) 
                        r_guard.reset( r); // guard if no global data
                }
            )
            //            aL=indexing_list { guard.reset(aL);}
        #(ax:ARRAYIX
                {
                    aL = ax->arrIxList;
                    assert( aL != NULL);
                    
                    guard.reset(aL);
                    
                    nExpr = aL->NParam();

                    if( nExpr == 0)
                    {
                        goto empty2;
                    }
                    //                 if( nExpr > 1)
                    //                 {
                    //                     ixExprList.reserve( nExpr);
                    //                     exprList.reserve( nExpr);
                    //                 }
                    //                if( nExpr == 0) goto empty;
                }
                (
                    ( s=indexable_expr
                    | s=lib_function_call
                        {
                            if( !callStack.back()->Contains( s)) 
                            exprList.push_back( s);
                        }
                    | s=indexable_tmp_expr { exprList.push_back( s);}
                    )
                    {
                        ixExprList.push_back( s);
                        if( ixExprList.size() == nExpr)
                        break; // allows some manual tuning
                    }
                )*
                //            { empty: ;}
            )
            {
empty2:
                //_retTree = ax;
                res = aL->Index( r, ixExprList);
//                 aL->Init( ixExprList);
//                 aL->SetVariable( r);
//                 res=r->Index( aL);
//                ClearTmpList();
            }
        )   
//     | res=expr //indexable_expr
    ;

// for l and r expr
tag_expr [DotAccessDescT* aD] // 2nd...
{
    BaseGDL* e;
}
    : #(EXPR e=expr
            {
                auto_ptr<BaseGDL> e_guard(e);
                
                SizeT tagIx;
                int ret=e->Scalar2index(tagIx);
                if( ret < 1)
                throw GDLException( _t, "Expression must be a scalar"
                    " >= 0 in this context: "+Name(e),true,false);
                
                aD->Add( tagIx);
            }
        )                       
    | i:IDENTIFIER
        {
            std::string tagName=i->getText();
            aD->Add( tagName);
        }
    ;

// for l and r expr
tag_array_expr  [DotAccessDescT* aD] // 2nd...
{
    ArrayIndexListT* aL;
}
	: #(ARRAYEXPR tag_expr[ aD] aL=arrayindex_list { aD->AddIx(aL);} )
    | tag_expr[ aD] { aD->AddIx(NULL);} 
    ;

r_dot_indexable_expr [DotAccessDescT* aD] returns [BaseGDL* res] // 1st
{
    BaseGDL** e;
}
    : #(EXPR res=expr { aD->SetOwner( true);}) // ({tag:0}).tag should work 
    | e=l_defined_simple_var { res = *e;}
    |   // we cant use l_sys_var here because of copy protection
        // could use sysvar and SetOwner( true), but this is quicker
        res=sys_var_nocopy // system variables are always defined    
    ;

r_dot_array_expr [DotAccessDescT* aD] // 1st
{
    ArrayIndexListT* aL;
    BaseGDL*         r;
    DStructGDL*      structR;
    ArrayIndexListGuard guard;
    bool isObj = callStack.back()->IsObject();
}
// NOTE: r is owned by aD or a l_... (r must not be deleted here)
    : #(ARRAYEXPR r=r_dot_indexable_expr[ aD] 
            aL=arrayindex_list { guard.reset(aL);} )   
        {
            // check here for object and get struct
            structR=dynamic_cast<DStructGDL*>(r);
            if( structR == NULL)
            {
                if( isObj)
                {
                    DStructGDL* oStruct = ObjectStructCheckAccess( r, _t);
                    
//                    DStructGDL* obj = oStruct->Index( aL);

                    if( aD->IsOwner()) delete r; 
                    aD->SetOwner( false); // object struct, not owned
                    
                    aD->Root( oStruct, guard.release()); 
//                    aD->Root( obj); 

//                     BaseGDL* obj = r->Index( aL);
//                     auto_ptr<BaseGDL> objGuard( obj); // new object -> guard
                    
//                     DStructGDL* oStruct = ObjectStructCheckAccess( obj, _t);

//                     // oStruct cannot be "Assoc_"
//                     if( aD->IsOwner()) delete r; 
//                     aD->SetOwner( false); // object structs are never owned
//                     aD->Root( oStruct); 
                }
                else
                {
                    throw GDLException( _t, "Expression must be a"
                        " STRUCT in this context: "+Name(r),true,false);
                }
            }
            else
            {
                if( r->IsAssoc())
                throw GDLException( _t, "File expression not allowed "
                    "in this context: "+Name(r),true,false);
                
                aD->Root( structR, guard.release()); 
            }
        }
    | r=r_dot_indexable_expr[ aD]
        {
            // check here for object and get struct
            structR = dynamic_cast<DStructGDL*>(r);
            if( structR == NULL)
            {
                if( isObj) // memeber access to object?
                {
                    DStructGDL* oStruct = ObjectStructCheckAccess( r, _t);

                    // oStruct cannot be "Assoc_"
                    if( aD->IsOwner()) delete r;
                    aD->SetOwner( false); // object structs are never owned
                    aD->Root( oStruct); 
                }
                else
                {
                    throw GDLException( _t, "Expression must be a"
                        " STRUCT in this context: "+Name(r),true,false);
                }
            }
            else
            {
                if( r->IsAssoc())
                {
                    throw GDLException( _t, "File expression not allowed "
                        "in this context: "+Name(r),true,false);
                }
                
                aD->Root(structR); 
            }
        }
    ;

// Entry function for struct access
//#(DOT array_expr (tag_array_expr)+)                     
dot_expr returns [BaseGDL* res]
{

	ProgNodeP rTree = _t->getNextSibling();
	//ProgNodeP 
    dot = _t;
// 	match(antlr::RefAST(_t),DOT);
	_t = _t->getFirstChild();
	
	SizeT nDot=dot->nDot;
	auto_ptr<DotAccessDescT> aD( new DotAccessDescT(nDot+1));
	
	r_dot_array_expr(_t, aD.get());
	_t = _retTree;
	for (; _t != NULL;) {
			tag_array_expr(_t, aD.get());
			_t = _retTree;
	}
	res= aD->Resolve();
	_retTree = rTree;
	return res;
}
    : #(dot:DOT 
//             { 
//                 SizeT nDot=dot->nDot;
//                 auto_ptr<DotAccessDescT> aD( new DotAccessDescT(nDot+1));
//             } 
            r_dot_array_expr[ aD.get()] 
            (tag_array_expr[ aD.get()] /* nDot times*/ )+ 
        )         
//         { res= aD->Resolve();}
    ;

// indexable expressions are used to minimize copying of data
// ( via Dup())
// owned by caller
indexable_tmp_expr returns [BaseGDL* res]
{
    BaseGDL*  e1;
}
	: #(q:QUESTION 
            { res = q->Eval(); }
//                 e1=expr
//             { 
//                 auto_ptr<BaseGDL> e1_guard(e1);

//                 if( e1->True())
//                 {   
//                     res=expr(_t);
//                 }
//                 else
//                 {
//                     _t=_t->GetNextSibling(); // jump over 1st expression
//                     res=expr(_t);
//                 }
//             }
        ) // trinary operator
    | (a:ARRAYEXPR { res = a->Eval();}) //res=array_expr
    | res=dot_expr
    | res=assign_expr
    | res=function_call
    | res=r_expr
    | res=lib_function_call_retnew 
    ;

// not owned by caller 
indexable_expr returns [BaseGDL* res]
{
    res = _t->EvalNC();
	_retTree = _t->getNextSibling();
    return res;

    BaseGDL** e2;
}
    : e2=l_defined_simple_var
        {
            res = *e2;
        }
    | res=sys_var_nocopy
    | res=constant_nocopy
    | e2=l_deref 
        { 
            if( *e2 == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(e2),true,false);
            res = *e2;
        }
    ;

// l_expr used as r_expr and true r_expr
expr returns [BaseGDL* res]
{
	
	assert( _t != NULL);

	if ( _t->getType() == FCALL_LIB) {
        BaseGDL* res=lib_function_call(_t);
		
		if( callStack.back()->Contains( res)) 
		res = res->Dup();

        return res;    		
	}
	else
	{
		return tmp_expr(_t);
	}
}
    : res=tmp_expr
    | res=lib_function_call
        {
            if( callStack.back()->Contains( res)) 
                res = res->Dup();
        }
    ;


// check_expr returns [BaseGDL* res]
//     : res=lib_function_call 
//     ;

// l_expr used as r_expr and true r_expr
tmp_expr returns [BaseGDL* res]
{
    BaseGDL*  e1;
    BaseGDL** e2;
}
    : e2=l_deref 
        { 
            if( *e2 == NULL)
            throw GDLException( _t, "Variable is undefined: "+Name(e2),true,false);
            
            res = (*e2)->Dup();
        }
	| #(q:QUESTION 
            { res = q->Eval();}
//                 e1=expr
//             { 
//                 auto_ptr<BaseGDL> e1_guard(e1);
//                 if( e1->True())
//                 {
//                     res=expr(_t);
//                 }
//                 else
//                 {
//                     _t=_t->GetNextSibling(); // jump over 1st expression
//                     res=expr(_t);
//                 }
//             }
        ) // trinary operator
    | (a:ARRAYEXPR { res = a->Eval();}) //res=array_expr
//    | res=array_expr
    | res=dot_expr
    | res=assign_expr
    | res=function_call
    | res=r_expr
    | res=constant
        // *********
    | res=simple_var                         
    | res=sys_var 
    | res=lib_function_call_retnew 
    ;

assign_expr returns [BaseGDL* res]
{
    BaseGDL** l;
    BaseGDL*  r;

	ProgNodeP startNode = _t;
	if( _t->getType() == ASSIGN) 
	{
// 		match(antlr::RefAST(_t),ASSIGN);
		_t = _t->getFirstChild();
		
		auto_ptr<BaseGDL> r_guard;
		
		if( _t->getType() == FCALL_LIB)
		{
			res=lib_function_call(_t);
			_t = _retTree;
			
			if( !callStack.back()->Contains( res)) 
			r_guard.reset( res);
			
		}
        else
		{
			res=tmp_expr(_t);
			_t = _retTree;
			
			r_guard.reset( res);
			
		}
		
		l=l_expr(_t, res);
		_t = _retTree;
		
		if( r_guard.get() == res) // owner
		r_guard.release();
		else
		res = res->Dup();
		
	}
    else if( _t->getType() == ASSIGN_ARRAYEXPR_MFCALL) 
        {

		_t = _t->getFirstChild();
		
		auto_ptr<BaseGDL> r_guard;
		
		if( _t->getType() == FCALL_LIB)
		{
			res=lib_function_call(_t);
			_t = _retTree;
			
			if( !callStack.back()->Contains( res)) 
			r_guard.reset( res);
			
		}
        else
		{
			res=tmp_expr(_t);
			_t = _retTree;
			
			r_guard.reset( res);
			
		}

        ProgNodeP mark = _t;

        // try MFCALL
        try
            {
                l=l_arrayexpr_mfcall_as_mfcall( mark);
    
                if( res != (*l))
                    {
                        delete *l;
                        *l = res->Dup();     
		
                        if( r_guard.get() == res) // owner
                            {
                                r_guard.release(); 
                            }
                        else
                            res = res->Dup();
                    }
            }
        catch( GDLException& ex)
            {
                // try ARRAYEXPR
                try
	                {
                        l=l_arrayexpr_mfcall_as_arrayexpr(mark, res);

                        if( r_guard.get() == res) // owner
                            r_guard.release();
                        else
                            res = res->Dup();
	                }
                catch( GDLException& ex2)
                    {
                        throw GDLException(ex.toString() + " or "+ex2.toString());
                    }
            }

        }
    else
	{
// 		match(antlr::RefAST(_t),ASSIGN_REPLACE);
		_t = _t->getFirstChild();
		
		auto_ptr<BaseGDL> r_guard;
		
		if( _t->getType() == FCALL_LIB)
		{

			res=lib_function_call(_t);
			_t = _retTree;
			
			if( !callStack.back()->Contains( res)) 
			r_guard.reset( res);
			
		}
        else
		{
			res=tmp_expr(_t);
			_t = _retTree;
			
			r_guard.reset( res);
			
		}

		
		switch ( _t->getType()) {
		case DEREF:
		{
			l=l_deref(_t);
			_t = _retTree;
			break;
		}
		case VAR:
		case VARPTR:
		{
			l=l_simple_var(_t);
			_t = _retTree;
			break;
		}
		default:
		{
			l=l_function_call(_t);
			_t = _retTree;
			break;
		}
		}
		
		if( res != (*l))
		{
		delete *l;
		*l = res->Dup();     
		
		if( r_guard.get() == res) // owner
		{
		r_guard.release(); 
		}
		else
		res = res->Dup();
		}
    }

	_retTree = startNode->getNextSibling();
	return res;
}

    : #(ASSIGN 
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( res=tmp_expr
                {
                    r_guard.reset( res);
                }
            | res=lib_function_call
                {
                    if( !callStack.back()->Contains( res)) 
                        r_guard.reset( res);
                }
            )
            l=l_expr[ res]
            { 
                if( r_guard.get() == res) // owner
                    r_guard.release();
                else
                    res = res->Dup();
            } // here res is returned!
        )
    | #(ASSIGN_ARRAYEXPR_MFCALL
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( res=tmp_expr
                {
                    r_guard.reset( res);
                }
            | res=lib_function_call
                {
                    if( !callStack.back()->Contains( res)) 
                        r_guard.reset( res);
                }
            )
            l=l_expr[ res]
            { 
                if( r_guard.get() == res) // owner
                    r_guard.release();
                else
                    res = res->Dup();
            } // here res is returned!
        )
    | #(ASSIGN_REPLACE 
            { 
                auto_ptr<BaseGDL> r_guard;
            } 
            ( res=tmp_expr
                {
                    r_guard.reset( res);
                }
            | res=lib_function_call
                {
                    if( !callStack.back()->Contains( res)) 
                        r_guard.reset( res);
                }
            )
            (
              l=l_function_call   // FCALL_LIB, MFCALL, MFCALL_PARENT, FCALL
            | l=l_deref           // DEREF
            | l=l_simple_var      // VAR, VARPTR
            )
            {
                if( res != (*l))
                {
                delete *l;
                *l = res->Dup();     

                if( r_guard.get() == res) // owner
                {
                    r_guard.release(); 
                }
                else
                    res = res->Dup();
                }
            }
        )
    ;



simple_var returns [BaseGDL* res]
{
	assert( _t != NULL);

	if( _t->getType() == VAR)
	{
		ProgNodeP var = _t;
		match(antlr::RefAST(_t),VAR);
		_t = _t->getNextSibling();
		
		BaseGDL* vData=callStack.back()->GetKW( var->varIx);
		
		if( vData == NULL)
		throw GDLException( _t, "Variable is undefined: "+var->getText(),true,false);
		
		res=vData->Dup();
		
	}
	else // VARPTR
	{
		ProgNodeP varPtr = _t;
		match(antlr::RefAST(_t),VARPTR);
		_t = _t->getNextSibling();
		
		BaseGDL* vData=varPtr->var->Data();
		
		if( vData == NULL)
		throw GDLException( _t, "Common block variable is undefined.",true,false);
		
		res=vData->Dup();
		
	}

	_retTree = _t;
	return res;
}
    : var:VAR // DNode.varIx is index into functions/procedures environment
        {
            BaseGDL* vData=callStack.back()->GetKW( var->varIx);
            
            if( vData == NULL)
            throw GDLException( _t, "Variable is undefined: "+var->getText(),true,false);
            
            res=vData->Dup();
        }
    | varPtr:VARPTR // DNode.var   is ptr to common block variable
        {
            BaseGDL* vData=varPtr->var->Data();
            
            if( vData == NULL)
            throw GDLException( _t, "Common block variable is undefined.",true,false);
            
            res=vData->Dup();
        }
    ;

sys_var returns [BaseGDL* res]
{
	BaseGDL* sv = sys_var_nocopy(_t);
	//_t = _retTree;
	
	//BaseGDL* 
    res=sv->Dup();
	
	//_retTree = _t;
	return res;
}
    : sv=sys_var_nocopy 
        {
            res=sv->Dup();
        }
    ;

sys_var_nocopy returns [BaseGDL* res]
{
	ProgNodeP& sysVarRef = _t;
	
// 	match(antlr::RefAST(_t),SYSVAR);
	
	if( sysVarRef->var == NULL) 
	{
	sysVarRef->var=FindInVarList(sysVarList,sysVarRef->getText());
	if( sysVarRef->var == NULL)		    
	throw GDLException( _t, "Not a legal system variable: !"+
	sysVarRef->getText(),true,false);
	}
	
	if( sysVarRef->getText() == "STIME") SysVar::UpdateSTime();
	
	// note: system variables are always defined
	
	_retTree = _t->getNextSibling();;
	return sysVarRef->var->Data(); // no ->Dup()
}
    : sysVar:SYSVAR 
        // ProgNodeP->getText() returns name which has to be searched 
        // in 'sysVarList' (objects.cpp) if ProgNodeP->var is NULL
        {
            if( sysVar->var == NULL) 
            {
                sysVar->var=FindInVarList(sysVarList,sysVar->getText());
                if( sysVar->var == NULL)		    
                throw GDLException( _t, "Not a legal system variable: !"+
                                    sysVar->getText(),true,false);
            }

            if( sysVar->getText() == "STIME") SysVar::UpdateSTime();

            // system variables are always defined
            res=sysVar->var->Data(); // no ->Dup()
        }
    ;

constant returns [BaseGDL* res]
{
	_retTree = _t->getNextSibling();
	return _t->cData->Dup(); 
}
  : CONSTANT
//         {
//             res=c->cData->Dup(); 
//         }
  ;

constant_nocopy returns [BaseGDL* res]
{
	//BaseGDL* 
	_retTree = _t->getNextSibling();
	return _t->cData; // no ->Dup(); 
}
    : c:CONSTANT
        {
            res=c->cData; // no ->Dup(); 
        }
  ;

lib_function_call returns[ BaseGDL* res]
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);
	
	ProgNodeP rTree = _t->getNextSibling();
// 	match(antlr::RefAST(_t),FCALL_LIB);

// 	match(antlr::RefAST(_t),IDENTIFIER);

	ProgNodeP& fl = _t;
	EnvT* newEnv=new EnvT( fl, fl->libFun);//libFunList[fl->funIx]);
	
    parameter_def(_t->getFirstChild(), newEnv);

	// push id.pro onto call stack
	callStack.push_back(newEnv);
	// make the call
	//BaseGDL* 
    res=static_cast<DLibFun*>(newEnv->GetPro())->Fun()(newEnv);
	// *** MUST always return a defined expression
//    if( res == NULL)
//       throw GDLException( _t, "");

	_retTree = rTree;
	return res;
}
	: #(fll:FCALL_LIB //fll:IDENTIFIER
            {
                //EnvT* 
                newEnv=new EnvT( fll, fll->libFun);//libFunList[fl->funIx]);
            }
            parameter_def[ newEnv]
            {
                // push id.pro onto call stack
                callStack.push_back(newEnv);
                // make the call
                res=static_cast<DLibFun*>(newEnv->GetPro())->Fun()(newEnv);
                //*** MUST always return a defined expression
            }
        )
    ;    

lib_function_call_retnew returns[ BaseGDL* res]
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);

	ProgNodeP rTree = _t->getNextSibling();

// 	match(antlr::RefAST(_t),FCALL_LIB_RETNEW);
//	_t = _t->getFirstChild();
// 	match(antlr::RefAST(_t),IDENTIFIER);
	EnvT* newEnv=new EnvT( _t, _t->libFun);//libFunList[fl->funIx]);

// 	_t =_t->getFirstChild();
	
// 	EnvT* newEnv=new EnvT( fl, fl->libFun);//libFunList[fl->funIx]);
    static int n_elementsIx = LibFunIx("N_ELEMENTS");
    static DLibFun* n_elementsFun = libFunList[n_elementsIx];

    if( _t->libFun == n_elementsFun)
        {
            parameter_def_n_elements(_t->getFirstChild(), newEnv);
        }
    else
        {
            parameter_def(_t->getFirstChild(), newEnv);
        }


//	parameter_def(_t->getFirstChild(), newEnv);
	
	// push id.pro onto call stack
	callStack.push_back(newEnv);
	// make the call
	//BaseGDL* 
    res=static_cast<DLibFun*>(newEnv->GetPro())->Fun()(newEnv);
	//*** MUST always return a defined expression
	
	_retTree = rTree;
	return res;
}
	: #(fll:FCALL_LIB_RETNEW //fll:IDENTIFIER
            {
                //EnvT* 
                newEnv=new EnvT( fll, fll->libFun);//libFunList[fl->funIx]);
            }
            parameter_def[ newEnv]
            {
                // push id.pro onto call stack
                callStack.push_back(newEnv);
                // make the call
                res=static_cast<DLibFun*>(newEnv->GetPro())->Fun()(newEnv);
                //*** MUST always return a defined expression
            }
        )
    ;    


function_call returns[ BaseGDL* res]
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);
    BaseGDL *self;
    EnvUDT*  newEnv;
    ProgNodeP startNode = _t;
    ProgNodeP mark;
}
    : ( 
          #(MFCALL 
                self=expr mp:IDENTIFIER
                {  
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv=new EnvUDT( self, mp);

                    self_guard.release();
                }
                parameter_def[ newEnv]
            )
        | #(MFCALL_PARENT 
                self=expr parent:IDENTIFIER p:IDENTIFIER
                {
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv=new EnvUDT( self, p,
                        parent->getText());

                    self_guard.release();
                }
                parameter_def[ newEnv]
            )
        | #(f:FCALL //f:IDENTIFIER
                {
                    SetFunIx( f);
                    
                    newEnv=new EnvUDT( f, funList[f->funIx]);
                }
                parameter_def[ newEnv]
            )
        | #(ARRAYEXPR_MFCALL
            {
                mark = _t;
                _t = _t->getNextSibling(); // skip DOT
            }

            self=expr mp2:IDENTIFIER

            {  
                auto_ptr<BaseGDL> self_guard(self);
        
                try {
                    newEnv=new EnvUDT( self, mp2);
                    self_guard.release();
                }
                catch( GDLException& ex)
                {
                    goto tryARRAYEXPR;
                }
            } 

            parameter_def[ newEnv]
            )
        )        
            {
                // push environment onto call stack
                callStack.push_back(newEnv);
            
                // make the call
                res=call_fun(static_cast<DSubUD*>(newEnv->GetPro())->GetTree());

                _retTree = startNode->getNextSibling();
                return res;

                tryARRAYEXPR:;
                //_t = mark;

                ProgNodeP dot = mark;
                // 	match(antlr::RefAST(_t),DOT);
                _t = mark->getFirstChild();
	
                SizeT nDot=dot->nDot;
                auto_ptr<DotAccessDescT> aD( new DotAccessDescT(nDot+1));
	
                r_dot_array_expr(_t, aD.get());
                _t = _retTree;
                for (; _t != NULL;) {
                    tag_array_expr(_t, aD.get());
                    _t = _retTree;
                }
                res= aD->Resolve();

                _retTree = startNode->getNextSibling();
                return res;
            }
	;	



l_arrayexpr_mfcall [BaseGDL* right] returns [BaseGDL** res]
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);
    BaseGDL *self;
    EnvUDT*  newEnv;
    ProgNodeP startNode = _t;
}
    : #(ARRAYEXPR_MFCALL
        {
            ProgNodeP mark = _t;
            _t = _t->getNextSibling(); // skip DOT
        }

        self=expr mp2:IDENTIFIER

        {  
            auto_ptr<BaseGDL> self_guard(self);
        
            try {
                newEnv=new EnvUDT( self, mp2, "", true);
                self_guard.release();
            }
            catch( GDLException& ex)
            {
                goto tryARRAYEXPR;
            }
        }    

            parameter_def[ newEnv]
        )
        {
            // push environment onto call stack
            callStack.push_back(newEnv);
            
            // make the call
            res=call_lfun(static_cast<DSubUD*>(
                    newEnv->GetPro())->GetTree());

            _retTree = startNode->getNextSibling();
            return res;

            tryARRAYEXPR:;
            _t = mark;
        }   
        #(dot:DOT  // struct assignment
            { 
                SizeT nDot=dot->nDot;
                auto_ptr<DotAccessDescT> aD( new DotAccessDescT(nDot+1));
            } 
            l_dot_array_expr[ aD.get()] 
            (tag_array_expr[ aD.get()] /* nDot times*/ )+ 
        )
        {
            if( right == NULL)
            throw GDLException( _t, 
                                "Struct expression not allowed in this context.",
                                true,false);
            
            aD->Assign( right);

            res=NULL;

            _retTree = startNode->getNextSibling();
            return res;
        }
    ;



l_arrayexpr_mfcall_as_arrayexpr [BaseGDL* right] returns [BaseGDL** res]
    : #(ARRAYEXPR_MFCALL
            #(dot:DOT  // struct assignment
            { 
                SizeT nDot=dot->nDot;
                auto_ptr<DotAccessDescT> aD( new DotAccessDescT(nDot+1));
            } 
            l_dot_array_expr[ aD.get()] 
            (tag_array_expr[ aD.get()] /* nDot times*/ )+ 
            )         
        )
        {
            if( right == NULL)
            throw GDLException( _t, 
                                "Struct expression not allowed in this context.",
                                true,false);
            
            aD->Assign( right);

            res=NULL;
        }
    ;

// function call can be l_values (within (#EXPR ...) only)
l_arrayexpr_mfcall_as_mfcall returns[ BaseGDL** res]
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);
    BaseGDL *self;
    EnvUDT*   newEnv;
}
    : #(ARRAYEXPR_MFCALL
                {
                    _t = _t->getNextSibling(); // skip DOT
                }
 
                self=expr mp2:IDENTIFIER
                {  
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv=new EnvUDT( self, mp2, "", true);

                    self_guard.release();
                }
                parameter_def[ newEnv]
        )
        {
            // push environment onto call stack
            callStack.push_back(newEnv);
            
            // make the call
            res=call_lfun(static_cast<DSubUD*>(
                    newEnv->GetPro())->GetTree());
        }   
	;	

// function call can be l_values (within (#EXPR ...) only)
l_function_call returns[ BaseGDL** res]
{ 
    // better than auto_ptr: auto_ptr wouldn't remove newEnv from the stack
    StackGuard<EnvStackT> guard(callStack);
    BaseGDL *self;
    BaseGDL *libRes;
    EnvUDT*   newEnv;
}

	: #(fl:FCALL_LIB //fl:IDENTIFIER
            {
                EnvT* newEnv=new EnvT( fl, fl->libFun);//libFunList[fl->funIx]);
            }
            parameter_def[ newEnv]
            {
                EnvT* callerEnv = static_cast<EnvT*>(callStack.back());
                // push id.pro onto call stack
                callStack.push_back(newEnv);
                // make the call
                BaseGDL* libRes = 
                static_cast<DLibFun*>(newEnv->GetPro())->Fun()(newEnv);
                
                res = callerEnv->GetPtrTo( libRes);
                if( res == NULL)
                throw GDLException( _t, "Library function must return a "
                    "l-value in this context: "+fl->getText());
            }
        )
    |
        (
        ( #(MFCALL 
                self=expr mp:IDENTIFIER
                {  
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv=new EnvUDT( self, mp, "", true);

                    self_guard.release();
                }
                parameter_def[ newEnv]
            )
        | #(MFCALL_PARENT 
                self=expr parent:IDENTIFIER p:IDENTIFIER
                {
                    auto_ptr<BaseGDL> self_guard(self);
                    
                    newEnv=new EnvUDT( self, p,
                        parent->getText(), true);

                    self_guard.release();
                }
                parameter_def[ newEnv]
            )

        | #(f:FCALL //f:IDENTIFIER
                {
                    SetFunIx( f);
                    
                    newEnv=new EnvUDT( f, funList[f->funIx], true);
                }
                parameter_def[ newEnv]
            )
        )
        {
            // push environment onto call stack
            callStack.push_back(newEnv);
            
            // make the call
            res=call_lfun(static_cast<DSubUD*>(
                    newEnv->GetPro())->GetTree());
        }   
        )
	;	

ref_parameter [EnvBaseT* actEnv] returns[ BaseGDL** ret]
{
	assert(_t != NULL);

		if ( _t->getType() == DEREF) {
			//ret=
            return l_deref( _t);//, actEnv);
// 			_t = _retTree;
		}
		else	
		//case VAR:
		//case VARPTR:
		{
			//ret=
            return l_simple_var(_t);
// 			_t = _retTree;
		}

//  	_retTree = _t;
// 	return ret;
}
    : ret=l_simple_var
    | ret=l_deref
    ;

// the environment is not on the callstack
parameter_def_n_elements [EnvBaseT* actEnv] 
{
    auto_ptr<EnvBaseT> guard(actEnv); 
    _retTree = _t;
//     bool interruptEnableIn = interruptEnable;
    if( _retTree != NULL)
        {
        if( _retTree->getType() == REF ||
            _retTree->getType() == REF_EXPR ||
            _retTree->getType() == REF_CHECK ||
            _retTree->getType() == PARAEXPR)
            {
                try{
//                     interruptEnable = false;
                    static_cast<ParameterNode*>(_retTree)->Parameter( actEnv);
//                     interruptEnable = interruptEnableIn;
                } 
                catch( GDLException& e)
                    {
//                         interruptEnable = interruptEnableIn;
                        if( actEnv->NParam() == 0) 
                            {
                                BaseGDL* nP = NULL;
                                actEnv->SetNextPar( nP);
                            }
                    }
            }
        }
    try{
        while(_retTree != NULL) {
            static_cast<ParameterNode*>(_retTree)->Parameter( actEnv);
        }    
    }
    catch( GDLException& e)
        {
            // update line number, currently set to caller->CallingNode()
            // because actEnv is not on the stack yet, 
            // report caller->Pro()'s name is ok, because we are not inside
            // the call yet
            e.SetErrorNodeP( actEnv->CallingNode());
            throw e;
        }

    actEnv->Extra(); // expand _EXTRA

	guard.release();
	
    return;
}
    : #(KEYDEF_REF_EXPR IDENTIFIER //ref_parameter
        )             
;

// the environment is not on the callstack
parameter_def [EnvBaseT* actEnv] 
{
    auto_ptr<EnvBaseT> guard(actEnv); 

    EnvBaseT* callerEnv = callStack.back();
    EnvBaseT* oldNewEnv = callerEnv->GetNewEnv();
	callerEnv->SetNewEnv( actEnv);

    try{

        _retTree = _t;
        while(_retTree != NULL) {
            static_cast<ParameterNode*>(_retTree)->Parameter( actEnv);
        }    

        actEnv->Extra(); // expand _EXTRA
        
    } 
    catch( GDLException& e)
        {
            callerEnv->SetNewEnv( oldNewEnv);
            // update line number, currently set to caller->CallingNode()
            // because actEnv is not on the stack yet, 
            // report caller->Pro()'s name is ok, because we are not inside
            // the call yet
            e.SetErrorNodeP( actEnv->CallingNode());
            throw e;
        }
    callerEnv->SetNewEnv( oldNewEnv);

	guard.release();
	
    return;
}
    : (  #(KEYDEF_REF IDENTIFIER //ref_parameter
            )
        )
	;

arrayindex_list returns [ArrayIndexListT* aL]
{
    ExprListT        exprList; // for cleanup
    IxExprListT      ixExprList;
    SizeT nExpr;
    BaseGDL* s;

	
	ProgNodeP retTree = _t->getNextSibling();
	ax = _t;
// 	match(antlr::RefAST(_t),ARRAYIX);
	_t = _t->getFirstChild();
	
	aL = ax->arrIxList;
	assert( aL != NULL);
	
	nExpr = aL->NParam();
	if( nExpr == 0)
	{
	aL->Init();
	_retTree = retTree;
	return aL;
	}
	
	while( _t != NULL) {

			switch ( _t->getType()) {
			case CONSTANT:
			case DEREF:
			case SYSVAR:
			case VAR:
			case VARPTR:
			{
				s=indexable_expr(_t);
//				_t = _retTree;
				break;
			}
			case FCALL_LIB:
			{
				s=lib_function_call(_t);
//				_t = _retTree;
				
				if( !callStack.back()->Contains( s)) 
				exprList.push_back( s);
				
				break;
			}
			default:
			{
				s=indexable_tmp_expr(_t);
//				_t = _retTree;
				exprList.push_back( s);
				break;
			}
			} // switch
			
			
			ixExprList.push_back( s);
			if( ixExprList.size() == nExpr)
                break; // allows some manual tuning

            _t = _t->getNextSibling();
	}

	aL->Init( ixExprList);
	
	_retTree = retTree;
	return aL;
}
	: #(ax:ARRAYIX
//             {
//                 aL = ax->arrIxList;
//                 assert( aL != NULL);

//                 nExpr = aL->NParam();
//                 if( nExpr == 0)
//                 {
//                     aL->Init();
//                     goto empty;
//                 }

// //                 if( nExpr > 1)
// //                 {
// //                     ixExprList.reserve( nExpr);
// //                     exprList.reserve( nExpr);
// //                 }
// //                if( nExpr == 0) goto empty;
//             }
            (
                ( s=indexable_expr
                | s=lib_function_call
                    {
                        if( !callStack.back()->Contains( s)) 
                        exprList.push_back( s);
                    }
                | s=indexable_tmp_expr { exprList.push_back( s);}
                )
//                 {
//                     ixExprList.push_back( s);
//                     if( ixExprList.size() == nExpr)
//                         break; // allows some manual tuning
//                 }
            )*
//            { empty: ;}
        )
//         {
//             aL->Init( ixExprList);
//             empty:
//         }
    ;

