// ---------------------------------------------------------------------
// $Id: Optimizer.cc 151 2008-06-04 01:35:38Z daaugusto $
//
//   Optimizer.cc (created on Thu Nov 17 18:25:35 BRT 2005)
// 
//   Genetic Algorithm File Fitter (gaffitter)
//
//   Copyright (C) 2005-2008 Douglas A. Augusto
// 
// This file is part of gaffitter.
// 
// gaffitter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 3 of the License, or (at
// your option) any later version.
// 
// gaffitter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with gaffitter; if not, see <http://www.gnu.org/licenses/>.
//
// ---------------------------------------------------------------------

#include "Optimizer.hh"

#include <utility>
#include <iterator>
#include <algorithm>

// --------------------------------------------------------------------
Params::Size_t
Optimizer::CalcSum(const vector<bool>& v) const
{
   Params::Size_t sum=0.0;

   for(unsigned int i=0; i<v.size(); ++i) if (v[i]) sum += m_files[i].Size();

   return sum;
}

// --------------------------------------------------------------------
bool
Optimizer::Output()
{
   // increment iter
   ++m_cur_iter;

   if (m_params.m_verbose)
   {
      cout << "> Best individual is: ";
      copy(m_best->begin(), m_best->end(), ostream_iterator<bool>(cout));
      cout << endl << flush;
   }

   // comparison function. Should be "by size" or "by name" (reverse or not)
   bool (*cmp_function)(const SizeName&, const SizeName&);

   // sort by name or size, ascending or descending
   if (m_params.m_sort_by_size)
      if (m_params.m_sort_reverse) 
         cmp_function = SizeName::CmpSizeRev;
      else 
         cmp_function = SizeName::CmpSize;
   else // sort by name
      if (m_params.m_sort_reverse)
         if (m_params.m_no_case)
            cmp_function = SizeName::CmpNameRevNocase;
         else
            cmp_function = SizeName::CmpNameRev;
      else
         if (m_params.m_no_case)
            cmp_function = SizeName::CmpNameNocase;
         else
            cmp_function = SizeName::CmpName;

  const int num_input_files = m_files.size();
  vector<SizeName> buffer; // output buffer
  Params::Size_t sum=0, grand_total=0;

  int v_index = 0; // because m_files items will be removed (and so 
                   // the vector will be automatically resized
  for (unsigned int i=0; i<m_best->size(); ++i)
  {
     grand_total += m_files[v_index].Size();

     if ((*m_best)[i]) // selected!
     {
        if (!m_params.m_hide_selected)
        {
           buffer.push_back(m_files[v_index]);
           sum += m_files[v_index].Size();
        }

        // remove it from m_files
        m_files.erase(m_files.begin()+v_index);
     }
     else ++v_index;
  }

  if (!m_params.m_hide_selected)
  {
     // sort the output
     sort(buffer.begin(), buffer.end(), cmp_function);

     // make a blank line (separator) after first iteration
     // In verbose mode this separator is always needed
     if (m_cur_iter > 1 || m_params.m_verbose) cout << endl;

     for (unsigned int i=0; i<buffer.size(); ++i)
     {
        if (m_params.m_enclose)
           cout << m_params.m_enclose_chr << buffer[i].Name() 
                << m_params.m_enclose_chr;
        else
           cout << buffer[i].Name();

        if (m_params.m_show_size) 
           cout << '\t' << m_params.PrettySize(buffer[i].Size());

        cout << m_params.m_delimit_chr;
     }

     // print the summary
     if (!m_params.m_hide_summary)
     {
        cout << endl << "[" << m_cur_iter 
             << "] Sum: " << m_params.PrettySize(sum) << " of "
             << m_params.PrettySize(grand_total) << ", Diff: "
             << m_params.PrettySize(m_params.m_target - sum)
             << (m_params.m_no_metric?", Items: ":", Files: ")
             << buffer.size() << "/" << num_input_files << endl;
     }
  }

  //---- continue to process? ------------------------------------------
  if (m_files.size() > 0)
  {
     if (m_params.m_show_unselected) // print unselected files (+ summary)
     {
        buffer.clear(); sum = 0;

        // copy m_files to buffer for sorting
        buffer.assign(m_files.begin(), m_files.end());

        // sort by name or size, ascending or descending
        sort(buffer.begin(), buffer.end(), cmp_function);

        /* Make a blank line after either the first iteration or if
         already exist a previous output from "selected files"

         In verbose mode this separator is always needed */
        if (m_cur_iter > 1 || !m_params.m_hide_selected || m_params.m_verbose) 
           cout << endl;

        // output
        for (unsigned int i=0; i<buffer.size(); ++i)
        {
           if (m_params.m_enclose)
              cout << m_params.m_enclose_chr << buffer[i].Name() 
                   << m_params.m_enclose_chr;
           else
              cout << buffer[i].Name();

           if (m_params.m_show_size) 
              cout << '\t' << m_params.PrettySize(buffer[i].Size());

           cout << m_params.m_delimit_chr;

           sum += buffer[i].Size();
        }

        // print the summary for the unselected files
        if (!m_params.m_hide_summary)
        {
           cout << endl << "[" << m_cur_iter << "] <UNSELECTED> Sum: " 
                << m_params.PrettySize(sum) << " of "
                << m_params.PrettySize(grand_total)
                << (m_params.m_no_metric?", Items: ":", Files: ")
                << m_files.size() << "/" << num_input_files
                << endl;
        }
     }

     // Check if the user want more iterations
     if (--m_params.m_iterations > 0) return true;
  }

  // Work complete; terminate the execution
  return false;
}

// --------------------------------------------------------------------
