/* -*- mode: c++ -*-

  This file is part of the Feel library

  Author(s): Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
       Date: 2009-11-13

  Copyright (C) 2009 Universit Joseph Fourier (Grenoble I)

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/**
   \file heatSink.hpp
   \author Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
   \date 2009-11-13
 */
#ifndef __HeatSink2D_H
#define __HeatSink2D_H 1

#include <boost/timer.hpp>
#include <boost/shared_ptr.hpp>

#include <feel/options.hpp>
#include <feel/feelcore/feel.hpp>

#include <feel/feelalg/backend.hpp>

#include <feel/feeldiscr/functionspace.hpp>
#include <feel/feeldiscr/region.hpp>
#include <feel/feelpoly/im.hpp>

#include <feel/feelfilters/gmsh.hpp>
#include <feel/feelfilters/exporter.hpp>
#include <feel/feelpoly/polynomialset.hpp>
#include <feel/feelalg/solvereigen.hpp>

#include <feel/feelvf/vf.hpp>
#include <feel/feelcrb/parameterspace.hpp>

#include <feel/feeldiscr/bdf2.hpp>

#include <Eigen/Core>
#include <Eigen/LU>
#include <Eigen/Dense>




namespace Feel
{

po::options_description
makeHeatSink2DOptions()
{
    po::options_description heatsink2doptions("HeatSink2D options");
    heatsink2doptions.add_options()
    // mesh parameters
    ("hsize", Feel::po::value<double>()->default_value( 0.0001 ),
     "first h value to start convergence")
    ("L_ref", Feel::po::value<double>()->default_value( 0.02 ),
     "dimensional length of the sink (in meters)")
    ("width", Feel::po::value<double>()->default_value( 0.0005 ),
     "dimensional width of the fin (in meters)")

	// density parameter
	("rho_s", Feel::po::value<double>()->default_value( 8940 ),
	 "density of the spreader's material in SI unit kg.m^{-3}")
	("rho_f", Feel::po::value<double>()->default_value( 8940 ),
	 "density of the fin's material in SI unit kg.m^{-3}")

	// heat capacities parameter
	("c_s", Feel::po::value<double>()->default_value( 385 ),
	 "heat capacity of the spreader's material in SI unit J.kg^{-1}.K^{-1}")
	("c_f", Feel::po::value<double>()->default_value( 385 ),
	 "heat capacity of the fin's material in SI unit J.kg^{-1}.K^{-1}")

	// physical coeff
    ("therm_coeff", Feel::po::value<double>()->default_value(1e3),
     "thermal coefficient")
    ("Tamb", Feel::po::value<double>()->default_value(300),
     "ambiant temperature")
    ("heat_flux", Feel::po::value<double>()->default_value(1e6),
     "heat flux generated by CPU")

    ("steady", Feel::po::value<bool>()->default_value(false),
     "if true : steady else unsteady")

    // export
    ("export-matlab", "export matrix and vectors in matlab" );
        ;
    return heatsink2doptions.add( Feel::feel_options() ).add(bdf_options("heatSink2d"));;
}
AboutData
makeHeatSink2DAbout( std::string const& str = "heatSink" )
{
    Feel::AboutData about( str.c_str(),
                           str.c_str(),
                           "0.1",
                           "heat sink Benchmark",
                           Feel::AboutData::License_GPL,
                           "Copyright (c) 2010,2011 Universit de Grenoble 1 (Joseph Fourier)");

    about.addAuthor("Christophe Prud'homme", "developer", "christophe.prudhomme@ujf-grenoble.fr", "");
    about.addAuthor("Stephane Veys", "developer", "stephane.veys@imag.fr", "");
    return about;
}



/**
 * \class HeatSink2D
 * \brief brief description
 *
 * @author Christophe Prud'homme
 * @see
 */
class HeatSink2D
{
public:


    /** @name Constants
     */
    //@{

    static const uint16_type Order = 5;
    static const uint16_type ParameterSpaceDimension = 3;

    //@}

    /** @name Typedefs
     */
    //@{

    typedef double value_type;

    typedef Backend<value_type> backend_type;
    typedef boost::shared_ptr<backend_type> backend_ptrtype;

    /*matrix*/
    typedef backend_type::sparse_matrix_type sparse_matrix_type;
    typedef backend_type::sparse_matrix_ptrtype sparse_matrix_ptrtype;
    typedef backend_type::vector_type vector_type;
    typedef backend_type::vector_ptrtype vector_ptrtype;

    typedef Eigen::Matrix<double,Eigen::Dynamic,Eigen::Dynamic> eigen_matrix_type;
    typedef eigen_matrix_type ematrix_type;
    typedef boost::shared_ptr<eigen_matrix_type> eigen_matrix_ptrtype;

    /*mesh*/
    typedef Simplex<2,1> entity_type;
    typedef Mesh<entity_type> mesh_type;
    typedef boost::shared_ptr<mesh_type> mesh_ptrtype;

    typedef FunctionSpace<mesh_type, fusion::vector<Lagrange<0, Scalar> >, Discontinuous> p0_space_type;
    typedef typename p0_space_type::element_type p0_element_type;

    /*basis*/
    typedef fusion::vector<Lagrange<Order, Scalar> > basis_type;

    /*space*/
    typedef FunctionSpace<mesh_type, basis_type, value_type> space_type;
    typedef boost::shared_ptr<space_type> space_ptrtype;
    typedef space_type functionspace_type;
    typedef space_ptrtype functionspace_ptrtype;
    typedef typename space_type::element_type element_type;
    typedef boost::shared_ptr<element_type> element_ptrtype;

    /* export */
    typedef Exporter<mesh_type> export_type;
    typedef boost::shared_ptr<export_type> export_ptrtype;


    /* parameter space */
    typedef ParameterSpace<ParameterSpaceDimension> parameterspace_type;
    typedef boost::shared_ptr<parameterspace_type> parameterspace_ptrtype;
    typedef parameterspace_type::element_type parameter_type;
    typedef parameterspace_type::element_ptrtype parameter_ptrtype;
    typedef parameterspace_type::sampling_type sampling_type;
    typedef parameterspace_type::sampling_ptrtype sampling_ptrtype;

    /* time discretization */
	typedef Bdf<space_type>  bdf_type;
	typedef boost::shared_ptr<bdf_type> bdf_ptrtype;


    typedef Eigen::VectorXd theta_vector_type;

    typedef boost::tuple<std::vector<sparse_matrix_ptrtype>, std::vector<std::vector<vector_ptrtype>  > > affine_decomposition_type;
    //@}

    /** @name Constructors, destructor
     */
    //@{

    //! default constructor
    HeatSink2D();

    //! constructor from command line
    HeatSink2D( po::variables_map const& vm );


    //! copy constructor
    HeatSink2D( HeatSink2D const & );
    //! destructor
    ~HeatSink2D(){}

    //! initialisation of the model
    void init();
    //@}

    /** @name Operator overloads
     */
    //@{

    //@}

    /** @name Accessors
     */
    //@{

    // \return the number of terms in affine decomposition of left hand
    // side bilinear form
    int Qa() const { return 5; }

    /**
     * there is at least one output which is the right hand side of the
     * primal problem
     *
     * \return number of outputs associated to the model
     * in our case we have a compliant output and 2 others outputs : average temperature on boundaries
     */
    int Nl() const { return 3; }

    /**
     * \param l the index of output
     * \return number of terms  in affine decomposition of the \p q th output term
     * in our case no outputs depend on parameters
     */
    int Ql( int l ) const { return 2;  }

    /**
     * \brief Returns the function space
     */
    space_ptrtype functionSpace() { return Xh; }

    //! return the parameter space
    parameterspace_ptrtype parameterSpace() const { return M_Dmu;}

    /**
     * \brief compute the theta coefficient for both bilinear and linear form
     * \param mu parameter to evaluate the coefficients
     * list of parameters : k_f k_s and L
     */
    boost::tuple<theta_vector_type, std::vector<theta_vector_type> >
    computeThetaq( parameter_type const& mu )
        {

            double kf = mu( 0 );
            double ks = mu( 1 );
            double L  = mu( 2 );
            double detJ = L/L_ref;


            M_thetaAq.resize( Qa() );
            M_thetaAq( 0 ) = detJ;
            M_thetaAq( 1 ) = 1;
            M_thetaAq( 2 ) = ks;
            M_thetaAq( 3 ) = kf*detJ;
            M_thetaAq( 4 ) = kf*(L_ref*L_ref)/(L*L)*detJ;

            M_thetaFq.resize( Nl() );

            M_thetaFq[0].resize( Ql(0) );
            M_thetaFq[0]( 0 ) = 1;
            M_thetaFq[0]( 1 ) = detJ;

            M_thetaFq[1].resize( Ql(1) );
            M_thetaFq[1]( 0 ) = 1;
            M_thetaFq[1]( 1 ) = detJ;

            M_thetaFq[2].resize( Ql(2) );
            M_thetaFq[2]( 0 ) = 1;
            M_thetaFq[2]( 1 ) = detJ;

           return boost::make_tuple( M_thetaAq, M_thetaFq );
        }

    /**
     * \brief return the coefficient vector
     */
    theta_vector_type const& thetaAq() const { return M_thetaAq; }

    /**
     * \brief return the coefficient vector
     */
    std::vector<theta_vector_type> const& thetaFq() const { return M_thetaFq; }

    /**
     * \brief return the coefficient vector \p q component
     *
     */
    value_type thetaAq( int q ) const
        {
            return M_thetaAq( q );
        }

    /**
     * \return the \p q -th term of the \p l -th output
     */
    value_type thetaL( int l, int q ) const
        {
            return M_thetaFq[l]( q );
        }

    //@}

    /** @name  Mutators
     */
    //@{

    /**
     * set the mesh characteristic length to \p s
     */
    void setMeshSize( double s ) { meshSize = s; }


    //@}

    /** @name  Methods
     */
    //@{

    /**
     * run the convergence test
     */

    /**
     * create a new matrix
     * \return the newly created matrix
     */
    sparse_matrix_ptrtype newMatrix() const;

    /**
     * \brief Returns the affine decomposition
     */
    affine_decomposition_type computeAffineDecomposition();

    /**
     * \brief solve the model for parameter \p mu
     * \param mu the model parameter
     * \param T the temperature field
     */
    void solve( parameter_type const& mu, element_ptrtype& T, int output_index=0 );

    eigen_matrix_type solutionMatrix(){return M_solution_matrix;}
    eigen_matrix_type dualSolutionMatrix(){return Mdu_solution_matrix;}
    void fillSolutionMatrix ( int Nsnap, parameter_type const& mu, int output_index);
    bool isSteady () {return steady;}
    void pfemMode (bool mode) {pfem = mode;}
    void assemble();
    void computeTimeStep(int Nsnap);

    /**
     * solve for a given parameter \p mu
     */
    void solve( parameter_type const& mu );

    /**
     * solve \f$ M u = f \f$
     */
    void l2solve( vector_ptrtype& u, vector_ptrtype const& f );


    /**
     * update the PDE system with respect to \param mu
     */
    void update( parameter_type const& mu , int output_index=0);
    //@}

    /**
     * export results to ensight format (enabled by  --export cmd line options)
     */
    void exportResults(double time, element_type& T );

    void solve( sparse_matrix_ptrtype& ,element_type& ,vector_ptrtype&  );

    /**
     * returns the scalar product of the boost::shared_ptr vector x and
     * boost::shared_ptr vector y
     */
    double scalarProduct( vector_ptrtype const& X, vector_ptrtype const& Y );

    /**
     * returns the scalar product of the vector x and vector y
     */
    double scalarProduct( vector_type const& x, vector_type const& y );

    /**
     * specific interface for OpenTURNS
     *
     * \param X input vector of size N
     * \param N size of input vector X
     * \param Y input vector of size P
     * \param P size of input vector Y
     */
    void run( const double * X, unsigned long N, double * Y, unsigned long P );

    /**
     * Given the output index \p output_index and the parameter \p mu, return
     * the value of the corresponding FEM output
     */
    value_type output( int output_index, parameter_type const& mu );

    gmsh_ptrtype makefin( double hsize, double width, double L_ref );

private:

	/* mesh parameters */
    double meshSize;
    double L_ref;//reference value for geometric parameter
    double width;

	/* density of the materials */
	double rho_s;
	double rho_f;

	/* heat capacity of the materials*/
	double c_s;
	double c_f;

	/* thermal coeff */
    double therm_coeff;

	/* ambien temperature, and heat flux (Q) */
    double Tamb;
    double heat_flux;

    /* surfaces*/
    double surface_gamma4, surface_gamma1;

    /* mesh, pointers and spaces */
    mesh_ptrtype mesh;
    space_ptrtype Xh;

    sparse_matrix_ptrtype D,M,A_du;
    vector_ptrtype F;

	/* time management */
	//bdf_ptrtype M_bdf;
    bool steady;

    boost::shared_ptr<export_type> exporter;


    po::variables_map M_vm;
    backend_ptrtype backend;

    bool M_do_export;
    //    export_ptrtype exporter;

    element_ptrtype pT;

    std::vector<sparse_matrix_ptrtype> M_Aq;
    std::vector<sparse_matrix_ptrtype> M_Aq_du;
    std::vector<std::vector<vector_ptrtype> > M_Fq;

    parameterspace_ptrtype M_Dmu;
    theta_vector_type M_thetaAq;
    std::vector<theta_vector_type> M_thetaFq;

	bdf_ptrtype M_bdf;
    double M_pod_time_step;
    bool M_fill_solution_matrix;
    bool pfem;
    double M_Nsnap;
    double M_initial_time_step;
    Eigen::Matrix<double,Eigen::Dynamic,Eigen::Dynamic> M_solution_matrix;
    Eigen::Matrix<double,Eigen::Dynamic,Eigen::Dynamic> Mdu_solution_matrix;


};
HeatSink2D::HeatSink2D()
    :
    backend( backend_type::build( BACKEND_PETSC ) ),
    steady( false ),
    meshSize( 1e-4 ),
    L_ref( 0.02 ),
    width( 5e-4 ),
    rho_s ( 8940 ),
    rho_f (8940 ),
    c_s ( 385 ),
    c_f ( 385 ),
    therm_coeff( 1e3 ),
    Tamb ( 300 ),
    heat_flux ( 1e6 ),
    M_do_export( true ),
    exporter( Exporter<mesh_type>::New( "gmsh" ) ),
    M_Dmu( new parameterspace_type )
{
  this->init();
}

HeatSink2D::HeatSink2D( po::variables_map const& vm )
    :
    M_vm( vm ),
    backend( backend_type::build( vm ) ),
    steady( vm["steady"].as<bool>() ),
    meshSize( vm["hsize"].as<double>() ),
    L_ref( vm["L_ref"].as<double>() ),
    width( vm["width"].as<double>() ),
    rho_s ( vm["rho_s"].as<double>() ),
    rho_f (vm["rho_f"].as<double>() ),
    c_s ( vm["c_s"].as<double>() ),
    c_f ( vm["c_f"].as<double>() ),
    therm_coeff( vm["therm_coeff"].as<double>() ),
    Tamb ( vm["Tamb"].as<double>() ),
    heat_flux ( vm["heat_flux"].as<double>() ),
    M_do_export( !vm.count( "no-export" ) ),
    exporter( Exporter<mesh_type>::New( vm, "heatsink" ) ),
    M_Dmu( new parameterspace_type )
{
  this->init();
}





/**
 * makefin creates the .geo file and returns a gmsh pointer type.
 * Be carefull, the problem considered here is dimensional
 *
 * \arg hsize : size of the mesh
 * \arg width : the width of the fin
 * \arg L : dimensional length of the sink (in meters)
 */
gmsh_ptrtype
HeatSink2D::makefin( double hsize, double width, double L_ref)
{
    std::ostringstream ostr;
        ostr << "Mesh.MshFileVersion = 2.1;\n"
             << "Point (1) = {0, 0, 0, " << hsize << "};\n"
             << "Point (2) = {0.001, 0, 0, " << hsize << "};\n"
             << "Point (3) = {0.001, 0.001, 0, " << hsize << "};\n"
             << "Point (4) = {"<< width <<", 0.001, 0, " << hsize << "};\n"
             << "Point (5) = {0, 0.001, 0, " << hsize << "};\n"
             << "Point (6) = {"<< width <<", "<<0.0005+L_ref<<", 0, " << hsize << "};\n"
             << "Point (7) = {0, "<<0.0005+L_ref<<", 0, " << hsize << "};\n"
             << "Line (1) = {1, 2};\n"
             << "Line (2) = {2, 3};\n"
             << "Line (3) = {3, 4};\n"
             << "Line (4) = {4, 5};\n"
             << "Line (5) = {4, 6};\n"
             << "Line (6) = {6, 7};\n"
             << "Line (7) = {7, 5};\n"
             << "Line (8) = {5, 1};\n"
             << "Line Loop (9) = {1, 2, 3, 4, 8};\n"
             << "Line Loop (10) = {5, 6, 7, -4};\n"
             << "Plane Surface (9) = {9};\n"
             << "Plane Surface (10) = {10};\n"
            // physical entities
			 << "Physical Line (\"gamma1\") = {5};\n"
			 << "Physical Line (\"gamma2\") = {6};\n"
			 << "Physical Line (\"gamma3\") = {4};\n"
			 << "Physical Line (\"gamma4\") = {1};\n"
			 << "Physical Line (\"gamma5\") = {3};\n"
			 << "Physical Line (\"gamma6\") = {7};\n"
			 << "Physical Line (\"gamma7\") = {2};\n"
             << "Physical Line (\"gamma8\") = {8};\n"
             << "Physical Surface (\"spreader_mesh\") = {9};\n"
             << "Physical Surface (\"fin_mesh\") = {10};\n";

    std::ostringstream nameStr;
    nameStr.precision( 3 );
    nameStr << "fin_sink";
    gmsh_ptrtype gmshp( new Gmsh );
    gmshp->setPrefix( nameStr.str() );
    gmshp->setDescription( ostr.str() );
    return gmshp;
}



void HeatSink2D::init()
{

    using namespace Feel::vf;

    M_fill_solution_matrix=false;//by default

    /*
     * First we create the mesh
     */
    mesh = createGMSHMesh ( _mesh = new mesh_type,
                            _desc = makefin( meshSize, width, L_ref),
                            _update=MESH_UPDATE_FACES | MESH_UPDATE_EDGES );


    /*
     * The function space and some associate elements are then defined
     */
    Xh = space_type::New( mesh );
    // allocate an element of Xh
    pT = element_ptrtype( new element_type( Xh ) );



    surface_gamma4 = integrate( _range= markedfaces(mesh,"gamma4"), _expr=cst(1.) ).evaluate()(0,0);
    surface_gamma1 = integrate( _range= markedfaces(mesh,"gamma1"), _expr=cst(1.) ).evaluate()(0,0);

    M_bdf = bdf( _space=Xh, _vm=M_vm, _name="heatSink2d" , _prefix="heatSink2d");

    M_Aq.resize( 5 );
    M_Aq[0] = backend->newMatrix( Xh, Xh );
    M_Aq[1] = backend->newMatrix( Xh, Xh );
    M_Aq[2] = backend->newMatrix( Xh, Xh );
    M_Aq[3] = backend->newMatrix( Xh, Xh );
    M_Aq[4] = backend->newMatrix( Xh, Xh );


    M_Aq_du.resize( 5 );
    M_Aq_du[0] = backend->newMatrix( Xh, Xh );
    M_Aq_du[1] = backend->newMatrix( Xh, Xh );
    M_Aq_du[2] = backend->newMatrix( Xh, Xh );
    M_Aq_du[3] = backend->newMatrix( Xh, Xh );
    M_Aq_du[4] = backend->newMatrix( Xh, Xh );

    //three outputs : one "compliant" and two others
    M_Fq.resize( 3 );
    //first : the compliant case.
    M_Fq[0].resize(2 );
    M_Fq[0][0] = backend->newVector( Xh );
    M_Fq[0][1] = backend->newVector( Xh );
    //second output : non compliant case.
    M_Fq[1].resize( 2 );
    M_Fq[1][0] = backend->newVector( Xh );
    M_Fq[1][1] = backend->newVector( Xh );
    //third output : non compliant case.
    M_Fq[2].resize( 2 );
    M_Fq[2][0] = backend->newVector( Xh );
    M_Fq[2][1] = backend->newVector( Xh );



    D = backend->newMatrix( Xh, Xh );
    F = backend->newVector( Xh );
    A_du = backend->newMatrix( Xh, Xh );

    //static const int N = 2;
    //3 parameters : k_f k_s and L
    Feel::ParameterSpace<3>::Element mu_min( M_Dmu );
    mu_min << 100, 100, 0.02;
    M_Dmu->setMin( mu_min );
    Feel::ParameterSpace<3>::Element mu_max( M_Dmu );
    mu_max << 500, 500, 0.02;
    M_Dmu->setMax( mu_max );


    Log() << "Number of dof " << Xh->nLocalDof() << "\n";

    if( steady )
    {
        assemble();
    }
    else
    {
        M_initial_time_step = M_bdf->timeStep();
    }

} // HeatSink2D::init



void
HeatSink2D::computeTimeStep(int Nsnap)
{

    M_Nsnap = Nsnap;
    M_pod_time_step = M_bdf->timeFinal()/(Nsnap);
    assemble();
}



void HeatSink2D::assemble()
{

    using namespace Feel::vf;

    element_type u( Xh, "u" );
    element_type v( Xh, "v" );

    if(pfem || !M_fill_solution_matrix) M_bdf->setTimeStep(M_initial_time_step);
    else M_bdf->setTimeStep(M_pod_time_step);

    // geometric transformation :
    // x --> x
    // y --> L/L_ref y
    // z --> z
    //
    // so the jacobian is given by
    //  ( 1  0        0 )
    //  ( 0  L/L_ref  0 )
    //  ( 0  0        1 )
    //
    // and its determinant is L/L_ref
    //
    // note that the inverse jacobian is given by
    //  ( 1  0        0 )
    //  ( 0  L_ref/L  0 )
    //  ( 0  0        1 )


    // here are constructed part which not vary in time and not depend on the geometric parameter

    // compliant case : right hand side
    // if there is no geometric parameter it should be :
    // form1( _test=Xh, _vector=M_Fq[0][0], _init=true ) = integrate( _range= markedfaces(mesh, "gamma1"), _expr= therm_coeff*Tamb*id(v));
    // but the length of gamma1 is a geometric parameter so we need to do computations on a reference domain and then write transformations



    form1( _test=Xh, _vector=M_Fq[0][1], _init=true) = integrate( _range=markedfaces(mesh,"gamma1"), _expr=therm_coeff*Tamb*id(v) );

    form2(Xh, Xh, M_Aq[0],_init=true) =
        integrate( _range=markedfaces(mesh, "gamma1"), _expr= therm_coeff*idt(u)*id(v) );


    form2( Xh, Xh, M_Aq[2], _init=true ) = integrate( _range= markedelements(mesh,"spreader_mesh"), _expr= gradt(u)*trans(grad(v)) );

    form2( Xh, Xh, M_Aq[3], _init=true) = integrate( _range= markedelements(mesh,"fin_mesh"),_expr=  dx(v)*dxt(u)  );
    form2( Xh, Xh, M_Aq[4], _init=true) = integrate( _range= markedelements(mesh,"fin_mesh"),_expr=  dy(v)*dyt(u)  );

    form2(Xh, Xh, M_Aq[1],_init=true) ;

    if(!steady)
    {
        form2(Xh, Xh, M_Aq[0]) += integrate( _range=markedelements(mesh, "fin_mesh"), _expr=rho_f*c_f*idt(u)*id(v)*M_bdf->polyDerivCoefficient(0) );

        form2(Xh, Xh, M_Aq[1]) =
            integrate( _range=markedelements(mesh, "spreader_mesh"), _expr=rho_s*c_s*idt(u)*id(v)*M_bdf->polyDerivCoefficient(0) );
    }

    form1( _test=Xh, _vector=M_Fq[1][0], _init=true ) = integrate( _range=markedfaces(mesh,"gamma4"), _expr=(1./surface_gamma4)*id(v) ) ;
    form1( _test=Xh, _vector=M_Fq[2][1], _init=true ) = integrate( _range=markedfaces(mesh,"gamma1"), _expr=(1./surface_gamma1)*id(v) ) ;


    M_Aq[0]->close();
    M_Aq[1]->close();
    M_Aq[2]->close();
    M_Aq[3]->close();
    M_Aq[4]->close();




    //for scalarProduct
    M = backend->newMatrix( Xh, Xh );
    form2( Xh, Xh, M, _init=true ) =
        integrate( elements(mesh), id(u)*idt(v) + grad(u)*trans(gradt(u)) );
    M->close();



    //adjoint
    form1( _test=Xh, _vector=M_Fq[1][1], _init=true ) = integrate( _range=markedfaces(mesh,"gamma1"), _expr=therm_coeff*id(v) );
    form1( _test=Xh, _vector=M_Fq[2][1] ) += integrate( _range=markedfaces(mesh,"gamma1"), _expr=therm_coeff*id(v) );

    form2( Xh, Xh, M_Aq_du[2], _init=true ) = integrate( _range= markedelements(mesh,"spreader_mesh"), _expr= gradt(u)*trans(grad(v)) );

    form2( Xh, Xh, M_Aq_du[3], _init=true) = integrate( _range= markedelements(mesh,"fin_mesh"),_expr=  dx(v)*dxt(u)  );

    form2( Xh, Xh, M_Aq_du[4], _init=true) = integrate( _range= markedelements(mesh,"fin_mesh"),_expr=  dy(v)*dyt(u)  );

    form2(Xh, Xh, M_Aq_du[0],_init=true) = integrate( _range=markedelements(mesh, "fin_mesh"), _expr=-rho_f*c_f*idt(u)*id(v)*M_bdf->polyDerivCoefficient(0) );
    form2(Xh, Xh, M_Aq_du[0]) += integrate( _range=markedfaces(mesh,"gamma1"), _expr=-rho_f*c_f*idt(u)*id(v) );

    form2(Xh, Xh, M_Aq_du[1], _init=true) =
        integrate( _range=markedelements(mesh, "spreader_mesh"), _expr=-rho_s*c_s*idt(u)*id(v)*M_bdf->polyDerivCoefficient(0) );


    M_Aq_du[0]->close();
    M_Aq_du[1]->close();
    M_Aq_du[2]->close();
    M_Aq_du[3]->close();
    M_Aq_du[4]->close();

}


typename HeatSink2D::sparse_matrix_ptrtype
HeatSink2D::newMatrix() const
{
    return backend->newMatrix( Xh, Xh );
}


typename HeatSink2D::affine_decomposition_type
HeatSink2D::computeAffineDecomposition()
{
    return boost::make_tuple( M_Aq, M_Fq );
}


void HeatSink2D::solve( sparse_matrix_ptrtype& D,
               element_type& u,
               vector_ptrtype& F )
{

    vector_ptrtype U( backend->newVector( u.functionSpace() ) );
    backend->solve( D, D, U, F );
    u = *U;
}


void HeatSink2D::exportResults(double time, element_type& T )
{
    if ( M_do_export )
    {
        Log() << "exportResults starts\n";
        exporter->step(time)->setMesh( T.functionSpace()->mesh() );
        exporter->step(time)->add( "T", T );
        exporter->save();
    }
}


void HeatSink2D::update( parameter_type const& mu, int output_index )
{

    *D = *M_Aq[0];

    for( size_type q = 1;q < M_Aq.size(); ++q )
    {
        //std::cout << "[affine decomp] scale q=" << q << " with " << M_thetaAq[q] << "\n";
        D->addMatrix( M_thetaAq[q], M_Aq[q] );
    }

    F->close();
    F->zero();
    if( output_index >= M_Fq.size() )
    {
        std::cout<<"[HeatSink2D::update] ERROR : output_index = "<<output_index<<" and M_Fq.size() = "<<M_Fq.size()<<std::endl;
        exit(0);
    }
    for( size_type q = 0;q < M_Fq[output_index].size(); ++q )
    {
        //std::cout << "[affine decomp] scale q=" << q << " with " << M_thetaFq[0][q] << "\n";
        F->add( M_thetaFq[output_index][q], M_Fq[output_index][q] );
    }


    *A_du = *M_Aq_du[0];
    for( size_type q = 1;q < M_Aq_du.size(); ++q )
    {
        A_du->addMatrix( M_thetaAq[q], M_Aq_du[q] );
    }


}



void HeatSink2D::fillSolutionMatrix (const int Nsnap, parameter_type const& mu, int output_index)
{

    M_fill_solution_matrix=true;
    M_bdf->setTimeStep(M_pod_time_step);

    const int Ndof = Xh->nDof();//number of dofs used
    M_solution_matrix.resize(Ndof,Nsnap);
    Mdu_solution_matrix.resize(Ndof,Nsnap);

    solve(mu , pT);
}


void HeatSink2D::solve( parameter_type const& mu )
{
    element_ptrtype T( new element_type( Xh ) );
    this->solve( mu, T );
}


void HeatSink2D::solve( parameter_type const& mu, element_ptrtype& T, int output_index )
{

    //verification
    if( pfem ) M_fill_solution_matrix=false;

    using namespace Feel::vf;
    element_type v( Xh, "v" );//test functions

    assemble();

    //there a no time-dependent parameters
    this->computeThetaq( mu );

    int column_index=0;//usefull to fill the solution matrix

    M_bdf->initialize(*T);
    if(steady)
    {
        form1( _test=Xh, _vector=M_Fq[0][0] , _init=true) +=
            integrate( _range=markedfaces(mesh,"gamma4"), _expr= heat_flux*(1)*id(v) );

        M_Fq[0][0]->close();
        M_Fq[0][1]->close();

        this->update( mu );
        backend->solve( _matrix=D,  _solution=T, _rhs=F, _prec=D );
        this->exportResults(1, *T );
    }
    else
    {
        for ( M_bdf->start(); !M_bdf->isFinished() ; M_bdf->next() )
        {

            auto bdf_poly = M_bdf->polyDeriv();
            form1( _test=Xh, _vector=M_Fq[0][0], _init=true ) =
                integrate( _range=markedelements(mesh, "spreader_mesh"), _expr=rho_s*c_s*idv(bdf_poly)*id(v) ) +
                integrate( _range=markedfaces(mesh,"gamma4"), _expr= heat_flux*(1-exp(-M_bdf->time()))*id(v) );
            form1( _test=Xh, _vector=M_Fq[0][1] ) +=
                integrate( _range=markedelements(mesh, "fin_mesh"), _expr=rho_f*c_f*idv(bdf_poly)*id(v)  ) ;

            M_Fq[0][0]->close();
            M_Fq[0][1]->close();

            this->update( mu );
            backend->solve( _matrix=D,  _solution=T, _rhs=F, _prec=D );
            this->exportResults(M_bdf->time(), *T );
            if(M_fill_solution_matrix)
            {
                     element_type solution=*T;
                    for(int i=0;i<solution.size();i++)
                    {
                        M_solution_matrix(i,column_index)=solution[i];
                        if (solution[i]>1e10)
                            {
                                Log()<<"[HeatSink2D::solve] problem with solution of model : solution["<<i<<"] = "<<solution[i]<<"\n";
                            }
                    }

            }
            column_index++;
            M_bdf->shiftRight( *T );
        }
        if(M_fill_solution_matrix)
        {
            //and now we compute dual solution
            column_index=0;
            M_pod_time_step=-M_pod_time_step;
            M_bdf->setTimeStep(M_pod_time_step);
            M_bdf->setTimeInitial(M_bdf->timeFinal());
            M_bdf->setTimeFinal(0);
            for ( M_bdf->start(); M_bdf->time()>=M_bdf->timeFinal(); M_bdf->next() )
            {

                M_Fq[0][0]->zero();
                M_Fq[2][0]->zero();

                auto bdf_poly = M_bdf->polyDeriv();
                for(int i=0;i<M_Fq[output_index].size(); i++) M_Fq[output_index][i]->scale(-1);
                form1( _test=Xh, _vector=M_Fq[output_index][0] ) += integrate( _range=markedelements(mesh, "spreader_mesh"), _expr=-rho_s*c_s*idv(bdf_poly)*id(v) );
                form1( _test=Xh, _vector=M_Fq[output_index][1] ) += integrate( _range=markedelements(mesh, "fin_mesh"), _expr=-rho_f*c_f*idv(bdf_poly)*id(v)  );

                M_Fq[0][0]->close();
                M_Fq[0][1]->close();
                M_Fq[1][0]->close();
                M_Fq[1][1]->close();
                M_Fq[2][0]->close();
                M_Fq[2][1]->close();


                this->update( mu, output_index );
                for(int i=0;i<M_Fq[output_index].size(); i++) M_Fq[output_index][i]->scale(-1);
                backend->solve( _matrix=A_du,  _solution=T, _rhs=F, _prec=A_du );
                //fill the matrix
                element_type adjoint=*T;
                for(int i=0;i<adjoint.size();i++)
                {
                    Mdu_solution_matrix(i,column_index)=adjoint[i];
                    if (adjoint[i]>1e10)
                    {
                        Log()<<"[HeatSink2D::solve] problem with adjoint of model : adjoint["<<i<<"] = "<<adjoint[i]<<"\n";
                    }
                }
                column_index++;
                M_bdf->shiftRight( *T );
            }
            M_pod_time_step=-M_pod_time_step;
            M_bdf->setTimeStep(M_pod_time_step);
            M_bdf->setTimeFinal(M_bdf->timeInitial());
            M_bdf->setTimeInitial(0);
        }
    }
}

void HeatSink2D::l2solve( vector_ptrtype& u, vector_ptrtype const& f )
{
    //std::cout << "l2solve(u,f)\n";
    backend->solve( _matrix=M,  _solution=u, _rhs=f, _prec=M );
    //std::cout << "l2solve(u,f) done\n";
}


double HeatSink2D::scalarProduct( vector_ptrtype const& x, vector_ptrtype const& y )
{
    return M->energy( x, y );
}

double HeatSink2D::scalarProduct( vector_type const& x, vector_type const& y )
{
    return M->energy( x, y );
}


void HeatSink2D::run( const double * X, unsigned long N, double * Y, unsigned long P )
{
    using namespace vf;
    Feel::ParameterSpace<3>::Element mu( M_Dmu );
    mu << X[0], X[1], X[2];
    static int do_init = true;
    if ( do_init )
    {
        meshSize = X[3];
        this->init();
        do_init = false;
    }
    this->solve( mu, pT );

    double mean = integrate( elements(mesh),idv(*pT) ).evaluate()(0,0);
    Y[0]=mean;
}



double HeatSink2D::output( int output_index, parameter_type const& mu )
{

    using namespace vf;

    element_type u( Xh, "u" );
    element_type v( Xh, "v" );

    this->solve( mu, pT );
    vector_ptrtype U( backend->newVector( Xh ) );
    *U = *pT;

    double s=0;
    for(int i=0;i<Ql(output_index);i++) s += M_thetaFq[output_index](i)*dot( M_Fq[output_index][i], U );

    std::cout<<"output with pfem : "<<s<<std::endl;
    return s;


}

}

#endif /* __HeatSink2D_H */


