/*******************************************************************************
 * Copyright (c) 2004, 2010 Tasktop Technologies and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tasktop Technologies - initial API and implementation
 *     IBM Corporation - helper methods from 
 *       org.eclipse.wst.common.frameworks.internal.ui.WTPActivityHelper 
 *******************************************************************************/

package org.eclipse.mylyn.internal.provisional.commons.ui;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Calendar;

import org.eclipse.core.commands.NotEnabledException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.mylyn.internal.commons.ui.CommonsUiPlugin;
import org.eclipse.mylyn.internal.commons.ui.Messages;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IPluginContribution;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.activities.IIdentifier;
import org.eclipse.ui.activities.IWorkbenchActivitySupport;
import org.eclipse.ui.browser.IWebBrowser;
import org.eclipse.ui.browser.IWorkbenchBrowserSupport;
import org.eclipse.ui.handlers.IHandlerService;
import org.eclipse.ui.internal.browser.WebBrowserPreference;
import org.eclipse.ui.internal.browser.WorkbenchBrowserSupport;
import org.eclipse.ui.services.IServiceLocator;
import org.eclipse.ui.texteditor.IWorkbenchActionDefinitionIds;

/**
 * @author Mik Kersten
 * @author Steffen Pingel
 */
public class WorkbenchUtil {

	public static final String GROUP_EDIT = "group.edit"; //$NON-NLS-1$

	public static final String GROUP_FILE = "group.file"; //$NON-NLS-1$

	public static final String GROUP_REFRESH = "group.refresh"; //$NON-NLS-1$

	public static final String GROUP_FILTER = "group.filter"; //$NON-NLS-1$

	public static final String GROUP_NAVIGATE = "group.navigate"; //$NON-NLS-1$

	public static final String GROUP_NEW = "group.new"; //$NON-NLS-1$

	public static final String GROUP_OPEN = "group.open"; //$NON-NLS-1$

	public static final String GROUP_PREFERENCES = "group.preferences"; //$NON-NLS-1$

	public static final String GROUP_PROPERTIES = "group.properties"; //$NON-NLS-1$

	public static final String GROUP_RUN = "group.run"; //$NON-NLS-1$

	// TODO e3.6 IProgressConstants2#SHOW_IN_TASKBAR_ICON_PROPERTY
	public static final QualifiedName SHOW_IN_TASKBAR_ICON_PROPERTY = new QualifiedName(
			"org.eclipse.ui.workbench.progress", "inTaskBarIcon"); //$NON-NLS-1$//$NON-NLS-2$

	// FIXME remove this again
	private static final boolean TEST_MODE;

	static {
		String application = System.getProperty("eclipse.application", ""); //$NON-NLS-1$ //$NON-NLS-2$
		if (application.length() > 0) {
			TEST_MODE = application.endsWith("testapplication"); //$NON-NLS-1$
		} else {
			// eclipse 3.3 does not the eclipse.application property
			String commands = System.getProperty("eclipse.commands", ""); //$NON-NLS-1$ //$NON-NLS-2$
			TEST_MODE = commands.contains("testapplication\n"); //$NON-NLS-1$
		}
	}

//	public static IViewPart getFromActivePerspective(String viewId) {
//		if (PlatformUI.isWorkbenchRunning()) {
//			IWorkbenchPage activePage = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
//			if (activePage != null) {
//				return activePage.findView(viewId);
//			}
//		}
//		return null;
//	}

	public static IViewPart showViewInActiveWindow(String viewId) {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if (window != null) {
			IWorkbenchPage page = window.getActivePage();
			if (page != null) {
				try {
					return page.showView(viewId);
				} catch (PartInitException e) {
					// ignore
				}
			}
		}
		return null;
	}

	/**
	 * Return the modal shell that is currently open. If there isn't one then return null.
	 * <p>
	 * <b>Note: Applied from patch on bug 99472.</b>
	 * 
	 * @param shell
	 *            A shell to exclude from the search. May be <code>null</code>.
	 * @return Shell or <code>null</code>.
	 */
	private static Shell getModalShellExcluding(Shell shell) {
		IWorkbench workbench = PlatformUI.getWorkbench();
		Shell[] shells = workbench.getDisplay().getShells();
		int modal = SWT.APPLICATION_MODAL | SWT.SYSTEM_MODAL | SWT.PRIMARY_MODAL;
		for (Shell shell2 : shells) {
			if (shell2.equals(shell)) {
				break;
			}
			// Do not worry about shells that will not block the user.
			if (shell2.isVisible()) {
				int style = shell2.getStyle();
				if ((style & modal) != 0) {
					return shell2;
				}
			}
		}
		return null;
	}

	/**
	 * Utility method to get the best parenting possible for a dialog. If there is a modal shell create it so as to
	 * avoid two modal dialogs. If not then return the shell of the active workbench window. If neither can be found
	 * return null.
	 * <p>
	 * <b>Note: Applied from patch on bug 99472.</b>
	 * 
	 * @return Shell or <code>null</code>
	 */
	public static Shell getShell() {
		if (!PlatformUI.isWorkbenchRunning() || PlatformUI.getWorkbench().isClosing()) {
			return null;
		}
		Shell modal = getModalShellExcluding(null);
		if (modal != null) {
			return modal;
		}
		return getNonModalShell();
	}

	/**
	 * Get the active non modal shell. If there isn't one return null.
	 * <p>
	 * <b>Note: Applied from patch on bug 99472.</b>
	 * 
	 * @return Shell
	 */
	private static Shell getNonModalShell() {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if (window == null) {
			IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
			if (windows.length > 0) {
				return windows[0].getShell();
			}
		} else {
			return window.getShell();
		}

		return null;
	}

	/**
	 * @return whether the UI is set up to filter contributions (has defined activity categories).
	 */
	public static final boolean isFiltering() {
		return !PlatformUI.getWorkbench().getActivitySupport().getActivityManager().getDefinedActivityIds().isEmpty();
	}

	public static boolean allowUseOf(Object object) {
		if (!isFiltering()) {
			return true;
		}
		if (object instanceof IPluginContribution) {
			IPluginContribution contribution = (IPluginContribution) object;
			if (contribution.getPluginId() != null) {
				IWorkbenchActivitySupport workbenchActivitySupport = PlatformUI.getWorkbench().getActivitySupport();
				IIdentifier identifier = workbenchActivitySupport.getActivityManager().getIdentifier(
						createUnifiedId(contribution));
				return identifier.isEnabled();
			}
		}
		if (object instanceof String) {
			IWorkbenchActivitySupport workbenchActivitySupport = PlatformUI.getWorkbench().getActivitySupport();
			IIdentifier identifier = workbenchActivitySupport.getActivityManager().getIdentifier((String) object);
			return identifier.isEnabled();
		}
		return true;
	}

	private static final String createUnifiedId(IPluginContribution contribution) {
		if (contribution.getPluginId() != null) {
			return contribution.getPluginId() + '/' + contribution.getLocalId();
		}
		return contribution.getLocalId();
	}

	/**
	 * Opens <code>location</code> in a web-browser according to the Eclipse workbench preferences.
	 * 
	 * @param location
	 *            the url to open
	 * @see #openUrl(String, int)
	 */
	public static void openUrl(String location) {
		openUrl(location, SWT.NONE);
	}

	/**
	 * Opens <code>location</code> in a web-browser according to the Eclipse workbench preferences.
	 * 
	 * @param location
	 *            the url to open
	 * @param customFlags
	 *            additional flags that are passed to {@link IWorkbenchBrowserSupport}, pass
	 *            {@link IWorkbenchBrowserSupport#AS_EXTERNAL} to force opening external browser
	 */
	public static void openUrl(String location, int customFlags) {
		try {
			URL url = null;
			if (location != null) {
				url = new URL(location);
			}
			if (WebBrowserPreference.getBrowserChoice() == WebBrowserPreference.EXTERNAL
					|| (customFlags & IWorkbenchBrowserSupport.AS_EXTERNAL) != 0) {
				try {
					IWorkbenchBrowserSupport support = PlatformUI.getWorkbench().getBrowserSupport();
					support.getExternalBrowser().openURL(url);
				} catch (PartInitException e) {
					Status status = new Status(IStatus.ERROR, CommonsUiPlugin.ID_PLUGIN,
							Messages.WorkbenchUtil_Browser_Initialization_Failed);
					CommonsUiPlugin.getDefault().getLog().log(status);
					if (!TEST_MODE) {
						MessageDialog.openError(getShell(), Messages.WorkbenchUtil_Open_Location_Title,
								status.getMessage());
					}
				}
			} else {
				IWebBrowser browser = null;
				int flags = customFlags;
				if (WorkbenchBrowserSupport.getInstance().isInternalWebBrowserAvailable()) {
					flags |= IWorkbenchBrowserSupport.AS_EDITOR | IWorkbenchBrowserSupport.LOCATION_BAR
							| IWorkbenchBrowserSupport.NAVIGATION_BAR;
				} else {
					flags |= IWorkbenchBrowserSupport.AS_EXTERNAL | IWorkbenchBrowserSupport.LOCATION_BAR
							| IWorkbenchBrowserSupport.NAVIGATION_BAR;
				}

				String generatedId = "org.eclipse.mylyn.web.browser-" + Calendar.getInstance().getTimeInMillis(); //$NON-NLS-1$
				browser = WorkbenchBrowserSupport.getInstance().createBrowser(flags, generatedId, null, null);
				browser.openURL(url);
			}
		} catch (PartInitException e) {
			Status status = new Status(IStatus.ERROR, CommonsUiPlugin.ID_PLUGIN,
					Messages.WorkbenchUtil_Browser_Initialization_Failed, e);
			CommonsUiPlugin.getDefault().getLog().log(status);
			if (!TEST_MODE) {
				MessageDialog.openError(getShell(), Messages.WorkbenchUtil_Open_Location_Title, status.getMessage());
			}
		} catch (MalformedURLException e) {
			if (location != null && location.trim().equals("")) { //$NON-NLS-1$
				Status status = new Status(IStatus.WARNING, CommonsUiPlugin.ID_PLUGIN,
						Messages.WorkbenchUtil_No_URL_Error, e);
				if (!TEST_MODE) {
					MessageDialog.openWarning(getShell(), Messages.WorkbenchUtil_Open_Location_Title,
							status.getMessage());
				} else {
					CommonsUiPlugin.getDefault().getLog().log(status);
				}
			} else {
				Status status = new Status(IStatus.ERROR, CommonsUiPlugin.ID_PLUGIN, NLS.bind(
						Messages.WorkbenchUtil_Invalid_URL_Error, location), e);
				if (!TEST_MODE) {
					MessageDialog.openError(getShell(), Messages.WorkbenchUtil_Open_Location_Title, status.getMessage());
				} else {
					CommonsUiPlugin.getDefault().getLog().log(status);
				}
			}
		}
	}

	public static void addDefaultGroups(IMenuManager menuManager) {
		menuManager.add(new Separator(GROUP_NEW));
		menuManager.add(new Separator(GROUP_OPEN));
		menuManager.add(new Separator(GROUP_EDIT));
		menuManager.add(new Separator(GROUP_FILE));
		menuManager.add(new Separator(GROUP_RUN));
		menuManager.add(new Separator(GROUP_NAVIGATE));
		menuManager.add(new Separator(GROUP_REFRESH));
		menuManager.add(new Separator(GROUP_FILTER));
		menuManager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
		menuManager.add(new Separator(GROUP_PROPERTIES));
	}

	public static Object openProperties(IServiceLocator serviceLocator) {
		IHandlerService service = (IHandlerService) serviceLocator.getService(IHandlerService.class);
		if (service != null) {
			try {
				return service.executeCommand(IWorkbenchActionDefinitionIds.PROPERTIES, null);
			} catch (NotEnabledException e) {
				// ignore
			} catch (Exception e) {
				CommonsUiPlugin.getDefault()
						.getLog()
						.log(new Status(IStatus.ERROR, CommonsUiPlugin.ID_PLUGIN,
								"Opening repository properties failed", e)); //$NON-NLS-1$
			}
		}
		return IStatus.CANCEL;
	}

}
